﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Runtime.InteropServices;

using Peak.Can.IsoTp;
using Peak.Can.Uds;
using System.Threading;

namespace server_uds_and_can
{
    class Program
    {
        static String OK_KO(bool test)
        {
            if (test)
                return "OK";
            return "KO";
        }
        static String UDS_STATUS_OK_KO(uds_status test)
        {
            return OK_KO(UDSApi.StatusIsOk_2013(test));
        }
        static String ISOTP_STATUS_OK_KO(cantp_status test)
        {
            return OK_KO(CanTpApi.StatusIsOk_2016(test));
        }

        const int BUFFER_SIZE = 256;
        const String ISOTP_PING_MSG = "PING";
        const UInt32 ISOTP_PING_MSG_SIZE = 4;
        const UInt32 CAN_ID_START_RANGE = 1;
        const UInt32 CAN_ID_STOP_RANGE = 10;
        const int PING_TIMING_MS = 500;

        /// <summary>Structure passed as thread parameters</summary>
        public struct task_params
        {
            /// <summary>Server channel handle</summary>
            public cantp_handle server_handle;
            /// <summary>Server address</summary>
            public UInt32 server_address;
            /// <summary>Determine if the thread should end or not</summary>
            public bool stop_task;
        };

        /// <summary>ISOTP server task: periodically send "PING" message using different CAN identifiers</summary>
        /// <param name="parameters">pointer on task_params structures</param>
        static void isotp_server_task(ref task_params parameters)
        {
            cantp_status status;
            cantp_msg tx_msg;
            UInt32 can_id;

            // Init variables
            tx_msg = new cantp_msg();
            can_id = CAN_ID_START_RANGE;

            // Send loop
            do
            {
                // Wait before sending the next message
                Thread.Sleep(PING_TIMING_MS);

                // Initialize ISOTP Tx message containing "PING"
                status = CanTpApi.MsgDataAlloc_2016(out tx_msg, cantp_msgtype.PCANTP_MSGTYPE_CAN);
                Console.WriteLine("[ISOTP] Allocate ISOTP tx message: {0}", ISOTP_STATUS_OK_KO(status));
                Byte[] bytes_array = Encoding.ASCII.GetBytes(ISOTP_PING_MSG);
                status = CanTpApi.MsgDataInit_2016(out tx_msg, can_id, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, ISOTP_PING_MSG_SIZE, bytes_array);
                Console.WriteLine("[ISOTP] Initialize ISOTP tx message: {0}", ISOTP_STATUS_OK_KO(status));

                // Send "PING" message
                status = CanTpApi.Write_2016(parameters.server_handle, ref tx_msg);
                Console.WriteLine("[ISOTP] Send ISOTP \"PING\" message (can id=0x{0:x}): {1}", can_id, ISOTP_STATUS_OK_KO(status));

                // Free messages
                status = CanTpApi.MsgDataFree_2016(ref tx_msg);
                Console.WriteLine("[ISOTP] Free ISOTP TX message: {0}", ISOTP_STATUS_OK_KO(status));

                // Update can id for next message
                can_id++;
                if (can_id > CAN_ID_STOP_RANGE)
                {
                    can_id = CAN_ID_START_RANGE;
                }
            } while (parameters.stop_task == false);
        }

        /// <summary>UDS server task: respond TesterPresent request</summary>
        /// <param name="parameters">pointer on task_params structures</param>
        static void uds_server_task(ref task_params parameters)
        {
            uds_msgconfig config_physical;
            bool wait_result;
            uds_status status;
            uds_status read_status;
            uds_msg request_msg;
            uds_msg response_msg;

            // Init variables
            config_physical = new uds_msgconfig();
            request_msg = new uds_msg();
            response_msg = new uds_msg();

            // Set server address parameter
            status = UDSApi.SetValue_2013(parameters.server_handle, uds_parameter.PUDS_PARAMETER_SERVER_ADDRESS, ref parameters.server_address, sizeof(UInt32));
            Console.WriteLine("[UDS] Set UDS server address: {0}", UDS_STATUS_OK_KO(status));

            // Set a receive event for UDS
            System.Threading.AutoResetEvent receive_event = new System.Threading.AutoResetEvent(false);
            if (IntPtr.Size == 4)
            {
                UInt32 tmp_buffer = Convert.ToUInt32(receive_event.SafeWaitHandle.DangerousGetHandle().ToInt32());
                status = UDSApi.SetValue_2013(parameters.server_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, ref tmp_buffer, sizeof(UInt32));
            }
            else if (IntPtr.Size == 8)
            {
                Int64 tmp_buffer = receive_event.SafeWaitHandle.DangerousGetHandle().ToInt64();
                byte[] byte_array = BitConverter.GetBytes(tmp_buffer);
                status = UDSApi.SetValue_2013(parameters.server_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, byte_array, sizeof(UInt64));
            }
            Console.WriteLine("[UDS] Set UDS receive event parameter: {0}", UDS_STATUS_OK_KO(status));

            // Initialize a physical configuration
            config_physical.can_id = 0xFFFFFFFF;
            config_physical.can_msgtype = cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD;
            config_physical.nai.protocol = uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL;
            config_physical.nai.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;
            config_physical.type = uds_msgtype.PUDS_MSGTYPE_USDT;
            config_physical.nai.extension_addr = 0;

            do
            {

                // Wait a receive event on receiver
                wait_result = receive_event.WaitOne(1000);

                // If we get a receive event
                if (wait_result)
                {
                    do
                    {

                        // Read first available message (no filtering based on message's type is set):
                        read_status = UDSApi.Read_2013(parameters.server_handle, out request_msg);
                        Console.WriteLine("[UDS] Try to read a message: {0}", UDS_STATUS_OK_KO(read_status));
                        if (UDSApi.StatusIsOk_2013(read_status))
                        {

                            // We receive a request, check its length and if it is not a loopback message and if it is a USDT message
                            if (request_msg.type == uds_msgtype.PUDS_MSGTYPE_USDT && request_msg.msg.Msgdata_any_Copy.length >= 1 && (request_msg.msg.Msgdata_any_Copy.flags & cantp_msgflag.PCANTP_MSGFLAG_LOOPBACK) == 0)
                            {

                                // This is a valid request, switch services
                                byte service_id;
                                bool get_sid = UDSApi.GetDataServiceId_2013(ref request_msg, out service_id);
                                if (get_sid)
                                {
                                    switch ((uds_service)service_id)
                                    {
                                        case uds_service.PUDS_SERVICE_SI_TesterPresent:

                                            // Allocate response message
                                            status = UDSApi.MsgAlloc_2013(out response_msg, config_physical, 2);
                                            Console.WriteLine("[UDS] Prepare response message for TesterPresent service: {0}", UDS_STATUS_OK_KO(status));

                                            if (UDSApi.StatusIsOk_2013(status))
                                            {
                                                // Fill parameters
                                                cantp_netaddrinfo nadr = request_msg.msg.Msgdata_isotp_Copy.netaddrinfo;
                                                nadr.target_addr = nadr.source_addr;
                                                nadr.source_addr = (UInt16)parameters.server_address;
                                                CanTpApi.setNetaddrinfo_2016(ref response_msg.msg, ref nadr);
                                                UDSApi.SetDataServiceId_2013(ref response_msg, (Byte)uds_service.PUDS_SERVICE_SI_TesterPresent + UDSApi.PUDS_SI_POSITIVE_RESPONSE);
                                                UDSApi.SetDataParameter_2013(ref response_msg, 0, 0);

                                                // Write response message
                                                status = UDSApi.Write_2013(parameters.server_handle, ref response_msg);
                                                Console.WriteLine("[UDS] Write response message for TesterPresent service: {0}", UDS_STATUS_OK_KO(status));
                                            }

                                            // Free response message (and clean memory in order to reallocate later)
                                            status = UDSApi.MsgFree_2013(ref response_msg);
                                            Console.WriteLine("[UDS] Free response message: {0}", UDS_STATUS_OK_KO(status));
                                            break;
                                        default:
                                            Console.WriteLine("[UDS] Unknown service (0x{0:x2})", service_id);
                                            break;
                                    }
                                }
                            }
                        }

                        // Free request message (and clean memory in order to reallocate later)
                        status = UDSApi.MsgFree_2013(ref request_msg);
                        Console.WriteLine("[UDS] Free request message: {0}", UDS_STATUS_OK_KO(status));
                    } while (!UDSApi.StatusIsOk_2013(read_status, uds_status.PUDS_STATUS_NO_MESSAGE, false));
                }
            } while (parameters.stop_task == false);

            // Close receive event
            if (IntPtr.Size == 4)
            {
                UInt32 tmp_buffer = 0;
                status = UDSApi.SetValue_2013(parameters.server_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, ref tmp_buffer, sizeof(UInt32));
            }
            else if (IntPtr.Size == 8)
            {
                Int64 tmp_buffer = 0;
                byte[] byte_array = BitConverter.GetBytes(tmp_buffer);
                status = UDSApi.SetValue_2013(parameters.server_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, byte_array, sizeof(UInt64));
            }
            Console.WriteLine("[UDS] Stop UDS receive event: {0}", UDS_STATUS_OK_KO(status));
            receive_event.Close();
            Console.WriteLine("[UDS] Close UDS receive event: {0}", UDS_STATUS_OK_KO(status));
        }

        /// <summary>Entry point of the program, start a small server which handle UDS testerpresent request and periodically send isotp messages</summary>
        /// <returns>By convention, return success.</returns>
        static void Main(string[] args)
        {
            uds_status status;
            task_params t_params;
            StringBuilder buffer = new StringBuilder(BUFFER_SIZE);
            Thread uds_server;
            Thread isotp_server;
            int keyboard_res;

            // Initialize variables
            t_params.server_handle = cantp_handle.PCANTP_HANDLE_USBBUS2; // TODO: modify the value according to your available PCAN devices.
            t_params.server_address = (UInt32)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_1;
            t_params.stop_task = false;

            // Print version informations
            status = UDSApi.GetValue_2013(cantp_handle.PCANTP_HANDLE_NONEBUS, uds_parameter.PUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE);
            Console.WriteLine("PCAN-UDS API Version - {0}: {1}", buffer, UDS_STATUS_OK_KO(status));

            // Initialize server
            status = UDSApi.Initialize_2013(t_params.server_handle, cantp_baudrate.PCANTP_BAUDRATE_500K, (cantp_hwtype)0, 0, 0);
            Console.WriteLine("Initialize channel: {0}", UDS_STATUS_OK_KO(status));

            // Start uds and isotp servers
            uds_server = new Thread(() => uds_server_task(ref t_params));
            uds_server.Start();
            isotp_server = new Thread(() => isotp_server_task(ref t_params));
            isotp_server.Start();

            // Read while user do not press Q
            Console.WriteLine("Start listening, press Q to quit.");
            t_params.stop_task = false;
            do
            {
                // Quit when user press Q
                if (Console.KeyAvailable)
                {
                    keyboard_res = Console.ReadKey().KeyChar;
                    if (keyboard_res == 'Q' || keyboard_res == 'q')
                    {
                        t_params.stop_task = true;
                    }
                }
            } while (t_params.stop_task == false);

            // Close threads
            isotp_server.Join();
            uds_server.Join();

            // Close channel
            status = UDSApi.Uninitialize_2013(t_params.server_handle);
            Console.WriteLine("Uninitialize channel: {0}", UDS_STATUS_OK_KO(status));

            //Exit
            Console.WriteLine("Press any key to continue...");
            Console.In.Read();
        }
    }
}
