﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Runtime.InteropServices;

using Peak.Can.IsoTp;
using Peak.Can.Uds;
using System.Threading;

namespace client_uds_and_can
{
    class Program
    {
        const int BUFFER_SIZE = 256;
        const int UDS_REQUEST_TIMING_MS = 500;
        const int CLIENT_EXECUTION_TIME_MS = 10000;

        static String OK_KO(bool test)
        {
            if (test)
                return "OK";
            return "KO";
        }
        static String UDS_STATUS_OK_KO(uds_status test)
        {
            return OK_KO(UDSApi.StatusIsOk_2013(test));
        }
        static String ISOTP_STATUS_OK_KO(cantp_status test)
        {
            return OK_KO(CanTpApi.StatusIsOk_2016(test));
        }

        /// <summary>Structure passed as thread parameters</summary>
        public struct task_params
        {
            /// <summary>Client channel handle</summary>
            public cantp_handle client_handle;
            /// <summary>Determine if the thread should end or not</summary>
            public bool stop_task;
        };

        /// <summary>UDS client task: request TesterPresent service several time</summary>
        /// <param name="parameters">pointer on task_params structures</param>
        static void uds_client_task(ref task_params parameters)
        {
            uds_status status;
            uds_msgconfig config_physical;
            uds_msg msg_request;
            uds_msg request_confirmation;
            uds_msg response;

            // Initialize variables
            config_physical = new uds_msgconfig();
            msg_request = new uds_msg();
            request_confirmation = new uds_msg();
            response = new uds_msg();

            // Initialize a physical configuration
            config_physical.can_id = (UInt32)uds_can_id.PUDS_CAN_ID_ISO_15765_4_PHYSICAL_REQUEST_1;
            config_physical.can_msgtype = cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD;
            config_physical.nai.protocol = uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL;
            config_physical.nai.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;
            config_physical.type = uds_msgtype.PUDS_MSGTYPE_USDT;
            config_physical.nai.source_addr = (UInt16)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
            config_physical.nai.target_addr = (UInt16)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_1;
            config_physical.nai.extension_addr = 0;

            // Execute TesterPresent and wait response
            do
            {
                // Wait before transmitting the next message
                Thread.Sleep(UDS_REQUEST_TIMING_MS);

                status = UDSApi.SvcTesterPresent_2013(parameters.client_handle, config_physical, out msg_request, UDSApi.uds_svc_param_tp.PUDS_SVC_PARAM_TP_ZSUBF);
                Console.WriteLine("[UDS] Execute TesterPresent service: {0}", UDS_STATUS_OK_KO(status));
                status = UDSApi.WaitForService_2013(parameters.client_handle, ref msg_request, out response, out request_confirmation);
                Console.WriteLine("[UDS] Wait for service: {0}", UDS_STATUS_OK_KO(status));
                status = UDSApi.MsgFree_2013(ref msg_request);
                Console.WriteLine("[UDS] Free request message: {0}", UDS_STATUS_OK_KO(status));
                status = UDSApi.MsgFree_2013(ref request_confirmation);
                Console.WriteLine("[UDS] Free request confirmation message: {0}", UDS_STATUS_OK_KO(status));
                status = UDSApi.MsgFree_2013(ref response);
                Console.WriteLine("[UDS] Free response message: {0}", UDS_STATUS_OK_KO(status));
            } while (parameters.stop_task == false);
        }

        /// <summary>ISOTP client task: read and print isotp messages</summary>
        /// <param name="parameters">pointer on task_params structures</param>
        static void isotp_client_task(ref task_params parameters)
        {
            // Initialize variables
            cantp_status status;
            cantp_msg rx_msg;
            bool wait_result;
            UInt32 isotp_param;
            rx_msg = new cantp_msg();

            // Configure isotp to get any messages (disable can identifier filtering)
            isotp_param = 0;
            status = CanTpApi.SetValue_2016(parameters.client_handle, cantp_parameter.PCANTP_PARAMETER_FILTER_CAN_ID, ref isotp_param, sizeof(UInt32));
            Console.WriteLine("[ISOTP] Disable can identifier filtering in isotp: {0}", ISOTP_STATUS_OK_KO(status));

            // Configure isotp to get a copy of UDS messages
            isotp_param = 1;
            status = CanTpApi.SetValue_2016(parameters.client_handle, cantp_parameter.PCANTP_PARAMETER_KEEP_HIGHER_LAYER_MESSAGES, ref isotp_param, sizeof(UInt32));
            Console.WriteLine("[ISOTP] Activate higher layer messages in isotp: {0}", ISOTP_STATUS_OK_KO(status));

            // Create a isotp receive event
            System.Threading.AutoResetEvent receive_event = new System.Threading.AutoResetEvent(false);
            if (IntPtr.Size == 4)
            {
                UInt32 tmp_buffer = Convert.ToUInt32(receive_event.SafeWaitHandle.DangerousGetHandle().ToInt32());
                status = CanTpApi.SetValue_2016(parameters.client_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, ref tmp_buffer, sizeof(UInt32));
            }
            else if (IntPtr.Size == 8)
            {
                Int64 tmp_buffer = receive_event.SafeWaitHandle.DangerousGetHandle().ToInt64();
                byte[] byte_array = BitConverter.GetBytes(tmp_buffer);
                status = CanTpApi.SetValue_2016(parameters.client_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, byte_array, sizeof(UInt64));
            }
            Console.WriteLine("[ISOTP] Set isotp receive event parameter: {0}", ISOTP_STATUS_OK_KO(status));

            // Read and print ISOTP messages
            do
            {
                wait_result = receive_event.WaitOne(1000);
                Console.WriteLine("[ISOTP] Wait a receive event from isotp: {0}", OK_KO(wait_result));

                // Read ISOTP messages
                do
                {
                    status = CanTpApi.Read_2016(parameters.client_handle, out rx_msg);

                    // Check if we received a message
                    if (!CanTpApi.StatusIsOk_2016(status, cantp_status.PCANTP_STATUS_NO_MESSAGE, false))
                    {
                        Console.WriteLine("[ISOTP] Read ISOTP message: {0}", ISOTP_STATUS_OK_KO(status));
                        Byte data_0;
                        bool get_data_0 = CanTpApi.getData_2016(ref rx_msg, 0, out data_0);
                        if (get_data_0 && (data_0 == (Byte)uds_service.PUDS_SERVICE_SI_TesterPresent || data_0 == (Byte)(uds_service.PUDS_SERVICE_SI_TesterPresent + UDSApi.PUDS_SI_POSITIVE_RESPONSE)))
                        {
                            // This message is a TesterPresent message, PCANTP_PARAMETER_KEEP_HIGHER_LAYER_MESSAGES
                            // option must be activated to get these messages.
                            Console.WriteLine("[ISOTP] Message is a TesterPresent service message.");
                        }
                        else if (CanTpApi.StatusIsOk_2016(status) && rx_msg.Msgdata_any_Copy.length < BUFFER_SIZE)
                        {
                            // This message is a CAN message received by ISOTP
                            StringBuilder msg_data = new StringBuilder();
                            for (int i = 0; i < (int)rx_msg.Msgdata_any_Copy.length; i++)
                            {
                                Byte data;
                                if (CanTpApi.getData_2016(ref rx_msg, i, out data))
                                {
                                    msg_data.Append(Convert.ToChar(data));
                                }
                            }
                            Console.WriteLine("[ISOTP] Message from 0x{0:x} can identifier contains \"{1}\"", rx_msg.can_info.can_id, msg_data);
                        }

                        status = CanTpApi.MsgDataFree_2016(ref rx_msg);
                        Console.WriteLine("[ISOTP] Free RX message: {0}", ISOTP_STATUS_OK_KO(status));
                    }
                } while (!CanTpApi.StatusIsOk_2016(status, cantp_status.PCANTP_STATUS_NO_MESSAGE, false));
            } while (parameters.stop_task == false);

            // Close receive event
            if (IntPtr.Size == 4)
            {
                UInt32 tmp_buffer = 0;
                status = CanTpApi.SetValue_2016(parameters.client_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, ref tmp_buffer, sizeof(UInt32));
            }
            else if (IntPtr.Size == 8)
            {
                Int64 tmp_buffer = 0;
                byte[] byte_array = BitConverter.GetBytes(tmp_buffer);
                status = CanTpApi.SetValue_2016(parameters.client_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, byte_array, sizeof(UInt64));
            }
            Console.WriteLine("[ISOTP] Stop ISOTP receive event: {0}", ISOTP_STATUS_OK_KO(status));
            receive_event.Close();
            Console.WriteLine("[ISOTP] Close ISOTP receive event: {0}", ISOTP_STATUS_OK_KO(status));
        }

        /// <summary>Entry point of the program, start a UDS channel, ask in the same time UDS TesterPresent and isotp request</summary>
        /// <returns>By convention, return success.</returns>
        static void Main(string[] args)
        {
            StringBuilder buffer = new StringBuilder(BUFFER_SIZE);
            uds_status status;
            cantp_handle client_handle;
            Thread uds_client;
            Thread isotp_client;
            task_params t_params;

            // Initialize variables
            client_handle = cantp_handle.PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.
            t_params.client_handle = client_handle;
            t_params.stop_task = false;

            // Print version informations
            status = UDSApi.GetValue_2013(cantp_handle.PCANTP_HANDLE_NONEBUS, uds_parameter.PUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE);
            Console.WriteLine("PCAN-UDS API Version - {0}: {1}", buffer, UDS_STATUS_OK_KO(status));

            // Initialize client channel
            status = UDSApi.Initialize_2013(client_handle, cantp_baudrate.PCANTP_BAUDRATE_500K, (cantp_hwtype)0, 0, 0);
            Console.WriteLine("Initialize channel: {0}", UDS_STATUS_OK_KO(status));

            // Start uds and isotp clients
            uds_client = new Thread(() => uds_client_task(ref t_params));
            uds_client.Start();

            isotp_client = new Thread(() => isotp_client_task(ref t_params));
            isotp_client.Start();

            Thread.Sleep(CLIENT_EXECUTION_TIME_MS);
            t_params.stop_task = true;

            isotp_client.Join();
            uds_client.Join();

            // Close channel
            status = UDSApi.Uninitialize_2013(client_handle);
            Console.WriteLine("Uninitialize channel: {0}", UDS_STATUS_OK_KO(status));

            // Exit
            Console.WriteLine("Press any key to continue...");
            Console.In.Read();
        }
    }
}
