﻿using System;
using System.Text;
using System.Collections.Generic;
using System.Runtime.InteropServices;

using Peak.Can.Uds;
using TPUDSCANHandle = System.UInt16;

namespace PcanUdsExample
{
    public class ApiUser
    {
        [DllImport("PCAN-ISO-TP.dll", EntryPoint = "CANTP_SetValue")]
        static extern uint IsoTp_SetValue(
            [MarshalAs(UnmanagedType.U2)]
            TPUDSCANHandle CanChannel,
            [MarshalAs(UnmanagedType.U1)]
            byte Parameter,
            ref UInt32 NumericBuffer,
            UInt32 BufferLength);


        public bool IsConnected { get { return mConnectedChannelHandle != 0; } }
        public bool IsCommunicating { get; private set; }
        public bool LocalAddressIsExtended { get { return mLocalNodeAddress > 0xFF; } }
        public uint LocalNodeAddress
        {
            get
            {
                if (LocalAddressIsExtended)
                    return mLocalNodeAddress ^ UDSApi.PUDS_SERVER_ADDR_FLAG_ENHANCED_ISO_15765_3;
                return mLocalNodeAddress;
            }
        }
        public TPUDSMsg Response { get; private set; }
        public TPUDSMsg[] FunctionalResponses { get; private set; }
        public uint ReceivedFunctionalResponses { get; private set; }
        public bool RequestIsFunctional { get; private set; }
        public TPUDSMsg UUDTResponse { get; private set; }
        public bool UUDTError { get { return mUUDTStatus != TPUDSStatus.PUDS_ERROR_OK && mUUDTStatus != TPUDSStatus.PUDS_ERROR_NO_MESSAGE; } }

        private TPUDSCANHandle mConnectedChannelHandle;
        private uint mLocalNodeAddress;
        private TPUDSMsg mRequest;
        private ServiceInformation mServiceInformationOfLastRequest;
        private TPUDSStatus mResponseStatus;
        private TPUDSStatus mUUDTStatus;


        public ApiUser()
        {
            mConnectedChannelHandle = UDSApi.PUDS_NONEBUS;
            mLocalNodeAddress = 0xF1;
            IsCommunicating = false;
            FunctionalResponses = new TPUDSMsg[20];
        }


        #region Connection
        public List<TPUDSCANHandle> GetAvailableChannels()
        {
            TPUDSStatus status;
            List<TPUDSCANHandle> availableChannels = new List<TPUDSCANHandle>();

            foreach (TPUDSCANHandle channel in UdsExampleUtils.AllChannelHandles)
            {
                if (channel == UDSApi.PUDS_NONEBUS)
                    continue;
                if (channel <= UDSApi.PUDS_DNGBUS1)
                    availableChannels.Add(channel);
                else
                {
                    status = UDSApi.GetValue(channel, TPUDSParameter.PUDS_PARAM_CHANNEL_CONDITION, out uint condition, sizeof(uint));

                    if ((status == TPUDSStatus.PUDS_ERROR_OK) &&
                        ((condition & UDSApi.PUDS_CHANNEL_AVAILABLE) == UDSApi.PUDS_CHANNEL_AVAILABLE))
                        availableChannels.Add(channel);
                }
            }

            return availableChannels;
        }

        public bool Connect(TPUDSCANHandle canHandle, TPUDSBaudrate baudrate)
        {
            TPUDSStatus status = UDSApi.Initialize(canHandle, baudrate);

            if (status != TPUDSStatus.PUDS_ERROR_OK)
            {
                ShowError(status);
                return false;
            }
            else
                mLocalNodeAddress = 0xF1;

            mConnectedChannelHandle = canHandle;

            return true;
        }

        public bool ConnectNonPlugAndPlay(TPUDSCANHandle canHandle, TPUDSBaudrate baudrate, TPUDSHWType hardwareType, UInt32 ioPort, UInt16 interrupt)
        {
            TPUDSStatus status = UDSApi.Initialize(canHandle, baudrate, hardwareType, ioPort, interrupt);

            if (status != TPUDSStatus.PUDS_ERROR_OK)
            {
                ShowError(status);
                return false;
            }
            else
                mLocalNodeAddress = 0xF1;

            mConnectedChannelHandle = canHandle;

            return true;
        }

        public bool Uninitialize()
        {
            if (!IsConnected)
                return false;

            TPUDSStatus status = UDSApi.Uninitialize(mConnectedChannelHandle);

            if (status != TPUDSStatus.PUDS_ERROR_OK)
            {
                ShowError(status);
                return false;
            }

            mConnectedChannelHandle = UDSApi.PUDS_NONEBUS;
            return true;
        }

        public bool GetHardwareStatus(out string statusOfDevice)
        {
            statusOfDevice = "";

            if (!IsConnected)
                return false;

            TPUDSStatus status = UDSApi.GetStatus(mConnectedChannelHandle);

            if (status != TPUDSStatus.PUDS_ERROR_OK)
                ShowError(status);
            else
            {
                string errorDiscription;

                switch (status)
                {
                    case TPUDSStatus.PUDS_ERROR_OK:
                        errorDiscription = "PUDS_ERROR_OK";
                        break;
                    case TPUDSStatus.PUDS_ERROR_NOT_INITIALIZED:
                        errorDiscription = "PUDS_ERROR_NOT_INITIALIZED";
                        break;
                    case TPUDSStatus.PUDS_ERROR_BUSLIGHT:
                        errorDiscription = "PUDS_ERROR_BUSLIGHT";
                        break;
                    case TPUDSStatus.PUDS_ERROR_BUSHEAVY:
                        errorDiscription = "PUDS_ERROR_BUSHEAVY";
                        break;
                    case TPUDSStatus.PUDS_ERROR_BUSOFF:
                        errorDiscription = "PUDS_ERROR_BUSOFF";
                        break;
                    default:
                        errorDiscription = "See Documentation";
                        break;
                }

                statusOfDevice = String.Format("Channel status: {0} ({1,8:X}h)", errorDiscription, status);
            }

            return status == TPUDSStatus.PUDS_ERROR_OK;
        }

        public bool ResetHardware()
        {
            if (!IsConnected)
                return false;

            TPUDSStatus status = UDSApi.Reset(mConnectedChannelHandle);

            if (status != TPUDSStatus.PUDS_ERROR_OK)
                ShowError(status);

            return status == TPUDSStatus.PUDS_ERROR_OK;
        }
        #endregion

        #region Parameter

        public bool GetLocalAddress(out uint address)
        {
            TPUDSStatus status = UDSApi.GetValue(mConnectedChannelHandle, TPUDSParameter.PUDS_PARAM_SERVER_ADDRESS, out address, sizeof(uint));

            if (status != TPUDSStatus.PUDS_ERROR_OK)
                ShowError(status);
            else
            {
                if (mLocalNodeAddress != address)
                    mLocalNodeAddress = address;

                if ((address & UDSApi.PUDS_SERVER_ADDR_FLAG_ENHANCED_ISO_15765_3) == UDSApi.PUDS_SERVER_ADDR_FLAG_ENHANCED_ISO_15765_3)
                    address ^= UDSApi.PUDS_SERVER_ADDR_FLAG_ENHANCED_ISO_15765_3;
            }

            return status == TPUDSStatus.PUDS_ERROR_OK;
        }

        public bool SetLocalAddress(uint address, bool isEnhanced)
        {
            //ISO 14229-1:2006 addresses need to be marked by the flag PUDS_SERVER_ADDR_FLAG_ENHANCED_ISO_15765_3
            if (isEnhanced)
                address |= UDSApi.PUDS_SERVER_ADDR_FLAG_ENHANCED_ISO_15765_3;

            TPUDSStatus status = UDSApi.SetValue(mConnectedChannelHandle, TPUDSParameter.PUDS_PARAM_SERVER_ADDRESS, ref address, sizeof(uint));

            if (status != TPUDSStatus.PUDS_ERROR_OK)
                ShowError(status);
            else
                mLocalNodeAddress = address;

            return status == TPUDSStatus.PUDS_ERROR_OK;
        }


        public bool AddAddressToFilter(uint address)
        {
            address |= UDSApi.PUDS_SERVER_FILTER_LISTEN;
            TPUDSStatus status = UDSApi.SetValue(mConnectedChannelHandle, TPUDSParameter.PUDS_PARAM_SERVER_FILTER, ref address, sizeof(uint));

            if (status != TPUDSStatus.PUDS_ERROR_OK)
                ShowError(status);

            return status == TPUDSStatus.PUDS_ERROR_OK;
        }

        public bool RemoveAddressFromFilter(uint address)
        {
            address |= UDSApi.PUDS_SERVER_FILTER_IGNORE;
            TPUDSStatus status = UDSApi.SetValue(mConnectedChannelHandle, TPUDSParameter.PUDS_PARAM_SERVER_FILTER, ref address, sizeof(uint));

            if (status != TPUDSStatus.PUDS_ERROR_OK)
                ShowError(status);

            return status == TPUDSStatus.PUDS_ERROR_OK;
        }


        public bool GetUIntParameter(TPUDSParameter parameter, out UInt32 value)
        {
            TPUDSStatus status = UDSApi.GetValue(mConnectedChannelHandle, parameter, out value, sizeof(UInt32));

            if (status != TPUDSStatus.PUDS_ERROR_OK)
                ShowError(status);

            return status == TPUDSStatus.PUDS_ERROR_OK;
        }

        public bool GetStringParameter(TPUDSParameter parameter, out string value)
        {
            StringBuilder stringBuilder = new StringBuilder(256);

            TPUDSStatus status = UDSApi.GetValue(mConnectedChannelHandle, parameter, stringBuilder, (UInt32)stringBuilder.Capacity);

            if (status != TPUDSStatus.PUDS_ERROR_OK)
                ShowError(status);

            value = stringBuilder.ToString();

            return status == TPUDSStatus.PUDS_ERROR_OK;
        }

        public bool GetApiVersion(out string information)
        {
            information = "";

            if (GetStringParameter(TPUDSParameter.PUDS_PARAM_API_VERSION, out string value))
            {
                information = "Api Version: " + value;
                return true;
            }

            return false;
        }


        public bool SetParameter(TPUDSParameter parameter, UInt32 value)
        {
            TPUDSStatus status = UDSApi.SetValue(mConnectedChannelHandle, parameter, ref value, sizeof(UInt32));

            if (status != TPUDSStatus.PUDS_ERROR_OK)
                ShowError(status);

            return status == TPUDSStatus.PUDS_ERROR_OK;
        }
        #endregion

        #region Mapping
        public bool AddMapping(MappingInformation mapping)
        {
            TPUDSStatus status = UDSApi.SetValue(mConnectedChannelHandle, TPUDSParameter.PUDS_PARAM_MAPPING_ADD, mapping.MappingPointer, mapping.PointerSize);

            if (status != TPUDSStatus.PUDS_ERROR_OK)
                ShowError(status);

            return status == TPUDSStatus.PUDS_ERROR_OK;
        }

        public bool RemoveMapping(MappingInformation mapping)
        {
            TPUDSStatus status = UDSApi.SetValue(mConnectedChannelHandle, TPUDSParameter.PUDS_PARAM_MAPPING_REMOVE, mapping.MappingPointerRemoval, mapping.PointerSize);

            if (status != TPUDSStatus.PUDS_ERROR_OK)
                ShowError(status);

            return status == TPUDSStatus.PUDS_ERROR_OK;
        }
        #endregion

        #region Send/Receive
        public bool SendService(ServiceInformation serviceInformation)
        {
            TPUDSStatus status = 0;
            mRequest.NETADDRINFO = serviceInformation.NetAddressInformation;
            mServiceInformationOfLastRequest = serviceInformation;
            RequestIsFunctional = serviceInformation.NetAddressInformation.TA_TYPE == TPUDSAddressingType.PUDS_ADDRESSING_FUNCTIONAL;

            if (UdsExampleUtils.ProtocolIsDetachedFromIdsAndAddresses(serviceInformation.NetAddressInformation.PROTOCOL))
                SetPriority(serviceInformation.Priority);

            byte[] byteBuffer;

            switch (serviceInformation.Service)
            {
                case TPUDSService.PUDS_SI_TesterPresent:
                    status = UDSApi.SvcTesterPresent(mConnectedChannelHandle, ref mRequest, UDSApi.TPUDSSvcParamTP.PUDS_SVC_PARAM_TP_ZSUBF);
                    break;
                case TPUDSService.PUDS_SI_DiagnosticSessionControl:
                    UDSApi.TPUDSSvcParamDSC paramDSC = (UDSApi.TPUDSSvcParamDSC)serviceInformation.ParametersOfService[0];
                    status = UDSApi.SvcDiagnosticSessionControl(mConnectedChannelHandle, ref mRequest, paramDSC);
                    break;
                case TPUDSService.PUDS_SI_ReadDataByIdentifier:
                    ushort[] identifiers = (ushort[])serviceInformation.ParametersOfService[0];
                    status = UDSApi.SvcReadDataByIdentifier(mConnectedChannelHandle, ref mRequest, identifiers, (ushort)identifiers.Length);
                    break;
                case TPUDSService.PUDS_SI_ClearDiagnosticInformation:
                    uint groupOfDtc = Convert.ToUInt32(serviceInformation.ParametersOfService[0]);
                    status = UDSApi.SvcClearDiagnosticInformation(mConnectedChannelHandle, ref mRequest, groupOfDtc);
                    break;
                case TPUDSService.PUDS_SI_ReadDTCInformation:
                    UDSApi.TPUDSSvcParamRDTCI rdtciType = (UDSApi.TPUDSSvcParamRDTCI)serviceInformation.ParametersOfService[0];
                    byte dtcStatusMask = Convert.ToByte(serviceInformation.ParametersOfService[1]);
                    status = UDSApi.SvcReadDTCInformation(mConnectedChannelHandle, ref mRequest, rdtciType, dtcStatusMask);
                    break;
                case TPUDSService.PUDS_SI_ControlDTCSetting:
                    UDSApi.TPUDSSvcParamCDTCS dtcSettingType = (UDSApi.TPUDSSvcParamCDTCS)serviceInformation.ParametersOfService[0];
                    byteBuffer = (byte[])serviceInformation.ParametersOfService[1];
                    status = UDSApi.SvcControlDTCSetting(mConnectedChannelHandle, ref mRequest, dtcSettingType, byteBuffer, (ushort)byteBuffer.Length);
                    break;
                case TPUDSService.PUDS_SI_CommunicationControl:
                    UDSApi.TPUDSSvcParamCC controlType = (UDSApi.TPUDSSvcParamCC)serviceInformation.ParametersOfService[0];
                    byte communicationType = Convert.ToByte(serviceInformation.ParametersOfService[1]);
                    status = UDSApi.SvcCommunicationControl(mConnectedChannelHandle, ref mRequest, controlType, communicationType);
                    break;
                case TPUDSService.PUDS_SI_SecurityAccess:
                    byte securityAccessType = Convert.ToByte(serviceInformation.ParametersOfService[0]);
                    byteBuffer = (byte[])serviceInformation.ParametersOfService[1];
                    status = UDSApi.SvcSecurityAccess(mConnectedChannelHandle, ref mRequest, securityAccessType, byteBuffer, (ushort)byteBuffer.Length);
                    break;
                case TPUDSService.PUDS_SI_RequestDownload:
                    byte compressionMethod = Convert.ToByte(serviceInformation.ParametersOfService[0]);
                    byte encryptingMethod = Convert.ToByte(serviceInformation.ParametersOfService[1]);
                    byte[] memoryAddress = (byte[])serviceInformation.ParametersOfService[2];
                    byte[] memorySize = (byte[])serviceInformation.ParametersOfService[3];
                    status = UDSApi.SvcRequestDownload(mConnectedChannelHandle, ref mRequest, compressionMethod, encryptingMethod, memoryAddress,
                                                            (byte)memoryAddress.Length, memorySize, (byte)memorySize.Length);
                    break;
                case TPUDSService.PUDS_SI_TransferData:
                    byte blockSequenceCounter = Convert.ToByte(serviceInformation.ParametersOfService[0]);
                    byteBuffer = (byte[])serviceInformation.ParametersOfService[1];
                    status = UDSApi.SvcTransferData(mConnectedChannelHandle, ref mRequest, blockSequenceCounter, byteBuffer, (ushort)byteBuffer.Length);
                    break;
                case TPUDSService.PUDS_SI_RequestTransferExit:
                    byteBuffer = (byte[])serviceInformation.ParametersOfService[0];
                    status = UDSApi.SvcRequestTransferExit(mConnectedChannelHandle, ref mRequest, byteBuffer, (ushort)byteBuffer.Length);
                    break;
                case TPUDSService.PUDS_SI_RoutineControl:
                    UDSApi.TPUDSSvcParamRC routineControlType = (UDSApi.TPUDSSvcParamRC)serviceInformation.ParametersOfService[0];
                    ushort routineIdentifier = Convert.ToUInt16(serviceInformation.ParametersOfService[1]);
                    byteBuffer = (byte[])serviceInformation.ParametersOfService[2];
                    status = UDSApi.SvcRoutineControl(mConnectedChannelHandle, ref mRequest, routineControlType, routineIdentifier, byteBuffer, (ushort)byteBuffer.Length);
                    break;
                case TPUDSService.PUDS_SI_ECUReset:
                    UDSApi.TPUDSSvcParamER resetType = (UDSApi.TPUDSSvcParamER)serviceInformation.ParametersOfService[0];
                    status = UDSApi.SvcECUReset(mConnectedChannelHandle, ref mRequest, resetType);
                    break;

                default:
                    return false;
            }



            if (status != TPUDSStatus.PUDS_ERROR_OK)
                mResponseStatus = status;

            return status == TPUDSStatus.PUDS_ERROR_OK;
        }

        private void SetPriority(byte prio)
        {
            uint isoTpStatus = 0, priority = prio;

            isoTpStatus = IsoTp_SetValue(mConnectedChannelHandle, 0xEE, ref priority, sizeof(uint));

            if (isoTpStatus != 0)
                System.Windows.Forms.MessageBox.Show("Error on set prio: " + isoTpStatus.ToString());
        }

        public bool WaitForResponse()
        {
            TPUDSStatus status;

            if (!RequestIsFunctional)
            {
                status = UDSApi.WaitForService(mConnectedChannelHandle, out TPUDSMsg response, ref mRequest, out TPUDSMsg transmitConfirmation);

                if (status == TPUDSStatus.PUDS_ERROR_OK)
                    Response = response;
            }
            else
            {
                status = UDSApi.WaitForServiceFunctional(mConnectedChannelHandle, FunctionalResponses, (uint)FunctionalResponses.Length, out uint receiveCount, true,
                                                            ref mRequest, out TPUDSMsg transmitConfirmation);

                if (status == TPUDSStatus.PUDS_ERROR_OK)
                    ReceivedFunctionalResponses = receiveCount;
            }

            if (status != TPUDSStatus.PUDS_ERROR_OK)
                mResponseStatus = status;
            else if (mServiceInformationOfLastRequest.Service == TPUDSService.PUDS_SI_DiagnosticSessionControl)
            {
                UDSApi.TPUDSSvcParamDSC paramDSC = (UDSApi.TPUDSSvcParamDSC)mServiceInformationOfLastRequest.ParametersOfService[0];
                if ((paramDSC != UDSApi.TPUDSSvcParamDSC.PUDS_SVC_PARAM_DSC_DS)
                    && mServiceInformationOfLastRequest.DisableTesterPresent)
                    DeactivateTesterPresent(mServiceInformationOfLastRequest.NetAddressInformation);
            }

            return status == TPUDSStatus.PUDS_ERROR_OK;
        }

        public bool DeactivateTesterPresent(TPUDSNetAddrInfo netAddrInfo)
        {
            //build pointer to session info...
            TPUDSSessionInfo sessionInfo = new TPUDSSessionInfo();
            sessionInfo.NETADDRINFO = netAddrInfo;
            int sessionSize = Marshal.SizeOf(sessionInfo);
            IntPtr sessionPtr = Marshal.AllocHGlobal(sessionSize);
            Marshal.StructureToPtr(sessionInfo, sessionPtr, false);
            //...to retreive current session info...
            TPUDSStatus status = UDSApi.GetValue(mConnectedChannelHandle, TPUDSParameter.PUDS_PARAM_SESSION_INFO, sessionPtr, (uint)sessionSize);
            //...if an error occured, show error message and exit method...
            if (status != TPUDSStatus.PUDS_ERROR_OK)
            {
                Marshal.FreeHGlobal(sessionPtr);
                ShowError(status);
                return false;
            }
            //...else safe current session info in variable...
            sessionInfo = (TPUDSSessionInfo)Marshal.PtrToStructure(sessionPtr, typeof(TPUDSSessionInfo));
            Marshal.FreeHGlobal(sessionPtr);
            //...override session type to prevent sending of tester present...
            sessionInfo.SESSION_TYPE = (byte)UDSApi.TPUDSSvcParamDSC.PUDS_SVC_PARAM_DSC_DS;
            //...get session pointer with changed session type...
            sessionPtr = Marshal.AllocHGlobal(sessionSize);
            Marshal.StructureToPtr(sessionInfo, sessionPtr, false);
            //...set session info to prevent the sending of tester present...
            status |= UDSApi.SetValue(mConnectedChannelHandle, TPUDSParameter.PUDS_PARAM_SESSION_INFO, sessionPtr, (uint)sessionSize);
            //...free space ocupied by the pointer
            Marshal.FreeHGlobal(sessionPtr);


            if (status != TPUDSStatus.PUDS_ERROR_OK)
                ShowError(status);

            return status == TPUDSStatus.PUDS_ERROR_OK;
        }


        public bool SendUUDT(MappingInformation information, int dlc, byte[] data)
        {
            TPUDSMsg uudtMessage = new TPUDSMsg();
            uudtMessage.NETADDRINFO = information.NetAddressInformation;
            uudtMessage.LEN = (ushort)(dlc + 4);
            uudtMessage.DATA = new byte[4095];
            uudtMessage.DATA[0] = (byte)((information.CanId >> 24) & 0xFF);
            uudtMessage.DATA[1] = (byte)((information.CanId >> 16) & 0xFF);
            uudtMessage.DATA[2] = (byte)((information.CanId >> 8) & 0xFF);
            uudtMessage.DATA[3] = (byte)((information.CanId >> 0) & 0xFF);
            Array.Copy(data, 0, uudtMessage.DATA, 4, dlc);
            TPUDSStatus status = UDSApi.Write(mConnectedChannelHandle, ref uudtMessage);

            if (status != TPUDSStatus.PUDS_ERROR_OK)
                ShowError(status);

            return status == TPUDSStatus.PUDS_ERROR_OK;
        }

        public bool ReceiveUUDT(MappingInformation argument)
        {
            mUUDTStatus = UDSApi.Read(mConnectedChannelHandle, out TPUDSMsg message);

            if (mUUDTStatus == TPUDSStatus.PUDS_ERROR_OK)
            {
                UUDTResponse = message;
                return true;
            }

            return false;
        }
        #endregion

        public void ShowResponseError(bool isUUDT = false)
        {
            if (isUUDT)
                ShowError(mUUDTStatus);
            else
                ShowError(mResponseStatus);
        }

        private void ShowError(TPUDSStatus status)
        {
            if (status == TPUDSStatus.PUDS_ERROR_OK)
                return;

            string errorText = "An error occurred (" + Enum.GetName(typeof(TPUDSStatus), status) + "):\n";

            switch (status)
            {
                case TPUDSStatus.PUDS_ERROR_NOT_INITIALIZED:
                    errorText += "The channel is not initialized";
                    break;
                case TPUDSStatus.PUDS_ERROR_ALREADY_INITIALIZED:
                    errorText += "The channel is already initialized";
                    break;
                case TPUDSStatus.PUDS_ERROR_NO_MEMORY:
                    errorText += "Failed to allocate memory";
                    break;
                case TPUDSStatus.PUDS_ERROR_OVERFLOW:
                    errorText += "A buffer overflow occured (too many channels initialized or too many messages in queue).";
                    break;
                case TPUDSStatus.PUDS_ERROR_TIMEOUT:
                    errorText += "Timeout while trying to access the PCAN-UDS API.";
                    break;
                case TPUDSStatus.PUDS_ERROR_NO_MESSAGE:
                    errorText += "No message is available.";
                    break;
                case TPUDSStatus.PUDS_ERROR_WRONG_PARAM:
                    errorText += "The given parameter is invalid";
                    break;
                case TPUDSStatus.PUDS_ERROR_BUSLIGHT:
                    errorText += "An error counter on the CAN bus reached the 'light' limit";
                    break;
                case TPUDSStatus.PUDS_ERROR_BUSHEAVY:
                    errorText += "An error counter on the CAN bus reached the 'heavy' limit";
                    break;
                case TPUDSStatus.PUDS_ERROR_BUSOFF:
                    errorText += "The CAN  controller is in bus-off state";
                    break;
                default:
                    if ((status & TPUDSStatus.PUDS_ERROR_CAN_ERROR) == TPUDSStatus.PUDS_ERROR_CAN_ERROR)
                    {
                        uint pcanBasicError = (uint)(status ^ TPUDSStatus.PUDS_ERROR_CAN_ERROR);
                        errorText += "An error occured in the underlying PCAN-Basic API.\nThe PCAN-Basic error code is: 0x" + pcanBasicError.ToString("X8");
                    }
                    else
                        errorText += "unknown error code.";
                    break;
            }

            System.Windows.Forms.MessageBox.Show(errorText, "Error", System.Windows.Forms.MessageBoxButtons.OK, System.Windows.Forms.MessageBoxIcon.Error);
        }

    }
}