﻿using System;
using System.Linq;
using System.Collections.Generic;
using System.ComponentModel;
using System.IO;
using System.Runtime.InteropServices;
using System.Text;
using System.Windows.Forms;

using Peak.Can.Uds;
using TPUDSCANHandle = System.UInt16;

namespace PcanUdsExample
{


    public partial class FormMain : Form
    {
        const string ByteValuesBufferInputText = "(8 bit values in hex separeted with commas, e.g. 00, 6, F7)";
        const string ShortValuesBufferInputText = "(16 bit values in hex separeted with commas, e.g. 00, 6, F7, F1D8)";

        private readonly ApiUser mApiUser;
        private List<MappingInformation> mMappings;
        private bool mSelectedHardwareIsNonPlugAndPlay;
        private BackgroundWorker mServiceHandler;
        private BackgroundWorker mRecevieUUDTWorker;
        private readonly MappingInformation[] DefaultMappings;
        private List<ResponseInformation> mResponsesReceived;
        private bool mMappingIsUUDT;


        public FormMain()
        {
            mApiUser = new ApiUser();
            mMappings = new List<MappingInformation>();

            mResponsesReceived = new List<ResponseInformation>();

            mServiceHandler = new BackgroundWorker();
            mServiceHandler.DoWork += SendServiceAndWaitForResponse;
            mServiceHandler.RunWorkerCompleted += HandleResponse;
            mServiceHandler.WorkerSupportsCancellation = true;

            mRecevieUUDTWorker = new BackgroundWorker();
            mRecevieUUDTWorker.DoWork += ReceiveUUDT;
            mRecevieUUDTWorker.RunWorkerCompleted += EndReceiveUUDT;
            mRecevieUUDTWorker.WorkerSupportsCancellation = true;

            
            DefaultMappings = new MappingInformation[14];

            InitializeComponent();

            FillConnectionUi();
            FillDefaultMappings();

            foreach (TPUDSParameter parameter in Enum.GetValues(typeof(TPUDSParameter)))
            {
                if (parameter == TPUDSParameter.PUDS_PARAM_SERVER_ADDRESS
                    || parameter == TPUDSParameter.PUDS_PARAM_SESSION_INFO
                    || parameter == TPUDSParameter.PUDS_PARAM_MAPPING_ADD
                    || parameter == TPUDSParameter.PUDS_PARAM_MAPPING_REMOVE)
                    continue;

                comboBoxParameter.Items.Add(new ComboBoxItem(Enum.GetName(typeof(TPUDSParameter), parameter), parameter));
            }
            comboBoxParameter.SelectedIndex = 0;

            comboBoxServiceToSend.Items.Add(new ComboBoxItem("TesterPresent (0x3E)", TPUDSService.PUDS_SI_TesterPresent));
            comboBoxServiceToSend.Items.Add(new ComboBoxItem("DiagnosticSessionControl (0x10)", TPUDSService.PUDS_SI_DiagnosticSessionControl));
            comboBoxServiceToSend.Items.Add(new ComboBoxItem("ReadDataByIdentifier (0x22)", TPUDSService.PUDS_SI_ReadDataByIdentifier));
            comboBoxServiceToSend.Items.Add(new ComboBoxItem("ClearDiagnosticInformation (0x14)", TPUDSService.PUDS_SI_ClearDiagnosticInformation));
            comboBoxServiceToSend.Items.Add(new ComboBoxItem("ReadDTCInformation (0x19)", TPUDSService.PUDS_SI_ReadDTCInformation));
            comboBoxServiceToSend.Items.Add(new ComboBoxItem("ControlDtcSetting (0x85)", TPUDSService.PUDS_SI_ControlDTCSetting));
            comboBoxServiceToSend.Items.Add(new ComboBoxItem("CommunicationControl (0x28)", TPUDSService.PUDS_SI_CommunicationControl));
            comboBoxServiceToSend.Items.Add(new ComboBoxItem("SecurityAccess (0x27)", TPUDSService.PUDS_SI_SecurityAccess));
            comboBoxServiceToSend.Items.Add(new ComboBoxItem("RequestDownload (0x34)", TPUDSService.PUDS_SI_RequestDownload));
            comboBoxServiceToSend.Items.Add(new ComboBoxItem("TransferData (0x36)", TPUDSService.PUDS_SI_TransferData));
            comboBoxServiceToSend.Items.Add(new ComboBoxItem("RequestTransferExit (0x37)", TPUDSService.PUDS_SI_RequestTransferExit));
            comboBoxServiceToSend.Items.Add(new ComboBoxItem("RoutineControl (0x31)", TPUDSService.PUDS_SI_RoutineControl));
            comboBoxServiceToSend.Items.Add(new ComboBoxItem("ECUReset (0x11)", TPUDSService.PUDS_SI_ECUReset));
            comboBoxServiceToSend.SelectedIndex = 0;

            UpdateUi();
        }

        private void FormMain_FormClosing(object sender, FormClosingEventArgs e)
        {
            if (mRecevieUUDTWorker.IsBusy)
                mRecevieUUDTWorker.CancelAsync();

            if (mApiUser.IsConnected)
                mApiUser.Uninitialize();
        }

        private void FillConnectionUi()
        {
            foreach (TPUDSBaudrate baudrate in Enum.GetValues(typeof(TPUDSBaudrate)))
                comboBoxBaudrate.Items.Add(new ComboBoxItem(UdsExampleUtils.GetBitrateString(baudrate), baudrate));
            comboBoxBaudrate.SelectedIndex = 2;

            foreach (TPUDSHWType hwType in Enum.GetValues(typeof(TPUDSHWType)))
                comboBoxHwType.Items.Add(new ComboBoxItem(Enum.GetName(typeof(TPUDSHWType), hwType), hwType));

            comboBoxIoPort.Items.Add(new ComboBoxItem("0100", 0x0100U));
            comboBoxIoPort.Items.Add(new ComboBoxItem("0120", 0x0120U));
            comboBoxIoPort.Items.Add(new ComboBoxItem("0140", 0x0140U));
            comboBoxIoPort.Items.Add(new ComboBoxItem("0200", 0x0200U));
            comboBoxIoPort.Items.Add(new ComboBoxItem("0220", 0x0220U));
            comboBoxIoPort.Items.Add(new ComboBoxItem("0240", 0x0240U));
            comboBoxIoPort.Items.Add(new ComboBoxItem("0260", 0x0260U));
            comboBoxIoPort.Items.Add(new ComboBoxItem("0278", 0x0278U));
            comboBoxIoPort.Items.Add(new ComboBoxItem("0280", 0x0280U));
            comboBoxIoPort.Items.Add(new ComboBoxItem("02A0", 0x02A0U));
            comboBoxIoPort.Items.Add(new ComboBoxItem("02C0", 0x02C0U));
            comboBoxIoPort.Items.Add(new ComboBoxItem("02E0", 0x02E0U));
            comboBoxIoPort.Items.Add(new ComboBoxItem("02E8", 0x02E8U));
            comboBoxIoPort.Items.Add(new ComboBoxItem("02F8", 0x02F8U));
            comboBoxIoPort.Items.Add(new ComboBoxItem("0300", 0x0300U));
            comboBoxIoPort.Items.Add(new ComboBoxItem("0320", 0x0320U));
            comboBoxIoPort.Items.Add(new ComboBoxItem("0340", 0x0340U));
            comboBoxIoPort.Items.Add(new ComboBoxItem("0360", 0x0360U));
            comboBoxIoPort.Items.Add(new ComboBoxItem("0378", 0x0378U));
            comboBoxIoPort.Items.Add(new ComboBoxItem("0380", 0x0380U));
            comboBoxIoPort.Items.Add(new ComboBoxItem("03BC", 0x03BCU));
            comboBoxIoPort.Items.Add(new ComboBoxItem("03E0", 0x03E0U));
            comboBoxIoPort.Items.Add(new ComboBoxItem("03E8", 0x03E8U));
            comboBoxIoPort.Items.Add(new ComboBoxItem("03F8", 0x03F8U));

            comboBoxInterrupt.Items.Add(new ComboBoxItem("3",  (UInt16)3));
            comboBoxInterrupt.Items.Add(new ComboBoxItem("4",  (UInt16)4));
            comboBoxInterrupt.Items.Add(new ComboBoxItem("5",  (UInt16)5));
            comboBoxInterrupt.Items.Add(new ComboBoxItem("7",  (UInt16)7));
            comboBoxInterrupt.Items.Add(new ComboBoxItem("9",  (UInt16)9));
            comboBoxInterrupt.Items.Add(new ComboBoxItem("10", (UInt16)10));
            comboBoxInterrupt.Items.Add(new ComboBoxItem("11", (UInt16)11));
            comboBoxInterrupt.Items.Add(new ComboBoxItem("12", (UInt16)12));
            comboBoxInterrupt.Items.Add(new ComboBoxItem("15", (UInt16)15));

            ButtonHwRefresh_Click(null, EventArgs.Empty);
        }

        private void FillDefaultMappings()
        {
            //this first five entrys can not be removed or added as the fixed CAN IDs do not use mapping, they are only represented as MappingInformation to simplify data handling

            // Physical 29bit CAN id requests with FIXED NORMAL addressing (Source and Target contained in CAN ID)
            DefaultMappings[0] = new MappingInformation(TPUDSAddressingType.PUDS_ADDRESSING_PHYSICAL, TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_29B);

            // Functional 29bit CAN id requests with FIXED NORMAL addressing (Source and Target contained in CAN ID)
            DefaultMappings[1] = new MappingInformation(TPUDSAddressingType.PUDS_ADDRESSING_FUNCTIONAL, TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_29B);

            // Physical 29bit CAN id requests with MIXED addressing (Source and Target contained in CAN ID)
            DefaultMappings[2] = new MappingInformation(TPUDSAddressingType.PUDS_ADDRESSING_PHYSICAL, TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_29B_REMOTE);

            // Functional 29bit CAN id requests with MIXED addressing (Source and Target contained in CAN ID)
            DefaultMappings[3] = new MappingInformation(TPUDSAddressingType.PUDS_ADDRESSING_FUNCTIONAL, TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_29B_REMOTE);

            // 29bit CAN id requests with ENHANCED addressing (Source and Target contained in CAN ID)
            DefaultMappings[4] = new MappingInformation(TPUDSAddressingType.PUDS_ADDRESSING_PHYSICAL, TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_3_29B);


            //Functional request using 11 bits CAN identifier and normal addressing, from External Test Equipment address to OBD functional address
            DefaultMappings[5] = new MappingInformation((uint)TPUDSCanId.PUDS_ISO_15765_4_CAN_ID_FUNCTIONAL_REQUEST, 0, (byte)TPUDSAddress.PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT, (byte)TPUDSAddress.PUDS_ISO_15765_4_ADDR_OBD_FUNCTIONAL, 0,
                                                            TPUDSAddressingType.PUDS_ADDRESSING_FUNCTIONAL, TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_11B);

            #region Physical 11bit CAN id requests between the External Test Equipment and the ECUs (#1-#8)
            //Physical requests and responses using 11 bits CAN identifier and normal addressing, between the External Test Equipment address and standard ECU #1 address
            DefaultMappings[6] = new MappingInformation((uint)TPUDSCanId.PUDS_ISO_15765_4_CAN_ID_PHYSICAL_REQUEST_1, (uint)TPUDSCanId.PUDS_ISO_15765_4_CAN_ID_PHYSICAL_RESPONSE_1,
                                                        (byte)TPUDSAddress.PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT, (byte)TPUDSAddress.PUDS_ISO_15765_4_ADDR_ECU_1, 0,
                                                        TPUDSAddressingType.PUDS_ADDRESSING_PHYSICAL, TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_11B, true);

            //Physical requests and responses using 11 bits CAN identifier and normal addressing, between the External Test Equipment address and standard ECU #2 address
            DefaultMappings[7] = new MappingInformation((uint)TPUDSCanId.PUDS_ISO_15765_4_CAN_ID_PHYSICAL_REQUEST_2, (uint)TPUDSCanId.PUDS_ISO_15765_4_CAN_ID_PHYSICAL_RESPONSE_2,
                                                        (byte)TPUDSAddress.PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT, (byte)TPUDSAddress.PUDS_ISO_15765_4_ADDR_ECU_2, 0,
                                                        TPUDSAddressingType.PUDS_ADDRESSING_PHYSICAL, TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_11B, true);

            //Physical requests and responses using 11 bits CAN identifier and normal addressing, between the External Test Equipment address and standard ECU #3 address
            DefaultMappings[8] = new MappingInformation((uint)TPUDSCanId.PUDS_ISO_15765_4_CAN_ID_PHYSICAL_REQUEST_3, (uint)TPUDSCanId.PUDS_ISO_15765_4_CAN_ID_PHYSICAL_RESPONSE_3,
                                                        (byte)TPUDSAddress.PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT, (byte)TPUDSAddress.PUDS_ISO_15765_4_ADDR_ECU_3, 0,
                                                        TPUDSAddressingType.PUDS_ADDRESSING_PHYSICAL, TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_11B, true);

            //Physical requests and responses using 11 bits CAN identifier and normal addressing, between the External Test Equipment address and standard ECU #4 address
            DefaultMappings[9] = new MappingInformation((uint)TPUDSCanId.PUDS_ISO_15765_4_CAN_ID_PHYSICAL_REQUEST_4, (uint)TPUDSCanId.PUDS_ISO_15765_4_CAN_ID_PHYSICAL_RESPONSE_4,
                                                        (byte)TPUDSAddress.PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT, (byte)TPUDSAddress.PUDS_ISO_15765_4_ADDR_ECU_4, 0,
                                                        TPUDSAddressingType.PUDS_ADDRESSING_PHYSICAL, TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_11B, true);

            //Physical requests and responses using 11 bits CAN identifier and normal addressing, between the External Test Equipment address and standard ECU #5 address
            DefaultMappings[10] = new MappingInformation((uint)TPUDSCanId.PUDS_ISO_15765_4_CAN_ID_PHYSICAL_REQUEST_5, (uint)TPUDSCanId.PUDS_ISO_15765_4_CAN_ID_PHYSICAL_RESPONSE_5,
                                                        (byte)TPUDSAddress.PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT, (byte)TPUDSAddress.PUDS_ISO_15765_4_ADDR_ECU_5, 0,
                                                        TPUDSAddressingType.PUDS_ADDRESSING_PHYSICAL, TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_11B, true);

            //Physical requests and responses using 11 bits CAN identifier and normal addressing, between the External Test Equipment address and standard ECU #6 address
            DefaultMappings[11] = new MappingInformation((uint)TPUDSCanId.PUDS_ISO_15765_4_CAN_ID_PHYSICAL_REQUEST_6, (uint)TPUDSCanId.PUDS_ISO_15765_4_CAN_ID_PHYSICAL_RESPONSE_6,
                                                        (byte)TPUDSAddress.PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT, (byte)TPUDSAddress.PUDS_ISO_15765_4_ADDR_ECU_6, 0,
                                                        TPUDSAddressingType.PUDS_ADDRESSING_PHYSICAL, TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_11B, true);

            //Physical requests and responses using 11 bits CAN identifier and normal addressing, between the External Test Equipment address and standard ECU #7 address
            DefaultMappings[12] = new MappingInformation((uint)TPUDSCanId.PUDS_ISO_15765_4_CAN_ID_PHYSICAL_REQUEST_7, (uint)TPUDSCanId.PUDS_ISO_15765_4_CAN_ID_PHYSICAL_RESPONSE_7,
                                                        (byte)TPUDSAddress.PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT, (byte)TPUDSAddress.PUDS_ISO_15765_4_ADDR_ECU_7, 0,
                                                        TPUDSAddressingType.PUDS_ADDRESSING_PHYSICAL, TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_11B, true);

            //Physical requests and responses using 11 bits CAN identifier and normal addressing, between the External Test Equipment address and standard ECU #8 address
            DefaultMappings[13] = new MappingInformation((uint)TPUDSCanId.PUDS_ISO_15765_4_CAN_ID_PHYSICAL_REQUEST_8, (uint)TPUDSCanId.PUDS_ISO_15765_4_CAN_ID_PHYSICAL_RESPONSE_8,
                                                        (byte)TPUDSAddress.PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT, (byte)TPUDSAddress.PUDS_ISO_15765_4_ADDR_ECU_8, 0,
                                                        TPUDSAddressingType.PUDS_ADDRESSING_PHYSICAL, TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_11B, true);
            #endregion
        }

        private void UpdateUi()
        {
            buttonHwInitialize.Enabled = !mApiUser.IsConnected;
            buttonHwRelease.Enabled = mApiUser.IsConnected;
            buttonHwRefresh.Enabled = !mApiUser.IsConnected;
            buttonHwReset.Enabled = mApiUser.IsConnected; 
            comboBoxChannel.Enabled = !mApiUser.IsConnected;
            comboBoxBaudrate.Enabled = !mApiUser.IsConnected;
            if (mSelectedHardwareIsNonPlugAndPlay)
            {
                comboBoxHwType.Enabled = !mApiUser.IsConnected;
                comboBoxIoPort.Enabled = !mApiUser.IsConnected;
                comboBoxInterrupt.Enabled = !mApiUser.IsConnected;
            }

            buttonInformationStatus.Enabled = mApiUser.IsConnected;

            buttonChannelNodeGet.Enabled = mApiUser.IsConnected;
            buttonChannelNodeSet.Enabled = mApiUser.IsConnected;

            buttonMappingAdd.Enabled = mApiUser.IsConnected;
            buttonMappingSave.Enabled = mApiUser.IsConnected;
            buttonMappingLoad.Enabled = mApiUser.IsConnected;
            buttonMappingFillExample.Enabled = mApiUser.IsConnected;


            groupBoxSend.Enabled = mApiUser.IsConnected;
            listViewResponse.Enabled = mApiUser.IsConnected;
            buttonSend.Enabled = !mApiUser.IsCommunicating;

            ComboBoxParameter_SelectedIndexChanged(null, EventArgs.Empty);
        }


        #region Connection GroupBox
        private void ComboBoxChannel_SelectedIndexChanged(object sender, EventArgs e)
        {
            mSelectedHardwareIsNonPlugAndPlay = (TPUDSCANHandle)(comboBoxChannel.SelectedItem as ComboBoxItem).Data <= UDSApi.PUDS_DNGBUS1;

            comboBoxHwType.Enabled = mSelectedHardwareIsNonPlugAndPlay;
            comboBoxIoPort.Enabled = mSelectedHardwareIsNonPlugAndPlay;
            comboBoxInterrupt.Enabled = mSelectedHardwareIsNonPlugAndPlay;

            if (mSelectedHardwareIsNonPlugAndPlay)
            {
                comboBoxHwType.SelectedIndex = 0;
                comboBoxIoPort.SelectedIndex = 0;
                comboBoxInterrupt.SelectedIndex = 0;
            }
            else
            {
                comboBoxHwType.SelectedItem = null;
                comboBoxIoPort.SelectedItem = null;
                comboBoxInterrupt.SelectedItem = null;
            }
        }

        private void ButtonHwRefresh_Click(object sender, EventArgs e)
        {
            var availableChannels = mApiUser.GetAvailableChannels();
            TPUDSCANHandle selectedHandle;

            if (comboBoxChannel.SelectedItem is ComboBoxItem)
                selectedHandle = (TPUDSCANHandle)(comboBoxChannel.SelectedItem as ComboBoxItem).Data;
            else
                selectedHandle = 0;

            comboBoxChannel.Items.Clear();

            foreach (var channel in availableChannels)
            {
                comboBoxChannel.Items.Add(new ComboBoxItem(UdsExampleUtils.GetHardwareChannelName(channel), channel));

                if (selectedHandle == channel)
                    comboBoxChannel.SelectedIndex = comboBoxChannel.Items.Count - 1;
            }

            if ((selectedHandle == 0) || (comboBoxChannel.SelectedItem == null))
                comboBoxChannel.SelectedIndex = comboBoxChannel.Items.Count - 1;
        }

        private void buttonReset_Click(object sender, EventArgs e)
        {
            mApiUser.ResetHardware();
            mResponsesReceived.Clear();
            listViewResponse.Items.Clear();
        }

        private void ButtonHwInitialize_Click(object sender, EventArgs e)
        {
            TPUDSCANHandle selectedHandle = (TPUDSCANHandle)(comboBoxChannel.SelectedItem as ComboBoxItem).Data;
            TPUDSBaudrate selectedBaudrate = (TPUDSBaudrate)(comboBoxBaudrate.SelectedItem as ComboBoxItem).Data;

            if (mSelectedHardwareIsNonPlugAndPlay)
            {
                TPUDSHWType selectedHwType = (TPUDSHWType)(comboBoxHwType.SelectedItem as ComboBoxItem).Data;
                UInt32 selectedIoPort = (UInt32)(comboBoxIoPort.SelectedItem as ComboBoxItem).Data;
                UInt16 selectedInterrupt = (UInt16)(comboBoxInterrupt.SelectedItem as ComboBoxItem).Data;
                mApiUser.ConnectNonPlugAndPlay(selectedHandle, selectedBaudrate, selectedHwType, selectedIoPort, selectedInterrupt);
            }
            else
                mApiUser.Connect(selectedHandle, selectedBaudrate);

            if (mApiUser.IsConnected)
            {
                numericUpDownLocalNodeAdress.Value = mApiUser.LocalNodeAddress;
                mMappings = new List<MappingInformation>();
                mMappings.AddRange(DefaultMappings);
                UpdateMappingsInUi();
            }

            listViewResponse.Items.Clear();
            mResponsesReceived.Clear();

            UpdateUi();
        }

        private void ButtonHwRelease_Click(object sender, EventArgs e)
        {
            mApiUser.Uninitialize();
            UpdateUi();
        } 
        #endregion
        

        #region ParameterTab

        private void NumericUpDownLocalNodeAdress_ValueChanged(object sender, EventArgs e)
        {
            if (numericUpDownLocalNodeAdress.Value >= 0x100)
                checkBoxEnhancedLocalAddress.Checked = true;
        }

        private void CheckBoxEnhancedLocalAddress_CheckedChanged(object sender, EventArgs e)
        {
            if (checkBoxEnhancedLocalAddress.Checked)
            {
                numericUpDownLocalNodeAdress.Maximum = 0x07FF;
            }
            else
            {
                if (numericUpDownLocalNodeAdress.Value > 0xFF)
                    numericUpDownLocalNodeAdress.Value = 0xFF;

                numericUpDownLocalNodeAdress.Maximum = 0x100;
            }
        }

        private void ButtonChannelNodeSet_Click(object sender, EventArgs e)
        {
            if (mApiUser.SetLocalAddress((uint)numericUpDownLocalNodeAdress.Value, checkBoxEnhancedLocalAddress.Checked))
            {
                numericUpDownLocalNodeAdress.Value = mApiUser.LocalNodeAddress;
                textBoxSourceAddress.Text = mApiUser.LocalNodeAddress.ToString("X2");
                AddToListBox("Local address set");

                if (mApiUser.IsConnected)
                    EnableSendGroupbox();
            }
            else
                AddToListBox("Failed to set local address");
        }

        private void ButtonChannelNodeGet_Click(object sender, EventArgs e)
        {
            if (mApiUser.GetLocalAddress(out uint localNode))
            {
                if (mApiUser.LocalAddressIsExtended)
                    checkBoxEnhancedLocalAddress.Checked = true;
                numericUpDownLocalNodeAdress.Value = localNode;
                AddToListBox("Fetched local address");
            }
            else
                AddToListBox("Failed to fetched local address");
        }


        private void ComboBoxParameter_SelectedIndexChanged(object sender, EventArgs e)
        {
            ComboBoxItem item = comboBoxParameter.SelectedItem as ComboBoxItem;

            if (item == null)
            {
                buttonSetParameter.Enabled = false;
                buttonGetParameter.Enabled = false;
                numericUpDownParamValue.Enabled = false;
                radioButtonParameterActive.Enabled = false;
                radioButtonParameterInactive.Enabled = false;
            }

            TPUDSParameter parameter = (TPUDSParameter)item.Data;

            if (mApiUser.IsConnected)
            {
                buttonGetParameter.Enabled = true;

                if (parameter != TPUDSParameter.PUDS_PARAM_SESSION_INFO &&
                     parameter != TPUDSParameter.PUDS_PARAM_API_VERSION &&
                     parameter != TPUDSParameter.PUDS_PARAM_RECEIVE_EVENT &&
                     parameter != TPUDSParameter.PUDS_PARAM_CHANNEL_CONDITION)
                {
                    buttonSetParameter.Enabled = true;
                    if (parameter == TPUDSParameter.PUDS_PARAM_SERVER_FILTER)
                    {
                        labelActivation.Text = "Operation";
                        radioButtonParameterActive.Text = "Add";
                        radioButtonParameterInactive.Text = "Remove";
                    }
                    else
                    {
                        labelActivation.Text = "Activation";
                        radioButtonParameterActive.Text = "Active";
                        radioButtonParameterInactive.Text = "Inactive";
                    }


                    radioButtonParameterActive.Enabled = (parameter == TPUDSParameter.PUDS_PARAM_SERVER_FILTER
                                                            || parameter == TPUDSParameter.PUDS_PARAM_DEBUG
                                                            || parameter == TPUDSParameter.PUDS_PARAM_CAN_DATA_PADDING);
                    radioButtonParameterInactive.Enabled = (parameter == TPUDSParameter.PUDS_PARAM_SERVER_FILTER
                                                            || parameter == TPUDSParameter.PUDS_PARAM_DEBUG
                                                            || parameter == TPUDSParameter.PUDS_PARAM_CAN_DATA_PADDING);
                    numericUpDownParamValue.Enabled = (parameter != TPUDSParameter.PUDS_PARAM_DEBUG && parameter != TPUDSParameter.PUDS_PARAM_CAN_DATA_PADDING);

                    AdjustParameterValueRange(parameter);
                }
                else
                {
                    buttonSetParameter.Enabled = false;
                    radioButtonParameterActive.Enabled = false;
                    radioButtonParameterInactive.Enabled = false;
                    numericUpDownParamValue.Enabled = false;
                }

            }
            else
            {
                buttonSetParameter.Enabled = parameter == TPUDSParameter.PUDS_PARAM_WFT_MAX;
                buttonGetParameter.Enabled = (parameter == TPUDSParameter.PUDS_PARAM_API_VERSION ||
                                                parameter == TPUDSParameter.PUDS_PARAM_CHANNEL_CONDITION ||
                                                parameter == TPUDSParameter.PUDS_PARAM_WFT_MAX);

                numericUpDownParamValue.Enabled = parameter == TPUDSParameter.PUDS_PARAM_WFT_MAX;

                if (parameter == TPUDSParameter.PUDS_PARAM_WFT_MAX)
                    AdjustParameterValueRange(TPUDSParameter.PUDS_PARAM_WFT_MAX);
            }
        }
        private void AdjustParameterValueRange(TPUDSParameter parameter)
        {
            switch (parameter)
            {
                case TPUDSParameter.PUDS_PARAM_TIMEOUT_REQUEST:
                case TPUDSParameter.PUDS_PARAM_TIMEOUT_RESPONSE:
                case TPUDSParameter.PUDS_PARAM_WFT_MAX:
                    numericUpDownParamValue.Maximum = 0xFFFFFFFF;
                    break;
                case TPUDSParameter.PUDS_PARAM_BLOCK_SIZE:
                case TPUDSParameter.PUDS_PARAM_PADDING_VALUE:
                case TPUDSParameter.PUDS_PARAM_SERVER_FILTER:
                    numericUpDownParamValue.Maximum = 0xFF;
                    break;
                case TPUDSParameter.PUDS_PARAM_SEPERATION_TIME:
                    numericUpDownParamValue.Maximum = 0x7F;
                    break;
            }
        }

        private uint GetParameterValueFromUi(TPUDSParameter parameter)
        {
            switch (parameter)
            {
                case TPUDSParameter.PUDS_PARAM_SERVER_FILTER:
                case TPUDSParameter.PUDS_PARAM_TIMEOUT_REQUEST:
                case TPUDSParameter.PUDS_PARAM_TIMEOUT_RESPONSE:
                case TPUDSParameter.PUDS_PARAM_BLOCK_SIZE:
                case TPUDSParameter.PUDS_PARAM_SEPERATION_TIME:
                case TPUDSParameter.PUDS_PARAM_WFT_MAX:
                case TPUDSParameter.PUDS_PARAM_PADDING_VALUE:
                    return (uint)numericUpDownParamValue.Value;

                case TPUDSParameter.PUDS_PARAM_DEBUG:
                case TPUDSParameter.PUDS_PARAM_CAN_DATA_PADDING:
                    if (radioButtonParameterActive.Checked)
                        return 1;
                    else
                        return 0;

                default:
                    return 0;
            }
        }
        private void ButtonSetParameter_Click(object sender, EventArgs e)
        {
            TPUDSParameter parameter = (TPUDSParameter)(comboBoxParameter.SelectedItem as ComboBoxItem).Data;
            uint value = GetParameterValueFromUi(parameter);

            if (parameter == TPUDSParameter.PUDS_PARAM_SERVER_FILTER)
            {
                if (radioButtonParameterActive.Checked)
                    mApiUser.AddAddressToFilter(value);
                else
                    mApiUser.RemoveAddressFromFilter(value);
            }

            if (mApiUser.SetParameter(parameter, value))
                AddToListBox(BuildParameterSetLine(parameter, value));
        }


        private void ButtonGetParameter_Click(object sender, EventArgs e)
        {
            TPUDSParameter parameter = (TPUDSParameter)(comboBoxParameter.SelectedItem as ComboBoxItem).Data;

            if (IsUIntParameter(parameter))
            {
                if (mApiUser.GetUIntParameter(parameter, out uint value))
                    AddToListBox(BuildUIntParameterLine(parameter, value));
            }
            else
            {
                if (mApiUser.GetStringParameter(parameter, out string value))
                    AddToListBox(BuildStringParameterLine(parameter, value));
            }
        }

        private bool IsUIntParameter(TPUDSParameter parameter)
        {
            return parameter != TPUDSParameter.PUDS_PARAM_API_VERSION;
        }



        private void ButtonInformationVersion_Click(object sender, EventArgs e)
        {
            if (mApiUser.GetStringParameter(TPUDSParameter.PUDS_PARAM_API_VERSION, out string version))
                AddToListBox("Api-Version: " + version);
        }

        private void ButtonInformationClear_Click(object sender, EventArgs e)
        {
            listBoxParameterInformation.Items.Clear();
        }

        private void ButtonInformationStatus_Click(object sender, EventArgs e)
        {
            if (mApiUser.GetHardwareStatus(out string status))
                AddToListBox(status);
            else
                AddToListBox("Could not fetch status of device");
        }
               
        private string BuildParameterSetLine(TPUDSParameter parameter, uint value)
        {
            if (parameter == TPUDSParameter.PUDS_PARAM_PADDING_VALUE)
                return Enum.GetName(typeof(TPUDSParameter), parameter) + " set to " + value.ToString("X") + "h";
            else
                return Enum.GetName(typeof(TPUDSParameter), parameter) + " set to " + value;
        }
        private string BuildStringParameterLine(TPUDSParameter parameter, string value)
        {
            if (parameter == TPUDSParameter.PUDS_PARAM_API_VERSION)
                return "Api Version: " + value;


            return "unknown parameter";
        }
        private string BuildUIntParameterLine(TPUDSParameter parameter, uint value)
        {
            string parameterLine;

            switch (parameter)
            {
                case TPUDSParameter.PUDS_PARAM_SERVER_FILTER:
                    parameterLine = "Server filter: " + value.ToString("X");
                    break;
                case TPUDSParameter.PUDS_PARAM_TIMEOUT_REQUEST:
                    parameterLine = "Request timeout: " + value;
                    break;
                case TPUDSParameter.PUDS_PARAM_TIMEOUT_RESPONSE:
                    parameterLine = "Response timeout: " + value;
                    break;
                case TPUDSParameter.PUDS_PARAM_RECEIVE_EVENT:
                    if (value != 0)
                        parameterLine = String.Format("Handle of receive event: 0x{0:X8}" + value);
                    else
                        parameterLine = "no receive event set";
                    break;
                case TPUDSParameter.PUDS_PARAM_BLOCK_SIZE:
                    if (value == 0)
                        parameterLine = "block size is unlimited";
                    else
                        parameterLine = "block size is limited to " + value + " consecutive frames";
                    break;
                case TPUDSParameter.PUDS_PARAM_SEPERATION_TIME:
                    if (value == 0)
                        parameterLine = "No seperation time is given for the consecutive frames.";
                    else
                        parameterLine = "A minimum of " + value + "ms is waited between two consecutive frames.";
                    break;
                case TPUDSParameter.PUDS_PARAM_DEBUG:
                    if (value == UDSApi.PUDS_DEBUG_NONE)
                        parameterLine = "Debugging is disabled";
                    else if (value == UDSApi.PUDS_DEBUG_CAN)
                        parameterLine = "Debugging is enabled";
                    else
                        parameterLine = "No debugging value returned";
                    break;
                case TPUDSParameter.PUDS_PARAM_CHANNEL_CONDITION:
                    parameterLine = BuildChannelConditionLine(value);
                    break;
                case TPUDSParameter.PUDS_PARAM_WFT_MAX:
                    parameterLine = "Maximum number of Wait FlowControl: " + value;
                    break;
                case TPUDSParameter.PUDS_PARAM_CAN_DATA_PADDING:
                    if (value == UDSApi.PUDS_CAN_DATA_PADDING_NONE)
                        parameterLine = "Data padding is disabled";
                    else if (value == UDSApi.PUDS_CAN_DATA_PADDING_ON)
                        parameterLine = "Data padding is enabled";
                    else
                        parameterLine = "No padding value returned";
                    break;
                case TPUDSParameter.PUDS_PARAM_PADDING_VALUE:
                    parameterLine = String.Format("Padded bytes are filled with value: {0,2:X}h", value);
                    break;
                default:
                    parameterLine = "unknown parameter";
                    break;
            }

            return parameterLine;
        }

        private string BuildChannelConditionLine(uint value)
        {
            if (value == UDSApi.PUDS_CHANNEL_AVAILABLE)
                return "Channel condition is: PUDS_CHANNEL_AVAILABLE";
            else if (value == UDSApi.PUDS_CHANNEL_OCCUPIED)
                return "Channel condition is: PUDS_CHANNEL_OCCUPIED";
            else if (value == UDSApi.PUDS_CHANNEL_UNAVAILABLE)
                return "Channel condition is: PUDS_CHANNEL_UNAVAILABLE";
            return "no channel condition value";
        }

        private void AddToListBox(string line)
        {
            listBoxParameterInformation.Items.Add(line);
            listBoxParameterInformation.SelectedIndex = listBoxParameterInformation.Items.Count - 1;
        }

        #endregion


        #region MappingTab
        private void UpdateMappingsInUi()
        {
            MappingInformation comboboxSelected = null;
            MappingInformation listviewSelected = null;

            if (comboBoxMappings.SelectedItem != null)
                comboboxSelected = (MappingInformation)(comboBoxMappings.SelectedItem as ComboBoxItem).Data;

            if (listViewMappings.SelectedItems.Count > 0)
                listviewSelected = (MappingInformation)listViewMappings.SelectedItems[0].Tag;

            comboBoxMappings.Items.Clear();
            listViewMappings.Items.Clear();

            foreach (MappingInformation mapping in mMappings)
            {
                ListViewItem item = new ListViewItem();
                item.Tag = mapping;
                item.Text = mapping.CanIdString;
                item.SubItems.Add(mapping.ResponseCanIdString);
                item.SubItems.Add(mapping.TargetTypeString);
                item.SubItems.Add(mapping.ProtocolString);
                item.SubItems.Add(mapping.SourceAddressString);
                item.SubItems.Add(mapping.TargetAddressString);
                item.SubItems.Add(mapping.RemoteAddressString);
                item.SubItems.Add(mapping.IsTwoWayString);
                listViewMappings.Items.Add(item);

                if (mapping.Equals(listviewSelected))
                    item.Selected = true;

                comboBoxMappings.Items.Add(new ComboBoxItem(mapping.Name, mapping));
                if (mapping.IsTwoWay)
                {
                    MappingInformation backwardsMapping = mapping.Backwards();
                    comboBoxMappings.Items.Add(new ComboBoxItem(backwardsMapping.Name, backwardsMapping));
                }
                if (mapping.Equals(comboboxSelected))
                    comboBoxMappings.SelectedIndex = comboBoxMappings.Items.Count - 1;

            }

            if (comboBoxMappings.SelectedItem == null)
                comboBoxMappings.SelectedIndex = 0;
        }

        private void ListViewMappings_SelectedIndexChanged(object sender, EventArgs e)
        {
            bool enableRemove = false;

            if (listViewMappings.SelectedIndices.Count != 0)
            {
                MappingInformation info = (MappingInformation)listViewMappings.SelectedItems[0].Tag;
                if (!info.IsFixedCanIdprotocol)
                    enableRemove = true;
            }

            buttonMappingRemove.Enabled = enableRemove;
        }

        private void ButtonMappingAdd_Click(object sender, EventArgs e)
        {
            FormMapping mapping = new FormMapping((uint)numericUpDownLocalNodeAdress.Value);

            if (mapping.ShowDialog() == DialogResult.OK)
            {
                if (mApiUser.AddMapping(mapping.MappingInformation))
                {
                    bool foundMapping = false;

                    for (int i = 0; i < mMappings.Count; i++)
                    {
                        if (mMappings[i].Backwards().Equals(mapping.MappingInformation))
                        {
                            mMappings[i].IsTwoWay = true;
                            foundMapping = true;
                        }
                    }

                    if (!foundMapping)
                        mMappings.Add(mapping.MappingInformation);

                    UpdateMappingsInUi();
                }
            }
        }
        private void ButtonMappingRemove_Click(object sender, EventArgs e)
        {
            MappingInformation mapping = (MappingInformation)listViewMappings.SelectedItems[0].Tag;

            if (mApiUser.RemoveMapping(mapping))
            {
                if (mapping.IsTwoWay)
                    if (!mApiUser.RemoveMapping(mapping.Backwards()))
                        mMappings.Add(mapping.Backwards());

                mMappings.Remove(mapping);
                UpdateMappingsInUi();

                ListViewMappings_SelectedIndexChanged(null, EventArgs.Empty);
            }
        }
        private void buttonMappingSave_Click(object sender, EventArgs e)
        {
            if (mMappings.Count == 0)
            {
                MessageBox.Show("Nothing to save!");
                return;
            }

            List<string> lines = new List<string>();

            lines.Add("CanId;ResponseCanId;TargetType;Protocol;SourceAddress;TargetAddress;RemoteAddress;TwoWay");

            foreach (MappingInformation information in mMappings)
            {
                if (UdsExampleUtils.ProtocolIsDetachedFromIdsAndAddresses(information.Protocol))
                    continue;

                StringBuilder line = new StringBuilder();
                line.Append(information.CanIdString);
                line.Append(";");
                line.Append(information.ResponseCanIdString);
                line.Append(";");
                line.Append(information.TargetTypeString);
                line.Append(";");
                line.Append(information.ProtocolString);
                line.Append(";");
                line.Append(information.SourceAddressString);
                line.Append(";");
                line.Append(information.TargetAddressString);
                line.Append(";");
                line.Append(information.RemoteAddressString);
                line.Append(";");
                line.Append(information.IsTwoWayString);
                lines.Add(line.ToString());
            }

            File.WriteAllLines("mapping.csv", lines);
        }
        private void buttonMappingLoad_Click(object sender, EventArgs e)
        {
            string[] lines;

            try
            {
                lines = File.ReadAllLines("mapping.csv");
            }
            catch (FileNotFoundException)
            {
                lines = null;
            }

            if (lines == null)
            {
                MessageBox.Show("No saved mappings found (file: mapping.csv).", "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }

            bool atLeastOneFaultyLine = false;
            List<MappingInformation> newMappings = new List<MappingInformation>();

            for (int i = 0; i < 5; i++)
            {
                newMappings.Add(DefaultMappings[i]);
            }

            for (int i = 1; i < lines.Length; i++)
            {
                string[] partsOfMapping = lines[i].Split(';').Select((string x) => { x = x.Trim('h'); return x; }).ToArray();

                if (partsOfMapping.Length != 8)
                {
                    atLeastOneFaultyLine = true;
                    break;
                }

                TPUDSProtocol protocol;
                TPUDSAddressingType targetType;

                if (!Enum.TryParse(partsOfMapping[3], out protocol) || !Enum.TryParse(partsOfMapping[2], out targetType))
                {
                    atLeastOneFaultyLine = true;
                    break;
                }

                if (UdsExampleUtils.ProtocolIsDetachedFromIdsAndAddresses(protocol))
                {
                    newMappings.Add(new MappingInformation(targetType, protocol));
                    break;
                }

                uint canId, responseCanId;
                byte sourceAddress, targetAddress, remoteAddress;
                bool isTwoWay;

                if (UInt32.TryParse(partsOfMapping[0], System.Globalization.NumberStyles.HexNumber, null, out canId)
                    && Byte.TryParse(partsOfMapping[4], System.Globalization.NumberStyles.HexNumber, null, out sourceAddress)
                    && Byte.TryParse(partsOfMapping[5], System.Globalization.NumberStyles.HexNumber, null, out targetAddress)
                    && Byte.TryParse(partsOfMapping[6], System.Globalization.NumberStyles.HexNumber, null, out remoteAddress)
                    )
                {
                    if ((targetType == TPUDSAddressingType.PUDS_ADDRESSING_PHYSICAL) && (protocol != TPUDSProtocol.PUDS_PROTOCOL_NONE))
                    {
                        if (UInt32.TryParse(partsOfMapping[1], System.Globalization.NumberStyles.HexNumber, null, out responseCanId)
                            && Boolean.TryParse(partsOfMapping[7], out isTwoWay))
                        {
                            newMappings.Add(new MappingInformation(canId, responseCanId, sourceAddress, targetAddress, remoteAddress, targetType, protocol, isTwoWay));
                        }
                        else
                        {
                            atLeastOneFaultyLine = true;
                            break;
                        }
                    }
                    else 
                      newMappings.Add(new MappingInformation(canId, 0, sourceAddress, targetAddress, remoteAddress, targetType, protocol));                    
                }
                else
                {
                    atLeastOneFaultyLine = true;
                    break;
                }
            }

            if (atLeastOneFaultyLine)
            {
                MessageBox.Show("The file \"mapping.csv\" contained faulty at least one faulty entry.\nAborted the loading.", "Warning", MessageBoxButtons.OK, MessageBoxIcon.Warning);
                return;
            }

            bool atLeastOneFailed = false;

            foreach (MappingInformation info in mMappings)
            {
                if (info.IsFixedCanIdprotocol)
                    continue;

                if (mApiUser.RemoveMapping(info))
                {
                    if (info.IsTwoWay)
                    {
                        if (!mApiUser.RemoveMapping(info.Backwards()))
                            atLeastOneFailed = true;
                    }
                }
                else
                    atLeastOneFailed = true;
            }

            if (atLeastOneFailed)
                MessageBox.Show("At least one mapping could not be removed.\n This may lead to unwanted aftereffects, if this is the case reset the mappings by releasing and intializing the hardware.", "Warning", MessageBoxButtons.OK, MessageBoxIcon.Warning);

            mMappings.Clear();
            mMappings.AddRange(newMappings);

            atLeastOneFailed = false;

            foreach (MappingInformation info in mMappings)
            {
                if (info.IsFixedCanIdprotocol)
                    continue;

                if (mApiUser.AddMapping(info))
                {
                    if (info.IsTwoWay)
                        if (!mApiUser.AddMapping(info.Backwards()))
                            atLeastOneFailed = true;
                }
                else
                    atLeastOneFailed = true;
            }

            if (atLeastOneFailed)
                MessageBox.Show("At least one mapping could not be added.\n This may lead to unwanted aftereffects, if this is the case reset the mappings by releasing and intializing the hardware.", "Warning", MessageBoxButtons.OK, MessageBoxIcon.Warning);

            UpdateMappingsInUi();
        }
        
        private void buttonMappingFillExample_Click(object sender, EventArgs e)
        {
            MappingInformation info = new MappingInformation(0x22, 0x11, 0xF1, 0x22, 0, TPUDSAddressingType.PUDS_ADDRESSING_PHYSICAL, TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_11B, true);

            if (!mMappings.Contains(info))
            {
                mApiUser.AddMapping(info);
                if (!mMappings.Contains(info.Backwards()))
                {
                    mApiUser.AddMapping(info.Backwards());
                    mMappings.Add(info);
                }
                else
                {
                    int index = mMappings.IndexOf(info.Backwards());
                    mMappings[index].IsTwoWay = true;
                }
            }
            else
            {
                int index = mMappings.IndexOf(info);
                
                if (!mMappings[index].IsTwoWay)
                {
                    mApiUser.AddMapping(info.Backwards());
                    mMappings[index].IsTwoWay = true;
                }
            }

            info = new MappingInformation(0x2F, 0, 0xF1, 0x33, 0, TPUDSAddressingType.PUDS_ADDRESSING_FUNCTIONAL, TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_11B);

            if (!mMappings.Contains(info))
            {
                mApiUser.AddMapping(info);
                mMappings.Add(info);
            }

            UpdateMappingsInUi();
        }
        #endregion


        #region ServiceTab

        private void ComboBoxServiceToSend_SelectedIndexChanged(object sender, EventArgs e)
        {
            ShowAndFillServiceInputs();
        }

        private void ShowAndFillServiceInputs()
        {
            TPUDSService service = (TPUDSService)(comboBoxServiceToSend.SelectedItem as ComboBoxItem).Data;

            FillEnumInput(service);

            bool numberInputOne = false, numberInputTwo = false, bufferInputOne = false, bufferInputTwo = false;
            uint numberInputOneMax = 0, numberInputTwoMax = 0;

            labelReadDTCInformation.Visible = false;

            switch (service)
            {
                case TPUDSService.PUDS_SI_SecurityAccess:
                    labelFirstServiceBuffer.Text = "Buffer";
                    bufferInputOne = true;
                    break;
                case TPUDSService.PUDS_SI_CommunicationControl:
                    labelSecondServiceNumber.Text = "Communication Type";
                    numberInputTwo = true;
                    numberInputTwoMax = 0xFF;
                    break;
                case TPUDSService.PUDS_SI_ControlDTCSetting:
                    labelFirstServiceBuffer.Text = "Buffer";
                    bufferInputOne = true;
                    break;
                case TPUDSService.PUDS_SI_ReadDataByIdentifier:
                    labelFirstServiceBuffer.Text = "Buffer";
                    bufferInputOne = true;
                    break;
                case TPUDSService.PUDS_SI_ClearDiagnosticInformation:
                    labelFirstServiceNumber.Text = "Group of DTCs";
                    numberInputOne = true;
                    numberInputOneMax = 0xFFFFFFFF;
                    break;
                case TPUDSService.PUDS_SI_ReadDTCInformation:
                    labelFirstServiceNumber.Text = "DTC status mask";
                    labelReadDTCInformation.Visible = true;
                    numberInputOne = true;
                    numberInputOneMax = 0xFF;
                    break;
                case TPUDSService.PUDS_SI_RoutineControl:
                    labelSecondServiceNumber.Text = "Routine Identifier";
                    labelFirstServiceBuffer.Text = "Options";
                    bufferInputOne = true;
                    numberInputTwo = true;
                    numberInputTwoMax = 0xFFFF;
                    break;
                case TPUDSService.PUDS_SI_RequestDownload:
                    labelFirstServiceNumber.Text = "Compression method";
                    labelSecondServiceNumber.Text = "Encryption method";
                    labelFirstServiceBuffer.Text = "Memory address";
                    labelSecondServiceBuffer.Text = "Memory size";
                    numberInputOne = true;
                    numberInputOneMax = 0xFF;
                    numberInputTwo = true;
                    numberInputTwoMax = 0xFF;
                    bufferInputOne = true;
                    bufferInputTwo = true;
                    break;
                case TPUDSService.PUDS_SI_TransferData:
                    labelFirstServiceBuffer.Text = "Buffer";
                    labelFirstServiceNumber.Text = "Block sequence counter";
                    numberInputOne = true;
                    numberInputOneMax = 0xFF;
                    bufferInputOne = true;
                    break;
                case TPUDSService.PUDS_SI_RequestTransferExit:
                    labelFirstServiceBuffer.Text = "Buffer";
                    bufferInputOne = true;
                    break;
            }

            if (service == TPUDSService.PUDS_SI_ReadDataByIdentifier)
                labelFirstServiceBufferHint.Text = ShortValuesBufferInputText;
            else
                labelFirstServiceBufferHint.Text = ByteValuesBufferInputText;

            labelFirstServiceNumber.Visible = numberInputOne;
            numericUpDownFirstServiceNumber.Visible = numberInputOne;
            numericUpDownFirstServiceNumber.Maximum = numberInputOneMax;
            labelSecondServiceNumber.Visible = numberInputTwo;
            numericUpDownSecondServiceNumber.Visible = numberInputTwo;
            numericUpDownSecondServiceNumber.Maximum = numberInputTwoMax;
            labelFirstServiceBuffer.Visible = bufferInputOne;
            labelFirstServiceBufferHint.Visible = bufferInputOne;
            textBoxFirstServiceBuffer.Visible = bufferInputOne;
            labelSecondServiceBuffer.Visible = bufferInputTwo;
            labelSecondServiceBufferHint.Visible = bufferInputTwo;
            textBoxSecondServiceBuffer.Visible = bufferInputTwo;

            checkBoxSendTesterPresent.Visible = (service == TPUDSService.PUDS_SI_DiagnosticSessionControl);
            checkBoxSendTesterPresent.Checked = true;
            checkBoxSendTesterPresent.Enabled = false;
        }
        private void FillEnumInput(TPUDSService service)
        {
            if (service == TPUDSService.PUDS_SI_TesterPresent
                || service == TPUDSService.PUDS_SI_ReadDataByIdentifier
                || service == TPUDSService.PUDS_SI_ClearDiagnosticInformation
                || service == TPUDSService.PUDS_SI_RequestDownload
                || service == TPUDSService.PUDS_SI_TransferData
                || service == TPUDSService.PUDS_SI_RequestTransferExit)
            {
                labelFirstServiceEnum.Visible = false;
                comboBoxFirstServiceEnum.Visible = false;
                comboBoxSecondServiceEnum.Visible = false;
                labelSecondServiceEnum.Visible = false;
                return;
            }

            labelFirstServiceEnum.Visible = true;
            comboBoxFirstServiceEnum.Visible = true;
            comboBoxFirstServiceEnum.Items.Clear();

            if (service == TPUDSService.PUDS_SI_SecurityAccess)
            {
                labelFirstServiceEnum.Text = "Security access type";
                comboBoxFirstServiceEnum.Items.Add(new ComboBoxItem("PUDS_SVC_PARAM_SA_RSD_1", UDSApi.PUDS_SVC_PARAM_SA_RSD_1));
                comboBoxFirstServiceEnum.Items.Add(new ComboBoxItem("PUDS_SVC_PARAM_SA_RSD_3", UDSApi.PUDS_SVC_PARAM_SA_RSD_3));
                comboBoxFirstServiceEnum.Items.Add(new ComboBoxItem("PUDS_SVC_PARAM_SA_RSD_5", UDSApi.PUDS_SVC_PARAM_SA_RSD_5));
                comboBoxFirstServiceEnum.Items.Add(new ComboBoxItem("PUDS_SVC_PARAM_SA_RSD_MIN", UDSApi.PUDS_SVC_PARAM_SA_RSD_MIN));
                comboBoxFirstServiceEnum.Items.Add(new ComboBoxItem("PUDS_SVC_PARAM_SA_RSD_MAX", UDSApi.PUDS_SVC_PARAM_SA_RSD_MAX));
                comboBoxFirstServiceEnum.Items.Add(new ComboBoxItem("PUDS_SVC_PARAM_SA_SK_2", UDSApi.PUDS_SVC_PARAM_SA_SK_2));
                comboBoxFirstServiceEnum.Items.Add(new ComboBoxItem("PUDS_SVC_PARAM_SA_SK_4", UDSApi.PUDS_SVC_PARAM_SA_SK_4));
                comboBoxFirstServiceEnum.Items.Add(new ComboBoxItem("PUDS_SVC_PARAM_SA_SK_6", UDSApi.PUDS_SVC_PARAM_SA_SK_6));
                comboBoxFirstServiceEnum.Items.Add(new ComboBoxItem("PUDS_SVC_PARAM_SA_SK_MIN", UDSApi.PUDS_SVC_PARAM_SA_SK_MIN));
                comboBoxFirstServiceEnum.Items.Add(new ComboBoxItem("PUDS_SVC_PARAM_SA_SK_MAX", UDSApi.PUDS_SVC_PARAM_SA_SK_MAX));
            }
            else if (service == TPUDSService.PUDS_SI_ReadDTCInformation)
            {
                labelFirstServiceEnum.Text = "DTC information type";
                comboBoxFirstServiceEnum.Items.Add(new ComboBoxItem("PUDS_SVC_PARAM_RDTCI_RNODTCBSM", UDSApi.TPUDSSvcParamRDTCI.PUDS_SVC_PARAM_RDTCI_RNODTCBSM));
                comboBoxFirstServiceEnum.Items.Add(new ComboBoxItem("PUDS_SVC_PARAM_RDTCI_RDTCBSM", UDSApi.TPUDSSvcParamRDTCI.PUDS_SVC_PARAM_RDTCI_RDTCBSM));
                comboBoxFirstServiceEnum.Items.Add(new ComboBoxItem("PUDS_SVC_PARAM_RDTCI_RMMDTCBSM", UDSApi.TPUDSSvcParamRDTCI.PUDS_SVC_PARAM_RDTCI_RMMDTCBSM));
                comboBoxFirstServiceEnum.Items.Add(new ComboBoxItem("PUDS_SVC_PARAM_RDTCI_RNOMMDTCBSM", UDSApi.TPUDSSvcParamRDTCI.PUDS_SVC_PARAM_RDTCI_RNOMMDTCBSM));
                comboBoxFirstServiceEnum.Items.Add(new ComboBoxItem("PUDS_SVC_PARAM_RDTCI_RNOOBDDTCBSM", UDSApi.TPUDSSvcParamRDTCI.PUDS_SVC_PARAM_RDTCI_RNOOBDDTCBSM));
                comboBoxFirstServiceEnum.Items.Add(new ComboBoxItem("PUDS_SVC_PARAM_RDTCI_ROBDDTCBSM", UDSApi.TPUDSSvcParamRDTCI.PUDS_SVC_PARAM_RDTCI_ROBDDTCBSM));
            }
            else
            {
                Type enumType;
                switch (service)
                {
                    case TPUDSService.PUDS_SI_DiagnosticSessionControl:
                        labelFirstServiceEnum.Text = "Session type";
                        enumType = typeof(UDSApi.TPUDSSvcParamDSC);
                        break;
                    case TPUDSService.PUDS_SI_ControlDTCSetting:
                        labelFirstServiceEnum.Text = "DTC setting type";
                        enumType = typeof(UDSApi.TPUDSSvcParamCDTCS);
                        break;
                    case TPUDSService.PUDS_SI_CommunicationControl:
                        labelFirstServiceEnum.Text = "Control type";
                        enumType = typeof(UDSApi.TPUDSSvcParamCC);
                        break;
                    case TPUDSService.PUDS_SI_RoutineControl:
                        labelFirstServiceEnum.Text = "Routine control type";
                        enumType = typeof(UDSApi.TPUDSSvcParamRC);

                        break;
                    case TPUDSService.PUDS_SI_ECUReset:
                        labelFirstServiceEnum.Text = "Reset type";
                        enumType = typeof(UDSApi.TPUDSSvcParamER);
                        break;
                    default:
                        return;
                }

                foreach (var value in Enum.GetValues(enumType))
                {
                    //if (service == TPUDSService.PUDS_SI_ReadDTCInformation)
                    //{
                    //    UDSApi.TPUDSSvcParamRDTCI enumVal = (UDSApi.TPUDSSvcParamRDTCI)value;
                    //    if (enumVal == UDSApi.TPUDSSvcParamRDTCI.PUDS_SVC_PARAM_RDTCI_RDTCSSBDTC)
                    //        || (enumVal == UDSApi.TPUDSSvcParamRDTCI.PUDS_SVC_PARAM_RDTCI_RDTCSSBRN)
                    //        || (enumVal == UDSApi.TPUDSSvcParamRDTCI.PUDS_SVC_PARAM_RDTCI_RDTCEDRBDN)
                    //        || (enumVal == UDSApi.TPUDSSvcParamRDTCI.)
                    //}
                    comboBoxFirstServiceEnum.Items.Add(new ComboBoxItem(Enum.GetName(enumType, value), value));
                }
            }

            comboBoxFirstServiceEnum.SelectedIndex = 0;

            if (service == TPUDSService.PUDS_SI_RoutineControl)
            {
                numericUpDownSecondServiceNumber.Maximum = 0xFFFF;

                labelSecondServiceEnum.Visible = true;
                labelSecondServiceEnum.Text = "Routine control";
                comboBoxSecondServiceEnum.Visible = true;
                comboBoxSecondServiceEnum.Items.Clear();

                foreach (UDSApi.TPUDSSvcParamRC_RID routineControl in Enum.GetValues(typeof(UDSApi.TPUDSSvcParamRC_RID)))
                {
                    comboBoxSecondServiceEnum.Items.Add(new ComboBoxItem(Enum.GetName(typeof(UDSApi.TPUDSSvcParamRC_RID), routineControl), routineControl));
                }

                comboBoxSecondServiceEnum.Items.Add(new ComboBoxItem("Non predefined value", (ushort)0));

                comboBoxSecondServiceEnum.SelectedIndex = 0;
            }
            else if (service == TPUDSService.PUDS_SI_CommunicationControl)
            {
                numericUpDownSecondServiceNumber.Maximum = 0xFF;

                labelSecondServiceEnum.Visible = true;
                labelSecondServiceEnum.Text = "Communication Type";
                comboBoxSecondServiceEnum.Visible = true;
                comboBoxSecondServiceEnum.Items.Clear();

                comboBoxSecondServiceEnum.Items.Add(new ComboBoxItem("PUDS_SVC_PARAM_CC_FLAG_APPL", UDSApi.PUDS_SVC_PARAM_CC_FLAG_APPL));
                comboBoxSecondServiceEnum.Items.Add(new ComboBoxItem("PUDS_SVC_PARAM_CC_FLAG_NWM", UDSApi.PUDS_SVC_PARAM_CC_FLAG_NWM));
                comboBoxSecondServiceEnum.Items.Add(new ComboBoxItem("PUDS_SVC_PARAM_CC_FLAG_DESCTIRNCN", UDSApi.PUDS_SVC_PARAM_CC_FLAG_DESCTIRNCN));
                comboBoxSecondServiceEnum.Items.Add(new ComboBoxItem("PUDS_SVC_PARAM_CC_FLAG_DENWRIRO", UDSApi.PUDS_SVC_PARAM_CC_FLAG_DENWRIRO));
                comboBoxSecondServiceEnum.Items.Add(new ComboBoxItem("PUDS_SVC_PARAM_CC_FLAG_DESNIBNN_MIN", UDSApi.PUDS_SVC_PARAM_CC_FLAG_DESNIBNN_MIN));
                comboBoxSecondServiceEnum.Items.Add(new ComboBoxItem("PUDS_SVC_PARAM_CC_FLAG_DESNIBNN_MAX", UDSApi.PUDS_SVC_PARAM_CC_FLAG_DESNIBNN_MAX));
                comboBoxSecondServiceEnum.Items.Add(new ComboBoxItem("Non predefined value", (byte)0));

                comboBoxSecondServiceEnum.SelectedIndex = 0;
            }
            else
            {
                labelSecondServiceEnum.Visible = false;
                comboBoxSecondServiceEnum.Visible = false;
            }
        }

        private void ButtonSend_Click(object sender, EventArgs e)
        {
            if (mMappingIsUUDT)
            {
                SendUUDT();
                return;
            }

            ServiceInformation serviceInformation = BuildServiceInformationFromUserInput();
            mServiceHandler.RunWorkerAsync(serviceInformation);
            buttonSend.Text = "Waiting..";
            buttonSend.Enabled = false;
        }
        private void SendUUDT()
        {
            MappingInformation information = (MappingInformation)(comboBoxMappings.SelectedItem as ComboBoxItem).Data;
            if (ConvertFirstBufferInputToArray(out byte[] data))
            {
                int dlc = (int)numericUpDownFirstServiceNumber.Value;

                if (data.Length != dlc)
                {
                    MessageBox.Show("DLC and data input do not match in length", "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    return;
                }

                mApiUser.SendUUDT(information, dlc, data);
            }
            else
                MessageBox.Show("Wrong buffer input", "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
        }
        private ServiceInformation BuildServiceInformationFromUserInput()
        {
            TPUDSService service = (TPUDSService)(comboBoxServiceToSend.SelectedItem as ComboBoxItem).Data;
            TPUDSNetAddrInfo netAddrInfo = ((MappingInformation)(comboBoxMappings.SelectedItem as ComboBoxItem).Data).NetAddressInformation;

            if (netAddrInfo.PROTOCOL == TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_29B)
            {
                netAddrInfo.SA = (byte)mApiUser.LocalNodeAddress;
                netAddrInfo.TA = (byte)numericUpDownTargetAddr.Value;
            }

            object[] parametersOfService = null;
            byte[] bufferByteInput;
            ushort[] bufferUshortInput;

            switch (service)
            {
                case TPUDSService.PUDS_SI_TesterPresent:
                    parametersOfService = new object[0];
                    break;
                case TPUDSService.PUDS_SI_DiagnosticSessionControl:
                    parametersOfService = new object[1];
                    parametersOfService[0] = (comboBoxFirstServiceEnum.SelectedItem as ComboBoxItem).Data;
                    break;
                case TPUDSService.PUDS_SI_ReadDataByIdentifier:
                    parametersOfService = new object[1];
                    if (ConvertFirstBufferInputToArray(out bufferUshortInput))
                        parametersOfService[0] = bufferUshortInput;
                    else
                        parametersOfService = null;
                    break;
                case TPUDSService.PUDS_SI_ClearDiagnosticInformation:
                    parametersOfService = new object[1];
                    parametersOfService[0] = numericUpDownFirstServiceNumber.Value;
                    break;
                case TPUDSService.PUDS_SI_ReadDTCInformation:
                    parametersOfService = new object[2];
                    parametersOfService[0] = (comboBoxFirstServiceEnum.SelectedItem as ComboBoxItem).Data;
                    parametersOfService[1] = numericUpDownFirstServiceNumber.Value;
                    break;
                case TPUDSService.PUDS_SI_ControlDTCSetting:
                    parametersOfService = new object[2];
                    parametersOfService[0] = (comboBoxFirstServiceEnum.SelectedItem as ComboBoxItem).Data;
                    if (ConvertFirstBufferInputToArray(out bufferByteInput))
                        parametersOfService[1] = bufferByteInput;
                    else
                        parametersOfService = null;
                    break;
                case TPUDSService.PUDS_SI_CommunicationControl:
                    parametersOfService = new object[2];
                    parametersOfService[0] = (comboBoxFirstServiceEnum.SelectedItem as ComboBoxItem).Data;
                    parametersOfService[1] = numericUpDownSecondServiceNumber.Value;
                    break;
                case TPUDSService.PUDS_SI_SecurityAccess:
                    parametersOfService = new object[2];
                    parametersOfService[0] = (comboBoxFirstServiceEnum.SelectedItem as ComboBoxItem).Data;
                    if (ConvertFirstBufferInputToArray(out bufferByteInput))
                        parametersOfService[1] = bufferByteInput;
                    else
                        parametersOfService = null;
                    break;
                case TPUDSService.PUDS_SI_RequestDownload:
                    parametersOfService = new object[4];
                    parametersOfService[0] = numericUpDownFirstServiceNumber.Value;
                    parametersOfService[1] = numericUpDownSecondServiceNumber.Value;
                    if (ConvertFirstBufferInputToArray(out bufferByteInput))
                    {
                        parametersOfService[2] = bufferByteInput;

                        if (ConvertSecondBufferInputToArray(out bufferByteInput))
                            parametersOfService[3] = bufferByteInput;
                        else
                            parametersOfService = null;
                    }
                    else
                        parametersOfService = null;
                    break;
                case TPUDSService.PUDS_SI_TransferData:
                    parametersOfService = new object[2];
                    parametersOfService[0] = numericUpDownFirstServiceNumber.Value;
                    if (ConvertFirstBufferInputToArray(out bufferByteInput))
                        parametersOfService[1] = bufferByteInput;
                    else
                        parametersOfService = null;
                    break;
                case TPUDSService.PUDS_SI_RequestTransferExit:
                    parametersOfService = new object[1];
                    if (ConvertFirstBufferInputToArray(out bufferByteInput))
                        parametersOfService[0] = bufferByteInput;
                    else
                        parametersOfService = null;
                    break;
                case TPUDSService.PUDS_SI_RoutineControl:
                    parametersOfService = new object[3];
                    parametersOfService[0] = (comboBoxFirstServiceEnum.SelectedItem as ComboBoxItem).Data;
                    parametersOfService[1] = numericUpDownSecondServiceNumber.Value;
                    if (ConvertFirstBufferInputToArray(out bufferByteInput))
                        parametersOfService[2] = bufferByteInput;
                    else
                        parametersOfService = null;
                    break;
                case TPUDSService.PUDS_SI_ECUReset:
                    parametersOfService = new object[1];
                    parametersOfService[0] = (comboBoxFirstServiceEnum.SelectedItem as ComboBoxItem).Data;
                    break;
            }

            if (parametersOfService == null)
                return null;

            return new ServiceInformation(service, parametersOfService, netAddrInfo, !checkBoxSendTesterPresent.Checked, (byte)numericUpDownPriority.Value);
        }
        private bool ConvertFirstBufferInputToArray(out ushort[] bufferInput)
        {
            if (textBoxFirstServiceBuffer.Text.Length == 0)
            {
                bufferInput = new ushort[0];
                return true;
            }

            string[] temp = textBoxFirstServiceBuffer.Text.Split(',');

            bufferInput = new ushort[temp.Length];

            for (int i = 0; i < temp.Length; i++)
            {
                if (UInt16.TryParse(temp[i], System.Globalization.NumberStyles.HexNumber, null, out ushort result))
                    bufferInput[i] = result;
                else
                    return false;
            }

            return true;
        }
        private bool ConvertFirstBufferInputToArray(out byte[] bufferInput)
        {
            if (textBoxFirstServiceBuffer.Text.Length == 0)
            {
                bufferInput = new byte[0];
                return true;
            }

            string[] temp = textBoxFirstServiceBuffer.Text.Split(',');

            bufferInput = new byte[temp.Length];

            for (int i = 0; i < temp.Length; i++)
            {
                if (Byte.TryParse(temp[i], System.Globalization.NumberStyles.HexNumber, null, out byte result))
                    bufferInput[i] = result;
                else
                    return false;
            }

            return true;
        }
        private bool ConvertSecondBufferInputToArray(out byte[] bufferInput)
        {
            if (textBoxSecondServiceBuffer.Text.Length == 0)
            {
                bufferInput = new byte[0];
                return true;
            }

            string[] temp = textBoxSecondServiceBuffer.Text.Split(',');

            bufferInput = new byte[temp.Length];

            for (int i = 0; i < temp.Length; i++)
            {
                if (Byte.TryParse(temp[i], System.Globalization.NumberStyles.HexNumber, null, out byte result))
                    bufferInput[i] = result;
                else
                    return false;
            }

            return true;
        }

        private void SendServiceAndWaitForResponse(object sender, DoWorkEventArgs e)
        {
            if (e.Argument == null)
                e.Result = -1;
            else if (!mApiUser.SendService((ServiceInformation)e.Argument))
                e.Result = -2;
            else
            {
                if (mApiUser.WaitForResponse())
                    e.Result = 1;
                else
                    e.Result = 0;
            }
        }
        private void HandleResponse(object sender, RunWorkerCompletedEventArgs e)
        {
            switch ((int)e.Result)
            {
                case -1:
                    MessageBox.Show("Wrong buffer input was given.", "Error");
                    break;
                case -2:
                    mApiUser.ShowResponseError();
                    break;
                case 0:
                    mApiUser.ShowResponseError();
                    break;
                case 1:
                    if (mApiUser.RequestIsFunctional)
                        ShowResponse(mApiUser.FunctionalResponses, mApiUser.ReceivedFunctionalResponses);
                    else
                        ShowResponse(mApiUser.Response);
                    break;
            }

            buttonSend.Text = "Send";
            buttonSend.Enabled = true;
        }

        private void ShowResponse(TPUDSMsg response)
        {
            ShowResponse(new TPUDSMsg[] { response }, 1);
        }
        private void ShowResponse(TPUDSMsg[] responses, uint amountOfResponses)
        {
            listViewResponse.BeginUpdate();

            for (int i = 0; i < amountOfResponses; i++)
            {
                AddResponseToList(responses[i]);
            }

            listViewResponse.EndUpdate();
        }
        private void AddResponseToList(TPUDSMsg response)
        {
            ResponseInformation responseInformation = new ResponseInformation(response);

            if (!mResponsesReceived.Contains(responseInformation))
            {
                mResponsesReceived.Add(responseInformation);
            }
            else
            {
                int index = mResponsesReceived.IndexOf(responseInformation);
                mResponsesReceived[index].Count++;
                responseInformation.Count = mResponsesReceived[index].Count;
            }

            bool itemFound = false;

            foreach (ListViewItem item in listViewResponse.Items)
            {
                if (responseInformation.Equals(item.Tag))
                {
                    item.SubItems[6].Text = responseInformation.ResultText;
                    item.SubItems[7].Text = responseInformation.CountText;
                    item.SubItems[8].Text = responseInformation.LengthText;
                    item.SubItems[9].Text = responseInformation.DataText;
                    itemFound = true;
                    break;
                }
            }

            if (!itemFound)
            {
                ListViewItem viewItem = new ListViewItem(responseInformation.SourceText);
                viewItem.SubItems.Add(responseInformation.TargetText);
                viewItem.SubItems.Add(responseInformation.RemoteText);
                viewItem.SubItems.Add(responseInformation.TargetTypeText);
                viewItem.SubItems.Add(responseInformation.ProtocolText);
                viewItem.SubItems.Add(responseInformation.ServiceIdentifierText);
                viewItem.SubItems.Add(responseInformation.ResultText);
                viewItem.SubItems.Add(responseInformation.CountText);
                viewItem.SubItems.Add(responseInformation.LengthText);
                viewItem.SubItems.Add(responseInformation.DataText);
                viewItem.Tag = responseInformation;
                listViewResponse.Items.Add(viewItem);
            }
        }


        private void buttonReceive_Click(object sender, EventArgs e)
        {
            if (!mRecevieUUDTWorker.IsBusy)
            {
                mRecevieUUDTWorker.RunWorkerAsync((MappingInformation)(comboBoxMappings.SelectedItem as ComboBoxItem).Data);
                buttonReceive.Text = "Stop";
            }
            else
            {
                mRecevieUUDTWorker.CancelAsync();
                buttonReceive.Text = "Receive";
            }

        }
        private void ReceiveUUDT(object sender, DoWorkEventArgs e)
        {
            BackgroundWorker worker = sender as BackgroundWorker;

            e.Result = 0;

            do
            {
                if (mApiUser.ReceiveUUDT((MappingInformation)e.Argument))
                {
                    e.Result = 1;
                    break;
                }
                else if (mApiUser.UUDTError)
                {
                    e.Result = -1;
                    break;
                }

                System.Threading.Thread.Sleep(100);

            } while (!worker.CancellationPending);
        }
        private void EndReceiveUUDT(object sender, RunWorkerCompletedEventArgs e)
        {
            if ((int)e.Result == 1)
                ShowResponse(mApiUser.UUDTResponse);

            if ((int)e.Result == -1)
                mApiUser.ShowResponseError(true);

            buttonReceive.Text = "Receive";
        }

        private void comboBoxMappings_SelectedIndexChanged(object sender, EventArgs e)
        {
            EnableSendGroupbox();
        }
        private void EnableSendGroupbox()
        {
            MappingInformation information = (MappingInformation)(comboBoxMappings.SelectedItem as ComboBoxItem).Data;

            if (!mServiceHandler.IsBusy)
            {
                if ((information.NetAddressInformation.SA != mApiUser.LocalNodeAddress) && (!information.IsFixedCanIdprotocol))
                    buttonSend.Enabled = false;
                else
                    buttonSend.Enabled = mApiUser.IsConnected;
            }

            if (information.IsUUDT)
            {
                mMappingIsUUDT = true;

                buttonReceive.Visible = true;

                if ((information.TargetAddress == mApiUser.LocalNodeAddress)|| (information.SourceAddress == mApiUser.LocalNodeAddress))
                    buttonReceive.Enabled = true;
                else
                    buttonReceive.Enabled = false;

                labelPriority.Visible = false;
                numericUpDownPriority.Visible = false;
                labelService.Visible = false;
                comboBoxServiceToSend.Visible = false;
                labelFirstServiceEnum.Visible = false;
                comboBoxFirstServiceEnum.Visible = false;
                labelSecondServiceEnum.Visible = false;
                comboBoxSecondServiceEnum.Visible = false;
                labelSecondServiceNumber.Visible = false;
                numericUpDownSecondServiceNumber.Visible = false;
                labelSecondServiceBuffer.Visible = false;
                labelFirstServiceBufferHint.Visible = false;
                textBoxSecondServiceBuffer.Visible = false;
                checkBoxSendTesterPresent.Visible = false;

                labelFirstServiceNumber.Text = "DLC";
                labelFirstServiceNumber.Visible = true;
                numericUpDownFirstServiceNumber.Visible = true;
                numericUpDownFirstServiceNumber.Maximum = 8;

                labelFirstServiceBuffer.Text = "Data of message";
                labelFirstServiceBuffer.Visible = true;
                labelFirstServiceBufferHint.Visible = true;
                textBoxFirstServiceBuffer.Visible = true;

                numericUpDownTargetAddr.Value = information.TargetAddress;
                numericUpDownRemoteAddr.Value = information.RemoteAddress;
                textBoxSourceAddress.Text = information.SourceAddress.ToString("X");
                numericUpDownTargetAddr.Enabled = false;
                numericUpDownRemoteAddr.Enabled = false;
                if (information.SourceAddress != mApiUser.LocalNodeAddress)
                    textBoxSourceAddress.BackColor = System.Drawing.Color.Red;
                else
                    textBoxSourceAddress.BackColor = System.Drawing.SystemColors.Control;
                return;
            }
            else
            {
                mMappingIsUUDT = false;

                buttonReceive.Visible = false;
                labelService.Visible = true;
                comboBoxServiceToSend.Visible = true;
                ComboBoxServiceToSend_SelectedIndexChanged(null, EventArgs.Empty);
            }

            if (!information.IsFixedCanIdprotocol)
            {
                numericUpDownTargetAddr.Value = information.TargetAddress;
                numericUpDownRemoteAddr.Value = information.RemoteAddress;
                textBoxSourceAddress.Text = information.SourceAddress.ToString("X");

                if (information.SourceAddress != mApiUser.LocalNodeAddress)
                    textBoxSourceAddress.BackColor = System.Drawing.Color.Red;
                else
                    textBoxSourceAddress.BackColor = System.Drawing.SystemColors.Control;
            }

            numericUpDownTargetAddr.Enabled = information.IsFixedCanIdprotocol;
            numericUpDownRemoteAddr.Enabled = information.IsFixedCanIdprotocol;
            labelPriority.Visible = information.IsFixedCanIdprotocol;
            numericUpDownPriority.Visible = information.IsFixedCanIdprotocol;
        }

        private void comboBoxFirstServiceEnum_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (mServiceHandler.IsBusy)
                return;

            TPUDSService service = (TPUDSService)(comboBoxServiceToSend.SelectedItem as ComboBoxItem).Data;

            if (service == TPUDSService.PUDS_SI_DiagnosticSessionControl)
            {
                UDSApi.TPUDSSvcParamDSC sessionType = (UDSApi.TPUDSSvcParamDSC)(comboBoxFirstServiceEnum.SelectedItem as ComboBoxItem).Data;

                checkBoxSendTesterPresent.Enabled = sessionType != UDSApi.TPUDSSvcParamDSC.PUDS_SVC_PARAM_DSC_DS;
            }

        }

        private void ComboBoxSecondServiceEnum_SelectedIndexChanged(object sender, EventArgs e)
        {
            TPUDSService service = ((TPUDSService)(comboBoxServiceToSend.SelectedItem as ComboBoxItem).Data);

            if (service == TPUDSService.PUDS_SI_RoutineControl)
            {
                ComboBoxItem item = comboBoxSecondServiceEnum.SelectedItem as ComboBoxItem;
                if (((ushort)item.Data != 0) && (numericUpDownSecondServiceNumber.Value != (ushort)item.Data))
                    numericUpDownSecondServiceNumber.Value = (ushort)item.Data;
            }
            else if (service == TPUDSService.PUDS_SI_CommunicationControl)
            {
                ComboBoxItem item = comboBoxSecondServiceEnum.SelectedItem as ComboBoxItem;
                if (((byte)item.Data != 0) && (numericUpDownSecondServiceNumber.Value != (byte)item.Data))
                    numericUpDownSecondServiceNumber.Value = (byte)item.Data;
            }
        }
        
        private void NumericUpDownSecondServiceNumber_ValueChanged(object sender, EventArgs e)
        {
            if (mServiceHandler.IsBusy)
                return;
         
            TPUDSService service = ((TPUDSService)(comboBoxServiceToSend.SelectedItem as ComboBoxItem).Data);

            if (service == TPUDSService.PUDS_SI_RoutineControl)
            {
                bool valueFound = false;

                foreach (ComboBoxItem comboboxItem in comboBoxSecondServiceEnum.Items)
                {
                    if ((ushort)comboboxItem.Data == numericUpDownSecondServiceNumber.Value)
                    {
                        comboBoxSecondServiceEnum.SelectedItem = comboboxItem;
                        valueFound = true;
                    }
                }

                if (!valueFound)
                    comboBoxSecondServiceEnum.SelectedIndex = comboBoxSecondServiceEnum.Items.Count - 1;

                ComboBoxItem item = comboBoxSecondServiceEnum.SelectedItem as ComboBoxItem;
                if ((ushort)item.Data != 0)
                    numericUpDownSecondServiceNumber.Value = (ushort)item.Data;
            }
            else if (service == TPUDSService.PUDS_SI_CommunicationControl)
            {
                bool valueFound = false;

                foreach (ComboBoxItem comboboxItem in comboBoxSecondServiceEnum.Items)
                {
                    if ((byte)comboboxItem.Data == numericUpDownSecondServiceNumber.Value)
                    {
                        comboBoxSecondServiceEnum.SelectedItem = comboboxItem;
                        valueFound = true;
                    }
                }

                if (!valueFound)
                    comboBoxSecondServiceEnum.SelectedIndex = comboBoxSecondServiceEnum.Items.Count - 1;

                ComboBoxItem item = comboBoxSecondServiceEnum.SelectedItem as ComboBoxItem;
                if ((byte)item.Data != 0)
                    numericUpDownSecondServiceNumber.Value = (byte)item.Data;
            }
        }
        #endregion

        private void contextMenuReset_Click(object sender, EventArgs e)
        {
            mApiUser.ResetHardware();
            mResponsesReceived.Clear();
            listViewResponse.Items.Clear();
        }
    }

    public class ComboBoxItem
    {
        public Object Data { get; set; }
        public string Text { get; set; }

        public ComboBoxItem(string text, object data)
        {
            Text = text;
            Data = data;
        }

        public override string ToString()
        {
            return Text;
        }
    }

    public class ServiceInformation
    {
        public TPUDSService Service { get; private set; }
        public object[] ParametersOfService { get; private set; }
        public TPUDSNetAddrInfo NetAddressInformation { get; private set; }
        public bool DisableTesterPresent { get; private set; }
        public byte Priority { get; private set; }

        public ServiceInformation(TPUDSService service, object[] parametersOfService, TPUDSNetAddrInfo netAddrInfo, bool disableTesterPresent = false, byte priority = 6)
        {
            Service = service;
            ParametersOfService = parametersOfService;
            NetAddressInformation = netAddrInfo;
            DisableTesterPresent = disableTesterPresent;
            Priority = priority;
        }
    }

    public class MappingInformation
    {
        private IntPtr mPointer;
        private TPUDSMsg mMessage;

        private byte mSourceAddress;
        private byte mTargetAddress;
        private byte mRemoteAddress;
        private TPUDSAddressingType mTargetType;
        private TPUDSProtocol mProtocol;
        private TPUDSNetAddrInfo mNetAddressInformation;

        public uint CanId { get; set; }
        public uint ResponseCanId { get; set; }
        public byte SourceAddress
        {
            get { return mSourceAddress; }
            set
            {
                mSourceAddress = value;
                mNetAddressInformation.SA = mSourceAddress;
            }
        }
        public byte TargetAddress
        {
            get { return mTargetAddress; }
            set
            {
                mTargetAddress = value;
                mNetAddressInformation.TA = mTargetAddress;
            }
        }
        public byte RemoteAddress
        {
            get { return mRemoteAddress; }
            set
            {
                mRemoteAddress = value;
                mNetAddressInformation.RA = mRemoteAddress;
            }
        }
        public TPUDSAddressingType TargetType
        {
            get { return mTargetType; }
            set
            {
                mTargetType = value;
                mNetAddressInformation.TA_TYPE = mTargetType;
            }
        }
        public TPUDSProtocol Protocol
        {
            get { return mProtocol; }
            set
            {
                mProtocol = value;
                mNetAddressInformation.PROTOCOL = mProtocol;
            }
        }

        public TPUDSNetAddrInfo NetAddressInformation => mNetAddressInformation;

        public IntPtr MappingPointer
        {
            get
            {
                mMessage.NETADDRINFO = NetAddressInformation;
                if (IsUUDT)
                    mMessage.LEN = 4;
                else
                    mMessage.LEN = 8;

                mMessage.DATA[0] = (byte)((CanId >> 24) & 0xFF);
                mMessage.DATA[1] = (byte)((CanId >> 16) & 0xFF);
                mMessage.DATA[2] = (byte)((CanId >> 8) & 0xFF);
                mMessage.DATA[3] = (byte)(CanId & 0xFF);

                if (!IsUUDT)
                {
                    mMessage.DATA[4] = (byte)((ResponseCanId >> 24) & 0xFF);
                    mMessage.DATA[5] = (byte)((ResponseCanId >> 16) & 0xFF);
                    mMessage.DATA[6] = (byte)((ResponseCanId >> 8) & 0xFF);
                    mMessage.DATA[7] = (byte)(ResponseCanId & 0xFF);
                }

                Marshal.StructureToPtr(mMessage, mPointer, false);

                return mPointer;
            }
        }
        public IntPtr MappingPointerRemoval
        {
            get
            {
                mMessage.NETADDRINFO = NetAddressInformation;
                mMessage.LEN = 4;

                mMessage.DATA[0] = (byte)((CanId >> 24) & 0xFF);
                mMessage.DATA[1] = (byte)((CanId >> 16) & 0xFF);
                mMessage.DATA[2] = (byte)((CanId >> 8) & 0xFF);
                mMessage.DATA[3] = (byte)(CanId & 0xFF);
                
                Marshal.StructureToPtr(mMessage, mPointer, false);

                return mPointer;
            }
        }
        public uint PointerSize { get { return (uint)Marshal.SizeOf(mMessage); } }

        public string CanIdString
        {
            get
            {
                if (!UdsExampleUtils.ProtocolIsDetachedFromIdsAndAddresses(Protocol))
                    return UdsExampleUtils.GetCanIdString(CanId, UdsExampleUtils.ProtocolIs29Bit(Protocol));
                else
                    return "-";
            }
        }
        public string ResponseCanIdString
        {
            get
            {
                if ((!UdsExampleUtils.ProtocolIsDetachedFromIdsAndAddresses(Protocol)) && (TargetType != TPUDSAddressingType.PUDS_ADDRESSING_FUNCTIONAL) && !IsUUDT)
                    return UdsExampleUtils.GetCanIdString(ResponseCanId, UdsExampleUtils.ProtocolIs29Bit(Protocol));
                else
                    return "-";
            }
        }
        public string TargetTypeString { get { return Enum.GetName(typeof(TPUDSAddressingType), TargetType); } }
        public string TargetTypeShortString { get { return UdsExampleUtils.GetTargetTypeString(TargetType); } }
        public string ProtocolString { get { return Enum.GetName(typeof(TPUDSProtocol), Protocol); } }
        public string ProtocolShortString { get { return UdsExampleUtils.GetProtocolString(Protocol); } }
        public string SourceAddressString 
        { 
            get 
            {
                if (!UdsExampleUtils.ProtocolIsDetachedFromIdsAndAddresses(Protocol))
                    return SourceAddress.ToString("X2") + "h";
                else
                    return "-";
            } 
        }
        public string TargetAddressString 
        { 
            get 
            { 
                if (!UdsExampleUtils.ProtocolIsDetachedFromIdsAndAddresses(Protocol))
                    return TargetAddress.ToString("X2") + "h";
                else
                    return "-";
            } 
        }
        public string RemoteAddressString 
        { 
            get 
            { 
                if (!UdsExampleUtils.ProtocolIsDetachedFromIdsAndAddresses(Protocol))
                    return RemoteAddress.ToString("X2") + "h";
                else
                    return "-";
            } 
        }
        public string IsTwoWayString
        {
            get
            {
                if (UdsExampleUtils.ProtocolIsDetachedFromIdsAndAddresses(Protocol) || (TargetType == TPUDSAddressingType.PUDS_ADDRESSING_FUNCTIONAL) || IsUUDT)
                    return "-";
                if (IsTwoWay)
                    return Boolean.TrueString;
                return Boolean.FalseString;
            }
        }

        public string Name
        {
            get
            {
                if (UdsExampleUtils.ProtocolIsDetachedFromIdsAndAddresses(Protocol))
                    return ProtocolShortString + ", " + TargetTypeShortString;

                if (IsUUDT)
                    return ProtocolShortString + " (UUDT): Can ID = " + CanIdString;

                if (IsRemote)
                    return ProtocolShortString + ", " + TargetTypeShortString + ": " + SourceAddressString + " -> " + RemoteAddressString + " (remote)";

                return ProtocolShortString + ", " + TargetTypeShortString + ": " + SourceAddressString + " -> " + TargetAddressString;
            }
        }

        public bool IsRemote { get { return UdsExampleUtils.ProtocolIsRemote(Protocol); } }
        public bool IsTwoWay { get; set; }

        //No mapping needs to be/can be added for protocols with fixed CAN IDs, they are only represented as MappingInformation to simplify data transactions within the application.
        public bool IsFixedCanIdprotocol { get { return UdsExampleUtils.ProtocolIsDetachedFromIdsAndAddresses(Protocol); } }
        public bool IsUUDT { get { return mProtocol == TPUDSProtocol.PUDS_PROTOCOL_NONE; } }

        public MappingInformation(bool isTwoWay = false)
        {
            mMessage = new TPUDSMsg();
            mMessage.DATA = new byte[4095];

            IsTwoWay = isTwoWay;

            mPointer = Marshal.AllocHGlobal(Marshal.SizeOf(mMessage));
        }
        public MappingInformation(uint canId, uint responseCanId, byte sourceAddress, byte targetAddress, byte remoteAddress, TPUDSAddressingType targetType, TPUDSProtocol protocol, bool isTwoWay = false) : this(isTwoWay)
        {
            CanId = canId;
            ResponseCanId = responseCanId;
            SourceAddress = sourceAddress;
            TargetAddress = targetAddress;
            RemoteAddress = remoteAddress;
            TargetType = targetType;
            Protocol = protocol;
        }
        public MappingInformation(TPUDSAddressingType targetType, TPUDSProtocol protocol) : this(false)
        {
            TargetType = targetType;
            Protocol = protocol;
            CanId = 0;
            ResponseCanId = 0;
            SourceAddress = 0;
            TargetAddress = 0;
            RemoteAddress = 0;
        }
        ~MappingInformation()
        {
            Marshal.FreeHGlobal(mPointer);
        }

        public MappingInformation Backwards()
        {
            return new MappingInformation(ResponseCanId, CanId, TargetAddress, SourceAddress, RemoteAddress, TargetType, Protocol);
        }

        public override bool Equals(object obj)
        {
            MappingInformation info = obj as MappingInformation;
            if (info == null)
                return false;

            if (UdsExampleUtils.ProtocolIsDetachedFromIdsAndAddresses(Protocol))
                return SourceAddress == info.SourceAddress &&
                   TargetAddress ==     info.TargetAddress &&
                   RemoteAddress ==     info.RemoteAddress &&
                   TargetType ==        info.TargetType &&
                   Protocol ==          info.Protocol;

            return CanId ==         info.CanId &&
                   ResponseCanId == info.ResponseCanId &&
                   SourceAddress == info.SourceAddress &&
                   TargetAddress == info.TargetAddress &&
                   RemoteAddress == info.RemoteAddress &&
                   TargetType ==    info.TargetType &&
                   Protocol ==      info.Protocol;
        }

        public override int GetHashCode()
        {
            var hashCode = -1246325849;
            hashCode = hashCode * -1521134295 + CanId.GetHashCode();
            hashCode = hashCode * -1521134295 + ResponseCanId.GetHashCode();
            hashCode = hashCode * -1521134295 + SourceAddress.GetHashCode();
            hashCode = hashCode * -1521134295 + TargetAddress.GetHashCode();
            hashCode = hashCode * -1521134295 + RemoteAddress.GetHashCode();
            hashCode = hashCode * -1521134295 + TargetType.GetHashCode();
            hashCode = hashCode * -1521134295 + Protocol.GetHashCode();
            return hashCode;
        }
    }

    public class ResponseInformation
    {
        public TPUDSMsg Response { get; private set; }

        public string SourceText { get { return Response.NETADDRINFO.SA.ToString("X2"); } }
        public string TargetText { get { return Response.NETADDRINFO.TA.ToString("X2"); } }
        public string RemoteText { get { return Response.NETADDRINFO.RA.ToString("X2"); } }
        public string TargetTypeText { get { return Enum.GetName(typeof(TPUDSAddressingType), Response.NETADDRINFO.TA_TYPE); } }
        public string ProtocolText { get { return Enum.GetName(typeof(TPUDSProtocol), Response.NETADDRINFO.PROTOCOL); } }
        public string ServiceIdentifierText { get { return (Response.NETADDRINFO.PROTOCOL == TPUDSProtocol.PUDS_PROTOCOL_NONE)? "- (UUDT Response)" : Response.ServiceID.ToString("X2"); } }
        public string ResultText { get { return Enum.GetName(typeof(TPUDSServiceResult), Response.RESULT); } }
        public string CountText { get { return Count.ToString(); } }
        public string LengthText { get { return Response.LEN.ToString(); } }
        public string DataText 
        { 
            get 
            {
                string data = "";

                for (int i = 0; i < Response.LEN; i++)
                {
                    data += Response.DATA[i].ToString("X2") + " ";
                }

                return data;
            } 
        }

        public int Count { get; set; }

        public ResponseInformation(TPUDSMsg response)
        {
            Response = response;
            Count = 1;
        }

        public override bool Equals(object obj)
        {
            return obj is ResponseInformation information &&
                   EqualityComparer<TPUDSNetAddrInfo>.Default.Equals(Response.NETADDRINFO, information.Response.NETADDRINFO) &&
                   Response.MSGTYPE == information.Response.MSGTYPE &&
                   (Response.ServiceID == information.Response.ServiceID || (Response.NETADDRINFO.PROTOCOL == information.Response.NETADDRINFO.PROTOCOL && Response.NETADDRINFO.PROTOCOL == TPUDSProtocol.PUDS_PROTOCOL_NONE));
        }
    }

    public static class UdsExampleUtils
    {
        public static readonly TPUDSCANHandle[] AllChannelHandles =
        {
            UDSApi.PUDS_NONEBUS,

            UDSApi.PUDS_ISABUS1,
            UDSApi.PUDS_ISABUS2,
            UDSApi.PUDS_ISABUS3,
            UDSApi.PUDS_ISABUS4,
            UDSApi.PUDS_ISABUS5,
            UDSApi.PUDS_ISABUS6,
            UDSApi.PUDS_ISABUS7,
            UDSApi.PUDS_ISABUS8,

            UDSApi.PUDS_DNGBUS1,

            UDSApi.PUDS_PCIBUS1,
            UDSApi.PUDS_PCIBUS2,
            UDSApi.PUDS_PCIBUS3,
            UDSApi.PUDS_PCIBUS4,
            UDSApi.PUDS_PCIBUS5,
            UDSApi.PUDS_PCIBUS6,
            UDSApi.PUDS_PCIBUS7,
            UDSApi.PUDS_PCIBUS8,
            UDSApi.PUDS_PCIBUS9,
            UDSApi.PUDS_PCIBUS10,
            UDSApi.PUDS_PCIBUS11,
            UDSApi.PUDS_PCIBUS12,
            UDSApi.PUDS_PCIBUS13,
            UDSApi.PUDS_PCIBUS14,
            UDSApi.PUDS_PCIBUS15,
            UDSApi.PUDS_PCIBUS16,

            UDSApi.PUDS_USBBUS1,
            UDSApi.PUDS_USBBUS2,
            UDSApi.PUDS_USBBUS3,
            UDSApi.PUDS_USBBUS4,
            UDSApi.PUDS_USBBUS5,
            UDSApi.PUDS_USBBUS6,
            UDSApi.PUDS_USBBUS7,
            UDSApi.PUDS_USBBUS8,
            UDSApi.PUDS_USBBUS9,
            UDSApi.PUDS_USBBUS10,
            UDSApi.PUDS_USBBUS11,
            UDSApi.PUDS_USBBUS12,
            UDSApi.PUDS_USBBUS13,
            UDSApi.PUDS_USBBUS14,
            UDSApi.PUDS_USBBUS15,
            UDSApi.PUDS_USBBUS16,

            UDSApi.PUDS_PCCBUS1,
            UDSApi.PUDS_PCCBUS2,

            UDSApi.PUDS_LANBUS1,
            UDSApi.PUDS_LANBUS2,
            UDSApi.PUDS_LANBUS3,
            UDSApi.PUDS_LANBUS4,
            UDSApi.PUDS_LANBUS5,
            UDSApi.PUDS_LANBUS6,
            UDSApi.PUDS_LANBUS7,
            UDSApi.PUDS_LANBUS8,
            UDSApi.PUDS_LANBUS9,
            UDSApi.PUDS_LANBUS10,
            UDSApi.PUDS_LANBUS11,
            UDSApi.PUDS_LANBUS12,
            UDSApi.PUDS_LANBUS13,
            UDSApi.PUDS_LANBUS14,
            UDSApi.PUDS_LANBUS15,
            UDSApi.PUDS_LANBUS16,
        };

        /// <summary>
        /// Represents a PCAN device
        /// </summary>
        public enum TPCANDevice : byte
        {
            /// <summary>
            /// Undefined, unknown or not selected PCAN device value
            /// </summary>
            PCAN_NONE = 0,
            /// <summary>
            /// PCAN Non-Plug and Play devices. NOT USED WITHIN PCAN-Basic API
            /// </summary>
            PCAN_PEAKCAN = 1,
            /// <summary>
            /// PCAN-ISA, PCAN-PC/104, and PCAN-PC/104-Plus
            /// </summary>
            PCAN_ISA = 2,
            /// <summary>
            /// PCAN-Dongle
            /// </summary>
            PCAN_DNG = 3,
            /// <summary>
            /// PCAN-PCI, PCAN-cPCI, PCAN-miniPCI, and PCAN-PCI Express
            /// </summary>
            PCAN_PCI = 4,
            /// <summary>
            /// PCAN-USB and PCAN-USB Pro
            /// </summary>
            PCAN_USB = 5,
            /// <summary>
            /// PCAN-PC Card
            /// </summary>
            PCAN_PCC = 6,
            /// <summary>
            /// PCAN Virtual hardware. NOT USED WITHIN PCAN-Basic API
            /// </summary>
            PCAN_VIRTUAL = 7,
            /// <summary>
            /// PCAN Gateway devices
            /// </summary>
            PCAN_LAN = 8
        }

        public static bool ProtocolIs29Bit(TPUDSProtocol protocol)
        {
            return protocol == TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_29B ||
                    protocol == TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_29B_REMOTE ||
                    protocol == TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_3_29B ||
                    protocol == TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_29B_NORMAL ||
                    protocol == TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_29B_EXTENDED;
        }
        public static bool ProtocolIsRemote(TPUDSProtocol protocol)
        {
            return protocol == TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_11B_REMOTE || protocol == TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_29B_REMOTE;
        }
        public static bool ProtocolIsDetachedFromIdsAndAddresses(TPUDSProtocol protocol)
        {
            return protocol == TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_29B
                || protocol == TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_29B_REMOTE
                || protocol == TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_3_29B;
        }

        public static string GetHardwareChannelName(TPUDSCANHandle handle)
        {
            TPCANDevice device;
            int channel;

            if (handle < 0x100)
            {
                device = (TPCANDevice)(handle >> 4);
                channel = handle & 0xF;
            }
            else
            {
                device = (TPCANDevice)(handle >> 8);
                channel = handle & 0xFF;
            }

            return String.Format("{0} {1} ({2:X2}h)", device, channel, handle);
        }
        public static string GetBitrateString(TPUDSBaudrate baudrate)
        {
            switch (baudrate)
            {
                case TPUDSBaudrate.PUDS_BAUD_1M:
                    return "1 MBit/s";
                case TPUDSBaudrate.PUDS_BAUD_800K:
                    return "800 kBit/s";
                case TPUDSBaudrate.PUDS_BAUD_500K:
                    return "500 kBit/s";
                case TPUDSBaudrate.PUDS_BAUD_250K:
                    return "250 kBit/s";
                case TPUDSBaudrate.PUDS_BAUD_125K:
                    return "125 kBit/s";
                case TPUDSBaudrate.PUDS_BAUD_100K:
                    return "100 kBit/s";
                case TPUDSBaudrate.PUDS_BAUD_95K:
                    return "95,238 kBit/s";
                case TPUDSBaudrate.PUDS_BAUD_83K:
                    return "83,333 kBit/s";
                case TPUDSBaudrate.PUDS_BAUD_50K:
                    return "50 kBit/s";
                case TPUDSBaudrate.PUDS_BAUD_47K:
                    return "47,619 kBit/s";
                case TPUDSBaudrate.PUDS_BAUD_33K:
                    return "33,333 kBit/s";
                case TPUDSBaudrate.PUDS_BAUD_20K:
                    return "20 kBit/s";
                case TPUDSBaudrate.PUDS_BAUD_10K:
                    return "10 kBit/s";
                case TPUDSBaudrate.PUDS_BAUD_5K:
                    return "5 kBit/s";
            }
            return baudrate.ToString();
        }
        public static string GetCanIdString(uint id, bool isExtended)
        {
            if (isExtended)
                return id.ToString("X8") + "h";
            else
                return id.ToString("X3") + "h";
        }
        public static string GetProtocolString(TPUDSProtocol protocol)
        {
            switch (protocol)
            {
                case TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_11B:
                    return "11bit normal";
                case TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_11B_REMOTE:
                    return "11bit mixed,remote";
                case TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_29B:
                    return "29bit fixed normal";
                case TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_29B_REMOTE:
                    return "29bit mixed,remote";
                case TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_3_29B:
                    return "29bit enhanced diagnostic";
                case TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_29B_NORMAL:
                    return "29bit normal";
                case TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_11B_EXTENDED:
                    return "11bit mixed,remote";
                case TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_29B_EXTENDED:
                    return "29bit enhanced diagnostic";
                default:
                case TPUDSProtocol.PUDS_PROTOCOL_NONE:
                    return "no protocol";
            }
        }
        public static string GetTargetTypeString(TPUDSAddressingType addressingType)
        {
            switch (addressingType)
            {
                case TPUDSAddressingType.PUDS_ADDRESSING_PHYSICAL:
                    return "physical addressing";
                case TPUDSAddressingType.PUDS_ADDRESSING_FUNCTIONAL:
                    return "functional addressing";
                default:
                    return "";
            }
        }
    }
}
