﻿using System;
using System.Linq;
using System.Drawing;
using System.Windows.Forms;

using Peak.Can.Uds;

namespace PcanUdsExample
{
    public partial class FormMapping : Form
    {
        public MappingInformation MappingInformation { get; private set; }
        private uint mSourceAddress;
        private bool mFixedCanId;

        private const string FixedCanAddressExplanation = "This protocol type has a fixed CAN ID, which is influenced by the source and target addresses and the priority used for the communication." +
                                                          "Because the CAN IDs are always the same, they can not be added as mapping." +
                                                          "The addresses and priority can be chosen anew for every service send in the \"Service\" tab.\n";
        private const string TwoDigitAddresses = "XX and YY are the target and source addresses, respectivly.";
        private const string ThreeDigitAddresses = "XXX and YYY are the target and source addresses, respectivly.";

        public FormMapping(uint source)
        {
            mFixedCanId = false;
            mSourceAddress = source;
            MappingInformation = new MappingInformation();

            InitializeComponent();

            textBoxLocalAddress.Text = mSourceAddress.ToString("X");

            foreach (TPUDSProtocol protocol in Enum.GetValues(typeof(TPUDSProtocol)))
            {
                if (!UdsExampleUtils.ProtocolIsDetachedFromIdsAndAddresses(protocol))
                    comboBoxProtocol.Items.Add(new ComboBoxItem(Enum.GetName(typeof(TPUDSProtocol), protocol), protocol));
            }

            comboBoxProtocol.Items.Add(new ComboBoxItem("----- undefinable mappings -----", null));

            foreach (TPUDSProtocol protocol in Enum.GetValues(typeof(TPUDSProtocol)))
            {
                if (UdsExampleUtils.ProtocolIsDetachedFromIdsAndAddresses(protocol))
                    comboBoxProtocol.Items.Add(new ComboBoxItem(Enum.GetName(typeof(TPUDSProtocol), protocol), protocol));
            }


            foreach (TPUDSAddressingType addressingType in Enum.GetValues(typeof(TPUDSAddressingType)))
            {
                comboBoxTargetType.Items.Add(new ComboBoxItem(Enum.GetName(typeof(TPUDSAddressingType), addressingType), addressingType));
            }

            comboBoxProtocol.SelectedIndex = 0;
            comboBoxTargetType.SelectedIndex = 0;
        }

        private void UpdateFixedCanId()
        {
            uint CanId = 0;

            if (comboBoxTargetType.SelectedItem == null)
                return;

            TPUDSAddressingType addressingType = (TPUDSAddressingType)(comboBoxTargetType.SelectedItem as ComboBoxItem).Data;
            TPUDSProtocol protocol = (TPUDSProtocol)(comboBoxProtocol.SelectedItem as ComboBoxItem).Data;

            switch (protocol)
            {
                case TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_29B:
                    if (addressingType == TPUDSAddressingType.PUDS_ADDRESSING_PHYSICAL)
                        CanId = 0x00DA0000;
                    else
                        CanId = 0x00DB0000;
                    break;
                case TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_2_29B_REMOTE:
                    if (addressingType == TPUDSAddressingType.PUDS_ADDRESSING_PHYSICAL)
                        CanId = 0x00CE0000;
                    else
                        CanId = 0x00CD0000;
                    break;
                case TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_3_29B:
                    CanId = 0x03000000;
                    break;
            }


            switch (Decimal.ToInt32(numericUpDownCanId.Value))
            {
                case 0:
                    CanId += 0x00000000;
                    break;
                case 1:
                    CanId += 0x04000000;
                    break;
                case 2:
                    CanId += 0x08000000;
                    break;
                case 3:
                    CanId += 0x0C000000;
                    break;
                case 4:
                    CanId += 0x10000000;
                    break;
                case 5:
                    CanId += 0x14000000;
                    break;
                case 6:
                    CanId += 0x18000000;
                    break;
                case 7:
                    CanId += 0x1C000000;
                    break;
            }

            if (protocol == TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_3_29B)
                labelFixedCanId.Text = CanId.ToString("X8").Replace("000000", "XXXYYY") + "h";
            else
                labelFixedCanId.Text = CanId.ToString("X8").Replace("0000", "XXYY") + "h";
        }

        #region Event Handlers
        /// <summary>
        /// Event handler called when selectin in comboBox "Message Type" is changed.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void comboBoxProtocol_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (comboBoxProtocol.SelectedItem == null)
            {
                DisableNumericUpDowns();
                return;
            }

            if ((comboBoxProtocol.SelectedItem is ComboBoxItem item) && (item.Data == null))
            {
                ChangeUiVisibility(false);
                return;
            }
            else
                ChangeUiVisibility(true);

            TPUDSProtocol protocol = (TPUDSProtocol)(comboBoxProtocol.SelectedItem as ComboBoxItem).Data;

            if (protocol != TPUDSProtocol.PUDS_PROTOCOL_NONE)
            {
                if (!comboBoxTargetType.Enabled)
                {
                    comboBoxTargetType.Enabled = true;
                    numericUpDownComponents_ValueChanged(null, EventArgs.Empty);
                }

                labelCanIdResp.Visible = true;
                numericUpDownCanIdResp.Visible = true;
            }
            else
            {
                if ((comboBoxTargetType.SelectedItem == null) || ((TPUDSAddressingType)(comboBoxTargetType.SelectedItem as ComboBoxItem).Data != TPUDSAddressingType.PUDS_ADDRESSING_PHYSICAL))
                {
                    foreach (ComboBoxItem comboboxItem in comboBoxTargetType.Items)
                    {
                        if ((TPUDSAddressingType)comboboxItem.Data == TPUDSAddressingType.PUDS_ADDRESSING_PHYSICAL)
                        {
                            comboBoxTargetType.SelectedItem = comboboxItem;
                            break;
                        }
                    }
                }

                labelCanIdResp.Visible = false;
                numericUpDownCanIdResp.Visible = false;

                comboBoxTargetType.Enabled = false;
            }


            if (UdsExampleUtils.ProtocolIsDetachedFromIdsAndAddresses(protocol))
            {
                buttonOk.Enabled = false;

                if (!mFixedCanId)
                {
                    labelFixedCanId.Visible = true;
                    labelTargetAddr.Visible = false;
                    labelRemoteAddr.Visible = false;
                    labelLocalAddress.Visible = false;
                    textBoxLocalAddress.Visible = false;
                    numericUpDownTargetAddr.Visible = false;
                    numericUpDownRemoteAddr.Visible = false;
                    mFixedCanId = true;
                    numericUpDownCanId.Maximum = 7;
                    numericUpDownCanId.Enabled = true;
                    labelCanId.Text = "Priority";
                    labelCanIdResp.Text = "CAN ID";
                }
                numericUpDownCanIdResp.Visible = false;
                if (protocol == TPUDSProtocol.PUDS_PROTOCOL_ISO_15765_3_29B)
                    labelSrcAddr.Text = FixedCanAddressExplanation + ThreeDigitAddresses;
                else
                    labelSrcAddr.Text = FixedCanAddressExplanation + TwoDigitAddresses;

                UpdateFixedCanId();
                return;
            }
            else if (mFixedCanId)
            {
                labelCanId.Text = "CAN ID (hex):";
                labelCanIdResp.Text = "CAN ID response (hex):";
                labelSrcAddr.Text = "Source address (hex):";
                mFixedCanId = false;
                labelFixedCanId.Visible = false;
                labelSrcAddr.Visible = true;
                labelTargetAddr.Visible = true;
                labelRemoteAddr.Visible = true;
                textBoxLocalAddress.Visible = true;
                labelLocalAddress.Visible = true;
                numericUpDownCanIdResp.Visible = true;
                numericUpDownTargetAddr.Visible = true;
                numericUpDownRemoteAddr.Visible = true;
                numericUpDownComponents_ValueChanged(null, EventArgs.Empty);
            }

            if (!mFixedCanId)
            {
                EnableNumericUpDowns();

                if (UdsExampleUtils.ProtocolIsRemote(protocol))
                    numericUpDownRemoteAddr.Enabled = true;
                else
                {
                    numericUpDownRemoteAddr.Enabled = false;
                    numericUpDownRemoteAddr.Value = 0;
                }

                if (UdsExampleUtils.ProtocolIs29Bit(protocol))
                {
                    numericUpDownCanId.Maximum = 0x1FFFFFFF;
                    numericUpDownCanIdResp.Maximum = 0x1FFFFFFF;
                }
                else
                {
                    numericUpDownCanId.Maximum = 0x7FF;
                    numericUpDownCanIdResp.Maximum = 0x7FF;
                }
            }
        }

        private void ChangeUiVisibility(bool visibility)
        {
            comboBoxTargetType.Enabled = visibility;
            numericUpDownTargetAddr.Enabled = visibility;
            numericUpDownRemoteAddr.Enabled = visibility;
            labelCanIdResp.Enabled = visibility;
            labelCanId.Enabled = visibility;
            numericUpDownCanId.Enabled = visibility;
            numericUpDownCanIdResp.Enabled = visibility;
            labelTargetType.Enabled = visibility;
            labelTargetAddr.Enabled = visibility;
            labelRemoteAddr.Enabled = visibility;
            buttonOk.Enabled = visibility;
            labelSrcAddr.Enabled = visibility;
            textBoxLocalAddress.Enabled = visibility;
            labelLocalAddress.Enabled = visibility;
            label2.Enabled = visibility;
            radioButtonSending.Enabled = visibility;
            radioButtonReceiving.Enabled = visibility;
            labelFixedCanId.Enabled = visibility;
        }



        /// <summary>
        /// Event handler called when selectin in comboBox "Target Addressing Type" is changed.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void comboBoxTargetType_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (comboBoxTargetType.SelectedItem == null)
            {
                DisableNumericUpDowns();
                return;
            }
            EnableNumericUpDowns();

            TPUDSAddressingType targetType = (TPUDSAddressingType)(comboBoxTargetType.SelectedItem as ComboBoxItem).Data;

            if (mFixedCanId)
            {
                UpdateFixedCanId();
                return;
            }


            numericUpDownComponents_ValueChanged(null, EventArgs.Empty);

            if (targetType == TPUDSAddressingType.PUDS_ADDRESSING_FUNCTIONAL)
            {
                numericUpDownCanIdResp.Enabled = false;
                numericUpDownCanIdResp.Maximum = 0xFFFFFFFF;
                numericUpDownCanIdResp.Value = 0xFFFFFFFF;
                radioButtonSending.Checked = true;
                radioButtonReceiving.Enabled = false;
                radioButtonSending.Enabled = false;
            }
            else
            {
                radioButtonReceiving.Enabled = true;
                radioButtonSending.Enabled = true;
            }
            comboBoxProtocol_SelectedIndexChanged(null, EventArgs.Empty);
        }

        private void DisableNumericUpDowns()
        {
            numericUpDownCanId.Enabled = false;
            numericUpDownCanIdResp.Enabled = false;
            numericUpDownTargetAddr.Enabled = false;
            numericUpDownRemoteAddr.Enabled = false;
        }
        private void EnableNumericUpDowns()
        {
            numericUpDownCanId.Enabled = true;
            numericUpDownCanIdResp.Enabled = true;
            numericUpDownTargetAddr.Enabled = true;
            numericUpDownRemoteAddr.Enabled = true;
        }

        /// <summary>
        /// Event handler called when value is changed in various numericUpDown components.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void numericUpDownComponents_ValueChanged(object sender, EventArgs e)
        {
            if (mFixedCanId)
            {
                UpdateFixedCanId();
            }
            else
            {
                if ((numericUpDownCanId.Value != numericUpDownCanIdResp.Value) && (mSourceAddress != numericUpDownTargetAddr.Value))
                    buttonOk.Enabled = true;
                else
                    buttonOk.Enabled = false;
            }
        }

        private void AssignMappingsValuesToDataStructure()
        {
            MappingInformation.CanId = (uint)numericUpDownCanId.Value;
            MappingInformation.ResponseCanId = (uint)numericUpDownCanIdResp.Value;
            if (radioButtonSending.Checked)
            {
                MappingInformation.SourceAddress = (byte)mSourceAddress;
                MappingInformation.TargetAddress = (byte)numericUpDownTargetAddr.Value;
            }
            else
            {
                MappingInformation.TargetAddress = (byte)mSourceAddress;
                MappingInformation.SourceAddress = (byte)numericUpDownTargetAddr.Value;
            }
            MappingInformation.RemoteAddress = (byte)numericUpDownRemoteAddr.Value;
            if (comboBoxProtocol.SelectedItem != null)
                MappingInformation.Protocol = (TPUDSProtocol)(comboBoxProtocol.SelectedItem as ComboBoxItem).Data;
            if (comboBoxTargetType.SelectedItem != null)
                MappingInformation.TargetType = (TPUDSAddressingType)(comboBoxTargetType.SelectedItem as ComboBoxItem).Data;
        }

        private void radioButtonSending_CheckedChanged(object sender, EventArgs e)
        {
            if (radioButtonSending.Checked)
            {
                textBoxLocalAddress.Location = new Point(textBoxLocalAddress.Location.X, 136);
                labelLocalAddress.Location = new Point(labelLocalAddress.Location.X, 139);

                numericUpDownTargetAddr.Location = new Point(numericUpDownTargetAddr.Location.X, 163);
            }
            else
            {
                numericUpDownTargetAddr.Location = new Point(numericUpDownTargetAddr.Location.X, 136);

                textBoxLocalAddress.Location = new Point(textBoxLocalAddress.Location.X, 163);
                labelLocalAddress.Location = new Point(labelLocalAddress.Location.X, 166);
            }
        }
        #endregion

        private void buttonOk_Click(object sender, EventArgs e)
        {
            AssignMappingsValuesToDataStructure();
        }

    }
}
