
#include "PCANBasicCLR.h"
#include "PCAN-ISO-TP-CLR_2016.h"
#include "PCAN-UDS-CLR_2013.h"


using namespace System;
using namespace System::IO;
using namespace System::Threading;
using namespace Peak::Can::IsoTp;
using namespace Peak::Can::Uds;

static String^ OK_KO(bool test)
{
	if (test)
		return "OK";
	return "KO";
}

static String^ STATUS_OK_KO(uds_status test)
{
	return OK_KO(UDSApi::StatusIsOk_2013(test));
}

static void process_request(cantp_handle tp_handle, UInt16 server_address, uds_msg request_msg);

/// <summary>Helper: convert cantp network address information into uds network information</summary>
/// <param name="nai">cantp network address information structure</param>
/// <param name="can_msgtype">cantp message type</param>
/// <returns>UDS network address information structure</returns>
static uds_netaddrinfo ConvertNai(cantp_netaddrinfo nai, cantp_can_msgtype can_msgtype)
{
	uds_netaddrinfo result = {};
	bool is_29bits = ((can_msgtype & cantp_can_msgtype::PCANTP_CAN_MSGTYPE_EXTENDED) == cantp_can_msgtype::PCANTP_CAN_MSGTYPE_EXTENDED);

	result.source_addr = nai.source_addr;
	result.target_addr = nai.target_addr;
	result.target_type = nai.target_type;
	result.extension_addr = nai.extension_addr;

	switch (nai.format)
	{
	case cantp_isotp_format::PCANTP_ISOTP_FORMAT_ENHANCED:
		result.protocol = uds_msgprotocol::PUDS_MSGPROTOCOL_ISO_15765_3_29B_ENHANCED;
		break;
	case cantp_isotp_format::PCANTP_ISOTP_FORMAT_EXTENDED:
		result.protocol = is_29bits ? uds_msgprotocol::PUDS_MSGPROTOCOL_ISO_15765_2_29B_EXTENDED : uds_msgprotocol::PUDS_MSGPROTOCOL_ISO_15765_2_11B_EXTENDED;
		break;
	case cantp_isotp_format::PCANTP_ISOTP_FORMAT_FIXED_NORMAL:
		result.protocol = uds_msgprotocol::PUDS_MSGPROTOCOL_ISO_15765_2_29B_FIXED_NORMAL;
		break;
	case cantp_isotp_format::PCANTP_ISOTP_FORMAT_MIXED:
		result.protocol = is_29bits ? uds_msgprotocol::PUDS_MSGPROTOCOL_ISO_15765_2_29B_REMOTE : uds_msgprotocol::PUDS_MSGPROTOCOL_ISO_15765_2_11B_REMOTE;
		break;
	case cantp_isotp_format::PCANTP_ISOTP_FORMAT_NORMAL:
		result.protocol = is_29bits ? uds_msgprotocol::PUDS_MSGPROTOCOL_ISO_15765_2_29B_NORMAL : uds_msgprotocol::PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL;
		break;
	case cantp_isotp_format::PCANTP_ISOTP_FORMAT_NONE:
	case cantp_isotp_format::PCANTP_ISOTP_FORMAT_UNKNOWN:
	default:
		result.protocol = uds_msgprotocol::PUDS_MSGPROTOCOL_NONE;
		break;
	}
	return result;
}

/// <summary>Helper: check if a service has a subfunction</summary>
/// <param name="service_identifier">service identifier</param>
/// <returns>boolean: yes (true) or not (false)</returns>
static bool HasSubFunction(Byte service_identifier)
{
	bool res;
	switch (service_identifier)
	{
	case (Byte)uds_service::PUDS_SERVICE_SI_DiagnosticSessionControl:
		res = true;
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_ECUReset:
		res = true;
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_SecurityAccess:
		res = true;
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_CommunicationControl:
		res = true;
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_TesterPresent:
		res = true;
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_AccessTimingParameter:
		res = true;
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_SecuredDataTransmission:
		res = false;
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_ControlDTCSetting:
		res = true;
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_ResponseOnEvent:
		res = true;
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_LinkControl:
		res = true;
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_ReadDataByIdentifier:
		res = false;
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_ReadMemoryByAddress:
		res = false;
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_ReadScalingDataByIdentifier:
		res = false;
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_ReadDataByPeriodicIdentifier:
		res = false;
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_DynamicallyDefineDataIdentifier:
		res = true;
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_WriteDataByIdentifier:
		res = false;
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_WriteMemoryByAddress:
		res = false;
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_ClearDiagnosticInformation:
		res = false;
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_ReadDTCInformation:
		res = true;
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_InputOutputControlByIdentifier:
		res = false;
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_RoutineControl:
		res = true;
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_RequestDownload:
		res = false;
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_RequestUpload:
		res = false;
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_TransferData:
		res = false;
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_RequestTransferExit:
		res = false;
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_RequestFileTransfer:
		res = false;
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_Authentication:
		res = true;
		break;
	default:
		res = false;
		break;
	}
	return res;
}

/// <summary>
/// Create the ECUReset response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing ECUReset parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
static uds_status create_SvcECUReset_response(uds_msgconfig config, uds_msg* request_msg, uds_msg* response_msg)
{
	uds_status status;

	// Allocate response message

	if (request_msg->links.param[0] == (Byte)UDSApi::uds_svc_param_er::PUDS_SVC_PARAM_ER_ERPSD)
	{
		status = UDSApi::MsgAlloc_2013(*response_msg, config, 3);
		response_msg->links.param[1] = 0x66;    // Power down time
	}
	else
	{
		status = UDSApi::MsgAlloc_2013(*response_msg, config, 2);
	}
	Console::WriteLine("Allocate ECUReset response: {0}", STATUS_OK_KO(status));

	if (UDSApi::StatusIsOk_2013(status))
	{
		// Fill parameters
		*response_msg->links.service_id = (Byte)(uds_service::PUDS_SERVICE_SI_ECUReset + UDSApi::PUDS_SI_POSITIVE_RESPONSE);
		response_msg->links.param[0] = request_msg->links.param[0];
	}

	return status;
}

/// <summary>
/// Create the WriteMemoryByAddress response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
static uds_status create_SvcWriteMemoryByAddress_response(uds_msgconfig config, uds_msg* request_msg, uds_msg* response_msg)
{
	uds_status status;
	UInt32 memory_size_size;
	UInt32 memory_address_size;
	UInt32 length;

	// Read memory_size_size & memory_address_size
	memory_size_size = (request_msg->links.param[0] >> 4) & 0xF;
	memory_address_size = request_msg->links.param[0] & 0xF;
	length = 2 + memory_size_size + memory_address_size;

	// Allocate message
	status = UDSApi::MsgAlloc_2013(*response_msg, config, length);
	Console::WriteLine("Allocate WriteMemoryByAddress response: {0}", STATUS_OK_KO(status));

	if (UDSApi::StatusIsOk_2013(status))
	{
		// Copy address and memory parameters
		response_msg->links.param[0] = request_msg->links.param[0];
		for (UInt32 i = 0; i < memory_address_size; i++)
		{
			response_msg->links.param[1 + i] = request_msg->links.param[1 + i];
		}
		for (UInt32 i = 0; i < memory_size_size; i++)
		{
			response_msg->links.param[1 + memory_address_size + i] = request_msg->links.param[1 + memory_address_size + i];
		}


		*response_msg->links.service_id = (Byte)(uds_service::PUDS_SERVICE_SI_WriteMemoryByAddress + UDSApi::PUDS_SI_POSITIVE_RESPONSE);
	}

	return status;
}

/// <summary>
/// Create the DynamicallyDefineDataIdentifier response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
static uds_status create_SvcDynamicallyDefineDataIdentifier_response(uds_msgconfig config, uds_msg* request_msg, uds_msg* response_msg)
{
	uds_status status;

	// Allocate and fill message
	if (request_msg->msg.msgdata.any->length >= 4) {
		status = UDSApi::MsgAlloc_2013(*response_msg, config, 4);
		Console::WriteLine("Allocate DynamicallyDefineDataIdentifier response: {0}", STATUS_OK_KO(status));

		if (UDSApi::StatusIsOk_2013(status))
		{
			// Copy data
			response_msg->links.param[0] = request_msg->links.param[0];
			response_msg->links.param[1] = request_msg->links.param[1];
			response_msg->links.param[2] = request_msg->links.param[2];
		}
	}
	else {
		// Clear all data identifier request
		status = UDSApi::MsgAlloc_2013(*response_msg, config, 2);
		Console::WriteLine("Allocate DynamicallyDefineDataIdentifier response: {0}", STATUS_OK_KO(status));

		if (UDSApi::StatusIsOk_2013(status))
		{
			// Copy subfunction
			response_msg->links.param[0] = request_msg->links.param[0];
		}
	}

	if (UDSApi::StatusIsOk_2013(status))
	{
		*response_msg->links.service_id = (Byte)(uds_service::PUDS_SERVICE_SI_DynamicallyDefineDataIdentifier + UDSApi::PUDS_SI_POSITIVE_RESPONSE);
	}
	return status;
}

/// <summary>
/// Create the dummy response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
static uds_status create_dummy_response(uds_msgconfig config, uds_msg* request_msg, uds_msg* response_msg)
{
	uds_status status;
	UInt32 length;

	Byte service_id = *request_msg->links.service_id;

	// Set a random message size
	Random^ rand = gcnew Random();
	length = (UInt32)1 + service_id + rand->Next(0, 200);
	length = length > CanTpApi::PCANTP_MAX_LENGTH_ISOTP2004 ? CanTpApi::PCANTP_MAX_LENGTH_ISOTP2004 : length;

	// Allocate response message
	status = UDSApi::MsgAlloc_2013(*response_msg, config, length);
	Console::WriteLine("Allocate dummy response: {0}", STATUS_OK_KO(status));

	if (UDSApi::StatusIsOk_2013(status))
	{
		// Fake a positive response
		*response_msg->links.service_id = (Byte)(service_id + UDSApi::PUDS_SI_POSITIVE_RESPONSE);

		// Fill with dummy data
		response_msg->links.param[0] = service_id;
		for (UInt32 i = 1; i < length - 1; i++)   // (length - 1) as positive response SID uses 1 byte
			response_msg->links.param[i] = (Byte)(i + 1);

		// Do not set PUDS_FLAG_SUPPRESS_POSITIVE_RESPONSE flag when service has a subfunction
		if (HasSubFunction(service_id))
		{
			response_msg->msg.msgdata.any->data[1] = 0;
		}
	}

	return status;
}

/// <summary>
/// Create the RequestTransferExit response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
static uds_status create_SvcRequestTransferExit_response(uds_msgconfig config, uds_msg* request_msg, uds_msg* response_msg)
{
	uds_status status;
	UInt32 length;

	// Allocate message
	Random^ rand = gcnew Random();
	length = 1 + 1 + rand->Next(0, 50);
	status = UDSApi::MsgAlloc_2013(*response_msg, config, length);
	Console::WriteLine("Allocate RequestTransferExit response: {0}", STATUS_OK_KO(status));

	if (UDSApi::StatusIsOk_2013(status))
	{
		// Fill with dummy data
		for (UInt32 i = 0; i < length - 1; i++)   // (length - 1) as positive response SID uses 1 byte
			response_msg->links.param[i] = (Byte)(i + 1);
		*response_msg->links.service_id = (Byte)(uds_service::PUDS_SERVICE_SI_RequestTransferExit + UDSApi::PUDS_SI_POSITIVE_RESPONSE);
	}

	return status;
}

/// <summary>
/// Create the SvcRequestUpload response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
static uds_status create_SvcRequestUpload_response(uds_msgconfig config, uds_msg* request_msg, uds_msg* response_msg)
{
	uds_status status;
	UInt32 length;

	// Allocate message
	Random^ rand = gcnew Random();
	length = 2 + 1 + rand->Next(0, 50);
	status = UDSApi::MsgAlloc_2013(*response_msg, config, length);
	Console::WriteLine("Allocate RequestUpload response: {0}", STATUS_OK_KO(status));

	if (UDSApi::StatusIsOk_2013(status))
	{
		// Max number of block length = 0xF
		response_msg->links.param[0] = 0xF0;

		// Fill with dummy data
		for (UInt32 i = 1; i < length - 1; i++)   // (length - 1) as positive response SID uses 1 byte
			response_msg->links.param[i] = (Byte)(i + 1);

		*response_msg->links.service_id = (Byte)(uds_service::PUDS_SERVICE_SI_RequestUpload + UDSApi::PUDS_SI_POSITIVE_RESPONSE);
	}

	return status;
}

/// <summary>
/// Create the SvcRequestDownload response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
static uds_status create_SvcRequestDownload_response(uds_msgconfig config, uds_msg* request_msg, uds_msg* response_msg)
{
	uds_status status;
	UInt32 length;

	// Allocate message
	Random^ rand = gcnew Random();
	length = 2 + 1 + rand->Next(0, 50);
	status = UDSApi::MsgAlloc_2013(*response_msg, config, length);
	Console::WriteLine("Allocate RequestDownload response: {0}", STATUS_OK_KO(status));

	if (UDSApi::StatusIsOk_2013(status))
	{
		// Max number of block length = 0xF
		response_msg->links.param[0] = 0xF0;

		// Fill with dummy data
		for (UInt32 i = 1; i < length - 1; i++)   // (length - 1) as positive response SID uses 1 byte
			response_msg->links.param[i] = (Byte)(i + 1);

		*response_msg->links.service_id = (Byte)(uds_service::PUDS_SERVICE_SI_RequestDownload + UDSApi::PUDS_SI_POSITIVE_RESPONSE);
	}

	return status;
}

/// <summary>
/// Create the RoutineControl response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
static uds_status create_SvcRoutineControl_response(uds_msgconfig config, uds_msg* request_msg, uds_msg* response_msg)
{
	uds_status status;

	// Allocate response
	status = UDSApi::MsgAlloc_2013(*response_msg, config, 4);
	Console::WriteLine("Allocate RoutineControl response: {0}", STATUS_OK_KO(status));

	if (UDSApi::StatusIsOk_2013(status))
	{
		// Copy data
		response_msg->links.param[0] = request_msg->links.param[0];
		response_msg->links.param[1] = request_msg->links.param[1];
		response_msg->links.param[2] = request_msg->links.param[2];

		// Routine status record not implemented

		*response_msg->links.service_id = (Byte)(uds_service::PUDS_SERVICE_SI_RoutineControl + UDSApi::PUDS_SI_POSITIVE_RESPONSE);
	}

	return status;
}

/// <summary>
/// Create the InputOutputControlByIdentifier response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
static uds_status create_SvcInputOutputControlByIdentifier_response(uds_msgconfig config, uds_msg* request_msg, uds_msg* response_msg)
{
	uds_status status;

	status = UDSApi::MsgAlloc_2013(*response_msg, config, 3);
	Console::WriteLine("Allocate InputOutputControlByIdentifier response: {0}", STATUS_OK_KO(status));

	if (UDSApi::StatusIsOk_2013(status))
	{
		// Copy data
		response_msg->links.param[0] = request_msg->links.param[0];
		response_msg->links.param[1] = request_msg->links.param[1];

		// Control status record not implemented
		*response_msg->links.service_id = (Byte)(uds_service::PUDS_SERVICE_SI_InputOutputControlByIdentifier + UDSApi::PUDS_SI_POSITIVE_RESPONSE);
	}

	return status;
}


/// <summary>
/// Create the ClearDiagnosticInformation response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
static uds_status create_SvcClearDiagnosticInformation_response(uds_msgconfig config, uds_msg* request_msg, uds_msg* response_msg)
{
	uds_status status;

	status = UDSApi::MsgAlloc_2013(*response_msg, config, 1);
	Console::WriteLine("Allocate ClearDiagnosticInformation response: {0}", STATUS_OK_KO(status));

	if (UDSApi::StatusIsOk_2013(status))
	{
		*response_msg->links.service_id = (Byte)(uds_service::PUDS_SERVICE_SI_ClearDiagnosticInformation + UDSApi::PUDS_SI_POSITIVE_RESPONSE);
	}

	return status;
}

/// <summary>
/// Create the WriteDataByIdentifier response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
static uds_status create_SvcWriteDataByIdentifier_response(uds_msgconfig config, uds_msg* request_msg, uds_msg* response_msg)
{
	uds_status status;

	status = UDSApi::MsgAlloc_2013(*response_msg, config, 3);
	Console::WriteLine("Allocate WriteDataByIdentifier response: {0}", STATUS_OK_KO(status));

	if (UDSApi::StatusIsOk_2013(status))
	{
		// Copy data
		response_msg->links.param[0] = request_msg->links.param[0];
		response_msg->links.param[1] = request_msg->links.param[1];
		*response_msg->links.service_id = (Byte)(uds_service::PUDS_SERVICE_SI_WriteDataByIdentifier + UDSApi::PUDS_SI_POSITIVE_RESPONSE);
	}

	return status;
}

/// <summary>
/// Create the ReadDataByPeriodicIdentifier response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
static uds_status create_SvcReadDataByPeriodicIdentifier_response(uds_msgconfig config, uds_msg* request_msg, uds_msg* response_msg)
{
	uds_status status;
	status = UDSApi::MsgAlloc_2013(*response_msg, config, 1);
	Console::WriteLine("Allocate ReadDataByPeriodicIdentifier response: {0}", STATUS_OK_KO(status));

	if (UDSApi::StatusIsOk_2013(status))
	{
		*response_msg->links.service_id = (Byte)uds_service::PUDS_SERVICE_SI_ReadDataByPeriodicIdentifier + UDSApi::PUDS_SI_POSITIVE_RESPONSE;
	}
	return status;
}

/// <summary>
/// Create the ReadMemoryByAddress response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
static uds_status create_SvcReadMemoryByAddress_response(uds_msgconfig config, uds_msg* request_msg, uds_msg* response_msg)
{
	uds_status status;
	UInt32 length;

	// Read memory_size_size = bits [7..4]
	length = 1 + ((request_msg->links.param[1] >> 4) & 0xF);

	// Allocate message
	status = UDSApi::MsgAlloc_2013(*response_msg, config, length);
	Console::WriteLine("Allocate ReadMemoryByAddress response: {0}", STATUS_OK_KO(status));

	if (UDSApi::StatusIsOk_2013(status))
	{
		// Fill with dummy data
		for (UInt32 i = 0; i < length - 1; i++)   // (length - 1) as positive response SID uses 1 byte
			response_msg->links.param[i] = i + 1;
		*response_msg->links.service_id = (Byte)uds_service::PUDS_SERVICE_SI_ReadMemoryByAddress + UDSApi::PUDS_SI_POSITIVE_RESPONSE;
	}

	return status;
}

/// <summary>
/// Create the ReadScalingDataByIdentifier response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
static uds_status create_SvcReadScalingDataByIdentifier_response(uds_msgconfig config, uds_msg* request_msg, uds_msg* response_msg)
{
	uds_status status;

	// Allocate message
	status = UDSApi::MsgAlloc_2013(*response_msg, config, 12);
	Console::WriteLine("Allocate ReadScalingDataByIdentifier response: {0}", STATUS_OK_KO(status));

	if (UDSApi::StatusIsOk_2013(status))
	{
		*response_msg->links.service_id = (Byte)(uds_service::PUDS_SERVICE_SI_ReadScalingDataByIdentifier + UDSApi::PUDS_SI_POSITIVE_RESPONSE);

		// Copy data
		response_msg->links.param[0] = request_msg->links.param[0];
		response_msg->links.param[1] = request_msg->links.param[1];

		// Create a formula Vehicule Speed = (0.75*x+30) km/h
		response_msg->links.param[2] = (0x0 << 4) | (0x1);	// unsigned numeric of 1 Bytes)
		response_msg->links.param[3] = 0x90;	// formula, 0 data bytes
		response_msg->links.param[4] = 0x00;	// formulaIdentifier = C0 * x + C1
		response_msg->links.param[5] = 0xE0;	// C0 high byte
		response_msg->links.param[6] = 0x4B;	// C0 low byte
		response_msg->links.param[7] = 0x00;	// C1 high byte
		response_msg->links.param[8] = 0x1E;	// C1 low byte
		response_msg->links.param[9] = 0xA0;	// unit/format, 0 data bytes
		response_msg->links.param[10] = 0x30;	// unit ID, km/h
	}

	return status;
}

/// <summary>
/// Create the ReadDataByIdentifier response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
static uds_status create_SvcReadDataByIdentifier_response(uds_msgconfig config, uds_msg* request_msg, uds_msg* response_msg)
{
	uds_status status;
	UInt32 length;
	int counter;

	// Allocate message
	length = ((request_msg->msg.msgdata.any->length - 1) * 7 / 2) + 1;
	status = UDSApi::MsgAlloc_2013(*response_msg, config, length);
	Console::WriteLine("Allocate ReadDataByIdentifier response: {0}", STATUS_OK_KO(status));

	if (UDSApi::StatusIsOk_2013(status))
	{
		counter = 0;
		for (UInt32 i = 0; i < request_msg->msg.msgdata.any->length - 1; i += 2)
		{
			// Copy data
			response_msg->links.param[counter++] = request_msg->links.param[i];
			response_msg->links.param[counter++] = request_msg->links.param[i + 1];
			// Data record: fill with dummy data
			for (int j = 0; j < 5; j++)
			{
				response_msg->links.param[counter++] = (Byte)(j + 'A');
			}
		}
		*response_msg->links.service_id = (Byte)uds_service::PUDS_SERVICE_SI_ReadDataByIdentifier + UDSApi::PUDS_SI_POSITIVE_RESPONSE;
	}

	return status;
}

/// <summary>
/// Create the LinkControl response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
static uds_status create_SvcLinkControl_response(uds_msgconfig config, uds_msg* request_msg, uds_msg* response_msg)
{
	uds_status status;
	status = UDSApi::MsgAlloc_2013(*response_msg, config, 2);
	Console::WriteLine("Allocate LinkControl response: {0}", STATUS_OK_KO(status));

	if (UDSApi::StatusIsOk_2013(status))
	{
		response_msg->links.param[0] = request_msg->links.param[0];
		*response_msg->links.service_id = (Byte)(uds_service::PUDS_SERVICE_SI_LinkControl + UDSApi::PUDS_SI_POSITIVE_RESPONSE);
	}

	return status;
}

/// <summary>
/// Create the ResponseOnEvent response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
static uds_status create_SvcResponseOnEvent_response(uds_msgconfig config, uds_msg* request_msg, uds_msg* response_msg)
{
	uds_status status;


	if (request_msg->links.param[0] == (Byte)UDSApi::uds_svc_param_roe::PUDS_SVC_PARAM_ROE_RAE)
	{
		status = UDSApi::MsgAlloc_2013(*response_msg, config, 3);
		Console::WriteLine("Allocate ResponseOnEvent response: {0}", STATUS_OK_KO(status));

		if (UDSApi::StatusIsOk_2013(status))
		{
			response_msg->links.param[1] = 0;   // Number of activated events
																	// Event type and service to respond to records not implemented
		}
	}
	else
	{
		status = UDSApi::MsgAlloc_2013(*response_msg, config, 4);
		Console::WriteLine("Allocate ResponseOnEvent response: {0}", STATUS_OK_KO(status));

		if (UDSApi::StatusIsOk_2013(status))
		{
			response_msg->links.param[1] = 0;  // Number of identified events
			response_msg->links.param[2] = request_msg->links.param[1];	// Event window time
																		// Event type and service to respond to records not implemented
		}
	}

	if (UDSApi::StatusIsOk_2013(status))
	{
		response_msg->links.param[0] = request_msg->links.param[0];
		*response_msg->links.service_id = (Byte)(uds_service::PUDS_SERVICE_SI_ResponseOnEvent + UDSApi::PUDS_SI_POSITIVE_RESPONSE);
	}

	return status;
}

/// <summary>
/// Create the ControlDTCSetting response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
static uds_status create_SvcControlDTCSetting_response(uds_msgconfig config, uds_msg* request_msg, uds_msg* response_msg)
{
	uds_status status;
	status = UDSApi::MsgAlloc_2013(*response_msg, config, 2);
	Console::WriteLine("Allocate ControlDTCSetting response: {0}", STATUS_OK_KO(status));

	if (UDSApi::StatusIsOk_2013(status))
	{
		response_msg->links.param[0] = request_msg->links.param[0];
		*response_msg->links.service_id = (Byte)(uds_service::PUDS_SERVICE_SI_ControlDTCSetting + UDSApi::PUDS_SI_POSITIVE_RESPONSE);
	}

	return status;
}

/// <summary>
/// Create the SecuredDataTransmission response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
static uds_status create_SvcSecuredDataTransmission_response(uds_msgconfig config, uds_msg* request_msg, uds_msg* response_msg)
{
	uds_status status;
	UInt32 length;

	// Allocate message
	Random^ rand = gcnew Random();
	length = 1 + rand->Next(0, 50);
	status = UDSApi::MsgAlloc_2013(*response_msg, config, length);
	Console::WriteLine("Allocate SecuredDataTransmission response: {0}", STATUS_OK_KO(status));

	if (UDSApi::StatusIsOk_2013(status))
	{
		// Fill with dummy data (check Security-SubLayer record defined in ISO-15764)
		for (UInt32 i = 0; i < length - 1; i++)   // (length - 1) as positive response SID uses 1 byte
			response_msg->links.param[i] = (Byte)(i + 1);

		*response_msg->links.service_id = (Byte)(uds_service::PUDS_SERVICE_SI_SecuredDataTransmission + UDSApi::PUDS_SI_POSITIVE_RESPONSE);
	}

	return status;
}

/// <summary>
/// Create the TesterPresent response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
static uds_status create_SvcTesterPresent_response(uds_msgconfig config, uds_msg* request_msg, uds_msg* response_msg)
{
	uds_status status;
	status = UDSApi::MsgAlloc_2013(*response_msg, config, 2);
	Console::WriteLine("Allocate TesterPresent response: {0}", STATUS_OK_KO(status));

	if (UDSApi::StatusIsOk_2013(status))
	{
		response_msg->links.param[0] = request_msg->links.param[0];
		*response_msg->links.service_id = (Byte)(uds_service::PUDS_SERVICE_SI_TesterPresent + UDSApi::PUDS_SI_POSITIVE_RESPONSE);
	}

	return status;
}

/// <summary>
/// Create the CommunicationControl response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
static uds_status create_SvcCommunicationControl_response(uds_msgconfig config, uds_msg* request_msg, uds_msg* response_msg)
{
	uds_status status;
	status = UDSApi::MsgAlloc_2013(*response_msg, config, 2);
	Console::WriteLine("Allocate CommunicationControl response: {0}", STATUS_OK_KO(status));

	if (UDSApi::StatusIsOk_2013(status))
	{
		response_msg->links.param[0] = request_msg->links.param[0];
		*response_msg->links.service_id = (Byte)(uds_service::PUDS_SERVICE_SI_CommunicationControl + UDSApi::PUDS_SI_POSITIVE_RESPONSE);
	}

	return status;
}

/// <summary>
/// Create the SecurityAccess response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
static uds_status create_SvcSecurityAccess_response(uds_msgconfig config, uds_msg* request_msg, uds_msg* response_msg)
{
	uds_status status;
	UInt32 length;
	UInt32 i;

	// Allocate response message

	if (request_msg->links.param[0] >= UDSApi::PUDS_SVC_PARAM_SA_RSD_MIN
		&& request_msg->links.param[0] <= UDSApi::PUDS_SVC_PARAM_SA_RSD_MAX
		&& request_msg->links.param[0] % 2 == 1)
	{

		// Request security seed are Even values
		// Fill with dummy data
		length = (UInt32)(1 + uds_service::PUDS_SERVICE_SI_SecurityAccess) > (UInt32)CanTpApi::PCANTP_MAX_LENGTH_ISOTP2004 ? (UInt32)CanTpApi::PCANTP_MAX_LENGTH_ISOTP2004 : (UInt32)(1 + uds_service::PUDS_SERVICE_SI_SecurityAccess);
		status = UDSApi::MsgAlloc_2013(*response_msg, config, length);
		Console::WriteLine("Allocate SecurityAccess response: {0}", STATUS_OK_KO(status));

		if (UDSApi::StatusIsOk_2013(status))
		{
			for (i = 1; i < length - 1; i++)    // (length - 1) as positive response SID uses 1 byte
				response_msg->links.param[i] = (Byte)(i + 1);
		}
	}
	else
	{
		status = UDSApi::MsgAlloc_2013(*response_msg, config, 2);
		Console::WriteLine("Allocate SecurityAccess response: {0}", STATUS_OK_KO(status));
	}

	if (UDSApi::StatusIsOk_2013(status))
	{
		response_msg->links.param[0] = request_msg->links.param[0];
		*response_msg->links.service_id = (Byte)(uds_service::PUDS_SERVICE_SI_SecurityAccess + UDSApi::PUDS_SI_POSITIVE_RESPONSE);
	}

	return status;
}


/// <summary>
/// Create the DiagnosticSessionControl response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
static uds_status create_SvcDiagnosticSessionControl_response(uds_msgconfig config, uds_msg* request_msg, uds_msg* response_msg)
{
	uds_status status;

	// Allocate response message
	status = UDSApi::MsgAlloc_2013(*response_msg, config, 6);
	Console::WriteLine("Allocate DiagnosticSessionControl response: {0}", STATUS_OK_KO(status));

	if (UDSApi::StatusIsOk_2013(status))
	{
		// Fill response
		response_msg->links.param[0] = request_msg->links.param[0];
		response_msg->links.param[1] = 0x00;	// P2Can_Server_Max = 0x0010
		response_msg->links.param[2] = 0x10;
		response_msg->links.param[3] = 0x03;	// P2*Can_Server_Max = 0x03E8
		response_msg->links.param[4] = 0xE8;
		*response_msg->links.service_id = (Byte)(uds_service::PUDS_SERVICE_SI_DiagnosticSessionControl + UDSApi::PUDS_SI_POSITIVE_RESPONSE);
	}

	return status;
}

/// <summary>
/// Create the TransferData response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
static uds_status create_SvcTransferData_response(cantp_handle tp_handle, uds_msgconfig config, uds_msg* request_msg, uds_msg* response_msg)
{
	uds_status status;
	UInt32 length;
	uds_msg pending_response_msg = {};

	// Initialize locale struct

	// Custom response to client_all_request example:
	//  a. service is requested functionally,
	//  b. 1st response is NRC response pending
	//  c. wait
	//  d. send correct response
	if (request_msg->msg.msgdata.isotp->netaddrinfo.target_type == cantp_isotp_addressing::PCANTP_ISOTP_ADDRESSING_FUNCTIONAL)
	{

		// Transmit a NRC response pending
		status = UDSApi::MsgAlloc_2013(pending_response_msg, config, 3);
		Console::WriteLine("Allocate TransferData pending response: {0}", STATUS_OK_KO(status));


		// Transmit a NRC response pending
		if (UDSApi::StatusIsOk_2013(status))
		{
			*pending_response_msg.links.service_id = (Byte)uds_service::PUDS_SERVICE_NR_SI;
			pending_response_msg.links.param[0] = (Byte)uds_service::PUDS_SERVICE_SI_TransferData;
			pending_response_msg.links.param[1] = UDSApi::PUDS_NRC_EXTENDED_TIMING;
		}
		status = UDSApi::Write_2013(tp_handle, pending_response_msg);

		Console::WriteLine(); Console::WriteLine("   ...Transmitting a NRC Response Pending message: {0}", (int)status);
		Console::WriteLine(); Console::Write("   ...simulating computation... (waiting ~{0}ms)", (int)UDSApi::PUDS_P2CAN_ENHANCED_SERVER_MAX_DEFAULT);
		Thread::Sleep((int)UDSApi::PUDS_P2CAN_ENHANCED_SERVER_MAX_DEFAULT - 100);

		// Reallocate response message
		status = UDSApi::MsgFree_2013(pending_response_msg);
		Console::WriteLine("Free pending response: {0}", STATUS_OK_KO(status));

		// Initialize real service response
		config.type = uds_msgtype::PUDS_MSGTYPE_USDT;
		length = CanTpApi::PCANTP_MAX_LENGTH_ISOTP2004;
		status = UDSApi::MsgAlloc_2013(*response_msg, config, length);
		Console::WriteLine("Allocate TransferData real response: {0}", STATUS_OK_KO(status));

	}
	else
	{
		length = (2 + request_msg->msg.msgdata.isotp->length) > CanTpApi::PCANTP_MAX_LENGTH_ISOTP2004 ? CanTpApi::PCANTP_MAX_LENGTH_ISOTP2004 : 2 + request_msg->msg.msgdata.isotp->length;
		status = UDSApi::MsgAlloc_2013(*response_msg, config, length);
		Console::WriteLine("Allocate TransferData response: {0}", STATUS_OK_KO(status));
	}

	if (UDSApi::StatusIsOk_2013(status))
	{
		response_msg->links.param[0] = request_msg->links.param[0];

		// Fill with dummy data
		for (UInt32 i = 1; i < length - 1; i++)   // (length - 1) as positive response SID uses 1 byte
			response_msg->links.param[i] = i + 1;
		*response_msg->links.service_id = (Byte)uds_service::PUDS_SERVICE_SI_TransferData + UDSApi::PUDS_SI_POSITIVE_RESPONSE;
	}

	return status;
}

/// <summary>
///	Process request and send a response
/// </summary>
/// <param name="tp_handle">CANTP channel</param>
/// <param name="server_address">Server address</param>
/// <param name="request_msg">Received request message</param>
static void process_request(cantp_handle tp_handle, UInt16 server_address, uds_msg request_msg)
{
	uds_msgconfig config = {};
	uds_msg response_msg = {};
	uds_status status = uds_status::PUDS_STATUS_OK;


	// Initialize config from request message network address information

	config.nai = ConvertNai(request_msg.msg.msgdata.isotp->netaddrinfo, request_msg.msg.can_info.can_msgtype);
	config.can_msgtype = request_msg.msg.can_info.can_msgtype;
	config.type = uds_msgtype::PUDS_MSGTYPE_USDT;
	config.nai.extension_addr = request_msg.msg.msgdata.isotp->netaddrinfo.extension_addr;

	// Set target and source addresses
	if (request_msg.msg.msgdata.isotp->netaddrinfo.target_type == cantp_isotp_addressing::PCANTP_ISOTP_ADDRESSING_FUNCTIONAL)
	{
		// response to functional addressing is set to TEST_EQUIPMENT
		config.nai.target_addr = (UInt16)uds_address::PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
	}
	else
	{
		config.nai.target_addr = request_msg.msg.msgdata.isotp->netaddrinfo.source_addr;
	}
	config.nai.target_type = cantp_isotp_addressing::PCANTP_ISOTP_ADDRESSING_PHYSICAL;
	config.nai.source_addr = server_address;

	// This is a valid request, switch services
	Byte service_id = *request_msg.links.service_id;
	switch (service_id)
	{
	case (Byte)uds_service::PUDS_SERVICE_SI_DiagnosticSessionControl:
		status = create_SvcDiagnosticSessionControl_response(config, &request_msg, &response_msg);
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_SecurityAccess:
		status = create_SvcSecurityAccess_response(config, &request_msg, &response_msg);
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_CommunicationControl:
		status = create_SvcCommunicationControl_response(config, &request_msg, &response_msg);
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_TesterPresent:
		status = create_SvcTesterPresent_response(config, &request_msg, &response_msg);
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_SecuredDataTransmission:
		status = create_SvcSecuredDataTransmission_response(config, &request_msg, &response_msg);
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_ControlDTCSetting:
		status = create_SvcControlDTCSetting_response(config, &request_msg, &response_msg);
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_ResponseOnEvent:
		status = create_SvcResponseOnEvent_response(config, &request_msg, &response_msg);
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_LinkControl:
		status = create_SvcLinkControl_response(config, &request_msg, &response_msg);
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_ReadDataByIdentifier:
		status = create_SvcReadDataByIdentifier_response(config, &request_msg, &response_msg);
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_ReadMemoryByAddress:
		status = create_SvcReadMemoryByAddress_response(config, &request_msg, &response_msg);
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_ReadScalingDataByIdentifier:
		status = create_SvcReadScalingDataByIdentifier_response(config, &request_msg, &response_msg);
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_ReadDataByPeriodicIdentifier:
		status = create_SvcReadDataByPeriodicIdentifier_response(config, &request_msg, &response_msg);
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_DynamicallyDefineDataIdentifier:
		status = create_SvcDynamicallyDefineDataIdentifier_response(config, &request_msg, &response_msg);
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_WriteDataByIdentifier:
		status = create_SvcWriteDataByIdentifier_response(config, &request_msg, &response_msg);
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_WriteMemoryByAddress:
		status = create_SvcWriteMemoryByAddress_response(config, &request_msg, &response_msg);
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_ClearDiagnosticInformation:
		status = create_SvcClearDiagnosticInformation_response(config, &request_msg, &response_msg);
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_InputOutputControlByIdentifier:
		status = create_SvcInputOutputControlByIdentifier_response(config, &request_msg, &response_msg);
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_RoutineControl:
		status = create_SvcRoutineControl_response(config, &request_msg, &response_msg);
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_RequestDownload:
		status = create_SvcRequestDownload_response(config, &request_msg, &response_msg);
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_RequestUpload:
		status = create_SvcRequestUpload_response(config, &request_msg, &response_msg);
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_TransferData:
		status = create_SvcTransferData_response(tp_handle, config, &request_msg, &response_msg);
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_RequestTransferExit:
		status = create_SvcRequestTransferExit_response(config, &request_msg, &response_msg);
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_ECUReset:
		status = create_SvcECUReset_response(config, &request_msg, &response_msg);
		break;
	case (Byte)uds_service::PUDS_SERVICE_SI_ReadDTCInformation:
		// Not yet implemented
	case (Byte)uds_service::PUDS_SERVICE_SI_AccessTimingParameter:
		// Not yet implemented
	case (Byte)uds_service::PUDS_SERVICE_SI_RequestFileTransfer:
		// Not yet implemented
	case (Byte)uds_service::PUDS_SERVICE_SI_Authentication:
		// Not yet implemented
	default:
		Console::WriteLine("Unknown service (0x{0:x})", service_id);
		status = create_dummy_response(config, &request_msg, &response_msg);
		break;
	}

	// Print allocation result
	Console::WriteLine("Allocate response message: {0}", STATUS_OK_KO(status));

	if (UDSApi::StatusIsOk_2013(status))
	{
		// Send response message
		status = UDSApi::Write_2013(tp_handle, response_msg);
		Console::WriteLine(); Console::WriteLine("   ...Transmitting response: {0}", (int)status);
	}

	// Free response message
	status = UDSApi::MsgFree_2013(response_msg);
	Console::WriteLine("Free response message: {0}", STATUS_OK_KO(status));
}

/// <summary>Print an UDS message</summary>
/// <param name="msg">Message to print</param>
static void display_message(uds_msg% msg)
{
	// A function that displays UDS messages

	if ((msg.msg.msgdata.isotp->netaddrinfo.msgtype & cantp_isotp_msgtype::PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION) == 0)
	{
		Console::WriteLine(); Console::WriteLine(" {0} message from 0x{1:X2} (to 0x{2:X2}, with extension 0x{3:X2}) - result: {4} - {5}",
			((msg.msg.msgdata.isotp->flags & cantp_msgflag::PCANTP_MSGFLAG_LOOPBACK) == 0) ? "Received UDS" : "Received Loopback",
			(int)msg.msg.msgdata.isotp->netaddrinfo.source_addr,
			(int)msg.msg.msgdata.isotp->netaddrinfo.target_addr,
			(int)msg.msg.msgdata.isotp->netaddrinfo.extension_addr,
			(int)msg.msg.msgdata.isotp->netstatus,
			msg.msg.msgdata.isotp->netstatus != cantp_netstatus::PCANTP_NETSTATUS_OK ? "PCANTP_NETSTATUS_OK !!!" : "OK !");
		// Display data
		Console::Write("  -> Length: {0}, Data= ", (int)msg.msg.msgdata.isotp->length);
		for (UInt32 i = 0; i < msg.msg.msgdata.isotp->length; i++)
		{
			Console::Write("{0:X2} ", (int)msg.msg.msgdata.isotp->data[i]);
		}
		Console::WriteLine("");
	}
	else
	{

		Console::WriteLine(); Console::WriteLine(" PENDING UDS message from 0x{0:X2} (to 0x{1:X2}, with extension 0x{2:X2}) -> length={3} ...",
			(int)msg.msg.msgdata.isotp->netaddrinfo.source_addr,
			(int)msg.msg.msgdata.isotp->netaddrinfo.target_addr,
			(int)msg.msg.msgdata.isotp->netaddrinfo.extension_addr,
			(int)msg.msg.msgdata.isotp->length);
	}
}

/// <summary>Entry point of the program, start a CAN UDS server simulation</summary>
int main(array<System::String^>^ args)
{

	const int BUFFER_SIZE = 256;
	StringBuilder^ buffer = gcnew StringBuilder(BUFFER_SIZE);
	uds_status status;
	cantp_handle tp_handle;
	UInt32 server_address;
	bool stop;
	bool wait_result;
	uds_status read_status;
	uds_msg request_msg = {};
	UInt32 timeout_value;

	// Initialize variables
	tp_handle = cantp_handle::PCANTP_HANDLE_USBBUS2; // TODO: modify the value according to your available PCAN devices.
	server_address = (UInt32)uds_address::PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_1;

	// Print version informations
	status = UDSApi::GetValue_2013(cantp_handle::PCANTP_HANDLE_NONEBUS, uds_parameter::PUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE);
	Console::WriteLine("PCAN-UDS API Version - {0}: {1}", buffer, STATUS_OK_KO(status));

	// Initialize channel
	status = UDSApi::Initialize_2013(tp_handle, cantp_baudrate::PCANTP_BAUDRATE_500K);
	Console::WriteLine("Initialize channel: {0}", STATUS_OK_KO(status));

	// Set server address parameter
	status = UDSApi::SetValue_2013(tp_handle, uds_parameter::PUDS_PARAMETER_SERVER_ADDRESS, server_address, sizeof(UInt32));
	Console::WriteLine("Set server address: {0}", STATUS_OK_KO(status));

	// Set timeout values
	timeout_value = CanTpApi::PCANTP_ISO_TIMEOUTS_15765_4;
	status = UDSApi::SetValue_2013(tp_handle, uds_parameter::PUDS_PARAMETER_ISO_TIMEOUTS, timeout_value, sizeof(UInt32));
	Console::WriteLine("Set ISO 15765-4 timeouts values: {0}", STATUS_OK_KO(status));

	// Print timeout values
	status = UDSApi::GetValue_2013(tp_handle, uds_parameter::PUDS_PARAMETER_TIMEOUT_REQUEST, timeout_value, sizeof(UInt32));
	Console::WriteLine("Get request timeout value ({0}ms): {1}", timeout_value, STATUS_OK_KO(status));
	status = UDSApi::GetValue_2013(tp_handle, uds_parameter::PUDS_PARAMETER_TIMEOUT_RESPONSE, timeout_value, sizeof(UInt32));
	Console::WriteLine("Get response timeout value ({0}ms): {1}", timeout_value, STATUS_OK_KO(status));

	// Set a receive event

	System::Threading::AutoResetEvent receive_event(false);
	if (IntPtr::Size == 4)
	{
		UInt32 tmp_buffer = Convert::ToUInt32(receive_event.SafeWaitHandle->DangerousGetHandle().ToInt32());
		status = UDSApi::SetValue_2013(tp_handle, uds_parameter::PUDS_PARAMETER_RECEIVE_EVENT, tmp_buffer, sizeof(UInt32));
	}
	else if (IntPtr::Size == 8)
	{
		Int64 tmp_buffer = receive_event.SafeWaitHandle->DangerousGetHandle().ToInt64();
		array<Byte>^ byte_array = BitConverter::GetBytes(tmp_buffer);
		status = UDSApi::SetValue_2013(tp_handle, uds_parameter::PUDS_PARAMETER_RECEIVE_EVENT, byte_array, sizeof(UInt64));
	}


	Console::WriteLine("Set receive event parameter: {0}", STATUS_OK_KO(status));


	// Read while user do not press Q
	Console::WriteLine("Start listening, press Q to quit.");
	stop = false;
	while (!stop)
	{

		// Wait a receive event on receiver
		//	note: timeout is used to check keyboard hit.
		wait_result = receive_event.WaitOne(2000);

		// If we get a receive event
		if (wait_result)
		{

			// Read messages
			do
			{
				read_status = UDSApi::Read_2013(tp_handle, request_msg);
				Console::WriteLine(" Try to read a message (status={0}): {1}", (int)read_status, STATUS_OK_KO(read_status));
				if (UDSApi::StatusIsOk_2013(read_status))
				{
					display_message(request_msg);

					// We receive a request, check if it is not a loopback message, if it is not a UUDT message, if the message is not pending, and do not respond if it is not ask
					if ((request_msg.msg.msgdata.isotp->flags & cantp_msgflag::PCANTP_MSGFLAG_LOOPBACK) == 0
						&& (request_msg.msg.msgdata.isotp->netaddrinfo.msgtype & cantp_isotp_msgtype::PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION) == 0
						&& request_msg.msg.msgdata.isotp->netstatus == cantp_netstatus::PCANTP_NETSTATUS_OK
						&& (request_msg.type & uds_msgtype::PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE) != uds_msgtype::PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE
						)
					{

						// Process response and send it
						process_request(tp_handle, (UInt16)server_address, request_msg);
					}
					else
					{
						Console::WriteLine();
						Console::WriteLine("   ...Skipping response...  (flags=0x{0:X}, type=0x{1:X}, netstatus=0x{2:X}, msgtype=0x{3:X})",
							(int)request_msg.msg.msgdata.isotp->flags, (int)request_msg.type, (int)request_msg.msg.msgdata.isotp->netstatus,
							(int)request_msg.msg.msgdata.isotp->netaddrinfo.msgtype);
					}
				}

				// Free request message (and clean memory in order to reallocate later)
				status = UDSApi::MsgFree_2013(request_msg);
				Console::WriteLine("Free request message: {0}", STATUS_OK_KO(status));
			} while (!UDSApi::StatusIsOk_2013(read_status, uds_status::PUDS_STATUS_NO_MESSAGE));
		}

		// Quit when user press Q
		if (Console::KeyAvailable)
		{
			char keyboard_res = (char)Console::ReadKey().KeyChar;
			if (keyboard_res == 'Q' || keyboard_res == 'q')
			{
				stop = true;
			}
		}
	}

	// Close receive event
	if (IntPtr::Size == 4)
	{
		UInt32 tmp_buffer = 0;
		status = UDSApi::SetValue_2013(tp_handle, uds_parameter::PUDS_PARAMETER_RECEIVE_EVENT, tmp_buffer, sizeof(UInt32));
	}
	else if (IntPtr::Size == 8)
	{
		Int64 tmp_buffer = 0;
		array<Byte>^ byte_array = BitConverter::GetBytes(tmp_buffer);
		status = UDSApi::SetValue_2013(tp_handle, uds_parameter::PUDS_PARAMETER_RECEIVE_EVENT, byte_array, sizeof(UInt64));
	}
	Console::WriteLine("Stop receive event: {0}", STATUS_OK_KO(status));
	receive_event.Close();
	Console::WriteLine("Close receive event: {0}", STATUS_OK_KO(status));

	// Close server
	status = UDSApi::Uninitialize_2013(tp_handle);
	Console::WriteLine("Uninitialize channel: {0}", STATUS_OK_KO(status));

	//Exit
	Console::WriteLine("Press any key to continue...");
	Console::Read();
}


