
#include "PCANBasicCLR.h"
#include "PCAN-ISO-TP-CLR_2016.h"
#include "PCAN-UDS-CLR_2013.h"


using namespace System;
using namespace System::IO;
using namespace Peak::Can::IsoTp;
using namespace Peak::Can::Uds;

static String^ OK_KO(bool test)
{
	if (test)
		return "OK";
	return "KO";
}

static String^ STATUS_OK_KO(uds_status test)
{
	return OK_KO(UDSApi::StatusIsOk_2013(test));
}

// Define Bitrate: SAE J2284-4: High-Speed CAN (HSC) for Vehicle Applications at 500 kbps with CAN FD Data at 2 Mbps
ref class Global
{
public:
	static const cantp_bitrate PCAN_BITRATE_SAE_J2284_4 = "f_clock=80000000,nom_brp=2,nom_tseg1=63,nom_tseg2=16,nom_sjw=16,data_brp=2,data_tseg1=15,data_tseg2=4,data_sjw=4";
	static String^ MSG = "PEAK";
	static const UInt32 MSG_SIZE = 4;
	static const UInt16 MAPPING_BUFFER_SIZE = 256;
	static const int BUFFER_SIZE = 256;
};

/// <summary>Helper: Check if a mapping list contains a certain mapping</summary>
/// <param name="mapping_list">Mapping list</param>
/// <param name="mapping_list_size">Number of mapping in the mapping list</param>
/// <param name="searched_mapping">Searched mapping</param>
/// <returns>Present (true) or not (false)</returns>
static bool mapping_list_contains(array<uds_mapping>^ mapping_list, UInt16 mapping_list_size, uds_mapping* searched_mapping)
{
	bool res = false;
	for (int i = 0; i < mapping_list_size; i++)
	{

		// If unique identifier are the same, the mapping is in the list
		if (mapping_list[i].uid == searched_mapping->uid)
		{
			res = true;
			break;
		}
	}
	return res;
}

/// <summary>Entry point of the program, start a small CAN UDS read/write example</summary>
int main(array<System::String^>^ args)
{
	StringBuilder^ buffer = gcnew StringBuilder(Global::BUFFER_SIZE);
	uds_status status;
	cantp_handle transmitter_handle;
	cantp_handle receiver_handle;
	uds_msg tx_msg = {};
	uds_msg rx_msg = {};
	uds_msgconfig msg_config = {};
	uds_mapping source_mapping = {};
	uds_mapping response_mapping = {};
	array<uds_mapping>^ mapping_buffer = gcnew array<uds_mapping>(Global::MAPPING_BUFFER_SIZE);
	UInt16 mapping_count;
	bool wait_result;

	// Initialize variables
	transmitter_handle = cantp_handle::PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.
	receiver_handle = cantp_handle::PCANTP_HANDLE_USBBUS2; // TODO: modify the value according to your available PCAN devices.
	mapping_count = 0;

	// Print version informations
	status = UDSApi::GetValue_2013(cantp_handle::PCANTP_HANDLE_NONEBUS, uds_parameter::PUDS_PARAMETER_API_VERSION, buffer, Global::BUFFER_SIZE);
	Console::WriteLine("PCAN-UDS API Version: {0} ({1})", buffer, STATUS_OK_KO(status));

	// Initialize transmitter and receiver
	status = UDSApi::InitializeFD_2013(transmitter_handle, Global::PCAN_BITRATE_SAE_J2284_4);
	Console::WriteLine("Initialize transmitter: {0}", STATUS_OK_KO(status));
	status = UDSApi::InitializeFD_2013(receiver_handle, Global::PCAN_BITRATE_SAE_J2284_4);
	Console::WriteLine("Initialize receiver: {0}", STATUS_OK_KO(status));

	// Create a receive event on receiver
	System::Threading::AutoResetEvent receive_event(false);
	if (IntPtr::Size == 4)
	{
		UInt32 tmp_buffer = Convert::ToUInt32(receive_event.SafeWaitHandle->DangerousGetHandle().ToInt32());
		status = UDSApi::SetValue_2013(receiver_handle, uds_parameter::PUDS_PARAMETER_RECEIVE_EVENT, tmp_buffer, sizeof(UInt32));
	}
	else if (IntPtr::Size == 8)
	{
		Int64 tmp_buffer = receive_event.SafeWaitHandle->DangerousGetHandle().ToInt64();
		array<Byte>^ byte_array = BitConverter::GetBytes(tmp_buffer);
		status = UDSApi::SetValue_2013(receiver_handle, uds_parameter::PUDS_PARAMETER_RECEIVE_EVENT, byte_array, sizeof(UInt64));
	}
	Console::WriteLine("Set receive event parameter: {0}", STATUS_OK_KO(status));

	// Initialize 29bits
	msg_config.can_id = 0x30;
	msg_config.can_msgtype = cantp_can_msgtype::PCANTP_CAN_MSGTYPE_STANDARD;
	// To use CAN FD with bitrate switch uncomment:
	// msg_config.can_msgtype |= (PCANTP_CAN_MSGTYPE_FD | PCANTP_CAN_MSGTYPE_BRS);
	msg_config.nai.protocol = uds_msgprotocol::PUDS_MSGPROTOCOL_ISO_15765_2_29B_NORMAL;
	msg_config.nai.target_type = cantp_isotp_addressing::PCANTP_ISOTP_ADDRESSING_PHYSICAL;
	msg_config.type = uds_msgtype::PUDS_MSGTYPE_USDT;
	msg_config.nai.source_addr = 0x10;
	msg_config.nai.target_addr = 0x20;

	// Allocate TX message with previous configuration
	status = UDSApi::MsgAlloc_2013(tx_msg, msg_config, Global::MSG_SIZE);
	if (UDSApi::StatusIsOk_2013(status))
	{
		for (int i = 0; i < Global::MSG_SIZE; i++)
		{
			tx_msg.msg.msgdata.any->data[i] = (Byte)(Global::MSG[i]);
		}
	}
	Console::WriteLine("Allocate TX message: {0}", STATUS_OK_KO(status));


	// Initialize source mapping
	source_mapping.can_id = msg_config.can_id;
	source_mapping.can_id_flow_ctrl = msg_config.can_id + 1;
	source_mapping.can_msgtype = msg_config.can_msgtype;
	source_mapping.can_tx_dlc = 8;
	source_mapping.nai = msg_config.nai;

	// Initialize response mapping
	response_mapping = source_mapping;
	response_mapping.can_id = source_mapping.can_id_flow_ctrl;
	response_mapping.can_id_flow_ctrl = source_mapping.can_id;
	response_mapping.nai.source_addr = source_mapping.nai.target_addr;
	response_mapping.nai.target_addr = source_mapping.nai.source_addr;

	// Add mappings on transmitter
	status = UDSApi::AddMapping_2013(transmitter_handle, source_mapping);
	Console::WriteLine("Add source mapping on transmitter: {0}", STATUS_OK_KO(status));
	status = UDSApi::AddMapping_2013(transmitter_handle, response_mapping);
	Console::WriteLine("Add response mapping on transmitter: {0}", STATUS_OK_KO(status));

	// Check if mappings are added on transmitter
	status = UDSApi::GetMappings_2013(transmitter_handle, mapping_buffer, (UInt16)Global::MAPPING_BUFFER_SIZE, mapping_count);
	Console::WriteLine("Get mappings on transmitter: {0}", STATUS_OK_KO(status));
	Console::WriteLine("Check added mappings on transmitter: {0}", OK_KO(mapping_list_contains(mapping_buffer, mapping_count, &source_mapping)
		&& mapping_list_contains(mapping_buffer, mapping_count, &response_mapping)));

	// Add mappings on receiver
	status = UDSApi::AddMapping_2013(receiver_handle, source_mapping);
	Console::WriteLine("Add source mapping on receiver: {0}", STATUS_OK_KO(status));
	status = UDSApi::AddMapping_2013(receiver_handle, response_mapping);
	Console::WriteLine("Add response mapping on receiver: {0}", STATUS_OK_KO(status));

	// Check if mappings are added on receiver
	status = UDSApi::GetMappings_2013(receiver_handle, mapping_buffer, Global::MAPPING_BUFFER_SIZE, mapping_count);
	Console::WriteLine("Get mappings on receiver: {0}", STATUS_OK_KO(status));
	Console::WriteLine("Check added mappings on receiver: {0}", OK_KO(mapping_list_contains(mapping_buffer, mapping_count, &source_mapping)
		&& mapping_list_contains(mapping_buffer, mapping_count, &response_mapping)));

	// Write message
	status = UDSApi::Write_2013(transmitter_handle, tx_msg);
	Console::WriteLine("Write \"PEAK\": {0}", STATUS_OK_KO(status));

	// Wait a receive event
	wait_result = receive_event.WaitOne(3000);
	Console::WriteLine("Wait a receive event: {0}", OK_KO(wait_result));

	// If we get a receive event
	if (wait_result)
	{

		// Read message
		status = UDSApi::Read_2013(receiver_handle, rx_msg);
		Console::WriteLine("Receive message: {0}", STATUS_OK_KO(status));

		// Check message (should contains "PEAK")
		bool msgok = false;
		if (UDSApi::StatusIsOk_2013(status) && rx_msg.msg.msgdata.any->length == Global::MSG_SIZE)
		{
			msgok = true;
			for (int i = 0; msgok && i < Global::MSG_SIZE; i++) {
				if (rx_msg.msg.msgdata.any->data[i] != Global::MSG[i]) {
					msgok = false;
					break;
				}
			}
		}
		if (msgok)
		{
			Console::WriteLine("Message contains \"{0}{1}{2}{3}\": OK", Convert::ToChar(rx_msg.msg.msgdata.any->data[0]), Convert::ToChar(rx_msg.msg.msgdata.any->data[1]),
				Convert::ToChar(rx_msg.msg.msgdata.any->data[2]), Convert::ToChar(rx_msg.msg.msgdata.any->data[3]));
		}
		else
		{
			Console::WriteLine("Message is corrupted: KO");
		}
	}

	// Close receive event
	if (IntPtr::Size == 4)
	{
		UInt32 tmp_buffer = 0;
		status = UDSApi::SetValue_2013(receiver_handle, uds_parameter::PUDS_PARAMETER_RECEIVE_EVENT, tmp_buffer, sizeof(UInt32));
	}
	else if (IntPtr::Size == 8)
	{
		Int64 tmp_buffer = 0;
		array<Byte>^ byte_array = BitConverter::GetBytes(tmp_buffer);
		status = UDSApi::SetValue_2013(receiver_handle, uds_parameter::PUDS_PARAMETER_RECEIVE_EVENT, byte_array, sizeof(UInt64));
	}
	Console::WriteLine("Stop receive event receiver: {0}", STATUS_OK_KO(status));
	receive_event.Close();
	Console::WriteLine("Close receive event: {0}", STATUS_OK_KO(status));

	// Free messages
	status = UDSApi::MsgFree_2013(tx_msg);
	Console::WriteLine("Free TX message: {0}", STATUS_OK_KO(status));
	status = UDSApi::MsgFree_2013(rx_msg);
	Console::WriteLine("Free RX message: {0}", STATUS_OK_KO(status));

	// Close transmitter and receiver
	status = UDSApi::Uninitialize_2013(transmitter_handle);
	Console::WriteLine("Uninitialize transmitter: {0}", STATUS_OK_KO(status));
	status = UDSApi::Uninitialize_2013(receiver_handle);
	Console::WriteLine("Uninitialize receiver: {0}", STATUS_OK_KO(status));

	// Exit
	Console::WriteLine("Press any key to continue...");
	Console::Read();
}


