
#include "PCANBasicCLR.h"
#include "PCAN-ISO-TP-CLR_2016.h"
#include "PCAN-UDS-CLR_2013.h"


using namespace System;
using namespace System::IO;
using namespace System::Threading;
using namespace System::Runtime::InteropServices;
using namespace Peak::Can::IsoTp;
using namespace Peak::Can::Uds;

static String^ OK_KO(bool test)
{
	if (test)
		return "OK";
	return "KO";
}

static String^ STATUS_OK_KO(int test)
{
	return OK_KO(UDSApi::StatusIsOk_2013((uds_status)test));
}

// Define Bitrate: SAE J2284-4: High-Speed CAN (HSC) for Vehicle Applications at 500 kbps with CAN FD Data at 2 Mbps
ref class Global
{
public:
	static const cantp_bitrate PCAN_BITRATE_SAE_J2284_4 = "f_clock=80000000,nom_brp=2,nom_tseg1=63,nom_tseg2=16,nom_sjw=16,data_brp=2,data_tseg1=15,data_tseg2=4,data_sjw=4";
	static const int BUFFER_SIZE = 256;
	static String^ MSG = "PEAK";
	static const UInt32 MSG_SIZE = 4;
};

/// <summary>
/// Entry point of the program, start a small CAN UDS / ISOTP read-write example
/// This example initialise two UDS channel with a specific mapping.
/// Then it inits an ISOTP mapping and write an ISOTP message.
/// In this case, PCAN-UDS should not receive the message, we need PCAN-ISOTP to read it.
/// </summary>
/// <returns>By convention, return success.</returns>
int main(array<System::String^>^ args)
{
	StringBuilder^ buffer = gcnew StringBuilder(Global::BUFFER_SIZE);
	int status;
	cantp_handle transmitter_handle;
	cantp_handle receiver_handle;
	cantp_msg tx_msg = {};
	uds_msg uds_rx_msg = {};
	cantp_msg rx_msg = {};
	uds_mapping source_mapping = {};
	cantp_mapping mapping = {};
	cantp_mapping reverse_mapping = {};
	uds_mapping response_mapping = {};
	cantp_timestamp timestamp_buffer = {};
	bool wait_result;
	cantp_can_msgtype can_msgtype;

	// Initialize variables
	transmitter_handle = cantp_handle::PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.
	receiver_handle = cantp_handle::PCANTP_HANDLE_USBBUS2; // TODO: modify the value according to your available PCAN devices.
	can_msgtype = PCANTP_CAN_MSGTYPE_STANDARD;
	// To use CAN FD with bitrate switch uncomment:
	// can_msgtype |= (PCANTP_CAN_MSGTYPE_FD | PCANTP_CAN_MSGTYPE_BRS);

	// Print version informations
	status = (int)UDSApi::GetValue_2013(cantp_handle::PCANTP_HANDLE_NONEBUS, PUDS_PARAMETER_API_VERSION, buffer, Global::BUFFER_SIZE);
	Console::WriteLine("PCAN-UDS API Version: {0} ({1})", buffer, STATUS_OK_KO(status));

	// Initialize transmitter and receiver
	status = (int)UDSApi::InitializeFD_2013(transmitter_handle, Global::PCAN_BITRATE_SAE_J2284_4);
	Console::WriteLine("Initialize transmitter: {0}", STATUS_OK_KO(status));
	status = (int)UDSApi::InitializeFD_2013(receiver_handle, Global::PCAN_BITRATE_SAE_J2284_4);
	Console::WriteLine("Initialize receiver: {0}", STATUS_OK_KO(status));

	// Create a receive event on receiver
	System::Threading::AutoResetEvent receive_event(false);
	if (IntPtr::Size == 4)
	{
		UInt32 tmp_buffer = Convert::ToUInt32(receive_event.SafeWaitHandle->DangerousGetHandle().ToInt32());
		status = (int)UDSApi::SetValue_2013(receiver_handle, uds_parameter::PUDS_PARAMETER_RECEIVE_EVENT, tmp_buffer, sizeof(UInt32));
	}
	else if (IntPtr::Size == 8)
	{
		Int64 tmp_buffer = receive_event.SafeWaitHandle->DangerousGetHandle().ToInt64();
		array<Byte>^ byte_array = BitConverter::GetBytes(tmp_buffer);
		status = (int)UDSApi::SetValue_2013(receiver_handle, uds_parameter::PUDS_PARAMETER_RECEIVE_EVENT, byte_array, sizeof(UInt64));
	}
	Console::WriteLine("Set receive event parameter: {0}", STATUS_OK_KO(status));

	// Initialize source mapping
	source_mapping.can_id = 0x30;
	source_mapping.can_id_flow_ctrl = source_mapping.can_id + 1;
	source_mapping.can_msgtype = can_msgtype;
	source_mapping.can_tx_dlc = 8;
	source_mapping.nai.protocol = uds_msgprotocol::PUDS_MSGPROTOCOL_ISO_15765_2_29B_NORMAL;
	source_mapping.nai.target_type = cantp_isotp_addressing::PCANTP_ISOTP_ADDRESSING_PHYSICAL;
	source_mapping.nai.source_addr = 0x10;
	source_mapping.nai.target_addr = 0x20;

	// Initialize response mapping
	response_mapping = source_mapping;
	response_mapping.can_id = source_mapping.can_id_flow_ctrl;
	response_mapping.can_id_flow_ctrl = source_mapping.can_id;
	response_mapping.nai.source_addr = source_mapping.nai.target_addr;
	response_mapping.nai.target_addr = source_mapping.nai.source_addr;

	// Add UDS mappings on transmitter
	status = (int)UDSApi::AddMapping_2013(transmitter_handle, source_mapping);
	Console::WriteLine("Add source mapping on transmitter: {0}", STATUS_OK_KO(status));
	status = (int)UDSApi::AddMapping_2013(transmitter_handle, response_mapping);
	Console::WriteLine("Add response mapping on transmitter: {0}", STATUS_OK_KO(status));

	// Add UDS mappings on and receiver
	status = (int)UDSApi::AddMapping_2013(receiver_handle, source_mapping);
	Console::WriteLine("Add source mapping on receiver: {0}", STATUS_OK_KO(status));
	status = (int)UDSApi::AddMapping_2013(receiver_handle, response_mapping);
	Console::WriteLine("Add response mapping on receiver: {0}", STATUS_OK_KO(status));

	// Create a simple isotp physical mapping:
	//    - Source 0xF1 (transmitter), target 0x01 (receiver), CAN id 0xA1, CAN ID flow control 0xA2
	//    - Diagnostic message in a classic format
	mapping.can_id = 0xA1;
	mapping.can_id_flow_ctrl = 0xA2;
	mapping.can_msgtype = can_msgtype;
	mapping.can_tx_dlc = 0x0;
	mapping.netaddrinfo.extension_addr = 0x00;
	mapping.netaddrinfo.format = cantp_isotp_format::PCANTP_ISOTP_FORMAT_NORMAL;
	mapping.netaddrinfo.msgtype = cantp_isotp_msgtype::PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC;
	mapping.netaddrinfo.source_addr = 0xF1;
	mapping.netaddrinfo.target_addr = 0x01;
	mapping.netaddrinfo.target_type = cantp_isotp_addressing::PCANTP_ISOTP_ADDRESSING_PHYSICAL;

	// Create the associated isotp reversed mapping:
	reverse_mapping = mapping;
	reverse_mapping.can_id = mapping.can_id_flow_ctrl;
	reverse_mapping.can_id_flow_ctrl = mapping.can_id;
	reverse_mapping.netaddrinfo.source_addr = mapping.netaddrinfo.target_addr;
	reverse_mapping.netaddrinfo.target_addr = mapping.netaddrinfo.source_addr;

	// Add ISOTP mappings on channels
	status = (int)CanTpApi::AddMapping_2016(transmitter_handle, mapping);
	Console::WriteLine("Add a simple isotp mapping on transmitter: {0}", STATUS_OK_KO(status));
	status = (int)CanTpApi::AddMapping_2016(transmitter_handle, reverse_mapping);
	Console::WriteLine("Add the reverse isotp mapping on transmitter: {0}", STATUS_OK_KO(status));
	status = (int)CanTpApi::AddMapping_2016(receiver_handle, mapping);
	Console::WriteLine("Add a simple isotp mapping on receiver: {0}", STATUS_OK_KO(status));
	status = (int)CanTpApi::AddMapping_2016(receiver_handle, reverse_mapping);
	Console::WriteLine("Add the reverse isotp mapping on receiver: {0}", STATUS_OK_KO(status));

	// Initialize ISOTP Tx message containing "PEAK"
	status = (int)CanTpApi::MsgDataAlloc_2016(tx_msg, cantp_msgtype::PCANTP_MSGTYPE_ISOTP);
	Console::WriteLine("Allocate tx ISOTP message: {0}", STATUS_OK_KO(status));
	array<Byte>^ msg_array = Encoding::ASCII->GetBytes(Global::MSG);
	status = (int)CanTpApi::MsgDataInit_2016(tx_msg, mapping.can_id, mapping.can_msgtype, Global::MSG_SIZE, msg_array, mapping.netaddrinfo);
	Console::WriteLine("Initialize tx message: {0}", STATUS_OK_KO(status));

	// Send "PEAK" ISOTP message
	status = (int)CanTpApi::Write_2016(transmitter_handle, tx_msg);
	Console::WriteLine("Write \"PEAK\": {0}", STATUS_OK_KO(status));

	// Wait a receive event on uds message (should not work because we sent a ISOTP message)
	wait_result = receive_event.WaitOne(3000);
	Console::WriteLine("Wait a receive event on UDS message (should not work because we sent an ISOTP message): {0}", OK_KO(wait_result));

	// We should receive a ISOTP message instead
	status = (int)CanTpApi::Read_2016(receiver_handle, rx_msg, timestamp_buffer, cantp_msgtype::PCANTP_MSGTYPE_NONE);
	Console::WriteLine("Read ISOTP message on receiver: {0}", STATUS_OK_KO(status));

	// Check message (should contains "PEAK")
	bool msgok = false;
	if (UDSApi::StatusIsOk_2013((uds_status)status) && rx_msg.msgdata.any->length == Global::MSG_SIZE)
	{
		msgok = true;
		for (int i = 0; msgok && i < Global::MSG_SIZE; i++)
		{
			if (rx_msg.msgdata.any->data[i] != Global::MSG[i])
			{
				msgok = false;
				break;
			}
		}
	}
	if (msgok)
	{
		Console::WriteLine("Message contains \"{0}{1}{2}{3}\": OK", Convert::ToChar(rx_msg.msgdata.any->data[0]), Convert::ToChar(rx_msg.msgdata.any->data[1]), Convert::ToChar(rx_msg.msgdata.any->data[2]), Convert::ToChar(rx_msg.msgdata.any->data[3]));
	}
	else
	{
		Console::WriteLine("Message is corrupted: KO");
	}

	// Close receive event
	if (IntPtr::Size == 4)
	{
		UInt32 tmp_buffer = 0;
		status = (int)UDSApi::SetValue_2013(receiver_handle, uds_parameter::PUDS_PARAMETER_RECEIVE_EVENT, tmp_buffer, sizeof(UInt32));
	}
	else if (IntPtr::Size == 8)
	{
		Int64 tmp_buffer = 0;
		array<Byte>^ byte_array = BitConverter::GetBytes(tmp_buffer);
		status = (int)UDSApi::SetValue_2013(receiver_handle, uds_parameter::PUDS_PARAMETER_RECEIVE_EVENT, byte_array, sizeof(UInt64));
	}
	Console::WriteLine("Stop receive event receiver: {0}", STATUS_OK_KO(status));
	receive_event.Close();
	Console::WriteLine("Close receive event: {0}", STATUS_OK_KO(status));

	// Free messages
	status = (int)CanTpApi::MsgDataFree_2016(tx_msg);
	Console::WriteLine("Free TX message: {0}", STATUS_OK_KO(status));
	status = (int)CanTpApi::MsgDataFree_2016(rx_msg);
	Console::WriteLine("Free RX message: {0}", STATUS_OK_KO(status));
	status = (int)UDSApi::MsgFree_2013(uds_rx_msg);
	Console::WriteLine("Free UDS/RX message: {0}", STATUS_OK_KO(status));

	// Close transmitter and receiver
	status = (int)UDSApi::Uninitialize_2013(transmitter_handle);
	Console::WriteLine("Uninitialize transmitter: {0}", STATUS_OK_KO(status));
	status = (int)UDSApi::Uninitialize_2013(receiver_handle);
	Console::WriteLine("Uninitialize receiver: {0}", STATUS_OK_KO(status));

	// Exit
	Console::WriteLine("Press any key to continue...");
	Console::Read();
}
