
#include "PCANBasicCLR.h"
#include "PCAN-ISO-TP-CLR_2016.h"
#include "PCAN-UDS-CLR_2013.h"


using namespace System;
using namespace System::IO;
using namespace System::Threading;
using namespace System::Runtime::InteropServices;
using namespace Peak::Can::IsoTp;
using namespace Peak::Can::Uds;

ref class Global
{
public:
	static const int BUFFER_SIZE = 256;
	static const int MSG_SIZE = 4;
	static String^ ISOTP_REQUEST_MSG = "PING";
	static String^ ISOTP_RESPONSE_MSG = "PONG";
};

static String^ OK_KO(bool test)
{
	if (test)
		return "OK";
	return "KO";
}
static String^ UDS_STATUS_OK_KO(uds_status test)
{
	return OK_KO(UDSApi::StatusIsOk_2013(test));
}
static String^ ISOTP_STATUS_OK_KO(cantp_status test)
{
	return OK_KO(CanTpApi::StatusIsOk_2016(test));
}

/// <summary>Structure passed as thread parameters</summary>
public ref struct task_params
{
public:
	/// <summary>Server channel handle</summary>
	cantp_handle server_handle;
	/// <summary>Server address</summary>
	UInt32 server_address;
	/// <summary>Determine if the thread should end or not</summary>
	bool stop_task;
};

/// <summary>ISOTP server task: respond "PING" message by sending "PONG"</summary>
/// <param name="parameters">pointer on task_params structures</param>
static void isotp_server_task(Object^ parameters)
{

	// Init variables
	cantp_msg rx_msg = {};
	cantp_msg tx_msg = {};
	cantp_mapping mapping = {};
	cantp_mapping reverse_mapping = {};
	task_params^ t_params = safe_cast<task_params^>(parameters);
	cantp_timestamp timestamp_buffer;
	bool wait_result;
	cantp_status status = cantp_status::PCANTP_STATUS_UNKNOWN;

	// Create a isotp receive event
	System::Threading::AutoResetEvent receive_event(false);
	if (IntPtr::Size == 4)
	{
		UInt32 tmp_buffer = Convert::ToUInt32(receive_event.SafeWaitHandle->DangerousGetHandle().ToInt32());
		status = CanTpApi::SetValue_2016(t_params->server_handle, cantp_parameter::PCANTP_PARAMETER_RECEIVE_EVENT, tmp_buffer, sizeof(UInt32));
	}
	else if (IntPtr::Size == 8)
	{
		Int64 tmp_buffer = receive_event.SafeWaitHandle->DangerousGetHandle().ToInt64();
		array<Byte>^ byte_array = BitConverter::GetBytes(tmp_buffer);
		status = CanTpApi::SetValue_2016(t_params->server_handle, cantp_parameter::PCANTP_PARAMETER_RECEIVE_EVENT, byte_array, sizeof(UInt64));
	}
	Console::WriteLine("[ISOTP] Set isotp receive event parameter: {0}", ISOTP_STATUS_OK_KO(status));

	// Create a simple isotp physical mapping:
	//    - Source 0xA1 (client), target 0xA2 (server), CAN id 0xA1, CAN ID flow control 0xA2
	//    - Diagnostic message in a classic format
	mapping.can_id = 0xA1;
	mapping.can_id_flow_ctrl = 0xA2;
	mapping.can_msgtype = cantp_can_msgtype::PCANTP_CAN_MSGTYPE_STANDARD;
	mapping.netaddrinfo.extension_addr = 0x00;
	mapping.netaddrinfo.format = cantp_isotp_format::PCANTP_ISOTP_FORMAT_NORMAL;
	mapping.netaddrinfo.msgtype = cantp_isotp_msgtype::PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC;
	mapping.netaddrinfo.source_addr = 0xA1;
	mapping.netaddrinfo.target_addr = 0xA2;
	mapping.netaddrinfo.target_type = cantp_isotp_addressing::PCANTP_ISOTP_ADDRESSING_PHYSICAL;

	// Create the associated isotp reversed mapping:
	reverse_mapping = mapping;
	reverse_mapping.can_id = mapping.can_id_flow_ctrl;
	reverse_mapping.can_id_flow_ctrl = mapping.can_id;
	reverse_mapping.netaddrinfo.source_addr = mapping.netaddrinfo.target_addr;
	reverse_mapping.netaddrinfo.target_addr = mapping.netaddrinfo.source_addr;

	// Add ISOTP mappings on channels
	status = CanTpApi::AddMapping_2016(t_params->server_handle, mapping);
	Console::WriteLine("[ISOTP] Add a simple isotp mapping: {0}", ISOTP_STATUS_OK_KO(status));
	status = CanTpApi::AddMapping_2016(t_params->server_handle, reverse_mapping);
	Console::WriteLine("[ISOTP] Add the reverse isotp mapping: {0}", ISOTP_STATUS_OK_KO(status));

	// Initialize ISOTP Tx message containing "PING"
	status = CanTpApi::MsgDataAlloc_2016(tx_msg, cantp_msgtype::PCANTP_MSGTYPE_ISOTP);
	Console::WriteLine("[ISOTP] Allocate ISOTP tx message: {0}", ISOTP_STATUS_OK_KO(status));
	array<Byte>^ msg_array = Encoding::ASCII->GetBytes(Global::ISOTP_RESPONSE_MSG);
	status = CanTpApi::MsgDataInit_2016(tx_msg, reverse_mapping.can_id, reverse_mapping.can_msgtype, Global::MSG_SIZE, msg_array, reverse_mapping.netaddrinfo);
	Console::WriteLine("[ISOTP] Initialize ISOTP tx message: {0}", ISOTP_STATUS_OK_KO(status));

	// Wait a receive event on isotp message
	do
	{
		wait_result = receive_event.WaitOne(1000);
		Console::WriteLine("[ISOTP] Wait a receive event from isotp: {0}", OK_KO(wait_result));

		// Read ISOTP messages
		do
		{
			status = CanTpApi::Read_2016(t_params->server_handle, rx_msg, timestamp_buffer, cantp_msgtype::PCANTP_MSGTYPE_NONE);
			Console::WriteLine("[ISOTP] Read ISOTP message: {0}", ISOTP_STATUS_OK_KO(status));

			// Check message (should contains "PING") and send "PONG"
			if (!CanTpApi::StatusIsOk_2016(status, cantp_status::PCANTP_STATUS_NO_MESSAGE))
			{
				if (CanTpApi::StatusIsOk_2016(status, cantp_status::PCANTP_STATUS_OK, false))
				{
					bool msgok = true;
					for (int i = 0; msgok && i < Global::MSG_SIZE; i++)
					{
						if (rx_msg.msgdata.any->data[i] != Global::ISOTP_REQUEST_MSG[i])
						{
							msgok = false;
							break;
						}
					}
					if (msgok)
					{
						Console::WriteLine("[ISOTP] Received message contains \"{0}{1}{2}{3}\": OK", Convert::ToChar(rx_msg.msgdata.any->data[0]), Convert::ToChar(rx_msg.msgdata.any->data[1]), Convert::ToChar(rx_msg.msgdata.any->data[2]), Convert::ToChar(rx_msg.msgdata.any->data[3]));
						status = CanTpApi::Write_2016(t_params->server_handle, tx_msg);
						Console::WriteLine("[ISOTP] Send ISOTP \"PONG\" message: {0}", ISOTP_STATUS_OK_KO(status));
					}
				}

				// Free RX message
				status = CanTpApi::MsgDataFree_2016(rx_msg);
				Console::WriteLine("[ISOTP] Free RX message: {0}", ISOTP_STATUS_OK_KO(status));
			}
		} while (!CanTpApi::StatusIsOk_2016(status, cantp_status::PCANTP_STATUS_NO_MESSAGE));
	} while (t_params->stop_task == false);

	// Close receive event
	if (IntPtr::Size == 4)
	{
		UInt32 tmp_buffer = 0;
		status = CanTpApi::SetValue_2016(t_params->server_handle, cantp_parameter::PCANTP_PARAMETER_RECEIVE_EVENT, tmp_buffer, sizeof(UInt32));
	}
	else if (IntPtr::Size == 8)
	{
		Int64 tmp_buffer = 0;
		array<Byte>^ byte_array = BitConverter::GetBytes(tmp_buffer);
		status = CanTpApi::SetValue_2016(t_params->server_handle, cantp_parameter::PCANTP_PARAMETER_RECEIVE_EVENT, byte_array, sizeof(UInt64));
	}
	Console::WriteLine("[ISOTP] Stop ISOTP receive event: {0}", ISOTP_STATUS_OK_KO(status));
	receive_event.Close();
	Console::WriteLine("[ISOTP] Close ISOTP receive event: {0}", ISOTP_STATUS_OK_KO(status));

	// Free messages
	status = CanTpApi::MsgDataFree_2016(tx_msg);
	Console::WriteLine("[ISOTP] Free ISOTP TX message: {0}", ISOTP_STATUS_OK_KO(status));
}

/// <summary>UDS server task: respond TesterPresent request</summary>
/// <param name="parameters">pointer on task_params structures</param>
static void uds_server_task(Object^ parameters)
{
	// Init variables
	uds_msgconfig config_physical = {};
	uds_msg request_msg = {};
	uds_msg response_msg = {};
	task_params^ t_params = safe_cast<task_params^>(parameters);
	bool wait_result;
	uds_status status;
	uds_status read_status;

	// Set server address parameter
	status = UDSApi::SetValue_2013(t_params->server_handle, uds_parameter::PUDS_PARAMETER_SERVER_ADDRESS, t_params->server_address, sizeof(UInt32));
	Console::WriteLine("[UDS] Set UDS server address: {0}", UDS_STATUS_OK_KO(status));

	// Set a receive event for UDS
	System::Threading::AutoResetEvent receive_event(false);
	if (IntPtr::Size == 4)
	{
		UInt32 tmp_buffer = Convert::ToUInt32(receive_event.SafeWaitHandle->DangerousGetHandle().ToInt32());
		status = UDSApi::SetValue_2013(t_params->server_handle, uds_parameter::PUDS_PARAMETER_RECEIVE_EVENT, tmp_buffer, sizeof(UInt32));
	}
	else if (IntPtr::Size == 8)
	{
		Int64 tmp_buffer = receive_event.SafeWaitHandle->DangerousGetHandle().ToInt64();
		array<Byte>^ byte_array = BitConverter::GetBytes(tmp_buffer);
		status = UDSApi::SetValue_2013(t_params->server_handle, uds_parameter::PUDS_PARAMETER_RECEIVE_EVENT, byte_array, sizeof(UInt64));
	}
	Console::WriteLine("[UDS] Set UDS receive event parameter: {0}", UDS_STATUS_OK_KO(status));

	// Initialize a physical configuration
	config_physical.can_id = 0xFFFFFFFF;
	config_physical.can_msgtype = cantp_can_msgtype::PCANTP_CAN_MSGTYPE_STANDARD;
	config_physical.nai.protocol = uds_msgprotocol::PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL;
	config_physical.nai.target_type = cantp_isotp_addressing::PCANTP_ISOTP_ADDRESSING_PHYSICAL;
	config_physical.type = uds_msgtype::PUDS_MSGTYPE_USDT;
	config_physical.nai.extension_addr = 0;

	do
	{
		// Wait a receive event on receiver
		wait_result = receive_event.WaitOne(1000);

		// If we get a receive event
		if (wait_result)
		{
			do
			{
				// Read first available message (no filtering based on message's type is set):
				read_status = UDSApi::Read_2013(t_params->server_handle, request_msg);

				Console::WriteLine("[UDS] Try to read a message: {0}", UDS_STATUS_OK_KO(read_status));
				if (UDSApi::StatusIsOk_2013(read_status))
				{

					// We receive a request, check its length and if it is not a loopback message and if it is a USDT message
					if (request_msg.type == uds_msgtype::PUDS_MSGTYPE_USDT && request_msg.msg.msgdata.any->length >= 1 && (request_msg.msg.msgdata.any->flags & cantp_msgflag::PCANTP_MSGFLAG_LOOPBACK) == 0)
					{

						// This is a valid request, switch services

						switch (*request_msg.links.service_id)
						{
						case uds_service::PUDS_SERVICE_SI_TesterPresent:

							// Allocate response message
							status = UDSApi::MsgAlloc_2013(response_msg, config_physical, 2);
							Console::WriteLine("[UDS] Prepare response message for TesterPresent service: {0}", UDS_STATUS_OK_KO(status));

							if (UDSApi::StatusIsOk_2013(status)) {
								// Fill parameters
								response_msg.msg.msgdata.isotp->netaddrinfo = request_msg.msg.msgdata.isotp->netaddrinfo;
								*response_msg.links.service_id = uds_service::PUDS_SERVICE_SI_TesterPresent + UDSApi::PUDS_SI_POSITIVE_RESPONSE;
								response_msg.links.param[0] = 0;
								response_msg.msg.msgdata.isotp->netaddrinfo.target_addr = request_msg.msg.msgdata.isotp->netaddrinfo.source_addr;
								response_msg.msg.msgdata.isotp->netaddrinfo.source_addr = t_params->server_address;

								// Write response message
								status = UDSApi::Write_2013(t_params->server_handle, response_msg);
								Console::WriteLine("[UDS] Write response message for TesterPresent service: {0}", UDS_STATUS_OK_KO(status));
							}

							// Free response message (and clean memory in order to reallocate later)
							status = UDSApi::MsgFree_2013(response_msg);
							Console::WriteLine("[UDS] Free response message: {0}", UDS_STATUS_OK_KO(status));
							break;
						default:
							Console::WriteLine("[UDS] Unknown service (0x{0:x2})", *request_msg.links.service_id);
							break;
						}
					}
				}

				// Free request message (and clean memory in order to reallocate later)
				status = UDSApi::MsgFree_2013(request_msg);

				Console::WriteLine("[UDS] Free request message: {0}", UDS_STATUS_OK_KO(status));
			} while (!UDSApi::StatusIsOk_2013(read_status, uds_status::PUDS_STATUS_NO_MESSAGE));
		}
	} while (t_params->stop_task == false);

	// Close receive event
	if (IntPtr::Size == 4)
	{
		UInt32 tmp_buffer = 0;
		status = UDSApi::SetValue_2013(t_params->server_handle, uds_parameter::PUDS_PARAMETER_RECEIVE_EVENT, tmp_buffer, sizeof(UInt32));
	}
	else if (IntPtr::Size == 8)
	{
		Int64 tmp_buffer = 0;
		array<Byte>^ byte_array = BitConverter::GetBytes(tmp_buffer);
		status = UDSApi::SetValue_2013(t_params->server_handle, uds_parameter::PUDS_PARAMETER_RECEIVE_EVENT, byte_array, sizeof(UInt64));
	}

	Console::WriteLine("[UDS] Stop UDS receive event: {0}", UDS_STATUS_OK_KO(status));
	receive_event.Close();
	Console::WriteLine("[UDS] Close UDS receive event: {0}", UDS_STATUS_OK_KO(status));
}

/// <summary>Entry point of the program, start a small server which handle UDS testerpresent request and isotp request</summary>
/// <returns>By convention, return success.</returns>
int main(array<System::String^>^ args)
{
	uds_status status;
	Thread^ uds_server;
	Thread^ isotp_server;
	task_params^ t_params = gcnew task_params();
	StringBuilder^ buffer = gcnew StringBuilder(Global::BUFFER_SIZE);
	int keyboard_res;

	// Initialize variables
	t_params->server_handle = cantp_handle::PCANTP_HANDLE_USBBUS2; // TODO: modify the value according to your available PCAN devices.
	t_params->server_address = (UInt32)uds_address::PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_1;
	t_params->stop_task = false;

	// Print version informations
	status = UDSApi::GetValue_2013(cantp_handle::PCANTP_HANDLE_NONEBUS, uds_parameter::PUDS_PARAMETER_API_VERSION, buffer, Global::BUFFER_SIZE);

	Console::WriteLine("PCAN-UDS API Version - {0}: {1}", buffer, UDS_STATUS_OK_KO(status));

	// Initialize server
	status = UDSApi::Initialize_2013(t_params->server_handle, cantp_baudrate::PCANTP_BAUDRATE_500K, (cantp_hwtype)0, 0, 0);

	Console::WriteLine("Initialize channel: {0}", UDS_STATUS_OK_KO(status));

	// Start uds and isotp servers
	uds_server = gcnew Thread(gcnew ParameterizedThreadStart(uds_server_task));
	uds_server->Start(t_params);
	isotp_server = gcnew Thread(gcnew ParameterizedThreadStart(isotp_server_task));
	isotp_server->Start(t_params);

	// Read while user do not press Q
	Console::WriteLine("Start listening, press Q to quit.");
	t_params->stop_task = false;
	do
	{
		// Quit when user press Q
		if (Console::KeyAvailable)
		{
			keyboard_res = Console::ReadKey().KeyChar;
			if (keyboard_res == 'Q' || keyboard_res == 'q')
			{
				t_params->stop_task = true;
			}
		}
	} while (t_params->stop_task == false);

	// Close threads
	isotp_server->Join();
	uds_server->Join();


	// Close channel
	status = UDSApi::Uninitialize_2013(t_params->server_handle);

	Console::WriteLine("Uninitialize channel: {0}", UDS_STATUS_OK_KO(status));

	// Exit
	Console::WriteLine("Press any key to continue...");
	Console::Read();
}
