
#include "PCANBasicCLR.h"
#include "PCAN-ISO-TP-CLR_2016.h"
#include "PCAN-UDS-CLR_2013.h"


using namespace System;
using namespace System::IO;
using namespace System::Threading;
using namespace System::Runtime::InteropServices;
using namespace Peak::Can::IsoTp;
using namespace Peak::Can::Uds;

ref class Global
{
public:
	static const int BUFFER_SIZE = 256;
	static const int MSG_SIZE = 4;
	static String^ ISOTP_RESPONSE_MSG = "PONG";
	static String^ ISOTP_REQUEST_MSG = "PING";
};

static String^ OK_KO(bool test)
{
	if (test)
		return "OK";
	return "KO";
}
static String^ UDS_STATUS_OK_KO(uds_status test)
{
	return OK_KO(UDSApi::StatusIsOk_2013(test));
}
static String^ ISOTP_STATUS_OK_KO(cantp_status test)
{
	return OK_KO(CanTpApi::StatusIsOk_2016(test));
}

/// <summary>UDS client task: request TesterPresent service several time</summary>
/// <param name="param">client channel handle</param>
static void uds_client_task(Object^ param)
{

	// Initialize variables
	uds_msgconfig config_physical = {};
	uds_msg msg_request = {};
	uds_msg request_confirmation = {};
	uds_msg response = {};
	cantp_handle client_handle = safe_cast<cantp_handle>(safe_cast<int>(param));
	uds_status status;
	int i;

	// Initialize a physical configuration
	config_physical.can_id = uds_can_id::PUDS_CAN_ID_ISO_15765_4_PHYSICAL_REQUEST_1;
	config_physical.can_msgtype = cantp_can_msgtype::PCANTP_CAN_MSGTYPE_STANDARD;
	config_physical.nai.protocol = uds_msgprotocol::PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL;
	config_physical.nai.target_type = cantp_isotp_addressing::PCANTP_ISOTP_ADDRESSING_PHYSICAL;
	config_physical.type = uds_msgtype::PUDS_MSGTYPE_USDT;
	config_physical.nai.source_addr = uds_address::PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
	config_physical.nai.target_addr = uds_address::PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_1;
	config_physical.nai.extension_addr = 0;

	// Execute TesterPresent and wait response
	for (i = 1; i <= 10; i++)
	{
		status = UDSApi::SvcTesterPresent_2013(client_handle, config_physical, msg_request);
		Console::WriteLine("[UDS] Execute TesterPresent service ({0}): {1}", i, UDS_STATUS_OK_KO(status));
		status = UDSApi::WaitForService_2013(client_handle, msg_request, response, request_confirmation);
		Console::WriteLine("[UDS] Wait for service ({0}): {1}", i, UDS_STATUS_OK_KO(status));
		status = UDSApi::MsgFree_2013(msg_request);
		Console::WriteLine("[UDS] Free request message: {0}", UDS_STATUS_OK_KO(status));
		status = UDSApi::MsgFree_2013(request_confirmation);
		Console::WriteLine("[UDS] Free request confirmation message: {0}", UDS_STATUS_OK_KO(status));
		status = UDSApi::MsgFree_2013(response);
		Console::WriteLine("[UDS] Free response message: {0}", UDS_STATUS_OK_KO(status));
	}
}

private ref class ReceptionWaitHandleWrapper : public System::Threading::WaitHandle
{
public:
	ReceptionWaitHandleWrapper(IntPtr handle)
	{
		this->SafeWaitHandle = gcnew Microsoft::Win32::SafeHandles::SafeWaitHandle(handle, false);
	}
};

/// <summary>ISOTP: wait an ISOTP "PONG" message</summary>
/// <param name="client_handle">client channel handle</param>
/// <param name="repeat_wait_counter">maximum number of WaitForSingleObject to do (to avoid infinite loop)</param>
/// <returns>if we received a PONG message (true) or not (false)</returns>
static bool wait_isotp_pong_response(cantp_handle client_handle, int repeat_wait_counter)
{

	// Init variables
	cantp_msg rx_msg = {};
	int repeat = 0;
	bool get_pong_msg = false;
	cantp_status status;
	bool wait_result;
	cantp_timestamp timestamp_buffer;

	// Get ISOTP receive event
	array<Byte>^ byte_array = gcnew array<Byte>(IntPtr::Size);
	status = CanTpApi::GetValue_2016(client_handle, cantp_parameter::PCANTP_PARAMETER_RECEIVE_EVENT, byte_array, sizeof(Byte) * IntPtr::Size);
	IntPtr handle_ptr = IntPtr::Zero;
	if (IntPtr::Size == 4)
	{
		handle_ptr = IntPtr(BitConverter::ToInt32(byte_array, 0));
	}
	else
	{
		handle_ptr = IntPtr(BitConverter::ToInt64(byte_array, 0));
	}
	ReceptionWaitHandleWrapper^ receive_event = gcnew ReceptionWaitHandleWrapper(handle_ptr);
	Console::WriteLine("[ISOTP] Get isotp receive event parameter: {0}", ISOTP_STATUS_OK_KO(status));

	// Wait a receive event on isotp message
	do
	{
		wait_result = receive_event->WaitOne(1000);
		Console::WriteLine("[ISOTP] Wait a receive event from isotp: {0}", OK_KO(wait_result));

		// Read ISOTP messages
		do
		{
			status = CanTpApi::Read_2016(client_handle, rx_msg, timestamp_buffer, cantp_msgtype::PCANTP_MSGTYPE_NONE);

			// Check if we received a "PONG" message
			if (!CanTpApi::StatusIsOk_2016(status, cantp_status::PCANTP_STATUS_NO_MESSAGE))
			{
				Console::WriteLine("[ISOTP] Read ISOTP message: {0}", ISOTP_STATUS_OK_KO(status));

				bool msgok = false;
				if (CanTpApi::StatusIsOk_2016(status) && rx_msg.msgdata.any->length == Global::MSG_SIZE)
				{
					msgok = true;
					for (int i = 0; msgok && i < Global::MSG_SIZE; i++) {
						if (rx_msg.msgdata.any->data[i] != Global::ISOTP_RESPONSE_MSG[i]) {
							msgok = false;
							break;
						}
					}
				}
				if (msgok)
				{
					Console::WriteLine("[ISOTP] Message contains \"{0}{1}{2}{3}\": OK", Convert::ToChar(rx_msg.msgdata.any->data[0]), Convert::ToChar(rx_msg.msgdata.any->data[1]), Convert::ToChar(rx_msg.msgdata.any->data[2]), Convert::ToChar(rx_msg.msgdata.any->data[3]));
					get_pong_msg = true;
				}

				status = CanTpApi::MsgDataFree_2016(rx_msg);
				Console::WriteLine("[ISOTP] Free RX message: {0}", ISOTP_STATUS_OK_KO(status));
			}
		} while (!CanTpApi::StatusIsOk_2016(status, cantp_status::PCANTP_STATUS_NO_MESSAGE));
		repeat++;
	} while (get_pong_msg == false && repeat < repeat_wait_counter);
	return get_pong_msg;
}

/// <summary>ISOTP client task: send "PING" and wait "PONG" message several times</summary>
/// <param name="param">client channel handle</param>
static void isotp_client_task(Object^ param)
{

	// Initialize variables
	cantp_msg tx_msg = {};
	cantp_mapping mapping = {};
	cantp_mapping reverse_mapping = {};
	cantp_status status = cantp_status::PCANTP_STATUS_UNKNOWN;
	cantp_handle client_handle = safe_cast<cantp_handle>(safe_cast<int>(param));
	bool response;
	int i;

	// Create a isotp receive event
	System::Threading::AutoResetEvent receive_event(false);
	if (IntPtr::Size == 4)
	{
		UInt32 tmp_buffer = Convert::ToUInt32(receive_event.SafeWaitHandle->DangerousGetHandle().ToInt32());
		status = CanTpApi::SetValue_2016(client_handle, cantp_parameter::PCANTP_PARAMETER_RECEIVE_EVENT, tmp_buffer, sizeof(UInt32));
	}
	else if (IntPtr::Size == 8)
	{
		Int64 tmp_buffer = receive_event.SafeWaitHandle->DangerousGetHandle().ToInt64();
		array<Byte>^ byte_array = BitConverter::GetBytes(tmp_buffer);
		status = CanTpApi::SetValue_2016(client_handle, cantp_parameter::PCANTP_PARAMETER_RECEIVE_EVENT, byte_array, sizeof(UInt64));
	}
	Console::WriteLine("[ISOTP] Set isotp receive event parameter: {0}", ISOTP_STATUS_OK_KO(status));

	// Create a simple isotp physical mapping:
	//    - Source 0xA1 (client), target 0xA2 (server), CAN id 0xA1, CAN ID flow control 0xA2
	//    - Diagnostic message in a classic format
	mapping.can_id = 0xA1;
	mapping.can_id_flow_ctrl = 0xA2;
	mapping.can_msgtype = cantp_can_msgtype::PCANTP_CAN_MSGTYPE_STANDARD;
	mapping.netaddrinfo.extension_addr = 0x00;
	mapping.netaddrinfo.format = cantp_isotp_format::PCANTP_ISOTP_FORMAT_NORMAL;
	mapping.netaddrinfo.msgtype = cantp_isotp_msgtype::PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC;
	mapping.netaddrinfo.source_addr = 0xA1;
	mapping.netaddrinfo.target_addr = 0xA2;
	mapping.netaddrinfo.target_type = cantp_isotp_addressing::PCANTP_ISOTP_ADDRESSING_PHYSICAL;

	// Create the associated isotp reversed mapping:
	reverse_mapping = mapping;
	reverse_mapping.can_id = mapping.can_id_flow_ctrl;
	reverse_mapping.can_id_flow_ctrl = mapping.can_id;
	reverse_mapping.netaddrinfo.source_addr = mapping.netaddrinfo.target_addr;
	reverse_mapping.netaddrinfo.target_addr = mapping.netaddrinfo.source_addr;

	// Add ISOTP mappings on channels
	status = CanTpApi::AddMapping_2016(client_handle, mapping);
	Console::WriteLine("[ISOTP] Add a simple isotp mapping: {0}", ISOTP_STATUS_OK_KO(status));
	status = CanTpApi::AddMapping_2016(client_handle, reverse_mapping);
	Console::WriteLine("[ISOTP] Add the reverse isotp mapping: {0}", ISOTP_STATUS_OK_KO(status));

	// Initialize ISOTP Tx message containing "PING"
	status = CanTpApi::MsgDataAlloc_2016(tx_msg, cantp_msgtype::PCANTP_MSGTYPE_ISOTP);
	Console::WriteLine("[ISOTP] Allocate ISOTP tx message: {0}", ISOTP_STATUS_OK_KO(status));
	array<Byte>^ msg_array = Encoding::ASCII->GetBytes(Global::ISOTP_REQUEST_MSG);
	status = CanTpApi::MsgDataInit_2016(tx_msg, mapping.can_id, mapping.can_msgtype, Global::MSG_SIZE, msg_array, mapping.netaddrinfo);
	Console::WriteLine("[ISOTP] Initialize ISOTP tx message: {0}", ISOTP_STATUS_OK_KO(status));

	// Send "PING" ISOTP message and wait "PONG" response
	for (i = 1; i <= 10; i++)
	{
		status = CanTpApi::Write_2016(client_handle, tx_msg);
		Console::WriteLine("[ISOTP] Send ISOTP \"PING\" message ({0}): {1}", i, ISOTP_STATUS_OK_KO(status));
		response = wait_isotp_pong_response(client_handle, 3);
		Console::WriteLine("[ISOTP] Get ISOTP \"PONG\" response ({0}): {1}", i, OK_KO(response));
	}

	// Close receive event
	if (IntPtr::Size == 4)
	{
		UInt32 tmp_buffer = 0;
		status = CanTpApi::SetValue_2016(client_handle, cantp_parameter::PCANTP_PARAMETER_RECEIVE_EVENT, tmp_buffer, sizeof(UInt32));
	}
	else if (IntPtr::Size == 8)
	{
		Int64 tmp_buffer = 0;
		array<Byte>^ byte_array = BitConverter::GetBytes(tmp_buffer);
		status = CanTpApi::SetValue_2016(client_handle, cantp_parameter::PCANTP_PARAMETER_RECEIVE_EVENT, byte_array, sizeof(UInt64));
	}
	Console::WriteLine("[ISOTP] Stop ISOTP receive event: {0}", ISOTP_STATUS_OK_KO(status));
	receive_event.Close();
	Console::WriteLine("[ISOTP] Close ISOTP receive event: {0}", ISOTP_STATUS_OK_KO(status));

	// Free messages
	status = CanTpApi::MsgDataFree_2016(tx_msg);
	Console::WriteLine("[ISOTP] Free ISOTP TX message: {0}", ISOTP_STATUS_OK_KO(status));
}

/// <summary>Entry point of the program, start a UDS channel, ask in the same time UDS TesterPresent and isotp request</summary>
/// <returns>By convention, return success.</returns>
int main(array<System::String^>^ args)
{
	StringBuilder^ buffer;
	uds_status status;
	cantp_handle client_handle;
	Thread^ uds_client;
	Thread^ isotp_client;

	// Initialize variables
	client_handle = cantp_handle::PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.
	buffer = gcnew StringBuilder(Global::BUFFER_SIZE);

	// Print version informations
	status = UDSApi::GetValue_2013(cantp_handle::PCANTP_HANDLE_NONEBUS, uds_parameter::PUDS_PARAMETER_API_VERSION, buffer, Global::BUFFER_SIZE);
	Console::WriteLine("PCAN-UDS API Version - {0}: {1}", buffer, UDS_STATUS_OK_KO(status));

	// Initialize client channel
	status = UDSApi::Initialize_2013(client_handle, cantp_baudrate::PCANTP_BAUDRATE_500K, (cantp_hwtype)0, 0, 0);
	Console::WriteLine("Initialize channel: {0}", UDS_STATUS_OK_KO(status));

	// Start uds and isotp clients
	uds_client = gcnew Thread(gcnew ParameterizedThreadStart(uds_client_task));
	uds_client->Start((int)client_handle);

	isotp_client = gcnew Thread(gcnew ParameterizedThreadStart(isotp_client_task));
	isotp_client->Start((int)client_handle);

	isotp_client->Join();
	uds_client->Join();

	// Close channel
	status = UDSApi::Uninitialize_2013(client_handle);
	Console::WriteLine("Uninitialize channel: {0}", UDS_STATUS_OK_KO(status));

	// Exit
	Console::WriteLine("Press any key to continue...");
	Console::Read();
}
