
#include "PCANBasicCLR.h"
#include "PCAN-ISO-TP-CLR_2016.h"
#include "PCAN-UDS-CLR_2013.h"


using namespace System;
using namespace System::IO;
using namespace System::Threading;
using namespace Peak::Can::IsoTp;
using namespace Peak::Can::Uds;


ref class Global
{
public:
	const static cantp_bitrate PCAN_BITRATE = "f_clock=40000000,nom_brp=2,nom_tseg1=63,nom_tseg2=16,nom_sjw=16,data_brp=2,data_tseg1=7,data_tseg2=2,data_sjw=2";
};

static String^ OK_KO(bool test)
{
	if (test)
		return "OK";
	return "KO";
}

static String^ STATUS_OK_KO(uds_status test)
{
	return OK_KO(UDSApi::StatusIsOk_2013(test));
}

/// <summary>
///  Entry point of the program, start a small server wich only support ReadDataByPeriodicIdentifier service.
///  This example use a specific addressing. It receives request from test equipement (0xF1 to 0xC1) in 29b fixed
///  normal addressing and sends responses for each periodic data identifier with 0x1F22C1F1 can identifier (UUDT).
/// </summary>
/// <returns>By convention, return success.</returns>
int main(array<System::String^>^ args)
{
	uds_status status;
	cantp_handle server_handle;
	UInt32 server_address;
	uds_msgconfig service_response_config = {};
	bool stop;
	bool wait_result;
	uds_status read_status;
	uds_msg request_msg = {};
	uds_msg service_response_msg = {};
	uds_msg periodic_response = {};
	uds_msgconfig periodic_msg_config = {};
	int keyboard_res;
	UInt32 padding_value;
	UInt32 can_tx_dl;
	Byte periodic_data_identifier;
	UInt32 timeout_value;
	UInt32 periodic_data_identifier_length;
	UInt32 i;

	// Initialize variables
	server_handle = PCANTP_HANDLE_USBBUS2; // TODO: modify the value according to your available PCAN devices.
	server_address = 0xC1;

	// Initialize server
	status = UDSApi::InitializeFD_2013(server_handle, Global::PCAN_BITRATE);
	Console::WriteLine("Initialize channel: {0}", STATUS_OK_KO(status));

	// Set server address parameter
	status = UDSApi::SetValue_2013(server_handle, uds_parameter::PUDS_PARAMETER_SERVER_ADDRESS, server_address, sizeof(UInt32));
	Console::WriteLine("Set server address: {0}", STATUS_OK_KO(status));

	// Set a padding value
	padding_value = 0xFF;
	status = UDSApi::SetValue_2013(server_handle, uds_parameter::PUDS_PARAMETER_CAN_PADDING_VALUE, padding_value, sizeof(UInt32));
	Console::WriteLine("Set padding value: {0}", STATUS_OK_KO(status));

	// Define CAN_TX_DL=15
	can_tx_dl = 15;
	status = UDSApi::SetValue_2013(server_handle, uds_parameter::PUDS_PARAMETER_CAN_TX_DL, can_tx_dl, sizeof(UInt32));
	Console::WriteLine("Set CAN TX DL: {0}", STATUS_OK_KO(status));

	// Set UDS timeouts
	timeout_value = 5000;
	status = UDSApi::SetValue_2013(server_handle, uds_parameter::PUDS_PARAMETER_TIMEOUT_REQUEST, timeout_value, sizeof(UInt32));
	Console::WriteLine("Set request timeout(ms): {0}", STATUS_OK_KO(status));
	status = UDSApi::SetValue_2013(server_handle, uds_parameter::PUDS_PARAMETER_TIMEOUT_RESPONSE, timeout_value, sizeof(UInt32));
	Console::WriteLine("Set response timeout(ms): {0}", STATUS_OK_KO(status));

	// Set a receive event
	System::Threading::AutoResetEvent receive_event(false);
	if (IntPtr::Size == 4)
	{
		UInt32 tmp_buffer = Convert::ToUInt32(receive_event.SafeWaitHandle->DangerousGetHandle().ToInt32());
		status = UDSApi::SetValue_2013(server_handle, uds_parameter::PUDS_PARAMETER_RECEIVE_EVENT, tmp_buffer, sizeof(UInt32));
	}
	else if (IntPtr::Size == 8)
	{
		Int64 tmp_buffer = receive_event.SafeWaitHandle->DangerousGetHandle().ToInt64();
		array<Byte>^ byte_array = BitConverter::GetBytes(tmp_buffer);
		status = UDSApi::SetValue_2013(server_handle, uds_parameter::PUDS_PARAMETER_RECEIVE_EVENT, byte_array, sizeof(UInt64));
	}
	Console::WriteLine("Set receive event parameter: {0}", STATUS_OK_KO(status));

	// Initialize service response configuration
	service_response_config.can_id = (UInt32)0xFFFFFFFF;
	service_response_config.can_msgtype = PCANTP_CAN_MSGTYPE_EXTENDED | PCANTP_CAN_MSGTYPE_FD | PCANTP_CAN_MSGTYPE_BRS;
	service_response_config.nai.protocol = PUDS_MSGPROTOCOL_ISO_15765_2_29B_FIXED_NORMAL;
	service_response_config.nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL;
	service_response_config.type = PUDS_MSGTYPE_USDT;
	service_response_config.nai.source_addr = (UInt16)server_address;
	service_response_config.nai.target_addr = (UInt16)PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
	service_response_config.nai.extension_addr = 0;

	// Initialize responses configuration (for each periodic data identifier contained in the request)
	periodic_msg_config.can_id = 0x1F22C1F1;
	periodic_msg_config.can_msgtype = PCANTP_CAN_MSGTYPE_EXTENDED | PCANTP_CAN_MSGTYPE_FD | PCANTP_CAN_MSGTYPE_BRS;
	periodic_msg_config.nai.protocol = PUDS_MSGPROTOCOL_ISO_15765_2_29B_NORMAL;
	periodic_msg_config.nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL;
	periodic_msg_config.type = PUDS_MSGTYPE_UUDT;
	periodic_msg_config.nai.source_addr = (UInt16)server_address;
	periodic_msg_config.nai.target_addr = (UInt16)PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
	periodic_msg_config.nai.extension_addr = 0;

	// Add a filter for 0x1F22C1F1 can id (in order to receive UUDT loopback messages)
	status = UDSApi::AddCanIdFilter_2013(server_handle, periodic_msg_config.can_id);
	Console::WriteLine("Add can identifier filter: {0}", STATUS_OK_KO(status));

	// Read while user do not press Q
	Console::WriteLine("Start listening, press Q to quit.");
	stop = false;
	do
	{

		// Wait a receive event on receiver
		//	note: timeout is used to check keyboard hit.
		wait_result = receive_event.WaitOne(1000);

		// If we get a receive event
		if (wait_result)
		{

			do
			{

				// Read first available message (no filtering based on message's type is set):
				read_status = UDSApi::Read_2013(server_handle, request_msg);
				Console::WriteLine("Try to read a message: {0}", STATUS_OK_KO(read_status));
				if (UDSApi::StatusIsOk_2013(read_status))
				{

					// We receive a request, check its length and if it is not a loopback message
					if (request_msg.msg.msgdata.any->length >= 1 && (request_msg.msg.msgdata.any->flags & PCANTP_MSGFLAG_LOOPBACK) == 0)
					{
						Byte service_id = request_msg.msg.msgdata.any->data[0];

						// This is a valid request, switch services
						switch (service_id)
						{
						case (Byte)uds_service::PUDS_SERVICE_SI_ReadDataByPeriodicIdentifier:

							// Allocates service response message
							status = UDSApi::MsgAlloc_2013(service_response_msg, service_response_config, 1);
							if (UDSApi::StatusIsOk_2013(status))
							{
								service_response_msg.msg.msgdata.any->data[0] = PUDS_SERVICE_SI_ReadDataByPeriodicIdentifier + UDSApi::PUDS_SI_POSITIVE_RESPONSE;
							}
							Console::WriteLine("Prepare response message for ReadDataByPeriodicIdentifier service: {0}", STATUS_OK_KO(status));

							// Write service response message
							status = UDSApi::Write_2013(server_handle, service_response_msg);
							Console::WriteLine("Write response message for ReadDataByPeriodicIdentifier service: {0}", STATUS_OK_KO(status));

							// Free response message (and clean memory in order to reallocate later)
							status = UDSApi::MsgFree_2013(service_response_msg);
							Console::WriteLine("Free response message: {0}", STATUS_OK_KO(status));

							// Sends a message for each data identifier with a specific addressing.
							periodic_data_identifier_length = request_msg.msg.msgdata.any->length - 2;
							for (i = 0; i < periodic_data_identifier_length; i++)
							{

								// Allocates and prepares message with dummy data
								status = UDSApi::MsgAlloc_2013(periodic_response, periodic_msg_config, 5);
								if (UDSApi::StatusIsOk_2013(status))
								{
									periodic_data_identifier = request_msg.links.param[i + 1];
									periodic_response.msg.msgdata.any->data[0] = periodic_data_identifier;
									periodic_response.msg.msgdata.any->data[1] = 0x12;
									periodic_response.msg.msgdata.any->data[2] = 0x34;
									periodic_response.msg.msgdata.any->data[3] = 0x56;
									periodic_response.msg.msgdata.any->data[4] = 0x78;

									Console::WriteLine("Allocates message for 0x{0:x2} periodic data identifier: {1}", periodic_data_identifier, STATUS_OK_KO(status));
									status = UDSApi::Write_2013(server_handle, periodic_response);
									Console::WriteLine("Write message for 0x{0:x2} periodic data identifier: {1}", periodic_data_identifier, STATUS_OK_KO(status));
									status = UDSApi::MsgFree_2013(periodic_response);
									Console::WriteLine("Free message for 0x{0:x2} periodic data identifier: {1}", periodic_data_identifier, STATUS_OK_KO(status));
								}
							}
							break;
						default:
							Console::WriteLine("Unknown service (0x{0:x2})", service_id);
							break;
						}
					}
				}

				// Free request message (in order to reallocate later)
				status = UDSApi::MsgFree_2013(request_msg);
				Console::WriteLine("Free request message: {0}", STATUS_OK_KO(status));
			} while (!UDSApi::StatusIsOk_2013(read_status, PUDS_STATUS_NO_MESSAGE, false));
		}

		// Quit when user press Q
		if (Console::KeyAvailable)
		{
			keyboard_res = Console::ReadKey().KeyChar;
			if (keyboard_res == 'Q' || keyboard_res == 'q')
			{
				stop = true;
			}
		}
	} while (stop == false);

	// Close receive event
	if (IntPtr::Size == 4)
	{
		UInt32 tmp_buffer = 0;
		status = UDSApi::SetValue_2013(server_handle, PUDS_PARAMETER_RECEIVE_EVENT, tmp_buffer, sizeof(UInt32));
	}
	else if (IntPtr::Size == 8)
	{
		Int64 tmp_buffer = 0;
		array<Byte>^ byte_array = BitConverter::GetBytes(tmp_buffer);
		status = UDSApi::SetValue_2013(server_handle, PUDS_PARAMETER_RECEIVE_EVENT, byte_array, sizeof(UInt64));
	}
	Console::WriteLine("Stop receive event: {0}", STATUS_OK_KO(status));
	receive_event.Close();
	Console::WriteLine("Close receive event: {0}", STATUS_OK_KO(status));

	// Close server
	status = UDSApi::Uninitialize_2013(server_handle);
	Console::WriteLine("Uninitialize channel: {0}", STATUS_OK_KO(status));

	// Exit
	Console::WriteLine("Press any key to continue...");
	Console::Read();
}


