
#include "PCANBasicCLR.h"
#include "PCAN-ISO-TP-CLR_2016.h"
#include "PCAN-UDS-CLR_2013.h"


using namespace System;
using namespace System::IO;
using namespace System::Threading;
using namespace System::Runtime::InteropServices;
using namespace Peak::Can::IsoTp;
using namespace Peak::Can::Uds;

ref class Global
{
public:
	static cantp_bitrate PCAN_BITRATE = "f_clock=40000000,nom_brp=2,nom_tseg1=63,nom_tseg2=16,nom_sjw=16,data_brp=2,data_tseg1=7,data_tseg2=2,data_sjw=2";
	static long long RESPONSES_TIMEOUT_MS = 5000;
};

static String^ OK_KO(bool test)
{
	if (test)
		return "OK";
	return "KO";
}

static String^ STATUS_OK_KO(uds_status test)
{
	return OK_KO(UDSApi::StatusIsOk_2013(test));
}


/// <summary>
///  Entry point of the program, start a UDS channel, ask ReadDataByPeriodicIdentifier service.
///  It sends a request from 0xF1 to 0xC1 address in 29b fixed with normal addressing. Then read
///  responses from 0x1F22C1F1 can identifier (UUDT).
/// </summary>
/// <returns>By convention, return success.</returns>
int main(array<System::String^>^ args)
{
	uds_status status;
	cantp_handle client_handle;
	UInt32 can_tx_dl;
	UInt32 timeout_value;
	Byte response_count;
	long long start_reading;
	UInt32 i;

	// Initialize variables
	client_handle = PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.
	uds_msgconfig request_config = {};
	uds_msg msg_request = {};
	uds_msg request_confirmation = {};
	uds_msg service_response = {};
	uds_msg periodic_response = {};

	// Initialize client
	status = UDSApi::InitializeFD_2013(client_handle, Global::PCAN_BITRATE);
	Console::WriteLine("Initialize channel: {0}", STATUS_OK_KO(status));

	// Define CAN_TX_DL
	can_tx_dl = 15;
	status = UDSApi::SetValue_2013(client_handle, PUDS_PARAMETER_CAN_TX_DL, can_tx_dl, sizeof(UInt32));
	Console::WriteLine("Set CAN TX DL: {0}", STATUS_OK_KO(status));

	// Check CAN_TX_DL
	can_tx_dl = 0;
	status = UDSApi::GetValue_2013(client_handle, PUDS_PARAMETER_CAN_TX_DL, can_tx_dl, sizeof(can_tx_dl));
	Console::WriteLine("Check new CAN TX DL value({0}): {1}", can_tx_dl, STATUS_OK_KO(status));

	// Set UDS timeouts
	timeout_value = 5000;
	status = UDSApi::SetValue_2013(client_handle, PUDS_PARAMETER_TIMEOUT_REQUEST, timeout_value, sizeof(UInt32));
	Console::WriteLine("Set request timeout(ms): {0}", STATUS_OK_KO(status));
	status = UDSApi::SetValue_2013(client_handle, PUDS_PARAMETER_TIMEOUT_RESPONSE, timeout_value, sizeof(UInt32));
	Console::WriteLine("Set response timeout(ms): {0}", STATUS_OK_KO(status));

	// Check timeouts
	timeout_value = 0;
	status = UDSApi::GetValue_2013(client_handle, PUDS_PARAMETER_TIMEOUT_REQUEST, timeout_value, sizeof(UInt32));
	Console::WriteLine("Get new request timeout({0}ms): {1}", timeout_value, STATUS_OK_KO(status));
	timeout_value = 0;
	status = UDSApi::GetValue_2013(client_handle, PUDS_PARAMETER_TIMEOUT_RESPONSE, timeout_value, sizeof(UInt32));
	Console::WriteLine("Get new response timeout({0}ms): {1}", timeout_value, STATUS_OK_KO(status));

	// Initialize the request configuration
	request_config.can_id = 0xFFFFFFFF;
	request_config.can_msgtype = PCANTP_CAN_MSGTYPE_EXTENDED | PCANTP_CAN_MSGTYPE_FD | PCANTP_CAN_MSGTYPE_BRS;
	request_config.nai.protocol = PUDS_MSGPROTOCOL_ISO_15765_2_29B_FIXED_NORMAL;
	request_config.nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL;
	request_config.type = PUDS_MSGTYPE_USDT;
	request_config.nai.source_addr = (UInt16)PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
	request_config.nai.target_addr = 0xC1;
	request_config.nai.extension_addr = 0;

	// Add a filter for 0x1F22C1F1 can id (in order to receive UUDT messages)
	status = UDSApi::AddCanIdFilter_2013(client_handle, 0x1F22C1F1);
	Console::WriteLine("Add can identifier filter: {0}", STATUS_OK_KO(status));

	// Execute ReadDataByPeriodicIdentifier and wait service response
	array<Byte>^ periodic_data_identifier = { 0x12, 0x34, 0x56, 0x78 };

	status = UDSApi::SvcReadDataByPeriodicIdentifier_2013(client_handle, request_config, msg_request, UDSApi::uds_svc_param_rdbpi::PUDS_SVC_PARAM_RDBPI_SAMR, periodic_data_identifier, 4);
	Console::WriteLine("Execute ReadDataByPeriodicIdentifier service: {0}", STATUS_OK_KO(status));
	status = UDSApi::WaitForService_2013(client_handle, msg_request, service_response, request_confirmation);
	Console::WriteLine("Wait for service (confirmation and response): {0}", STATUS_OK_KO(status));

	// Read responses for each given periodic data identifier
	response_count = 0;
	start_reading = DateTime::UtcNow.Ticks / TimeSpan::TicksPerMillisecond;
	do
	{
		status = UDSApi::Read_2013(client_handle, periodic_response);
		if (UDSApi::StatusIsOk_2013(status, PUDS_STATUS_OK, false) && periodic_response.msg.msgdata.any->length > 0)
		{
			response_count++;
			Console::Write("Read response for 0x{0:x2} periodic data identifier: [", periodic_response.msg.msgdata.any->data[0]);
			for (i = 0; i < periodic_response.msg.msgdata.any->length; i++)
			{
				Console::Write(" 0x{0:x2}", periodic_response.msg.msgdata.any->data[i]);
			}
			Console::WriteLine(" ]");
			status = UDSApi::MsgFree_2013(periodic_response);
			Console::WriteLine("Free response buffer: {0}", STATUS_OK_KO(status));
		}

		// Wait for 4 responses or timeout
	} while (response_count < 4 && (DateTime::UtcNow.Ticks / TimeSpan::TicksPerMillisecond) - start_reading < Global::RESPONSES_TIMEOUT_MS);

	// Free message structures
	status = UDSApi::MsgFree_2013(msg_request);
	Console::WriteLine("Free request message: {0}", STATUS_OK_KO(status));
	status = UDSApi::MsgFree_2013(request_confirmation);
	Console::WriteLine("Free request confirmation: {0}", STATUS_OK_KO(status));
	status = UDSApi::MsgFree_2013(service_response);
	Console::WriteLine("Free service response: {0}", STATUS_OK_KO(status));

	// Close client
	status = UDSApi::Uninitialize_2013(client_handle);
	Console::WriteLine("Uninitialize channel: {0}", STATUS_OK_KO(status));

	// Exit
	Console::WriteLine("Press any key to continue...");
	Console::Read();
}
