
#include "PCANBasicCLR.h"
#include "PCAN-ISO-TP-CLR_2016.h"
#include "PCAN-UDS-CLR_2013.h"


using namespace System;
using namespace System::IO;
using namespace System::Threading;
using namespace Peak::Can::IsoTp;
using namespace Peak::Can::Uds;

ref class Global
{
public:
	const static int BUFFER_SIZE = 256;
	const static UInt32 ISOTP_PING_MSG_SIZE = 4;
	const static UInt32 CAN_ID_START_RANGE = 1;
	const static UInt32 CAN_ID_STOP_RANGE = 10;
	const int static PING_TIMING_MS = 500;
	static String^ ISOTP_PING_MSG = "PING";
};

static String^ OK_KO(bool test)
{
	if (test)
		return "OK";
	return "KO";
}

static String^ UDS_STATUS_OK_KO(uds_status test)
{
	return OK_KO(UDSApi::StatusIsOk_2013(test));
}

static String^ ISOTP_STATUS_OK_KO(cantp_status test)
{
	return OK_KO(CanTpApi::StatusIsOk_2016(test));
}

/// <summary>Structure passed as thread parameters</summary>
public ref struct task_params
{
public:
	/// <summary>Server channel handle</summary>
	cantp_handle server_handle;
	/// <summary>Server address</summary>
	UInt32 server_address;
	/// <summary>Determine if the thread should end or not</summary>
	bool stop_task;
};

/// <summary>ISOTP server task: periodically send "PING" message using different CAN identifiers</summary>
/// <param name="parameters">pointer on task_params structures</param>
static void isotp_server_task(Object^ parameters)
{
	// Init variables
	cantp_status status;
	UInt32 can_id = Global::CAN_ID_START_RANGE;
	task_params^ t_params = safe_cast<task_params^>(parameters);
	cantp_msg tx_msg = {};

	// Send loop
	do
	{
		// Wait before sending the next message
		Thread::Sleep(Global::PING_TIMING_MS);

		// Initialize ISOTP Tx message containing "PING"
		status = CanTpApi::MsgDataAlloc_2016(tx_msg, PCANTP_MSGTYPE_CAN);
		Console::WriteLine("[ISOTP] Allocate ISOTP tx message: {0}", ISOTP_STATUS_OK_KO(status));
		array<Byte>^ bytes_array = Encoding::ASCII->GetBytes(Global::ISOTP_PING_MSG);
		status = CanTpApi::MsgDataInit_2016(tx_msg, can_id, PCANTP_CAN_MSGTYPE_STANDARD, Global::ISOTP_PING_MSG_SIZE, bytes_array);
		Console::WriteLine("[ISOTP] Initialize ISOTP tx message: {0}", ISOTP_STATUS_OK_KO(status));

		// Send "PING" message
		status = CanTpApi::Write_2016(t_params->server_handle, tx_msg);
		Console::WriteLine("[ISOTP] Send ISOTP \"PING\" message (can id=0x{0:x}): {1}", can_id, ISOTP_STATUS_OK_KO(status));

		// Free messages
		status = CanTpApi::MsgDataFree_2016(tx_msg);
		Console::WriteLine("[ISOTP] Free ISOTP TX message: {0}", ISOTP_STATUS_OK_KO(status));

		// Update can id for next message
		can_id++;
		if (can_id > Global::CAN_ID_STOP_RANGE)
		{
			can_id = Global::CAN_ID_START_RANGE;
		}
	} while (t_params->stop_task == false);
}

/// <summary>UDS server task: respond TesterPresent request</summary>
/// <param name="parameters">pointer on task_params structures</param>
static void uds_server_task(Object^ parameters)
{
	// Init variables
	uds_msgconfig config_physical = {};
	bool wait_result;
	uds_status status;
	uds_status read_status;
	uds_msg request_msg = {};
	uds_msg response_msg = {};
	task_params^ t_params = safe_cast<task_params^>(parameters);

	// Set server address parameter
	status = UDSApi::SetValue_2013(t_params->server_handle, PUDS_PARAMETER_SERVER_ADDRESS, t_params->server_address, sizeof(UInt32));
	Console::WriteLine("[UDS] Set UDS server address: {0}", UDS_STATUS_OK_KO(status));

	// Set a receive event for UDS
	System::Threading::AutoResetEvent receive_event(false);
	if (IntPtr::Size == 4)
	{
		UInt32 tmp_buffer = Convert::ToUInt32(receive_event.SafeWaitHandle->DangerousGetHandle().ToInt32());
		status = UDSApi::SetValue_2013(t_params->server_handle, PUDS_PARAMETER_RECEIVE_EVENT, tmp_buffer, sizeof(UInt32));
	}
	else if (IntPtr::Size == 8)
	{
		Int64 tmp_buffer = receive_event.SafeWaitHandle->DangerousGetHandle().ToInt64();
		array<Byte>^ byte_array = BitConverter::GetBytes(tmp_buffer);
		status = UDSApi::SetValue_2013(t_params->server_handle, PUDS_PARAMETER_RECEIVE_EVENT, byte_array, sizeof(UInt64));
	}
	Console::WriteLine("[UDS] Set UDS receive event parameter: {0}", UDS_STATUS_OK_KO(status));

	// Initialize a physical configuration
	config_physical.can_id = 0xFFFFFFFF;
	config_physical.can_msgtype = PCANTP_CAN_MSGTYPE_STANDARD;
	config_physical.nai.protocol = PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL;
	config_physical.nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL;
	config_physical.type = PUDS_MSGTYPE_USDT;
	config_physical.nai.extension_addr = 0;

	do
	{

		// Wait a receive event on receiver
		wait_result = receive_event.WaitOne(1000);

		// If we get a receive event
		if (wait_result)
		{
			do
			{

				// Read first available message (no filtering based on message's type is set):
				read_status = UDSApi::Read_2013(t_params->server_handle, request_msg);
				Console::WriteLine("[UDS] Try to read a message: {0}", UDS_STATUS_OK_KO(read_status));
				if (UDSApi::StatusIsOk_2013(read_status))
				{

					// We receive a request, check its length and if it is not a loopback message and if it is a USDT message
					if (request_msg.type == PUDS_MSGTYPE_USDT && request_msg.msg.msgdata.any->length >= 1 && (request_msg.msg.msgdata.any->flags & PCANTP_MSGFLAG_LOOPBACK) == 0)
					{

						// This is a valid request, switch services
						switch (*request_msg.links.service_id)
						{
						case PUDS_SERVICE_SI_TesterPresent:

							// Allocate response message
							status = UDSApi::MsgAlloc_2013(response_msg, config_physical, 2);
							Console::WriteLine("[UDS] Prepare response message for TesterPresent service: {0}", UDS_STATUS_OK_KO(status));

							if (UDSApi::StatusIsOk_2013(status))
							{
								// Fill parameters
								cantp_netaddrinfo nadr = request_msg.msg.msgdata.isotp->netaddrinfo;
								nadr.target_addr = nadr.source_addr;
								nadr.source_addr = (UInt16)t_params->server_address;
								response_msg.msg.msgdata.isotp->netaddrinfo = nadr;
								*response_msg.links.service_id = PUDS_SERVICE_SI_TesterPresent + UDSApi::PUDS_SI_POSITIVE_RESPONSE;
								response_msg.links.param[0] = 0;

								// Write response message
								status = UDSApi::Write_2013(t_params->server_handle, response_msg);
								Console::WriteLine("[UDS] Write response message for TesterPresent service: {0}", UDS_STATUS_OK_KO(status));
							}

							// Free response message (and clean memory in order to reallocate later)
							status = UDSApi::MsgFree_2013(response_msg);
							Console::WriteLine("[UDS] Free response message: {0}", UDS_STATUS_OK_KO(status));
							break;
						default:
							Console::WriteLine("[UDS] Unknown service (0x{0:x2})", request_msg.msg.msgdata.any->data[0]);
							break;
						}
					}
				}

				// Free request message (and clean memory in order to reallocate later)
				status = UDSApi::MsgFree_2013(request_msg);
				Console::WriteLine("[UDS] Free request message: {0}", UDS_STATUS_OK_KO(status));
			} while (!UDSApi::StatusIsOk_2013(read_status, PUDS_STATUS_NO_MESSAGE, false));
		}
	} while (t_params->stop_task == false);

	// Close receive event
	if (IntPtr::Size == 4)
	{
		UInt32 tmp_buffer = 0;
		status = UDSApi::SetValue_2013(t_params->server_handle, PUDS_PARAMETER_RECEIVE_EVENT, tmp_buffer, sizeof(UInt32));
	}
	else if (IntPtr::Size == 8)
	{
		Int64 tmp_buffer = 0;
		array<Byte>^ byte_array = BitConverter::GetBytes(tmp_buffer);
		status = UDSApi::SetValue_2013(t_params->server_handle, PUDS_PARAMETER_RECEIVE_EVENT, byte_array, sizeof(UInt64));
	}
	Console::WriteLine("[UDS] Stop UDS receive event: {0}", UDS_STATUS_OK_KO(status));
	receive_event.Close();
	Console::WriteLine("[UDS] Close UDS receive event: {0}", UDS_STATUS_OK_KO(status));
}

/// <summary>Entry point of the program, start a small server which handle UDS testerpresent request and periodically send isotp messages</summary>
/// <returns>By convention, return success.</returns>
int main(array<System::String^>^ args)
{
	uds_status status;
	task_params^ t_params = gcnew task_params();
	StringBuilder^ buffer = gcnew StringBuilder(Global::BUFFER_SIZE);
	Thread^ uds_server;
	Thread^ isotp_server;
	int keyboard_res;

	// Initialize variables
	t_params->server_handle = PCANTP_HANDLE_USBBUS2; // TODO: modify the value according to your available PCAN devices.
	t_params->server_address = (UInt32)PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_1;
	t_params->stop_task = false;

	// Print version informations
	status = UDSApi::GetValue_2013(PCANTP_HANDLE_NONEBUS, PUDS_PARAMETER_API_VERSION, buffer, Global::BUFFER_SIZE);
	Console::WriteLine("PCAN-UDS API Version - {0}: {1}", buffer, UDS_STATUS_OK_KO(status));

	// Initialize server
	status = UDSApi::Initialize_2013(t_params->server_handle, PCANTP_BAUDRATE_500K, (cantp_hwtype)0, 0, 0);
	Console::WriteLine("Initialize channel: {0}", UDS_STATUS_OK_KO(status));

	// Start uds and isotp servers
	uds_server = gcnew Thread(gcnew ParameterizedThreadStart(uds_server_task));
	uds_server->Start(t_params);
	isotp_server = gcnew Thread(gcnew ParameterizedThreadStart(isotp_server_task));
	isotp_server->Start(t_params);

	// Read while user do not press Q
	Console::WriteLine("Start listening, press Q to quit.");
	t_params->stop_task = false;
	do
	{
		// Quit when user press Q
		if (Console::KeyAvailable)
		{
			keyboard_res = Console::ReadKey().KeyChar;
			if (keyboard_res == 'Q' || keyboard_res == 'q')
			{
				t_params->stop_task = true;
			}
		}
	} while (t_params->stop_task == false);

	// Close threads
	isotp_server->Join();
	uds_server->Join();

	// Close channel
	status = UDSApi::Uninitialize_2013(t_params->server_handle);
	Console::WriteLine("Uninitialize channel: {0}", UDS_STATUS_OK_KO(status));

	//Exit
	Console::WriteLine("Press any key to continue...");
	Console::Read();
}

