
#include "PCANBasicCLR.h"
#include "PCAN-ISO-TP-CLR_2016.h"
#include "PCAN-UDS-CLR_2013.h"


using namespace System;
using namespace System::IO;
using namespace System::Threading;
using namespace System::Runtime::InteropServices;
using namespace Peak::Can::IsoTp;
using namespace Peak::Can::Uds;

const int BUFFER_SIZE = 256;
const int UDS_REQUEST_TIMING_MS = 500;
const int CLIENT_EXECUTION_TIME_MS = 10000;

static String^ OK_KO(bool test)
{
	if (test)
		return "OK";
	return "KO";
}

static String^ UDS_STATUS_OK_KO(uds_status test)
{
	return OK_KO(UDSApi::StatusIsOk_2013(test));
}

static String^ ISOTP_STATUS_OK_KO(cantp_status test)
{
	return OK_KO(CanTpApi::StatusIsOk_2016(test));
}

/// <summary>Structure passed as thread parameters</summary>
public ref struct task_params
{
	/// <summary>Client channel handle</summary>
	cantp_handle client_handle;
	/// <summary>Determine if the thread should end or not</summary>
	bool stop_task;
};

/// <summary>UDS client task: request TesterPresent service several time</summary>
/// <param name="parameters">pointer on task_params structures</param>
static void uds_client_task(Object^ parameters)
{
	// Initialize variables
	uds_status status;
	uds_msgconfig config_physical = {};
	uds_msg msg_request = {};
	uds_msg request_confirmation = {};
	uds_msg response = {};
	task_params^ t_params = safe_cast<task_params^>(parameters);

	// Initialize a physical configuration
	config_physical.can_id = (UInt32)PUDS_CAN_ID_ISO_15765_4_PHYSICAL_REQUEST_1;
	config_physical.can_msgtype = PCANTP_CAN_MSGTYPE_STANDARD;
	config_physical.nai.protocol = PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL;
	config_physical.nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL;
	config_physical.type = PUDS_MSGTYPE_USDT;
	config_physical.nai.source_addr = (UInt16)PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
	config_physical.nai.target_addr = (UInt16)PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_1;
	config_physical.nai.extension_addr = 0;

	// Execute TesterPresent and wait response
	do
	{
		// Wait before transmitting the next message
		Thread::Sleep(UDS_REQUEST_TIMING_MS);

		status = UDSApi::SvcTesterPresent_2013(t_params->client_handle, config_physical, msg_request, UDSApi::uds_svc_param_tp::PUDS_SVC_PARAM_TP_ZSUBF);
		Console::WriteLine("[UDS] Execute TesterPresent service: {0}", UDS_STATUS_OK_KO(status));
		status = UDSApi::WaitForService_2013(t_params->client_handle, msg_request, response, request_confirmation);
		Console::WriteLine("[UDS] Wait for service: {0}", UDS_STATUS_OK_KO(status));
		status = UDSApi::MsgFree_2013(msg_request);
		Console::WriteLine("[UDS] Free request message: {0}", UDS_STATUS_OK_KO(status));
		status = UDSApi::MsgFree_2013(request_confirmation);
		Console::WriteLine("[UDS] Free request confirmation message: {0}", UDS_STATUS_OK_KO(status));
		status = UDSApi::MsgFree_2013(response);
		Console::WriteLine("[UDS] Free response message: {0}", UDS_STATUS_OK_KO(status));
	} while (t_params->stop_task == false);
}

/// <summary>ISOTP client task: read and print isotp messages</summary>
/// <param name="parameters">pointer on task_params structures</param>
static void isotp_client_task(Object^ parameters)
{
	// Initialize variables
	cantp_status status;
	cantp_msg rx_msg = {};
	bool wait_result;
	UInt32 isotp_param;
	task_params^ t_params = safe_cast<task_params^>(parameters);

	// Configure isotp to get any messages (disable can identifier filtering)
	isotp_param = 0;
	status = CanTpApi::SetValue_2016(t_params->client_handle, PCANTP_PARAMETER_FILTER_CAN_ID, isotp_param, sizeof(UInt32));
	Console::WriteLine("[ISOTP] Disable can identifier filtering in isotp: {0}", ISOTP_STATUS_OK_KO(status));

	// Configure isotp to get a copy of UDS messages
	isotp_param = 1;
	status = CanTpApi::SetValue_2016(t_params->client_handle, PCANTP_PARAMETER_KEEP_HIGHER_LAYER_MESSAGES, isotp_param, sizeof(UInt32));
	Console::WriteLine("[ISOTP] Activate higher layer messages in isotp: {0}", ISOTP_STATUS_OK_KO(status));

	// Create a isotp receive event
	System::Threading::AutoResetEvent receive_event(false);
	if (IntPtr::Size == 4)
	{
		UInt32 tmp_buffer = Convert::ToUInt32(receive_event.SafeWaitHandle->DangerousGetHandle().ToInt32());
		status = CanTpApi::SetValue_2016(t_params->client_handle, PCANTP_PARAMETER_RECEIVE_EVENT, tmp_buffer, sizeof(UInt32));
	}
	else if (IntPtr::Size == 8)
	{
		Int64 tmp_buffer = receive_event.SafeWaitHandle->DangerousGetHandle().ToInt64();
		array<Byte>^ byte_array = BitConverter::GetBytes(tmp_buffer);
		status = CanTpApi::SetValue_2016(t_params->client_handle, PCANTP_PARAMETER_RECEIVE_EVENT, byte_array, sizeof(UInt64));
	}
	Console::WriteLine("[ISOTP] Set isotp receive event parameter: {0}", ISOTP_STATUS_OK_KO(status));

	// Read and print ISOTP messages
	do
	{
		wait_result = receive_event.WaitOne(1000);
		Console::WriteLine("[ISOTP] Wait a receive event from isotp: {0}", OK_KO(wait_result));

		// Read ISOTP messages
		do
		{
			status = CanTpApi::Read_2016(t_params->client_handle, rx_msg);

			// Check if we received a message
			if (!CanTpApi::StatusIsOk_2016(status, PCANTP_STATUS_NO_MESSAGE, false))
			{
				Console::WriteLine("[ISOTP] Read ISOTP message: {0}", ISOTP_STATUS_OK_KO(status));
				if (rx_msg.msgdata.any->length > 0 && (rx_msg.msgdata.any->data[0] == PUDS_SERVICE_SI_TesterPresent || rx_msg.msgdata.any->data[0] == PUDS_SERVICE_SI_TesterPresent + UDSApi::PUDS_SI_POSITIVE_RESPONSE))
				{
					// This message is a TesterPresent message, PCANTP_PARAMETER_KEEP_HIGHER_LAYER_MESSAGES
					// option must be activated to get these messages.
					Console::WriteLine("[ISOTP] Message is a TesterPresent service message.");
				}
				else if (CanTpApi::StatusIsOk_2016(status) && rx_msg.msgdata.any->length < BUFFER_SIZE)
				{
					// This message is a CAN message received by ISOTP
					String^ msg_data = gcnew String((char*)rx_msg.msgdata.any->data, 0, rx_msg.msgdata.any->length);
					Console::WriteLine("[ISOTP] Message from 0x{0:x} can identifier contains \"{1}\"", rx_msg.can_info.can_id, msg_data);
				}

				status = CanTpApi::MsgDataFree_2016(rx_msg);
				Console::WriteLine("[ISOTP] Free RX message: {0}", ISOTP_STATUS_OK_KO(status));
			}
		} while (!CanTpApi::StatusIsOk_2016(status, PCANTP_STATUS_NO_MESSAGE, false));
	} while (t_params->stop_task == false);

	// Close receive event
	if (IntPtr::Size == 4)
	{
		UInt32 tmp_buffer = 0;
		status = CanTpApi::SetValue_2016(t_params->client_handle, PCANTP_PARAMETER_RECEIVE_EVENT, tmp_buffer, sizeof(UInt32));
	}
	else if (IntPtr::Size == 8)
	{
		Int64 tmp_buffer = 0;
		array<Byte>^ byte_array = BitConverter::GetBytes(tmp_buffer);
		status = CanTpApi::SetValue_2016(t_params->client_handle, PCANTP_PARAMETER_RECEIVE_EVENT, byte_array, sizeof(UInt64));
	}
	Console::WriteLine("[ISOTP] Stop ISOTP receive event: {0}", ISOTP_STATUS_OK_KO(status));
	receive_event.Close();
	Console::WriteLine("[ISOTP] Close ISOTP receive event: {0}", ISOTP_STATUS_OK_KO(status));
}

/// <summary>Entry point of the program, start a UDS channel, ask in the same time UDS TesterPresent and isotp request</summary>
/// <returns>By convention, return success.</returns>
int main(array<System::String^>^ args)
{
	StringBuilder^ buffer = gcnew StringBuilder(BUFFER_SIZE);
	uds_status status;
	cantp_handle client_handle;
	Thread^ uds_client;
	Thread^ isotp_client;
	task_params^ t_params = gcnew task_params();

	// Initialize variables
	client_handle = PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.
	t_params->client_handle = client_handle;
	t_params->stop_task = false;

	// Print version informations
	status = UDSApi::GetValue_2013(PCANTP_HANDLE_NONEBUS, PUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE);
	Console::WriteLine("PCAN-UDS API Version - {0}: {1}", buffer, UDS_STATUS_OK_KO(status));

	// Initialize client channel
	status = UDSApi::Initialize_2013(client_handle, PCANTP_BAUDRATE_500K, (cantp_hwtype)0, 0, 0);
	Console::WriteLine("Initialize channel: {0}", UDS_STATUS_OK_KO(status));

	// Start uds and isotp clients
	uds_client = gcnew Thread(gcnew ParameterizedThreadStart(uds_client_task));
	uds_client->Start(t_params);
	isotp_client = gcnew Thread(gcnew ParameterizedThreadStart(isotp_client_task));
	isotp_client->Start(t_params);

	Thread::Sleep(CLIENT_EXECUTION_TIME_MS);
	t_params->stop_task = true;

	isotp_client->Join();
	uds_client->Join();

	// Close channel
	status = UDSApi::Uninitialize_2013(client_handle);
	Console::WriteLine("Uninitialize channel: {0}", UDS_STATUS_OK_KO(status));

	// Exit
	Console::WriteLine("Press any key to continue...");
	Console::Read();
}
