#include "stdafx.h"

#include "lib/PCAN-UDS_2013.h"

#define OK_KO(test) (test)?"OK":"KO"
#define STATUS_OK_KO(test) OK_KO(UDS_StatusIsOk_2013(test, PUDS_STATUS_OK, false))

#define BUFFER_SIZE 256

/// <summary>Helper: convert cantp network address information into uds network information</summary>
/// <param name="nai">cantp network address information structure</param>
/// <param name="can_msgtype">cantp message type</param>
/// <returns>UDS network address information structure</returns>
uds_netaddrinfo ConvertNai(cantp_netaddrinfo nai, cantp_can_msgtype can_msgtype) {
	uds_netaddrinfo result;
	bool is_29bits = ((can_msgtype & PCANTP_CAN_MSGTYPE_EXTENDED) == PCANTP_CAN_MSGTYPE_EXTENDED);

	result.source_addr = nai.source_addr;
	result.target_addr = nai.target_addr;
	result.target_type = nai.target_type;
	result.extension_addr = nai.extension_addr;

	switch (nai.format) {
	case PCANTP_ISOTP_FORMAT_ENHANCED:
		result.protocol = PUDS_MSGPROTOCOL_ISO_15765_3_29B_ENHANCED;
		break;
	case PCANTP_ISOTP_FORMAT_EXTENDED:
		result.protocol = is_29bits ? PUDS_MSGPROTOCOL_ISO_15765_2_29B_EXTENDED : PUDS_MSGPROTOCOL_ISO_15765_2_11B_EXTENDED;
		break;
	case PCANTP_ISOTP_FORMAT_FIXED_NORMAL:
		result.protocol = PUDS_MSGPROTOCOL_ISO_15765_2_29B_FIXED_NORMAL;
		break;
	case PCANTP_ISOTP_FORMAT_MIXED:
		result.protocol = is_29bits ? PUDS_MSGPROTOCOL_ISO_15765_2_29B_REMOTE : PUDS_MSGPROTOCOL_ISO_15765_2_11B_REMOTE;
		break;
	case PCANTP_ISOTP_FORMAT_NORMAL:
		result.protocol = is_29bits ? PUDS_MSGPROTOCOL_ISO_15765_2_29B_NORMAL : PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL;
		break;
	case PCANTP_ISOTP_FORMAT_NONE:
	case PCANTP_ISOTP_FORMAT_UNKNOWN:
	default:
		result.protocol = PUDS_MSGPROTOCOL_NONE;
		break;
	}
	return result;
}

/// <summary>Helper: check if a service has a subfunction</summary>
/// <param name="service_identifier">service identifier</param>
/// <returns>boolean: yes (true) or not (false)</returns>
bool HasSubFunction(uint8_t service_identifier) {
	bool res;
	switch (service_identifier) {
	case PUDS_SI_DiagnosticSessionControl:
		res = true;
		break;
	case PUDS_SI_ECUReset:
		res = true;
		break;
	case PUDS_SI_SecurityAccess:
		res = true;
		break;
	case PUDS_SI_CommunicationControl:
		res = true;
		break;
	case PUDS_SI_TesterPresent:
		res = true;
		break;
	case PUDS_SI_AccessTimingParameter:
		res = true;
		break;
	case PUDS_SI_SecuredDataTransmission:
		res = false;
		break;
	case PUDS_SI_ControlDTCSetting:
		res = true;
		break;
	case PUDS_SI_ResponseOnEvent:
		res = true;
		break;
	case PUDS_SI_LinkControl:
		res = true;
		break;
	case PUDS_SI_ReadDataByIdentifier:
		res = false;
		break;
	case PUDS_SI_ReadMemoryByAddress:
		res = false;
		break;
	case PUDS_SI_ReadScalingDataByIdentifier:
		res = false;
		break;
	case PUDS_SI_ReadDataByPeriodicIdentifier:
		res = false;
		break;
	case PUDS_SI_DynamicallyDefineDataIdentifier:
		res = true;
		break;
	case PUDS_SI_WriteDataByIdentifier:
		res = false;
		break;
	case PUDS_SI_WriteMemoryByAddress:
		res = false;
		break;
	case PUDS_SI_ClearDiagnosticInformation:
		res = false;
		break;
	case PUDS_SI_ReadDTCInformation:
		res = true;
		break;
	case PUDS_SI_InputOutputControlByIdentifier:
		res = false;
		break;
	case PUDS_SI_RoutineControl:
		res = true;
		break;
	case PUDS_SI_RequestDownload:
		res = false;
		break;
	case PUDS_SI_RequestUpload:
		res = false;
		break;
	case PUDS_SI_TransferData:
		res = false;
		break;
	case PUDS_SI_RequestTransferExit:
		res = false;
		break;
	case PUDS_SI_RequestFileTransfer:
		res = false;
		break;
	case PUDS_SI_Authentication:
		res = true;
		break;
	default:
		res = false;
		break;
	}
	return res;
}

/// <summary>
/// Create the ECUReset response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing ECUReset parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
uds_status create_SvcECUReset_response(uds_msgconfig config, uds_msg *request_msg, uds_msg *response_msg) {
	uds_status status;

	// Allocate response message
	if (request_msg->links.param[0] == PUDS_SVC_PARAM_ER_ERPSD) {
		status = UDS_MsgAlloc_2013(response_msg, config, 3);
		response_msg->links.param[1] = 0x66;	// Power down time
	}
	else {
		status = UDS_MsgAlloc_2013(response_msg, config, 2);
	}
	printf("Allocate ECUReset response: %s\n", STATUS_OK_KO(status));

	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
		// Fill parameters
		*response_msg->links.service_id = PUDS_SI_ECUReset + PUDS_SI_POSITIVE_RESPONSE;
		response_msg->links.param[0] = request_msg->links.param[0];
	}

	return status;
}

/// <summary>
/// Create the WriteMemoryByAddress response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
uds_status create_SvcWriteMemoryByAddress_response(uds_msgconfig config, uds_msg *request_msg, uds_msg *response_msg) {
	uds_status status;
	uint32_t memory_size_size;
	uint32_t memory_address_size;
	uint32_t length;

	// Read memory_size_size & memory_address_size
	memory_size_size = (request_msg->links.param[0] >> 4) & 0xF;
	memory_address_size = request_msg->links.param[0] & 0xF;
	length = 2 + memory_size_size + memory_address_size;

	// Allocate message
	status = UDS_MsgAlloc_2013(response_msg, config, length);
	printf("Allocate WriteMemoryByAddress response: %s\n", STATUS_OK_KO(status));

	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
		// Copy address and memory parameters
		response_msg->links.param[0] = request_msg->links.param[0];
		memcpy(&response_msg->links.param[1],
			&request_msg->links.param[1], memory_address_size);
		memcpy(&response_msg->links.param[1 + memory_address_size],
			&request_msg->links.param[1 + memory_address_size], memory_size_size);
		*response_msg->links.service_id = PUDS_SI_WriteMemoryByAddress + PUDS_SI_POSITIVE_RESPONSE;
	}

	return status;
}

/// <summary>
/// Create the DynamicallyDefineDataIdentifier response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
uds_status create_SvcDynamicallyDefineDataIdentifier_response(uds_msgconfig config, uds_msg *request_msg, uds_msg *response_msg) {
	uds_status status;

	// Allocate and fill message
	if (request_msg->msg.msgdata.any->length >= 4) {
		status = UDS_MsgAlloc_2013(response_msg, config, 4);
		printf("Allocate DynamicallyDefineDataIdentifier response: %s\n", STATUS_OK_KO(status));

		if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
			// Copy data
			*response_msg->links.service_id = PUDS_SI_DynamicallyDefineDataIdentifier + PUDS_SI_POSITIVE_RESPONSE;
			response_msg->links.param[0] = request_msg->links.param[0];
			response_msg->links.param[1] = request_msg->links.param[1];
			response_msg->links.param[2] = request_msg->links.param[2];
		}
	}
	else {
		// Clear all data identifier request
		status = UDS_MsgAlloc_2013(response_msg, config, 2);
		printf("Allocate DynamicallyDefineDataIdentifier response: %s\n", STATUS_OK_KO(status));

		if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
			// Copy subfunction
			*response_msg->links.service_id = PUDS_SI_DynamicallyDefineDataIdentifier + PUDS_SI_POSITIVE_RESPONSE;
			response_msg->links.param[0] = request_msg->links.param[0];
		}
	}

	return status;
}

/// <summary>
/// Create the dummy response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
uds_status create_dummy_response(uds_msgconfig config, uds_msg *request_msg, uds_msg *response_msg) {
	uds_status status;
	uint32_t length;


	// Set a random message size
	length = 1 + *request_msg->links.service_id + rand() % 200;
	length = length > PCANTP_MAX_LENGTH_ISOTP2004 ? PCANTP_MAX_LENGTH_ISOTP2004 : length;

	// Allocate response message
	status = UDS_MsgAlloc_2013(response_msg, config, length);
	printf("Allocate dummy response: %s\n", STATUS_OK_KO(status));

	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
		// Fake a positive response
		*response_msg->links.service_id = *request_msg->links.service_id + PUDS_SI_POSITIVE_RESPONSE;

		// Fill with dummy data
		response_msg->links.param[0] = *request_msg->links.service_id;
		for (uint32_t i = 1; i < length - 1; i++)	// (length - 1) as positive response SID uses 1 byte
			response_msg->links.param[i] = i + 1;

		// Do not set PUDS_FLAG_SUPPRESS_POSITIVE_RESPONSE flag when service has a subfunction
		if (HasSubFunction(*request_msg->links.service_id)) {
			response_msg->msg.msgdata.any->data[1] = 0;
		}
	}

	return status;
}

/// <summary>
/// Create the RequestTransferExit response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
uds_status create_SvcRequestTransferExit_response(uds_msgconfig config, uds_msg *response_msg) {
	uds_status status;
	uint32_t length;

	// Allocate message
	length = 1 + 1 + rand() % 50;
	status = UDS_MsgAlloc_2013(response_msg, config, length);
	printf("Allocate RequestTransferExit response: %s\n", STATUS_OK_KO(status));

	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
		// Fill with dummy data
		for (uint32_t i = 0; i < length - 1; i++)	// (length - 1) as positive response SID uses 1 byte
			response_msg->links.param[i] = i + 1;
		*response_msg->links.service_id = PUDS_SI_RequestTransferExit + PUDS_SI_POSITIVE_RESPONSE;
	}

	return status;
}

/// <summary>
/// Create the SvcRequestUpload response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
uds_status create_SvcRequestUpload_response(uds_msgconfig config, uds_msg *response_msg) {
	uds_status status;
	uint32_t length;

	// Allocate message
	length = 2 + 1 + rand() % 50;
	status = UDS_MsgAlloc_2013(response_msg, config, length);
	printf("Allocate RequestUpload response: %s\n", STATUS_OK_KO(status));

	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
		// Max number of block length = 0xF
		response_msg->links.param[0] = 0xF0;

		// Fill with dummy data
		for (uint32_t i = 1; i < length - 1; i++)	// (length - 1) as positive response SID uses 1 byte
			response_msg->links.param[i] = i + 1;

		*response_msg->links.service_id = PUDS_SI_RequestUpload + PUDS_SI_POSITIVE_RESPONSE;
	}

	return status;
}

/// <summary>
/// Create the SvcRequestDownload response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
uds_status create_SvcRequestDownload_response(uds_msgconfig config, uds_msg *response_msg) {
	uds_status status;
	uint32_t length;

	// Allocate message
	length = 2 + 1 + rand() % 50;
	status = UDS_MsgAlloc_2013(response_msg, config, length);
	printf("Allocate RequestDownload response: %s\n", STATUS_OK_KO(status));

	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
		// Max number of block length = 0xF
		response_msg->links.param[0] = 0xF0;

		// Fill with dummy data
		for (uint32_t i = 1; i < length - 1; i++)	// (length - 1) as positive response SID uses 1 byte
			response_msg->links.param[i] = i + 1;

		*response_msg->links.service_id = PUDS_SI_RequestDownload + PUDS_SI_POSITIVE_RESPONSE;
	}

	return status;
}

/// <summary>
/// Create the RoutineControl response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
uds_status create_SvcRoutineControl_response(uds_msgconfig config, uds_msg *request_msg, uds_msg *response_msg) {
	uds_status status;

	// Allocate response
	status = UDS_MsgAlloc_2013(response_msg, config, 4);
	printf("Allocate RoutineControl response: %s\n", STATUS_OK_KO(status));

	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
		// Copy data
		response_msg->links.param[0] = request_msg->links.param[0];
		response_msg->links.param[1] = request_msg->links.param[1];
		response_msg->links.param[2] = request_msg->links.param[2];

		// Routine status record not implemented

		*response_msg->links.service_id = PUDS_SI_RoutineControl + PUDS_SI_POSITIVE_RESPONSE;
	}

	return status;
}

/// <summary>
/// Create the InputOutputControlByIdentifier response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
uds_status create_SvcInputOutputControlByIdentifier_response(uds_msgconfig config, uds_msg *request_msg, uds_msg *response_msg) {
	uds_status status;

	status = UDS_MsgAlloc_2013(response_msg, config, 3);
	printf("Allocate InputOutputControlByIdentifier response: %s\n", STATUS_OK_KO(status));

	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
		// Copy data
		response_msg->links.param[0] = request_msg->links.param[0];
		response_msg->links.param[1] = request_msg->links.param[1];

		// Control status record not implemented
		*response_msg->links.service_id = PUDS_SI_InputOutputControlByIdentifier + PUDS_SI_POSITIVE_RESPONSE;
	}
	return status;
}

/// <summary>
/// Create the ClearDiagnosticInformation response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
uds_status create_SvcClearDiagnosticInformation_response(uds_msgconfig config, uds_msg *response_msg) {
	uds_status status;

	status = UDS_MsgAlloc_2013(response_msg, config, 1);
	printf("Allocate ClearDiagnosticInformation response: %s\n", STATUS_OK_KO(status));

	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
		*response_msg->links.service_id = PUDS_SI_ClearDiagnosticInformation + PUDS_SI_POSITIVE_RESPONSE;
	}

	return status;
}

/// <summary>
/// Create the WriteDataByIdentifier response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
uds_status create_SvcWriteDataByIdentifier_response(uds_msgconfig config, uds_msg *request_msg, uds_msg *response_msg) {
	uds_status status;

	status = UDS_MsgAlloc_2013(response_msg, config, 3);
	printf("Allocate WriteDataByIdentifier response: %s\n", STATUS_OK_KO(status));

	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
		// Copy data
		response_msg->links.param[0] = request_msg->links.param[0];
		response_msg->links.param[1] = request_msg->links.param[1];
		*response_msg->links.service_id = PUDS_SI_WriteDataByIdentifier + PUDS_SI_POSITIVE_RESPONSE;
	}

	return status;
}

/// <summary>
/// Create the ReadDataByPeriodicIdentifier response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
uds_status create_SvcReadDataByPeriodicIdentifier_response(uds_msgconfig config, uds_msg *response_msg) {
	uds_status status;
	status = UDS_MsgAlloc_2013(response_msg, config, 1);
	printf("Allocate ReadDataByPeriodicIdentifier response: %s\n", STATUS_OK_KO(status));

	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
		*response_msg->links.service_id = PUDS_SI_ReadDataByPeriodicIdentifier + PUDS_SI_POSITIVE_RESPONSE;
	}

	return status;
}

/// <summary>
/// Create the ReadMemoryByAddress response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
uds_status create_SvcReadMemoryByAddress_response(uds_msgconfig config, uds_msg *request_msg, uds_msg *response_msg) {
	uds_status status;
	uint32_t length;

	// Read memory_size_size = bits [7..4]
	length = 1 + ((request_msg->links.param[1] >> 4) & 0xF);

	// Allocate message
	status = UDS_MsgAlloc_2013(response_msg, config, length);
	printf("Allocate ReadMemoryByAddress response: %s\n", STATUS_OK_KO(status));

	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
		// Fill with dummy data
		for (uint32_t i = 0; i < length - 1; i++)	// (length - 1) as positive response SID uses 1 byte
			response_msg->links.param[i] = i + 1;
		*response_msg->links.service_id = PUDS_SI_ReadMemoryByAddress + PUDS_SI_POSITIVE_RESPONSE;
	}

	return status;
}

/// <summary>
/// Create the ReadScalingDataByIdentifier response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
uds_status create_SvcReadScalingDataByIdentifier_response(uds_msgconfig config, uds_msg *request_msg, uds_msg *response_msg) {
	uds_status status;

	// Allocate message
	status = UDS_MsgAlloc_2013(response_msg, config, 12);
	printf("Allocate ReadScalingDataByIdentifier response: %s\n", STATUS_OK_KO(status));

	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
		*response_msg->links.service_id = PUDS_SI_ReadScalingDataByIdentifier + PUDS_SI_POSITIVE_RESPONSE;

		// Copy data
		response_msg->links.param[0] = request_msg->links.param[0];
		response_msg->links.param[1] = request_msg->links.param[1];

		// Create a formula Vehicule Speed = (0.75*x+30) km/h
		response_msg->links.param[2] = (0x0 << 4) | (0x1);	// unsigned numeric of 1 Bytes)
		response_msg->links.param[3] = 0x90;	// formula, 0 data bytes
		response_msg->links.param[4] = 0x00;	// formulaIdentifier = C0 * x + C1
		response_msg->links.param[5] = 0xE0;	// C0 high byte
		response_msg->links.param[6] = 0x4B;	// C0 low byte
		response_msg->links.param[7] = 0x00;	// C1 high byte
		response_msg->links.param[8] = 0x1E;	// C1 low byte
		response_msg->links.param[9] = 0xA0;	// unit/format, 0 data bytes
		response_msg->links.param[10] = 0x30;	// unit ID, km/h
	}

	return status;
}

/// <summary>
/// Create the ReadDataByIdentifier response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
uds_status create_SvcReadDataByIdentifier_response(uds_msgconfig config, uds_msg *request_msg, uds_msg *response_msg) {
	uds_status status;
	uint32_t length;
	uint32_t counter;

	// Allocate message
	length = ((request_msg->msg.msgdata.any->length - 1) * 7 / 2) + 1;
	status = UDS_MsgAlloc_2013(response_msg, config, length);
	printf("Allocate ReadDataByIdentifier response: %s\n", STATUS_OK_KO(status));

	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
		counter = 0;
		for (uint32_t i = 0; i < request_msg->msg.msgdata.any->length - 1; i += 2)
		{
			// Copy data
			response_msg->links.param[counter++] = request_msg->links.param[i];
			response_msg->links.param[counter++] = request_msg->links.param[i + 1];
			// Data record: fill with dummy data
			for (int j = 0; j < 5; j++) {
				response_msg->links.param[counter++] = j + 'A';
			}
		}
		*response_msg->links.service_id = PUDS_SI_ReadDataByIdentifier + PUDS_SI_POSITIVE_RESPONSE;
	}

	return status;
}

/// <summary>
/// Create the LinkControl response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
uds_status create_SvcLinkControl_response(uds_msgconfig config, uds_msg *request_msg, uds_msg *response_msg) {
	uds_status status;
	status = UDS_MsgAlloc_2013(response_msg, config, 2);
	printf("Allocate LinkControl response: %s\n", STATUS_OK_KO(status));

	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
		response_msg->links.param[0] = request_msg->links.param[0];
		*response_msg->links.service_id = PUDS_SI_LinkControl + PUDS_SI_POSITIVE_RESPONSE;
	}

	return status;
}

/// <summary>
/// Create the ResponseOnEvent response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
uds_status create_SvcResponseOnEvent_response(uds_msgconfig config, uds_msg *request_msg, uds_msg *response_msg) {
	uds_status status;

	if (request_msg->links.param[0] == PUDS_SVC_PARAM_ROE_RAE)
	{
		status = UDS_MsgAlloc_2013(response_msg, config, 3);
		printf("Allocate ResponseOnEvent response: %s\n", STATUS_OK_KO(status));

		if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
			response_msg->links.param[1] = 0;	// Number of activated events
												// Event type and service to respond to records not implemented
		}
	}
	else
	{
		status = UDS_MsgAlloc_2013(response_msg, config, 4);
		printf("Allocate ResponseOnEvent response: %s\n", STATUS_OK_KO(status));
		if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
			response_msg->links.param[1] = 0;	// Number of identified events
			response_msg->links.param[2] = request_msg->links.param[1];	// Event window time
																		// Event type and service to respond to records not implemented
		}
	}
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
		response_msg->links.param[0] = request_msg->links.param[0];
		*response_msg->links.service_id = PUDS_SI_ResponseOnEvent + PUDS_SI_POSITIVE_RESPONSE;
	}

	return status;
}

/// <summary>
/// Create the ControlDTCSetting response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
uds_status create_SvcControlDTCSetting_response(uds_msgconfig config, uds_msg *request_msg, uds_msg *response_msg) {
	uds_status status;
	status = UDS_MsgAlloc_2013(response_msg, config, 2);
	printf("Allocate ControlDTCSetting response: %s\n", STATUS_OK_KO(status));

	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
		response_msg->links.param[0] = request_msg->links.param[0];
		*response_msg->links.service_id = PUDS_SI_ControlDTCSetting + PUDS_SI_POSITIVE_RESPONSE;
	}
	return status;
}

/// <summary>
/// Create the SecuredDataTransmission response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
uds_status create_SvcSecuredDataTransmission_response(uds_msgconfig config, uds_msg *response_msg) {
	uds_status status;
	uint32_t length;

	// Allocate message
	length = 1 + rand() % 50;
	status = UDS_MsgAlloc_2013(response_msg, config, length);
	printf("Allocate SecuredDataTransmission response: %s\n", STATUS_OK_KO(status));

	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
		// Fill with dummy data (check Security-SubLayer record defined in ISO-15764)
		for (uint32_t i = 0; i < length - 1; i++)	// (length - 1) as positive response SID uses 1 byte
			response_msg->links.param[i] = i + 1;

		*response_msg->links.service_id = PUDS_SI_SecuredDataTransmission + PUDS_SI_POSITIVE_RESPONSE;
	}

	return status;
}

/// <summary>
/// Create the TesterPresent response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
uds_status create_SvcTesterPresent_response(uds_msgconfig config, uds_msg *request_msg, uds_msg *response_msg) {
	uds_status status;
	status = UDS_MsgAlloc_2013(response_msg, config, 2);
	printf("Allocate TesterPresent response: %s\n", STATUS_OK_KO(status));

	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
		response_msg->links.param[0] = request_msg->links.param[0];
		*response_msg->links.service_id = PUDS_SI_TesterPresent + PUDS_SI_POSITIVE_RESPONSE;
	}

	return status;
}

/// <summary>
/// Create the CommunicationControl response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
uds_status create_SvcCommunicationControl_response(uds_msgconfig config, uds_msg *request_msg, uds_msg *response_msg) {
	uds_status status;
	status = UDS_MsgAlloc_2013(response_msg, config, 2);
	printf("Allocate CommunicationControl response: %s\n", STATUS_OK_KO(status));

	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
		response_msg->links.param[0] = request_msg->links.param[0];
		*response_msg->links.service_id = PUDS_SI_CommunicationControl + PUDS_SI_POSITIVE_RESPONSE;
	}

	return status;
}

/// <summary>
/// Create the SecurityAccess response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
uds_status create_SvcSecurityAccess_response(uds_msgconfig config, uds_msg *request_msg, uds_msg *response_msg) {
	uds_status status;
	uint32_t length;
	uint32_t i;

	// Allocate response message
	if (request_msg->links.param[0] >= PUDS_SVC_PARAM_SA_RSD_MIN
		&& request_msg->links.param[0] <= PUDS_SVC_PARAM_SA_RSD_MAX
		&& request_msg->links.param[0] % 2 == 1) {

		// Request security seed are Even values
		// Fill with dummy data
		length = (1 + PUDS_SI_SecurityAccess) > PCANTP_MAX_LENGTH_ISOTP2004 ? PCANTP_MAX_LENGTH_ISOTP2004 : (1 + PUDS_SI_SecurityAccess);
		status = UDS_MsgAlloc_2013(response_msg, config, length);
		printf("Allocate SecurityAccess response: %s\n", STATUS_OK_KO(status));

		if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
			for (i = 1; i < length - 1; i++)	// (length - 1) as positive response SID uses 1 byte
				response_msg->links.param[i] = i + 1;
		}
	}
	else {
		status = UDS_MsgAlloc_2013(response_msg, config, 2);
		printf("Allocate SecurityAccess response: %s\n", STATUS_OK_KO(status));
	}
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
		response_msg->links.param[0] = request_msg->links.param[0];
		*response_msg->links.service_id = PUDS_SI_SecurityAccess + PUDS_SI_POSITIVE_RESPONSE;
	}

	return status;
}


/// <summary>
/// Create the DiagnosticSessionControl response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
uds_status create_SvcDiagnosticSessionControl_response(uds_msgconfig config, uds_msg *request_msg, uds_msg *response_msg) {
	uds_status status;

	// Allocate response message
	status = UDS_MsgAlloc_2013(response_msg, config, 6);
	printf("Allocate DiagnosticSessionControl response: %s\n", STATUS_OK_KO(status));

	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
		// Fill response
		response_msg->links.param[0] = request_msg->links.param[0];
		response_msg->links.param[1] = 0x00;	// P2Can_Server_Max = 0x0010
		response_msg->links.param[2] = 0x10;
		response_msg->links.param[3] = 0x03;	// P2*Can_Server_Max = 0x03E8
		response_msg->links.param[4] = 0xE8;
		*response_msg->links.service_id = PUDS_SI_DiagnosticSessionControl + PUDS_SI_POSITIVE_RESPONSE;
	}

	return status;
}

/// <summary>
/// Create the TransferData response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
uds_status create_SvcTransferData_response(cantp_handle tp_handle, uds_msgconfig config, uds_msg *request_msg, uds_msg *response_msg) {
	uds_status status;
	uint32_t length;
	uds_msg pending_response_msg;

	// Initialize locale struct
	memset(&pending_response_msg, 0, sizeof(uds_msg));

	// Custom response to client_all_request example:
	//  a. service is requested functionally,
	//  b. 1st response is NRC response pending
	//  c. wait
	//  d. send correct response
	if (request_msg->msg.msgdata.isotp->netaddrinfo.target_type == PCANTP_ISOTP_ADDRESSING_FUNCTIONAL) {

		// Transmit a NRC response pending
		status = UDS_MsgAlloc_2013(&pending_response_msg, config, 3);
		printf("Allocate TransferData pending response: %s\n", STATUS_OK_KO(status));

		// Transmit a NRC response pending
		if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
			*pending_response_msg.links.service_id = PUDS_NR_SI;
			pending_response_msg.links.param[0] = PUDS_SI_TransferData;
			pending_response_msg.links.param[1] = PUDS_NRC_EXTENDED_TIMING;
		}
		status = UDS_Write_2013(tp_handle, &pending_response_msg);

		printf("\n   ...Transmitting a NRC Response Pending message: %i\n", (int)status);
		printf("\n   ...simulating computation... (waiting ~%dms)", PUDS_P2CAN_ENHANCED_SERVER_MAX_DEFAULT);
		Sleep(PUDS_P2CAN_ENHANCED_SERVER_MAX_DEFAULT - 100);

		// Reallocate response message
		status = UDS_MsgFree_2013(&pending_response_msg);
		printf("Free pending response: %s\n", STATUS_OK_KO(status));

		// Initialize real service response
		config.type = PUDS_MSGTYPE_USDT;
		length = PCANTP_MAX_LENGTH_ISOTP2004;
		status = UDS_MsgAlloc_2013(response_msg, config, length);
		printf("Allocate TransferData real response: %s\n", STATUS_OK_KO(status));
	}
	else {
		length = (2 + request_msg->msg.msgdata.any->length) > PCANTP_MAX_LENGTH_ISOTP2004 ? PCANTP_MAX_LENGTH_ISOTP2004 : 2 + request_msg->msg.msgdata.any->length;
		status = UDS_MsgAlloc_2013(response_msg, config, length);
		printf("Allocate TransferData response: %s\n", STATUS_OK_KO(status));
	}

	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
		response_msg->links.param[0] = request_msg->links.param[0];

		// Fill with dummy data
		for (uint32_t i = 1; i < length - 1; i++)	// (length - 1) as positive response SID uses 1 byte
			response_msg->links.param[i] = i + 1;
		*response_msg->links.service_id = PUDS_SI_TransferData + PUDS_SI_POSITIVE_RESPONSE;
	}

	return status;
}


/// <summary>
///		Process request and send a response
/// </summary>
/// <param name="tp_handle">CANTP channel</param>
/// <param name="server_address">Server address</param>
/// <param name="request_msg">Received request message</param>
void process_request(cantp_handle tp_handle, uint16_t server_address, uds_msg *request_msg) {
	uds_msgconfig config;
	uds_msg response_msg;
	uds_status status;

	// Initialize config from request message network address information
	memset(&config, 0, sizeof(uds_msgconfig));
	config.nai = ConvertNai(request_msg->msg.msgdata.isotp->netaddrinfo, request_msg->msg.can_info.can_msgtype);
	config.can_msgtype = request_msg->msg.can_info.can_msgtype;
	config.type = PUDS_MSGTYPE_USDT;
	config.nai.extension_addr = request_msg->msg.msgdata.isotp->netaddrinfo.extension_addr;

	memset(&response_msg, 0, sizeof(uds_msg));

	// Set target and source addresses
	if (request_msg->msg.msgdata.isotp->netaddrinfo.target_type == PCANTP_ISOTP_ADDRESSING_FUNCTIONAL) {
		// response to functional addressing is set to TEST_EQUIPMENT
		config.nai.target_addr = PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
	}
	else {
		config.nai.target_addr = request_msg->msg.msgdata.isotp->netaddrinfo.source_addr;
	}
	config.nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL;
	config.nai.source_addr = server_address;

	// This is a valid request, switch services
	switch (*request_msg->links.service_id)
	{
	case PUDS_SI_DiagnosticSessionControl:
		status = create_SvcDiagnosticSessionControl_response(config, request_msg, &response_msg);
		break;
	case PUDS_SI_SecurityAccess:
		status = create_SvcSecurityAccess_response(config, request_msg, &response_msg);
		break;
	case PUDS_SI_CommunicationControl:
		status = create_SvcCommunicationControl_response(config, request_msg, &response_msg);
		break;
	case PUDS_SI_TesterPresent:
		status = create_SvcTesterPresent_response(config, request_msg, &response_msg);
		break;
	case PUDS_SI_SecuredDataTransmission:
		status = create_SvcSecuredDataTransmission_response(config, &response_msg);
		break;
	case PUDS_SI_ControlDTCSetting:
		status = create_SvcControlDTCSetting_response(config, request_msg, &response_msg);
		break;
	case PUDS_SI_ResponseOnEvent:
		status = create_SvcResponseOnEvent_response(config, request_msg, &response_msg);
		break;
	case PUDS_SI_LinkControl:
		status = create_SvcLinkControl_response(config, request_msg, &response_msg);
		break;
	case PUDS_SI_ReadDataByIdentifier:
		status = create_SvcReadDataByIdentifier_response(config, request_msg, &response_msg);
		break;
	case PUDS_SI_ReadMemoryByAddress:
		status = create_SvcReadMemoryByAddress_response(config, request_msg, &response_msg);
		break;
	case PUDS_SI_ReadScalingDataByIdentifier:
		status = create_SvcReadScalingDataByIdentifier_response(config, request_msg, &response_msg);
		break;
	case PUDS_SI_ReadDataByPeriodicIdentifier:
		status = create_SvcReadDataByPeriodicIdentifier_response(config, &response_msg);
		break;
	case PUDS_SI_DynamicallyDefineDataIdentifier:
		status = create_SvcDynamicallyDefineDataIdentifier_response(config, request_msg, &response_msg);
		break;
	case PUDS_SI_WriteDataByIdentifier:
		status = create_SvcWriteDataByIdentifier_response(config, request_msg, &response_msg);
		break;
	case PUDS_SI_WriteMemoryByAddress:
		status = create_SvcWriteMemoryByAddress_response(config, request_msg, &response_msg);
		break;
	case PUDS_SI_ClearDiagnosticInformation:
		status = create_SvcClearDiagnosticInformation_response(config, &response_msg);
		break;
	case PUDS_SI_InputOutputControlByIdentifier:
		status = create_SvcInputOutputControlByIdentifier_response(config, request_msg, &response_msg);
		break;
	case PUDS_SI_RoutineControl:
		status = create_SvcRoutineControl_response(config, request_msg, &response_msg);
		break;
	case PUDS_SI_RequestDownload:
		status = create_SvcRequestDownload_response(config, &response_msg);
		break;
	case PUDS_SI_RequestUpload:
		status = create_SvcRequestUpload_response(config, &response_msg);
		break;
	case PUDS_SI_TransferData:
		status = create_SvcTransferData_response(tp_handle, config, request_msg, &response_msg);
		break;
	case PUDS_SI_RequestTransferExit:
		status = create_SvcRequestTransferExit_response(config, &response_msg);
		break;
	case PUDS_SI_ECUReset:
		status = create_SvcECUReset_response(config, request_msg, &response_msg);
		break;
	case PUDS_SI_ReadDTCInformation:
		// Not yet implemented
	case PUDS_SI_AccessTimingParameter:
		// Not yet implemented
	case PUDS_SI_RequestFileTransfer:
		// Not yet implemented
	case PUDS_SI_Authentication:
		// Not yet implemented
	default:
		printf("Unknown service (0x%02X)\n", *request_msg->links.service_id);
		status = create_dummy_response(config, request_msg, &response_msg);
		break;
	}

	// Print allocation result
	printf("Allocate response message: %s\n", STATUS_OK_KO(status));

	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
		// Send response message
		status = UDS_Write_2013(tp_handle, &response_msg);
		printf("\n   ...Transmitting response: %i\n\n", (int)status);
	}

	// Free response message
	status = UDS_MsgFree_2013(&response_msg);
	printf("Free response message: %s\n", STATUS_OK_KO(status));
}

/// <summary>Print an UDS message</summary>
/// <param name="msg">Message to print</param>
void display_message(uds_msg *msg) {
	// A function that displays UDS messages

	char buffer[BUFSIZ];
	memset(buffer, 0, BUFSIZ);
	if ((msg->msg.msgdata.isotp->netaddrinfo.msgtype & PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION) == 0)
	{
		printf("\n[%llu] %s message from 0x%02X (to 0x%02X, with extension 0x%02X) - result: %i - %s\n", GetTickCount64(),
			((msg->msg.msgdata.any->flags & PCANTP_MSGFLAG_LOOPBACK) == 0) ? "Received UDS" : "Received Loopback",
			(int)msg->msg.msgdata.isotp->netaddrinfo.source_addr,
			(int)msg->msg.msgdata.isotp->netaddrinfo.target_addr,
			(int)msg->msg.msgdata.isotp->netaddrinfo.extension_addr,
			(int)msg->msg.msgdata.any->netstatus,
			msg->msg.msgdata.any->netstatus != PCANTP_NETSTATUS_OK ? "PCANTP_NETSTATUS_OK !!!" : "OK !");
		// Display data
		setvbuf(stdout, buffer, _IOLBF, BUFSIZ);
		printf("\t-> Length: %i, Data= ", (int)msg->msg.msgdata.any->length);
		for (uint32_t i = 0; i < msg->msg.msgdata.any->length; i++) {
			printf("%02X ", (int)msg->msg.msgdata.any->data[i]);
			fflush(stdout);
		}
		printf("\n\n");
		setvbuf(stdout, NULL, _IONBF, 0);
	}
	else
	{
		printf("\n[%llu] PENDING UDS message from 0x%02X (to 0x%02X, with extension 0x%02X) -> length=%i ...\n", GetTickCount64(),
			(int)msg->msg.msgdata.isotp->netaddrinfo.source_addr,
			(int)msg->msg.msgdata.isotp->netaddrinfo.target_addr,
			(int)msg->msg.msgdata.isotp->netaddrinfo.extension_addr,
			(int)msg->msg.msgdata.any->length);
	}
}

/// <summary>Entry point of the program, start a CAN UDS server simulation</summary>
int main()
{
	uds_status status;
	cantp_handle tp_handle;
	uint32_t server_address;
	uint64_t null_handle;
	char buffer[BUFFER_SIZE];
	HANDLE receive_event;
	BOOL boolean_status;
	bool stop;
	DWORD wait_result;
	uds_status read_status;
	uds_msg request_msg;
	int keyboard_res;
	uint32_t timeout_value;
	uint8_t tmp_buffer;

	// Initialize variables
	tp_handle = PCANTP_HANDLE_USBBUS2; // TODO: modify the value according to your available PCAN devices.
	null_handle = 0;
	server_address = PUDS_ISO_15765_4_ADDR_ECU_1;
	buffer[0] = '\0';
	memset(&request_msg, 0, sizeof(uds_msg));
	srand((unsigned int)time(NULL));

	// Print version informations
	status = UDS_GetValue_2013(PCANTP_HANDLE_NONEBUS, PUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE);
	printf("PCAN-UDS API Version - %s: %s\n", buffer, STATUS_OK_KO(status));

	// Initialize channel
	status = UDS_Initialize_2013(tp_handle, PCANTP_BAUDRATE_500K, (cantp_hwtype)0, 0, 0);
	printf("Initialize channel: %s\n", STATUS_OK_KO(status));

	// Set server address parameter
	status = UDS_SetValue_2013(tp_handle, PUDS_PARAMETER_SERVER_ADDRESS, &server_address, sizeof(uint32_t));
	printf("Set server address: %s\n", STATUS_OK_KO(status));

	// Set timeout values
	tmp_buffer = PCANTP_ISO_TIMEOUTS_15765_4;
	status = UDS_SetValue_2013(tp_handle, PUDS_PARAMETER_ISO_TIMEOUTS, &tmp_buffer, sizeof(tmp_buffer));
	printf("Set ISO 15765-4 timeouts values: %s\n", STATUS_OK_KO(status));

	// Print timeout values
	status = UDS_GetValue_2013(tp_handle, PUDS_PARAMETER_TIMEOUT_REQUEST, &timeout_value, sizeof(uint32_t));
	printf("Get request timeout value (%ums): %s\n", timeout_value, STATUS_OK_KO(status));
	status = UDS_GetValue_2013(tp_handle, PUDS_PARAMETER_TIMEOUT_RESPONSE, &timeout_value, sizeof(uint32_t));
	printf("Get response timeout value (%ums): %s\n", timeout_value, STATUS_OK_KO(status));

	// Set a receive event
	receive_event = CreateEvent(NULL, FALSE, FALSE, NULL);
	status = UDS_SetValue_2013(tp_handle, PUDS_PARAMETER_RECEIVE_EVENT, &receive_event, sizeof(receive_event));
	printf("Set receive event parameter: %s\n", STATUS_OK_KO(status));

	// Read while user do not press Q
	printf("Start listening, press Q to quit.\n");
	stop = false;
	while (!stop) {

		// Wait a receive event on receiver
		//	note: timeout is used to check keyboard hit.
		wait_result = WaitForSingleObject(receive_event, 2000);

		// If we get a receive event
		if (wait_result == WAIT_OBJECT_0) {

			// Read messages
			do {
				read_status = UDS_Read_2013(tp_handle, &request_msg, NULL, NULL);
				printf("[%llu] Try to read a message (status=%i): %s\n", GetTickCount64(), (int)read_status, STATUS_OK_KO(read_status));
				if (UDS_StatusIsOk_2013(read_status, PUDS_STATUS_OK, false)) {
					display_message(&request_msg);

					// We receive a request, check if it is not a loopback message, if it is not a UUDT message, if the message is not pending, and do not respond if it is not ask
					if ((request_msg.msg.msgdata.any->flags & PCANTP_MSGFLAG_LOOPBACK) == 0
						&& (request_msg.msg.msgdata.isotp->netaddrinfo.msgtype & PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION) == 0
						&& request_msg.msg.msgdata.any->netstatus == PCANTP_NETSTATUS_OK
						&& (request_msg.type&PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE) != PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE
						) {

						// Process response and send it
						process_request(tp_handle, (uint16_t)server_address, &request_msg);
					}
					else {
						printf("\n[%llu]   ...Skipping response...  (flags=0x%X, type=0x%X, netstatus=0x%X, msgtype=0x%X)\n\n", GetTickCount64(), request_msg.msg.msgdata.any->flags, request_msg.type, request_msg.msg.msgdata.any->netstatus, request_msg.msg.msgdata.isotp->netaddrinfo.msgtype);
					}
				}

				// Free request message (and clean memory in order to reallocate later)
				status = UDS_MsgFree_2013(&request_msg);
				printf("Free request message: %s\n", STATUS_OK_KO(status));
			} while (!UDS_StatusIsOk_2013(read_status, PUDS_STATUS_NO_MESSAGE, false));
		}

		// Quit when user press Q
		if (_kbhit()) {
			keyboard_res = _getch();
			if (keyboard_res == 'Q' || keyboard_res == 'q') {
				stop = true;
			}
		}
	}

	// Close receive event
	status = UDS_SetValue_2013(tp_handle, PUDS_PARAMETER_RECEIVE_EVENT, &null_handle, sizeof(HANDLE));
	printf("Stop receive event: %s\n", STATUS_OK_KO(status));
	boolean_status = CloseHandle(receive_event);
	printf("Close receive event: %s\n", OK_KO(boolean_status));

	// Close server
	status = UDS_Uninitialize_2013(tp_handle);
	printf("Uninitialize channel: %s\n", STATUS_OK_KO(status));

	//Exit
	system("PAUSE");
	return EXIT_SUCCESS;
}

