#include "stdafx.h"
#include <stdlib.h>
#include <windows.h>
#include "lib/PCAN-UDS_2013.h"

// Console handling
#define USE_GETCH false
#define CLEAR_CONSOLE if (USE_GETCH) system("cls");

// A global counter to keep track of the number of failed tests (see display_uds_msg function)
int g_nbErr = 0;


/// <summary> A simple function that waits for user input</summary>
void waitGetch() {
	if (USE_GETCH) {
		system("PAUSE");
		CLEAR_CONSOLE;
	}
}

/// <summary>A function that displays UDS Request and Response messages (and count error if no response)</summary>
/// <param name="request">Request message</param>
/// <param name="response">Received response message</param>
/// <param name="no_response_expected">if no response is expected, do not increment error counter</param>
void display_uds_msg(uds_msg *request, uds_msg *response, bool no_response_expected) {
	char buffer[512];
	memset(buffer, 0, sizeof(char) * 512);
	if (request != NULL && request->msg.msgdata.isotp != NULL) {
		printf("\nUDS request from 0x%04X (to 0x%04X, with extension address 0x%02X) - result: %i - %s\n",
			request->msg.msgdata.isotp->netaddrinfo.source_addr, request->msg.msgdata.isotp->netaddrinfo.target_addr,
			request->msg.msgdata.isotp->netaddrinfo.extension_addr, request->msg.msgdata.any->netstatus,
			(request->msg.msgdata.any->netstatus != PCANTP_NETSTATUS_OK) ? "ERROR !!!" : "OK !");
		// display data
		setvbuf(stdout, buffer, _IOLBF, BUFSIZ);
		printf("\t-> Length: %u, Data= ",
			request->msg.msgdata.any->length);
		for (uint32_t i = 0; i < request->msg.msgdata.any->length; i++) {
			printf("%02X ", request->msg.msgdata.any->data[i]);
			fflush(stdout);
		}
		printf("\n");
		setvbuf(stdout, NULL, _IONBF, 0);
	}
	if (response != NULL && response->msg.msgdata.isotp != NULL) {
		printf("\nUDS RESPONSE from 0x%04X (to 0x%04X, with extension address 0x%02X) - result: %i - %s\n",
			response->msg.msgdata.isotp->netaddrinfo.source_addr, response->msg.msgdata.isotp->netaddrinfo.target_addr,
			response->msg.msgdata.isotp->netaddrinfo.extension_addr, response->msg.msgdata.any->netstatus,
			(response->msg.msgdata.any->netstatus != PCANTP_NETSTATUS_OK) ? "ERROR !!!" : "OK !");
		// display data
		setvbuf(stdout, buffer, _IOLBF, BUFSIZ);
		printf("\t-> Length: %u, Data= ", response->msg.msgdata.any->length);
		for (uint32_t i = 0; i < response->msg.msgdata.any->length; i++) {
			printf("%02X ", response->msg.msgdata.any->data[i]);
			fflush(stdout);
		}
		printf("\n");
		setvbuf(stdout, NULL, _IONBF, 0);
	}
	else if (!no_response_expected) {
		printf("\n /!\\ ERROR: NO UDS RESPONSE !!\n\n");
		g_nbErr++;
	}
}

/// <summary>Helper: Inverts the bytes of a 32 bits numeric value</summary>
/// <param name="v">Value to revert</param>
/// <returns>Reverted value</returns>
uint32_t Reverse32(const uint32_t *v) {
	const uint8_t *b = (const uint8_t*)v;
	return b[0] << 24 | b[1] << 16 | b[2] << 8 | b[3];
}

/// <summary>Call UDS Service DiagnosticSessionControl</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testDiagnosticSessionControl(cantp_handle channel, uds_msgconfig config) {
	uds_status status;
	uds_sessioninfo session_info;
	uds_msg request;
	uds_msg response;
	uds_msg confirmation;

	memset(&request, 0, sizeof(uds_msg));
	memset(&response, 0, sizeof(uds_msg));
	memset(&confirmation, 0, sizeof(uds_msg));
	memset(&session_info, 0, sizeof(uds_sessioninfo));

	CLEAR_CONSOLE;
	printf("\n\n*** UDS Service: DiagnosticSessionControl ***\n");

	// Read default session information Server is not yet known (status will be PUDS_ERROR_NOT_INITIALIZED)
	// yet the API will still set session info to default values
	session_info.nai = config.nai;
	status = UDS_GetValue_2013(channel, PUDS_PARAMETER_SESSION_INFO, &session_info, sizeof(uds_sessioninfo));
	printf(" Diagnostic Session Information: %i, 0x%04X => %d = [%04X; %04X]\n", status, session_info.nai.target_addr,
		session_info.session_type, session_info.timeout_p2can_server_max, session_info.timeout_enhanced_p2can_server_max);
	waitGetch();

	// Set Diagnostic session to DEFAULT (to get session information)
	printf("\n\nSetting a DEFAULT Diagnostic Session:\n");
	status = UDS_SvcDiagnosticSessionControl_2013(channel, config, &request, PUDS_SVC_PARAM_DSC_DS);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcDiagnosticSessionControl_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);

	// Read current session information
	memset(&session_info, 0, sizeof(session_info));
	session_info.nai = config.nai;
	status = UDS_GetValue_2013(channel, PUDS_PARAMETER_SESSION_INFO, &session_info, sizeof(uds_sessioninfo));
	printf(" Diagnostic Session Information: %i, 0x%04X => %d = [%04X; %04X]\n", status, session_info.nai.target_addr,
		session_info.session_type, session_info.timeout_p2can_server_max, session_info.timeout_enhanced_p2can_server_max);
	waitGetch();

	// Set Diagnostic session to PROGRAMMING
	printf("\n\nSetting a ECUPS Diagnostic Session:\n");
	status = UDS_SvcDiagnosticSessionControl_2013(channel, config, &request, PUDS_SVC_PARAM_DSC_ECUPS);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcDiagnosticSessionControl_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);

	// Read current session information
	memset(&session_info, 0, sizeof(session_info));
	session_info.nai = config.nai;
	status = UDS_GetValue_2013(channel, PUDS_PARAMETER_SESSION_INFO, &session_info, sizeof(uds_sessioninfo));
	printf(" Diagnostic Session Information: %i, 0x%04X => %d = [%04X; %04X]\n", status, session_info.nai.target_addr,
		session_info.session_type, session_info.timeout_p2can_server_max, session_info.timeout_enhanced_p2can_server_max);
	printf(" Assert that Auto TesterPresent Frame is sent...\n");
	Sleep(2000);
	printf(" Should transmit an Auto TesterPresent Frame\n");
	Sleep(2000);
	printf(" Should transmit an Auto TesterPresent Frame\n");
	waitGetch();

	// Set Diagnostic session back to DEFAULT
	printf("\n\nSetting a DEFAULT Diagnostic Session:\n");
	status = UDS_SvcDiagnosticSessionControl_2013(channel, config, &request, PUDS_SVC_PARAM_DSC_DS);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcDiagnosticSessionControl_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	printf(" Assert that NO Auto TesterPresent Frame is sent...\n");
	Sleep(2000);
	printf(" Should NOT transmit an Auto TesterPresent Frame\n");
	Sleep(2000);
	printf(" Should NOT transmit an Auto TesterPresent Frame\n");
	waitGetch();
}

/// <summary>Call UDS Service ECUReset</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testECUReset(cantp_handle channel, uds_msgconfig config) {
	uds_status status;
	uds_msg request;
	uds_msg response;
	uds_msg confirmation;

	memset(&request, 0, sizeof(uds_msg));
	memset(&response, 0, sizeof(uds_msg));
	memset(&confirmation, 0, sizeof(uds_msg));

	CLEAR_CONSOLE;
	printf("\n\n*** UDS Service: ECUReset ***\n");

	// Sends a physical ECUReset message
	printf("\n\nSends a physical ECUReset message: \n");
	status = UDS_SvcECUReset_2013(channel, config, &request, PUDS_SVC_PARAM_ER_SR);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcECUReset_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();
}

/// <summary>Call UDS Service SecurityAccess</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testSecurityAccess(cantp_handle channel, uds_msgconfig config) {
	uds_status status;
	uds_msg request;
	uds_msg response;
	uint32_t security_access_data;
	uint32_t value_little_endian;
	uds_msg confirmation;

	memset(&request, 0, sizeof(uds_msg));
	memset(&response, 0, sizeof(uds_msg));
	memset(&confirmation, 0, sizeof(uds_msg));

	CLEAR_CONSOLE;
	printf("\n\n*** UDS Service: SecurityAccess ***\n");

	// Sends a physical SecurityAccess message
	printf("\n\nSends a physical SecurityAccess message: \n");
	value_little_endian = 0xF0A1B2C3;
	security_access_data = Reverse32(&value_little_endian); // use helper function to set MSB as 1st byte in the buffer (Win32 uses little endian format)
	status = UDS_SvcSecurityAccess_2013(channel, config, &request, PUDS_SVC_PARAM_SA_RSD_1, (uint8_t*)&security_access_data, 4);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcSecurityAccess_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();
}

/// <summary>Call UDS Service CommunicationControl</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testCommunicationControl(cantp_handle channel, uds_msgconfig config) {
	uds_status status;
	uds_msg request;
	uds_msg response;
	uds_msg confirmation;

	memset(&request, 0, sizeof(uds_msg));
	memset(&response, 0, sizeof(uds_msg));
	memset(&confirmation, 0, sizeof(uds_msg));

	CLEAR_CONSOLE;
	printf("\n\n*** UDS Service: CommunicationControl ***\n");

	// Sends a physical CommunicationControl message
	printf("\n\nSends a physical CommunicationControl message: \n");
	status = UDS_SvcCommunicationControl_2013(channel, config, &request,
		PUDS_SVC_PARAM_CC_ERXTX, PUDS_SVC_PARAM_CC_FLAG_APPL | PUDS_SVC_PARAM_CC_FLAG_NWM | PUDS_SVC_PARAM_CC_FLAG_DENWRIRO, 0);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcCommunicationControl_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();
}

/// <summary>UDS Call Service TesterPresent</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testTesterPresent(cantp_handle channel, uds_msgconfig config) {
	uds_status status;
	uds_msg request;
	uds_msg response;
	uds_msg confirmation;
	uint32_t response_count = 0;

	memset(&request, 0, sizeof(uds_msg));
	memset(&response, 0, sizeof(uds_msg));
	memset(&confirmation, 0, sizeof(uds_msg));

	CLEAR_CONSOLE;
	printf("\n\n*** UDS Service: TesterPresent ***\n");

	// Sends a physical TesterPresent message
	printf("\n\nSends a physical TesterPresent message: \n");
	status = UDS_SvcTesterPresent_2013(channel, config, &request, PUDS_SVC_PARAM_TP_ZSUBF);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcTesterPresent_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();

	// Sends a physical TesterPresent message with no positive response
	config.type = PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE;
	printf("\n\nSends a physical TesterPresent message with no positive response:\n");
	status = UDS_SvcTesterPresent_2013(channel, config, &request, PUDS_SVC_PARAM_TP_ZSUBF);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcTesterPresent_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, true);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();

	// Sends a functional TesterPresent message
	config.type = PUDS_MSGTYPE_USDT;
	config.nai.target_type = PCANTP_ISOTP_ADDRESSING_FUNCTIONAL;
	config.nai.target_addr = PUDS_ISO_15765_4_ADDR_OBD_FUNCTIONAL;
	printf("\n\nSends a functional TesterPresent message: \n");
	status = UDS_SvcTesterPresent_2013(channel, config, &request, PUDS_SVC_PARAM_TP_ZSUBF);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForServiceFunctional_2013(channel, &request, 1, true, &response, &response_count, &confirmation);
	printf(" UDS_SvcTesterPresent_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();

	// Sends a functional TesterPresent message with no positive response
	config.type = PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE;
	config.nai.target_type = PCANTP_ISOTP_ADDRESSING_FUNCTIONAL;
	config.nai.target_addr = PUDS_ISO_15765_4_ADDR_OBD_FUNCTIONAL;
	printf("\n\nSends a functional TesterPresent message with no positive response:\n");
	status = UDS_SvcTesterPresent_2013(channel, config, &request, PUDS_SVC_PARAM_TP_ZSUBF);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForServiceFunctional_2013(channel, &request, 1, true, &response, &response_count, &confirmation);
	printf(" UDS_SvcTesterPresent_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false) && response_count != 0)
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, true);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();
}

/// <summary>Call UDS Service SecuredDataTransmission</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testSecuredDataTransmission(cantp_handle channel, uds_msgconfig config) {

	uds_status status;
	uds_msg request;
	uds_msg response;
	uint32_t security_data_request_record;
	uint32_t value_little_endian;
	uds_msg ecureset_request;
	uds_msg confirmation;
	uint16_t administrative_parameter;
	uint8_t signature_encryption_calculation;
	uint16_t anti_replay_counter;
	uint8_t internal_service_identifier;
	uint8_t service_specific_parameters[4];
	uint32_t service_specific_parameters_size;
	uint8_t signature_mac[6];
	uint16_t signature_size;

	memset(&request, 0, sizeof(uds_msg));
	memset(&response, 0, sizeof(uds_msg));
	memset(&confirmation, 0, sizeof(uds_msg));
	memset(&ecureset_request, 0, sizeof(uds_msg));

	CLEAR_CONSOLE;
	printf("\n\n*** UDS Service: SecuredDataTransmission ***\n");

	// Sends a physical SecuredDataTransmission/2013 message
	printf("\n\nSends a physical SecuredDataTransmission/2013 message: \n");
	value_little_endian = 0xF0A1B2C3;
	security_data_request_record = Reverse32(&value_little_endian); // use helper function to set MSB as 1st byte in the buffer (Win32 uses little endian format)
	status = UDS_SvcSecuredDataTransmission_2013(channel, config, &request, (uint8_t*)&security_data_request_record, 4);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcSecuredDataTransmission_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();

	// Sends a physical SecuredDataTransmission/2013 message prepared with PUDS_ONLY_PREPARE_REQUEST option
	printf("\n\nSends a physical SecuredDataTransmission/2013 prepared with PUDS_ONLY_PREPARE_REQUEST option: \n");
	status = UDS_SvcECUReset_2013(PUDS_ONLY_PREPARE_REQUEST, config, &ecureset_request, PUDS_SVC_PARAM_ER_HR);
	printf(" Prepare ECUReset request for SecuredDataTransmission: %i\n", status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_SvcSecuredDataTransmission_2013(channel, config, &request, ecureset_request.msg.msgdata.any->data, ecureset_request.msg.msgdata.any->length);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcSecuredDataTransmission_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&ecureset_request);
	printf(" Free prepared message: %i\n", status);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();

	// Sends a physical SecuredDataTransmission/2020 message
	printf("\n\nSends a physical SecuredDataTransmission/2020 message: \n");
	administrative_parameter = PUDS_SVC_PARAM_APAR_REQUEST_MSG_FLAG | PUDS_SVC_PARAM_APAR_REQUEST_RESPONSE_SIGNATURE_FLAG | PUDS_SVC_PARAM_APAR_SIGNED_MSG_FLAG;
	signature_encryption_calculation = 0x0;
	anti_replay_counter = 0x0124;
	internal_service_identifier = 0x2E;
	service_specific_parameters[0] = 0xF1;
	service_specific_parameters[1] = 0x23;
	service_specific_parameters[2] = 0xAA;
	service_specific_parameters[3] = 0x55;
	service_specific_parameters_size = 4;
	signature_mac[0] = 0xDB;
	signature_mac[1] = 0xD1;
	signature_mac[2] = 0x0E;
	signature_mac[3] = 0xDC;
	signature_mac[4] = 0x55;
	signature_mac[5] = 0xAA;
	signature_size = 0x0006;
	status = UDS_SvcSecuredDataTransmission_2020(channel, config, &request, administrative_parameter, signature_encryption_calculation, anti_replay_counter, internal_service_identifier, service_specific_parameters, service_specific_parameters_size, signature_mac, signature_size);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcSecuredDataTransmission_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();
}

/// <summary>Call UDS Service ControlDTCSetting</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testControlDTCSetting(cantp_handle channel, uds_msgconfig config) {
	uds_status status;
	uds_msg request;
	uds_msg response;
	uint32_t dtc_setting_control_option_record;
	uint32_t value_little_endian;
	uds_msg confirmation;

	memset(&request, 0, sizeof(uds_msg));
	memset(&response, 0, sizeof(uds_msg));
	memset(&confirmation, 0, sizeof(uds_msg));

	CLEAR_CONSOLE;
	printf("\n\n*** UDS Service: ControlDTCSetting ***\n");

	// Sends a physical ControlDTCSetting message
	printf("\n\nSends a physical ControlDTCSetting message: \n");
	value_little_endian = 0xF1A1B2EE;
	dtc_setting_control_option_record = Reverse32(&value_little_endian); // use helper function to set MSB as 1st byte in the buffer (Win32 uses little endian format)
	status = UDS_SvcControlDTCSetting_2013(channel, config, &request, PUDS_SVC_PARAM_CDTCS_OFF, (uint8_t*)&dtc_setting_control_option_record, 3);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcControlDTCSetting_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();
}

/// <summary>Call UDS Service ResponseOnEvent</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testResponseOnEvent(cantp_handle channel, uds_msgconfig config) {
	uds_status status;
	uds_msg request;
	uds_msg response;
	uds_msg confirmation;
	uint8_t event_type_record[50];
	uint8_t service_to_respond_to_record[50];

	memset(&request, 0, sizeof(uds_msg));
	memset(&response, 0, sizeof(uds_msg));
	memset(&confirmation, 0, sizeof(uds_msg));

	CLEAR_CONSOLE;
	printf("\n\n*** UDS Service: ResponseOnEvent ***\n");

	// Sends a physical ResponseOnEvent message
	printf("\n\nSends a physical ResponseOnEvent message: \n");
	event_type_record[0] = 0x08;
	service_to_respond_to_record[0] = PUDS_SI_ReadDTCInformation;
	service_to_respond_to_record[1] = PUDS_SVC_PARAM_RDTCI_RNODTCBSM;
	service_to_respond_to_record[2] = 0x01;
	status = UDS_SvcResponseOnEvent_2013(channel, config, &request, PUDS_SVC_PARAM_ROE_ONDTCS, FALSE, 0x08, event_type_record,
		PUDS_SVC_PARAM_ROE_OTI_LEN, service_to_respond_to_record, 3);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcResponseOnEvent_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();
}

/// <summary>Call UDS Service LinkControl</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testLinkControl(cantp_handle channel, uds_msgconfig config) {

	uds_status status;
	uds_msg request;
	uds_msg response;
	uds_msg confirmation;

	memset(&request, 0, sizeof(uds_msg));
	memset(&response, 0, sizeof(uds_msg));
	memset(&confirmation, 0, sizeof(uds_msg));

	CLEAR_CONSOLE;
	printf("\n\n*** UDS Service: LinkControl ***\n");

	// Sends a physical LinkControl message
	printf("\n\nSends a physical LinkControl message (Verify Fixed Baudrate): \n");
	status = UDS_SvcLinkControl_2013(channel, config, &request, PUDS_SVC_PARAM_LC_VBTWFBR, PUDS_SVC_PARAM_LC_BAUDRATE_CAN_500K, 0);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcLinkControl_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);

	// Sends a physical LinkControl message
	printf("\n\nSends a physical LinkControl message (Verify Specific Baudrate): \n");
	status = UDS_SvcLinkControl_2013(channel, config, &request, PUDS_SVC_PARAM_LC_VBTWSBR, 0, 500000); // 500K = 0x0007a120
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcLinkControl_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);

	// Sends a physical LinkControl message
	printf("\n\nSends a physical LinkControl message (Transition): \n");
	status = UDS_SvcLinkControl_2013(channel, config, &request, PUDS_SVC_PARAM_LC_TB, 0, 0);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcLinkControl_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();
}

/// <summary>Call UDS Service ReadDataByIdentifier</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testReadDataByIdentifier(cantp_handle channel, uds_msgconfig config) {

	uds_status status;
	uds_msg request;
	uds_msg response;
	uint16_t data_identifier[2] = { PUDS_SVC_PARAM_DI_ADSDID, PUDS_SVC_PARAM_DI_ECUMDDID };
	uds_msg confirmation;

	memset(&request, 0, sizeof(uds_msg));
	memset(&response, 0, sizeof(uds_msg));
	memset(&confirmation, 0, sizeof(uds_msg));

	CLEAR_CONSOLE;
	printf("\n\n*** UDS Service: ReadDataByIdentifier ***\n");

	// Sends a physical ReadDataByIdentifier message
	printf("\n\nSends a physical ReadDataByIdentifier message: \n");
	status = UDS_SvcReadDataByIdentifier_2013(channel, config, &request, data_identifier, 2);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcReadDataByIdentifier_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();
}

/// <summary>Call UDS Service ReadMemoryByAddress</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testReadMemoryByAddress(cantp_handle channel, uds_msgconfig config) {
	uds_status status;
	uds_msg request;
	uds_msg response;
	uds_msg confirmation;
	uint8_t memory_address_buffer[10];
	uint8_t memory_size_buffer[10];
	uint8_t memory_address_size = 10;
	uint8_t memory_size_size = 3;

	memset(&request, 0, sizeof(uds_msg));
	memset(&response, 0, sizeof(uds_msg));
	memset(&confirmation, 0, sizeof(uds_msg));
	memset(memory_address_buffer, 0, sizeof(uint8_t) * 10);
	memset(memory_size_buffer, 0, sizeof(uint8_t) * 10);

	CLEAR_CONSOLE;
	printf("\n\n*** UDS Service: ReadMemoryByAddress ***\n");

	// Sends a physical ReadMemoryByAddress message
	printf("\n\nSends a physical ReadMemoryByAddress message: \n");
	for (int i = 0; i < memory_address_size; i++) {
		memory_address_buffer[i] = 'A' + i;
		memory_size_buffer[i] = '1' + i;
	}
	status = UDS_SvcReadMemoryByAddress_2013(channel, config, &request, memory_address_buffer, memory_address_size, memory_size_buffer, memory_size_size);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcReadMemoryByAddress_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();
}

/// <summary>Call UDS Service ReadScalingDataByIdentifier</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testReadScalingDataByIdentifier(cantp_handle channel, uds_msgconfig config) {
	uds_status status;
	uds_msg request;
	uds_msg response;
	uds_msg confirmation;

	memset(&request, 0, sizeof(uds_msg));
	memset(&response, 0, sizeof(uds_msg));
	memset(&confirmation, 0, sizeof(uds_msg));

	CLEAR_CONSOLE;
	printf("\n\n*** UDS Service: ReadScalingDataByIdentifier ***\n");

	// Sends a physical ReadScalingDataByIdentifier message
	printf("\n\nSends a physical ReadScalingDataByIdentifier message: \n");
	status = UDS_SvcReadScalingDataByIdentifier_2013(channel, config, &request, PUDS_SVC_PARAM_DI_BSFPDID);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcReadScalingDataByIdentifier_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();
}

/// <summary>Call UDS Service ReadDataByPeriodicIdentifier</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testReadDataByPeriodicIdentifier(cantp_handle channel, uds_msgconfig config) {
	uds_status status;
	uds_msg request;
	uds_msg response;
	uds_msg confirmation;
	uint8_t periodic_data_identifier[10];
	uint16_t periodic_data_identifier_size = 10;

	memset(&request, 0, sizeof(uds_msg));
	memset(&response, 0, sizeof(uds_msg));
	memset(periodic_data_identifier, 0, sizeof(uint8_t) * 10);
	memset(&confirmation, 0, sizeof(uds_msg));

	CLEAR_CONSOLE;
	printf("\n\n*** UDS Service: ReadDataByPeriodicIdentifier ***\n");

	// Sends a physical ReadScalingDataByIdentifier message
	printf("\n\nSends a physical ReadDataByPeriodicIdentifier message: \n");
	for (int i = 0; i < periodic_data_identifier_size; i++)
		periodic_data_identifier[i] = 'A' + i;
	status = UDS_SvcReadDataByPeriodicIdentifier_2013(channel, config, &request, PUDS_SVC_PARAM_RDBPI_SAMR, periodic_data_identifier, periodic_data_identifier_size);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcReadDataByPeriodicIdentifier_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();
}

/// <summary>Call UDS Service DynamicallyDefineDataIdentifier</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testDynamicallyDefineDataIdentifier(cantp_handle channel, uds_msgconfig config) {
	uds_status status;
	uds_msg request;
	uds_msg response;
	uds_msg confirmation;
	uint16_t source_data_identifier[20];
	uint8_t memory_size[20];
	uint8_t position_in_source_data_record[20];
	uint16_t number_of_elements = 10;
	uint8_t memory_address_buffer[15];
	uint8_t memory_size_buffer[9];
	uint8_t memory_address_size;
	uint8_t memory_size_size;

	memset(&request, 0, sizeof(uds_msg));
	memset(&response, 0, sizeof(uds_msg));
	memset(&confirmation, 0, sizeof(uds_msg));
	memset(source_data_identifier, 0, sizeof(uint16_t) * 20);
	memset(memory_size, 0, sizeof(uint8_t) * 20);
	memset(position_in_source_data_record, 0, sizeof(uint8_t) * 20);
	memset(memory_address_buffer, 0, sizeof(uint8_t) * 15);
	memset(memory_size_buffer, 0, sizeof(uint8_t) * 9);

	CLEAR_CONSOLE;
	printf("\n\n*** UDS Service: DynamicallyDefineDataIdentifier ***\n");

	// Sends a physical DynamicallyDefineDataIdentifierDBID message
	printf("\n\nSends a physical DynamicallyDefineDataIdentifierDBID message: \n");
	for (int i = 0; i < number_of_elements; i++) {
		source_data_identifier[i] = ((0xF0 + i) << 8) + ('A' + i);
		memory_size[i] = i + 1;
		position_in_source_data_record[i] = 100 + i;
	}
	status = UDS_SvcDynamicallyDefineDataIdentifierDBID_2013(channel, config, &request, PUDS_SVC_PARAM_DI_CDDID, source_data_identifier,
		memory_size, position_in_source_data_record, number_of_elements);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcDynamicallyDefineDataIdentifierDBID_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();

	// Sends a physical UDS_SvcDynamicallyDefineDataIdentifierDBMA message
	printf("\n\nSends a physical UDS_SvcDynamicallyDefineDataIdentifierDBMA_2013 message: \n");
	number_of_elements = 3;
	memory_address_size = 5;
	memory_size_size = 3;
	for (int j = 0; j < number_of_elements; j++) {
		for (int i = 0; i < memory_address_size; i++)
			memory_address_buffer[memory_address_size * j + i] = (10 * j) + i + 1;
		for (int i = 0; i < memory_size_size; i++)
			memory_size_buffer[memory_size_size * j + i] = 100 + (10 * j) + i + 1;
	}
	status = UDS_SvcDynamicallyDefineDataIdentifierDBMA_2013(channel, config, &request, PUDS_SVC_PARAM_DI_CESWNDID, memory_address_size,
		memory_size_size, memory_address_buffer, memory_size_buffer, number_of_elements);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcDynamicallyDefineDataIdentifierDBMA_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();

	// Sends a physical UDS_SvcDynamicallyDefineDataIdentifierCDDDI message
	printf("\n\nSends a physical UDS_SvcDynamicallyDefineDataIdentifierCDDDI_2013 message: \n");
	status = UDS_SvcDynamicallyDefineDataIdentifierCDDDI_2013(channel, config, &request, PUDS_SVC_PARAM_DI_CESWNDID);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcDynamicallyDefineDataIdentifierCDDDI_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();

	// Sends a physical UDS_SvcDynamicallyDefineDataIdentifierClearAllDDDI_2013 message
	printf("\n\nSends a physical UDS_SvcDynamicallyDefineDataIdentifierClearAllDDDI_2013 message: \n");
	status = UDS_SvcDynamicallyDefineDataIdentifierClearAllDDDI_2013(channel, config, &request);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcDynamicallyDefineDataIdentifierClearAllDDDI_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();
}

/// <summary>Call UDS Service WriteDataByIdentifier</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testWriteDataByIdentifier(cantp_handle channel, uds_msgconfig config) {
	uds_status status;
	uds_msg request;
	uds_msg response;
	uds_msg confirmation;
	uint8_t data_record[10];
	uint16_t data_record_size = 10;

	memset(&request, 0, sizeof(uds_msg));
	memset(&response, 0, sizeof(uds_msg));
	memset(&confirmation, 0, sizeof(uds_msg));
	memset(data_record, 0, sizeof(uint8_t) * 10);

	CLEAR_CONSOLE;
	printf("\n\n*** UDS Service: WriteDataByIdentifier ***\n");

	// Sends a physical WriteDataByIdentifier message
	printf("\n\nSends a physical WriteDataByIdentifier message: \n");
	for (int i = 0; i < data_record_size; i++)
		data_record[i] = 'A' + i;
	status = UDS_SvcWriteDataByIdentifier_2013(channel, config, &request, PUDS_SVC_PARAM_DI_ASFPDID, data_record, data_record_size);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcWriteDataByIdentifier_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();
}

/// <summary>Call UDS Service WriteMemoryByAddress</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testWriteMemoryByAddress(cantp_handle channel, uds_msgconfig config) {
	uds_status status;
	uds_msg request;
	uds_msg response;
	uds_msg confirmation;
	uint8_t data_record[50];
	uint8_t memory_address_buffer[50];
	uint8_t memory_size_buffer[50];
	uint16_t data_record_size = 50;
	uint8_t memory_address_size = 5;
	uint8_t memory_size_size = 3;

	memset(&request, 0, sizeof(uds_msg));
	memset(&response, 0, sizeof(uds_msg));
	memset(&confirmation, 0, sizeof(uds_msg));
	memset(data_record, 0, sizeof(uint8_t) * 50);
	memset(memory_address_buffer, 0, sizeof(uint8_t) * 50);
	memset(memory_size_buffer, 0, sizeof(uint8_t) * 50);

	CLEAR_CONSOLE;
	printf("\n\n*** UDS Service: WriteMemoryByAddress ***\n");

	// Sends a physical WriteMemoryByAddress message
	printf("\n\nSends a physical WriteMemoryByAddress message: \n");
	for (int i = 0; i < data_record_size; i++) {
		data_record[i] = i + 1;
		memory_address_buffer[i] = 'A' + i;
		memory_size_buffer[i] = 10 + i;
	}
	status = UDS_SvcWriteMemoryByAddress_2013(channel, config, &request, memory_address_buffer, memory_address_size, memory_size_buffer, memory_size_size,
		data_record, data_record_size);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcWriteMemoryByAddress_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();
}

/// <summary>Call UDS Service ClearDiagnosticInformation</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testClearDiagnosticInformation(cantp_handle channel, uds_msgconfig config) {
	uds_status status;
	uds_msg request;
	uds_msg response;
	uds_msg confirmation;

	memset(&request, 0, sizeof(uds_msg));
	memset(&response, 0, sizeof(uds_msg));
	memset(&confirmation, 0, sizeof(uds_msg));

	CLEAR_CONSOLE;
	printf("\n\n*** UDS Service: ClearDiagnosticInformation ***\n");

	// Sends a physical ClearDiagnosticInformation message
	printf("\n\nSends a physical ClearDiagnosticInformation message: \n");
	status = UDS_SvcClearDiagnosticInformation_2013(channel, config, &request, 0xF1A2B3);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcClearDiagnosticInformation_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();

	// Sends a physical ClearDiagnosticInformation message with memory selection parameter (2020)
	printf("\n\nSends a physical ClearDiagnosticInformation message with memory selection parameter: \n");
	status = UDS_SvcClearDiagnosticInformation_2020(channel, config, &request, 0xF1A2B3, 0x42);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcClearDiagnosticInformation_2020: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();
}

/// <summary>Call UDS Service ReadDTCInformation</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testReadDTCInformation(cantp_handle channel, uds_msgconfig config) {
	uds_status status;
	uds_msg request;
	uds_msg response;
	uds_msg confirmation;

	memset(&request, 0, sizeof(uds_msg));
	memset(&response, 0, sizeof(uds_msg));
	memset(&confirmation, 0, sizeof(uds_msg));

	CLEAR_CONSOLE;
	printf("\n\n*** UDS Service: ReadDTCInformation ***\n");

	// Sends a physical ReadDTCInformation message
	printf("\n\nSends a physical ReadDTCInformation message: \n");
	status = UDS_SvcReadDTCInformation_2013(channel, config, &request, PUDS_SVC_PARAM_RDTCI_RNODTCBSM, 0xF1);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcReadDTCInformation_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();

	// Sends a physical ReadDTCInformationRDTCSSBDTC message
	printf("\n\nSends a physical ReadDTCInformationRDTCSSBDTC message: \n");
	status = UDS_SvcReadDTCInformationRDTCSSBDTC_2013(channel, config, &request, 0x00A1B2B3, 0x12);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcReadDTCInformationRDTCSSBDTC_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();

	// Sends a physical ReadDTCInformationRDTCSSBRN message
	printf("\n\nSends a physical ReadDTCInformationRDTCSSBRN message: \n");
	status = UDS_SvcReadDTCInformationRDTCSSBRN_2013(channel, config, &request, 0x12);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcReadDTCInformationRDTCSSBRN_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();

	// Sends a physical ReadDTCInformationReportExtended message
	printf("\n\nSends a physical ReadDTCInformationReportExtended message: \n");
	status = UDS_SvcReadDTCInformationReportExtended_2013(channel, config, &request, PUDS_SVC_PARAM_RDTCI_RDTCEDRBDN, 0x00A1B2B3,
		0x12);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcReadDTCInformationReportExtended_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();

	// Sends a physical ReadDTCInformationReportSeverity message
	printf("\n\nSends a physical ReadDTCInformationReportSeverity message: \n");
	status = UDS_SvcReadDTCInformationReportSeverity_2013(channel, config, &request, PUDS_SVC_PARAM_RDTCI_RNODTCBSMR, 0xF1, 0x12);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcReadDTCInformationReportSeverity_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();

	// Sends a physical ReadDTCInformationRSIODTC message
	printf("\n\nSends a physical ReadDTCInformationRSIODTC message: \n");
	status = UDS_SvcReadDTCInformationRSIODTC_2013(channel, config, &request, 0xF1A1B2B3);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcReadDTCInformationRSIODTC_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();

	// Sends a physical ReadDTCInformationNoParam message
	printf("\n\nSends a physical ReadDTCInformationNoParam message: \n");
	status = UDS_SvcReadDTCInformationNoParam_2013(channel, config, &request, PUDS_SVC_PARAM_RDTCI_RSUPDTC);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcReadDTCInformationNoParam_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();

	// Sends a physical UDS_SvcReadDTCInformationRDTCEDBR_2013 message
	printf("\n\nSends a physical UDS_SvcReadDTCInformationRDTCEDBR_2013 message: \n");
	status = UDS_SvcReadDTCInformationRDTCEDBR_2013(channel, config, &request, 0x12);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcReadDTCInformationRDTCEDBR_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();

	// Sends a physical UDS_SvcReadDTCInformationRUDMDTCBSM_2013 message
	printf("\n\nSends a physical UDS_SvcReadDTCInformationRUDMDTCBSM_2013 message: \n");
	status = UDS_SvcReadDTCInformationRUDMDTCBSM_2013(channel, config, &request, 0x12, 0x34);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcReadDTCInformationRUDMDTCBSM_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();

	// Sends a physical UDS_SvcReadDTCInformationRUDMDTCSSBDTC_2013 message
	printf("\n\nSends a physical UDS_SvcReadDTCInformationRUDMDTCSSBDTC_2013 message: \n");
	status = UDS_SvcReadDTCInformationRUDMDTCSSBDTC_2013(channel, config, &request, 0x00123456, 0x78, 0x9A);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcReadDTCInformationRUDMDTCSSBDTC_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();

	// Sends a physical UDS_SvcReadDTCInformationRUDMDTCEDRBDN_2013 message
	printf("\n\nSends a physical UDS_SvcReadDTCInformationRUDMDTCEDRBDN_2013 message: \n");
	status = UDS_SvcReadDTCInformationRUDMDTCEDRBDN_2013(channel, config, &request, 0x00123456, 0x78, 0x9A);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcReadDTCInformationRUDMDTCEDRBDN_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();

	// Sends a physical UDS_SvcReadDTCInformationRDTCEDI_2020 message
	printf("\n\nSends a physical UDS_SvcReadDTCInformationRDTCEDI_2020 message: \n");
	status = UDS_SvcReadDTCInformationRDTCEDI_2020(channel, config, &request, 0x12);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcReadDTCInformationRDTCEDI_2020: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();

	// Sends a physical UDS_SvcReadDTCInformationRWWHOBDDTCBMR_2013 message
	printf("\n\nSends a physical UDS_SvcReadDTCInformationRWWHOBDDTCBMR_2013 message: \n");
	status = UDS_SvcReadDTCInformationRWWHOBDDTCBMR_2013(channel, config, &request, 0x12, 0x34, 0x56);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcReadDTCInformationRWWHOBDDTCBMR_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();

	// Sends a physical UDS_SvcReadDTCInformationRWWHOBDDTCWPS_2013 message
	printf("\n\nSends a physical UDS_SvcReadDTCInformationRWWHOBDDTCWPS_2013 message: \n");
	status = UDS_SvcReadDTCInformationRWWHOBDDTCWPS_2013(channel, config, &request, 0x12);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcReadDTCInformationRWWHOBDDTCWPS_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();

	// Sends a physical UDS_SvcReadDTCInformationRDTCBRGI_2020 message
	printf("\n\nSends a physical UDS_SvcReadDTCInformationRDTCBRGI_2020 message: \n");
	status = UDS_SvcReadDTCInformationRDTCBRGI_2020(channel, config, &request, 0x12, 0x34);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcReadDTCInformationRDTCBRGI_2020: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();
}

/// <summary>Call UDS Service InputOutputControlByIdentifier</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testInputOutputControlByIdentifier(cantp_handle channel, uds_msgconfig config) {
	uds_status status;
	uds_msg request;
	uds_msg response;
	uds_msg confirmation;
	uint8_t control_option_record[10];
	uint8_t control_enable_mask_record[10];
	uint16_t control_option_record_size = 10;
	uint16_t control_enable_mask_record_size = 5;

	memset(&request, 0, sizeof(uds_msg));
	memset(&response, 0, sizeof(uds_msg));
	memset(&confirmation, 0, sizeof(uds_msg));
	memset(control_option_record, 0, sizeof(uint8_t) * 10);
	memset(control_enable_mask_record, 0, sizeof(uint8_t) * 10);

	CLEAR_CONSOLE;
	printf("\n\n*** UDS Service: InputOutputControlByIdentifier ***\n");

	// Sends a physical InputOutputControlByIdentifier message
	printf("\n\nSends a physical InputOutputControlByIdentifier message: \n");
	for (int i = 0; i < control_option_record_size; i++) {
		control_option_record[i] = 'A' + i;
		control_enable_mask_record[i] = 10 + i;
	}
	status = UDS_SvcInputOutputControlByIdentifier_2013(channel, config, &request, PUDS_SVC_PARAM_DI_SSECUSWVNDID, control_option_record,
		control_option_record_size, control_enable_mask_record, control_enable_mask_record_size);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcInputOutputControlByIdentifier_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();
}

/// <summary>Call UDS Service RoutineControl</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testRoutineControl(cantp_handle channel, uds_msgconfig config) {

	uds_status status;
	uds_msg request;
	uds_msg response;
	uds_msg confirmation;
	uint8_t routine_control_option_record[10];
	uint16_t routine_control_option_record_size = 10;

	memset(&request, 0, sizeof(uds_msg));
	memset(&response, 0, sizeof(uds_msg));
	memset(&confirmation, 0, sizeof(uds_msg));
	memset(routine_control_option_record, 0, sizeof(uint8_t) * 10);

	CLEAR_CONSOLE;
	printf("\n\n*** UDS Service: RoutineControl ***\n");

	// Sends a physical RoutineControl message
	printf("\n\nSends a physical RoutineControl message: \n");
	for (int i = 0; i < routine_control_option_record_size; i++) {
		routine_control_option_record[i] = 'A' + i;
	}
	status = UDS_SvcRoutineControl_2013(channel, config, &request, PUDS_SVC_PARAM_RC_RRR, 0xF1A2, routine_control_option_record, routine_control_option_record_size);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcRoutineControl_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();
}

/// <summary>Call UDS Service RequestDownload</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testRequestDownload(cantp_handle channel, uds_msgconfig config) {
	uds_status status;
	uds_msg request;
	uds_msg response;
	uds_msg confirmation;
	uint8_t memory_address_buffer[15];
	uint8_t memory_addres_size[15];
	uint8_t memory_address_size = 15;
	uint8_t memory_size_size = 8;

	memset(&request, 0, sizeof(uds_msg));
	memset(&response, 0, sizeof(uds_msg));
	memset(&confirmation, 0, sizeof(uds_msg));
	memset(memory_address_buffer, 0, sizeof(uint8_t) * 15);
	memset(memory_addres_size, 0, sizeof(uint8_t) * 15);

	CLEAR_CONSOLE;
	printf("\n\n*** UDS Service: RequestDownload ***\n");

	// Sends a physical RequestDownload message
	printf("\n\nSends a physical RequestDownload message: \n");
	for (int i = 0; i < memory_address_size; i++) {
		memory_address_buffer[i] = 'A' + i;
		memory_addres_size[i] = 10 + i;
	}
	status = UDS_SvcRequestDownload_2013(channel, config, &request, 0x01, 0x02, memory_address_buffer, memory_address_size, memory_addres_size,
		memory_size_size);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcRequestDownload_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();
}

/// <summary>Call UDS Service RequestUpload</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testRequestUpload(cantp_handle channel, uds_msgconfig config) {
	uds_status status;
	uds_msg request;
	uds_msg response;
	uds_msg confirmation;
	uint8_t memory_address_buffer[4];
	uint8_t memory_size_buffer[4];
	uint8_t memory_address_buffer_size = 4;
	uint8_t memory_size_size = 4;

	memset(&request, 0, sizeof(uds_msg));
	memset(&response, 0, sizeof(uds_msg));
	memset(&confirmation, 0, sizeof(uds_msg));
	memset(memory_address_buffer, 0, sizeof(uint8_t) * 4);
	memset(memory_size_buffer, 0, sizeof(uint8_t) * 4);

	CLEAR_CONSOLE;
	printf("\n\n*** UDS Service: RequestUpload ***\n");

	// Sends a physical RequestUpload message
	printf("\n\nSends a physical RequestUpload message: \n");
	for (int i = 0; i < memory_size_size; i++) {
		memory_address_buffer[i] = 'A' + i;
		memory_size_buffer[i] = 10 + i;
	}
	status = UDS_SvcRequestUpload_2013(channel, config, &request, 0x01, 0x02, memory_address_buffer, memory_address_buffer_size, memory_size_buffer,
		memory_size_size);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcRequestUpload_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();
}

/// <summary>Call UDS Service RequestTransferExit</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testRequestTransferExit(cantp_handle channel, uds_msgconfig config) {
	uds_status status;
	uds_msg request;
	uds_msg response;
	uds_msg confirmation;
	uint8_t record[20];
	uint8_t record_size = 20;

	memset(&request, 0, sizeof(uds_msg));
	memset(&response, 0, sizeof(uds_msg));
	memset(&confirmation, 0, sizeof(uds_msg));
	memset(record, 0, sizeof(uint8_t) * 20);

	CLEAR_CONSOLE;
	printf("\n\n*** UDS Service: RequestTransferExit ***\n");

	// Sends a physical RequestTransferExit message
	printf("\n\nSends a physical RequestTransferExit message: \n");
	for (int i = 0; i < record_size; i++) {
		record[i] = 'A' + i;
	}
	status = UDS_SvcRequestTransferExit_2013(channel, config, &request, record, record_size);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcRequestTransferExit_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();
}

/// <summary>Call UDS Service TransferData</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testTransferData(cantp_handle channel, uds_msgconfig config) {
	uds_status status;
	uds_msg request;
	uds_msg response;
	uds_msg confirmation;
	uint8_t record[50];
	uint8_t record_size = 50;

	memset(&request, 0, sizeof(uds_msg));
	memset(&response, 0, sizeof(uds_msg));
	memset(&confirmation, 0, sizeof(uds_msg));
	memset(record, 0, sizeof(uint8_t) * 50);

	CLEAR_CONSOLE;
	printf("\n\n*** UDS Service: TransferData ***\n");

	// Sends a physical TransferData message
	printf("\n\nSends a physical TransferData message: \n");
	for (int i = 0; i < record_size; i++) {
		record[i] = 'A' + i;
	}
	status = UDS_SvcTransferData_2013(channel, config, &request, 0x01, record, record_size);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcTransferData_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();
}

/// <summary>Call UDS Service TransferData with MAX_DATA length</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testTransferDataBigMessage(cantp_handle channel, uds_msgconfig config) {
	uds_status status;
	uds_msg request;
	uds_msg response;
	uds_msg confirmation;
	uint8_t record[4095];
	uint16_t record_size = 4093;

	memset(&request, 0, sizeof(uds_msg));
	memset(&response, 0, sizeof(uds_msg));
	memset(&confirmation, 0, sizeof(uds_msg));
	memset(record, 0, sizeof(uint8_t) * 4095);

	CLEAR_CONSOLE;
	printf("\n\n*** UDS Service: TransferData with MAX_DATA ***\n");

	// Sends a physical TransferData message with the maximum data available. The goal is to show that
	// UDS_WaitForService_2013 does not return a TIMEOUT error although the transmit and receive time of all the
	// data will be longer than the default time to get a response.
	printf("\n\nSends a physical TransferData message (length=%d): \n", record_size);
	for (int i = 0; i < record_size; i++)
		record[i] = 'A' + i;
	status = UDS_SvcTransferData_2013(channel, config, &request, 0x01, record, record_size);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcTransferData_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();
}

/// <summary>Call UDS Service TransferData</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testTransferDataMultipleFunctionalMessage(cantp_handle channel, uds_msgconfig config) {
	uds_status status;
	uds_msg request;
	uds_msg response;
	uds_msg confirmation;
	uint8_t record[5];
	uint16_t record_size = 5;
	uint32_t response_count = 0;

	memset(&request, 0, sizeof(uds_msg));
	memset(&response, 0, sizeof(uds_msg));
	memset(&confirmation, 0, sizeof(uds_msg));
	memset(record, 0, sizeof(uint8_t) * 5);

	CLEAR_CONSOLE;
	printf("\n\n*** UDS Service: TransferData with functional message***\n");

	// Initialize request message
	config.nai.target_addr = PUDS_ISO_15765_4_ADDR_OBD_FUNCTIONAL;
	config.nai.target_type = PCANTP_ISOTP_ADDRESSING_FUNCTIONAL;

	// Sends a functional TransferData message. The goal is to show that UDS_WaitForServiceFunctional_2013 waits long
	// enough to fetch all possible ECU responses.
	printf("\n\nSends a functional TransferData message: \n");
	for (int i = 0; i < record_size; i++) {
		record[i] = 'A' + i;
	}
	status = UDS_SvcTransferData_2013(channel, config, &request, 0x01, record, record_size);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForServiceFunctional_2013(channel, &request, 1, true, &response, &response_count, &confirmation);
	printf(" UDS_SvcTransferData_2013: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
		display_uds_msg(&confirmation, NULL, true);
		printf("\n Received %u UDS responses:\n", response_count);
		display_uds_msg(NULL, &response, false);
	}
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();
}

/// <summary>Sample to use event</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testUsingEvent(cantp_handle channel, uds_msgconfig config) {
	uds_status status;
	uds_status read_status;
	uds_msg request;
	uds_msg response;
	HANDLE receive_event;
	HANDLE tmp_event_handle;
	DWORD wait_result;
	bool stop;

	memset(&request, 0, sizeof(uds_msg));
	memset(&response, 0, sizeof(uds_msg));

	// set event handler
	receive_event = CreateEvent(NULL, false, false, NULL);
	status = UDS_SetValue_2013(channel, PUDS_PARAMETER_RECEIVE_EVENT, &receive_event, sizeof(receive_event));
	if (!UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
		printf("Failed to set event, aborting...");
		CloseHandle(receive_event);
		waitGetch();
		return;
	}

	CLEAR_CONSOLE;
	printf("\n\n*** UDS Service with Event: TesterPresent ***\n");

	// Sends a physical TesterPresent message
	printf("\n\nSends a physical TesterPresent message: \n");
	status = UDS_SvcTesterPresent_2013(channel, config, &request, PUDS_SVC_PARAM_TP_ZSUBF);
	printf(" UDS_SvcTesterPresent_2013: %i\n", (int)status);

	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {

		// Instead of calling WaitForService function, this sample demonstrates how event can be used. But note that
		// the use of a thread listening to notifications and making the read operations is preferred.
		stop = false;

		// wait until we receive expected response
		do {
			wait_result = WaitForSingleObject(receive_event, INFINITE);
			if (wait_result == WAIT_OBJECT_0) {

				// read all messages
				do {
					read_status = UDS_Read_2013(channel, &response, NULL, NULL);
					if (UDS_StatusIsOk_2013(read_status, PUDS_STATUS_OK, false))
					{
						// this is a simple message check (type and sender/receiver address): to filter UDS request
						// confirmation and get first message from target, but real use-case should check that the UDS
						// service ID matches the request
						if (response.msg.msgdata.isotp->netaddrinfo.msgtype == PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC
							&& response.msg.msgdata.isotp->netaddrinfo.source_addr == config.nai.target_addr
							&& response.msg.msgdata.isotp->netaddrinfo.target_addr == config.nai.source_addr) {
							stop = true;
							display_uds_msg(&request, &response, false);
						}
					}
					// Free response message
					status = UDS_MsgFree_2013(&response);
					printf(" Free response message: %i\n", status);
				} while (!UDS_StatusIsOk_2013(read_status, PUDS_STATUS_NO_MESSAGE, false));
			}
		} while (!stop);
	}
	waitGetch();

	// Free request message
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);

	// Uninitialize event
	tmp_event_handle = 0;
	status = UDS_SetValue_2013(channel, PUDS_PARAMETER_RECEIVE_EVENT, &tmp_event_handle, sizeof(tmp_event_handle));
	printf(" Remove receive event: %i\n", status);
	CloseHandle(receive_event);
}

/// <summary>Call UDS Service RequestFileTransfer</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testRequestFileTransfer(cantp_handle channel, uds_msgconfig config)
{
	uds_status status;
	uds_msg request;
	uds_msg response;
	uds_msg confirmation;
	uint8_t file_size_uncompressed[2] = { 0xD, 0x00 };
	uint8_t file_size_compressed[2] = { 0xA, 0x00 };
	uint8_t file_size_length = 2;
	char *sfile_name = "toto.txt";
	uint16_t file_name_length = 8;

	memset(&request, 0, sizeof(request));
	memset(&response, 0, sizeof(response));
	memset(&confirmation, 0, sizeof(uds_msg));

	CLEAR_CONSOLE;
	printf("*** UDS Service: RequestFileTransfer ***\n");

	// Sends a physical RequestFileTransfer message
	printf("\n\nSends a physical RequestFileTransfer message: \n");
	status = UDS_SvcRequestFileTransfer_2013(channel, config, &request, PUDS_SVC_PARAM_RFT_MOOP_REPLFILE, file_name_length,
		(uint8_t*)sfile_name, 0, 0, file_size_length, file_size_uncompressed, file_size_compressed);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcRequestFileTransfer_2013: %d\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %d\n", (int)status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %d\n", (int)status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();
}

/// <summary>Call UDS Service AccessTimingParameter</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testAccessTimingParameter(cantp_handle channel, uds_msgconfig config)
{
	uds_status status;
	uds_msg request;
	uds_msg response;
	uds_msg confirmation;
	uint8_t request_record[2] = { 0xAB, 0xCD };
	uint32_t record_size = 2;

	memset(&request, 0, sizeof(request));
	memset(&response, 0, sizeof(response));
	memset(&confirmation, 0, sizeof(uds_msg));

	CLEAR_CONSOLE;
	printf("\n\n\n*** UDS Service: AccessTimingParameter ***\n");

	// Sends a physical AccessTimingParameter message
	printf("\n\nSends a physical AccessTimingParameter message: \n");
	status = UDS_SvcAccessTimingParameter_2013(channel, config, &request, PUDS_SVC_PARAM_ATP_RCATP, request_record, record_size);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcAccessTimingParameter_2013: %d\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %d\n", (int)status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %d\n", (int)status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();
}

/// <summary>Call UDS Service Authentication</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
void testAuthentication(cantp_handle channel, uds_msgconfig config)
{
	uds_status status;
	uds_msg request;
	uds_msg response;
	uds_msg confirmation;

	uint8_t communication_configuration = 0;
	uint8_t certificate_client[2] = { 0x12,0x34 };
	uint16_t certificate_client_size = 2;
	uint8_t challenge_client[2] = { 0x56,0x78 };
	uint16_t challenge_client_size = 2;
	uint8_t proof_of_ownership_client[2] = { 0x9A,0xBC };
	uint16_t proof_of_ownership_client_size = 2;
	uint8_t ephemeral_public_key_client[2] = { 0xDE,0xF0 };
	uint16_t ephemeral_public_key_client_size = 2;
	uint8_t algorithm_indicator[16] = { 0x00, 0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,0x0C,0x0D,0x0E,0x0F };
	uint8_t additional_parameter[2] = { 0xAA,0xBB };
	uint16_t additional_parameter_size = 2;

	memset(&request, 0, sizeof(uds_msg));
	memset(&response, 0, sizeof(uds_msg));
	memset(&confirmation, 0, sizeof(uds_msg));

	CLEAR_CONSOLE;
	printf("\n\n*** UDS Service: Authentication ***\n");

	// Sends a physical Authentication/deAuthenticate message
	printf("\n\nSends a physical Authentication/deAuthenticate message: \n");
	status = UDS_SvcAuthenticationDA_2020(channel, config, &request);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcAuthenticationDA_2020: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();


	// Sends a physical Authentication/verifyCertificateUnidirectional message
	printf("\n\nSends a physical Authentication/verifyCertificateUnidirectional message: \n");
	status = UDS_SvcAuthenticationVCU_2020(channel, config, &request, communication_configuration, certificate_client, certificate_client_size, challenge_client, challenge_client_size);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcAuthenticationVCU_2020: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();


	// Sends a physical Authentication/verifyCertificateBidirectional message
	printf("\n\nSends a physical Authentication/verifyCertificateBidirectional message: \n");
	status = UDS_SvcAuthenticationVCB_2020(channel, config, &request, communication_configuration, certificate_client, certificate_client_size, challenge_client, challenge_client_size);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcAuthenticationVCB_2020: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();


	// Sends a physical Authentication/proofOfOwnership message
	printf("\n\nSends a physical Authentication/proofOfOwnership message: \n");
	status = UDS_SvcAuthenticationPOWN_2020(channel, config, &request, proof_of_ownership_client, proof_of_ownership_client_size, ephemeral_public_key_client, ephemeral_public_key_client_size);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcAuthenticationPOWN_2020: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();


	// Sends a physical Authentication/requestChallengeForAuthentication message
	printf("\n\nSends a physical Authentication/requestChallengeForAuthentication message: \n");
	status = UDS_SvcAuthenticationRCFA_2020(channel, config, &request, communication_configuration, algorithm_indicator);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcAuthenticationRCFA_2020: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();


	// Sends a physical Authentication/verifyProofOfOwnershipUnidirectional message
	printf("\n\nSends a physical Authentication/verifyProofOfOwnershipUnidirectional message: \n");
	status = UDS_SvcAuthenticationVPOWNU_2020(channel, config, &request, algorithm_indicator, proof_of_ownership_client, proof_of_ownership_client_size, challenge_client, challenge_client_size, additional_parameter, additional_parameter_size);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcAuthenticationVPOWNU_2020: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();


	// Sends a physical Authentication/verifyProofOfOwnershipBidirectional message
	printf("\n\nSends a physical Authentication/verifyProofOfOwnershipBidirectional message: \n");
	status = UDS_SvcAuthenticationVPOWNB_2020(channel, config, &request, algorithm_indicator, proof_of_ownership_client, proof_of_ownership_client_size, challenge_client, challenge_client_size, additional_parameter, additional_parameter_size);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcAuthenticationVPOWNB_2020: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();


	// Sends a physical Authentication/authenticationConfiguration message
	printf("\n\nSends a physical Authentication/authenticationConfiguration message: \n");
	status = UDS_SvcAuthenticationAC_2020(channel, config, &request);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		status = UDS_WaitForService_2013(channel, &request, &response, &confirmation);
	printf(" UDS_SvcAuthenticationAC_2020: %i\n", (int)status);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false))
		display_uds_msg(&confirmation, &response, false);
	else
		display_uds_msg(&request, NULL, false);

	// Free messages
	status = UDS_MsgFree_2013(&request);
	printf(" Free request message: %i\n", status);
	status = UDS_MsgFree_2013(&response);
	printf(" Free response message: %i\n", status);
	status = UDS_MsgFree_2013(&confirmation);
	printf(" Free confirmation message: %i\n", status);
	waitGetch();
}

/// <summary>Entry point of the program, start a CAN UDS client</summary>
int main() {
	cantp_handle client_handle;
	uds_status status;
	uds_msgconfig config;
	uint8_t tmp_buffer;

	// Set the PCAN-Channel to use
	client_handle = PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.

	// Initializing of the UDS Communication session
	status = UDS_Initialize_2013(client_handle, PCANTP_BAUDRATE_500K, (cantp_hwtype)0, 0, 0);
	printf("Initialize UDS: %i\n", (int)status);

	// Define TimeOuts
	tmp_buffer = PCANTP_ISO_TIMEOUTS_15765_4;
	status = UDS_SetValue_2013(client_handle, PUDS_PARAMETER_ISO_TIMEOUTS, &tmp_buffer, sizeof(tmp_buffer));
	printf("Set ISO 15765-4 timeouts values: %d\n", (int)status);
	waitGetch();

	// Define Network Address Information used for all the tests
	memset(&config, 0, sizeof(uds_msgconfig));
	config.can_id = -1;
	config.can_msgtype = PCANTP_CAN_MSGTYPE_STANDARD;
	config.nai.protocol = PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL;
	config.nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL;
	config.type = PUDS_MSGTYPE_USDT;
	config.nai.source_addr = PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
	config.nai.target_addr = PUDS_ISO_15765_4_ADDR_ECU_1;

	// The following functions call UDS Services
	testDiagnosticSessionControl(client_handle, config);
	testECUReset(client_handle, config);
	testSecurityAccess(client_handle, config);
	testCommunicationControl(client_handle, config);
	testTesterPresent(client_handle, config);
	testSecuredDataTransmission(client_handle, config);
	testControlDTCSetting(client_handle, config);
	testResponseOnEvent(client_handle, config);
	testLinkControl(client_handle, config);
	testReadDataByIdentifier(client_handle, config);
	testReadMemoryByAddress(client_handle, config);
	testReadScalingDataByIdentifier(client_handle, config);
	testReadDataByPeriodicIdentifier(client_handle, config);
	testDynamicallyDefineDataIdentifier(client_handle, config);
	testWriteDataByIdentifier(client_handle, config);
	testWriteMemoryByAddress(client_handle, config);
	testClearDiagnosticInformation(client_handle, config);
	testReadDTCInformation(client_handle, config);
	testInputOutputControlByIdentifier(client_handle, config);
	testRoutineControl(client_handle, config);
	testRequestDownload(client_handle, config);
	testRequestUpload(client_handle, config);
	testTransferData(client_handle, config);
	testRequestTransferExit(client_handle, config);
	testAccessTimingParameter(client_handle, config);
	testRequestFileTransfer(client_handle, config);
	testAuthentication(client_handle, config);

	// Miscellaneous examples
	testTransferDataBigMessage(client_handle, config);
	testTransferDataMultipleFunctionalMessage(client_handle, config);
	testUsingEvent(client_handle, config);

	// Uninitialize channel
	status = UDS_Uninitialize_2013(client_handle);
	printf(" Unitialize channel: %i\n", status);

	// Display a small report and quit
	if (g_nbErr > 0)
		printf("\nERROR: %d errors occurred.\n\n", g_nbErr);
	else
		printf("\nALL Transmissions succeeded !\n\n");
	system("PAUSE");
	return EXIT_SUCCESS;
}

