#include "stdafx.h"

#include "lib/PCAN-UDS_2013.h"

#define OK_KO(test) (test)?"OK":"KO"
#define STATUS_OK_KO(test) OK_KO(UDS_StatusIsOk_2013(test, PUDS_STATUS_OK, false))

// Define Bitrate: SAE J2284-4: High-Speed CAN (HSC) for Vehicle Applications at 500 kbps with CAN FD Data at 2 Mbps
#define PCAN_BITRATE_SAE_J2284_4     "f_clock=80000000,nom_brp=2,nom_tseg1=63,nom_tseg2=16,nom_sjw=16,data_brp=2,data_tseg1=15,data_tseg2=4,data_sjw=4"
#define BUFFER_SIZE 256

#define MSG "PEAK"
#define MSG_SIZE 4
#define MAPPING_BUFFER_SIZE 256

/// <summary>Helper: Check if a mapping list contains a certain mapping</summary>
/// <param name="mapping_list">Mapping list</param>
/// <param name="mapping_list_size">Number of mapping in the mapping list</param>
/// <param name="searched_mapping">Searched mapping</param>
/// <returns>Present (true) or not (false)</returns>
bool mapping_list_contains(uds_mapping *mapping_list, uint16_t mapping_list_size, uds_mapping *searched_mapping) {
	bool res = false;
	for (int i = 0; i < mapping_list_size; i++) {

		// If unique identifier are the same, the mapping is in the list
		if (mapping_list[i].uid == searched_mapping->uid) {
			res = true;
			break;
		}
	}
	return res;
}

/// <summary>Entry point of the program, start a small CAN UDS read/write example</summary>
int main()
{
	char buffer[BUFFER_SIZE];
	uds_status status;
	cantp_handle transmitter_handle;
	cantp_handle receiver_handle;
	uds_msg tx_msg;
	uds_msg rx_msg;
	uds_msgconfig msg_config;
	uds_mapping source_mapping;
	uds_mapping response_mapping;
	uds_mapping mapping_buffer[MAPPING_BUFFER_SIZE];
	uint16_t mapping_count;
	HANDLE receive_event;
	DWORD wait_result;
	BOOL boolean_status;
	uint64_t null_handle;

	// Initialize variables
	transmitter_handle = PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.
	receiver_handle = PCANTP_HANDLE_USBBUS2; // TODO: modify the value according to your available PCAN devices.
	null_handle = 0;
	mapping_count = 0;
	memset(mapping_buffer, 0, sizeof(uds_mapping)*MAPPING_BUFFER_SIZE);
	memset(&response_mapping, 0, sizeof(uds_mapping));
	memset(&source_mapping, 0, sizeof(uds_mapping));
	memset(&msg_config, 0, sizeof(uds_msgconfig));
	memset(&tx_msg, 0, sizeof(uds_msg));
	memset(&rx_msg, 0, sizeof(uds_msg));
	buffer[0] = '\0';

	// Print version informations
	status = UDS_GetValue_2013(PCANTP_HANDLE_NONEBUS, PUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE);
	printf("PCAN-UDS API Version: %s (%s)\n", buffer, STATUS_OK_KO(status));

	// Initialize transmitter and receiver
	status = UDS_InitializeFD_2013(transmitter_handle, PCAN_BITRATE_SAE_J2284_4);
	printf("Initialize transmitter: %s\n", STATUS_OK_KO(status));
	status = UDS_InitializeFD_2013(receiver_handle, PCAN_BITRATE_SAE_J2284_4);
	printf("Initialize receiver: %s\n", STATUS_OK_KO(status));

	// Create a receive event on receiver
	receive_event = CreateEvent(NULL, FALSE, FALSE, NULL);
	status = UDS_SetValue_2013(receiver_handle, PUDS_PARAMETER_RECEIVE_EVENT, &receive_event, sizeof(receive_event));
	printf("Set receive event parameter: %s\n", STATUS_OK_KO(status));

	// Initialize 29bits
	msg_config.can_id = 0x30;
	msg_config.can_msgtype = PCANTP_CAN_MSGTYPE_STANDARD;
	// To use CAN FD with bitrate switch uncomment:
	// msg_config.can_msgtype |= (PCANTP_CAN_MSGTYPE_FD | PCANTP_CAN_MSGTYPE_BRS);
	msg_config.nai.protocol = PUDS_MSGPROTOCOL_ISO_15765_2_29B_NORMAL;
	msg_config.nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL;
	msg_config.type = PUDS_MSGTYPE_USDT;
	msg_config.nai.source_addr = 0x10;
	msg_config.nai.target_addr = 0x20;

	// Allocate TX message with previous configuration
	status = UDS_MsgAlloc_2013(&tx_msg, msg_config, MSG_SIZE);
	if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
		memcpy(tx_msg.msg.msgdata.any->data, MSG, MSG_SIZE);
	}
	printf("Allocate TX message: %s\n", STATUS_OK_KO(status));

	// Initialize source mapping
	source_mapping.can_id = msg_config.can_id;
	source_mapping.can_id_flow_ctrl = msg_config.can_id + 1;
	source_mapping.can_msgtype = msg_config.can_msgtype;
	source_mapping.can_tx_dlc = 8;
	source_mapping.nai = msg_config.nai;

	// Initialize response mapping
	response_mapping = source_mapping;
	response_mapping.can_id = source_mapping.can_id_flow_ctrl;
	response_mapping.can_id_flow_ctrl = source_mapping.can_id;
	response_mapping.nai.source_addr = source_mapping.nai.target_addr;
	response_mapping.nai.target_addr = source_mapping.nai.source_addr;

	// Add mappings on transmitter
	status = UDS_AddMapping_2013(transmitter_handle, &source_mapping);
	printf("Add source mapping on transmitter: %s\n", STATUS_OK_KO(status));
	status = UDS_AddMapping_2013(transmitter_handle, &response_mapping);
	printf("Add response mapping on transmitter: %s\n", STATUS_OK_KO(status));

	// Check if mappings are added on transmitter
	status = UDS_GetMappings_2013(transmitter_handle, mapping_buffer, MAPPING_BUFFER_SIZE, &mapping_count);
	printf("Get mappings on transmitter: %s\n", STATUS_OK_KO(status));
	printf("Check added mappings on transmitter: %s\n", OK_KO(mapping_list_contains(mapping_buffer, mapping_count, &source_mapping) && mapping_list_contains(mapping_buffer, mapping_count, &response_mapping)));

	// Add mappings on receiver
	status = UDS_AddMapping_2013(receiver_handle, &source_mapping);
	printf("Add source mapping on receiver: %s\n", STATUS_OK_KO(status));
	status = UDS_AddMapping_2013(receiver_handle, &response_mapping);
	printf("Add response mapping on receiver: %s\n", STATUS_OK_KO(status));

	// Check if mappings are added on receiver
	status = UDS_GetMappings_2013(receiver_handle, mapping_buffer, MAPPING_BUFFER_SIZE, &mapping_count);
	printf("Get mappings on receiver: %s\n", STATUS_OK_KO(status));
	printf("Check added mappings on receiver: %s\n", OK_KO(mapping_list_contains(mapping_buffer, mapping_count, &source_mapping) && mapping_list_contains(mapping_buffer, mapping_count, &response_mapping)));

	// Write message
	status = UDS_Write_2013(transmitter_handle, &tx_msg);
	printf("Write \"PEAK\": %s\n", STATUS_OK_KO(status));

	// Wait a receive event
	wait_result = WaitForSingleObject(receive_event, 3000);
	printf("Wait a receive event: %s\n", OK_KO(wait_result == WAIT_OBJECT_0));

	// If we get a receive event
	if (wait_result == WAIT_OBJECT_0) {

		// Read message
		status = UDS_Read_2013(receiver_handle, &rx_msg, NULL, NULL);
		printf("Receive message: %s\n", STATUS_OK_KO(status));

		// Check message (should contains "PEAK")
		if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false) && rx_msg.msg.msgdata.any->length == MSG_SIZE && memcmp(tx_msg.msg.msgdata.any->data, rx_msg.msg.msgdata.any->data, MSG_SIZE) == 0) {
			printf("Message contains \"%c%c%c%c\": OK\n", rx_msg.msg.msgdata.any->data[0], rx_msg.msg.msgdata.any->data[1], rx_msg.msg.msgdata.any->data[2], rx_msg.msg.msgdata.any->data[3]);
		}
		else {
			printf("Message is corrupted: KO\n");
		}
	}

	// Close receive event
	status = UDS_SetValue_2013(receiver_handle, PUDS_PARAMETER_RECEIVE_EVENT, &null_handle, sizeof(HANDLE));
	printf("Stop receive event receiver: %s\n", STATUS_OK_KO(status));
	boolean_status = CloseHandle(receive_event);
	printf("Close receive event: %s\n", OK_KO(boolean_status));

	// Free messages
	status = UDS_MsgFree_2013(&tx_msg);
	printf("Free TX message: %s\n", STATUS_OK_KO(status));
	status = UDS_MsgFree_2013(&rx_msg);
	printf("Free RX message: %s\n", STATUS_OK_KO(status));

	// Close transmitter and receiver
	status = UDS_Uninitialize_2013(transmitter_handle);
	printf("Uninitialize transmitter: %s\n", STATUS_OK_KO(status));
	status = UDS_Uninitialize_2013(receiver_handle);
	printf("Uninitialize receiver: %s\n", STATUS_OK_KO(status));

	// Exit
	system("PAUSE");
	return EXIT_SUCCESS;
}

