#include "stdafx.h"

#include "lib/PCAN-UDS_2013.h"

#define OK_KO(test) (test)?"OK":"KO"
#define STATUS_OK_KO(test) OK_KO(UDS_StatusIsOk_2013((uds_status)test, PUDS_STATUS_OK, false))

// Define Bitrate: SAE J2284-4: High-Speed CAN (HSC) for Vehicle Applications at 500 kbps with CAN FD Data at 2 Mbps
#define PCAN_BITRATE_SAE_J2284_4     "f_clock=80000000,nom_brp=2,nom_tseg1=63,nom_tseg2=16,nom_sjw=16,data_brp=2,data_tseg1=15,data_tseg2=4,data_sjw=4"
#define BUFFER_SIZE 256

#define MSG "PEAK"
#define MSG_SIZE 4

/// <summary>
/// Entry point of the program, start a small CAN UDS / ISOTP read-write example
/// This example initialise two UDS channel with a specific mapping.
/// Then it inits an ISOTP mapping and write an ISOTP message.
/// In this case, PCAN-UDS should not receive the message, we need PCAN-ISOTP to read it.
/// </summary>
/// <returns>By convention, return success.</returns>
int main()
{
	char buffer[BUFFER_SIZE];
	int status;
	cantp_handle transmitter_handle;
	cantp_handle receiver_handle;
	cantp_msg tx_msg;
	uds_msg uds_rx_msg;
	cantp_msg rx_msg;
	uds_mapping source_mapping;
	cantp_mapping mapping;
	cantp_mapping reverse_mapping;
	uds_mapping response_mapping;
	HANDLE receive_event;
	DWORD wait_result;
	BOOL boolean_status;
	uint64_t null_handle;
	cantp_can_msgtype can_msgtype;

	// Initialize variables
	transmitter_handle = PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.
	receiver_handle = PCANTP_HANDLE_USBBUS2; // TODO: modify the value according to your available PCAN devices.
	null_handle = 0;
	memset(&response_mapping, 0, sizeof(uds_mapping));
	memset(&source_mapping, 0, sizeof(uds_mapping));
	memset(&tx_msg, 0, sizeof(cantp_msg));
	memset(&uds_rx_msg, 0, sizeof(uds_msg));
	memset(&mapping, 0, sizeof(cantp_mapping));
	memset(&reverse_mapping, 0, sizeof(cantp_mapping));
	memset(&rx_msg, 0, sizeof(cantp_msg));
	buffer[0] = '\0';
	can_msgtype = PCANTP_CAN_MSGTYPE_STANDARD;
	// To use CAN FD with bitrate switch uncomment:
	// can_msgtype |= (PCANTP_CAN_MSGTYPE_FD | PCANTP_CAN_MSGTYPE_BRS);

	// Print version informations
	status = UDS_GetValue_2013(PCANTP_HANDLE_NONEBUS, PUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE);
	printf("PCAN-UDS API Version: %s (%s)\n", buffer, STATUS_OK_KO(status));

	// Initialize transmitter and receiver
	status = UDS_InitializeFD_2013(transmitter_handle, PCAN_BITRATE_SAE_J2284_4);
	printf("Initialize transmitter: %s\n", STATUS_OK_KO(status));
	status = UDS_InitializeFD_2013(receiver_handle, PCAN_BITRATE_SAE_J2284_4);
	printf("Initialize receiver: %s\n", STATUS_OK_KO(status));

	// Create a receive event on receiver
	receive_event = CreateEvent(NULL, FALSE, FALSE, NULL);
	status = UDS_SetValue_2013(receiver_handle, PUDS_PARAMETER_RECEIVE_EVENT, &receive_event, sizeof(receive_event));
	printf("Set receive event parameter: %s\n", STATUS_OK_KO(status));

	// Initialize source mapping
	source_mapping.can_id = 0x30;
	source_mapping.can_id_flow_ctrl = source_mapping.can_id + 1;
	source_mapping.can_msgtype = can_msgtype;
	source_mapping.can_tx_dlc = 8;
	source_mapping.nai.protocol = PUDS_MSGPROTOCOL_ISO_15765_2_29B_NORMAL;
	source_mapping.nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL;
	source_mapping.nai.source_addr = 0x10;
	source_mapping.nai.target_addr = 0x20;

	// Initialize response mapping
	response_mapping = source_mapping;
	response_mapping.can_id = source_mapping.can_id_flow_ctrl;
	response_mapping.can_id_flow_ctrl = source_mapping.can_id;
	response_mapping.nai.source_addr = source_mapping.nai.target_addr;
	response_mapping.nai.target_addr = source_mapping.nai.source_addr;

	// Add UDS mappings on transmitter
	status = UDS_AddMapping_2013(transmitter_handle, &source_mapping);
	printf("Add source mapping on transmitter: %s\n", STATUS_OK_KO(status));
	status = UDS_AddMapping_2013(transmitter_handle, &response_mapping);
	printf("Add response mapping on transmitter: %s\n", STATUS_OK_KO(status));

	// Add UDS mappings on and receiver
	status = UDS_AddMapping_2013(receiver_handle, &source_mapping);
	printf("Add source mapping on receiver: %s\n", STATUS_OK_KO(status));
	status = UDS_AddMapping_2013(receiver_handle, &response_mapping);
	printf("Add response mapping on receiver: %s\n", STATUS_OK_KO(status));

	// Create a simple isotp physical mapping:
	//    - Source 0xF1 (transmitter), target 0x01 (receiver), CAN id 0xA1, CAN ID flow control 0xA2
	//    - Diagnostic message in a classic format
	mapping.can_id = 0xA1;
	mapping.can_id_flow_ctrl = 0xA2;
	mapping.can_msgtype = can_msgtype;
	mapping.can_tx_dlc = 0x0;
	mapping.netaddrinfo.extension_addr = 0x00;
	mapping.netaddrinfo.format = PCANTP_ISOTP_FORMAT_NORMAL;
	mapping.netaddrinfo.msgtype = PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC;
	mapping.netaddrinfo.source_addr = 0xF1;
	mapping.netaddrinfo.target_addr = 0x01;
	mapping.netaddrinfo.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL;

	// Create the associated isotp reversed mapping:
	reverse_mapping = mapping;
	reverse_mapping.can_id = mapping.can_id_flow_ctrl;
	reverse_mapping.can_id_flow_ctrl = mapping.can_id;
	reverse_mapping.netaddrinfo.source_addr = mapping.netaddrinfo.target_addr;
	reverse_mapping.netaddrinfo.target_addr = mapping.netaddrinfo.source_addr;

	// Add ISOTP mappings on channels
	status = CANTP_AddMapping_2016(transmitter_handle, &mapping);
	printf("Add a simple isotp mapping on transmitter: %s\n", STATUS_OK_KO(status));
	status = CANTP_AddMapping_2016(transmitter_handle, &reverse_mapping);
	printf("Add the reverse isotp mapping on transmitter: %s\n", STATUS_OK_KO(status));
	status = CANTP_AddMapping_2016(receiver_handle, &mapping);
	printf("Add a simple isotp mapping on receiver: %s\n", STATUS_OK_KO(status));
	status = CANTP_AddMapping_2016(receiver_handle, &reverse_mapping);
	printf("Add the reverse isotp mapping on receiver: %s\n", STATUS_OK_KO(status));

	// Initialize ISOTP Tx message containing "PEAK"
	status = CANTP_MsgDataAlloc_2016(&tx_msg, PCANTP_MSGTYPE_ISOTP);
	printf("Allocate tx ISOTP message: %s\n", STATUS_OK_KO(status));
	status = CANTP_MsgDataInit_2016(&tx_msg, mapping.can_id, mapping.can_msgtype, MSG_SIZE, MSG, &mapping.netaddrinfo);
	printf("Initialize tx message: %s\n", STATUS_OK_KO(status));

	// Send "PEAK" ISOTP message
	status = CANTP_Write_2016(transmitter_handle, &tx_msg);
	printf("Write \"PEAK\": %s\n", STATUS_OK_KO(status));

	// Wait a receive event on uds message (should not work because we sent a ISOTP message)
	wait_result = WaitForSingleObject(receive_event, 3000);
	printf("Wait a receive event on UDS message (should not work because we sent an ISOTP message): %s\n", OK_KO(wait_result == WAIT_OBJECT_0));

	// We should receive a ISOTP message instead
	status = CANTP_Read_2016(receiver_handle, &rx_msg, NULL, PCANTP_MSGTYPE_NONE);
	printf("Read ISOTP message on receiver: %s\n", STATUS_OK_KO(status));

	// Check message (should contains "PEAK")
	if (UDS_StatusIsOk_2013((uds_status)status, PUDS_STATUS_OK, false) && rx_msg.msgdata.any->length == MSG_SIZE && memcmp(tx_msg.msgdata.any->data, rx_msg.msgdata.any->data, MSG_SIZE) == 0) {
		printf("Message contains \"%c%c%c%c\": OK\n", rx_msg.msgdata.any->data[0], rx_msg.msgdata.any->data[1], rx_msg.msgdata.any->data[2], rx_msg.msgdata.any->data[3]);
	}
	else {
		printf("Message is corrupted: KO\n");
	}

	// Close receive event
	status = UDS_SetValue_2013(receiver_handle, PUDS_PARAMETER_RECEIVE_EVENT, &null_handle, sizeof(HANDLE));
	printf("Stop receive event receiver: %s\n", STATUS_OK_KO(status));
	boolean_status = CloseHandle(receive_event);
	printf("Close receive event: %s\n", OK_KO(boolean_status));

	// Free messages
	status = CANTP_MsgDataFree_2016(&tx_msg);
	printf("Free TX message: %s\n", STATUS_OK_KO(status));
	status = CANTP_MsgDataFree_2016(&rx_msg);
	printf("Free RX message: %s\n", STATUS_OK_KO(status));
	status = UDS_MsgFree_2013(&uds_rx_msg);
	printf("Free UDS/RX message: %s\n", STATUS_OK_KO(status));

	// Close transmitter and receiver
	status = UDS_Uninitialize_2013(transmitter_handle);
	printf("Uninitialize transmitter: %s\n", STATUS_OK_KO(status));
	status = UDS_Uninitialize_2013(receiver_handle);
	printf("Uninitialize receiver: %s\n", STATUS_OK_KO(status));

	// Exit
	system("PAUSE");
	return EXIT_SUCCESS;
}

