#include "stdafx.h"

#include "lib/PCAN-UDS_2013.h"

#define OK_KO(test) (test)?"OK":"KO"
#define UDS_STATUS_OK_KO(test) OK_KO(UDS_StatusIsOk_2013(test, PUDS_STATUS_OK, false))
#define ISOTP_STATUS_OK_KO(test) OK_KO(CANTP_StatusIsOk_2016(test, PCANTP_STATUS_OK, false))

#define BUFFER_SIZE 256
#define ISOTP_REQUEST_MSG "PING"
#define ISOTP_RESPONSE_MSG "PONG"
#define MSG_SIZE 4

/// <summary>UDS client task: request TesterPresent service several time</summary>
/// <param name="param">client channel handle</param>
DWORD WINAPI uds_client_task(LPVOID param) {
	uds_status status;
	int i;
	uds_msgconfig config_physical;
	uds_msg msg_request;
	uds_msg request_confirmation;
	uds_msg response;
	cantp_handle client_handle;

	// Initialize variables
	memset(&config_physical, 0, sizeof(config_physical));
	memset(&msg_request, 0, sizeof(msg_request));
	memset(&request_confirmation, 0, sizeof(request_confirmation));
	memset(&response, 0, sizeof(response));
	client_handle = *(cantp_handle*)param;

	// Initialize a physical configuration
	config_physical.can_id = PUDS_ISO_15765_4_CAN_ID_PHYSICAL_REQUEST_1;
	config_physical.can_msgtype = PCANTP_CAN_MSGTYPE_STANDARD;
	config_physical.nai.protocol = PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL;
	config_physical.nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL;
	config_physical.type = PUDS_MSGTYPE_USDT;
	config_physical.nai.source_addr = PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
	config_physical.nai.target_addr = PUDS_ISO_15765_4_ADDR_ECU_1;
	config_physical.nai.extension_addr = 0;

	// Execute TesterPresent and wait response
	for (i = 1; i <= 10; i++) {
		status = UDS_SvcTesterPresent_2013(client_handle, config_physical, &msg_request, PUDS_SVC_PARAM_TP_ZSUBF);
		printf("[UDS] Execute TesterPresent service (%d): %s\n", i, UDS_STATUS_OK_KO(status));
		status = UDS_WaitForService_2013(client_handle, &msg_request, &response, &request_confirmation);
		printf("[UDS] Wait for service (%d): %s\n", i, UDS_STATUS_OK_KO(status));
		status = UDS_MsgFree_2013(&msg_request);
		printf("[UDS] Free request message: %s\n", UDS_STATUS_OK_KO(status));
		status = UDS_MsgFree_2013(&request_confirmation);
		printf("[UDS] Free request confirmation message: %s\n", UDS_STATUS_OK_KO(status));
		status = UDS_MsgFree_2013(&response);
		printf("[UDS] Free response message: %s\n", UDS_STATUS_OK_KO(status));
	}
	return 0;
}

/// <summary>ISOTP: wait an ISOTP "PONG" message</summary>
/// <param name="client_handle">client channel handle</param>
/// <param name="repeat_wait_counter">maximum number of WaitForSingleObject to do (to avoid infinite loop)</param>
/// <returns>if we received a PONG message (true) or not (false)</returns>
bool wait_isotp_pong_response(cantp_handle client_handle, int repeat_wait_counter) {
	cantp_status status;
	HANDLE receive_event;
	DWORD wait_result;
	cantp_msg rx_msg;
	int repeat;
	bool get_pong_msg;

	// Init variables
	memset(&rx_msg, 0, sizeof(cantp_msg));
	repeat = 0;
	get_pong_msg = false;

	// Get ISOTP receive event
	status = CANTP_GetValue_2016(client_handle, PCANTP_PARAMETER_RECEIVE_EVENT, &receive_event, sizeof(receive_event));
	printf("[ISOTP] Get isotp receive event parameter: %s\n", ISOTP_STATUS_OK_KO(status));

	// Wait a receive event on isotp message
	do {
		wait_result = WaitForSingleObject(receive_event, 1000);
		printf("[ISOTP] Wait a receive event from isotp: %s\n", OK_KO(wait_result == WAIT_OBJECT_0));

		// Read ISOTP messages
		do {
			status = CANTP_Read_2016(client_handle, &rx_msg, NULL, PCANTP_MSGTYPE_NONE);

			// Check if we received a "PONG" message
			if (!CANTP_StatusIsOk_2016(status, PCANTP_STATUS_NO_MESSAGE, false)) {
				printf("[ISOTP] Read ISOTP message: %s\n", ISOTP_STATUS_OK_KO(status));
				if (CANTP_StatusIsOk_2016(status, PCANTP_STATUS_OK, false) && rx_msg.msgdata.any->length == MSG_SIZE && memcmp(rx_msg.msgdata.any->data, ISOTP_RESPONSE_MSG, MSG_SIZE) == 0) {
					printf("[ISOTP] Message contains \"%c%c%c%c\": OK\n", rx_msg.msgdata.any->data[0], rx_msg.msgdata.any->data[1], rx_msg.msgdata.any->data[2], rx_msg.msgdata.any->data[3]);
					get_pong_msg = true;
				}

				status = CANTP_MsgDataFree_2016(&rx_msg);
				printf("[ISOTP] Free RX message: %s\n", ISOTP_STATUS_OK_KO(status));
			}
		} while (!CANTP_StatusIsOk_2016(status, PCANTP_STATUS_NO_MESSAGE, false));
		repeat++;
	} while (get_pong_msg == false && repeat < repeat_wait_counter);
	return get_pong_msg;
}

/// <summary>ISOTP client task: send "PING" and wait "PONG" message several times</summary>
/// <param name="param">client channel handle</param>
DWORD WINAPI isotp_client_task(LPVOID param) {
	cantp_status status;
	cantp_msg tx_msg;
	cantp_mapping mapping;
	cantp_mapping reverse_mapping;
	HANDLE receive_event;
	BOOL boolean_status;
	uint64_t null_handle;
	cantp_handle client_handle;
	bool response;
	int i;

	// Initialize variables
	memset(&tx_msg, 0, sizeof(cantp_msg));
	memset(&mapping, 0, sizeof(cantp_mapping));
	memset(&reverse_mapping, 0, sizeof(cantp_mapping));
	client_handle = *(cantp_handle*)param;
	null_handle = 0;

	// Create a isotp receive event
	receive_event = CreateEvent(NULL, FALSE, FALSE, NULL);
	status = CANTP_SetValue_2016(client_handle, PCANTP_PARAMETER_RECEIVE_EVENT, &receive_event, sizeof(receive_event));
	printf("[ISOTP] Set isotp receive event parameter: %s\n", ISOTP_STATUS_OK_KO(status));

	// Create a simple isotp physical mapping:
	//    - Source 0xA1 (client), target 0xA2 (server), CAN id 0xA1, CAN ID flow control 0xA2
	//    - Diagnostic message in a classic format
	mapping.can_id = 0xA1;
	mapping.can_id_flow_ctrl = 0xA2;
	mapping.can_msgtype = PCANTP_CAN_MSGTYPE_STANDARD;
	mapping.netaddrinfo.extension_addr = 0x00;
	mapping.netaddrinfo.format = PCANTP_ISOTP_FORMAT_NORMAL;
	mapping.netaddrinfo.msgtype = PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC;
	mapping.netaddrinfo.source_addr = 0xA1;
	mapping.netaddrinfo.target_addr = 0xA2;
	mapping.netaddrinfo.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL;

	// Create the associated isotp reversed mapping:
	reverse_mapping = mapping;
	reverse_mapping.can_id = mapping.can_id_flow_ctrl;
	reverse_mapping.can_id_flow_ctrl = mapping.can_id;
	reverse_mapping.netaddrinfo.source_addr = mapping.netaddrinfo.target_addr;
	reverse_mapping.netaddrinfo.target_addr = mapping.netaddrinfo.source_addr;

	// Add ISOTP mappings on channels
	status = CANTP_AddMapping_2016(client_handle, &mapping);
	printf("[ISOTP] Add a simple isotp mapping: %s\n", ISOTP_STATUS_OK_KO(status));
	status = CANTP_AddMapping_2016(client_handle, &reverse_mapping);
	printf("[ISOTP] Add the reverse isotp mapping: %s\n", ISOTP_STATUS_OK_KO(status));

	// Initialize ISOTP Tx message containing "PING"
	status = CANTP_MsgDataAlloc_2016(&tx_msg, PCANTP_MSGTYPE_ISOTP);
	printf("[ISOTP] Allocate ISOTP tx message: %s\n", ISOTP_STATUS_OK_KO(status));
	status = CANTP_MsgDataInit_2016(&tx_msg, mapping.can_id, mapping.can_msgtype, MSG_SIZE, ISOTP_REQUEST_MSG, &mapping.netaddrinfo);
	printf("[ISOTP] Initialize ISOTP tx message: %s\n", ISOTP_STATUS_OK_KO(status));

	// Send "PING" ISOTP message and wait "PONG" response
	for (i = 1; i <= 10; i++) {
		status = CANTP_Write_2016(client_handle, &tx_msg);
		printf("[ISOTP] Send ISOTP \"PING\" message (%d): %s\n", i, ISOTP_STATUS_OK_KO(status));
		response = wait_isotp_pong_response(client_handle, 3);
		printf("[ISOTP] Get ISOTP \"PONG\" response (%d): %s\n", i, OK_KO(response));
	}

	// Close receive event
	status = CANTP_SetValue_2016(client_handle, PCANTP_PARAMETER_RECEIVE_EVENT, &null_handle, sizeof(HANDLE));
	printf("[ISOTP] Stop ISOTP receive event: %s\n", ISOTP_STATUS_OK_KO(status));
	boolean_status = CloseHandle(receive_event);
	printf("[ISOTP] Close ISOTP receive event: %s\n", OK_KO((int)boolean_status != 0));

	// Free messages
	status = CANTP_MsgDataFree_2016(&tx_msg);
	printf("[ISOTP] Free ISOTP TX message: %s\n", ISOTP_STATUS_OK_KO(status));

	return 0;
}

/// <summary>Entry point of the program, start a UDS channel, ask in the same time UDS TesterPresent and isotp request</summary>
/// <returns>By convention, return success.</returns>
int main()
{
	char buffer[BUFFER_SIZE];
	uds_status status;
	cantp_handle client_handle;
	HANDLE uds_client;
	HANDLE isotp_client;
	DWORD uds_client_id;
	DWORD isotp_client_id;

	// Initialize variables
	client_handle = PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.
	buffer[0] = '\0';

	// Print version informations
	status = UDS_GetValue_2013(PCANTP_HANDLE_NONEBUS, PUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE);
	printf("PCAN-UDS API Version - %s: %s\n", buffer, UDS_STATUS_OK_KO(status));

	// Initialize client channel
	status = UDS_Initialize_2013(client_handle, PCANTP_BAUDRATE_500K, (cantp_hwtype)0, 0, 0);
	printf("Initialize channel: %s\n", UDS_STATUS_OK_KO(status));

	// Start uds and isotp clients
	uds_client = CreateThread(NULL, 0, uds_client_task, &client_handle, 0, &uds_client_id);
	if (uds_client != NULL) {
		isotp_client = CreateThread(NULL, 0, isotp_client_task, &client_handle, 0, &isotp_client_id);
		if (isotp_client != NULL) {
			WaitForSingleObject(isotp_client, INFINITE);
			CloseHandle(isotp_client);
		}
		WaitForSingleObject(uds_client, INFINITE);
		CloseHandle(uds_client);
	}

	// Close channel
	status = UDS_Uninitialize_2013(client_handle);
	printf("Uninitialize channel: %s\n", UDS_STATUS_OK_KO(status));

	// Exit
	system("PAUSE");
	return EXIT_SUCCESS;
}


