#include "stdafx.h"

#include "lib/PCAN-UDS_2013.h"

#define OK_KO(test) (test)?"OK":"KO"
#define STATUS_OK_KO(test) OK_KO(UDS_StatusIsOk_2013(test, PUDS_STATUS_OK, false))
#define PCAN_BITRATE  "f_clock=40000000,nom_brp=2,nom_tseg1=63,nom_tseg2=16,nom_sjw=16,data_brp=2,data_tseg1=7,data_tseg2=2,data_sjw=2"

/// <summary>
///  Entry point of the program, start a small server wich only support ReadDataByPeriodicIdentifier service.
///  This example use a specific addressing. It receives request from test equipement (0xF1 to 0xC1) in 29b fixed
///  normal addressing and sends responses for each periodic data identifier with 0x1F22C1F1 can identifier (UUDT).
/// </summary>
/// <returns>By convention, return success.</returns>
int main()
{
	uds_status status;
	cantp_handle server_handle;
	uint16_t server_address;
	uint64_t null_handle;
	HANDLE receive_event;
	BOOL boolean_status;
	uds_msgconfig service_response_config;
	bool stop;
	DWORD wait_result;
	uds_status read_status;
	uds_msg request_msg;
	uds_msg service_response_msg;
	uds_msg periodic_response;
	uds_msgconfig periodic_msg_config;
	int keyboard_res;
	uint8_t padding_value;
	uint8_t can_tx_dl;
	uint8_t periodic_data_identifier;
	uint32_t timeout_value;
	uint32_t periodic_data_identifier_length;
	uint32_t i;

	// Initialize variables
	server_handle = PCANTP_HANDLE_USBBUS2; // TODO: modify the value according to your available PCAN devices.
	null_handle = 0;
	server_address = 0xC1;
	memset(&service_response_config, 0, sizeof(uds_msgconfig));
	memset(&periodic_msg_config, 0, sizeof(uds_msgconfig));
	memset(&request_msg, 0, sizeof(uds_msg));
	memset(&service_response_msg, 0, sizeof(uds_msg));

	// Initialize server
	status = UDS_InitializeFD_2013(server_handle, PCAN_BITRATE);
	printf("Initialize channel: %s\n", STATUS_OK_KO(status));

	// Set server address parameter
	status = UDS_SetValue_2013(server_handle, PUDS_PARAMETER_SERVER_ADDRESS, &server_address, sizeof(uint32_t));
	printf("Set server address: %s\n", STATUS_OK_KO(status));

	// Set a padding value
	padding_value = 0xFF;
	status = UDS_SetValue_2013(server_handle, PUDS_PARAMETER_CAN_PADDING_VALUE, &padding_value, sizeof(padding_value));
	printf("Set padding value: %s\n", STATUS_OK_KO(status));

	// Define CAN_TX_DL=15
	can_tx_dl = 15;
	status = UDS_SetValue_2013(server_handle, PUDS_PARAMETER_CAN_TX_DL, &can_tx_dl, sizeof(can_tx_dl));
	printf("Set CAN TX DL: %s\n", STATUS_OK_KO(status));

	// Set UDS timeouts
	timeout_value = 5000;
	status = UDS_SetValue_2013(server_handle, PUDS_PARAMETER_TIMEOUT_REQUEST, &timeout_value, sizeof(timeout_value));
	printf("Set request timeout(ms): %s\n", STATUS_OK_KO(status));
	status = UDS_SetValue_2013(server_handle, PUDS_PARAMETER_TIMEOUT_RESPONSE, &timeout_value, sizeof(timeout_value));
	printf("Set response timeout(ms): %s\n", STATUS_OK_KO(status));

	// Set a receive event
	receive_event = CreateEvent(NULL, FALSE, FALSE, NULL);
	status = UDS_SetValue_2013(server_handle, PUDS_PARAMETER_RECEIVE_EVENT, &receive_event, sizeof(receive_event));
	printf("Set receive event parameter: %s\n", STATUS_OK_KO(status));

	// Initialize service response configuration
	service_response_config.can_id = (uint32_t)-1;
	service_response_config.can_msgtype = PCANTP_CAN_MSGTYPE_EXTENDED | PCANTP_CAN_MSGTYPE_FD | PCANTP_CAN_MSGTYPE_BRS;
	service_response_config.nai.protocol = PUDS_MSGPROTOCOL_ISO_15765_2_29B_FIXED_NORMAL;
	service_response_config.nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL;
	service_response_config.type = PUDS_MSGTYPE_USDT;
	service_response_config.nai.source_addr = server_address;
	service_response_config.nai.target_addr = PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
	service_response_config.nai.extension_addr = 0;

	// Initialize responses configuration (for each periodic data identifier contained in the request)
	periodic_msg_config.can_id = 0x1F22C1F1;
	periodic_msg_config.can_msgtype = PCANTP_CAN_MSGTYPE_EXTENDED | PCANTP_CAN_MSGTYPE_FD | PCANTP_CAN_MSGTYPE_BRS;
	periodic_msg_config.nai.protocol = PUDS_MSGPROTOCOL_ISO_15765_2_29B_NORMAL;
	periodic_msg_config.nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL;
	periodic_msg_config.type = PUDS_MSGTYPE_UUDT;
	periodic_msg_config.nai.source_addr = server_address;
	periodic_msg_config.nai.target_addr = PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
	periodic_msg_config.nai.extension_addr = 0;

	// Add a filter for 0x1F22C1F1 can id (in order to receive UUDT loopback messages)
	status = UDS_AddCanIdFilter_2013(server_handle, periodic_msg_config.can_id);
	printf("Add can identifier filter: %s\n", STATUS_OK_KO(status));

	// Read while user do not press Q
	printf("Start listening, press Q to quit.\n");
	stop = false;
	do {

		// Wait a receive event on receiver
		//	note: timeout is used to check keyboard hit.
		wait_result = WaitForSingleObject(receive_event, 1000);

		// If we get a receive event
		if (wait_result == WAIT_OBJECT_0) {

			do {

				// Read first available message (no filtering based on message's type is set):
				read_status = UDS_Read_2013(server_handle, &request_msg, NULL, NULL);
				printf("Try to read a message: %s\n", STATUS_OK_KO(read_status));
				if (UDS_StatusIsOk_2013(read_status, PUDS_STATUS_OK, false)) {

					// We receive a request, check its length and if it is not a loopback message
					if (request_msg.msg.msgdata.any->length >= 1 && (request_msg.msg.msgdata.any->flags & PCANTP_MSGFLAG_LOOPBACK) == 0) {

						// This is a valid request, switch services
						switch (*request_msg.links.service_id) {
						case PUDS_SI_ReadDataByPeriodicIdentifier:

							// Allocates service response message
							status = UDS_MsgAlloc_2013(&service_response_msg, service_response_config, 1);
							if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
								*service_response_msg.links.service_id = PUDS_SI_ReadDataByPeriodicIdentifier + PUDS_SI_POSITIVE_RESPONSE;
							}
							printf("Prepare response message for ReadDataByPeriodicIdentifier service: %s\n", STATUS_OK_KO(status));

							// Write service response message
							status = UDS_Write_2013(server_handle, &service_response_msg);
							printf("Write response message for ReadDataByPeriodicIdentifier service: %s\n", STATUS_OK_KO(status));

							// Free response message (and clean memory in order to reallocate later)
							status = UDS_MsgFree_2013(&service_response_msg);
							printf("Free response message: %s\n", STATUS_OK_KO(status));

							// Sends a message for each data identifier with a specific addressing.
							periodic_data_identifier_length = request_msg.msg.msgdata.any->length - 2;
							for (i = 0; i < periodic_data_identifier_length; i++) {

								// Allocates and prepares message with dummy data
								memset(&periodic_response, 0, sizeof(uds_msg));
								status = UDS_MsgAlloc_2013(&periodic_response, periodic_msg_config, 5);
								if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
									periodic_data_identifier = request_msg.links.param[1 + i];
									periodic_response.msg.msgdata.any->data[0] = periodic_data_identifier;
									periodic_response.msg.msgdata.any->data[1] = 0x12;
									periodic_response.msg.msgdata.any->data[2] = 0x34;
									periodic_response.msg.msgdata.any->data[3] = 0x56;
									periodic_response.msg.msgdata.any->data[4] = 0x78;

									printf("Allocates message for 0x%02x periodic data identifier: %s\n", periodic_data_identifier, STATUS_OK_KO(status));
									status = UDS_Write_2013(server_handle, &periodic_response);
									printf("Write message for 0x%02x periodic data identifier: %s\n", periodic_data_identifier, STATUS_OK_KO(status));
									status = UDS_MsgFree_2013(&periodic_response);
									printf("Free message for 0x%02x periodic data identifier: %s\n", periodic_data_identifier, STATUS_OK_KO(status));
								}
							}
							break;
						default:
							printf("Unknown service (0x%02x)\n", *request_msg.links.service_id);
							break;
						}
					}
				}

				// Free request message (in order to reallocate later)
				status = UDS_MsgFree_2013(&request_msg);
				printf("Free request message: %s\n", STATUS_OK_KO(status));
			} while (!UDS_StatusIsOk_2013(read_status, PUDS_STATUS_NO_MESSAGE, false));
		}

		// Quit when user press Q
		if (_kbhit()) {
			keyboard_res = _getch();
			if (keyboard_res == 'Q' || keyboard_res == 'q') {
				stop = true;
			}
		}
	} while (stop == false);

	// Close receive event
	status = UDS_SetValue_2013(server_handle, PUDS_PARAMETER_RECEIVE_EVENT, &null_handle, sizeof(HANDLE));
	printf("Stop receive event: %s\n", STATUS_OK_KO(status));
	boolean_status = CloseHandle(receive_event);
	printf("Close receive event: %s\n", OK_KO((int)boolean_status != 0));

	// Close server
	status = UDS_Uninitialize_2013(server_handle);
	printf("Uninitialize channel: %s\n", STATUS_OK_KO(status));

	// Exit
	system("PAUSE");
	return EXIT_SUCCESS;
}

