#include "stdafx.h"

#include "lib/PCAN-UDS_2013.h"

#define OK_KO(test) (test)?"OK":"KO"
#define STATUS_OK_KO(test) OK_KO(UDS_StatusIsOk_2013(test, PUDS_STATUS_OK, false))
#define PCAN_BITRATE  "f_clock=40000000,nom_brp=2,nom_tseg1=63,nom_tseg2=16,nom_sjw=16,data_brp=2,data_tseg1=7,data_tseg2=2,data_sjw=2"
#define RESPONSES_TIMEOUT_MS 5000

/// <summary>
///  Entry point of the program, start a UDS channel, ask ReadDataByPeriodicIdentifier service.
///  It sends a request from 0xF1 to 0xC1 address in 29b fixed with normal addressing. Then read
///  responses from 0x1F22C1F1 can identifier (UUDT).
/// </summary>
/// <returns>By convention, return success.</returns>
int main()
{
	uds_status status;
	cantp_handle client_handle;
	uds_msgconfig request_config;
	uds_msg msg_request;
	uds_msg request_confirmation;
	uds_msg service_response;
	uint8_t can_tx_dl;
	uint32_t timeout_value;
	uds_msg periodic_response;
	uint8_t response_count;
	uint64_t start_reading;
	uint32_t i;

	// Initialize variables
	client_handle = PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.
	memset(&request_config, 0, sizeof(uds_msgconfig));
	memset(&msg_request, 0, sizeof(uds_msg));
	memset(&request_confirmation, 0, sizeof(uds_msg));
	memset(&service_response, 0, sizeof(uds_msg));
	memset(&periodic_response, 0, sizeof(uds_msg));

	// Initialize client
	status = UDS_InitializeFD_2013(client_handle, PCAN_BITRATE);
	printf("Initialize channel: %s\n", STATUS_OK_KO(status));

	// Define CAN_TX_DL
	can_tx_dl = 15;
	status = UDS_SetValue_2013(client_handle, PUDS_PARAMETER_CAN_TX_DL, &can_tx_dl, sizeof(can_tx_dl));
	printf("Set CAN TX DL: %s\n", STATUS_OK_KO(status));

	// Check CAN_TX_DL
	can_tx_dl = 0;
	status = UDS_GetValue_2013(client_handle, PUDS_PARAMETER_CAN_TX_DL, &can_tx_dl, sizeof(can_tx_dl));
	printf("Check new CAN TX DL value(%u): %s\n", can_tx_dl, STATUS_OK_KO(status));

	// Set UDS timeouts
	timeout_value = 5000;
	status = UDS_SetValue_2013(client_handle, PUDS_PARAMETER_TIMEOUT_REQUEST, &timeout_value, sizeof(timeout_value));
	printf("Set request timeout(ms): %s\n", STATUS_OK_KO(status));
	status = UDS_SetValue_2013(client_handle, PUDS_PARAMETER_TIMEOUT_RESPONSE, &timeout_value, sizeof(timeout_value));
	printf("Set response timeout(ms): %s\n", STATUS_OK_KO(status));

	// Check timeouts
	timeout_value = 0;
	status = UDS_GetValue_2013(client_handle, PUDS_PARAMETER_TIMEOUT_REQUEST, &timeout_value, sizeof(timeout_value));
	printf("Get new request timeout(%ums): %s\n", timeout_value, STATUS_OK_KO(status));
	timeout_value = 0;
	status = UDS_GetValue_2013(client_handle, PUDS_PARAMETER_TIMEOUT_RESPONSE, &timeout_value, sizeof(timeout_value));
	printf("Get new response timeout(%ums): %s\n", timeout_value, STATUS_OK_KO(status));

	// Initialize the request configuration
	request_config.can_id = (uint32_t)-1;
	request_config.can_msgtype = PCANTP_CAN_MSGTYPE_EXTENDED | PCANTP_CAN_MSGTYPE_FD | PCANTP_CAN_MSGTYPE_BRS;
	request_config.nai.protocol = PUDS_MSGPROTOCOL_ISO_15765_2_29B_FIXED_NORMAL;
	request_config.nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL;
	request_config.type = PUDS_MSGTYPE_USDT;
	request_config.nai.source_addr = PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
	request_config.nai.target_addr = 0xC1;
	request_config.nai.extension_addr = 0;

	// Add a filter for 0x1F22C1F1 can id (in order to receive UUDT messages)
	status = UDS_AddCanIdFilter_2013(client_handle, 0x1F22C1F1);
	printf("Add can identifier filter: %s\n", STATUS_OK_KO(status));

	// Execute ReadDataByPeriodicIdentifier and wait service response
	uint8_t periodic_data_identifier[4] = { 0x12,0x34,0x56,0x78 };
	status = UDS_SvcReadDataByPeriodicIdentifier_2013(client_handle, request_config, &msg_request, PUDS_SVC_PARAM_RDBPI_SAMR, periodic_data_identifier, 4);
	printf("Execute ReadDataByPeriodicIdentifier service: %s\n", STATUS_OK_KO(status));
	status = UDS_WaitForService_2013(client_handle, &msg_request, &service_response, &request_confirmation);
	printf("Wait for service (confirmation and response): %s\n", STATUS_OK_KO(status));

	// Read responses for each given periodic data identifier
	response_count = 0;
	start_reading = GetTickCount64();
	do {
		status = UDS_Read_2013(client_handle, &periodic_response, NULL, NULL);
		if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false) && periodic_response.msg.msgdata.any->length > 0)
		{
			response_count++;
			printf("Read response for 0x%02x periodic data identifier: [", periodic_response.msg.msgdata.any->data[0]);
			for (i = 0; i < periodic_response.msg.msgdata.any->length; i++)
				printf(" 0x%02x", periodic_response.msg.msgdata.any->data[i]);
			printf(" ]\n");
			status = UDS_MsgFree_2013(&periodic_response);
			printf("Free response buffer: %s\n", STATUS_OK_KO(status));
		}

		// Wait for 4 responses or timeout
	} while (response_count < 4 && GetTickCount64() - start_reading < RESPONSES_TIMEOUT_MS);

	// Free message structures
	status = UDS_MsgFree_2013(&msg_request);
	printf("Free request message: %s\n", STATUS_OK_KO(status));
	status = UDS_MsgFree_2013(&request_confirmation);
	printf("Free request confirmation: %s\n", STATUS_OK_KO(status));
	status = UDS_MsgFree_2013(&service_response);
	printf("Free service response: %s\n", STATUS_OK_KO(status));

	// Close client
	status = UDS_Uninitialize_2013(client_handle);
	printf("Uninitialize channel: %s\n", STATUS_OK_KO(status));

	// Exit
	system("PAUSE");
	return EXIT_SUCCESS;
}

