#include "stdafx.h"

#include "lib/PCAN-UDS_2013.h"

#define OK_KO(test) (test)?"OK":"KO"
#define UDS_STATUS_OK_KO(test) OK_KO(UDS_StatusIsOk_2013(test, PUDS_STATUS_OK, false))
#define ISOTP_STATUS_OK_KO(test) OK_KO(CANTP_StatusIsOk_2016(test, PCANTP_STATUS_OK, false))

#define BUFFER_SIZE 256
#define ISOTP_PING_MSG "PING"
#define ISOTP_PING_MSG_SIZE 4
#define CAN_ID_START_RANGE 1
#define CAN_ID_STOP_RANGE 10
#define PING_TIMING_MS 500

/// <summary>Structure passed as thread parameters</summary>
struct task_params {
	/// <summary>Server channel handle</summary>
	cantp_handle server_handle;
	/// <summary>Server address</summary>
	uint32_t server_address;
	/// <summary>Determine if the thread should end or not</summary>
	bool stop_task;
};

/// <summary>ISOTP server task: periodically send "PING" message using different CAN identifiers</summary>
/// <param name="parameters">pointer on task_params structures</param>
DWORD WINAPI isotp_server_task(LPVOID parameters) {
	struct task_params* t_params;
	cantp_status status;
	cantp_msg tx_msg;
	uint32_t can_id;

	// Init variables
	t_params = (struct task_params*)parameters;
	memset(&tx_msg, 0, sizeof(cantp_msg));
	can_id = CAN_ID_START_RANGE;

	// Send loop
	do {
		// Wait before sending the next message
		Sleep(PING_TIMING_MS);

		// Initialize ISOTP Tx message containing "PING"
		status = CANTP_MsgDataAlloc_2016(&tx_msg, PCANTP_MSGTYPE_CAN);
		printf("[ISOTP] Allocate ISOTP tx message: %s\n", ISOTP_STATUS_OK_KO(status));
		status = CANTP_MsgDataInit_2016(&tx_msg, can_id, PCANTP_CAN_MSGTYPE_STANDARD, ISOTP_PING_MSG_SIZE, ISOTP_PING_MSG, NULL);
		printf("[ISOTP] Initialize ISOTP tx message: %s\n", ISOTP_STATUS_OK_KO(status));

		// Send "PING" message
		status = CANTP_Write_2016(t_params->server_handle, &tx_msg);
		printf("[ISOTP] Send ISOTP \"PING\" message (can id=0x%x): %s\n", can_id, ISOTP_STATUS_OK_KO(status));

		// Free messages
		status = CANTP_MsgDataFree_2016(&tx_msg);
		printf("[ISOTP] Free ISOTP TX message: %s\n", ISOTP_STATUS_OK_KO(status));

		// Update can id for next message
		can_id++;
		if (can_id > CAN_ID_STOP_RANGE) {
			can_id = CAN_ID_START_RANGE;
		}
	} while (t_params->stop_task == false);

	return 0;
}

/// <summary>UDS server task: respond TesterPresent request</summary>
/// <param name="parameters">pointer on task_params structures</param>
DWORD WINAPI uds_server_task(LPVOID parameters) {
	struct task_params* t_params;
	uds_msgconfig config_physical;
	DWORD wait_result;
	uds_status status;
	HANDLE receive_event;
	uds_status read_status;
	uds_msg request_msg;
	uds_msg response_msg;
	uint64_t null_handle;
	bool boolean_status;

	// Init variables
	t_params = (struct task_params*)parameters;
	null_handle = 0;
	memset(&config_physical, 0, sizeof(uds_msgconfig));
	memset(&request_msg, 0, sizeof(uds_msg));
	memset(&response_msg, 0, sizeof(uds_msg));

	// Set server address parameter
	status = UDS_SetValue_2013(t_params->server_handle, PUDS_PARAMETER_SERVER_ADDRESS, &t_params->server_address, sizeof(uint32_t));
	printf("[UDS] Set UDS server address: %s\n", UDS_STATUS_OK_KO(status));

	// Set a receive event for UDS
	receive_event = CreateEvent(NULL, FALSE, FALSE, NULL);
	status = UDS_SetValue_2013(t_params->server_handle, PUDS_PARAMETER_RECEIVE_EVENT, &receive_event, sizeof(receive_event));
	printf("[UDS] Set UDS receive event parameter: %s\n", UDS_STATUS_OK_KO(status));

	// Initialize a physical configuration
	config_physical.can_id = (uint32_t)-1;
	config_physical.can_msgtype = PCANTP_CAN_MSGTYPE_STANDARD;
	config_physical.nai.protocol = PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL;
	config_physical.nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL;
	config_physical.type = PUDS_MSGTYPE_USDT;
	config_physical.nai.extension_addr = 0;

	do {

		// Wait a receive event on receiver
		wait_result = WaitForSingleObject(receive_event, 1000);

		// If we get a receive event
		if (wait_result == WAIT_OBJECT_0) {
			do {

				// Read first available message (no filtering based on message's type is set):
				read_status = UDS_Read_2013(t_params->server_handle, &request_msg, NULL, NULL);
				printf("[UDS] Try to read a message: %s\n", UDS_STATUS_OK_KO(read_status));
				if (UDS_StatusIsOk_2013(read_status, PUDS_STATUS_OK, false)) {

					// We receive a request, check its length and if it is not a loopback message and if it is a USDT message
					if (request_msg.type == PUDS_MSGTYPE_USDT && request_msg.msg.msgdata.any->length >= 1 && (request_msg.msg.msgdata.any->flags & PCANTP_MSGFLAG_LOOPBACK) == 0) {

						// This is a valid request, switch services
						switch (*request_msg.links.service_id) {
						case PUDS_SI_TesterPresent:

							// Allocate response message
							status = UDS_MsgAlloc_2013(&response_msg, config_physical, 2);
							printf("[UDS] Prepare response message for TesterPresent service: %s\n", UDS_STATUS_OK_KO(status));

							if (UDS_StatusIsOk_2013(status, PUDS_STATUS_OK, false)) {
								// Fill parameters
								memcpy(&response_msg.msg.msgdata.isotp->netaddrinfo, &request_msg.msg.msgdata.isotp->netaddrinfo, sizeof(cantp_netaddrinfo));
								*response_msg.links.service_id = PUDS_SI_TesterPresent + PUDS_SI_POSITIVE_RESPONSE;
								response_msg.links.param[0] = 0;
								response_msg.msg.msgdata.isotp->netaddrinfo.target_addr = request_msg.msg.msgdata.isotp->netaddrinfo.source_addr;
								response_msg.msg.msgdata.isotp->netaddrinfo.source_addr = (uint16_t)t_params->server_address;

								// Write response message
								status = UDS_Write_2013(t_params->server_handle, &response_msg);
								printf("[UDS] Write response message for TesterPresent service: %s\n", UDS_STATUS_OK_KO(status));
							}

							// Free response message (and clean memory in order to reallocate later)
							status = UDS_MsgFree_2013(&response_msg);
							printf("[UDS] Free response message: %s\n", UDS_STATUS_OK_KO(status));
							break;
						default:
							printf("[UDS] Unknown service (0x%02x)\n", *request_msg.links.service_id);
							break;
						}
					}
				}

				// Free request message (and clean memory in order to reallocate later)
				status = UDS_MsgFree_2013(&request_msg);
				printf("[UDS] Free request message: %s\n", UDS_STATUS_OK_KO(status));
			} while (!UDS_StatusIsOk_2013(read_status, PUDS_STATUS_NO_MESSAGE, false));
		}
	} while (t_params->stop_task == false);

	// Close receive event
	status = UDS_SetValue_2013(t_params->server_handle, PUDS_PARAMETER_RECEIVE_EVENT, &null_handle, sizeof(HANDLE));
	printf("[UDS] Stop UDS receive event: %s\n", UDS_STATUS_OK_KO(status));
	boolean_status = (CloseHandle(receive_event) == TRUE) ? true : false;
	printf("[UDS] Close UDS receive event: %s\n", OK_KO(boolean_status != false));

	return 0;
}

/// <summary>Entry point of the program, start a small server which handle UDS testerpresent request and periodically send isotp messages</summary>
/// <returns>By convention, return success.</returns>
int main()
{
	uds_status status;
	struct task_params t_params;
	char buffer[BUFFER_SIZE];
	HANDLE uds_server;
	HANDLE isotp_server;
	DWORD uds_server_id;
	DWORD isotp_server_id;
	int keyboard_res;

	// Initialize variables
	t_params.server_handle = PCANTP_HANDLE_USBBUS2; // TODO: modify the value according to your available PCAN devices.
	t_params.server_address = PUDS_ISO_15765_4_ADDR_ECU_1;
	t_params.stop_task = false;
	buffer[0] = '\0';

	// Print version informations
	status = UDS_GetValue_2013(PCANTP_HANDLE_NONEBUS, PUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE);
	printf("PCAN-UDS API Version - %s: %s\n", buffer, UDS_STATUS_OK_KO(status));

	// Initialize server
	status = UDS_Initialize_2013(t_params.server_handle, PCANTP_BAUDRATE_500K, (cantp_hwtype)0, 0, 0);
	printf("Initialize channel: %s\n", UDS_STATUS_OK_KO(status));

	// Start uds and isotp servers
	uds_server = CreateThread(NULL, 0, uds_server_task, &t_params, 0, &uds_server_id);
	if (uds_server != NULL) {
		isotp_server = CreateThread(NULL, 0, isotp_server_task, &t_params, 0, &isotp_server_id);
		if (isotp_server != NULL) {

			// Read while user do not press Q
			printf("Start listening, press Q to quit.\n");
			t_params.stop_task = false;
			do {
				// Quit when user press Q
				if (_kbhit()) {
					keyboard_res = _getch();
					if (keyboard_res == 'Q' || keyboard_res == 'q') {
						t_params.stop_task = true;
					}
				}
			} while (t_params.stop_task == false);

			// Close threads
			WaitForSingleObject(isotp_server, INFINITE);
			CloseHandle(isotp_server);
		}
		WaitForSingleObject(uds_server, INFINITE);
		CloseHandle(uds_server);
	}

	// Close channel
	status = UDS_Uninitialize_2013(t_params.server_handle);
	printf("Uninitialize channel: %s\n", UDS_STATUS_OK_KO(status));

	//Exit
	system("PAUSE");
	return EXIT_SUCCESS;
}

