#include "stdafx.h"

#include "lib/PCAN-UDS_2013.h"

#define OK_KO(test) (test)?"OK":"KO"
#define UDS_STATUS_OK_KO(test) OK_KO(UDS_StatusIsOk_2013(test, PUDS_STATUS_OK, false))
#define ISOTP_STATUS_OK_KO(test) OK_KO(CANTP_StatusIsOk_2016(test, PCANTP_STATUS_OK, false))

#define BUFFER_SIZE 256
#define UDS_REQUEST_TIMING_MS 500
#define CLIENT_EXECUTION_TIME_MS 10000

/// <summary>Structure passed as thread parameters</summary>
struct task_params {
	/// <summary>Client channel handle</summary>
	cantp_handle client_handle;
	/// <summary>Determine if the thread should end or not</summary>
	bool stop_task;
};

/// <summary>UDS client task: request TesterPresent service several time</summary>
/// <param name="parameters">pointer on task_params structures</param>
DWORD WINAPI uds_client_task(LPVOID parameters) {
	uds_status status;
	uds_msgconfig config_physical;
	uds_msg msg_request;
	uds_msg request_confirmation;
	uds_msg response;
	struct task_params* t_params;

	// Initialize variables
	memset(&config_physical, 0, sizeof(config_physical));
	memset(&msg_request, 0, sizeof(msg_request));
	memset(&request_confirmation, 0, sizeof(request_confirmation));
	memset(&response, 0, sizeof(response));
	t_params = (struct task_params*)parameters;

	// Initialize a physical configuration
	config_physical.can_id = PUDS_ISO_15765_4_CAN_ID_PHYSICAL_REQUEST_1;
	config_physical.can_msgtype = PCANTP_CAN_MSGTYPE_STANDARD;
	config_physical.nai.protocol = PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL;
	config_physical.nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL;
	config_physical.type = PUDS_MSGTYPE_USDT;
	config_physical.nai.source_addr = PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
	config_physical.nai.target_addr = PUDS_ISO_15765_4_ADDR_ECU_1;
	config_physical.nai.extension_addr = 0;

	// Execute TesterPresent and wait response
	do {
		// Wait before transmitting the next message
		Sleep(UDS_REQUEST_TIMING_MS);

		status = UDS_SvcTesterPresent_2013(t_params->client_handle, config_physical, &msg_request, PUDS_SVC_PARAM_TP_ZSUBF);
		printf("[UDS] Execute TesterPresent service: %s\n", UDS_STATUS_OK_KO(status));
		status = UDS_WaitForService_2013(t_params->client_handle, &msg_request, &response, &request_confirmation);
		printf("[UDS] Wait for service: %s\n", UDS_STATUS_OK_KO(status));
		status = UDS_MsgFree_2013(&msg_request);
		printf("[UDS] Free request message: %s\n", UDS_STATUS_OK_KO(status));
		status = UDS_MsgFree_2013(&request_confirmation);
		printf("[UDS] Free request confirmation message: %s\n", UDS_STATUS_OK_KO(status));
		status = UDS_MsgFree_2013(&response);
		printf("[UDS] Free response message: %s\n", UDS_STATUS_OK_KO(status));
	} while (t_params->stop_task == false);
	return 0;
}

/// <summary>ISOTP client task: read and print isotp messages</summary>
/// <param name="parameters">pointer on task_params structures</param>
DWORD WINAPI isotp_client_task(LPVOID parameters) {
	cantp_status status;
	HANDLE receive_event;
	BOOL boolean_status;
	uint64_t null_handle;
	struct task_params* t_params;
	cantp_msg rx_msg;
	DWORD wait_result;
	char buffer[BUFFER_SIZE];
	uint8_t isotp_param;

	// Initialize variables
	null_handle = 0;
	t_params = (struct task_params*)parameters;
	memset(&rx_msg, 0, sizeof(cantp_msg));
	buffer[0] = '\0';

	// Configure isotp to get any messages (disable can identifier filtering)
	isotp_param = 0;
	status = CANTP_SetValue_2016(t_params->client_handle, PCANTP_PARAMETER_FILTER_CAN_ID, &isotp_param, sizeof(isotp_param));
	printf("[ISOTP] Disable can identifier filtering in isotp: %s\n", ISOTP_STATUS_OK_KO(status));

	// Configure isotp to get a copy of UDS messages
	isotp_param = 1;
	status = CANTP_SetValue_2016(t_params->client_handle, PCANTP_PARAMETER_KEEP_HIGHER_LAYER_MESSAGES, &isotp_param, sizeof(isotp_param));
	printf("[ISOTP] Activate higher layer messages in isotp: %s\n", ISOTP_STATUS_OK_KO(status));

	// Create a isotp receive event
	receive_event = CreateEvent(NULL, FALSE, FALSE, NULL);
	status = CANTP_SetValue_2016(t_params->client_handle, PCANTP_PARAMETER_RECEIVE_EVENT, &receive_event, sizeof(receive_event));
	printf("[ISOTP] Set isotp receive event parameter: %s\n", ISOTP_STATUS_OK_KO(status));

	// Read and print ISOTP messages
	do {
		wait_result = WaitForSingleObject(receive_event, 1000);
		printf("[ISOTP] Wait a receive event from isotp: %s\n", OK_KO(wait_result == WAIT_OBJECT_0));

		// Read ISOTP messages
		do {
			status = CANTP_Read_2016(t_params->client_handle, &rx_msg, NULL, PCANTP_MSGTYPE_NONE);

			// Check if we received a message
			if (!CANTP_StatusIsOk_2016(status, PCANTP_STATUS_NO_MESSAGE, false)) {
				printf("[ISOTP] Read ISOTP message: %s\n", ISOTP_STATUS_OK_KO(status));
				if (rx_msg.msgdata.any->length > 0 && (rx_msg.msgdata.any->data[0] == PUDS_SI_TesterPresent || rx_msg.msgdata.any->data[0] == PUDS_SI_TesterPresent + PUDS_SI_POSITIVE_RESPONSE)) {
					// This message is a TesterPresent message, PCANTP_PARAMETER_KEEP_HIGHER_LAYER_MESSAGES
					// option must be activated to get these messages.
					printf("[ISOTP] Message is a TesterPresent service message.\n");
				}
				else if (CANTP_StatusIsOk_2016(status, PCANTP_STATUS_OK, false) && rx_msg.msgdata.any->length < BUFFER_SIZE) {
					// This message is a CAN message received by ISOTP
					memcpy(buffer, rx_msg.msgdata.any->data, rx_msg.msgdata.any->length);
					buffer[rx_msg.msgdata.any->length] = '\0';
					printf("[ISOTP] Message from 0x%x can identifier contains \"%s\"\n", rx_msg.can_info.can_id, buffer);
				}

				status = CANTP_MsgDataFree_2016(&rx_msg);
				printf("[ISOTP] Free RX message: %s\n", ISOTP_STATUS_OK_KO(status));
			}
		} while (!CANTP_StatusIsOk_2016(status, PCANTP_STATUS_NO_MESSAGE, false));
	} while (t_params->stop_task == false);

	// Close receive event
	status = CANTP_SetValue_2016(t_params->client_handle, PCANTP_PARAMETER_RECEIVE_EVENT, &null_handle, sizeof(HANDLE));
	printf("[ISOTP] Stop ISOTP receive event: %s\n", ISOTP_STATUS_OK_KO(status));
	boolean_status = CloseHandle(receive_event);
	printf("[ISOTP] Close ISOTP receive event: %s\n", OK_KO((int)boolean_status != 0));

	return 0;
}

/// <summary>Entry point of the program, start a UDS channel, ask in the same time UDS TesterPresent and isotp request</summary>
/// <returns>By convention, return success.</returns>
int main()
{
	char buffer[BUFFER_SIZE];
	uds_status status;
	cantp_handle client_handle;
	HANDLE uds_client;
	HANDLE isotp_client;
	DWORD uds_client_id;
	DWORD isotp_client_id;
	struct task_params t_params;

	// Initialize variables
	client_handle = PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.
	buffer[0] = '\0';
	t_params.client_handle = client_handle;
	t_params.stop_task = false;

	// Print version informations
	status = UDS_GetValue_2013(PCANTP_HANDLE_NONEBUS, PUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE);
	printf("PCAN-UDS API Version - %s: %s\n", buffer, UDS_STATUS_OK_KO(status));

	// Initialize client channel
	status = UDS_Initialize_2013(client_handle, PCANTP_BAUDRATE_500K, (cantp_hwtype)0, 0, 0);
	printf("Initialize channel: %s\n", UDS_STATUS_OK_KO(status));

	// Start uds and isotp clients
	uds_client = CreateThread(NULL, 0, uds_client_task, &t_params, 0, &uds_client_id);
	if (uds_client != NULL) {
		isotp_client = CreateThread(NULL, 0, isotp_client_task, &t_params, 0, &isotp_client_id);
		if (isotp_client != NULL) {
			Sleep(CLIENT_EXECUTION_TIME_MS);
			t_params.stop_task = true;
			WaitForSingleObject(isotp_client, INFINITE);
			CloseHandle(isotp_client);
		}
		WaitForSingleObject(uds_client, INFINITE);
		CloseHandle(uds_client);
	}

	// Close channel
	status = UDS_Uninitialize_2013(client_handle);
	printf("Uninitialize channel: %s\n", UDS_STATUS_OK_KO(status));

	// Exit
	system("PAUSE");
	return EXIT_SUCCESS;
}


