// PCUServer.cpp : Defines the entry point for the console application.
//
#include "stdafx.h"
#include <conio.h>
#include <stdlib.h>

#include "lib/PCAN-UDS.h"

// Inverts the bytes of a 16 bits numeric value
WORD Reverse16(const BYTE *v)
{
	return v[0] << 8 | v[1];
}

// A function that displays UDS messages
void displayMessage(TPUDSMsg * Message, bool isRx)
{
	char buffer[500];

	if (Message->MSGTYPE != PUDS_MESSAGE_TYPE_INDICATION)
	{
		printf("\n%s UDS message from 0x%02x (to 0x%02x, with RA 0x%02x) - result: %i - %s\n",
			isRx ? "Received" : "Transmitted",
			(int)Message->NETADDRINFO.SA,
			(int)Message->NETADDRINFO.TA,
			(int)Message->NETADDRINFO.RA,
			(int)Message->RESULT,
			Message->RESULT != PUDS_RESULT_N_OK ? "ERROR !!!" : "OK !");
		// display data
		setvbuf(stdout, buffer, _IOLBF, BUFSIZ);
		printf("\t\\-> Length: %i, Data= ", (int)Message->LEN);
		for (int i = 0; i < Message->LEN; i++) {
			printf("%02x ", (int)Message->DATA.RAW[i]);
			fflush(stdout);
		}
		printf("\n");
		setvbuf(stdout, NULL, _IONBF, 0);
	}
	else
	{
		printf("\nPENDING UDS message from 0x%02x (to 0x%02x, with RA 0x%02x) -> LEN=%i ...\n",
			(int)Message->NETADDRINFO.SA,
			(int)Message->NETADDRINFO.TA,
			(int)Message->NETADDRINFO.RA,
			(int)Message->LEN);
	}
}

// This function generates and transmits UDS responses
void transmitResponse(TPUDSCANHandle Channel, BYTE serverAddr, TPUDSMsg * UDSRequest)
{
	TPUDSStatus Status;
	TPUDSMsg Message = {};
	BYTE SI = UDSRequest->DATA.REQUEST.SI;
	WORD lDataIdentifier;
	WORD lCount;
	WORD lMemorySizeLength;
	WORD lMemoryAddressLength;

	// copy N_AI
	memcpy(&Message.NETADDRINFO, &UDSRequest->NETADDRINFO, sizeof(TPUDSNetAddrInfo));
	if (UDSRequest->NETADDRINFO.TA_TYPE == PUDS_ADDRESSING_FUNCTIONAL)
		// response to functional addressing is set to TEST_EQUIPMENT
		Message.NETADDRINFO.TA = PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
	else
		Message.NETADDRINFO.TA = UDSRequest->NETADDRINFO.SA;
	Message.NETADDRINFO.SA = serverAddr;
	Message.NETADDRINFO.TA_TYPE = PUDS_ADDRESSING_PHYSICAL;
	// all responses are positive
	Message.DATA.POSITIVE.SI = UDSRequest->DATA.REQUEST.SI + PUDS_SI_POSITIVE_RESPONSE;

	// customize message response based on the UDS Service ID (see ISO 14229-1)
	switch (SI)
	{
	case PUDS_SI_DiagnosticSessionControl:
		Message.DATA.POSITIVE.PARAM[0] = UDSRequest->DATA.REQUEST.PARAM[0];
		Message.DATA.POSITIVE.PARAM[1] = 0x00;	// P2Can_Server_Max = 0x0010
		Message.DATA.POSITIVE.PARAM[2] = 0x10;
		Message.DATA.POSITIVE.PARAM[3] = 0x03;	// P2*Can_Server_Max = 0x03E8
		Message.DATA.POSITIVE.PARAM[4] = 0xE8;
		Message.LEN = 6;
		break;
	case PUDS_SI_ECUReset:
		Message.DATA.POSITIVE.PARAM[0] = UDSRequest->DATA.REQUEST.PARAM[0];
		Message.LEN = 2;
		if (UDSRequest->DATA.REQUEST.PARAM[0] == PUDS_SVC_PARAM_ER_ERPSD)
		{
			Message.DATA.POSITIVE.PARAM[1] = 0x66;	// power down time
			Message.LEN = 3;
		}
		break;
	case PUDS_SI_SecurityAccess:
		Message.DATA.POSITIVE.PARAM[0] = UDSRequest->DATA.REQUEST.PARAM[0];
		Message.LEN = 2;
		if (UDSRequest->DATA.REQUEST.PARAM[0] >= PUDS_SVC_PARAM_SA_RSD_MIN
			&& UDSRequest->DATA.REQUEST.PARAM[0] <= PUDS_SVC_PARAM_SA_RSD_MAX
			&& UDSRequest->DATA.REQUEST.PARAM[0] % 2 == 1) {	// Request security seed are Even values
			// fill with dummy data
			Message.LEN = (1 + SI) > PUDS_MAX_DATA ? PUDS_MAX_DATA : (1 + SI);
			for (int i = 1; i < Message.LEN - 1; i++)	// (LEN - 1) as POSITIVE.SI uses 1 byte
				Message.DATA.POSITIVE.PARAM[i] = i + 1;
		}
		break;
	case PUDS_SI_CommunicationControl:
		Message.DATA.POSITIVE.PARAM[0] = UDSRequest->DATA.REQUEST.PARAM[0];
		Message.LEN = 2;
		break;
	case PUDS_SI_TesterPresent:
		Message.DATA.POSITIVE.PARAM[0] = UDSRequest->DATA.REQUEST.PARAM[0];
		Message.LEN = 2;
		break;
	case PUDS_SI_SecuredDataTransmission:
		// fill with dummy data (check Security-SubLayer record defined in ISO-15764)
		Message.LEN = 1 + rand() % 50;
		for (int i = 0; i < Message.LEN - 1; i++)	// (LEN - 1) as POSITIVE.SI uses 1 byte
			Message.DATA.POSITIVE.PARAM[i] = i + 1;
		break;
	case PUDS_SI_ControlDTCSetting:
		Message.DATA.POSITIVE.PARAM[0] = UDSRequest->DATA.REQUEST.PARAM[0];
		Message.LEN = 2;
		break;
	case PUDS_SI_ResponseOnEvent:
		Message.DATA.POSITIVE.PARAM[0] = UDSRequest->DATA.REQUEST.PARAM[0];
		if (UDSRequest->DATA.REQUEST.PARAM[0] == PUDS_SVC_PARAM_ROE_RAE)
		{
			Message.DATA.POSITIVE.PARAM[1] = 0;	// # of activated events
			Message.LEN = 3;
			// EventType and ServiceToRespondTo Records not implemented
		}
		else
		{
			Message.DATA.POSITIVE.PARAM[1] = 0;	// # of identified events
			Message.DATA.POSITIVE.PARAM[2] = UDSRequest->DATA.REQUEST.PARAM[1];	// # event window time
			Message.LEN = 4;
			// EventType and ServiceToRespondTo Records not implemented
		}
		break;
	case PUDS_SI_LinkControl:
		Message.DATA.POSITIVE.PARAM[0] = UDSRequest->DATA.REQUEST.PARAM[0];
		Message.LEN = 2;
		break;
	case PUDS_SI_ReadDataByIdentifier:
		lCount = 0;
		for (int i = 0; i < UDSRequest->LEN - 1; i += 2)
		{
			// Use helper function to read network data (in big endian format) to WORD value (in windows i.e. little endian format)
			lDataIdentifier = Reverse16(&UDSRequest->DATA.REQUEST.PARAM[i]);
			// copy DataIdentifier
			Message.DATA.POSITIVE.PARAM[lCount++] = UDSRequest->DATA.REQUEST.PARAM[i];
			Message.DATA.POSITIVE.PARAM[lCount++] = UDSRequest->DATA.REQUEST.PARAM[i + 1];
			// DataRecord : fill with dummy data
			for (int j = 0; j < 5; j++) {
				Message.DATA.POSITIVE.PARAM[lCount++] = j + 'A';
			}
		}
		Message.LEN = lCount + 1;
		break;
	case PUDS_SI_ReadMemoryByAddress:
		// read memorySize = bits [7..4]
		Message.LEN = 1 + ((UDSRequest->DATA.REQUEST.PARAM[1] >> 4) & 0xF);
		// fill with dummy data
		for (int i = 0; i < Message.LEN - 1; i++)	// (LEN - 1) as POSITIVE.SI uses 1 byte
			Message.DATA.POSITIVE.PARAM[i] = i + 1;
		break;
	case PUDS_SI_ReadScalingDataByIdentifier:
		// Use helper function to read network data (in big endian format) to WORD value (in windows i.e. little endian format)
		lDataIdentifier = Reverse16(&UDSRequest->DATA.REQUEST.PARAM[0]);
		// copy DataIdentifier
		Message.DATA.POSITIVE.PARAM[0] = UDSRequest->DATA.REQUEST.PARAM[0];
		Message.DATA.POSITIVE.PARAM[1] = UDSRequest->DATA.REQUEST.PARAM[1];
		// create a formula Vehicule Speed = (0.75*x+30) km/h
		Message.DATA.POSITIVE.PARAM[2] = (0x0 << 4) | (0x1);	// unSignedNumeric of 1 Bytes)
		Message.DATA.POSITIVE.PARAM[3] = 0x90;	// formula, 0 data bytes
		Message.DATA.POSITIVE.PARAM[4] = 0x00;	// formulaIdentifier = C0 * x + C1
		Message.DATA.POSITIVE.PARAM[5] = 0xE0;	// C0 high byte
		Message.DATA.POSITIVE.PARAM[6] = 0x4B;	// C0 low byte
		Message.DATA.POSITIVE.PARAM[7] = 0x00;	// C1 high byte
		Message.DATA.POSITIVE.PARAM[8] = 0x1E;	// C1 low byte
		Message.DATA.POSITIVE.PARAM[9] = 0xA0;	// unit/format, 0 data bytes
		Message.DATA.POSITIVE.PARAM[10] = 0x30;	// unit ID, km/h
		Message.LEN = 11;
		break;
	case PUDS_SI_ReadDataByPeriodicIdentifier:
		Message.LEN = 1;
		break;
	case PUDS_SI_DynamicallyDefineDataIdentifier:
		Message.LEN = 4;
		Message.DATA.POSITIVE.PARAM[0] = UDSRequest->DATA.REQUEST.PARAM[0];
		// Use helper function to read network data (in big endian format) to WORD value (in windows i.e. little endian format)
		lDataIdentifier = Reverse16(&UDSRequest->DATA.REQUEST.PARAM[0]);
		// copy DataIdentifier
		Message.DATA.POSITIVE.PARAM[1] = UDSRequest->DATA.REQUEST.PARAM[1];
		Message.DATA.POSITIVE.PARAM[2] = UDSRequest->DATA.REQUEST.PARAM[2];
		break;
	case PUDS_SI_WriteDataByIdentifier:
		Message.LEN = 3;
		// Use helper function to read network data (in big endian format) to WORD value (in windows i.e. little endian format)
		lDataIdentifier = Reverse16(&UDSRequest->DATA.REQUEST.PARAM[0]);
		// copy DataIdentifier
		Message.DATA.POSITIVE.PARAM[0] = UDSRequest->DATA.REQUEST.PARAM[0];
		Message.DATA.POSITIVE.PARAM[1] = UDSRequest->DATA.REQUEST.PARAM[1];
		break;
	case PUDS_SI_WriteMemoryByAddress:
		// read MemorySizeLength & MemoryAddressLength
		lMemorySizeLength = (UDSRequest->DATA.REQUEST.PARAM[0] >> 4) & 0xF;
		lMemoryAddressLength = UDSRequest->DATA.REQUEST.PARAM[0] & 0xF;
		Message.LEN = 2 + lMemorySizeLength + lMemoryAddressLength;
		// copy Address and Memory parameters
		Message.DATA.REQUEST.PARAM[0] = UDSRequest->DATA.REQUEST.PARAM[0];
		memcpy(&Message.DATA.REQUEST.PARAM[1],
			&UDSRequest->DATA.REQUEST.PARAM[1], lMemoryAddressLength);
		memcpy(&Message.DATA.REQUEST.PARAM[1 + lMemoryAddressLength],
			&UDSRequest->DATA.REQUEST.PARAM[1 + lMemoryAddressLength], lMemorySizeLength);
		break;
	case PUDS_SI_ClearDiagnosticInformation:
		Message.LEN = 1;
		break;
	case PUDS_SI_InputOutputControlByIdentifier:
		Message.LEN = 3;
		// Use helper function to read network data (in big endian format) to WORD value (in windows i.e. little endian format)
		lDataIdentifier = Reverse16(&UDSRequest->DATA.REQUEST.PARAM[0]);
		// copy DataIdentifier
		Message.DATA.POSITIVE.PARAM[0] = UDSRequest->DATA.REQUEST.PARAM[0];
		Message.DATA.POSITIVE.PARAM[1] = UDSRequest->DATA.REQUEST.PARAM[1];
		// ControlStatus Record not implemented
		break;
	case PUDS_SI_RoutineControl:
		Message.LEN = 4;
		Message.DATA.POSITIVE.PARAM[0] = UDSRequest->DATA.REQUEST.PARAM[0];
		// Use helper function to read network data (in big endian format) to WORD value (in windows i.e. little endian format)
		lDataIdentifier = Reverse16(&UDSRequest->DATA.REQUEST.PARAM[0]);
		// copy DataIdentifier
		Message.DATA.POSITIVE.PARAM[1] = UDSRequest->DATA.REQUEST.PARAM[1];
		Message.DATA.POSITIVE.PARAM[2] = UDSRequest->DATA.REQUEST.PARAM[2];
		// RoutineStatus Record not implemented
		break;
	case PUDS_SI_RequestDownload:
	case PUDS_SI_RequestUpload:
		Message.LEN = 2 + 1 + rand() % 50;
		Message.DATA.POSITIVE.PARAM[0] = 0xF0;	// max number of block length = 0xF
		// fill with dummy data
		for (int i = 1; i < Message.LEN - 1; i++)	// (LEN - 1) as POSITIVE.SI uses 1 byte
			Message.DATA.POSITIVE.PARAM[i] = i + 1;
		break;
	case PUDS_SI_TransferData:
		Message.LEN = (2 + UDSRequest->LEN) > PUDS_MAX_DATA ? PUDS_MAX_DATA : 2 + UDSRequest->LEN;
		// custom response to PCUClient example:
		//  a. service is requested functionally,
		//  b. 1st response is NRC ResponsePending
		//  c. wait
		//  d. send correct response
		if (UDSRequest->NETADDRINFO.TA_TYPE == PUDS_ADDRESSING_FUNCTIONAL) {

			// Transmit a NRC ResponsePending response
			Message.LEN = 3;
			Message.DATA.NEGATIVE.NR_SI = PUDS_NR_SI;
			Message.DATA.NEGATIVE.SI = SI;
			Message.DATA.NEGATIVE.NRC = PUDS_NRC_EXTENDED_TIMING;
			Status = UDS_Write(Channel, &Message);
			printf("\n   ...Transmitting a NRC Response Pending message: %i\n", (int)Status);
			printf("\n   ...simulating computation... (waiting ~%dms)", PUDS_P2CAN_ENHANCED_SERVER_MAX);
			Sleep(PUDS_P2CAN_ENHANCED_SERVER_MAX - 100);
			// initialize real service response
			Message.LEN = PUDS_MAX_DATA;
			Message.DATA.POSITIVE.SI = UDSRequest->DATA.REQUEST.SI + PUDS_SI_POSITIVE_RESPONSE;
		}
		Message.DATA.POSITIVE.PARAM[0] = UDSRequest->DATA.REQUEST.PARAM[0];
		// fill with dummy data
		for (int i = 1; i < Message.LEN - 1; i++)	// (LEN - 1) as POSITIVE.SI uses 1 byte
			Message.DATA.POSITIVE.PARAM[i] = i + 1;
		break;
	case PUDS_SI_RequestTransferExit:
		Message.LEN = 1 + 1 + rand() % 50;
		// fill with dummy data
		for (int i = 0; i < Message.LEN - 1; i++)	// (LEN - 1) as POSITIVE.SI uses 1 byte
			Message.DATA.POSITIVE.PARAM[i] = i + 1;
		break;
	default:
		// fill with dummy data
		Message.LEN = (1 + SI) > PUDS_MAX_DATA ? PUDS_MAX_DATA : (1 + SI);
		Message.DATA.POSITIVE.PARAM[0] = SI;
		for (int i = 1; i < Message.LEN - 1; i++)	// (LEN - 1) as POSITIVE.SI uses 1 byte
			Message.DATA.POSITIVE.PARAM[i] = i + 1;
		break;
	}

	// Transmit UDS response
	Status = UDS_Write(Channel, &Message);
	printf("\n   ...Transmitting response: %i\n", (int)Status);
	memcpy(UDSRequest, &Message, sizeof(TPUDSMsg));
}

/// <summary>Main entry-point for this application.</summary>
/// <param name="argc">The argc.</param>
/// <param name="argv">[in,out] If non-null, the argv. The application can take 2 arguments: USB device number and Server ECU number</param>
/// <returns>Exit status</returns>
int _tmain(int argc, _TCHAR* argv[])
{
	TPUDSStatus Status;
	TPUDSCANHandle Channel;
	int nbErr = 0;
	WORD param = 0x0;
	BYTE serverAddr = PUDS_ISO_15765_4_ADDR_ECU_1;
	char buff[50] = {};

	// Show version information
	UDS_GetValue(0, PUDS_PARAM_API_VERSION, &buff, 50);
	printf("PCAN-UDS API Version: %s\n", buff);

	// Sets the default PCAN-Channel to use (PCAN-USB Channel 2)
	//
	Channel = PUDS_USBBUS2;

	// Sets server address and channel from application arguments if specified
	if (argc == 3) {
		Channel = PUDS_USBBUS1 - 1 + _ttoi(argv[1]);
		serverAddr = (BYTE)_ttoi(argv[2]);
	}

	// Initializing of the UDS Communication session
	//
	Status = UDS_Initialize(Channel, PUDS_BAUD_250K, 0, 0, 0);
	printf("Initialize UDS: %i (chan. 0x%02x)\n", (int)Status, Channel);

	// Define server address and filtered address
	//
	param = serverAddr;
	Status = UDS_SetValue(Channel, PUDS_PARAM_SERVER_ADDRESS, &param, sizeof(param));
	printf("Set ServerAddress: %i\n", (int)Status);
	Status = UDS_GetValue(Channel, PUDS_PARAM_SERVER_ADDRESS, &param, sizeof(param));
	printf("ServerAddress = 0x%02x\n", param);
	// listen to the standard OBD functional address
	param = PUDS_ISO_15765_4_ADDR_OBD_FUNCTIONAL | PUDS_SERVER_FILTER_LISTEN;
	Status = UDS_SetValue(Channel, PUDS_PARAM_SERVER_FILTER, &param, sizeof(param));
	printf("Set Filtered Address: %i\n", (int)Status);

	printf("\nNote: press 'c' or 'C' to clear screen,");
	printf("\nNote: press 'q', 'Q' or '<Escape>' to quit...\n\n");

	// Message Polling
	//
	TPUDSMsg Message;
	bool bStop = false;
	while (!bStop)
	{
		// check message
		Status = UDS_Read(Channel, &Message);
		if (Status == PUDS_ERROR_OK) {
			// display message
			displayMessage(&Message, Message.NETADDRINFO.SA != serverAddr);
			// check if an automatic reply should be sent
			if (Message.NETADDRINFO.SA != serverAddr	// do not reply to Tx Confirmation from this server
				&& Message.RESULT == PUDS_RESULT_N_OK)	// and reply if there is no Network error
			{
				// do not reply if it was requested not to
				if (Message.NO_POSITIVE_RESPONSE_MSG == PUDS_SUPPR_POS_RSP_MSG_INDICATION_BIT)
				{
					printf("\n   ...Skipping response...\n");
				}
				// do not reply to incoming message notification
				else if (Message.MSGTYPE != PUDS_MESSAGE_TYPE_INDICATION)
				{
					transmitResponse(Channel, serverAddr, &Message);
				}
			}
		}
		Sleep(1);
		// check exit request
		if (_kbhit()) {
			switch (_getch()) {
			case 'q':
			case 'Q':
			case 27:	//Escape
				bStop = true;
				break;
			case 'c':
			case 'C':
				system("cls");
				printf("ServerAddress = 0x%x\n", serverAddr);
				printf("\nNote: press 'c' or 'C' to clear screen,");
				printf("\nNote: press 'q', 'Q' or '<Escape>' to quit...\n\n");
				break;
			}
		}
	}

	// Display a small report
	if (nbErr > 0) {
		printf("\nERROR : %d errors occured.\n\n", nbErr);
		printf("\n\nPress <Enter> to quit...");
		getchar();
	}

	// Release channel
	UDS_Uninitialize(Channel);
	return 0;
}

