program client_request_ecu_reset_USDT;

{$APPTYPE CONSOLE}

uses
  SysUtils,
  ShellApi,
  windows,
  PUDS_2013 in 'PUDS_2013.pas',
  PCANBasic in 'PCANBasic.pas',
  PCANTP_2016 in 'PCANTP_2016.pas';

type
  Puds_sessioninfo = ^uds_sessioninfo;

Function OK_KO(test: Boolean): String;
begin
  If test Then
    result := 'OK'
  else
    result := 'KO';
End;

Function STATUS_OK_KO(test: uds_status): String;
begin
  result := OK_KO(TUDSApi.StatusIsOk_2013(test));
End;

/// <summary>Helper: check if two UDS session info structure are the same</summary>
/// <param name="session_1">Session to compare</param>
/// <param name="session_2">Session to compare</param>
/// <returns>Same session info (true) or not (false)</returns>
Function is_same_uds_sessioninfo(session_1: Puds_sessioninfo;
  session_2: Puds_sessioninfo): Boolean;
var
  res: Boolean;
begin
  res := false;
  If (session_1.nai.protocol = session_2.nai.protocol) And
    (session_1.nai.target_type = session_2.nai.target_type) And
    (session_1.nai.source_addr = session_2.nai.source_addr) And
    (session_1.nai.target_addr = session_2.nai.target_addr) And
    (session_1.nai.extension_addr = session_2.nai.extension_addr) And
    (session_1.can_msg_type = session_2.can_msg_type) And
    (session_1.session_type = session_2.session_type) And
    (session_1.timeout_p2can_server_max = session_2.timeout_p2can_server_max)
    And (session_1.timeout_enhanced_p2can_server_max = session_2.
    timeout_enhanced_p2can_server_max) And
    (session_1.s3_client_ms = session_2.s3_client_ms) Then

    res := True;

  result := res;
End;

/// <summary>Entry point of the program, start a UDS channel, ask ECureset service and set new session info</summary>
var

  buffer: array [0 .. 255] of AnsiChar;
  status: uds_status;
  tp_handle: cantp_handle;
  config_physical: uds_msgconfig;
  msg_request: uds_msg;
  request_confirmation: uds_msg;
  response: uds_msg;
  new_sessioninfo: uds_sessioninfo;
  read_sessioninfo: uds_sessioninfo;
  dummy: char;
  FORCE_CUSTOM_SESSION: Bool;

begin
  try
    FORCE_CUSTOM_SESSION := True;

    // Initialize variables
    FillChar(msg_request, sizeof(msg_request), 0);
    FillChar(request_confirmation, sizeof(request_confirmation), 0);
    FillChar(response, sizeof(response), 0);
    buffer[0] := #0;

    // TODO: modify the value according to your available PCAN devices.
    tp_handle := cantp_handle.PCANTP_HANDLE_USBBUS1;

    // Print version informations
    status := TUDSApi.GetValue_2013(cantp_handle.PCANTP_HANDLE_NONEBUS,
      uds_parameter.PUDS_PARAMETER_API_VERSION, buffer, 256);
    Writeln(Format('PCAN-UDS API Version - %s: %s',
      [buffer, STATUS_OK_KO(status)]));

    // Initialize client
    status := TUDSApi.Initialize_2013(tp_handle,
      cantp_baudrate.PCANTP_BAUDRATE_500K);
    Writeln(Format('Initialize channel: %s', [STATUS_OK_KO(status)]));

    // Initialize a physical configuration
    FillChar(config_physical, sizeof(config_physical), 0);
    config_physical.can_id :=
      UInt32(uds_can_id.PUDS_CAN_ID_ISO_15765_4_PHYSICAL_REQUEST_1);
    config_physical.can_msgtype :=
      cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD;
    config_physical.nai.protocol :=
      uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL;
    config_physical.nai.target_type :=
      cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;
    config_physical.typem := uds_msgtype.PUDS_MSGTYPE_USDT;
    config_physical.nai.source_addr :=
      UInt16(uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT);
    config_physical.nai.target_addr :=
      UInt16(uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_1);
    config_physical.nai.extension_addr := 0;

    // Execute ECUReset and wait response
    status := TUDSApi.SvcECUReset_2013(tp_handle, config_physical, msg_request,
      uds_svc_param_er.PUDS_SVC_PARAM_ER_HR);
    Writeln(Format('Execute ECUReset service: %s', [STATUS_OK_KO(status)]));
    status := TUDSApi.WaitForService_2013(tp_handle, @msg_request, response,
      @request_confirmation);
    Writeln(Format('Wait for service: %s', [STATUS_OK_KO(status)]));

    if FORCE_CUSTOM_SESSION then
    begin

      // Some custom ECU may be in a non default session after reset
      // to inform the UDS API, it is possible to override the current session information:

      // Initialize new session info
      new_sessioninfo.can_msg_type :=
        cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD;
      new_sessioninfo.session_type := $5;
      new_sessioninfo.timeout_p2can_server_max := 1;
      new_sessioninfo.timeout_enhanced_p2can_server_max := 2;
      new_sessioninfo.s3_client_ms := 8;
      new_sessioninfo.nai := config_physical.nai;

      // Set new session info
      status := TUDSApi.SetValue_2013(tp_handle,
        uds_parameter.PUDS_PARAMETER_SESSION_INFO, Pointer(@new_sessioninfo),
        UInt32(sizeof(new_sessioninfo)));
      Writeln(Format('Set new session info: %s', [STATUS_OK_KO(status)]));

      // Get session info parameter and check if the new value is added
      Fillchar(read_sessioninfo, sizeof(read_sessioninfo), 0);
      read_sessioninfo.nai := config_physical.nai;
      status := TUDSApi.GetSessionInformation_2013(tp_handle, read_sessioninfo);
      Writeln(Format('Get session info: %s', [STATUS_OK_KO(status)]));
      Writeln(Format('Check added session info: %s',
        [OK_KO(is_same_uds_sessioninfo(@read_sessioninfo, @new_sessioninfo))]));
    end;

    // Free message structures
    status := TUDSApi.MsgFree_2013(&msg_request);
    Writeln(Format('Free request message: %s', [STATUS_OK_KO(status)]));
    status := TUDSApi.MsgFree_2013(&request_confirmation);
    Writeln(Format('Free request confirmation: %s', [STATUS_OK_KO(status)]));
    status := TUDSApi.MsgFree_2013(&response);
    Writeln(Format('Free response message: %s', [STATUS_OK_KO(status)]));

    // Close client
    status := TUDSApi.Uninitialize_2013(tp_handle);
    Writeln(Format('Uninitialize channel: %s', [STATUS_OK_KO(status)]));

    // Exit
    Writeln('Press any key to continue...');
    Readln(dummy);

  except
    on E: Exception do
      Writeln(E.Classname, ': ', E.Message);
  end;

end.
