program server_response_ecu_reset_UUDT;

{$APPTYPE CONSOLE}

uses
  SysUtils,
  ShellApi,
  windows,
  PUDS_2013 in 'PUDS_2013.pas',
  PCANBasic in 'PCANBasic.pas',
  PCANTP_2016 in 'PCANTP_2016.pas';

Function OK_KO(test: Boolean): String;
begin
  If test Then
    result := 'OK'
  else
    result := 'KO';
End;

Function STATUS_OK_KO(test: uds_status): String;
begin
  result := OK_KO(TUDSApi.StatusIsOk_2013(test));
End;

Function KeyPressed: Boolean;
Const
  buffer_size = 20;
Var
  zone: Array [1 .. buffer_size] Of TInputRecord;
  is_waiting: Cardinal;
  i: Cardinal;
  han: THandle;
Begin
  han := GetStdHandle(STD_INPUT_HANDLE);
  PeekConsoleInput(han, zone[1], buffer_size, is_waiting);
  result := false;
  i := 1;
  While Not result And (i <= is_waiting) And (i <= buffer_size) Do
  Begin
    result := (zone[i].EventType = 1) And (zone[i].Event.KeyEvent.bKeyDown) And
      (zone[i].Event.KeyEvent.AsciiChar <> #0);
    inc(i);
  End;
  If (is_waiting <> 0) And Not result Then
    ReadConsoleInput(STD_INPUT_HANDLE, zone[1], is_waiting, i);
End;

Function ReadKey: AnsiChar;
Var
  zone: TInputRecord;
  nb_read: Cardinal;
  han: THandle;
Begin
  han := GetStdHandle(STD_INPUT_HANDLE);
  result := #0;
  Repeat
    ReadConsoleInput(han, zone, 1, nb_read);
    If (nb_read = 1) And (zone.EventType = 1) And (zone.Event.KeyEvent.bKeyDown) And
      (zone.Event.KeyEvent.AsciiChar <> #0) Then
      result := zone.Event.KeyEvent.AsciiChar;
  Until result <> #0;
End;

/// <summary>
/// Entry point of the program, start a small server wich only support ECUReset service.
/// This server listen only 0x123 can id and respond with 0x124 can id in UUDT mode.
/// </summary>
var
  status: uds_status;
  tp_handle: cantp_handle;
  server_address: UInt32;
  buffer: array [0 .. 255] of AnsiChar;
  config_physical: uds_msgconfig;
  stopit: Boolean;
  wait_result: UInt32;
  read_status: uds_status;
  request_msg: uds_msg;
  response_msg: uds_msg;
  receive_event, null_handle: THandle;
  req_any: cantp_msgdata;
  req_srv_id: Byte;
  param0: Byte;
  keyboard_res: AnsiChar;
  data_pointer: PByte;
  dummy: char;

begin
  try
    // Initialize variables
    FillChar(request_msg, sizeof(request_msg), 0);
    FillChar(response_msg, sizeof(response_msg), 0);
    buffer[0] := #0;

    // TODO: modify the value according to your available PCAN devices.
    tp_handle := cantp_handle.PCANTP_HANDLE_USBBUS2;

    server_address := UInt32(uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_1);

    // Print version informations
    status := TUDSApi.GetValue_2013(cantp_handle.PCANTP_HANDLE_NONEBUS,
      uds_parameter.PUDS_PARAMETER_API_VERSION, buffer, 256);
    Writeln(Format('PCAN-UDS API Version - %s: %s',
      [buffer, STATUS_OK_KO(status)]));

    // Initialize server
    status := TUDSApi.Initialize_2013(tp_handle,
      cantp_baudrate.PCANTP_BAUDRATE_500K);
    Writeln(Format('Initialize channel: %s', [STATUS_OK_KO(status)]));

    // Set server address parameter
    status := TUDSApi.SetValue_2013(tp_handle,
      uds_parameter.PUDS_PARAMETER_SERVER_ADDRESS, PLongWord(@server_address),
      SizeOf(server_address));
    Writeln(Format('Set server address: %s', [STATUS_OK_KO(status)]));

    // Set a receive event

    receive_event := CreateEvent(nil, false, false, '');
    status := TUDSApi.SetValue_2013(tp_handle,
      uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, PByte(@receive_event),
      SizeOf(receive_event));

    Writeln(Format('Set receive event parameter: %s', [STATUS_OK_KO(status)]));

    // Add filter on 0x123 and 0x124 can id
    status := TUDSApi.AddCanIdFilter_2013(tp_handle, $123);
    Writeln(Format('Add can id filter (0x123): %s', [STATUS_OK_KO(status)]));
    status := TUDSApi.AddCanIdFilter_2013(tp_handle, $124);
    Writeln(Format('Add can id filter (0x124): %s', [STATUS_OK_KO(status)]));

    // Initialize the response configuration
    FillChar(config_physical, sizeof(config_physical), 0);
    config_physical.can_id := $124;
    config_physical.can_msgtype :=
      cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD;
    config_physical.nai.protocol :=
      uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL;
    config_physical.nai.target_type :=
      cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;;
    config_physical.typem := uds_msgtype.PUDS_MSGTYPE_UUDT;

    // Read while user do not press Q
    Writeln('Start listening, press Q to quit.');
    stopit := false;

    repeat

      // Wait a receive event on receiver
      // note: timeout is used to check keyboard hit.
      wait_result := WaitForSingleObject(receive_event, 1000);

      // If we get a receive event
      If wait_result = WAIT_OBJECT_0 Then
      begin
        repeat

          // Read first available message (no filtering based on message's type is set):
          read_status := TUDSApi.Read_2013(tp_handle, request_msg);
          Writeln(Format('Try to read a message: %s',
            [STATUS_OK_KO(read_status)]));

          If TUDSApi.StatusIsOk_2013(read_status) Then
          begin
            Fillchar(req_any, SizeOf(cantp_msgdata), 0);
            If request_msg.msg.msgdata_any <> nil Then
              req_any := request_msg.msg.msgdata_any^;

            // We receive a request, check its length and if it is not a loopback message and if it is a UUDT message
            If ((UInt32(request_msg.typem) And
              UInt32(uds_msgtype.PUDS_MSGTYPE_UUDT))
              = UInt32(uds_msgtype.PUDS_MSGTYPE_UUDT)) And (req_any.length >= 1)
              And ((UInt32(req_any.flags) And
              UInt32(cantp_msgflag.PCANTP_MSGFLAG_LOOPBACK)) = 0) Then
            begin
              req_srv_id := request_msg.links.service_id^;

              // This is a valid request, switch services
              Case req_srv_id of
                Byte(uds_service.PUDS_SERVICE_SI_ECUReset):
                  begin
                    // Allocate response message
                    param0 := request_msg.links.param^;

                    If param0 = Byte
                      (uds_svc_param_er.PUDS_SVC_PARAM_ER_ERPSD) Then
                    begin
                      status := TUDSApi.MsgAlloc_2013(response_msg,
                        config_physical, 3);
                      data_pointer := PByte(response_msg.links.param);
                      inc(data_pointer);
                      data_pointer^ := $66; // power down time
                    end
                    Else
                      status := TUDSApi.MsgAlloc_2013(response_msg,
                        config_physical, 2);

                    Writeln(Format
                      ('Prepare response message for ECUReset service: %s',
                      [STATUS_OK_KO(status)]));

                    if TUDSApi.StatusIsOk_2013(status) Then
                    begin
                      // Fill parameters
                      response_msg.links.service_id^ :=
                        Byte(uds_service.PUDS_SERVICE_SI_ECUReset) +
                        TUDSApi.PUDS_SI_POSITIVE_RESPONSE;
                      param0 := request_msg.links.param^;
                      response_msg.links.param^ := param0;

                      // Write response message
                      status := TUDSApi.Write_2013(tp_handle, @response_msg);
                      Writeln(Format
                        ('Write response message for ECUReset service: %s',
                        [STATUS_OK_KO(status)]));
                    end;

                    // Free response message (and clean memory in order to reallocate later)
                    status := TUDSApi.MsgFree_2013(response_msg);
                    Writeln(Format('Free response message: %s',
                      [STATUS_OK_KO(status)]));
                  end
              Else
                Writeln(Format('Unknown service (0x%x)', [req_srv_id]));
              End;
            end;
          end;
          // Free request message (and clean memory in order to reallocate later)
          status := TUDSApi.MsgFree_2013(request_msg);
          Writeln(Format('Free request message: %s', [STATUS_OK_KO(status)]));
        until TUDSApi.StatusIsOk_2013(read_status, uds_status.PUDS_STATUS_NO_MESSAGE);
      end;

      // Quit when user press Q
      If KeyPressed() Then
      begin
        keyboard_res := ReadKey();
        If (keyboard_res = 'Q') Or (keyboard_res = 'q') Then
          stopit := true;
      end;
    until stopit = true;

    // Close receive event
    null_handle := 0;
    status := TUDSApi.SetValue_2013(tp_handle,
      uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, PByte(@null_handle),
      SizeOf(null_handle));
    Writeln(Format('Stop receive event: %s', [STATUS_OK_KO(status)]));
    CloseHandle(receive_event);
    Writeln(Format('Close receive event: %s', [STATUS_OK_KO(status)]));

    // Close server
    status := TUDSApi.Uninitialize_2013(tp_handle);
    Writeln(Format('Uninitialize channel: %s', [STATUS_OK_KO(status)]));

    // Exit
    Writeln('Press any key to continue...');
    Readln(dummy);
  except
    on E: Exception do
      Writeln(E.Classname, ': ', E.Message);
  end;
end.
