program server_simulator;

{$APPTYPE CONSOLE}

uses
  SysUtils,
  ShellApi,
  windows,
  PUDS_2013 in 'PUDS_2013.pas',
  PCANBasic in 'PCANBasic.pas',
  PCANTP_2016 in 'PCANTP_2016.pas';

Function OK_KO(test: Boolean): String;
begin
  If test Then
    result := 'OK'
  else
    result := 'KO';
End;

Function STATUS_OK_KO(test: uds_status): String;
begin
  result := OK_KO(TUDSApi.StatusIsOk_2013(test));
End;

/// <summary>Helper: convert cantp network address information into uds network information</summary>
/// <param name="nai">cantp network address information structure</param>
/// <param name="can_msgtype">cantp message type</param>
/// <returns>UDS network address information structure</returns>
Function ConvertNai(nai: cantp_netaddrinfo; can_msgtype: cantp_can_msgtype)
  : uds_netaddrinfo;
var
  results: uds_netaddrinfo;
  is_29bits: Boolean;
begin
  if (UInt32(can_msgtype) And
    UInt32(cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED))
    = UInt32(cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED) then
    is_29bits := true
  else
    is_29bits := false;

  results.source_addr := nai.source_addr;
  results.target_addr := nai.target_addr;
  results.target_type := nai.target_type;
  results.extension_addr := nai.extension_addr;

  Case nai.format of
    cantp_isotp_format.PCANTP_ISOTP_FORMAT_ENHANCED:
      results.protocol :=
        uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_3_29B_ENHANCED;

    cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED:
      If is_29bits then
        results.protocol :=
          uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_29B_EXTENDED
      else
        results.protocol :=
          uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_11B_EXTENDED;

    cantp_isotp_format.PCANTP_ISOTP_FORMAT_FIXED_NORMAL:
      results.protocol :=
        uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_29B_FIXED_NORMAL;

    cantp_isotp_format.PCANTP_ISOTP_FORMAT_MIXED:
      If is_29bits then
        results.protocol :=
          uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_29B_REMOTE
      else
        results.protocol :=
          uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_11B_REMOTE;

    cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL:
      If is_29bits then
        results.protocol :=
          uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_29B_NORMAL
      else
        results.protocol :=
          uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL;
  Else
    results.protocol := uds_msgprotocol.PUDS_MSGPROTOCOL_NONE;
  End;

  result := results;
End;

/// <summary>Helper: check if a service has a subfunction</summary>
/// <param name="service_identifier">service identifier</param>
/// <returns>boolean: yes (true) or not (false)</returns>
Function HasSubFunction(service_identifier: Byte): Boolean;
var
  res: Boolean;
  service_id: uds_service;
begin
  service_id := uds_service(service_identifier);
  Case service_id of
    uds_service.PUDS_SERVICE_SI_DiagnosticSessionControl:
      res := true;
    uds_service.PUDS_SERVICE_SI_ECUReset:
      res := true;
    uds_service.PUDS_SERVICE_SI_SecurityAccess:
      res := true;
    uds_service.PUDS_SERVICE_SI_CommunicationControl:
      res := true;
    uds_service.PUDS_SERVICE_SI_TesterPresent:
      res := true;
    uds_service.PUDS_SERVICE_SI_AccessTimingParameter:
      res := true;
    uds_service.PUDS_SERVICE_SI_SecuredDataTransmission:
      res := false;
    uds_service.PUDS_SERVICE_SI_ControlDTCSetting:
      res := true;
    uds_service.PUDS_SERVICE_SI_ResponseOnEvent:
      res := true;
    uds_service.PUDS_SERVICE_SI_LinkControl:
      res := true;
    uds_service.PUDS_SERVICE_SI_ReadDataByIdentifier:
      res := false;
    uds_service.PUDS_SERVICE_SI_ReadMemoryByAddress:
      res := false;
    uds_service.PUDS_SERVICE_SI_ReadScalingDataByIdentifier:
      res := false;
    uds_service.PUDS_SERVICE_SI_ReadDataByPeriodicIdentifier:
      res := false;
    uds_service.PUDS_SERVICE_SI_DynamicallyDefineDataIdentifier:
      res := true;
    uds_service.PUDS_SERVICE_SI_WriteDataByIdentifier:
      res := false;
    uds_service.PUDS_SERVICE_SI_WriteMemoryByAddress:
      res := false;
    uds_service.PUDS_SERVICE_SI_ClearDiagnosticInformation:
      res := false;
    uds_service.PUDS_SERVICE_SI_ReadDTCInformation:
      res := true;
    uds_service.PUDS_SERVICE_SI_InputOutputControlByIdentifier:
      res := false;
    uds_service.PUDS_SERVICE_SI_RoutineControl:
      res := true;
    uds_service.PUDS_SERVICE_SI_RequestDownload:
      res := false;
    uds_service.PUDS_SERVICE_SI_RequestUpload:
      res := false;
    uds_service.PUDS_SERVICE_SI_TransferData:
      res := false;
    uds_service.PUDS_SERVICE_SI_RequestTransferExit:
      res := false;
    uds_service.PUDS_SERVICE_SI_RequestFileTransfer:
      res := false;
    uds_service.PUDS_SERVICE_SI_Authentication:
      res := true;
  Else
    res := false;
  End;

  result := res;
End;

/// <summary>
/// Create the ECUReset response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing ECUReset parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
Function create_SvcECUReset_response(config: uds_msgconfig;
  var request_msg: uds_msg; var response_msg: uds_msg): uds_status;
var
  status: uds_status;
  param0: Byte;
  data_pointer: PByte;
begin

  // Allocate response message
  param0 := request_msg.links.param^;

  If param0 = Byte(uds_svc_param_er.PUDS_SVC_PARAM_ER_ERPSD) Then
  begin
    status := TUDSApi.MsgAlloc_2013(response_msg, config, 3);
    data_pointer := PByte(response_msg.links.param);
    inc(data_pointer);
    data_pointer^ := $66; // Power down time
  end
  Else
    status := TUDSApi.MsgAlloc_2013(response_msg, config, 2);

  Writeln(format('Allocate ECUReset response: %s', [STATUS_OK_KO(status)]));

  if TUDSApi.StatusIsOk_2013(status) then
  begin
    // Fill parameters
    response_msg.links.service_id^ := Byte(uds_service.PUDS_SERVICE_SI_ECUReset)
      + Byte(TUDSApi.PUDS_SI_POSITIVE_RESPONSE);
    response_msg.links.param^ := param0;
  end;

  result := status;
End;

/// <summary>
/// Create the WriteMemoryByAddress response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
Function create_SvcWriteMemoryByAddress_response(config: uds_msgconfig;
  var request_msg: uds_msg; var response_msg: uds_msg): uds_status;
var
  status: uds_status;
  memory_size_size: UInt32;
  memory_address_size: UInt32;
  length: integer;
  param0: Byte;
  req_data_pointer, resp_data_pointer: PByte;
  i: UInt32;
begin

  // Read memory_size_size & memory_address_size
  param0 := request_msg.links.param^;
  memory_size_size := UInt32(param0 Shr 4 And $F);
  memory_address_size := UInt32(param0 And $F);
  length := 2 + memory_size_size + memory_address_size;

  // Allocate message
  status := TUDSApi.MsgAlloc_2013(response_msg, config, length);
  Writeln(format('Allocate WriteMemoryByAddress response: %s',
    [STATUS_OK_KO(status)]));

  if TUDSApi.StatusIsOk_2013(status) then
  begin
    // Copy address and memory parameters
    response_msg.links.param^ := param0;

    req_data_pointer := PByte(request_msg.links.param);
    resp_data_pointer := PByte(response_msg.links.param);
    inc(req_data_pointer);
    inc(resp_data_pointer);
    i := 0;
    While i <= memory_address_size - 1 do
    begin
      resp_data_pointer^ := req_data_pointer^;
      inc(req_data_pointer);
      inc(resp_data_pointer);
      i := i + 1;
    end;

    i := 0;
    while i <= memory_size_size - 1 do
    begin
      resp_data_pointer^ := req_data_pointer^;
      inc(req_data_pointer);
      inc(resp_data_pointer);
      i := i + 1;
    end;
    response_msg.links.service_id^ :=
      Byte(uds_service.PUDS_SERVICE_SI_WriteMemoryByAddress) +
      Byte(TUDSApi.PUDS_SI_POSITIVE_RESPONSE);
  end;

  result := status;
End;

/// <summary>
/// Create the DynamicallyDefineDataIdentifier response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
Function create_SvcDynamicallyDefineDataIdentifier_response
  (config: uds_msgconfig; var request_msg: uds_msg; var response_msg: uds_msg)
  : uds_status;
var
  status: uds_status;
  req_data_pointer, resp_data_pointer: PByte;

begin
  // Allocate and fill message
  if request_msg.msg.msgdata_any.length >= 4 then
  begin
    status := TUDSApi.MsgAlloc_2013(response_msg, config, 4);
    Writeln(format('Allocate DynamicallyDefineDataIdentifier response: %s',
      [STATUS_OK_KO(status)]));

    if TUDSApi.StatusIsOk_2013(status) then
    begin
      // Copy data
      req_data_pointer := PByte(request_msg.links.param);
      resp_data_pointer := PByte(response_msg.links.param);
      resp_data_pointer^ := req_data_pointer^;
      inc(req_data_pointer);
      inc(resp_data_pointer);
      resp_data_pointer^ := req_data_pointer^;
      inc(req_data_pointer);
      inc(resp_data_pointer);
      resp_data_pointer^ := req_data_pointer^;
    end;
  end
  else
  begin
    // Clear all data identifier request
    status := TUDSApi.MsgAlloc_2013(response_msg, config, 2);
    Writeln(format('Allocate DynamicallyDefineDataIdentifier response: %s',
      [STATUS_OK_KO(status)]));

    if TUDSApi.StatusIsOk_2013(status) then
    begin
      // Copy subfunction
      req_data_pointer := PByte(request_msg.links.param);
      resp_data_pointer := PByte(response_msg.links.param);
      resp_data_pointer^ := req_data_pointer^;
    end;
  end;

  if TUDSApi.StatusIsOk_2013(status) then
  begin
    response_msg.links.service_id^ :=
      Byte(uds_service.PUDS_SERVICE_SI_DynamicallyDefineDataIdentifier) +
      Byte(TUDSApi.PUDS_SI_POSITIVE_RESPONSE);
  end;

  result := status;
End;

/// <summary>
/// Create the dummy response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
Function create_dummy_response(config: uds_msgconfig; var request_msg: uds_msg;
  var response_msg: uds_msg): uds_status;
var
  status: uds_status;
  length: integer;
  service_id: Byte;
  i: integer;
  data_pointer: PByte;
begin

  service_id := request_msg.links.service_id^;

  // Set a random message size
  length := 1 + service_id + Random(200);

  If length > TCanTpApi.PCANTP_MAX_LENGTH_ISOTP2004 then
    length := TCanTpApi.PCANTP_MAX_LENGTH_ISOTP2004;

  // Allocate response message
  status := TUDSApi.MsgAlloc_2013(response_msg, config, length);
  Writeln(format('Allocate dummy response: %s', [STATUS_OK_KO(status)]));

  if TUDSApi.StatusIsOk_2013(status) then
  begin
    // Fake a positive response
    response_msg.links.service_id^ := service_id +
      TUDSApi.PUDS_SI_POSITIVE_RESPONSE;

    // Fill with dummy data
    response_msg.links.param^ := service_id;
    data_pointer := PByte(response_msg.links.param);
    inc(data_pointer);
    i := 1;
    while i <= length - 2 do
    // (length - 1) as positive response SID uses 1 byte
    begin
      data_pointer^ := i + 1;
      inc(data_pointer);
      i := i + 1;
    end;

    // Do not set PUDS_FLAG_SUPPRESS_POSITIVE_RESPONSE flag when service has a subfunction
    If HasSubFunction(service_id) Then
    begin
      data_pointer := PByte(response_msg.msg.msgdata_isotp.data);
      inc(data_pointer);
      data_pointer^ := 0;
    End;
  end;

  result := status;
End;

/// <summary>
/// Create the RequestTransferExit response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
Function create_SvcRequestTransferExit_response(config: uds_msgconfig;
  var request_msg: uds_msg; var response_msg: uds_msg): uds_status;
var
  status: uds_status;
  length: integer;
  i: integer;
  resp_data_pointer: PByte;
begin

  // Allocate message
  length := 1 + 1 + Random(50);
  status := TUDSApi.MsgAlloc_2013(response_msg, config, length);
  Writeln(format('Allocate RequestTransferExit response: %s',
    [STATUS_OK_KO(status)]));

  if TUDSApi.StatusIsOk_2013(status) then
  begin
    // Fill with dummy data
    resp_data_pointer := PByte(response_msg.links.param);
    i := 0;
    while i <= length - 2 do
    // (length - 1) as positive response SID uses 1 byte
    begin
      resp_data_pointer^ := i + 1;
      inc(resp_data_pointer);
      i := i + 1;
    end;

    response_msg.links.service_id^ :=
      Byte(uds_service.PUDS_SERVICE_SI_RequestTransferExit) +
      Byte(TUDSApi.PUDS_SI_POSITIVE_RESPONSE);
  end;

  result := status;
End;

/// <summary>
/// Create the SvcRequestUpload response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
Function create_SvcRequestUpload_response(config: uds_msgconfig;
  var request_msg: uds_msg; var response_msg: uds_msg): uds_status;
var
  status: uds_status;
  i: integer;
  length: integer;
  resp_data_pointer: PByte;
begin

  // Allocate message
  length := 2 + 1 + Random(50);
  status := TUDSApi.MsgAlloc_2013(response_msg, config, length);
  Writeln(format('Allocate RequestUpload response: %s',
    [STATUS_OK_KO(status)]));

  if TUDSApi.StatusIsOk_2013(status) then
  begin
    // Max number of block length = 0xF
    response_msg.links.param^ := $F0;

    // Fill with dummy data
    resp_data_pointer := PByte(response_msg.links.param);
    inc(resp_data_pointer);
    i := 1;
    while i <= length - 2 do
    // (length - 1) as positive response SID uses 1 byte
    begin
      resp_data_pointer^ := i + 1;
      inc(resp_data_pointer);
      i := i + 1;
    end;

    response_msg.links.service_id^ :=
      Byte(uds_service.PUDS_SERVICE_SI_RequestUpload) +
      Byte(TUDSApi.PUDS_SI_POSITIVE_RESPONSE);
  end;

  result := status;
End;

/// <summary>
/// Create the SvcRequestDownload response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
Function create_SvcRequestDownload_response(config: uds_msgconfig;
  var request_msg: uds_msg; var response_msg: uds_msg): uds_status;
var
  status: uds_status;
  i: integer;
  length: integer;
  resp_data_pointer: PByte;
begin

  // Allocate message

  length := 2 + 1 + Random(50);
  status := TUDSApi.MsgAlloc_2013(response_msg, config, length);
  Writeln(format('Allocate RequestDownload response: %s',
    [STATUS_OK_KO(status)]));

  if TUDSApi.StatusIsOk_2013(status) then
  begin
    // Max number of block length = 0xF
    response_msg.links.param^ := $F0;

    // Fill with dummy data
    resp_data_pointer := PByte(response_msg.links.param);
    inc(resp_data_pointer);
    i := 1;
    while i <= length - 2 do
    // (length - 1) as positive response SID uses 1 byte
    begin
      resp_data_pointer^ := i + 1;
      inc(resp_data_pointer);
      i := i + 1;
    end;

    response_msg.links.service_id^ :=
      Byte(uds_service.PUDS_SERVICE_SI_RequestDownload) +
      Byte(TUDSApi.PUDS_SI_POSITIVE_RESPONSE);
  end;

  result := status;
End;

/// <summary>
/// Create the RoutineControl response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
Function create_SvcRoutineControl_response(config: uds_msgconfig;
  var request_msg: uds_msg; var response_msg: uds_msg): uds_status;
var
  status: uds_status;
  req_data_pointer, resp_data_pointer: PByte;
begin

  // Allocate response
  status := TUDSApi.MsgAlloc_2013(response_msg, config, 4);
  Writeln(format('Allocate RoutineControl response: %s',
    [STATUS_OK_KO(status)]));

  if TUDSApi.StatusIsOk_2013(status) then
  begin
    // Copy data
    req_data_pointer := PByte(request_msg.links.param);
    resp_data_pointer := PByte(response_msg.links.param);
    resp_data_pointer^ := req_data_pointer^;
    inc(req_data_pointer);
    inc(resp_data_pointer);
    resp_data_pointer^ := req_data_pointer^;
    inc(req_data_pointer);
    inc(resp_data_pointer);
    resp_data_pointer^ := req_data_pointer^;

    // Routine status record not implemented

    response_msg.links.service_id^ :=
      Byte(uds_service.PUDS_SERVICE_SI_RoutineControl) +
      Byte(TUDSApi.PUDS_SI_POSITIVE_RESPONSE);
  end;

  result := status;
End;

/// <summary>
/// Create the InputOutputControlByIdentifier response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
Function create_SvcInputOutputControlByIdentifier_response
  (config: uds_msgconfig; var request_msg: uds_msg; var response_msg: uds_msg)
  : uds_status;
var
  status: uds_status;
  req_data_pointer, resp_data_pointer: PByte;
begin
  status := TUDSApi.MsgAlloc_2013(response_msg, config, 3);
  Writeln(format('Allocate InputOutputControlByIdentifier response: %s',
    [STATUS_OK_KO(status)]));

  if TUDSApi.StatusIsOk_2013(status) then
  begin
    // Copy data
    req_data_pointer := PByte(request_msg.links.param);
    resp_data_pointer := PByte(response_msg.links.param);

    resp_data_pointer^ := req_data_pointer^;
    inc(resp_data_pointer);
    inc(req_data_pointer);
    resp_data_pointer^ := req_data_pointer^;

    // Control status record not implemented
    response_msg.links.service_id^ :=
      Byte(uds_service.PUDS_SERVICE_SI_InputOutputControlByIdentifier) +
      Byte(TUDSApi.PUDS_SI_POSITIVE_RESPONSE);
  end;

  result := status;
End;

/// <summary>
/// Create the ClearDiagnosticInformation response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
Function create_SvcClearDiagnosticInformation_response(config: uds_msgconfig;
  var request_msg: uds_msg; var response_msg: uds_msg): uds_status;
var
  status: uds_status;
begin
  status := TUDSApi.MsgAlloc_2013(response_msg, config, 1);
  Writeln(format('Allocate ClearDiagnosticInformation response: %s',
    [STATUS_OK_KO(status)]));

  if TUDSApi.StatusIsOk_2013(status) then
  begin
    response_msg.links.service_id^ :=
      Byte(uds_service.PUDS_SERVICE_SI_ClearDiagnosticInformation) +
      Byte(TUDSApi.PUDS_SI_POSITIVE_RESPONSE);
  end;

  result := status;
End;

/// <summary>
/// Create the WriteDataByIdentifier response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
Function create_SvcWriteDataByIdentifier_response(config: uds_msgconfig;
  var request_msg: uds_msg; var response_msg: uds_msg): uds_status;
var
  status: uds_status;
  req_data_pointer, resp_data_pointer: PByte;
begin
  status := TUDSApi.MsgAlloc_2013(response_msg, config, 3);
  Writeln(format('Allocate WriteDataByIdentifier response: %s',
    [STATUS_OK_KO(status)]));

  if TUDSApi.StatusIsOk_2013(status) then
  begin
    // Copy data
    req_data_pointer := PByte(request_msg.links.param);
    resp_data_pointer := PByte(response_msg.links.param);
    resp_data_pointer^ := req_data_pointer^;
    inc(req_data_pointer);
    inc(resp_data_pointer);
    resp_data_pointer^ := req_data_pointer^;
    response_msg.links.service_id^ :=
      Byte(uds_service.PUDS_SERVICE_SI_WriteDataByIdentifier) +
      Byte(TUDSApi.PUDS_SI_POSITIVE_RESPONSE);
  end;

  result := status;
End;

/// <summary>
/// Create the ReadDataByPeriodicIdentifier response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
Function create_SvcReadDataByPeriodicIdentifier_response(config: uds_msgconfig;
  var request_msg: uds_msg; var response_msg: uds_msg): uds_status;
var
  status: uds_status;
begin

  status := TUDSApi.MsgAlloc_2013(response_msg, config, 1);
  Writeln(format('Allocate ReadDataByPeriodicIdentifier response: %s',
    [STATUS_OK_KO(status)]));

  if TUDSApi.StatusIsOk_2013(status) then
  begin
    response_msg.links.service_id^ :=
      Byte(uds_service.PUDS_SERVICE_SI_ReadDataByPeriodicIdentifier) +
      Byte(TUDSApi.PUDS_SI_POSITIVE_RESPONSE);
  end;

  result := status;
End;

/// <summary>
/// Create the ReadMemoryByAddress response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
Function create_SvcReadMemoryByAddress_response(config: uds_msgconfig;
  var request_msg: uds_msg; var response_msg: uds_msg): uds_status;
var
  status: uds_status;
  length: integer;
  param: Byte;
  data_pointer: PByte;
  i: integer;
begin
  // Read memory_size_size = bits [7..4]
  data_pointer := PByte(request_msg.links.param);
  inc(data_pointer);
  param := data_pointer^;

  length := UInt32(1 + (param Shr 4 And $F));

  // Allocate message
  status := TUDSApi.MsgAlloc_2013(response_msg, config, length);
  Writeln(format('Allocate ReadMemoryByAddress response: %s',
    [STATUS_OK_KO(status)]));

  if TUDSApi.StatusIsOk_2013(status) then
  begin
    // Fill with dummy data
    data_pointer := PByte(response_msg.links.param);
    i := 0;
    while i <= length - 2 do
    // (length - 1) as positive response SID uses 1 byte
    begin
      data_pointer^ := i + 1;
      inc(data_pointer);
      i := i + 1;
    end;

    response_msg.links.service_id^ :=
      Byte(uds_service.PUDS_SERVICE_SI_ReadMemoryByAddress) +
      Byte(TUDSApi.PUDS_SI_POSITIVE_RESPONSE);
  end;

  result := status;
End;

/// <summary>
/// Create the ReadScalingDataByIdentifier response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
Function create_SvcReadScalingDataByIdentifier_response(config: uds_msgconfig;
  var request_msg: uds_msg; var response_msg: uds_msg): uds_status;
var
  status: uds_status;
  req_data_pointer, resp_data_pointer: PByte;
begin

  // Allocate message
  status := TUDSApi.MsgAlloc_2013(response_msg, config, 12);
  Writeln(format('Allocate ReadScalingDataByIdentifier response: %s',
    [STATUS_OK_KO(status)]));

  if TUDSApi.StatusIsOk_2013(status) then
  begin
    response_msg.links.service_id^ :=
      Byte(uds_service.PUDS_SERVICE_SI_ReadScalingDataByIdentifier) +
      Byte(TUDSApi.PUDS_SI_POSITIVE_RESPONSE);

    // Copy data
    req_data_pointer := PByte(request_msg.links.param);
    resp_data_pointer := PByte(response_msg.links.param);
    resp_data_pointer^ := req_data_pointer^;
    inc(req_data_pointer);
    inc(resp_data_pointer);
    resp_data_pointer^ := req_data_pointer^;
    inc(resp_data_pointer);

    // Create a formula Vehicule Speed = (0.75*x+30) km/h
    resp_data_pointer^ := $0 Shl 4 Or $1; // unsigned numeric of 1 Bytes)
    inc(resp_data_pointer);
    resp_data_pointer^ := $90; // formula, 0 data bytes
    inc(resp_data_pointer);
    resp_data_pointer^ := $0; // formulaIdentifier = C0 * x + C1
    inc(resp_data_pointer);
    resp_data_pointer^ := $E0; // C0 high byte
    inc(resp_data_pointer);
    resp_data_pointer^ := $4B; // C0 low byte
    inc(resp_data_pointer);
    resp_data_pointer^ := $0; // C1 high byte
    inc(resp_data_pointer);
    resp_data_pointer^ := $1E; // C1 low byte
    inc(resp_data_pointer);
    resp_data_pointer^ := $A0; // unit/format, 0 data bytes
    inc(resp_data_pointer);
    resp_data_pointer^ := $30; // unit ID, km/h
  end;

  result := status;
End;

/// <summary>
/// Create the ReadDataByIdentifier response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
Function create_SvcReadDataByIdentifier_response(config: uds_msgconfig;
  var request_msg: uds_msg; var response_msg: uds_msg): uds_status;
var
  status: uds_status;
  length: integer;

  req_data: cantp_msgdata;
  i, j: UInt32;
  req_data_pointer, resp_data_pointer: PByte;
begin

  If request_msg.msg.msgdata_any <> nil Then
    req_data := request_msg.msg.msgdata_any^;

  // Allocate message
  length := (req_data.length - 1) * 7 Div 2 + 1;
  status := TUDSApi.MsgAlloc_2013(response_msg, config, length);
  Writeln(format('Allocate ReadDataByIdentifier response: %s',
    [STATUS_OK_KO(status)]));;

  if TUDSApi.StatusIsOk_2013(status) then
  begin
    req_data_pointer := PByte(request_msg.links.param);
    resp_data_pointer := PByte(response_msg.links.param);
    i := 0;
    while i <= req_data.length - 2 do
    begin
      // Copy data
      resp_data_pointer^ := req_data_pointer^;
      inc(resp_data_pointer);
      inc(req_data_pointer);
      resp_data_pointer^ := req_data_pointer^;
      inc(resp_data_pointer);
      inc(req_data_pointer);

      // Data record: fill with dummy data
      For j := 0 To 4 do
      begin
        resp_data_pointer^ := Ord('A') + j;
        inc(resp_data_pointer);
      end;
      i := i + 2;
    end;

    response_msg.links.service_id^ :=
      Byte(uds_service.PUDS_SERVICE_SI_ReadDataByIdentifier) +
      Byte(TUDSApi.PUDS_SI_POSITIVE_RESPONSE);
  end;

  result := status;
End;

/// <summary>
/// Create the LinkControl response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
Function create_SvcLinkControl_response(config: uds_msgconfig;
  var request_msg: uds_msg; var response_msg: uds_msg): uds_status;
var
  status: uds_status;
begin
  status := TUDSApi.MsgAlloc_2013(response_msg, config, 2);
  Writeln(format('Allocate LinkControl response: %s', [STATUS_OK_KO(status)]));

  if TUDSApi.StatusIsOk_2013(status) then
  begin
    response_msg.links.param^ := request_msg.links.param^;
    response_msg.links.service_id^ :=
      Byte(uds_service.PUDS_SERVICE_SI_LinkControl) +
      Byte(TUDSApi.PUDS_SI_POSITIVE_RESPONSE);
  end;

  result := status;
End;

/// <summary>
/// Create the ResponseOnEvent response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
Function create_SvcResponseOnEvent_response(config: uds_msgconfig;
  var request_msg: uds_msg; var response_msg: uds_msg): uds_status;
var
  status: uds_status;
  req_data_pointer, resp_data_pointer: PByte;
  param0, param1: Byte;
begin
  req_data_pointer := PByte(request_msg.links.param);
  param0 := req_data_pointer^;
  inc(req_data_pointer);
  param1 := req_data_pointer^;

  If param0 = Byte(uds_svc_param_roe.PUDS_SVC_PARAM_ROE_RAE) Then
  begin
    status := TUDSApi.MsgAlloc_2013(response_msg, config, 3);
    Writeln(format('Allocate ResponseOnEvent response: %s',
      [STATUS_OK_KO(status)]));

    if TUDSApi.StatusIsOk_2013(status) then
    begin
      resp_data_pointer := PByte(response_msg.links.param);
      inc(resp_data_pointer);
      resp_data_pointer^ := 0; // Number of activated events
      // Event type and service to respond to records not implemented
    end;
  end
  Else
  begin
    status := TUDSApi.MsgAlloc_2013(response_msg, config, 4);
    Writeln(format('Allocate ResponseOnEvent response: %s',
      [STATUS_OK_KO(status)]));

    if TUDSApi.StatusIsOk_2013(status) then
    begin
      resp_data_pointer := PByte(response_msg.links.param);
      inc(resp_data_pointer);
      resp_data_pointer^ := 0; // Number of identified events
      inc(resp_data_pointer);
      resp_data_pointer^ := param1; // Event window time
      // Event type and service to respond to records not implemented
    end;
  End;

  if TUDSApi.StatusIsOk_2013(status) then
  begin
    response_msg.links.param^ := param0;
    response_msg.links.service_id^ :=
      Byte(uds_service.PUDS_SERVICE_SI_ResponseOnEvent) +
      Byte(TUDSApi.PUDS_SI_POSITIVE_RESPONSE);
  end;
  result := status;
End;

/// <summary>
/// Create the ControlDTCSetting response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
Function create_SvcControlDTCSetting_response(config: uds_msgconfig;
  var request_msg: uds_msg; var response_msg: uds_msg): uds_status;
var
  status: uds_status;
  param0: Byte;
begin
  status := TUDSApi.MsgAlloc_2013(response_msg, config, 2);
  Writeln(format('Allocate ControlDTCSetting response: %s',
    [STATUS_OK_KO(status)]));

  if TUDSApi.StatusIsOk_2013(status) then
  begin
    param0 := request_msg.links.param^;
    response_msg.links.param^ := param0;
    response_msg.links.service_id^ :=
      Byte(uds_service.PUDS_SERVICE_SI_ControlDTCSetting) +
      Byte(TUDSApi.PUDS_SI_POSITIVE_RESPONSE);
  end;
  result := status;
End;

/// <summary>
/// Create the SecuredDataTransmission response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
Function create_SvcSecuredDataTransmission_response(config: uds_msgconfig;
  var request_msg: uds_msg; var response_msg: uds_msg): uds_status;
var
  status: uds_status;
  i: integer;
  length: integer;
  resp_data_pointer: PByte;

begin

  // Allocate message
  length := 1 + Random(50);
  status := TUDSApi.MsgAlloc_2013(response_msg, config, length);
  Writeln(format('Allocate SecuredDataTransmission response: %s',
    [STATUS_OK_KO(status)]));

  if TUDSApi.StatusIsOk_2013(status) then
  begin
    // Fill with dummy data (check Security-SubLayer record defined in ISO-15764)
    resp_data_pointer := PByte(response_msg.links.param);

    i := 0;
    while i <= length - 2 do
    // (length - 1) as positive response SID uses 1 byte
    begin
      resp_data_pointer^ := i + 1;
      inc(resp_data_pointer);
      i := i + 1;
    end;

    response_msg.links.service_id^ :=
      Byte(uds_service.PUDS_SERVICE_SI_SecuredDataTransmission) +
      Byte(TUDSApi.PUDS_SI_POSITIVE_RESPONSE);
  end;

  result := status;
End;

/// <summary>
/// Create the TesterPresent response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
Function create_SvcTesterPresent_response(config: uds_msgconfig;
  var request_msg: uds_msg; var response_msg: uds_msg): uds_status;
var
  status: uds_status;
  param0: Byte;
begin
  status := TUDSApi.MsgAlloc_2013(response_msg, config, 2);
  Writeln(format('Allocate TesterPresent response: %s',
    [STATUS_OK_KO(status)]));

  if TUDSApi.StatusIsOk_2013(status) then
  begin
    param0 := request_msg.links.param^;
    response_msg.links.param^ := param0;
    response_msg.links.service_id^ :=
      Byte(uds_service.PUDS_SERVICE_SI_TesterPresent) +
      Byte(TUDSApi.PUDS_SI_POSITIVE_RESPONSE);
  end;

  result := status;
End;

/// <summary>
/// Create the CommunicationControl response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
Function create_SvcCommunicationControl_response(config: uds_msgconfig;
  var request_msg: uds_msg; var response_msg: uds_msg): uds_status;
var
  status: uds_status;
  param0: Byte;
begin
  status := TUDSApi.MsgAlloc_2013(response_msg, config, 2);
  Writeln(format('Allocate CommunicationControl response: %s',
    [STATUS_OK_KO(status)]));

  if TUDSApi.StatusIsOk_2013(status) then
  begin
    param0 := request_msg.links.param^;
    response_msg.links.param^ := param0;
    response_msg.links.service_id^ :=
      Byte(uds_service.PUDS_SERVICE_SI_CommunicationControl) +
      Byte(TUDSApi.PUDS_SI_POSITIVE_RESPONSE);
  end;

  result := status;
End;

/// <summary>
/// Create the SecurityAccess response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
Function create_SvcSecurityAccess_response(config: uds_msgconfig;
  var request_msg: uds_msg; var response_msg: uds_msg): uds_status;
var
  status: uds_status;
  length: integer;
  i: integer;
  param0: Byte;
  resp_data_pointer: PByte;
begin

  // Allocate response message
  param0 := request_msg.links.param^;

  If (param0 >= TUDSApi.PUDS_SVC_PARAM_SA_RSD_MIN) And
    (param0 <= TUDSApi.PUDS_SVC_PARAM_SA_RSD_MAX) And (param0 Mod 2 = 1) Then
  begin
    // Request security seed are Even values
    // Fill with dummy data
    length := 1 + UInt32(uds_service.PUDS_SERVICE_SI_SecurityAccess);
    status := TUDSApi.MsgAlloc_2013(response_msg, config, length);
    Writeln(format('Allocate SecurityAccess response: %s',
      [STATUS_OK_KO(status)]));

    if TUDSApi.StatusIsOk_2013(status) then
    begin
      resp_data_pointer := PByte(response_msg.links.param);
      inc(resp_data_pointer);
      i := 1;
      while i <= length - 2 do
      // (length - 1) as positive response SID uses 1 byte
      begin
        resp_data_pointer^ := i + 1;
        inc(resp_data_pointer);
        i := i + 1;
      end
    end;
  end
  Else
  begin
    status := TUDSApi.MsgAlloc_2013(response_msg, config, 2);
    Writeln(format('Allocate SecurityAccess response: %s',
      [STATUS_OK_KO(status)]));
  End;

  if TUDSApi.StatusIsOk_2013(status) then
  begin
    response_msg.links.param^ := param0;
    response_msg.links.service_id^ :=
      Byte(uds_service.PUDS_SERVICE_SI_SecurityAccess) +
      Byte(TUDSApi.PUDS_SI_POSITIVE_RESPONSE);
  end;

  result := status;
End;

/// <summary>
/// Create the DiagnosticSessionControl response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
Function create_SvcDiagnosticSessionControl_response(config: uds_msgconfig;
  var request_msg: uds_msg; var response_msg: uds_msg): uds_status;
var
  status: uds_status;
  param0: Byte;
  resp_data_pointer: PByte;
begin

  // Allocate response message
  status := TUDSApi.MsgAlloc_2013(response_msg, config, 6);
  Writeln(format('Allocate DiagnosticSessionControl response: %s',
    [STATUS_OK_KO(status)]));

  if TUDSApi.StatusIsOk_2013(status) then
  begin
    // Fill response
    param0 := request_msg.links.param^;
    response_msg.links.param^ := param0;
    resp_data_pointer := PByte(response_msg.links.param);
    inc(resp_data_pointer);

    resp_data_pointer^ := $0; // P2Can_Server_Max = 0x0010
    inc(resp_data_pointer);

    resp_data_pointer^ := $10;
    inc(resp_data_pointer);
    resp_data_pointer^ := $3; // P2*Can_Server_Max = 0x03E8
    inc(resp_data_pointer);
    resp_data_pointer^ := $E8;
    response_msg.links.service_id^ :=
      Byte(uds_service.PUDS_SERVICE_SI_DiagnosticSessionControl) +
      Byte(TUDSApi.PUDS_SI_POSITIVE_RESPONSE);
  end;

  result := status;
End;

/// <summary>
/// Create the TransferData response message
/// </summary>
/// <param name="config">Input, UDS configuration</param>
/// <param name="request_msg">Input, received request, containing request parameters</param>
/// <param name="response_msg">Output, contains the generated response</param>
/// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
Function create_SvcTransferData_response(tp_handle: cantp_handle;
  config: uds_msgconfig; var request_msg: uds_msg; var response_msg: uds_msg)
  : uds_status;
var
  status: uds_status;
  length: integer;
  pending_response_msg: uds_msg;
  data_pointer: PByte;
  i: integer;
  req_data: cantp_msgdata_isotp;

begin
  FillChar(pending_response_msg, sizeof(pending_response_msg), 0);

  // Initialize locale struct
  If request_msg.msg.msgdata_isotp <> nil Then
    req_data := request_msg.msg.msgdata_isotp^;

  // Custom response to client_all_request example:
  // a. service is requested functionally,
  // b. 1st response is NRC response pending
  // c. wait
  // d. send correct response
  If req_data.netaddrinfo.target_type = cantp_isotp_addressing.
    PCANTP_ISOTP_ADDRESSING_FUNCTIONAL Then
  begin
    // Transmit a NRC response pending
    status := TUDSApi.MsgAlloc_2013(pending_response_msg, config, 3);
    Writeln(format('Allocate TransferData pending response: %s',
      [STATUS_OK_KO(status)]));

    // Transmit a NRC response pending
    if TUDSApi.StatusIsOk_2013(status) then
    begin
      pending_response_msg.links.service_id^ :=
        Byte(uds_service.PUDS_SERVICE_NR_SI);
      data_pointer := PByte(pending_response_msg.links.param);
      data_pointer^ := Byte(uds_service.PUDS_SERVICE_SI_TransferData);
      inc(data_pointer);
      data_pointer^ := TUDSApi.PUDS_NRC_EXTENDED_TIMING;
    end;
    status := TUDSApi.Write_2013(tp_handle, @pending_response_msg);
    Writeln('');
    Writeln(format('   ...Transmitting a NRC Response Pending message: %d',
      [integer(status)]));
    Writeln('');
    Write(format('   ...simulating computation... (waiting ~%dms)',
      [integer(TUDSApi.PUDS_P2CAN_ENHANCED_SERVER_MAX_DEFAULT)]));
    Sleep(integer(TUDSApi.PUDS_P2CAN_ENHANCED_SERVER_MAX_DEFAULT) - 100);

    // Reallocate response message
    status := TUDSApi.MsgFree_2013(pending_response_msg);
    Writeln(format('Free pending response: %s', [STATUS_OK_KO(status)]));

    // Initialize real service response
    config.typem := uds_msgtype.PUDS_MSGTYPE_USDT;
    length := TCanTpApi.PCANTP_MAX_LENGTH_ISOTP2004;
    status := TUDSApi.MsgAlloc_2013(response_msg, config, length);
    Writeln(format('Allocate TransferData real response: %s',
      [STATUS_OK_KO(status)]));
  end
  Else
  begin
    If 2 + req_data.length > TCanTpApi.PCANTP_MAX_LENGTH_ISOTP2004 then
      length := TCanTpApi.PCANTP_MAX_LENGTH_ISOTP2004
    else
      length := 2 + req_data.length;
    status := TUDSApi.MsgAlloc_2013(response_msg, config, length);
    Writeln(format('Allocate TransferData response: %s',
      [STATUS_OK_KO(status)]));
  end;

  if TUDSApi.StatusIsOk_2013(status) then
  begin
    response_msg.links.param^ := request_msg.links.param^;

    // Fill with dummy data
    data_pointer := PByte(response_msg.links.param);
    inc(data_pointer);
    i := 1;
    while i <= length - 2 do
    // (length - 1) as positive response SID uses 1 byte
    begin
      data_pointer^ := (i + 1) And $FF;
      inc(data_pointer);
      i := i + 1;
    end;

    response_msg.links.service_id^ :=
      Byte(uds_service.PUDS_SERVICE_SI_TransferData) +
      Byte(TUDSApi.PUDS_SI_POSITIVE_RESPONSE);
  end;

  result := status;
End;

/// <summary>
/// Process request and send a response
/// </summary>
/// <param name="tp_handle">CANTP channel</param>
/// <param name="server_address">Server address</param>
/// <param name="request_msg">Received request message</param>
procedure process_request(tp_handle: cantp_handle; server_address: UInt16;
  var request_msg: uds_msg);
var
  config: uds_msgconfig;
  response_msg: uds_msg;
  status: uds_status;
  req_isotp: cantp_msgdata_isotp;
  service_id: Byte;
  service_id_: uds_service;
begin

  FillChar(response_msg, sizeof(response_msg), 0);
  FillChar(config, sizeof(config), 0);

  // Initialize config from request message network address information
  If request_msg.msg.msgdata_isotp <> nil Then
    req_isotp := request_msg.msg.msgdata_isotp^;

  config.nai := ConvertNai(req_isotp.netaddrinfo,
    request_msg.msg.can_info.can_msgtype);
  config.can_msgtype := request_msg.msg.can_info.can_msgtype;
  config.typem := uds_msgtype.PUDS_MSGTYPE_USDT;
  config.nai.extension_addr := req_isotp.netaddrinfo.extension_addr;

  // Set target and source addresses
  If req_isotp.netaddrinfo.target_type = cantp_isotp_addressing.
    PCANTP_ISOTP_ADDRESSING_FUNCTIONAL Then
    // response to functional addressing is set to TEST_EQUIPMENT
    config.nai.target_addr :=
      UInt16(uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT)
  Else
    config.nai.target_addr := req_isotp.netaddrinfo.source_addr;

  config.nai.target_type :=
    cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;
  config.nai.source_addr := server_address;

  // This is a valid request, switch services
  service_id := request_msg.links.service_id^;
  service_id_ := uds_service(service_id);
  Case service_id_ of
    uds_service.PUDS_SERVICE_SI_DiagnosticSessionControl:
      status := create_SvcDiagnosticSessionControl_response(config, request_msg,
        response_msg);
    uds_service.PUDS_SERVICE_SI_SecurityAccess:
      status := create_SvcSecurityAccess_response(config, request_msg,
        response_msg);
    uds_service.PUDS_SERVICE_SI_CommunicationControl:
      status := create_SvcCommunicationControl_response(config, request_msg,
        response_msg);
    uds_service.PUDS_SERVICE_SI_TesterPresent:
      status := create_SvcTesterPresent_response(config, request_msg,
        response_msg);
    uds_service.PUDS_SERVICE_SI_SecuredDataTransmission:
      status := create_SvcSecuredDataTransmission_response(config, request_msg,
        response_msg);
    uds_service.PUDS_SERVICE_SI_ControlDTCSetting:
      status := create_SvcControlDTCSetting_response(config, request_msg,
        response_msg);
    uds_service.PUDS_SERVICE_SI_ResponseOnEvent:
      status := create_SvcResponseOnEvent_response(config, request_msg,
        response_msg);
    uds_service.PUDS_SERVICE_SI_LinkControl:
      status := create_SvcLinkControl_response(config, request_msg,
        response_msg);
    uds_service.PUDS_SERVICE_SI_ReadDataByIdentifier:
      status := create_SvcReadDataByIdentifier_response(config, request_msg,
        response_msg);
    uds_service.PUDS_SERVICE_SI_ReadMemoryByAddress:
      status := create_SvcReadMemoryByAddress_response(config, request_msg,
        response_msg);
    uds_service.PUDS_SERVICE_SI_ReadScalingDataByIdentifier:
      status := create_SvcReadScalingDataByIdentifier_response(config,
        request_msg, response_msg);
    uds_service.PUDS_SERVICE_SI_ReadDataByPeriodicIdentifier:
      status := create_SvcReadDataByPeriodicIdentifier_response(config,
        request_msg, response_msg);
    uds_service.PUDS_SERVICE_SI_DynamicallyDefineDataIdentifier:
      status := create_SvcDynamicallyDefineDataIdentifier_response(config,
        request_msg, response_msg);
    uds_service.PUDS_SERVICE_SI_WriteDataByIdentifier:
      status := create_SvcWriteDataByIdentifier_response(config, request_msg,
        response_msg);
    uds_service.PUDS_SERVICE_SI_WriteMemoryByAddress:
      status := create_SvcWriteMemoryByAddress_response(config, request_msg,
        response_msg);
    uds_service.PUDS_SERVICE_SI_ClearDiagnosticInformation:
      status := create_SvcClearDiagnosticInformation_response(config,
        request_msg, response_msg);
    uds_service.PUDS_SERVICE_SI_InputOutputControlByIdentifier:
      status := create_SvcInputOutputControlByIdentifier_response(config,
        request_msg, response_msg);
    uds_service.PUDS_SERVICE_SI_RoutineControl:
      status := create_SvcRoutineControl_response(config, request_msg,
        response_msg);
    uds_service.PUDS_SERVICE_SI_RequestDownload:
      status := create_SvcRequestDownload_response(config, request_msg,
        response_msg);
    uds_service.PUDS_SERVICE_SI_RequestUpload:
      status := create_SvcRequestUpload_response(config, request_msg,
        response_msg);
    uds_service.PUDS_SERVICE_SI_TransferData:
      status := create_SvcTransferData_response(tp_handle, config, request_msg,
        response_msg);
    uds_service.PUDS_SERVICE_SI_RequestTransferExit:
      status := create_SvcRequestTransferExit_response(config, request_msg,
        response_msg);
    uds_service.PUDS_SERVICE_SI_ECUReset:
      status := create_SvcECUReset_response(config, request_msg, response_msg);
    uds_service.PUDS_SERVICE_SI_ReadDTCInformation:
      begin
        // Not yet implemented
        Writeln(format('Unknown service (0x%x)', [service_id]));
        status := create_dummy_response(config, request_msg, response_msg);
      end;
    uds_service.PUDS_SERVICE_SI_AccessTimingParameter:
      begin
        // Not yet implemented
        Writeln(format('Unknown service (0x%x)', [service_id]));
        status := create_dummy_response(config, request_msg, response_msg);
      end;
    uds_service.PUDS_SERVICE_SI_RequestFileTransfer:
      begin
        // Not yet implemented
        Writeln(format('Unknown service (0x%x)', [service_id]));
        status := create_dummy_response(config, request_msg, response_msg);
      end;
    uds_service.PUDS_SERVICE_SI_Authentication:
      begin
        // Not yet implemented
        Writeln(format('Unknown service (0x%x)', [service_id]));
        status := create_dummy_response(config, request_msg, response_msg);
      end;
  Else
    begin
      Writeln(format('Unknown service (0x%x)', [service_id]));
      status := create_dummy_response(config, request_msg, response_msg);
    end;
  End;

  // Print allocation result
  Writeln(format('Allocate response message: %s', [STATUS_OK_KO(status)]));

  // Send response message
  if TUDSApi.StatusIsOk_2013(status) then
  begin
    status := TUDSApi.Write_2013(tp_handle, @response_msg);
    Writeln('');
    Writeln(format('   ...Transmitting response: %d', [integer(status)]));
  end;

  // Free response message
  status := TUDSApi.MsgFree_2013(response_msg);
  Writeln(format('Free response message: %s', [STATUS_OK_KO(status)]));
End;

/// <summary>Print an UDS message</summary>
/// <param name="msg">Message to print</param>
procedure display_message(msg: Puds_msg);
// A function that displays UDS messages
var
  msg_isotp: cantp_msgdata_isotp;
  s1, s2: String;
  val: Byte;
  i: integer;
  data_pointer: PByte;
begin

  If msg.msg.msgdata_isotp <> nil Then
    msg_isotp := msg.msg.msgdata_isotp^;

  If (UInt32(msg_isotp.netaddrinfo.msgtype) And
    UInt32(cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION)) = 0 Then
  begin
    If (UInt32(msg_isotp.flags) And
      UInt32(cantp_msgflag.PCANTP_MSGFLAG_LOOPBACK)) = 0 then
      s1 := 'Received UDS'
    else
      s1 := 'Received Loopback';
    If msg_isotp.netstatus <> cantp_netstatus.PCANTP_NETSTATUS_OK then
      s2 := 'PCANTP_NETSTATUS_OK !!!'
    else
      s2 := 'OK !';

    Writeln('');
    Writeln(format
      (' %s message from 0x%.2x (to 0x%.2x, with extension 0x%.2x) - result: %d - %s',
      [s1, integer(msg_isotp.netaddrinfo.source_addr),
      integer(msg_isotp.netaddrinfo.target_addr),
      integer(msg_isotp.netaddrinfo.extension_addr),
      integer(msg_isotp.netstatus), s2]));
    // Display data
    Write(format('  -> Length: %d, Data= ', [integer(msg_isotp.length)]));

    data_pointer := PByte(msg.msg.msgdata_isotp.data);
    For i := 0 To msg_isotp.length - 1 do
    begin
      val := data_pointer^;
      Write(format('%.2x ', [val]));
      inc(data_pointer);
    end;

    Writeln('');
  end
  Else
  begin
    Writeln('');
    Writeln(format
      (' PENDING UDS message from 0x%.2x (to 0x%.2x, with extension 0x%.2x) -> length=%d ...',
      [integer(msg_isotp.netaddrinfo.source_addr),
      integer(msg_isotp.netaddrinfo.target_addr),
      integer(msg_isotp.netaddrinfo.extension_addr),
      integer(msg_isotp.length)]));
  End;
End;

Function KeyPressed: Boolean;
Const
  buffer_size = 20;
Var
  zone: Array [1 .. buffer_size] Of TInputRecord;
  is_waiting: Cardinal;
  i: Cardinal;
  han: THandle;
Begin
  han := GetStdHandle(STD_INPUT_HANDLE);
  PeekConsoleInput(han, zone[1], buffer_size, is_waiting);
  result := false;
  i := 1;
  While Not result And (i <= is_waiting) And (i <= buffer_size) Do
  Begin
    result := (zone[i].EventType = 1) And (zone[i].Event.KeyEvent.bKeyDown) And
      (zone[i].Event.KeyEvent.AsciiChar <> #0);
    inc(i);
  End;
  If (is_waiting <> 0) And Not result Then
    ReadConsoleInput(STD_INPUT_HANDLE, zone[1], is_waiting, i);
End;

Function ReadKey: AnsiChar;
Var
  zone: TInputRecord;
  nb_read: Cardinal;
  han: THandle;
Begin
  han := GetStdHandle(STD_INPUT_HANDLE);
  result := #0;
  Repeat
    ReadConsoleInput(han, zone, 1, nb_read);
    If (nb_read = 1) And (zone.EventType = 1) And (zone.Event.KeyEvent.bKeyDown)
      And (zone.Event.KeyEvent.AsciiChar <> #0) Then
      result := zone.Event.KeyEvent.AsciiChar;
  Until result <> #0;
End;

/// <summary>Entry point of the program, start a CAN UDS server simulation</summary>
var
  buffer: array [0 .. 255] of AnsiChar;
  status: uds_status;
  tp_handle: cantp_handle;
  server_address: UInt32;
  stopit: Boolean;
  wait_result: UInt32;
  read_status: uds_status;
  request_msg: uds_msg;
  timeout_value: UInt32;
  receive_event, null_handle: THandle;
  req_isotp: cantp_msgdata_isotp;
  dummy: Char;
  keyboard_res: AnsiChar;

begin
  try
    // Initialize variables
    tp_handle := cantp_handle.PCANTP_HANDLE_USBBUS2;
    Randomize;
    FillChar(request_msg, sizeof(request_msg), #0);
    buffer[0] := #0;

    // TODO: modify the value according to your available PCAN devices.
    server_address := UInt32(uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_1);

    // Print version informations
    status := TUDSApi.GetValue_2013(cantp_handle.PCANTP_HANDLE_NONEBUS,
      uds_parameter.PUDS_PARAMETER_API_VERSION, buffer, 256);
    Writeln(format('PCAN-UDS API Version - %s: %s',
      [buffer, STATUS_OK_KO(status)]));

    // Initialize channel
    status := TUDSApi.Initialize_2013(tp_handle,
      cantp_baudrate.PCANTP_BAUDRATE_500K);
    Writeln(format('Initialize channel: %s', [STATUS_OK_KO(status)]));

    // Set server address parameter
    status := TUDSApi.SetValue_2013(tp_handle,
      uds_parameter.PUDS_PARAMETER_SERVER_ADDRESS, PLongWord(@server_address),
      UInt32(sizeof(UInt32)));
    Writeln(format('Set server address: %s', [STATUS_OK_KO(status)]));

    // Set timeout values
    timeout_value := TCanTpApi.PCANTP_ISO_TIMEOUTS_15765_4;
    status := TUDSApi.SetValue_2013(tp_handle,
      uds_parameter.PUDS_PARAMETER_ISO_TIMEOUTS, PLongWord(@timeout_value),
      UInt32(sizeof(UInt32)));
    Writeln(format('Set ISO 15765-4 timeouts values: %s',
      [STATUS_OK_KO(status)]));

    // Print timeout values
    status := TUDSApi.GetValue_2013(tp_handle,
      uds_parameter.PUDS_PARAMETER_TIMEOUT_REQUEST, PLongWord(@timeout_value),
      UInt32(sizeof(UInt32)));
    Writeln(format('Get request timeout value (%dms): %s',
      [timeout_value, STATUS_OK_KO(status)]));
    status := TUDSApi.GetValue_2013(tp_handle,
      uds_parameter.PUDS_PARAMETER_TIMEOUT_RESPONSE, PLongWord(@timeout_value),
      UInt32(sizeof(UInt32)));

    Writeln(format('Get response timeout value (%dms): %s',
      [timeout_value, STATUS_OK_KO(status)]));

    // Set a receive event
    receive_event := CreateEvent(nil, false, false, '');
    status := TUDSApi.SetValue_2013(tp_handle,
      uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, PByte(@receive_event),
      sizeof(receive_event));

    Writeln(format('Set receive event parameter: %s', [STATUS_OK_KO(status)]));

    // Read while user do not press Q
    Writeln('Start listening, press Q to quit.');
    stopit := false;

    While Not stopit do
    begin
      // Wait a receive event on receiver
      // note: timeout is used to check keyboard hit.
      wait_result := WaitForSingleObject(receive_event, 2000);

      // If we get a receive event
      If wait_result = WAIT_OBJECT_0 Then
      begin
        // Read messages
        read_status := PUDS_STATUS_OK;
        while not TUDSApi.StatusIsOk_2013(read_status,
          uds_status.PUDS_STATUS_NO_MESSAGE) do
        begin
          read_status := TUDSApi.Read_2013(tp_handle, request_msg);
          Writeln(format(' Try to read a message (status=%d): %s',
            [integer(read_status), STATUS_OK_KO(read_status)]));

          If TUDSApi.StatusIsOk_2013(read_status) Then
          begin
            display_message(@request_msg);

            if request_msg.msg.msgdata_isotp <> nil Then
              req_isotp := request_msg.msg.msgdata_isotp^;

            // We receive a request, check if it is not a loopback message, if it is not a UUDT message, if the message is not pending, and do not respond if it is not ask
            if ((UInt32(req_isotp.flags) And
              UInt32(cantp_msgflag.PCANTP_MSGFLAG_LOOPBACK)) = 0) And
              ((UInt32(req_isotp.netaddrinfo.msgtype) And
              UInt32(cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION))
              = 0) And (req_isotp.netstatus = cantp_netstatus.
              PCANTP_NETSTATUS_OK) And
              ((UInt32(request_msg.typem) And
              UInt32(uds_msgtype.PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE)) <>
              UInt32(uds_msgtype.PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE)) Then

              // Process response and send it
              process_request(tp_handle, UInt16(server_address), request_msg)
            else
            begin
              Writeln('');
              Writeln(format
                ('   ...Skipping response...  (flags=0x%x, type=0x%x, netstatus=0x%x, msgtype=0x%x)',
                [integer(req_isotp.flags), integer(request_msg.typem),
                integer(req_isotp.netstatus),
                integer(req_isotp.netaddrinfo.msgtype)]));
            end;
          end;
          // Free request message (and clean memory in order to reallocate later)
          status := TUDSApi.MsgFree_2013(request_msg);
          Writeln(format('Free request message: %s', [STATUS_OK_KO(status)]));
        end;
      end;

      // Quit when user press Q
      If KeyPressed() Then
      begin
        keyboard_res := ReadKey();
        If (keyboard_res = 'Q') Or (keyboard_res = 'q') Then
          stopit := true;
      end;
    end;

    // Close receive event
    null_handle := 0;
    status := TUDSApi.SetValue_2013(tp_handle,
      uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, PByte(@null_handle),
      sizeof(null_handle));

    Writeln(format('Stop receive event: %s', [STATUS_OK_KO(status)]));
    CloseHandle(receive_event);
    Writeln(format('Close receive event: %s', [STATUS_OK_KO(status)]));

    // Close server
    status := TUDSApi.Uninitialize_2013(tp_handle);
    Writeln(format('Uninitialize channel: %s', [STATUS_OK_KO(status)]));

    // Exit
    Writeln('Press any key to continue...');
    Readln(dummy);

  except
    on E: Exception do
      Writeln(E.Classname, ': ', E.Message);
  end;

end.
