program client_all_request;

{$APPTYPE CONSOLE}

uses
  SysUtils,
  ShellApi,
  windows,
  PUDS_2013 in 'PUDS_2013.pas',
  PCANBasic in 'PCANBasic.pas',
  PCANTP_2016 in 'PCANTP_2016.pas';

var
  // Console handling
  USE_GETCH: Boolean = False;

  // A global counter to keep track of the number of failed tests (see display_uds_msg function)
  g_nbErr: Integer = 0;

Function OK_KO(test: Boolean): String;
begin
  If test Then
    result := 'OK'
  else
    result := 'KO';
End;

Function STATUS_OK_KO(test: uds_status): String;
begin
  result := OK_KO(TUDSApi.StatusIsOk_2013(test));
End;

// <summary> A simple function that waits for user input</summary>
procedure waitGetch();
var
  dummy: Char;
begin
  If USE_GETCH Then
  begin
    Write(sLineBreak + ' Press <Enter> to continue...');
    readln(dummy);
  End;
End;

Procedure display_uds_msg_response(response: Puds_msg;
  no_response_expected: Boolean); forward;
Procedure display_uds_msg_request(request: Puds_msg); overload; forward;

/// <summary>A function that displays UDS Request and Response messages (and count error if no response)</summary>
/// <param name="request">Request message</param>
/// <param name="response">Received response message</param>
/// <param name="no_response_expected">if no response is expected, do not increment error counter</param>
Procedure display_uds_msg(request: Puds_msg; response: Puds_msg;
  no_response_expected: Boolean);
begin
  display_uds_msg_request(request);
  display_uds_msg_response(response, no_response_expected);
End;

Procedure display_uds_msg_response(response: Puds_msg;
  no_response_expected: Boolean);
var
  resp_isotp: cantp_msgdata_isotp;
  s1: String;
  i: Integer;
  data_pointer: PByte;
begin
  If response.msg.msgdata_isotp <> nil Then
  begin
    If response.msg.msgdata_isotp <> nil Then
      resp_isotp := response.msg.msgdata_isotp^;
    If resp_isotp.netstatus <> cantp_netstatus.PCANTP_NETSTATUS_OK then
      s1 := 'ERROR !!!'
    else
      s1 := 'OK !';
    Writeln(Format(sLineBreak +
      'UDS RESPONSE from 0x%.4x (to 0x%.4x, with extension address 0x%.2x) - result: %d - %s',
      [resp_isotp.netaddrinfo.source_addr, resp_isotp.netaddrinfo.target_addr,
      resp_isotp.netaddrinfo.extension_addr,
      Integer(resp_isotp.netstatus), s1]));
    // display data
    Write(Format('-> Length: %d, Data= ', [resp_isotp.length]));

    data_pointer := PByte(response.msg.msgdata_isotp.data);
    For i := 0 To resp_isotp.length - 1 do
    begin
      Write(Format('%.2x ', [data_pointer^]));
      inc(data_pointer);
    end;

    Writeln('');
  end
  Else If Not no_response_expected Then
  begin
    Writeln(sLineBreak + ' /!\ ERROR: NO UDS RESPONSE !!');
    g_nbErr := g_nbErr + 1;
  end;
End;

Procedure display_uds_msg_request(request: Puds_msg); overload;
var
  req_isotp: cantp_msgdata_isotp;
  s1: String;
  i: Integer;
  data_pointer: PByte;

begin

  If request.msg.msgdata_isotp <> nil Then
  begin
    req_isotp := request.msg.msgdata_isotp^;
    If req_isotp.netstatus <> cantp_netstatus.PCANTP_NETSTATUS_OK then
      s1 := 'ERROR !!!'
    else
      s1 := 'OK !';

    Writeln(Format(sLineBreak +
      'UDS request from 0x%.4x (to 0x%.4x, with extension address 0x%.2x) - result: %d - %s',
      [req_isotp.netaddrinfo.source_addr, req_isotp.netaddrinfo.target_addr,
      req_isotp.netaddrinfo.extension_addr, Integer(req_isotp.netstatus), s1]));
    // display data
    Write(Format('-> Length: %d, Data= ', [req_isotp.length]));

    data_pointer := PByte(request.msg.msgdata_isotp.data);

    For i := 0 To req_isotp.length - 1 do
    begin

      Write(Format('%.2x ', [data_pointer^]));
      inc(data_pointer);
    end;

    Writeln('');
  end;
End;

Procedure display_uds_msg_request(request: Puds_msg;
  no_response_expected: Boolean); overload;
begin
  display_uds_msg_request(request);

  If Not no_response_expected Then
  begin
    Writeln(sLineBreak + ' /!\ ERROR: NO UDS RESPONSE !!');
    g_nbErr := g_nbErr + 1;
  End;
End;

/// <summary>Helper: Inverts the bytes of a 32 bits numeric value</summary>
/// <param name="v">Value to revert</param>
/// <returns>Reverted value</returns>
Function Reverse32(v: UInt32): UInt32;
begin
  result := ((v And $FF) Shl 24) Or ((v And $FF00) Shl 8) Or
    ((v And $FF0000) Shr 8) Or ((v And $FF000000) Shr 24 And $FF);
End;

Procedure UInt32ToBytes(dw_buffer: UInt32; byte_buffer: PByte);
begin
  byte_buffer^ := Byte(dw_buffer And $FF);
  inc(byte_buffer);
  byte_buffer^ := Byte((dw_buffer And $FF00) Shr 8);
  inc(byte_buffer);
  byte_buffer^ := Byte((dw_buffer And $FF0000) Shr 16);
  inc(byte_buffer);
  byte_buffer^ := Byte((dw_buffer And $FF000000) Shr 24);
End;

/// <summary>Call UDS Service DiagnosticSessionControl</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testDiagnosticSessionControl(channel: cantp_handle;
  config: uds_msgconfig);
var
  status: uds_status;
  session_info: uds_sessioninfo;
  request: uds_msg;
  response: uds_msg;
  confirmation: uds_msg;
begin
  Writeln(sLineBreak + sLineBreak +
    '*** UDS Service: DiagnosticSessionControl ***');

  FillChar(session_info, sizeof(session_info), 0);
  FillChar(request, sizeof(request), 0);
  FillChar(response, sizeof(response), 0);
  FillChar(confirmation, sizeof(confirmation), 0);

  // Read default session information Server is not yet known (status will be PUDS_ERROR_NOT_INITIALIZED)
  // yet the API will still set session info to default values
  session_info.nai := config.nai;
  status := TUDSApi.GetValue_2013(channel,
    uds_parameter.PUDS_PARAMETER_SESSION_INFO, Pointer(@session_info),
    UInt32(sizeof(session_info)));
  Writeln(Format
    (' Diagnostic Session Information: %d, 0x%.4x => %d = [%.4x; %.4x]',
    [Integer(status), session_info.nai.target_addr, session_info.session_type,
    session_info.timeout_p2can_server_max,
    session_info.timeout_enhanced_p2can_server_max]));
  waitGetch();

  // Set Diagnostic session to DEFAULT (to get session information)
  Writeln(sLineBreak + sLineBreak + 'Setting a DEFAULT Diagnostic Session:');
  status := TUDSApi.SvcDiagnosticSessionControl_2013(channel, config, request,
    uds_svc_param_dsc.PUDS_SVC_PARAM_DSC_DS);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcDiagnosticSessionControl_2013: %d',
    [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));

  // Read current session information
  FillChar(session_info, sizeof(session_info), 0);
  session_info.nai := config.nai;
  status := TUDSApi.GetValue_2013(channel,
    uds_parameter.PUDS_PARAMETER_SESSION_INFO, Pointer(@session_info),
    UInt32(sizeof(session_info)));
  Writeln(Format
    (' Diagnostic Session Information: %d, 0x%.4x => %d = [%.4x; %.4x]',
    [Integer(status), session_info.nai.target_addr, session_info.session_type,
    session_info.timeout_p2can_server_max,
    session_info.timeout_enhanced_p2can_server_max]));
  waitGetch();

  // Set Diagnostic session to PROGRAMMING
  Writeln(sLineBreak + sLineBreak + 'Setting a ECUPS Diagnostic Session:');
  status := TUDSApi.SvcDiagnosticSessionControl_2013(channel, config, request,
    uds_svc_param_dsc.PUDS_SVC_PARAM_DSC_ECUPS);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcDiagnosticSessionControl_2013: %d',
    [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));

  // Read current session information
  FillChar(session_info, sizeof(session_info), 0);
  session_info.nai := config.nai;
  status := TUDSApi.GetValue_2013(channel,
    uds_parameter.PUDS_PARAMETER_SESSION_INFO, Pointer(@session_info),
    UInt32(sizeof(session_info)));
  Writeln(Format
    (' Diagnostic Session Information: %d, 0x%.4x => %d = [%.4x; %.4x]',
    [Integer(status), session_info.nai.target_addr, session_info.session_type,
    session_info.timeout_p2can_server_max,
    session_info.timeout_enhanced_p2can_server_max]));
  Writeln(' Assert that Auto TesterPresent Frame is sent...');
  Sleep(2000);
  Writeln(' Should transmit an Auto TesterPresent Frame');
  Sleep(2000);
  Writeln(' Should transmit an Auto TesterPresent Frame');
  waitGetch();

  // Set Diagnostic session back to DEFAULT
  Writeln(sLineBreak + sLineBreak + 'Setting a DEFAULT Diagnostic Session:');
  status := TUDSApi.SvcDiagnosticSessionControl_2013(channel, config, request,
    uds_svc_param_dsc.PUDS_SVC_PARAM_DSC_DS);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcDiagnosticSessionControl_2013: %d',
    [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  Writeln(' Assert that NO Auto TesterPresent Frame is sent...');
  Sleep(2000);
  Writeln(' Should NOT transmit an Auto TesterPresent Frame');
  Sleep(2000);
  Writeln(' Should NOT transmit an Auto TesterPresent Frame');
  waitGetch();
end;

/// <summary>Call UDS Service ECUReset</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testECUReset(channel: cantp_handle; config: uds_msgconfig);
var
  status: uds_status;
  request: uds_msg;
  response: uds_msg;
  confirmation: uds_msg;
begin
  Writeln(sLineBreak + sLineBreak + '*** UDS Service: ECUReset ***');

  FillChar(request, sizeof(request), 0);
  FillChar(response, sizeof(response), 0);
  FillChar(confirmation, sizeof(confirmation), 0);

  // Sends a physical ECUReset message
  Writeln(sLineBreak + sLineBreak + 'Sends a physical ECUReset message: ');
  status := TUDSApi.SvcECUReset_2013(channel, config, request,
    uds_svc_param_er.PUDS_SVC_PARAM_ER_SR);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcECUReset_2013: %d', [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();
end;

/// <summary>Call UDS Service SecurityAccess</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testSecurityAccess(channel: cantp_handle; config: uds_msgconfig);
var
  status: uds_status;
  request: uds_msg;
  response: uds_msg;
  confirmation: uds_msg;
  dw_buffer: UInt32;
  security_access_data: array [0 .. 3] of Byte;
  value_little_endian: UInt32;
begin
  Writeln(sLineBreak + sLineBreak + '*** UDS Service: SecurityAccess ***');

  FillChar(request, sizeof(request), 0);
  FillChar(response, sizeof(response), 0);
  FillChar(confirmation, sizeof(confirmation), 0);

  // Sends a physical SecurityAccess message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical SecurityAccess message: ');
  value_little_endian := $F0A1B2C3;
  dw_buffer := Reverse32(value_little_endian);
  // use helper function to set MSB as 1st byte in the buffer (Win32 uses little endian format)
  UInt32ToBytes(dw_buffer, PByte(@security_access_data));
  status := TUDSApi.SvcSecurityAccess_2013(channel, config, request,
    TUDSApi.PUDS_SVC_PARAM_SA_RSD_1, PByte(@security_access_data), 4);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcSecurityAccess_2013: %d', [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();
end;

/// <summary>Call UDS Service CommunicationControl</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testCommunicationControl(channel: cantp_handle;
  config: uds_msgconfig);
var
  status: uds_status;
  request: uds_msg;
  response: uds_msg;
  confirmation: uds_msg;
begin
  Writeln(sLineBreak + sLineBreak +
    '*** UDS Service: CommunicationControl ***');

  FillChar(request, sizeof(request), 0);
  FillChar(response, sizeof(response), 0);
  FillChar(confirmation, sizeof(confirmation), 0);

  // Sends a physical CommunicationControl message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical CommunicationControl message: ');
  status := TUDSApi.SvcCommunicationControl_2013(channel, config, request,
    uds_svc_param_cc.PUDS_SVC_PARAM_CC_ERXTX,
    TUDSApi.PUDS_SVC_PARAM_CC_FLAG_APPL Or TUDSApi.PUDS_SVC_PARAM_CC_FLAG_NWM Or
    TUDSApi.PUDS_SVC_PARAM_CC_FLAG_DENWRIRO);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcCommunicationControl_2013: %d', [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();
end;

/// <summary>UDS Call Service TesterPresent</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testTesterPresent(channel: cantp_handle; config: uds_msgconfig);
var
  status: uds_status;
  request: uds_msg;
  tresponse: array [0 .. 0] of uds_msg;
  response: uds_msg;
  confirmation: uds_msg;
  response_count: UInt32;
  max_msg_count: UInt32;
begin
  response_count := 0;
  FillChar(request, sizeof(request), 0);
  FillChar(response, sizeof(response), 0);
  FillChar(tresponse[0], sizeof(tresponse[0]), 0);
  FillChar(confirmation, sizeof(confirmation), 0);
  Writeln(sLineBreak + sLineBreak + '*** UDS Service: TesterPresent ***');

  // Sends a physical TesterPresent message
  Writeln(sLineBreak + sLineBreak + 'Sends a physical TesterPresent message: ');
  status := TUDSApi.SvcTesterPresent_2013(channel, config, request,
    uds_svc_param_tp.PUDS_SVC_PARAM_TP_ZSUBF);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcTesterPresent_2013: %d', [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a physical TesterPresent message with no positive response
  config.typem := uds_msgtype.PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE;
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical TesterPresent message with no positive response:');
  status := TUDSApi.SvcTesterPresent_2013(channel, config, request,
    uds_svc_param_tp.PUDS_SVC_PARAM_TP_ZSUBF);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcTesterPresent_2013: %d', [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, True);

  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a functional TesterPresent message
  config.typem := uds_msgtype.PUDS_MSGTYPE_USDT;
  config.nai.target_type :=
    cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_FUNCTIONAL;
  config.nai.target_addr :=
    UInt16(uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_OBD_FUNCTIONAL);
  Writeln(sLineBreak + sLineBreak +
    'Sends a functional TesterPresent message: ');
  status := TUDSApi.SvcTesterPresent_2013(channel, config, request,
    uds_svc_param_tp.PUDS_SVC_PARAM_TP_ZSUBF);
  max_msg_count := 1;
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForServiceFunctional_2013(channel, @request,
      max_msg_count, True, Puds_msg(@tresponse), response_count, @confirmation);
  Writeln(Format(' UDS_SvcTesterPresent_2013: %d', [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @(tresponse[0]), False)
  Else
    display_uds_msg_request(@request, False);

  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(tresponse[0]);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a functional TesterPresent message with no positive response
  config.typem := uds_msgtype.PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE;
  config.nai.target_type :=
    cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_FUNCTIONAL;
  config.nai.target_addr :=
    UInt16(uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_OBD_FUNCTIONAL);
  Writeln(sLineBreak + sLineBreak +
    'Sends a functional TesterPresent message with no positive response:');
  status := TUDSApi.SvcTesterPresent_2013(channel, config, request,
    uds_svc_param_tp.PUDS_SVC_PARAM_TP_ZSUBF);
  max_msg_count := 1;
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForServiceFunctional_2013(channel, @request,
      max_msg_count, True, Puds_msg(@tresponse), response_count, @confirmation);
  Writeln(Format(' UDS_SvcTesterPresent_2013: %d', [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) And (response_count <> 0) Then
    display_uds_msg(@confirmation, @(tresponse[0]), False)
  Else
    display_uds_msg_request(@request, True);

  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(tresponse[0]);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();
end;

/// <summary>Call UDS Service SecuredDataTransmission</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testSecuredDataTransmission(channel: cantp_handle;
  config: uds_msgconfig);
var
  status: uds_status;
  request: uds_msg;
  response: uds_msg;
  confirmation: uds_msg;
  ecureset_request: uds_msg;
  dw_buffer: UInt32;
  security_data_request_record: array [0 .. 3] of Byte;
  value_little_endian: UInt32;
  administrative_parameter: UInt16;
  signature_encryption_calculation: UInt8;
  anti_replay_counter: UInt16;
  internal_service_identifier: UInt8;
  service_specific_parameters: array [0 .. 3] of Byte;
  service_specific_parameters_size: UInt32;
  signature_mac: array [0 .. 5] of Byte;
  signature_size: UInt16;
begin
  Writeln(sLineBreak + sLineBreak +
    '*** UDS Service: SecuredDataTransmission ***');

  FillChar(request, sizeof(request), 0);
  FillChar(response, sizeof(response), 0);
  FillChar(confirmation, sizeof(confirmation), 0);
  FillChar(ecureset_request, sizeof(ecureset_request), 0);

  // Sends a physical SecuredDataTransmission/2013 message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical SecuredDataTransmission/2013 message: ');
  value_little_endian := $F0A1B2C3;
  dw_buffer := Reverse32(value_little_endian);
  // use helper function to set MSB as 1st byte in the buffer (Win32 uses little endian format)
  UInt32ToBytes(dw_buffer, PByte(@security_data_request_record));
  status := TUDSApi.SvcSecuredDataTransmission_2013(channel, config, request,
    PByte(@security_data_request_record), 4);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcSecuredDataTransmission_2013: %d',
    [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a physical SecuredDataTransmission/2013 message prepared with PUDS_ONLY_PREPARE_REQUEST option
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical SecuredDataTransmission/2013 prepared with PUDS_ONLY_PREPARE_REQUEST option: ');
  status := TUDSApi.SvcECUReset_2013(TUDSApi.PUDS_ONLY_PREPARE_REQUEST, config,
    &ecureset_request, PUDS_SVC_PARAM_ER_HR);
  Writeln(Format(' Prepare ECUReset request for SecuredDataTransmission: %d',
    [Integer(status)]));
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    status := TUDSApi.SvcSecuredDataTransmission_2013(channel, config, &request,
      PByte(ecureset_request.msg.msgdata_any.data),
      ecureset_request.msg.msgdata_any.length);
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcSecuredDataTransmission_2013: %d',
    [Integer(status)]));
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    display_uds_msg(@confirmation, @response, False)
  else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(&ecureset_request);
  Writeln(Format(' Free prepared message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(&request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(&response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(&confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a physical SecuredDataTransmission/2020 message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical SecuredDataTransmission/2020 message: ');
  administrative_parameter := TUDSApi.PUDS_SVC_PARAM_APAR_REQUEST_MSG_FLAG or
    TUDSApi.PUDS_SVC_PARAM_APAR_REQUEST_RESPONSE_SIGNATURE_FLAG or
    TUDSApi.PUDS_SVC_PARAM_APAR_SIGNED_MSG_FLAG;
  signature_encryption_calculation := $0;
  anti_replay_counter := $0124;
  internal_service_identifier := $2E;
  service_specific_parameters[0] := $F1;
  service_specific_parameters[1] := $23;
  service_specific_parameters[2] := $AA;
  service_specific_parameters[3] := $55;
  service_specific_parameters_size := 4;
  signature_mac[0] := $DB;
  signature_mac[1] := $D1;
  signature_mac[2] := $0E;
  signature_mac[3] := $DC;
  signature_mac[4] := $55;
  signature_mac[5] := $AA;
  signature_size := $0006;
  status := TUDSApi.SvcSecuredDataTransmission_2020(channel, config, &request,
    administrative_parameter, signature_encryption_calculation,
    anti_replay_counter, internal_service_identifier,
    PByte(@service_specific_parameters), service_specific_parameters_size,
    PByte(@signature_mac), signature_size);
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcSecuredDataTransmission_2013: %d',
    [Integer(status)]));
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    display_uds_msg(@confirmation, @response, False)
  else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(&request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(&response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(&confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();
end;

/// <summary>Call UDS Service ControlDTCSetting</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testControlDTCSetting(channel: cantp_handle; config: uds_msgconfig);
var
  status: uds_status;
  request: uds_msg;
  response: uds_msg;
  confirmation: uds_msg;
  dw_buffer: UInt32;
  dtc_setting_control_option_record: array [0 .. 4] of Byte;
  value_little_endian: UInt32;
begin
  Writeln(sLineBreak + sLineBreak + '*** UDS Service: ControlDTCSetting ***');

  FillChar(request, sizeof(request), 0);
  FillChar(response, sizeof(response), 0);
  FillChar(confirmation, sizeof(confirmation), 0);

  // Sends a physical ControlDTCSetting message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical ControlDTCSetting message: ');
  value_little_endian := $F1A1B2EE;
  dw_buffer := Reverse32(value_little_endian);
  // use helper function to set MSB as 1st byte in the buffer (Win32 uses little endian format)
  UInt32ToBytes(dw_buffer, PByte(@dtc_setting_control_option_record));
  status := TUDSApi.SvcControlDTCSetting_2013(channel, config, request,
    uds_svc_param_cdtcs.PUDS_SVC_PARAM_CDTCS_OFF,
    PByte(@dtc_setting_control_option_record), 3);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcControlDTCSetting_2013: %d', [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch()
end;

/// <summary>Call UDS Service ResponseOnEvent</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testResponseOnEvent(channel: cantp_handle; config: uds_msgconfig);
var
  status: uds_status;
  request: uds_msg;
  response: uds_msg;
  confirmation: uds_msg;
  event_type_record: array [0 .. 49] of Byte;
  service_to_respond_to_record: array [0 .. 49] of Byte;
begin
  Writeln(sLineBreak + sLineBreak + '*** UDS Service: ResponseOnEvent ***');

  FillChar(request, sizeof(request), 0);
  FillChar(response, sizeof(response), 0);
  FillChar(confirmation, sizeof(confirmation), 0);

  // Sends a physical ResponseOnEvent message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical ResponseOnEvent message: ');
  event_type_record[0] := $8;
  service_to_respond_to_record[0] :=
    Byte(uds_service.PUDS_SERVICE_SI_ReadDTCInformation);
  service_to_respond_to_record[1] :=
    Byte(uds_svc_param_rdtci.PUDS_SVC_PARAM_RDTCI_RNODTCBSM);
  service_to_respond_to_record[2] := $1;
  status := TUDSApi.SvcResponseOnEvent_2013(channel, config, request,
    uds_svc_param_roe.PUDS_SVC_PARAM_ROE_ONDTCS, False, $8,
    PByte(@event_type_record), TUDSApi.PUDS_SVC_PARAM_ROE_OTI_LEN,
    PByte(@service_to_respond_to_record), 3);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcResponseOnEvent_2013: %d', [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();
end;

/// <summary>Call UDS Service LinkControl</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testLinkControl(channel: cantp_handle; config: uds_msgconfig);
var
  status: uds_status;
  request: uds_msg;
  response: uds_msg;
  confirmation: uds_msg;
begin
  Writeln(sLineBreak + sLineBreak + '*** UDS Service: LinkControl ***');

  FillChar(request, sizeof(request), 0);
  FillChar(response, sizeof(response), 0);
  FillChar(confirmation, sizeof(confirmation), 0);

  // Sends a physical LinkControl message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical LinkControl message (Verify Fixed Baudrate): ');
  status := TUDSApi.SvcLinkControl_2013(channel, config, request,
    uds_svc_param_lc.PUDS_SVC_PARAM_LC_VBTWFBR,
    uds_svc_param_lc_baudrate_identifier.
    PUDS_SVC_PARAM_LC_BAUDRATE_CAN_500K, 0);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcLinkControl_2013: %d', [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));

  // Sends a physical LinkControl message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical LinkControl message (Verify Specific Baudrate): ');
  status := TUDSApi.SvcLinkControl_2013(channel, config, request,
    uds_svc_param_lc.PUDS_SVC_PARAM_LC_VBTWSBR,
    uds_svc_param_lc_baudrate_identifier(0), 500000); // 500K = 0x0007a120
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcLinkControl_2013: %d', [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));

  // Sends a physical LinkControl message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical LinkControl message (Transition): ');
  status := TUDSApi.SvcLinkControl_2013(channel, config, request,
    uds_svc_param_lc.PUDS_SVC_PARAM_LC_TB,
    uds_svc_param_lc_baudrate_identifier(0), 0);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcLinkControl_2013: %d', [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch()
end;

/// <summary>Call UDS Service ReadDataByIdentifier</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testReadDataByIdentifier(channel: cantp_handle;
  config: uds_msgconfig);
var
  status: uds_status;
  request: uds_msg;
  response: uds_msg;
  confirmation: uds_msg;
  data_identifier: array [0 .. 1] of uds_svc_param_di;
begin
  Writeln(sLineBreak + sLineBreak +
    '*** UDS Service: ReadDataByIdentifier ***');

  FillChar(request, sizeof(request), 0);
  FillChar(response, sizeof(response), 0);
  FillChar(confirmation, sizeof(confirmation), 0);
  data_identifier[0] := uds_svc_param_di.PUDS_SVC_PARAM_DI_ADSDID;
  data_identifier[1] := uds_svc_param_di.PUDS_SVC_PARAM_DI_ECUMDDID;

  // Sends a physical ReadDataByIdentifier message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical ReadDataByIdentifier message: ');
  status := TUDSApi.SvcReadDataByIdentifier_2013(channel, config, request,
    Puds_svc_param_di(@data_identifier), 2);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcReadDataByIdentifier_2013: %d', [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();
End;

/// <summary>Call UDS Service ReadMemoryByAddress</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testReadMemoryByAddress(channel: cantp_handle; config: uds_msgconfig);
var
  status: uds_status;
  request: uds_msg;
  response: uds_msg;
  confirmation: uds_msg;
  memory_address_buffer: array [0 .. 9] of Byte;
  memory_size_buffer: array [0 .. 9] of Byte;
  memory_address_size: Byte;
  memory_size_size: Byte;
  i: Integer;
begin
  memory_address_size := 10;
  memory_size_size := 3;
  FillChar(request, sizeof(request), 0);
  FillChar(response, sizeof(response), 0);
  FillChar(confirmation, sizeof(confirmation), 0);
  Writeln(sLineBreak + sLineBreak + '*** UDS Service: ReadMemoryByAddress ***');

  // Sends a physical ReadMemoryByAddress message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical ReadMemoryByAddress message: ');

  For i := 0 To memory_address_size - 1 do
  begin
    memory_address_buffer[i] := Ord('A') + i;
    memory_size_buffer[i] := Ord('1') + i;
  end;

  status := TUDSApi.SvcReadMemoryByAddress_2013(channel, config, request,
    PByte(@memory_address_buffer), memory_address_size,
    PByte(@memory_size_buffer), memory_size_size);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcReadMemoryByAddress_2013: %d', [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();
end;

/// <summary>Call UDS Service ReadScalingDataByIdentifier</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testReadScalingDataByIdentifier(channel: cantp_handle;
  config: uds_msgconfig);
var
  status: uds_status;
  request: uds_msg;
  response: uds_msg;
  confirmation: uds_msg;
begin
  FillChar(request, sizeof(request), 0);
  FillChar(response, sizeof(response), 0);
  FillChar(confirmation, sizeof(confirmation), 0);

  Writeln(sLineBreak + sLineBreak +
    '*** UDS Service: ReadScalingDataByIdentifier ***');

  // Sends a physical ReadScalingDataByIdentifier message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical ReadScalingDataByIdentifier message: ');
  status := TUDSApi.SvcReadScalingDataByIdentifier_2013(channel, config,
    request, uds_svc_param_di.PUDS_SVC_PARAM_DI_BSFPDID);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcReadScalingDataByIdentifier_2013: %d',
    [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch()
end;

/// <summary>Call UDS Service ReadDataByPeriodicIdentifier</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testReadDataByPeriodicIdentifier(channel: cantp_handle;
  config: uds_msgconfig);
var
  status: uds_status;
  request: uds_msg;
  response: uds_msg;
  confirmation: uds_msg;
  periodic_data_identifier: array [0 .. 9] of Byte;
  periodic_data_identifier_size: UInt16;
  i: Integer;
begin
  periodic_data_identifier_size := 10;
  FillChar(request, sizeof(request), 0);
  FillChar(response, sizeof(response), 0);
  FillChar(confirmation, sizeof(confirmation), 0);
  Writeln(sLineBreak + sLineBreak +
    '*** UDS Service: ReadDataByPeriodicIdentifier ***');

  // Sends a physical ReadScalingDataByIdentifier message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical ReadDataByPeriodicIdentifier message: ');

  For i := 0 To periodic_data_identifier_size - 1 do
    periodic_data_identifier[i] := Ord('A') + i;

  status := TUDSApi.SvcReadDataByPeriodicIdentifier_2013(channel, config,
    request, uds_svc_param_rdbpi.PUDS_SVC_PARAM_RDBPI_SAMR,
    PByte(@periodic_data_identifier), periodic_data_identifier_size);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcReadDataByPeriodicIdentifier_2013: %d',
    [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch()
end;

/// <summary>Call UDS Service DynamicallyDefineDataIdentifier</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testDynamicallyDefineDataIdentifier(channel: cantp_handle;
  config: uds_msgconfig);
var
  status: uds_status;
  request: uds_msg;
  response: uds_msg;
  confirmation: uds_msg;
  source_data_identifier: array [0 .. 9] of UInt16;
  memory_size: array [0 .. 9] of Byte;
  position_in_source_data_record: array [0 .. 9] of Byte;
  number_of_elements: UInt16;
  memory_address_buffer: array [0 .. 14] of Byte;
  memory_size_buffer: array [0 .. 8] of Byte;
  memory_address_size: Byte;
  memory_size_size: Byte;
  i, j: Integer;
begin
  number_of_elements := 10;
  FillChar(request, sizeof(request), 0);
  FillChar(response, sizeof(response), 0);
  FillChar(confirmation, sizeof(confirmation), 0);

  Writeln(sLineBreak + sLineBreak +
    '*** UDS Service: DynamicallyDefineDataIdentifier ***');

  // Sends a physical DynamicallyDefineDataIdentifierDBID message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical DynamicallyDefineDataIdentifierDBID message: ');

  For i := 0 To number_of_elements - 1 do
  begin
    source_data_identifier[i] := UInt16(($F0 + i) Shl 8) + Ord('A') + i;
    memory_size[i] := Byte(i + 1);
    position_in_source_data_record[i] := Byte(100 + i);
  end;

  status := TUDSApi.SvcDynamicallyDefineDataIdentifierDBID_2013(channel, config,
    request, uds_svc_param_di.PUDS_SVC_PARAM_DI_CDDID,
    PUInt16(@source_data_identifier), PByte(@memory_size),
    PByte(@position_in_source_data_record), number_of_elements);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcDynamicallyDefineDataIdentifierDBID_2013: %d',
    [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a physical UDS_SvcDynamicallyDefineDataIdentifierDBMA message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical UDS_SvcDynamicallyDefineDataIdentifierDBMA_2013 message: ');
  number_of_elements := 3;
  memory_address_size := 5;
  memory_size_size := 3;

  For j := 0 To number_of_elements - 1 do
  begin
    For i := 0 To memory_address_size - 1 do
      memory_address_buffer[memory_address_size * j + i] :=
        Byte(10 * j + i + 1);

    For i := 0 To memory_size_size - 1 do
      memory_size_buffer[memory_size_size * j + i] :=
        Byte(100 + 10 * j + i + 1);

  end;

  status := TUDSApi.SvcDynamicallyDefineDataIdentifierDBMA_2013(channel, config,
    request, uds_svc_param_di.PUDS_SVC_PARAM_DI_CESWNDID, memory_address_size,
    memory_size_size, PByte(@memory_address_buffer), PByte(@memory_size_buffer),
    number_of_elements);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcDynamicallyDefineDataIdentifierDBMA_2013: %d',
    [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a physical UDS_SvcDynamicallyDefineDataIdentifierCDDDI message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical UDS_SvcDynamicallyDefineDataIdentifierCDDDI_2013 message: ');
  status := TUDSApi.SvcDynamicallyDefineDataIdentifierCDDDI_2013(channel,
    config, request, uds_svc_param_di.PUDS_SVC_PARAM_DI_CESWNDID);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcDynamicallyDefineDataIdentifierCDDDI_2013: %d',
    [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a physical UDS_SvcDynamicallyDefineDataIdentifierClearAllDDDI_2013 message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical UDS_SvcDynamicallyDefineDataIdentifierClearAllDDDI_2013 message: ');
  status := TUDSApi.SvcDynamicallyDefineDataIdentifierClearAllDDDI_2013(channel,
    config, request);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcDynamicallyDefineDataIdentifierClearAllDDDI_2013: %d',
    [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch()
end;

/// <summary>Call UDS Service WriteDataByIdentifier</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testWriteDataByIdentifier(channel: cantp_handle;
  config: uds_msgconfig);
var
  status: uds_status;
  request: uds_msg;
  response: uds_msg;
  confirmation: uds_msg;
  data_record: array [0 .. 9] of Byte;
  data_record_size: UInt16;
  i: Integer;
begin
  data_record_size := 10;
  FillChar(request, sizeof(request), 0);
  FillChar(response, sizeof(response), 0);
  FillChar(confirmation, sizeof(confirmation), 0);
  Writeln(sLineBreak + sLineBreak +
    '*** UDS Service: WriteDataByIdentifier ***');

  // Sends a physical WriteDataByIdentifier message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical WriteDataByIdentifier message: ');

  For i := 0 To data_record_size - 1 do
    data_record[i] := Ord('A') + i;

  status := TUDSApi.SvcWriteDataByIdentifier_2013(channel, config, request,
    uds_svc_param_di.PUDS_SVC_PARAM_DI_ASFPDID, PByte(@data_record),
    data_record_size);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcWriteDataByIdentifier_2013: %d', [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();
end;

//
/// <summary>Call UDS Service WriteMemoryByAddress</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testWriteMemoryByAddress(channel: cantp_handle;
  config: uds_msgconfig);
var
  status: uds_status;
  request: uds_msg;
  response: uds_msg;
  confirmation: uds_msg;
  data_record: array [0 .. 49] of Byte;
  memory_address_buffer: array [0 .. 49] of Byte;
  memory_size_buffer: array [0 .. 49] of Byte;
  data_record_size: UInt16;
  memory_address_size: Byte;
  memory_size_size: Byte;
  i: Integer;
begin
  FillChar(request, sizeof(request), 0);
  FillChar(response, sizeof(response), 0);
  FillChar(confirmation, sizeof(confirmation), 0);
  data_record_size := 50;
  memory_address_size := 5;
  memory_size_size := 3;
  Writeln(sLineBreak + sLineBreak +
    '*** UDS Service: WriteMemoryByAddress ***');

  // Sends a physical WriteMemoryByAddress message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical WriteMemoryByAddress message: ');

  For i := 0 To data_record_size - 1 do
  begin
    data_record[i] := Byte(i + 1);
    memory_address_buffer[i] := Ord('A') + i;
    memory_size_buffer[i] := Byte(10 + i);
  end;

  status := TUDSApi.SvcWriteMemoryByAddress_2013(channel, config, request,
    PByte(@memory_address_buffer), memory_address_size,
    PByte(@memory_size_buffer), memory_size_size, PByte(@data_record),
    data_record_size);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcWriteMemoryByAddress_2013: %d', [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();
end;

/// <summary>Call UDS Service ClearDiagnosticInformation</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testClearDiagnosticInformation(channel: cantp_handle;
  config: uds_msgconfig);
var
  status: uds_status;
  request: uds_msg;
  response: uds_msg;
  confirmation: uds_msg;
begin
  Writeln(sLineBreak + sLineBreak +
    '*** UDS Service: ClearDiagnosticInformation ***');

  FillChar(request, sizeof(request), 0);
  FillChar(response, sizeof(response), 0);
  FillChar(confirmation, sizeof(confirmation), 0);

  // Sends a physical ClearDiagnosticInformation message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical ClearDiagnosticInformation message: ');
  status := TUDSApi.SvcClearDiagnosticInformation_2013(channel, config,
    request, $F1A2B3);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcClearDiagnosticInformation_2013: %d',
    [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a physical ClearDiagnosticInformation message with memory selection parameter (2020)
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical ClearDiagnosticInformation message with memory selection parameter: ');
  status := TUDSApi.SvcClearDiagnosticInformation_2020(channel, config, request,
    $F1A2B3, $42);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcClearDiagnosticInformation_2020: %d',
    [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();
end;

/// <summary>Call UDS Service ReadDTCInformation</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testReadDTCInformation(channel: cantp_handle; config: uds_msgconfig);
var
  status: uds_status;
  request: uds_msg;
  response: uds_msg;
  confirmation: uds_msg;
begin
  FillChar(request, sizeof(request), 0);
  FillChar(response, sizeof(response), 0);
  FillChar(confirmation, sizeof(confirmation), 0);
  Writeln(sLineBreak + sLineBreak + '*** UDS Service: ReadDTCInformation ***');

  // Sends a physical ReadDTCInformation message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical ReadDTCInformation message: ');
  status := TUDSApi.SvcReadDTCInformation_2013(channel, config, request,
    uds_svc_param_rdtci.PUDS_SVC_PARAM_RDTCI_RNODTCBSM, $F1);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcReadDTCInformation_2013: %d', [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a physical ReadDTCInformationRDTCSSBDTC message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical ReadDTCInformationRDTCSSBDTC message: ');
  status := TUDSApi.SvcReadDTCInformationRDTCSSBDTC_2013(channel, config,
    request, $A1B2B3, $12);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcReadDTCInformationRDTCSSBDTC_2013: %d',
    [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a physical ReadDTCInformationRDTCSSBRN message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical ReadDTCInformationRDTCSSBRN message: ');
  status := TUDSApi.SvcReadDTCInformationRDTCSSBRN_2013(channel, config,
    request, $12);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcReadDTCInformationRDTCSSBRN_2013: %d',
    [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a physical ReadDTCInformationReportExtended message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical ReadDTCInformationReportExtended message: ');
  status := TUDSApi.SvcReadDTCInformationReportExtended_2013(channel, config,
    request, uds_svc_param_rdtci.PUDS_SVC_PARAM_RDTCI_RDTCEDRBDN, $A1B2B3, $12);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcReadDTCInformationReportExtended_2013: %d',
    [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a physical ReadDTCInformationReportSeverity message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical ReadDTCInformationReportSeverity message: ');
  status := TUDSApi.SvcReadDTCInformationReportSeverity_2013(channel, config,
    request, uds_svc_param_rdtci.PUDS_SVC_PARAM_RDTCI_RNODTCBSMR, $F1, $12);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcReadDTCInformationReportSeverity_2013: %d',
    [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a physical ReadDTCInformationRSIODTC message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical ReadDTCInformationRSIODTC message: ');
  status := TUDSApi.SvcReadDTCInformationRSIODTC_2013(channel, config, request,
    $F1A1B2B3);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcReadDTCInformationRSIODTC_2013: %d',
    [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a physical ReadDTCInformationNoParam message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical ReadDTCInformationNoParam message: ');
  status := TUDSApi.SvcReadDTCInformationNoParam_2013(channel, config, request,
    uds_svc_param_rdtci.PUDS_SVC_PARAM_RDTCI_RSUPDTC);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcReadDTCInformationNoParam_2013: %d',
    [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a physical UDS_SvcReadDTCInformationRDTCEDBR_2013 message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical UDS_SvcReadDTCInformationRDTCEDBR_2013 message: ');
  status := TUDSApi.SvcReadDTCInformationRDTCEDBR_2013(channel, config,
    &request, $12);
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcReadDTCInformationRDTCEDBR_2013: %d',
    [Integer(status)]));
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    display_uds_msg(@confirmation, @response, False)
  else
    display_uds_msg_request(@request, False);
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a physical UDS_SvcReadDTCInformationRUDMDTCBSM_2013 message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical UDS_SvcReadDTCInformationRUDMDTCBSM_2013 message: ');
  status := TUDSApi.SvcReadDTCInformationRUDMDTCBSM_2013(channel, config,
    &request, $12, $34);
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcReadDTCInformationRUDMDTCBSM_2013: %d',
    [Integer(status)]));
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    display_uds_msg(@confirmation, @response, False)
  else
    display_uds_msg_request(@request, False);
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a physical UDS_SvcReadDTCInformationRUDMDTCSSBDTC_2013 message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical UDS_SvcReadDTCInformationRUDMDTCSSBDTC_2013 message: ');
  status := TUDSApi.SvcReadDTCInformationRUDMDTCSSBDTC_2013(channel, config,
    &request, $00123456, $78, $9A);
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcReadDTCInformationRUDMDTCSSBDTC_2013: %d',
    [Integer(status)]));
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    display_uds_msg(@confirmation, @response, False)
  else
    display_uds_msg_request(@request, False);
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a physical UDS_SvcReadDTCInformationRUDMDTCEDRBDN_2013 message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical UDS_SvcReadDTCInformationRUDMDTCEDRBDN_2013 message: ');
  status := TUDSApi.SvcReadDTCInformationRUDMDTCEDRBDN_2013(channel, config,
    &request, $00123456, $78, $9A);
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcReadDTCInformationRUDMDTCEDRBDN_2013: %d',
    [Integer(status)]));
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    display_uds_msg(@confirmation, @response, False)
  else
    display_uds_msg_request(@request, False);
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a physical UDS_SvcReadDTCInformationRDTCEDI_2020 message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical UDS_SvcReadDTCInformationRDTCEDI_2020 message: ');
  status := TUDSApi.SvcReadDTCInformationRDTCEDI_2020(channel, config,
    &request, $12);
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcReadDTCInformationRDTCEDI_2020: %d',
    [Integer(status)]));
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    display_uds_msg(@confirmation, @response, False)
  else
    display_uds_msg_request(@request, False);
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a physical UDS_SvcReadDTCInformationRWWHOBDDTCBMR_2013 message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical UDS_SvcReadDTCInformationRWWHOBDDTCBMR_2013 message: ');
  status := TUDSApi.SvcReadDTCInformationRWWHOBDDTCBMR_2013(channel, config,
    &request, $12, $34, $56);
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcReadDTCInformationRWWHOBDDTCBMR_2013: %d',
    [Integer(status)]));
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    display_uds_msg(@confirmation, @response, False)
  else
    display_uds_msg_request(@request, False);
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a physical UDS_SvcReadDTCInformationRWWHOBDDTCWPS_2013 message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical UDS_SvcReadDTCInformationRWWHOBDDTCWPS_2013 message: ');
  status := TUDSApi.SvcReadDTCInformationRWWHOBDDTCWPS_2013(channel, config,
    &request, $12);
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcReadDTCInformationRWWHOBDDTCWPS_2013: %d',
    [Integer(status)]));
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    display_uds_msg(@confirmation, @response, False)
  else
    display_uds_msg_request(@request, False);
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a physical UDS_SvcReadDTCInformationRDTCBRGI_2020 message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical UDS_SvcReadDTCInformationRDTCBRGI_2020 message: ');
  status := TUDSApi.SvcReadDTCInformationRDTCBRGI_2020(channel, config,
    &request, $12, $34);
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcReadDTCInformationRDTCBRGI_2020: %d',
    [Integer(status)]));
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    display_uds_msg(@confirmation, @response, False)
  else
    display_uds_msg_request(@request, False);
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();
end;

/// <summary>Call UDS Service InputOutputControlByIdentifier</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testInputOutputControlByIdentifier(channel: cantp_handle;
  config: uds_msgconfig);
var

  status: uds_status;
  request: uds_msg;
  response: uds_msg;
  confirmation: uds_msg;
  control_option_record: array [0 .. 9] of Byte;
  control_enable_mask_record: array [0 .. 9] of Byte;
  control_option_record_size: UInt16;
  control_enable_mask_record_size: UInt16;
  i: Integer;
begin
  FillChar(request, sizeof(request), 0);
  FillChar(response, sizeof(response), 0);
  FillChar(confirmation, sizeof(confirmation), 0);
  control_option_record_size := 10;
  control_enable_mask_record_size := 5;

  Writeln(sLineBreak + sLineBreak +
    '*** UDS Service: InputOutputControlByIdentifier ***');

  // Sends a physical InputOutputControlByIdentifier message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical InputOutputControlByIdentifier message: ');

  For i := 0 To control_option_record_size - 1 do
  begin
    control_option_record[i] := Ord('A') + i;
    control_enable_mask_record[i] := Byte(10 + i);
  end;

  status := TUDSApi.SvcInputOutputControlByIdentifier_2013(channel, config,
    request, uds_svc_param_di.PUDS_SVC_PARAM_DI_SSECUSWVNDID,
    PByte(@control_option_record), control_option_record_size,
    PByte(@control_enable_mask_record), control_enable_mask_record_size);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcInputOutputControlByIdentifier_2013: %d',
    [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();
end;

/// <summary>Call UDS Service RoutineControl</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testRoutineControl(channel: cantp_handle; config: uds_msgconfig);
var
  status: uds_status;
  request: uds_msg;
  response: uds_msg;
  confirmation: uds_msg;
  routine_control_option_record: array [0 .. 9] of Byte;
  routine_control_option_record_size: UInt16;
  i: Integer;
begin
  FillChar(request, sizeof(request), 0);
  FillChar(response, sizeof(response), 0);
  FillChar(confirmation, sizeof(confirmation), 0);
  routine_control_option_record_size := 10;
  Writeln(sLineBreak + sLineBreak + '*** UDS Service: RoutineControl ***');

  // Sends a physical RoutineControl message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical RoutineControl message: ');

  For i := 0 To routine_control_option_record_size - 1 do
    routine_control_option_record[i] := Ord('A') + i;

  status := TUDSApi.SvcRoutineControl_2013(channel, config, request,
    uds_svc_param_rc.PUDS_SVC_PARAM_RC_RRR, uds_svc_param_rc_rid($F1A2),
    PByte(@routine_control_option_record), routine_control_option_record_size);

  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcRoutineControl_2013: %d', [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();
end;

/// <summary>Call UDS Service RequestDownload</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testRequestDownload(channel: cantp_handle; config: uds_msgconfig);
var
  status: uds_status;
  request: uds_msg;
  response: uds_msg;
  confirmation: uds_msg;
  memory_address_buffer: array [0 .. 14] of Byte;
  memory_size_buffer: array [0 .. 14] of Byte;
  memory_address_size: Byte;
  memory_size_size: Byte;
  i: Integer;
begin
  FillChar(request, sizeof(request), 0);
  FillChar(response, sizeof(response), 0);
  FillChar(confirmation, sizeof(confirmation), 0);
  memory_address_size := 15;
  memory_size_size := 8;
  Writeln(sLineBreak + sLineBreak + '*** UDS Service: RequestDownload ***');

  // Sends a physical RequestDownload message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical RequestDownload message: ');

  For i := 0 To memory_address_size - 1 do
  begin
    memory_address_buffer[i] := Ord('A') + i;
    memory_size_buffer[i] := Byte(10 + i);
  end;

  status := TUDSApi.SvcRequestDownload_2013(channel, config, request, $1, $2,
    PByte(@memory_address_buffer), memory_address_size,
    PByte(@memory_size_buffer), memory_size_size);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcRequestDownload_2013: %d', [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();
end;

/// <summary>Call UDS Service RequestUpload</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testRequestUpload(channel: cantp_handle; config: uds_msgconfig);
var
  status: uds_status;
  request: uds_msg;
  response: uds_msg;
  confirmation: uds_msg;
  memory_address_buffer: array [0 .. 3] of Byte;
  memory_size_buffer: array [0 .. 3] of Byte;
  memory_address_size: Byte;
  memory_size_size: Byte;
  i: Integer;
begin
  FillChar(request, sizeof(request), 0);
  FillChar(response, sizeof(response), 0);
  FillChar(confirmation, sizeof(confirmation), 0);
  memory_address_size := 4;
  memory_size_size := 4;
  Writeln(sLineBreak + sLineBreak + '*** UDS Service: RequestUpload ***');

  // Sends a physical RequestUpload message
  Writeln(sLineBreak + sLineBreak + 'Sends a physical RequestUpload message: ');

  For i := 0 To memory_size_size - 1 do
  begin
    memory_address_buffer[i] := Ord('A') + i;
    memory_size_buffer[i] := Byte(10 + i);
  end;

  status := TUDSApi.SvcRequestUpload_2013(channel, config, request, $1, $2,
    PByte(@memory_address_buffer), memory_address_size,
    PByte(@memory_size_buffer), memory_size_size);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcRequestUpload_2013: %d', [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();
end;

/// <summary>Call UDS Service RequestTransferExit</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testRequestTransferExit(channel: cantp_handle; config: uds_msgconfig);
var
  status: uds_status;
  request: uds_msg;
  response: uds_msg;
  confirmation: uds_msg;
  parameter_record: array [0 .. 19] of Byte;
  parameter_record_size: Byte;
  i: Integer;
begin
  FillChar(request, sizeof(request), 0);
  FillChar(response, sizeof(response), 0);
  FillChar(confirmation, sizeof(confirmation), 0);
  parameter_record_size := 20;
  Writeln(sLineBreak + sLineBreak + '*** UDS Service: RequestTransferExit ***');

  // Sends a physical RequestTransferExit message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical RequestTransferExit message: ');

  For i := 0 To parameter_record_size - 1 do
    parameter_record[i] := Ord('A') + i;

  status := TUDSApi.SvcRequestTransferExit_2013(channel, config, request,
    PByte(@parameter_record), parameter_record_size);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcRequestTransferExit_2013: %d', [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();
end;

/// <summary>Call UDS Service TransferData</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testTransferData(channel: cantp_handle; config: uds_msgconfig);
var
  status: uds_status;
  request: uds_msg;
  response: uds_msg;
  confirmation: uds_msg;
  buffer: array [0 .. 49] of Byte;
  buffer_size: Byte;
  i: Integer;
begin
  FillChar(request, sizeof(request), 0);
  FillChar(response, sizeof(response), 0);
  FillChar(confirmation, sizeof(confirmation), 0);
  buffer_size := 50;
  Writeln(sLineBreak + sLineBreak + '*** UDS Service: TransferData ***');

  // Sends a physical TransferData message
  Writeln(sLineBreak + sLineBreak + 'Sends a physical TransferData message: ');

  For i := 0 To buffer_size - 1 do
    buffer[i] := Ord('A') + i;

  status := TUDSApi.SvcTransferData_2013(channel, config, request, $1,
    PByte(@buffer), buffer_size);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcTransferData_2013: %d', [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();
end;

/// <summary>Call UDS Service TransferData with MAX_DATA length</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testTransferDataBigMessage(channel: cantp_handle;
  config: uds_msgconfig);
var
  status: uds_status;
  request: uds_msg;
  response: uds_msg;
  confirmation: uds_msg;
  buffer: array [0 .. 4092] of Byte;
  buffer_size: UInt16;
  i: Integer;
begin
  FillChar(request, sizeof(request), 0);
  FillChar(response, sizeof(response), 0);
  FillChar(confirmation, sizeof(confirmation), 0);
  buffer_size := 4093;
  Writeln(sLineBreak + sLineBreak +
    '*** UDS Service: TransferData with MAX_DATA ***');

  // Sends a physical TransferData message with the maximum data available. The goal is to show that
  // WaitForService_2013 does not return a TIMEOUT error although the transmit and receive time of all the
  // data will be longer than the default time to get a response.
  Writeln(Format(sLineBreak + sLineBreak +
    'Sends a physical TransferData message (length=%d): ', [buffer_size]));

  For i := 0 To buffer_size - 1 do
    buffer[i] := (Ord('A') + i) And $FF;

  status := TUDSApi.SvcTransferData_2013(channel, config, request, $1,
    PByte(@buffer), buffer_size);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcTransferData_2013: %d', [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();
end;

/// <summary>Call UDS Service TransferData</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testTransferDataMultipleFunctionalMessage(channel: cantp_handle;
  config: uds_msgconfig);

var
  status: uds_status;
  request: uds_msg;
  confirmation: uds_msg;
  tresponse: array [0 .. 0] of uds_msg;
  empty_msg: uds_msg;
  buffer: array [0 .. 4] of Byte;
  buffer_size: UInt32;
  response_count: UInt32;
  i: Integer;
  max_msg_count: UInt32;
begin
  FillChar(request, sizeof(request), 0);
  FillChar(empty_msg, sizeof(empty_msg), 0);
  FillChar(confirmation, sizeof(confirmation), 0);
  max_msg_count := 1;
  For i := 0 To max_msg_count - 1 do
    tresponse[i] := empty_msg;
  buffer_size := 5;
  response_count := 0;
  Writeln(sLineBreak + sLineBreak +
    '*** UDS Service: TransferData with functional message***');

  // Initialize request message
  config.nai.target_addr :=
    UInt16(uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_OBD_FUNCTIONAL);
  config.nai.target_type :=
    cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_FUNCTIONAL;

  // Sends a functional TransferData message. The goal is to show that UDS_WaitForServiceFunctional_2013 waits long
  // enough to fetch all possible ECU responses.
  Writeln(sLineBreak + sLineBreak +
    'Sends a functional TransferData message: ');

  For i := 0 To buffer_size - 1 do
    buffer[i] := (Ord('A') + i) And $FF;

  status := TUDSApi.SvcTransferData_2013(channel, config, request, $1,
    PByte(@buffer), buffer_size);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForServiceFunctional_2013(channel, @request,
      max_msg_count, True, Puds_msg(@tresponse), response_count, @confirmation);
  Writeln(Format(' UDS_SvcTransferData_2013: %d', [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
  begin
    display_uds_msg_request(@request, True);
    Writeln(Format(sLineBreak + ' Received %d UDS responses:',
      [response_count]));
    display_uds_msg_response(@(tresponse[0]), False);
  end
  Else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(tresponse[0]);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();
End;

/// <summary>Sample to use event</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testUsingEvent(channel: cantp_handle; config: uds_msgconfig);
var
  status: uds_status;
  read_status: uds_status;
  request: uds_msg;
  response: uds_msg;
  receive_event: THANDLE;
  empty_event: THANDLE;
  res: UInt32;
  stop: bool;
begin
  FillChar(request, sizeof(request), 0);
  FillChar(response, sizeof(response), 0);

  // set event handler
  receive_event := CreateEvent(nil, False, False, '');
  status := TUDSApi.SetValue_2013(channel, PUDS_PARAMETER_RECEIVE_EVENT,
    PByte(@receive_event), sizeof(receive_event));
  if not TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False) then
  begin
    Writeln('Failed to set event, aborting...');
    CloseHandle(receive_event);
    waitGetch();
  end
  else
  begin
    Writeln('*** UDS Service with Event: TesterPresent ***');

    // Sends a physical TesterPresent message
    Writeln('Sends a physical TesterPresent message: ');
    status := TUDSApi.SvcTesterPresent_2013(channel, config, &request,
      PUDS_SVC_PARAM_TP_ZSUBF);
    Writeln(Format(' UDS_SvcTesterPresent_2013: %d', [Integer(status)]));

    if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    begin

      // Instead of calling WaitForService function, this sample demonstrates how event can be used. But note that
      // the use of a thread listening to notifications and making the read operations is preferred.
      stop := False;

      // wait until we receive expected response
      repeat
        res := WaitForSingleObject(receive_event, INFINITE);
        if res = WAIT_OBJECT_0 then
        begin

          // read all messages
          repeat
            read_status := TUDSApi.Read_2013(channel, &response);
            if (TUDSApi.StatusIsOk_2013(read_status, PUDS_STATUS_OK, False))
            then
            begin
              // this is a simple message check (type and sender/receiver address): to filter UDS request
              // confirmation and get first message from target, but real use-case should check that the UDS
              // service ID matches the request
              if (response.msg.msgdata_isotp.netaddrinfo.msgtype =
                PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC) and
                (response.msg.msgdata_isotp.netaddrinfo.source_addr = config.
                nai.target_addr) and
                (response.msg.msgdata_isotp.netaddrinfo.target_addr = config.
                nai.source_addr) then
              begin
                stop := True;
                display_uds_msg(@request, @response, False);
              end;
            end;

            // Free response message
            status := TUDSApi.MsgFree_2013(&response);
            Writeln(Format(' Free response message: %d', [Integer(status)]));
          until TUDSApi.StatusIsOk_2013(read_status, PUDS_STATUS_NO_MESSAGE);
        end;
      until stop;
    end;
    waitGetch();

    // Free request message
    status := TUDSApi.MsgFree_2013(&request);
    Writeln(Format(' Free request message: %d', [Integer(status)]));

    // Uninitialize event
    empty_event := 0;
    status := TUDSApi.SetValue_2013(channel, PUDS_PARAMETER_RECEIVE_EVENT,
      PByte(@empty_event), sizeof(empty_event));
    Writeln(Format(' Remove receive event: %d', [Integer(status)]));
    CloseHandle(receive_event);
  end;
End;

/// <summary>Call UDS Service RequestFileTransfer</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testRequestFileTransfer(channel: cantp_handle; config: uds_msgconfig);
var
  status: uds_status;
  request: uds_msg;
  response: uds_msg;
  confirmation: uds_msg;
  file_size_uncompressed: array [0 .. 1] of Byte;
  file_size_compressed: array [0 .. 1] of Byte;
  file_size_length: Byte;
  sfile_name: AnsiString;
  file_name_length: UInt16;
begin
  file_size_uncompressed[0] := $D;
  file_size_uncompressed[1] := $0;
  file_size_compressed[0] := $A;
  file_size_compressed[1] := $0;
  file_size_length := 2;
  sfile_name := 'toto.txt';
  file_name_length := length(sfile_name);
  FillChar(request, sizeof(request), 0);
  FillChar(response, sizeof(response), 0);
  FillChar(confirmation, sizeof(confirmation), 0);

  Writeln(sLineBreak + sLineBreak + '*** UDS Service: RequestFileTransfer ***');

  // Sends a physical RequestFileTransfer message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical RequestFileTransfer message: ');
  status := TUDSApi.SvcRequestFileTransfer_2013(channel, config, request,
    uds_svc_param_rft_moop.PUDS_SVC_PARAM_RFT_MOOP_REPLFILE, file_name_length,
    PAnsiChar(sfile_name), Byte(0), Byte(0), file_size_length,
    PByte(@file_size_uncompressed), PByte(@file_size_compressed));
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcRequestFileTransfer_2013: %d', [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();
end;

/// <summary>Call UDS Service AccessTimingParameter</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testAccessTimingParameter(channel: cantp_handle;
  config: uds_msgconfig);
var
  status: uds_status;
  request: uds_msg;
  response: uds_msg;
  confirmation: uds_msg;
  request_record: array [0 .. 1] of Byte;
  record_size: UInt32;
begin
  FillChar(request, sizeof(request), 0);
  FillChar(response, sizeof(response), 0);
  FillChar(confirmation, sizeof(confirmation), 0);
  request_record[0] := $AB;
  request_record[1] := $CD;
  record_size := 2;

  Writeln(sLineBreak + sLineBreak +
    '*** UDS Service: AccessTimingParameter ***');

  // Sends a physical AccessTimingParameter message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical AccessTimingParameter message: ');
  status := TUDSApi.SvcAccessTimingParameter_2013(channel, config, request,
    uds_svc_param_atp.PUDS_SVC_PARAM_ATP_RCATP, PByte(@request_record),
    record_size);
  If TUDSApi.StatusIsOk_2013(status) Then
    status := TUDSApi.WaitForService_2013(channel, @request, response,
      @confirmation);
  Writeln(Format(' UDS_SvcAccessTimingParameter_2013: %d', [Integer(status)]));

  If TUDSApi.StatusIsOk_2013(status) Then
    display_uds_msg(@confirmation, @response, False)
  Else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch()
end;

/// <summary>Call UDS Service Authentication</summary>
/// <param name="channel">cantp channel handle</param>
/// <param name="config">Configuration of the request message (type, network address information...)</param>
Procedure testAuthentication(channel: cantp_handle; config: uds_msgconfig);
var
  status: uds_status;
  request: uds_msg;
  response: uds_msg;
  confirmation: uds_msg;

  communication_configuration: Byte;
  certificate_client: array [0 .. 1] of Byte;
  certificate_client_size: UInt16;
  challenge_client: array [0 .. 1] of Byte;
  challenge_client_size: UInt16;
  proof_of_ownership_client: array [0 .. 1] of Byte;
  proof_of_ownership_client_size: UInt16;
  ephemeral_public_key_client: array [0 .. 1] of Byte;
  ephemeral_public_key_client_size: UInt16;
  algorithm_indicator: array [0 .. 15] of Byte;
  additional_parameter: array [0 .. 1] of Byte;
  additional_parameter_size: UInt16;
begin
  communication_configuration := 0;
  certificate_client[0] := $12;
  certificate_client[1] := $34;
  certificate_client_size := 2;
  challenge_client[0] := $56;
  challenge_client[1] := $78;
  challenge_client_size := 2;
  proof_of_ownership_client[0] := $9A;
  proof_of_ownership_client[1] := $BC;
  proof_of_ownership_client_size := 2;
  ephemeral_public_key_client[0] := $DE;
  ephemeral_public_key_client[1] := $F0;
  ephemeral_public_key_client_size := 2;
  algorithm_indicator[0] := $00;
  algorithm_indicator[1] := $01;
  algorithm_indicator[2] := $02;
  algorithm_indicator[3] := $03;
  algorithm_indicator[4] := $04;
  algorithm_indicator[5] := $05;
  algorithm_indicator[6] := $06;
  algorithm_indicator[7] := $07;
  algorithm_indicator[8] := $08;
  algorithm_indicator[9] := $09;
  algorithm_indicator[10] := $0A;
  algorithm_indicator[11] := $0B;
  algorithm_indicator[12] := $0C;
  algorithm_indicator[13] := $0D;
  algorithm_indicator[14] := $0E;
  algorithm_indicator[15] := $0F;

  additional_parameter[0] := $AA;
  additional_parameter[1] := $BB;
  additional_parameter_size := 2;

  FillChar(request, sizeof(request), 0);
  FillChar(response, sizeof(response), 0);
  FillChar(confirmation, sizeof(confirmation), 0);

  Writeln(sLineBreak + sLineBreak + '*** UDS Service: Authentication ***');

  // Sends a physical Authentication/deAuthenticate message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical Authentication/deAuthenticate message: ');
  status := TUDSApi.SvcAuthenticationDA_2020(channel, config, &request);
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    status := TUDSApi.WaitForService_2013(channel, @request, &response,
      @confirmation);
  Writeln(Format(' UDS_SvcAuthenticationDA_2020: %d', [Integer(status)]));
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    display_uds_msg(@confirmation, @response, False)
  else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(&request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(&response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(&confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a physical Authentication/verifyCertificateUnidirectional message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical Authentication/verifyCertificateUnidirectional message: ');
  status := TUDSApi.SvcAuthenticationVCU_2020(channel, config, &request,
    communication_configuration, PByte(@certificate_client),
    certificate_client_size, PByte(@challenge_client), challenge_client_size);
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    status := TUDSApi.WaitForService_2013(channel, @request, &response,
      @confirmation);
  Writeln(Format(' UDS_SvcAuthenticationVCU_2020: %d', [Integer(status)]));
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    display_uds_msg(@confirmation, @response, False)
  else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(&request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(&response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(&confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a physical Authentication/verifyCertificateBidirectional message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical Authentication/verifyCertificateBidirectional message: ');
  status := TUDSApi.SvcAuthenticationVCB_2020(channel, config, &request,
    communication_configuration, PByte(@certificate_client),
    certificate_client_size, PByte(@challenge_client), challenge_client_size);
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    status := TUDSApi.WaitForService_2013(channel, @request, &response,
      @confirmation);
  Writeln(Format(' UDS_SvcAuthenticationVCB_2020: %d', [Integer(status)]));
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    display_uds_msg(@confirmation, @response, False)
  else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(&request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(&response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(&confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a physical Authentication/proofOfOwnership message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical Authentication/proofOfOwnership message: ');
  status := TUDSApi.SvcAuthenticationPOWN_2020(channel, config, &request,
    PByte(@proof_of_ownership_client), proof_of_ownership_client_size,
    PByte(@ephemeral_public_key_client), ephemeral_public_key_client_size);
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    status := TUDSApi.WaitForService_2013(channel, @request, &response,
      @confirmation);
  Writeln(Format(' UDS_SvcAuthenticationPOWN_2020: %d', [Integer(status)]));
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    display_uds_msg(@confirmation, @response, False)
  else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(&request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(&response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(&confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a physical Authentication/requestChallengeForAuthentication message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical Authentication/requestChallengeForAuthentication message: ');
  status := TUDSApi.SvcAuthenticationRCFA_2020(channel, config, &request,
    communication_configuration, PByte(@algorithm_indicator));
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    status := TUDSApi.WaitForService_2013(channel, @request, &response,
      @confirmation);
  Writeln(Format(' UDS_SvcAuthenticationRCFA_2020: %d', [Integer(status)]));
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    display_uds_msg(@confirmation, @response, False)
  else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(&request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(&response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(&confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a physical Authentication/verifyProofOfOwnershipUnidirectional message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical Authentication/verifyProofOfOwnershipUnidirectional message: ');
  status := TUDSApi.SvcAuthenticationVPOWNU_2020(channel, config, &request,
    PByte(@algorithm_indicator), PByte(@proof_of_ownership_client),
    proof_of_ownership_client_size, PByte(@challenge_client),
    challenge_client_size, PByte(@additional_parameter),
    additional_parameter_size);
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    status := TUDSApi.WaitForService_2013(channel, @request, &response,
      @confirmation);
  Writeln(Format(' UDS_SvcAuthenticationVPOWNU_2020: %d', [Integer(status)]));
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    display_uds_msg(@confirmation, @response, False)
  else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(&request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(&response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(&confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a physical Authentication/verifyProofOfOwnershipBidirectional message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical Authentication/verifyProofOfOwnershipBidirectional message: ');
  status := TUDSApi.SvcAuthenticationVPOWNB_2020(channel, config, &request,
    PByte(@algorithm_indicator), PByte(@proof_of_ownership_client),
    proof_of_ownership_client_size, PByte(@challenge_client),
    challenge_client_size, PByte(@additional_parameter),
    additional_parameter_size);
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    status := TUDSApi.WaitForService_2013(channel, @request, &response,
      @confirmation);
  Writeln(Format(' UDS_SvcAuthenticationVPOWNB_2020: %d', [Integer(status)]));
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    display_uds_msg(@confirmation, @response, False)
  else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(&request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(&response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(&confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();

  // Sends a physical Authentication/authenticationConfiguration message
  Writeln(sLineBreak + sLineBreak +
    'Sends a physical Authentication/authenticationConfiguration message: ');
  status := TUDSApi.SvcAuthenticationAC_2020(channel, config, &request);
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    status := TUDSApi.WaitForService_2013(channel, @request, &response,
      @confirmation);
  Writeln(Format(' UDS_SvcAuthenticationAC_2020: %d', [Integer(status)]));
  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, False)) then
    display_uds_msg(@confirmation, @response, False)
  else
    display_uds_msg_request(@request, False);

  // Free messages
  status := TUDSApi.MsgFree_2013(&request);
  Writeln(Format(' Free request message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(&response);
  Writeln(Format(' Free response message: %d', [Integer(status)]));
  status := TUDSApi.MsgFree_2013(&confirmation);
  Writeln(Format(' Free confirmation message: %d', [Integer(status)]));
  waitGetch();
end;

/// <summary>Entry point of the program, start a CAN UDS client</summary>
var
  client_handle: cantp_handle;
  status: uds_status;
  timeout_value: UInt32;
  config: uds_msgconfig;
  dummy: Char;

begin
  try

    // Set the PCAN-Channel to use
    client_handle := cantp_handle.PCANTP_HANDLE_USBBUS1;
    // TODO: modify the value according to your available PCAN devices.

    // Initializing of the UDS Communication session
    status := TUDSApi.Initialize_2013(client_handle,
      cantp_baudrate.PCANTP_BAUDRATE_500K);
    Writeln(Format('Initialize UDS: %d', [Integer(status)]));

    // Define TimeOuts
    timeout_value := TCantpApi.PCANTP_ISO_TIMEOUTS_15765_4;
    status := TUDSApi.SetValue_2013(client_handle, PUDS_PARAMETER_ISO_TIMEOUTS,
      PLongWord(@timeout_value), sizeof(timeout_value));
    Writeln(Format(' Set ISO 15765-4 timeouts values: %d', [Integer(status)]));
    waitGetch();

    // Define Network Address Information used for all the tests
    FillChar(config, sizeof(config), 0);
    config.can_id := $FFFFFFFF;
    config.can_msgtype := cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD;
    config.nai.protocol :=
      uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL;
    config.nai.target_type :=
      cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;
    config.typem := uds_msgtype.PUDS_MSGTYPE_USDT;
    config.nai.source_addr :=
      UInt16(uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT);
    config.nai.target_addr :=
      UInt16(uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_1);

    // The following functions call UDS Services
    testDiagnosticSessionControl(client_handle, config);

    testECUReset(client_handle, config);
    testSecurityAccess(client_handle, config);
    testCommunicationControl(client_handle, config);
    testTesterPresent(client_handle, config);
    testSecuredDataTransmission(client_handle, config);
    testControlDTCSetting(client_handle, config);
    testResponseOnEvent(client_handle, config);
    testLinkControl(client_handle, config);
    testReadDataByIdentifier(client_handle, config);
    testReadMemoryByAddress(client_handle, config);
    testReadScalingDataByIdentifier(client_handle, config);
    testReadDataByPeriodicIdentifier(client_handle, config);
    testDynamicallyDefineDataIdentifier(client_handle, config);
    testWriteDataByIdentifier(client_handle, config);
    testWriteMemoryByAddress(client_handle, config);
    testClearDiagnosticInformation(client_handle, config);
    testReadDTCInformation(client_handle, config);
    testInputOutputControlByIdentifier(client_handle, config);
    testRoutineControl(client_handle, config);
    testRequestDownload(client_handle, config);
    testRequestUpload(client_handle, config);
    testTransferData(client_handle, config);
    testRequestTransferExit(client_handle, config);
    testAccessTimingParameter(client_handle, config);
    testRequestFileTransfer(client_handle, config);
    testAuthentication(client_handle, config);

    // Miscellaneous examples
    testTransferDataBigMessage(client_handle, config);
    testTransferDataMultipleFunctionalMessage(client_handle, config);
    testUsingEvent(client_handle, config);

    // Uninitialize channel
    status := TUDSApi.Uninitialize_2013(client_handle);
    Writeln(Format(' Unitialize channel: %d', [Integer(status)]));

    // Display a small report and quit
    If g_nbErr > 0 Then
      Writeln(Format(sLineBreak + 'ERROR: %d errors occurred.', [g_nbErr]))
    Else
      Writeln(sLineBreak + 'ALL Transmissions succeeded !');

    Writeln(sLineBreak + sLineBreak + 'Press <Enter> to quit...');
    readln(dummy);

  except
    on E: Exception do
      Writeln(E.Classname, ': ', E.Message);
  end;

end.
