program read_write_uds;

{$APPTYPE CONSOLE}

uses
  SysUtils,
  ShellApi,
  windows,
  PUDS_2013 in 'PUDS_2013.pas',
  PCANBasic in 'PCANBasic.pas',
  PCANTP_2016 in 'PCANTP_2016.pas';

const
  PCAN_BITRATE_SAE_J2284_4: AnsiString = 'f_clock=80000000,nom_brp=2,nom_tseg1=63,nom_tseg2=16,nom_sjw=16,data_brp=2,data_tseg1=15,data_tseg2=4,data_sjw=4';
  // Define Bitrate: SAE J2284-4: High-Speed CAN (HSC) for Vehicle Applications at 500 kbps with CAN FD Data at 2 Mbps
  MSG: String = 'PEAK';
  MSG_SIZE: UInt32 = 4;

Function OK_KO(test: Boolean): String;
begin
  If test Then
    result := 'OK'
  else
    result := 'KO';
End;

Function STATUS_OK_KO(test: uds_status): String;
begin
  result := OK_KO(TUDSApi.StatusIsOk_2013(test));
End;

/// <summary>Helper: Check if a mapping list contains a certain mapping</summary>
/// <param name="mapping_list">Mapping list</param>
/// <param name="mapping_list_size">Number of mapping in the mapping list</param>
/// <param name="searched_mapping">Searched mapping</param>
/// <returns>Present (true) or not (false)</returns>
Function mapping_list_contains(mapping_list: Puds_mapping; mapping_list_size: UInt16; searched_mapping: Puds_mapping): Boolean;
var
  res: Boolean;
  i: integer;
  mapPointer: Puds_mapping;
begin
  res := False;
  mapPointer := mapping_list;
  For i := 0 To mapping_list_size - 1 do
  begin

    // If unique identifier are the same, the mapping is in the list
    If mapPointer^.uid = searched_mapping.uid Then
    begin
      res := True;
      break;
    End;
    inc(mapPointer);
  end;
  result := res;
End;

/// <summary>Entry point of the program, start a small CAN UDS read/write example</summary>
var
  buffer: array [0 .. 255] of ansichar;
  status: uds_status;
  transmitter_handle: cantp_handle;
  receiver_handle: cantp_handle;
  tx_msg: uds_msg;
  rx_msg: uds_msg;
  msg_config: uds_msgconfig;
  source_mapping: uds_mapping;
  response_mapping: uds_mapping;
  mapping_buffer: array [0 .. 255] of uds_mapping;
  mapping_count: UInt16;
  wait_result: UInt32;
  receive_event: THandle;

  msgok: Boolean;
  i: UInt32;
  data_pointer: PByte;
  null_handle: THandle;
  dummy: char;

begin
  try

    // Initialize variables
    transmitter_handle := cantp_handle.PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.
    receiver_handle := cantp_handle.PCANTP_HANDLE_USBBUS2; // TODO: modify the value according to your available PCAN devices.
    mapping_count := 0;
    FillChar(tx_msg, sizeof(tx_msg), 0);
    FillChar(rx_msg, sizeof(rx_msg), 0);
    buffer[0] := #0;

    // Print version informations
    status := TUDSApi.GetValue_2013(cantp_handle.PCANTP_HANDLE_NONEBUS, uds_parameter.PUDS_PARAMETER_API_VERSION, buffer, 256);
    Writeln(Format('PCAN-UDS API Version: %s (%s)', [buffer, STATUS_OK_KO(status)]));

    // Initialize transmitter and receiver
    status := TUDSApi.InitializeFD_2013(transmitter_handle, PAnsiChar(PCAN_BITRATE_SAE_J2284_4));
    Writeln(Format('Initialize transmitter: %s', [STATUS_OK_KO(status)]));
    status := TUDSApi.InitializeFD_2013(receiver_handle, PAnsiChar(PCAN_BITRATE_SAE_J2284_4));
    Writeln(Format('Initialize receiver: %s', [STATUS_OK_KO(status)]));

    // Create a receive event on receiver
    receive_event := CreateEvent(nil, False, False, '');
    status := TUDSApi.SetValue_2013(receiver_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, PByte(@receive_event), sizeof(receive_event));

    Writeln(Format('Set receive event parameter: %s', [STATUS_OK_KO(status)]));

    // Initialize 29bits
    FillChar(msg_config, sizeof(msg_config), 0);
    msg_config.can_id := $30;
    msg_config.can_msgtype := cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD;
    // To use CAN FD with bitrate switch uncomment:
    // msg_config.can_msgtype := cantp_can_msgtype(UInt32(msg_config.can_msgtype) or UInt32(PCANTP_CAN_MSGTYPE_FD) or UInt32(PCANTP_CAN_MSGTYPE_BRS));
    msg_config.nai.protocol := uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_29B_NORMAL;
    msg_config.nai.target_type := cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;
    msg_config.typem := uds_msgtype.PUDS_MSGTYPE_USDT;
    msg_config.nai.source_addr := $10;
    msg_config.nai.target_addr := $20;

    // Allocate TX message with previous configuration
    status := TUDSApi.MsgAlloc_2013(tx_msg, msg_config, MSG_SIZE);
    if TUDSApi.StatusIsOk_2013(status) then
    begin
      data_pointer := PByte(tx_msg.MSG.msgdata_any^.data);
      for i := 1 to MSG_SIZE do
        begin
          data_pointer^ := Byte(MSG[i]);
          inc(data_pointer);
        end;
    end;
    Writeln(Format('Allocate TX message: %s', [STATUS_OK_KO(status)]));

    // Initialize source mapping
    FillChar(source_mapping, sizeof(source_mapping), 0);
    source_mapping.can_id := msg_config.can_id;
    source_mapping.can_id_flow_ctrl := msg_config.can_id + 1;
    source_mapping.can_msgtype := msg_config.can_msgtype;
    source_mapping.can_tx_dlc := 8;
    source_mapping.nai := msg_config.nai;

    // Initialize response mapping
    FillChar(response_mapping, sizeof(response_mapping), 0);
    response_mapping := source_mapping;
    response_mapping.can_id := source_mapping.can_id_flow_ctrl;
    response_mapping.can_id_flow_ctrl := source_mapping.can_id;
    response_mapping.nai.source_addr := source_mapping.nai.target_addr;
    response_mapping.nai.target_addr := source_mapping.nai.source_addr;

    // Add mappings on transmitter
    status := TUDSApi.AddMapping_2013(transmitter_handle, @source_mapping);
    Writeln(Format('Add source mapping on transmitter: %s', [STATUS_OK_KO(status)]));
    status := TUDSApi.AddMapping_2013(transmitter_handle, @response_mapping);
    Writeln(Format('Add response mapping on transmitter: %s', [STATUS_OK_KO(status)]));

    // Check if mappings are added on transmitter
    status := TUDSApi.GetMappings_2013(transmitter_handle, @mapping_buffer, 256, @mapping_count);
    Writeln(Format('Get mappings on transmitter: %s', [STATUS_OK_KO(status)]));
    Writeln(Format('Check added mappings on transmitter: %s', [OK_KO(mapping_list_contains(@mapping_buffer, mapping_count, @source_mapping) And mapping_list_contains(@mapping_buffer, mapping_count,
      @response_mapping))]));

    // Add mappings on receiver
    status := TUDSApi.AddMapping_2013(receiver_handle, @source_mapping);
    Writeln(Format('Add source mapping on receiver: %s', [STATUS_OK_KO(status)]));
    status := TUDSApi.AddMapping_2013(receiver_handle, @response_mapping);
    Writeln(Format('Add response mapping on receiver: %s', [STATUS_OK_KO(status)]));

    // Check if mappings are added on receiver
    status := TUDSApi.GetMappings_2013(receiver_handle, @mapping_buffer, 256, @mapping_count);
    Writeln(Format('Get mappings on receiver: %s', [STATUS_OK_KO(status)]));
    Writeln(Format('Check added mappings on receiver: %s', [OK_KO(mapping_list_contains(@mapping_buffer, mapping_count, @source_mapping) And mapping_list_contains(@mapping_buffer, mapping_count,
      @response_mapping))]));

    // Write message
    status := TUDSApi.Write_2013(transmitter_handle, @tx_msg);
    Writeln(Format('Write "PEAK": %s', [STATUS_OK_KO(status)]));

    // Wait a receive event
    wait_result := WaitForSingleObject(receive_event, 3000);
    Writeln(Format('Wait a receive event: %s', [OK_KO(wait_result = WAIT_OBJECT_0)]));

    // If we get a receive event
    if wait_result = WAIT_OBJECT_0 then
    begin
      // Read message
      status := TUDSApi.Read_2013(receiver_handle, rx_msg);
      Writeln(Format('Receive message: %s', [STATUS_OK_KO(status)]));

      // Check message (should contains "PEAK")
      msgok := False;

      If (TUDSApi.StatusIsOk_2013(status)) And (rx_msg.MSG.msgdata_any^.length = MSG_SIZE) Then
      begin
        msgok := (rx_msg.MSG.msgdata_any^.data <> nil);
        If msgok Then
        begin
          i := 1;
          data_pointer := PByte(rx_msg.MSG.msgdata_any^.data);
          While msgok And (i <= MSG_SIZE) do
          begin
            If data_pointer^ = Byte(MSG[i]) Then
              msgok := True
            Else
              msgok := False;
            i := i+1;
            inc(data_pointer);
          End;
        End;
      End;

      If msgok Then
        Writeln(Format('Message contains "%s": OK', [MSG]))
      Else
        Writeln('Message is corrupted: KO');

    End;

    // Close receive event
    null_handle := 0;
    status := TUDSApi.SetValue_2013(receiver_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, PByte(@null_handle), sizeof(null_handle));

    Writeln(Format('Stop receive event receiver: %s', [STATUS_OK_KO(status)]));
    CloseHandle(receive_event);
    Writeln(Format('Close receive event: %s', [STATUS_OK_KO(status)]));

    // Free messages
    status := TUDSApi.MsgFree_2013(tx_msg);
    Writeln(Format('Free TX message: %s', [STATUS_OK_KO(status)]));
    status := TUDSApi.MsgFree_2013(rx_msg);
    Writeln(Format('Free RX message: %s', [STATUS_OK_KO(status)]));

    // Close transmitter and receiver
    status := TUDSApi.Uninitialize_2013(transmitter_handle);
    Writeln(Format('Uninitialize transmitter: %s', [STATUS_OK_KO(status)]));
    status := TUDSApi.Uninitialize_2013(receiver_handle);;
    Writeln(Format('Uninitialize receiver: %s', [STATUS_OK_KO(status)]));

    // Exit
    Writeln('Press any key to continue...');
    Readln(dummy);

  except
    on E: Exception do
      Writeln(E.Classname, ': ', E.Message);
  end;

end.
