program read_write_uds_and_isotp;

{$APPTYPE CONSOLE}

uses
  SysUtils,
  ShellApi,
  windows,
  PUDS_2013 in 'PUDS_2013.pas',
  PCANBasic in 'PCANBasic.pas',
  PCANTP_2016 in 'PCANTP_2016.pas';

const
  // Define Bitrate: SAE J2284-4: High-Speed CAN (HSC) for Vehicle Applications at 500 kbps with CAN FD Data at 2 Mbps
  PCAN_BITRATE_SAE_J2284_4
    : AnsiString =
    'f_clock=80000000,nom_brp=2,nom_tseg1=63,nom_tseg2=16,nom_sjw=16,data_brp=2,data_tseg1=15,data_tseg2=4,data_sjw=4';
  BUFFER_SIZE: UInt32 = 256;
  MSG: array [0 .. 3] of Byte = (Byte('P'), Byte('E'), Byte('A'), Byte('K'));
  MSG_SIZE: UInt32 = 4;

Function OK_KO(test: Boolean): String;
begin
  If test Then
    result := 'OK'
  else
    result := 'KO';
End;

Function STATUS_OK_KO(test: integer): String;
begin
  result := OK_KO(TUDSApi.StatusIsOk_2013(uds_status(test)));
End;

/// <summary>
/// Entry point of the program, start a small CAN UDS / ISOTP read-write example
/// This example initialise two UDS channel with a specific mapping.
/// Then it inits an ISOTP mapping and write an ISOTP message.
/// In this case, PCAN-UDS should not receive the message, we need PCAN-ISOTP to read it.
/// </summary>
/// <returns>By convention, return success.</returns>
var
  buffer: array [0 .. 255] of ansichar;
  status: integer;
  transmitter_handle: cantp_handle;
  receiver_handle: cantp_handle;
  tx_msg: cantp_msg;
  uds_rx_msg: uds_msg;
  rx_msg: cantp_msg;
  source_mapping: uds_mapping;
  mapping: cantp_mapping;
  reverse_mapping: cantp_mapping;
  response_mapping: uds_mapping;
  receive_event: THANDLE;
  wait_result: UInt32;
  null_handle: THANDLE;
  msgok: Boolean;
  data_pointer: PByte;
  i: UInt32;
  dummy: char;
  can_msgtype: cantp_can_msgtype;
begin
  try

    // Initialize variables
    null_handle := 0;
    FillChar(response_mapping, sizeof(response_mapping), 0);
    FillChar(source_mapping, sizeof(source_mapping), 0);
    FillChar(tx_msg, sizeof(tx_msg), 0);
    FillChar(uds_rx_msg, sizeof(uds_rx_msg), 0);
    FillChar(mapping, sizeof(mapping), 0);
    FillChar(reverse_mapping, sizeof(reverse_mapping), 0);
    FillChar(rx_msg, sizeof(rx_msg), 0);
    buffer[0] := #0;

    // TODO: modify the value according to your available PCAN devices.
    transmitter_handle := PCANTP_HANDLE_USBBUS1;

    // TODO: modify the value according to your available PCAN devices.
    receiver_handle := PCANTP_HANDLE_USBBUS2;

    can_msgtype := PCANTP_CAN_MSGTYPE_STANDARD;
    // To use CAN FD with bitrate switch uncomment:
    // can_msgtype := cantp_can_msgtype(UInt32(can_msgtype) or UInt32(PCANTP_CAN_MSGTYPE_FD) or UInt32(PCANTP_CAN_MSGTYPE_BRS));

    // Print version informations
    status := integer(TUDSApi.GetValue_2013(PCANTP_HANDLE_NONEBUS,
      PUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE));
    Writeln(Format('PCAN-UDS API Version: %s (%s)',
      [buffer, STATUS_OK_KO(status)]));

    // Initialize transmitter and receiver
    status := integer(TUDSApi.InitializeFD_2013(transmitter_handle,
      PAnsiChar(PCAN_BITRATE_SAE_J2284_4)));
    Writeln(Format('Initialize transmitter: %s', [STATUS_OK_KO(status)]));
    status := integer(TUDSApi.InitializeFD_2013(receiver_handle,
      PAnsiChar(PCAN_BITRATE_SAE_J2284_4)));
    Writeln(Format('Initialize receiver: %s', [STATUS_OK_KO(status)]));

    // Create a receive event on receiver
    receive_event := CreateEvent(nil, FALSE, FALSE, nil);
    status := integer(TUDSApi.SetValue_2013(receiver_handle,
      PUDS_PARAMETER_RECEIVE_EVENT, PByte(@receive_event),
      sizeof(receive_event)));
    Writeln(Format('Set receive event parameter: %s', [STATUS_OK_KO(status)]));

    // Initialize source mapping
    source_mapping.can_id := $30;
    source_mapping.can_id_flow_ctrl := source_mapping.can_id + 1;
    source_mapping.can_msgtype := can_msgtype;
    source_mapping.can_tx_dlc := 8;
    source_mapping.nai.protocol := PUDS_MSGPROTOCOL_ISO_15765_2_29B_NORMAL;
    source_mapping.nai.target_type := PCANTP_ISOTP_ADDRESSING_PHYSICAL;
    source_mapping.nai.source_addr := $10;
    source_mapping.nai.target_addr := $20;

    // Initialize response mapping
    response_mapping := source_mapping;
    response_mapping.can_id := source_mapping.can_id_flow_ctrl;
    response_mapping.can_id_flow_ctrl := source_mapping.can_id;
    response_mapping.nai.source_addr := source_mapping.nai.target_addr;
    response_mapping.nai.target_addr := source_mapping.nai.source_addr;

    // Add UDS mappings on transmitter
    status := integer(TUDSApi.AddMapping_2013(transmitter_handle,
      @source_mapping));
    Writeln(Format('Add source mapping on transmitter: %s',
      [STATUS_OK_KO(status)]));
    status := integer(TUDSApi.AddMapping_2013(transmitter_handle,
      @response_mapping));
    Writeln(Format('Add response mapping on transmitter: %s',
      [STATUS_OK_KO(status)]));

    // Add UDS mappings on and receiver
    status := integer(TUDSApi.AddMapping_2013(receiver_handle,
      @source_mapping));
    Writeln(Format('Add source mapping on receiver: %s',
      [STATUS_OK_KO(status)]));
    status := integer(TUDSApi.AddMapping_2013(receiver_handle,
      @response_mapping));
    Writeln(Format('Add response mapping on receiver: %s',
      [STATUS_OK_KO(status)]));

    // Create a simple isotp physical mapping:
    // - Source 0xF1 (transmitter), target 0x01 (receiver), CAN id 0xA1, CAN ID flow control 0xA2
    // - Diagnostic message in a classic format
    mapping.can_id := $A1;
    mapping.can_id_flow_ctrl := $A2;
    mapping.can_msgtype := can_msgtype;
    mapping.can_tx_dlc := $0;
    mapping.netaddrinfo.extension_addr := $0;
    mapping.netaddrinfo.Format := PCANTP_ISOTP_FORMAT_NORMAL;
    mapping.netaddrinfo.msgtype := PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC;
    mapping.netaddrinfo.source_addr := $F1;
    mapping.netaddrinfo.target_addr := $01;
    mapping.netaddrinfo.target_type := PCANTP_ISOTP_ADDRESSING_PHYSICAL;

    // Create the associated isotp reversed mapping:
    reverse_mapping := mapping;
    reverse_mapping.can_id := mapping.can_id_flow_ctrl;
    reverse_mapping.can_id_flow_ctrl := mapping.can_id;
    reverse_mapping.netaddrinfo.source_addr := mapping.netaddrinfo.target_addr;
    reverse_mapping.netaddrinfo.target_addr := mapping.netaddrinfo.source_addr;

    // Add ISOTP mappings on channels
    status := integer(TCanTpApi.AddMapping_2016(transmitter_handle, @mapping));
    Writeln(Format('Add a simple isotp mapping on transmitter: %s',
      [STATUS_OK_KO(status)]));
    status := integer(TCanTpApi.AddMapping_2016(transmitter_handle,
      @reverse_mapping));
    Writeln(Format('Add the reverse isotp mapping on transmitter: %s',
      [STATUS_OK_KO(status)]));
    status := integer(TCanTpApi.AddMapping_2016(receiver_handle, @mapping));
    Writeln(Format('Add a simple isotp mapping on receiver: %s',
      [STATUS_OK_KO(status)]));
    status := integer(TCanTpApi.AddMapping_2016(receiver_handle,
      @reverse_mapping));
    Writeln(Format('Add the reverse isotp mapping on receiver: %s',
      [STATUS_OK_KO(status)]));

    // Initialize ISOTP Tx message containing "PEAK"
    status := integer(TCanTpApi.MsgDataAlloc_2016(tx_msg,
      PCANTP_MSGTYPE_ISOTP));
    Writeln(Format('Allocate tx ISOTP message: %s', [STATUS_OK_KO(status)]));

    status := integer(TCanTpApi.MsgDataInit_2016(&tx_msg, mapping.can_id,
      mapping.can_msgtype, MSG_SIZE, PByte(@MSG), @mapping.netaddrinfo));
    Writeln(Format('Initialize tx message: %s', [STATUS_OK_KO(status)]));

    // Send "PEAK" ISOTP message
    status := integer(TCanTpApi.Write_2016(transmitter_handle, tx_msg));
    Writeln(Format('Write "PEAK": %s', [STATUS_OK_KO(status)]));

    // Wait a receive event on uds message (should not work because we sent a ISOTP message)
    wait_result := WaitForSingleObject(receive_event, 3000);
    Writeln(Format
      ('Wait a receive event on UDS message (should not work because we sent an ISOTP message): %s',
      [OK_KO(wait_result = WAIT_OBJECT_0)]));

    // We should receive a ISOTP message instead
    status := integer(TCanTpApi.Read_2016(receiver_handle, rx_msg, nil,
      PCANTP_MSGTYPE_NONE));
    Writeln(Format('Read ISOTP message on receiver: %s',
      [STATUS_OK_KO(status)]));

    // Check message (should contains "PEAK")
    msgok := FALSE;
    if (TUDSApi.StatusIsOk_2013(uds_status(status), PUDS_STATUS_OK, FALSE)) And
      (rx_msg.msgdata_any^.length = MSG_SIZE) then
    begin
      msgok := (rx_msg.msgdata_any^.data <> nil);
      If msgok Then
      begin
        i := 0;
        data_pointer := PByte(rx_msg.msgdata_any^.data);
        While msgok And (i < MSG_SIZE) do
        begin
          If data_pointer^ = MSG[i] Then
            msgok := True
          Else
            msgok := FALSE;
          i := i + 1;
          inc(data_pointer);
        End;
      End;
    end;
    If msgok Then
    begin
      Writeln('Message contains "PEAK": OK');
    end
    else
    begin
      Writeln('Message is corrupted: KO');
    end;

    // Close receive event
    status := integer(TUDSApi.SetValue_2013(receiver_handle,
      PUDS_PARAMETER_RECEIVE_EVENT, PByte(@null_handle), sizeof(null_handle)));
    Writeln(Format('Stop receive event receiver: %s', [STATUS_OK_KO(status)]));
    CloseHandle(receive_event);
    Writeln(Format('Close receive event: %s', [STATUS_OK_KO(status)]));

    // Free messages
    status := integer(TCanTpApi.MsgDataFree_2016(tx_msg));
    Writeln(Format('Free TX message: %s', [STATUS_OK_KO(status)]));
    status := integer(TCanTpApi.MsgDataFree_2016(rx_msg));
    Writeln(Format('Free RX message: %s', [STATUS_OK_KO(status)]));
    status := integer(TUDSApi.MsgFree_2013(uds_rx_msg));
    Writeln(Format('Free UDS/RX message: %s', [STATUS_OK_KO(status)]));

    // Close transmitter and receiver
    status := integer(TUDSApi.Uninitialize_2013(transmitter_handle));
    Writeln(Format('Uninitialize transmitter: %s', [STATUS_OK_KO(status)]));
    status := integer(TUDSApi.Uninitialize_2013(receiver_handle));
    Writeln(Format('Uninitialize receiver: %s', [STATUS_OK_KO(status)]));

    // Exit
    Writeln('Press any key to continue...');
    Readln(dummy);

  except
    on E: Exception do
      Writeln(E.Classname, ': ', E.Message);
  end;

end.
