program client_uds_and_isotp;

{$APPTYPE CONSOLE}

uses
  SysUtils,
  ShellApi,
  windows,
  PUDS_2013 in 'PUDS_2013.pas',
  PCANBasic in 'PCANBasic.pas',
  PCANTP_2016 in 'PCANTP_2016.pas';

const
  BUFFER_SIZE: Integer = 256;
  ISOTP_REQUEST_MSG: array [0 .. 3] of Byte = (Byte('P'), Byte('I'), Byte('N'),
    Byte('G'));
  ISOTP_RESPONSE_MSG: array [0 .. 3] of Byte = (Byte('P'), Byte('O'), Byte('N'),
    Byte('G'));
  MSG_SIZE: UInt32 = 4;

Function OK_KO(test: Boolean): String;
begin
  if test then
    result := 'OK'
  else
    result := 'KO';
end;

Function UDS_STATUS_OK_KO(test: uds_status): String;
begin
  result := OK_KO(TUDSApi.StatusIsOk_2013(test, PUDS_STATUS_OK, false))
end;

Function ISOTP_STATUS_OK_KO(test: cantp_status): String;
begin
  result := OK_KO(TCanTpApi.StatusIsOk_2016(test, PCANTP_STATUS_OK, false))
end;

/// <summary>UDS client task: request TesterPresent service several time</summary>
/// <param name="param">client channel handle</param>
Procedure uds_client_task(param: pointer); stdcall;
var
  status: uds_status;
  i: Integer;
  config_physical: uds_msgconfig;
  msg_request: uds_msg;
  request_confirmation: uds_msg;
  response: uds_msg;
  client_handle: ^cantp_handle;
begin

  // Initialize variables
  FillChar(config_physical, sizeof(config_physical), 0);
  FillChar(msg_request, sizeof(msg_request), 0);
  FillChar(request_confirmation, sizeof(request_confirmation), 0);
  FillChar(response, sizeof(response), 0);
  client_handle := param;

  // Initialize a physical configuration
  config_physical.can_id :=
    Cardinal(PUDS_CAN_ID_ISO_15765_4_PHYSICAL_REQUEST_1);
  config_physical.can_msgtype := PCANTP_CAN_MSGTYPE_STANDARD;
  config_physical.nai.protocol := PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL;
  config_physical.nai.target_type := PCANTP_ISOTP_ADDRESSING_PHYSICAL;
  config_physical.typem := PUDS_MSGTYPE_USDT;
  config_physical.nai.source_addr :=
    Word(PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT);
  config_physical.nai.target_addr := Word(PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_1);
  config_physical.nai.extension_addr := 0;

  // Execute TesterPresent and wait response
  for i := 1 to 10 do
  begin
    status := TUDSApi.SvcTesterPresent_2013(client_handle^, config_physical,
      &msg_request, PUDS_SVC_PARAM_TP_ZSUBF);
    Writeln(Format('[UDS] Execute TesterPresent service (%d): %s',
      [i, UDS_STATUS_OK_KO(status)]));
    status := TUDSApi.WaitForService_2013(client_handle^, @msg_request,
      &response, @request_confirmation);
    Writeln(Format('[UDS] Wait for service (%d): %s',
      [i, UDS_STATUS_OK_KO(status)]));
    status := TUDSApi.MsgFree_2013(&msg_request);
    Writeln(Format('[UDS] Free request message: %s',
      [UDS_STATUS_OK_KO(status)]));
    status := TUDSApi.MsgFree_2013(&request_confirmation);
    Writeln(Format('[UDS] Free request confirmation message: %s',
      [UDS_STATUS_OK_KO(status)]));
    status := TUDSApi.MsgFree_2013(&response);
    Writeln(Format('[UDS] Free response message: %s',
      [UDS_STATUS_OK_KO(status)]));
  end;
end;

/// <summary>ISOTP: wait an ISOTP "PONG" message</summary>
/// <param name="client_handle">client channel handle</param>
/// <param name="repeat_wait_counter">maximum number of WaitForSingleObject to do (to avoid infinite loop)</param>
/// <returns>if we received a PONG message (true) or not (false)</returns>
Function wait_isotp_pong_response(client_handle: cantp_handle;
  repeat_wait_counter: Integer): Boolean;
var
  status: cantp_status;
  receive_event: THANDLE;
  wait_result: DWORD;
  rx_msg: cantp_msg;
  repeat_iter: Integer;
  get_pong_msg: Boolean;
  msgok: Boolean;
  data_pointer: PByte;
  i: UInt32;
begin

  // Init variables
  FillChar(rx_msg, sizeof(cantp_msg), 0);
  repeat_iter := 0;
  get_pong_msg := false;

  // Get ISOTP receive event
  status := TCanTpApi.GetValue_2016(client_handle,
    PCANTP_PARAMETER_RECEIVE_EVENT, PByte(@receive_event),
    sizeof(receive_event));
  Writeln(Format('[ISOTP] Get isotp receive event parameter: %s',
    [ISOTP_STATUS_OK_KO(status)]));

  // Wait a receive event on isotp message
  repeat
    wait_result := WaitForSingleObject(receive_event, 1000);
    Writeln(Format('[ISOTP] Wait a receive event from isotp: %s',
      [OK_KO(wait_result = WAIT_OBJECT_0)]));

    // Read ISOTP messages
    repeat
      status := TCanTpApi.Read_2016(client_handle, &rx_msg, nil,
        PCANTP_MSGTYPE_NONE);

      // Check if we received a "PONG" message
      if not TCanTpApi.StatusIsOk_2016(status, PCANTP_STATUS_NO_MESSAGE, false)
      then
      begin
        Writeln(Format('[ISOTP] Read ISOTP message: %s',
          [ISOTP_STATUS_OK_KO(status)]));
        if (TUDSApi.StatusIsOk_2013(uds_status(status), PUDS_STATUS_OK, false))
          And (rx_msg.msgdata_any^.length = MSG_SIZE) then
        begin
          msgok := (rx_msg.msgdata_any^.data <> nil);
          If msgok Then
          begin
            i := 0;
            data_pointer := PByte(rx_msg.msgdata_any^.data);
            While msgok And (i < MSG_SIZE) do
            begin
              If data_pointer^ = ISOTP_RESPONSE_MSG[i] Then
                msgok := True
              Else
                msgok := false;
              i := i + 1;
              inc(data_pointer);
            End;
          End;
          If msgok Then
          begin
            Writeln('[ISOTP] Message contains "PONG": OK');
            get_pong_msg := True;
          end;
        end;

        status := TCanTpApi.MsgDataFree_2016(&rx_msg);
        Writeln(Format('[ISOTP] Free RX message: %s',
          [ISOTP_STATUS_OK_KO(status)]));
      end;
    until not TCanTpApi.StatusIsOk_2016(status,
      PCANTP_STATUS_NO_MESSAGE, false);
    repeat_iter := repeat_iter + 1;
  until not((get_pong_msg = false) And (repeat_iter < repeat_wait_counter));
  result := get_pong_msg;
end;

/// <summary>ISOTP client task: send "PING" and wait "PONG" message several times</summary>
/// <param name="param">client channel handle</param>
Procedure isotp_client_task(param: pointer); stdcall;
var
  status: cantp_status;
  tx_msg: cantp_msg;
  mapping: cantp_mapping;
  reverse_mapping: cantp_mapping;
  receive_event: THANDLE;
  null_handle: THANDLE;
  client_handle: ^cantp_handle;
  response: bool;
  i: Integer;
begin

  // Initialize variables
  FillChar(tx_msg, sizeof(tx_msg), 0);
  FillChar(mapping, sizeof(mapping), 0);
  FillChar(reverse_mapping, sizeof(reverse_mapping), 0);
  client_handle := param;
  null_handle := 0;

  // Create a isotp receive event
  receive_event := CreateEvent(nil, false, false, nil);
  status := TCanTpApi.SetValue_2016(client_handle^,
    PCANTP_PARAMETER_RECEIVE_EVENT, PByte(@receive_event),
    sizeof(receive_event));
  Writeln(Format('[ISOTP] Set isotp receive event parameter: %s',
    [ISOTP_STATUS_OK_KO(status)]));

  // Create a simple isotp physical mapping:
  // - Source 0xA1 (client), target 0xA2 (server), CAN id 0xA1, CAN ID flow control 0xA2
  // - Diagnostic message in a classic format
  mapping.can_id := $A1;
  mapping.can_id_flow_ctrl := $A2;
  mapping.can_msgtype := PCANTP_CAN_MSGTYPE_STANDARD;
  mapping.netaddrinfo.extension_addr := $00;
  mapping.netaddrinfo.Format := PCANTP_ISOTP_FORMAT_NORMAL;
  mapping.netaddrinfo.msgtype := PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC;
  mapping.netaddrinfo.source_addr := $A1;
  mapping.netaddrinfo.target_addr := $A2;
  mapping.netaddrinfo.target_type := PCANTP_ISOTP_ADDRESSING_PHYSICAL;

  // Create the associated isotp reversed mapping:
  reverse_mapping := mapping;
  reverse_mapping.can_id := mapping.can_id_flow_ctrl;
  reverse_mapping.can_id_flow_ctrl := mapping.can_id;
  reverse_mapping.netaddrinfo.source_addr := mapping.netaddrinfo.target_addr;
  reverse_mapping.netaddrinfo.target_addr := mapping.netaddrinfo.source_addr;

  // Add ISOTP mappings on channels
  status := TCanTpApi.AddMapping_2016(client_handle^, @mapping);
  Writeln(Format('[ISOTP] Add a simple isotp mapping: %s',
    [ISOTP_STATUS_OK_KO(status)]));
  status := TCanTpApi.AddMapping_2016(client_handle^, @reverse_mapping);
  Writeln(Format('[ISOTP] Add the reverse isotp mapping: %s',
    [ISOTP_STATUS_OK_KO(status)]));

  // Initialize ISOTP Tx message containing "PING"
  status := TCanTpApi.MsgDataAlloc_2016(&tx_msg, PCANTP_MSGTYPE_ISOTP);
  Writeln(Format('[ISOTP] Allocate ISOTP tx message: %s',
    [ISOTP_STATUS_OK_KO(status)]));
  status := TCanTpApi.MsgDataInit_2016(&tx_msg, mapping.can_id,
    mapping.can_msgtype, MSG_SIZE, PByte(@ISOTP_REQUEST_MSG),
    @mapping.netaddrinfo);
  Writeln(Format('[ISOTP] Initialize ISOTP tx message: %s',
    [ISOTP_STATUS_OK_KO(status)]));

  // Send "PING" ISOTP message and wait "PONG" response
  for i := 1 to 10 do
  begin
    status := TCanTpApi.Write_2016(client_handle^, &tx_msg);
    Writeln(Format('[ISOTP] Send ISOTP "PING" message (%d): %s',
      [i, ISOTP_STATUS_OK_KO(status)]));
    response := wait_isotp_pong_response(client_handle^, 3);
    Writeln(Format('[ISOTP] Get ISOTP "PONG" response (%d): %s',
      [i, OK_KO(response)]));
  end;

  // Close receive event
  status := TCanTpApi.SetValue_2016(client_handle^,
    PCANTP_PARAMETER_RECEIVE_EVENT, PByte(@null_handle), sizeof(null_handle));
  Writeln(Format('[ISOTP] Stop ISOTP receive event: %s',
    [ISOTP_STATUS_OK_KO(status)]));
  CloseHandle(receive_event);
  Writeln(Format('[ISOTP] Close ISOTP receive event: %s',
    [ISOTP_STATUS_OK_KO(status)]));

  // Free messages
  status := TCanTpApi.MsgDataFree_2016(&tx_msg);
  Writeln(Format('[ISOTP] Free ISOTP TX message: %s',
    [ISOTP_STATUS_OK_KO(status)]));
end;

/// <summary>Entry point of the program, start a UDS channel, ask in the same time UDS TesterPresent and isotp request</summary>
/// <returns>By convention, return success.</returns>
var
  buffer: array [0 .. 255] of ansichar;
  status: uds_status;
  client_handle: cantp_handle;
  uds_client: THANDLE;
  isotp_client: THANDLE;
  uds_client_id: DWORD;
  isotp_client_id: DWORD;
  dummy: char;

begin
  try

    // Initialize variables
    buffer[0] := #0;
    // TODO: modify the value according to your available PCAN devices.
    client_handle := PCANTP_HANDLE_USBBUS1;

    // Print version informations
    status := TUDSApi.GetValue_2013(PCANTP_HANDLE_NONEBUS,
      PUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE);
    Writeln(Format('PCAN-UDS API Version - %s: %s',
      [buffer, UDS_STATUS_OK_KO(status)]));

    // Initialize client channel
    status := TUDSApi.Initialize_2013(client_handle, PCANTP_BAUDRATE_500K);
    Writeln(Format('Initialize channel: %s', [UDS_STATUS_OK_KO(status)]));

    // Start uds and isotp clients
    uds_client := CreateThread(nil, 0, @uds_client_task, @client_handle, 0,
      uds_client_id);
    if (uds_client <> 0) then
    begin
      isotp_client := CreateThread(nil, 0, @isotp_client_task, @client_handle,
        0, isotp_client_id);
      if (isotp_client <> 0) then
      begin
        WaitForSingleObject(isotp_client, INFINITE);
        CloseHandle(isotp_client);
      end;

      WaitForSingleObject(uds_client, INFINITE);
      CloseHandle(uds_client);
    end;

    // Close channel
    status := TUDSApi.Uninitialize_2013(client_handle);
    Writeln(Format('Uninitialize channel: %s', [UDS_STATUS_OK_KO(status)]));

    // Exit
    Writeln('Press any key to continue...');
    Readln(dummy);
  except
    on E: Exception do begin
      Writeln(E.Classname, ': ', E.Message);
    end;
  end;

end.
