program server_response_ReadDataByPeriodicIdentifier;

{$APPTYPE CONSOLE}

uses
  SysUtils,
  ShellApi,
  windows,
  PUDS_2013 in 'PUDS_2013.pas',
  PCANBasic in 'PCANBasic.pas',
  PCANTP_2016 in 'PCANTP_2016.pas';

Function OK_KO(test: Boolean): String;
begin
  If test Then
    result := 'OK'
  else
    result := 'KO';
End;

Function STATUS_OK_KO(test: uds_status): String;
begin
  result := OK_KO(TUDSApi.StatusIsOk_2013(test));
End;

Function KeyPressed: Boolean;
Const
  buffer_size = 20;
Var
  zone: Array [1 .. buffer_size] Of TInputRecord;
  is_waiting: Cardinal;
  i: Cardinal;
  han: THandle;
Begin
  han := GetStdHandle(STD_INPUT_HANDLE);
  PeekConsoleInput(han, zone[1], buffer_size, is_waiting);
  result := false;
  i := 1;
  While Not result And (i <= is_waiting) And (i <= buffer_size) Do
  Begin
    result := (zone[i].EventType = 1) And (zone[i].Event.KeyEvent.bKeyDown) And
      (zone[i].Event.KeyEvent.AsciiChar <> #0);
    inc(i);
  End;
  If (is_waiting <> 0) And Not result Then
    ReadConsoleInput(STD_INPUT_HANDLE, zone[1], is_waiting, i);
End;

Function ReadKey: AnsiChar;
Var
  zone: TInputRecord;
  nb_read: Cardinal;
  han: THandle;
Begin
  han := GetStdHandle(STD_INPUT_HANDLE);
  result := #0;
  Repeat
    ReadConsoleInput(han, zone, 1, nb_read);
    If (nb_read = 1) And (zone.EventType = 1) And (zone.Event.KeyEvent.bKeyDown)
      And (zone.Event.KeyEvent.AsciiChar <> #0) Then
      result := zone.Event.KeyEvent.AsciiChar;
  Until result <> #0;
End;

const
  PCAN_BITRATE: AnsiString =
    'f_clock=40000000,nom_brp=2,nom_tseg1=63,nom_tseg2=16,nom_sjw=16,data_brp=2,data_tseg1=7,data_tseg2=2,data_sjw=2';

  /// <summary>
  /// Entry point of the program, start a small server wich only support ReadDataByPeriodicIdentifier service.
  /// This example use a specific addressing. It receives request from test equipement (0xF1 to 0xC1) in 29b fixed
  /// normal addressing and sends responses for each periodic data identifier with 0x1F22C1F1 can identifier (UUDT).
  /// </summary>
  /// <returns>By convention, return success.</returns>
var
  status: uds_status;
  server_handle: cantp_handle;
  server_address: UInt16;
  null_handle: UInt64;
  receive_event: THandle;
  service_response_config: uds_msgconfig;
  stopit: Boolean;
  wait_result: UInt32;
  read_status: uds_status;
  request_msg: uds_msg;
  service_response_msg: uds_msg;
  periodic_response: uds_msg;
  periodic_msg_config: uds_msgconfig;
  keyboard_res: AnsiChar;
  padding_value: UInt8;
  can_tx_dl: UInt8;
  periodic_data_identifier: UInt8;
  timeout_value: UInt32;
  periodic_data_identifier_length: UInt32;
  i: UInt32;
  req_srv_id: Byte;
  req_any: cantp_msgdata;
  resp_data_pointer: PByte;
  dummy: char;
  req_parameters: PByteArray;

begin
  try
    // Initialize variables
    server_handle := PCANTP_HANDLE_USBBUS2;
    // TODO: modify the value according to your available PCAN devices.
    null_handle := 0;
    server_address := $C1;
    FillChar(service_response_config, sizeof(service_response_config), 0);
    FillChar(periodic_msg_config, sizeof(periodic_msg_config), 0);
    FillChar(request_msg, sizeof(request_msg), 0);
    FillChar(service_response_msg, sizeof(service_response_msg), 0);

    // Initialize server
    status := TUDSApi.InitializeFD_2013(server_handle, PAnsiChar(PCAN_BITRATE));
    Writeln(Format('Initialize channel: %s', [STATUS_OK_KO(status)]));

    // Set server address parameter
    status := TUDSApi.SetValue_2013(server_handle,
      PUDS_PARAMETER_SERVER_ADDRESS, PByte(@server_address),
      sizeof(server_address));
    Writeln(Format('Set server address: %s', [STATUS_OK_KO(status)]));

    // Set a padding value
    padding_value := $FF;
    status := TUDSApi.SetValue_2013(server_handle,
      PUDS_PARAMETER_CAN_PADDING_VALUE, PByte(@padding_value),
      sizeof(padding_value));
    Writeln(Format('Set padding value: %s', [STATUS_OK_KO(status)]));

    // Define CAN_TX_DL=15
    can_tx_dl := 15;
    status := TUDSApi.SetValue_2013(server_handle, PUDS_PARAMETER_CAN_TX_DL,
      PByte(@can_tx_dl), sizeof(can_tx_dl));
    Writeln(Format('Set CAN TX DL: %s', [STATUS_OK_KO(status)]));

    // Set UDS timeouts
    timeout_value := 5000;
    status := TUDSApi.SetValue_2013(server_handle,
      PUDS_PARAMETER_TIMEOUT_REQUEST, PByte(@timeout_value),
      sizeof(timeout_value));
    Writeln(Format('Set request timeout(ms): %s', [STATUS_OK_KO(status)]));
    status := TUDSApi.SetValue_2013(server_handle,
      PUDS_PARAMETER_TIMEOUT_RESPONSE, PByte(@timeout_value),
      sizeof(timeout_value));
    Writeln(Format('Set response timeout(ms): %s', [STATUS_OK_KO(status)]));

    // Set a receive event
    receive_event := CreateEvent(nil, false, false, nil);
    status := TUDSApi.SetValue_2013(server_handle, PUDS_PARAMETER_RECEIVE_EVENT,
      PByte(@receive_event), sizeof(receive_event));
    Writeln(Format('Set receive event parameter: %s', [STATUS_OK_KO(status)]));

    // Initialize service response configuration
    service_response_config.can_id := $FFFFFFFF;
    service_response_config.can_msgtype :=
      cantp_can_msgtype(UInt32(PCANTP_CAN_MSGTYPE_EXTENDED) or
      UInt32(PCANTP_CAN_MSGTYPE_FD) or UInt32(PCANTP_CAN_MSGTYPE_BRS));
    service_response_config.nai.protocol :=
      PUDS_MSGPROTOCOL_ISO_15765_2_29B_FIXED_NORMAL;
    service_response_config.nai.target_type := PCANTP_ISOTP_ADDRESSING_PHYSICAL;
    service_response_config.typem := PUDS_MSGTYPE_USDT;
    service_response_config.nai.source_addr := server_address;
    service_response_config.nai.target_addr :=
      UInt16(PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT);
    service_response_config.nai.extension_addr := 0;

    // Initialize responses configuration (for each periodic data identifier contained in the request)
    periodic_msg_config.can_id := $1F22C1F1;
    periodic_msg_config.can_msgtype :=
      cantp_can_msgtype(UInt32(PCANTP_CAN_MSGTYPE_EXTENDED) or
      UInt32(PCANTP_CAN_MSGTYPE_FD) or UInt32(PCANTP_CAN_MSGTYPE_BRS));
    periodic_msg_config.nai.protocol := PUDS_MSGPROTOCOL_ISO_15765_2_29B_NORMAL;
    periodic_msg_config.nai.target_type := PCANTP_ISOTP_ADDRESSING_PHYSICAL;
    periodic_msg_config.typem := PUDS_MSGTYPE_UUDT;
    periodic_msg_config.nai.source_addr := server_address;
    periodic_msg_config.nai.target_addr :=
      UInt16(PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT);
    periodic_msg_config.nai.extension_addr := 0;

    // Add a filter for 0x1F22C1F1 can id (in order to receive UUDT loopback messages)
    status := TUDSApi.AddCanIdFilter_2013(server_handle,
      periodic_msg_config.can_id);
    Writeln(Format('Add can identifier filter: %s', [STATUS_OK_KO(status)]));

    // Read while user do not press Q
    Writeln('Start listening, press Q to quit.');
    stopit := false;
    repeat

      // Wait a receive event on receiver
      // note: timeout is used to check keyboard hit.
      wait_result := WaitForSingleObject(receive_event, 1000);

      // If we get a receive event
      if wait_result = WAIT_OBJECT_0 Then
      begin

        repeat

          // Read first available message (no filtering based on message's type is set):
          read_status := TUDSApi.Read_2013(server_handle, &request_msg);
          Writeln(Format('Try to read a message: %s',
            [STATUS_OK_KO(read_status)]));
          if (TUDSApi.StatusIsOk_2013(read_status, PUDS_STATUS_OK, false)) Then
          begin
            FillChar(req_any, sizeof(cantp_msgdata), 0);
            If request_msg.msg.msgdata_any <> nil Then
              req_any := request_msg.msg.msgdata_any^;

            // We receive a request, check its length and if it is not a loopback message
            if (req_any.length >= 1) and
              ((UInt32(req_any.flags) and UInt32(PCANTP_MSGFLAG_LOOPBACK))
              = 0) Then
            begin

              // This is a valid request, switch services
              req_srv_id := request_msg.links.service_id^;
              Case req_srv_id of
                Byte(PUDS_SERVICE_SI_ReadDataByPeriodicIdentifier):
                  begin
                    // Allocates service response message
                    status := TUDSApi.MsgAlloc_2013(&service_response_msg,
                      service_response_config, 1);
                    if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, false))
                    then
                    begin
                      service_response_msg.links.service_id^ :=
                        Byte(Byte(PUDS_SERVICE_SI_ReadDataByPeriodicIdentifier)
                        + TUDSApi.PUDS_SI_POSITIVE_RESPONSE);
                    end;
                    Writeln(Format
                      ('Prepare response message for ReadDataByPeriodicIdentifier service: %s',
                      [STATUS_OK_KO(status)]));

                    // Write service response message
                    status := TUDSApi.Write_2013(server_handle,
                      @service_response_msg);
                    Writeln(Format
                      ('Write response message for ReadDataByPeriodicIdentifier service: %s',
                      [STATUS_OK_KO(status)]));

                    // Free response message (and clean memory in order to reallocate later)
                    status := TUDSApi.MsgFree_2013(&service_response_msg);
                    Writeln(Format('Free response message: %s',
                      [STATUS_OK_KO(status)]));

                    // Sends a message for each data identifier with a specific addressing.
                    periodic_data_identifier_length := req_any.length - 2;
                    for i := 0 to periodic_data_identifier_length - 1 do
                    begin

                      // Allocates and prepares message with dummy data
                      FillChar(periodic_response, sizeof(periodic_response), 0);

                      status := TUDSApi.MsgAlloc_2013(&periodic_response,
                        periodic_msg_config, 5);
                      if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, false))
                      then
                      begin
                        req_parameters := PByteArray(request_msg.links.param);
                        periodic_data_identifier := req_parameters[1 + i];
                        resp_data_pointer :=
                          PByte(periodic_response.msg.msgdata_any.data);
                        resp_data_pointer^ := periodic_data_identifier;
                        inc(resp_data_pointer);
                        resp_data_pointer^ := $12;
                        inc(resp_data_pointer);
                        resp_data_pointer^ := $34;
                        inc(resp_data_pointer);
                        resp_data_pointer^ := $56;
                        inc(resp_data_pointer);
                        resp_data_pointer^ := $78;

                        Writeln(Format
                          ('Allocates message for 0x%02x periodic data identifier: %s',
                          [integer(periodic_data_identifier),
                          STATUS_OK_KO(status)]));
                        status := TUDSApi.Write_2013(server_handle,
                          @periodic_response);
                        Writeln(Format
                          ('Write message for 0x%02x periodic data identifier: %s',
                          [integer(periodic_data_identifier),
                          STATUS_OK_KO(status)]));
                        status := TUDSApi.MsgFree_2013(&periodic_response);
                        Writeln(Format
                          ('Free message for 0x%02x periodic data identifier: %s',
                          [integer(periodic_data_identifier),
                          STATUS_OK_KO(status)]));
                      end;
                    end;
                  end
              else
                Writeln(Format('Unknown service (0x%02x)',
                  [integer(req_srv_id)]));
              end;
            end;
          end;

          // Free request message (in order to reallocate later)
          status := TUDSApi.MsgFree_2013(&request_msg);
          Writeln(Format('Free request message: %s', [STATUS_OK_KO(status)]));
        until TUDSApi.StatusIsOk_2013(read_status,
          PUDS_STATUS_NO_MESSAGE, false);
      end;

      // Quit when user press Q
      If KeyPressed() Then
      begin
        keyboard_res := ReadKey();
        If (keyboard_res = 'Q') Or (keyboard_res = 'q') Then
          stopit := true;
      end;
    until stopit = true;

    // Close receive event
    status := TUDSApi.SetValue_2013(server_handle, PUDS_PARAMETER_RECEIVE_EVENT,
      PByte(@null_handle), sizeof(null_handle));
    Writeln(Format('Stop receive event: %s', [STATUS_OK_KO(status)]));
    CloseHandle(receive_event);
    Writeln(Format('Close receive event: %s', [STATUS_OK_KO(status)]));

    // Close server
    status := TUDSApi.Uninitialize_2013(server_handle);
    Writeln(Format('Uninitialize channel: %s', [STATUS_OK_KO(status)]));

    // Exit
    Writeln('Press any key to continue...');
    Readln(dummy);

  except
    on E: Exception do
      Writeln(E.Classname, ': ', E.Message);
  end;

end.
