program client_request_ReadDataByPeriodicIdentifier;

{$APPTYPE CONSOLE}

uses
  SysUtils,
  ShellApi,
  windows,
  PUDS_2013 in 'PUDS_2013.pas',
  PCANBasic in 'PCANBasic.pas',
  PCANTP_2016 in 'PCANTP_2016.pas';

const
  PCAN_BITRATE: AnsiString =
    'f_clock=40000000,nom_brp=2,nom_tseg1=63,nom_tseg2=16,nom_sjw=16,data_brp=2,data_tseg1=7,data_tseg2=2,data_sjw=2';

const
  RESPONSES_TIMEOUT_MS: UInt64 = 5000;

Function OK_KO(test: Boolean): String;
begin
  If test Then
    result := 'OK'
  else
    result := 'KO';
End;

Function STATUS_OK_KO(test: uds_status): String;
begin
  result := OK_KO(TUDSApi.StatusIsOk_2013(test));
End;

Function KeyPressed: Boolean;
Const
  buffer_size = 20;
Var
  zone: Array [1 .. buffer_size] Of TInputRecord;
  is_waiting: Cardinal;
  i: Cardinal;
  han: THandle;
Begin
  han := GetStdHandle(STD_INPUT_HANDLE);
  PeekConsoleInput(han, zone[1], buffer_size, is_waiting);
  result := false;
  i := 1;
  While Not result And (i <= is_waiting) And (i <= buffer_size) Do
  Begin
    result := (zone[i].EventType = 1) And (zone[i].Event.KeyEvent.bKeyDown) And
      (zone[i].Event.KeyEvent.AsciiChar <> #0);
    inc(i);
  End;
  If (is_waiting <> 0) And Not result Then
    ReadConsoleInput(STD_INPUT_HANDLE, zone[1], is_waiting, i);
End;

Function ReadKey: AnsiChar;
Var
  zone: TInputRecord;
  nb_read: Cardinal;
  han: THandle;
Begin
  han := GetStdHandle(STD_INPUT_HANDLE);
  result := #0;
  Repeat
    ReadConsoleInput(han, zone, 1, nb_read);
    If (nb_read = 1) And (zone.EventType = 1) And (zone.Event.KeyEvent.bKeyDown)
      And (zone.Event.KeyEvent.AsciiChar <> #0) Then
      result := zone.Event.KeyEvent.AsciiChar;
  Until result <> #0;
End;

/// <summary>
/// Entry point of the program, start a UDS channel, ask ReadDataByPeriodicIdentifier service.
/// It sends a request from 0xF1 to 0xC1 address in 29b fixed with normal addressing. Then read
/// responses from 0x1F22C1F1 can identifier (UUDT).
/// </summary>
/// <returns>By convention, return success.</returns>
var
  status: uds_status;
  client_handle: cantp_handle;
  request_config: uds_msgconfig;
  msg_request: uds_msg;
  request_confirmation: uds_msg;
  service_response: uds_msg;
  can_tx_dl: UInt8;
  timeout_value: UInt32;
  periodic_response: uds_msg;
  response_count: UInt8;
  start_reading: UInt64;
  i: UInt32;
  periodic_data_identifier: array [0 .. 3] of Byte;
  dummy: char;
  resp_data: PByteArray;

begin
  try

    // Initialize variables
    client_handle := PCANTP_HANDLE_USBBUS1;
    // TODO: modify the value according to your available PCAN devices.
    FillChar(&request_config, sizeof(request_config), 0);
    FillChar(&msg_request, sizeof(msg_request), 0);
    FillChar(&request_confirmation, sizeof(request_confirmation), 0);
    FillChar(&service_response, sizeof(service_response), 0);
    FillChar(&periodic_response, sizeof(periodic_response), 0);

    // Initialize client
    status := TUDSApi.InitializeFD_2013(client_handle, PAnsiChar(PCAN_BITRATE));
    Writeln(Format('Initialize channel: %s', [STATUS_OK_KO(status)]));

    // Define CAN_TX_DL
    can_tx_dl := 15;
    status := TUDSApi.SetValue_2013(client_handle, PUDS_PARAMETER_CAN_TX_DL,
      PByte(@can_tx_dl), sizeof(can_tx_dl));
    Writeln(Format('Set CAN TX DL: %s', [STATUS_OK_KO(status)]));

    // Check CAN_TX_DL
    can_tx_dl := 0;
    status := TUDSApi.GetValue_2013(client_handle, PUDS_PARAMETER_CAN_TX_DL,
      PByte(@can_tx_dl), sizeof(can_tx_dl));
    Writeln(Format('Check new CAN TX DL value(%d): %s', [integer(can_tx_dl),
      STATUS_OK_KO(status)]));

    // Set UDS timeouts
    timeout_value := 5000;
    status := TUDSApi.SetValue_2013(client_handle,
      PUDS_PARAMETER_TIMEOUT_REQUEST, PByte(@timeout_value),
      sizeof(timeout_value));
    Writeln(Format('Set request timeout(ms): %s', [STATUS_OK_KO(status)]));
    status := TUDSApi.SetValue_2013(client_handle,
      PUDS_PARAMETER_TIMEOUT_RESPONSE, PByte(@timeout_value),
      sizeof(timeout_value));
    Writeln(Format('Set response timeout(ms): %s', [STATUS_OK_KO(status)]));

    // Check timeouts
    timeout_value := 0;
    status := TUDSApi.GetValue_2013(client_handle,
      PUDS_PARAMETER_TIMEOUT_REQUEST, PByte(@timeout_value),
      sizeof(timeout_value));
    Writeln(Format('Get new request timeout(%dms): %s', [integer(timeout_value),
      STATUS_OK_KO(status)]));
    timeout_value := 0;
    status := TUDSApi.GetValue_2013(client_handle,
      PUDS_PARAMETER_TIMEOUT_RESPONSE, PByte(@timeout_value),
      sizeof(timeout_value));
    Writeln(Format('Get new response timeout(%dms): %s',
      [integer(timeout_value), STATUS_OK_KO(status)]));

    // Initialize the request configuration
    request_config.can_id := $FFFFFFFF;
    request_config.can_msgtype :=
      cantp_can_msgtype(UInt32(PCANTP_CAN_MSGTYPE_EXTENDED) or
      UInt32(PCANTP_CAN_MSGTYPE_FD) or UInt32(PCANTP_CAN_MSGTYPE_BRS));
    request_config.nai.protocol :=
      PUDS_MSGPROTOCOL_ISO_15765_2_29B_FIXED_NORMAL;
    request_config.nai.target_type := PCANTP_ISOTP_ADDRESSING_PHYSICAL;
    request_config.typem := PUDS_MSGTYPE_USDT;
    request_config.nai.source_addr :=
      UInt16(PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT);
    request_config.nai.target_addr := $C1;
    request_config.nai.extension_addr := 0;

    // Add a filter for 0x1F22C1F1 can id (in order to receive UUDT messages)
    status := TUDSApi.AddCanIdFilter_2013(client_handle, $1F22C1F1);
    Writeln(Format('Add can identifier filter: %s', [STATUS_OK_KO(status)]));

    // Execute ReadDataByPeriodicIdentifier and wait service response
    periodic_data_identifier[0] := $12;
    periodic_data_identifier[1] := $34;
    periodic_data_identifier[2] := $56;
    periodic_data_identifier[3] := $78;
    status := TUDSApi.SvcReadDataByPeriodicIdentifier_2013(client_handle,
      request_config, &msg_request, PUDS_SVC_PARAM_RDBPI_SAMR,
      @periodic_data_identifier, 4);
    Writeln(Format('Execute ReadDataByPeriodicIdentifier service: %s',
      [STATUS_OK_KO(status)]));
    status := TUDSApi.WaitForService_2013(client_handle, @msg_request,
      service_response, @request_confirmation);
    Writeln(Format('Wait for service (confirmation and response): %s',
      [STATUS_OK_KO(status)]));

    // Read responses for each given periodic data identifier
    response_count := 0;
    start_reading := GetTickCount();
    repeat
      status := TUDSApi.Read_2013(client_handle, &periodic_response);
      if ((TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, false)) and
        (periodic_response.msg.msgdata_any.length > 0)) then
      begin
        response_count := response_count + 1;
        resp_data := PByteArray(periodic_response.msg.msgdata_any.data);
        Write(Format('Read response for 0x%02x periodic data identifier: [',
          [integer(resp_data[0])]));
        for i := 0 To periodic_response.msg.msgdata_any.length - 1 do
        begin
          Write(Format(' 0x%02x', [integer(resp_data[i])]));
        end;
        Writeln(' ]');
        status := TUDSApi.MsgFree_2013(&periodic_response);
        Writeln(Format('Free response buffer: %s', [STATUS_OK_KO(status)]));
      end;

      // Wait for 4 responses or timeout
    until not((response_count < 4) and (GetTickCount() - start_reading <
      RESPONSES_TIMEOUT_MS));

    // Free message structures
    status := TUDSApi.MsgFree_2013(&msg_request);
    Writeln(Format('Free request message: %s', [STATUS_OK_KO(status)]));
    status := TUDSApi.MsgFree_2013(&request_confirmation);
    Writeln(Format('Free request confirmation: %s', [STATUS_OK_KO(status)]));
    status := TUDSApi.MsgFree_2013(&service_response);
    Writeln(Format('Free service response: %s', [STATUS_OK_KO(status)]));

    // Close client
    status := TUDSApi.Uninitialize_2013(client_handle);
    Writeln(Format('Uninitialize channel: %s', [STATUS_OK_KO(status)]));

    // Exit
    Writeln('Press any key to continue...');
    Readln(dummy);

  except
    on E: Exception do
      Writeln(E.Classname, ': ', E.Message);
  end;

end.
