program server_uds_and_can;

{$APPTYPE CONSOLE}

uses
  SysUtils,
  ShellApi,
  windows,
  PUDS_2013 in 'PUDS_2013.pas',
  PCANBasic in 'PCANBasic.pas',
  PCANTP_2016 in 'PCANTP_2016.pas';

const
  BUFFER_SIZE: Integer = 256;
  ISOTP_PING_MSG: array [0 .. 3] of Byte = (Byte('P'), Byte('I'), Byte('N'),
    Byte('G'));
  ISOTP_PING_MSG_SIZE: Integer = 4;
  CAN_ID_START_RANGE: UInt32 = 1;
  CAN_ID_STOP_RANGE: UInt32 = 10;
  PING_TIMING_MS: Integer = 500;

Function OK_KO(test: Boolean): String;
begin
  if (test) then
    result := 'OK'
  else
    result := 'KO';
end;

Function UDS_STATUS_OK_KO(test: uds_status): String;
begin
  result := OK_KO(TUDSApi.StatusIsOk_2013(test, PUDS_STATUS_OK, false));
end;

Function ISOTP_STATUS_OK_KO(test: cantp_status): String;
begin
  result := OK_KO(TCanTpApi.StatusIsOk_2016(test, PCANTP_STATUS_OK, false));
end;

Function KeyPressed: Boolean;
Const
  BUFFER_SIZE = 20;
Var
  zone: Array [1 .. BUFFER_SIZE] Of TInputRecord;
  is_waiting: Cardinal;
  i: Cardinal;
  han: THandle;
Begin
  han := GetStdHandle(STD_INPUT_HANDLE);
  PeekConsoleInput(han, zone[1], BUFFER_SIZE, is_waiting);
  result := false;
  i := 1;
  While Not result And (i <= is_waiting) And (i <= BUFFER_SIZE) Do
  Begin
    result := (zone[i].EventType = 1) And (zone[i].Event.KeyEvent.bKeyDown) And
      (zone[i].Event.KeyEvent.AsciiChar <> #0);
    inc(i);
  End;
  If (is_waiting <> 0) And Not result Then
    ReadConsoleInput(STD_INPUT_HANDLE, zone[1], is_waiting, i);
End;

Function ReadKey: AnsiChar;
Var
  zone: TInputRecord;
  nb_read: Cardinal;
  han: THandle;
Begin
  han := GetStdHandle(STD_INPUT_HANDLE);
  result := #0;
  Repeat
    ReadConsoleInput(han, zone, 1, nb_read);
    If (nb_read = 1) And (zone.EventType = 1) And (zone.Event.KeyEvent.bKeyDown)
      And (zone.Event.KeyEvent.AsciiChar <> #0) Then
      result := zone.Event.KeyEvent.AsciiChar;
  Until result <> #0;
End;

/// <summary>Structure passed as thread parameters</summary>
Type
  task_params = record
    /// <summary>Server channel handle</summary>
    server_handle: cantp_handle;
    /// <summary>Server address</summary>
    server_address: UInt32;
    /// <summary>Determine if the thread should end or not</summary>
    stop_task: bool;
  end;

  /// <summary>ISOTP server task: periodically send "PING" message using different CAN identifiers</summary>
  /// <param name="parameters">pointer on task_params structures</param>
Procedure isotp_server_task(parameters: pointer); stdcall;
var
  t_params: ^task_params;
  status: cantp_status;
  tx_msg: cantp_msg;
  can_id: UInt32;
begin

  // Init variables
  t_params := parameters;
  Fillchar(&tx_msg, sizeof(tx_msg), 0);
  can_id := CAN_ID_START_RANGE;

  // Send loop
  repeat
    // Wait before sending the next message
    Sleep(PING_TIMING_MS);

    // Initialize ISOTP Tx message containing "PING"
    status := TCanTpApi.MsgDataAlloc_2016(&tx_msg, PCANTP_MSGTYPE_CAN);
    Writeln(Format('[ISOTP] Allocate ISOTP tx message: %s',
      [ISOTP_STATUS_OK_KO(status)]));
    status := TCanTpApi.MsgDataInit_2016(&tx_msg, can_id,
      PCANTP_CAN_MSGTYPE_STANDARD, ISOTP_PING_MSG_SIZE, @ISOTP_PING_MSG, nil);
    Writeln(Format('[ISOTP] Initialize ISOTP tx message: %s',
      [ISOTP_STATUS_OK_KO(status)]));

    // Send "PING" message
    status := TCanTpApi.Write_2016(t_params.server_handle, &tx_msg);
    Writeln(Format('[ISOTP] Send ISOTP "PING" message (can id=0x%s): %s',
      [LowerCase(Format('%x', [Integer(can_id)])),
      ISOTP_STATUS_OK_KO(status)]));

    // Free messages
    status := TCanTpApi.MsgDataFree_2016(&tx_msg);
    Writeln(Format('[ISOTP] Free ISOTP TX message: %s',
      [ISOTP_STATUS_OK_KO(status)]));

    // Update can id for next message
    can_id := can_id + 1;
    if (can_id > CAN_ID_STOP_RANGE) Then
    begin
      can_id := CAN_ID_START_RANGE;
    end;
  until t_params.stop_task = true;
end;

/// <summary>UDS server task: respond TesterPresent request</summary>
/// <param name="parameters">pointer on task_params structures</param>
Procedure uds_server_task(parameters: pointer); stdcall;
var
  t_params: ^task_params;
  config_physical: uds_msgconfig;
  wait_result: DWORD;
  status: uds_status;
  receive_event: THandle;
  read_status: uds_status;
  request_msg: uds_msg;
  response_msg: uds_msg;
  null_handle: THandle;
  req_srv_id: Byte;
begin

  // Init variables
  t_params := parameters;
  null_handle := 0;
  Fillchar(&config_physical, sizeof(config_physical), 0);
  Fillchar(&request_msg, sizeof(request_msg), 0);
  Fillchar(&response_msg, sizeof(response_msg), 0);

  // Set server address parameter
  status := TUDSApi.SetValue_2013(t_params.server_handle,
    PUDS_PARAMETER_SERVER_ADDRESS, PByte(@t_params.server_address),
    sizeof(UInt32));
  Writeln(Format('[UDS] Set UDS server address: %s',
    [UDS_STATUS_OK_KO(status)]));

  // Set a receive event for UDS
  receive_event := CreateEvent(nil, false, false, nil);
  status := TUDSApi.SetValue_2013(t_params.server_handle,
    PUDS_PARAMETER_RECEIVE_EVENT, PByte(@receive_event), sizeof(receive_event));
  Writeln(Format('[UDS] Set UDS receive event parameter: %s',
    [UDS_STATUS_OK_KO(status)]));

  // Initialize a physical configuration
  config_physical.can_id := $FFFFFFFF;
  config_physical.can_msgtype := PCANTP_CAN_MSGTYPE_STANDARD;
  config_physical.nai.protocol := PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL;
  config_physical.nai.target_type := PCANTP_ISOTP_ADDRESSING_PHYSICAL;
  config_physical.typem := PUDS_MSGTYPE_USDT;
  config_physical.nai.extension_addr := 0;

  repeat

    // Wait a receive event on receiver
    wait_result := WaitForSingleObject(receive_event, 1000);

    // If we get a receive event
    if wait_result = WAIT_OBJECT_0 then
    begin
      repeat

        // Read first available message (no filtering based on message's type is set):
        read_status := TUDSApi.Read_2013(t_params.server_handle, &request_msg);
        Writeln(Format('[UDS] Try to read a message: %s',
          [UDS_STATUS_OK_KO(read_status)]));
        if (TUDSApi.StatusIsOk_2013(read_status, PUDS_STATUS_OK, false)) then
        begin

          // We receive a request, check its length and if it is not a loopback message and if it is a USDT message
          if ((request_msg.typem = PUDS_MSGTYPE_USDT) and
            (request_msg.msg.msgdata_any.length >= 1) and
            ((UInt32(request_msg.msg.msgdata_any.flags) and
            UInt32(PCANTP_MSGFLAG_LOOPBACK)) = 0)) then
          begin
            req_srv_id := request_msg.links.service_id^;

            // This is a valid request, switch services
            Case req_srv_id of
              Byte(uds_service.PUDS_SERVICE_SI_TesterPresent):
                begin

                  // Allocate response message
                  status := TUDSApi.MsgAlloc_2013(&response_msg,
                    config_physical, 2);
                  Writeln(Format
                    ('[UDS] Prepare response message for TesterPresent service: %s',
                    [UDS_STATUS_OK_KO(status)]));

                  if (TUDSApi.StatusIsOk_2013(status, PUDS_STATUS_OK, false))
                  then

                  begin
                    // Fill parameters
                    response_msg.msg.msgdata_isotp^.netaddrinfo :=
                      &request_msg.msg.msgdata_isotp^.netaddrinfo;
                    response_msg.links.service_id^ :=
                      Byte(PUDS_SERVICE_SI_TesterPresent) +
                      Byte(TUDSApi.PUDS_SI_POSITIVE_RESPONSE);
                    response_msg.links.param^ := Byte(0);
                    response_msg.msg.msgdata_isotp^.netaddrinfo.target_addr :=
                      request_msg.msg.msgdata_isotp^.netaddrinfo.source_addr;
                    response_msg.msg.msgdata_isotp^.netaddrinfo.source_addr :=
                      t_params.server_address;

                    // Write response message
                    status := TUDSApi.Write_2013(t_params.server_handle,
                      @response_msg);
                    Writeln(Format
                      ('[UDS] Write response message for TesterPresent service: %s',
                      [UDS_STATUS_OK_KO(status)]));
                  end;

                  // Free response message (and clean memory in order to reallocate later)
                  status := TUDSApi.MsgFree_2013(&response_msg);
                  Writeln(Format('[UDS] Free response message: %s',
                    [UDS_STATUS_OK_KO(status)]));
                end
            Else
              Writeln(Format('[UDS] Unknown service (0x%02x)',
                [Integer(req_srv_id)]));
            end;
          end;
        end;

        // Free request message (and clean memory in order to reallocate later)
        status := TUDSApi.MsgFree_2013(&request_msg);
        Writeln(Format('[UDS] Free request message: %s',
          [UDS_STATUS_OK_KO(status)]));
      until TUDSApi.StatusIsOk_2013(read_status, PUDS_STATUS_NO_MESSAGE, false);
    end;
  until t_params.stop_task = true;

  // Close receive event
  status := TUDSApi.SetValue_2013(t_params.server_handle,
    PUDS_PARAMETER_RECEIVE_EVENT, PByte(@null_handle), sizeof(null_handle));
  Writeln(Format('[UDS] Stop UDS receive event: %s',
    [UDS_STATUS_OK_KO(status)]));
  CloseHandle(receive_event);
  Writeln(Format('[UDS] Close UDS receive event: %s',
    [UDS_STATUS_OK_KO(status)]));
end;

/// <summary>Entry point of the program, start a small server which handle UDS testerpresent request and periodically send isotp messages</summary>
/// <returns>By convention, return success.</returns>
var
  status: uds_status;
  t_params: task_params;
  buffer: array [0 .. 255] of AnsiChar;
  uds_server: THandle;
  isotp_server: THandle;
  uds_server_id: DWORD;
  isotp_server_id: DWORD;
  keyboard_res: AnsiChar;
  dummy: char;

begin
  try
    // Initialize variables
    t_params.server_address :=
      UInt16(uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_1);
    t_params.stop_task := false;
    buffer[0] := #0;

    // TODO: modify the value according to your available PCAN devices.
    t_params.server_handle := PCANTP_HANDLE_USBBUS2;

    // Print version informations
    status := TUDSApi.GetValue_2013(PCANTP_HANDLE_NONEBUS,
      PUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE);
    Writeln(Format('PCAN-UDS API Version - %s: %s',
      [buffer, UDS_STATUS_OK_KO(status)]));

    // Initialize server
    status := TUDSApi.Initialize_2013(t_params.server_handle,
      PCANTP_BAUDRATE_500K);
    Writeln(Format('Initialize channel: %s', [UDS_STATUS_OK_KO(status)]));

    // Start uds and isotp servers
    uds_server := CreateThread(nil, 0, @uds_server_task, @t_params, 0,
      &uds_server_id);
    if (uds_server <> 0) then
    begin
      isotp_server := CreateThread(nil, 0, @isotp_server_task, @t_params, 0,
        &isotp_server_id);
      if (isotp_server <> 0) then
      begin

        // Read while user do not press Q
        Writeln('Start listening, press Q to quit.');
        t_params.stop_task := false;
        repeat
          // Quit when user press Q
          If KeyPressed() Then
          begin
            keyboard_res := ReadKey();
            If (keyboard_res = 'Q') Or (keyboard_res = 'q') Then
              t_params.stop_task := true;
          end;
        until t_params.stop_task = true;

        // Close threads
        WaitForSingleObject(isotp_server, INFINITE);
        CloseHandle(isotp_server);
      end;
      WaitForSingleObject(uds_server, INFINITE);
      CloseHandle(uds_server);
    end;

    // Close channel
    status := TUDSApi.Uninitialize_2013(t_params.server_handle);
    Writeln(Format('Uninitialize channel: %s', [UDS_STATUS_OK_KO(status)]));

    // Exit
    Writeln('Press any key to continue...');
    Readln(dummy);
  except
    on E: Exception do
      Writeln(E.Classname, ': ', E.Message);
  end;

end.
