program client_uds_and_can;

{$APPTYPE CONSOLE}

uses
  SysUtils,
  ShellApi,
  windows,
  PUDS_2013 in 'PUDS_2013.pas',
  PCANBasic in 'PCANBasic.pas',
  PCANTP_2016 in 'PCANTP_2016.pas';

const
  BUFFER_SIZE: UInt32 = 256;
  UDS_REQUEST_TIMING_MS: Integer = 500;
  CLIENT_EXECUTION_TIME_MS: Integer = 10000;

Function OK_KO(test: Boolean): String;
begin
  if (test) then
    result := 'OK'
  else
    result := 'KO';
end;

Function UDS_STATUS_OK_KO(test: uds_status): String;
begin
  result := OK_KO(TUDSApi.StatusIsOk_2013(test, PUDS_STATUS_OK, false));
end;

Function ISOTP_STATUS_OK_KO(test: cantp_status): String;
begin
  result := OK_KO(TCanTpApi.StatusIsOk_2016(test, PCANTP_STATUS_OK, false));
end;

Function KeyPressed: Boolean;
Const
  BUFFER_SIZE = 20;
Var
  zone: Array [1 .. BUFFER_SIZE] Of TInputRecord;
  is_waiting: Cardinal;
  i: Cardinal;
  han: THandle;
Begin
  han := GetStdHandle(STD_INPUT_HANDLE);
  PeekConsoleInput(han, zone[1], BUFFER_SIZE, is_waiting);
  result := false;
  i := 1;
  While Not result And (i <= is_waiting) And (i <= BUFFER_SIZE) Do
  Begin
    result := (zone[i].EventType = 1) And (zone[i].Event.KeyEvent.bKeyDown) And
      (zone[i].Event.KeyEvent.AsciiChar <> #0);
    inc(i);
  End;
  If (is_waiting <> 0) And Not result Then
    ReadConsoleInput(STD_INPUT_HANDLE, zone[1], is_waiting, i);
End;

Function ReadKey: AnsiChar;
Var
  zone: TInputRecord;
  nb_read: Cardinal;
  han: THandle;
Begin
  han := GetStdHandle(STD_INPUT_HANDLE);
  result := #0;
  Repeat
    ReadConsoleInput(han, zone, 1, nb_read);
    If (nb_read = 1) And (zone.EventType = 1) And (zone.Event.KeyEvent.bKeyDown)
      And (zone.Event.KeyEvent.AsciiChar <> #0) Then
      result := zone.Event.KeyEvent.AsciiChar;
  Until result <> #0;
End;

/// <summary>Structure passed as thread parameters</summary>
Type
  task_params = record
    /// <summary>Client channel handle</summary>
    client_handle: cantp_handle;
    /// <summary>Determine if the thread should end or not</summary>
    stop_task: bool;
  end;

  /// <summary>UDS client task: request TesterPresent service several time</summary>
  /// <param name="parameters">pointer on task_params structures</param>
Procedure uds_client_task(parameters: pointer); stdcall;
var
  status: uds_status;
  config_physical: uds_msgconfig;
  msg_request: uds_msg;
  request_confirmation: uds_msg;
  response: uds_msg;
  t_params: ^task_params;
begin
  // Initialize variables
  FillChar(&config_physical, sizeof(config_physical), 0);
  FillChar(&msg_request, sizeof(msg_request), 0);
  FillChar(&request_confirmation, sizeof(request_confirmation), 0);
  FillChar(&response, sizeof(response), 0);
  t_params := parameters;

  // Initialize a physical configuration
  config_physical.can_id :=
    UInt32(uds_can_id.PUDS_CAN_ID_ISO_15765_4_PHYSICAL_REQUEST_1);
  config_physical.can_msgtype := PCANTP_CAN_MSGTYPE_STANDARD;
  config_physical.nai.protocol := PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL;
  config_physical.nai.target_type := PCANTP_ISOTP_ADDRESSING_PHYSICAL;
  config_physical.typem := PUDS_MSGTYPE_USDT;
  config_physical.nai.source_addr :=
    UInt16(uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT);
  config_physical.nai.target_addr :=
    UInt16(uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_1);
  config_physical.nai.extension_addr := 0;

  // Execute TesterPresent and wait response
  repeat
    // Wait before transmitting the next message
    Sleep(UDS_REQUEST_TIMING_MS);

    status := TUDSApi.SvcTesterPresent_2013(t_params.client_handle,
      config_physical, &msg_request, PUDS_SVC_PARAM_TP_ZSUBF);
    Writeln(Format('[UDS] Execute TesterPresent service: %s',
      [UDS_STATUS_OK_KO(status)]));
    status := TUDSApi.WaitForService_2013(t_params.client_handle, @msg_request,
      response, @request_confirmation);
    Writeln(Format('[UDS] Wait for service: %s', [UDS_STATUS_OK_KO(status)]));
    status := TUDSApi.MsgFree_2013(&msg_request);
    Writeln(Format('[UDS] Free request message: %s',
      [UDS_STATUS_OK_KO(status)]));
    status := TUDSApi.MsgFree_2013(&request_confirmation);
    Writeln(Format('[UDS] Free request confirmation message: %s',
      [UDS_STATUS_OK_KO(status)]));
    status := TUDSApi.MsgFree_2013(&response);
    Writeln(Format('[UDS] Free response message: %s',
      [UDS_STATUS_OK_KO(status)]));
  until t_params.stop_task = true;
end;

/// <summary>ISOTP client task: read and print isotp messages</summary>
/// <param name="parameters">pointer on task_params structures</param>
Procedure isotp_client_task(parameters: pointer); stdcall;
var
  status: cantp_status;
  receive_event: THandle;
  null_handle: UInt64;
  t_params: ^task_params;
  rx_msg: cantp_msg;
  wait_result: DWORD;
  buffer: array [0 .. 255] of Char;
  isotp_param: UInt8;
  i: Integer;
  data_pointer: PByte;
begin
  // Initialize variables
  null_handle := 0;
  t_params := parameters;
  FillChar(&rx_msg, sizeof(rx_msg), 0);
  buffer[0] := #0;

  // Configure isotp to get any messages (disable can identifier filtering)
  isotp_param := 0;
  status := TCanTpApi.SetValue_2016(t_params.client_handle,
    PCANTP_PARAMETER_FILTER_CAN_ID, PByte(@isotp_param), sizeof(isotp_param));
  Writeln(Format('[ISOTP] Disable can identifier filtering in isotp: %s',
    [ISOTP_STATUS_OK_KO(status)]));

  // Configure isotp to get a copy of UDS messages
  isotp_param := 1;
  status := TCanTpApi.SetValue_2016(t_params.client_handle,
    PCANTP_PARAMETER_KEEP_HIGHER_LAYER_MESSAGES, PByte(@isotp_param),
    sizeof(isotp_param));
  Writeln(Format('[ISOTP] Activate higher layer messages in isotp: %s',
    [ISOTP_STATUS_OK_KO(status)]));

  // Create a isotp receive event
  receive_event := CreateEvent(nil, false, false, nil);
  status := TCanTpApi.SetValue_2016(t_params.client_handle,
    PCANTP_PARAMETER_RECEIVE_EVENT, PByte(@receive_event),
    sizeof(receive_event));
  Writeln(Format('[ISOTP] Set isotp receive event parameter: %s',
    [ISOTP_STATUS_OK_KO(status)]));

  // Read and print ISOTP messages
  repeat
    wait_result := WaitForSingleObject(receive_event, 1000);
    Writeln(Format('[ISOTP] Wait a receive event from isotp: %s',
      [OK_KO(wait_result = WAIT_OBJECT_0)]));

    // Read ISOTP messages
    repeat
      status := TCanTpApi.Read_2016(t_params.client_handle, &rx_msg);

      // Check if we received a message
      if not TCanTpApi.StatusIsOk_2016(status, PCANTP_STATUS_NO_MESSAGE, false)
      then
      begin
        Writeln(Format('[ISOTP] Read ISOTP message: %s',
          [ISOTP_STATUS_OK_KO(status)]));
        if ((rx_msg.msgdata_any.length > 0) and
          ((rx_msg.msgdata_any.data^ = Byte
          (uds_service.PUDS_SERVICE_SI_TesterPresent)) or
          (rx_msg.msgdata_any.data^ = Byte
          (uds_service.PUDS_SERVICE_SI_TesterPresent) +
          TUDSApi.PUDS_SI_POSITIVE_RESPONSE))) then
        begin
          // This message is a TesterPresent message, PCANTP_PARAMETER_KEEP_HIGHER_LAYER_MESSAGES
          // option must be activated to get these messages.
          Writeln('[ISOTP] Message is a TesterPresent service message.');
        end
        else if (TCanTpApi.StatusIsOk_2016(status, PCANTP_STATUS_OK, false) and
          (rx_msg.msgdata_any.length < BUFFER_SIZE)) then
        begin
          // This message is a CAN message received by ISOTP
          data_pointer := PByte(rx_msg.msgdata_any.data);
          for i := 0 to rx_msg.msgdata_any.length - 1 do
          begin
            buffer[i] := Char(data_pointer^);
            inc(data_pointer);
          end;
          buffer[rx_msg.msgdata_any.length] := #0;
          Writeln(Format
            ('[ISOTP] Message from 0x%s can identifier contains "%s"',
            [Lowercase(Format('%x', [Integer(rx_msg.can_info.can_id)])),
            buffer]));
        end;

        status := TCanTpApi.MsgDataFree_2016(&rx_msg);
        Writeln(Format('[ISOTP] Free RX message: %s',
          [ISOTP_STATUS_OK_KO(status)]));
      end;
    until TCanTpApi.StatusIsOk_2016(status, PCANTP_STATUS_NO_MESSAGE, false);
  until t_params.stop_task = true;

  // Close receive event
  status := TCanTpApi.SetValue_2016(t_params.client_handle,
    PCANTP_PARAMETER_RECEIVE_EVENT, PByte(@null_handle), sizeof(null_handle));
  Writeln(Format('[ISOTP] Stop ISOTP receive event: %s',
    [ISOTP_STATUS_OK_KO(status)]));
  CloseHandle(receive_event);
  Writeln(Format('[ISOTP] Close ISOTP receive event: %s',
    [ISOTP_STATUS_OK_KO(status)]));
end;

/// <summary>Entry point of the program, start a UDS channel, ask in the same time UDS TesterPresent and isotp request</summary>
/// <returns>By convention, return success.</returns>
var
  buffer: array [0 .. 255] of AnsiChar;
  status: uds_status;
  client_handle: cantp_handle;
  uds_client: THandle;
  isotp_client: THandle;
  uds_client_id: DWORD;
  isotp_client_id: DWORD;
  t_params: task_params;
  dummy: Char;

begin
  try
    // Initialize variables
    buffer[0] := #0;
    t_params.stop_task := false;

    // TODO: modify the value according to your available PCAN devices.
    client_handle := PCANTP_HANDLE_USBBUS1;
    t_params.client_handle := client_handle;

    // Print version informations
    status := TUDSApi.GetValue_2013(PCANTP_HANDLE_NONEBUS,
      PUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE);
    Writeln(Format('PCAN-UDS API Version - %s: %s',
      [buffer, UDS_STATUS_OK_KO(status)]));

    // Initialize client channel
    status := TUDSApi.Initialize_2013(client_handle, PCANTP_BAUDRATE_500K);
    Writeln(Format('Initialize channel: %s', [UDS_STATUS_OK_KO(status)]));

    // Start uds and isotp clients
    uds_client := CreateThread(nil, 0, @uds_client_task, @t_params, 0,
      &uds_client_id);
    if (uds_client <> 0) then
    begin
      isotp_client := CreateThread(nil, 0, @isotp_client_task, @t_params, 0,
        &isotp_client_id);
      if (isotp_client <> 0) then
      begin
        Sleep(CLIENT_EXECUTION_TIME_MS);
        t_params.stop_task := true;
        WaitForSingleObject(isotp_client, INFINITE);
        CloseHandle(isotp_client);
      end;
      WaitForSingleObject(uds_client, INFINITE);
      CloseHandle(uds_client);
    end;

    // Close channel
    status := TUDSApi.Uninitialize_2013(client_handle);
    Writeln(Format('Uninitialize channel: %s', [UDS_STATUS_OK_KO(status)]));

    // Exit
    Writeln('Press any key to continue...');
    Readln(dummy);

  except
    on E: Exception do
      Writeln(E.Classname, ': ', E.Message);
  end;

end.
