import time
import copy
from PCAN_UDS_2013 import *

IS_WINDOWS = platform.system() == 'Windows'

# Support events
if not IS_WINDOWS:
	import select
	__LIBC_OBJ = cdll.LoadLibrary("libc.so.6")

WAIT_OBJECT_0 = 0
def NULL_HANDLE():
	return c_void_p(0) if IS_WINDOWS else c_int(0)

# Support keyboard
if IS_WINDOWS:
	from msvcrt import getch, kbhit
	def install_keyboardHit():
		return 0
	def desinstall_keyboardHit(old_settings):
		pass
	def keyboardHit():
		return kbhit() != 0
	def getInput():
		key = getch()
		try:
			key_decode = key.decode('utf-8')
			return key_decode
		except UnicodeDecodeError:
			return key
else:
	import tty
	import termios
	def install_keyboardHit():
		old_settings = termios.tcgetattr(sys.stdin)
		tty.setcbreak(sys.stdin.fileno())
		return old_settings
	def desinstall_keyboardHit(old_settings):
		termios.tcsetattr(sys.stdin, termios.TCSADRAIN, old_settings)
	def keyboardHit():
		return select.select([sys.stdin], [], [], 0) == ([sys.stdin], [], [])
	def getInput():
		return sys.stdin.read(1)

# UDS library
objPCANUds = PCAN_UDS_2013()

# Help functions
def OK_KO(test):
	return "OK" if test else "KO"

def STATUS_OK_KO(test):
	return OK_KO(objPCANUds.StatusIsOk_2013(test, PUDS_STATUS_OK, False))

# Definitions
BUFFER_SIZE = 256

#
# Main entry point of the program, start a small server wich only support ECUReset service
# This server listen only 0x123 can id and respond with 0x124 can id in UUDT mode.
#

try:
	old_settings = install_keyboardHit()

	# Initialize variables
	tp_handle = PCANTP_HANDLE_USBBUS2 # TODO : modify the value according to your available PCAN devices.
	server_address = c_uint16(PUDS_ISO_15765_4_ADDR_ECU_1)

	# Print version informations
	buffer = create_string_buffer(BUFFER_SIZE)
	status = objPCANUds.GetValue_2013(PCANTP_HANDLE_NONEBUS, PUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE)
	print("PCAN-UDS API Version - %s: %s" %(buffer.value, STATUS_OK_KO(status)))

	# Initialize server
	status = objPCANUds.Initialize_2013(tp_handle, PCANTP_BAUDRATE_500K, 0, 0, 0)
	print("Initialize channel: %s" %(STATUS_OK_KO(status)))

	# Set server address parameter
	status = objPCANUds.SetValue_2013(tp_handle, PUDS_PARAMETER_SERVER_ADDRESS, server_address, sizeof(server_address))
	print("Set server address: %s" %(STATUS_OK_KO(status)))

	# Set a receive event
	if IS_WINDOWS:
		receive_event = c_void_p(windll.kernel32.CreateEventA(None, 0,0,None))
		res = objPCANUds.SetValue_2013(tp_handle, PUDS_PARAMETER_RECEIVE_EVENT, receive_event, sizeof(receive_event))
		print("Set receive event parameter: %s" %( STATUS_OK_KO(res)))
	else:
		receive_event = NULL_HANDLE()
		res = objPCANUds.GetValue_2013(tp_handle, PUDS_PARAMETER_RECEIVE_EVENT, receive_event, sizeof(receive_event))
		print("Get receive event parameter: %s" %( STATUS_OK_KO(res)))

	# Add filter on 0x123 and 0x124 can id
	status = objPCANUds.AddCanIdFilter_2013(tp_handle, 0x123)
	print("Add can id filter (0x123): %s" %(STATUS_OK_KO(status)))
	status = objPCANUds.AddCanIdFilter_2013(tp_handle, 0x124)
	print("Add can id filter (0x124): %s" %(STATUS_OK_KO(status)))

	# Initialize the response configuration
	config_physical = uds_msgconfig()
	config_physical.can_id = 0x124
	config_physical.can_msgtype = PCANTP_CAN_MSGTYPE_STANDARD
	config_physical.nai.protocol = PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL
	config_physical.nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL
	config_physical.type = PUDS_MSGTYPE_UUDT

	# Read while user do not press Q
	print("Start listening, press Q to quit.")
	stop = False
	while stop == False:

		# Wait a receive event on receiver
		#	note: timeout is used to check keyboard hit.
		if IS_WINDOWS:
			windows_wait_result = c_uint64(windll.kernel32.WaitForSingleObject(receive_event, 1000))
			wait_result = True if windows_wait_result.value == WAIT_OBJECT_0 else False
		else:
			readable, _, _ = select.select([receive_event.value], [], [], 3)
			wait_result = True if len(readable) > 0 else False

		# If we get a receive event
		if wait_result:

			request_msg = uds_msg()
			response_msg = uds_msg()
			read_status = PUDS_STATUS_OK
			while not objPCANUds.StatusIsOk_2013(read_status, PUDS_STATUS_NO_MESSAGE, False):

				# Read first available message (no filtering based on message's type is set):
				read_status = objPCANUds.Read_2013(tp_handle, request_msg, None, None)
				print("Try to read a message: %s" %(STATUS_OK_KO(read_status)))
				if objPCANUds.StatusIsOk_2013(read_status, PUDS_STATUS_OK, False):

					# We receive a request, check its length and if it is not a loopback message and if it is a UUDT message
					if (request_msg.type & PUDS_MSGTYPE_UUDT.value) == PUDS_MSGTYPE_UUDT.value and request_msg.msg.msgdata.any.contents.length >= 1 and (request_msg.msg.msgdata.any.contents.flags & PCANTP_MSGFLAG_LOOPBACK.value) == 0:

						# This is a valid request, switch services
						service_id = request_msg.links.service_id.contents.value
						if service_id == objPCANUds.PUDS_SI_ECUReset:

							# Allocate response message
							if request_msg.links.param[0] == objPCANUds.PUDS_SVC_PARAM_ER_ERPSD:
								status = objPCANUds.MsgAlloc_2013(response_msg, config_physical, 3)
								response_msg.links.param[1] = 0x66	# power down time
							else:
								status = objPCANUds.MsgAlloc_2013(response_msg, config_physical, 2)
							print("Prepare response message for ECUReset service: %s" %(STATUS_OK_KO(status)))

							if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):

								# Fill parameters
								response_msg.links.service_id[0] = c_ubyte(objPCANUds.PUDS_SI_ECUReset + objPCANUds.PUDS_SI_POSITIVE_RESPONSE)
								response_msg.links.param[0] = request_msg.links.param[0]

								# Write response message
								status = objPCANUds.Write_2013(tp_handle, response_msg)
								print("Write response message for ECUReset service: %s" %(STATUS_OK_KO(status)))

							# Free response message (and clean memory in order to reallocate later)
							status = objPCANUds.MsgFree_2013(response_msg)
							print("Free response message: %s" %(STATUS_OK_KO(status)))

						else:
							print("Unknown service (0x%02x)" %(service_id))

				# Free request message (and clean memory in order to reallocate later)
				status = objPCANUds.MsgFree_2013(request_msg)
				print("Free request message: %s" %(STATUS_OK_KO(status)))


		# Quit when user press Q
		if keyboardHit():
			keyboard_res = getInput()
			if keyboard_res == 'Q' or keyboard_res == 'q':
				stop = True

	# Close receive event
	if IS_WINDOWS:
		status = objPCANUds.SetValue_2013(tp_handle, PUDS_PARAMETER_RECEIVE_EVENT, NULL_HANDLE(), sizeof(NULL_HANDLE()))
		print("Stop receive event: %s" %(STATUS_OK_KO(status)))
		boolean_status = windll.kernel32.CloseHandle(receive_event)
		print("Close receive event: %s" %(OK_KO(boolean_status != 0)))

	# Close server
	status = objPCANUds.Uninitialize_2013(tp_handle)
	print("Uninitialize channel: %s" %(STATUS_OK_KO(status)))

	# Exit
	print("Press any key to continue...")
	getInput()

finally:
	desinstall_keyboardHit(old_settings)



