import time
from time import sleep
import random
from PCAN_UDS_2013 import *

IS_WINDOWS = platform.system() == 'Windows'

# Support events
if not IS_WINDOWS:
	import select
	__LIBC_OBJ = cdll.LoadLibrary("libc.so.6")

WAIT_OBJECT_0 = 0
def NULL_HANDLE():
	return c_void_p(0) if IS_WINDOWS else c_int(0)

# Support keyboard
if IS_WINDOWS:
	from msvcrt import getch, kbhit
	def install_keyboardHit():
		return 0
	def desinstall_keyboardHit(old_settings):
		pass
	def keyboardHit():
		return kbhit() != 0
	def getInput():
		key = getch()
		try:
			key_decode = key.decode('utf-8')
			return key_decode
		except UnicodeDecodeError:
			return key
else:
	import tty
	import termios
	def install_keyboardHit():
		old_settings = termios.tcgetattr(sys.stdin)
		tty.setcbreak(sys.stdin.fileno())
		return old_settings
	def desinstall_keyboardHit(old_settings):
		termios.tcsetattr(sys.stdin, termios.TCSADRAIN, old_settings)
	def keyboardHit():
		return select.select([sys.stdin], [], [], 0) == ([sys.stdin], [], [])
	def getInput():
		return sys.stdin.read(1)

# UDS library
objPCANUds = PCAN_UDS_2013()

# Help functions
def getTimeIndicationInMilliseconds():
	"""
	Get a time indication in milliseconds
	"""
	return time.time()*1000

def OK_KO(test):
	return "OK" if test else "KO"

def STATUS_OK_KO(test):
	return OK_KO(objPCANUds.StatusIsOk_2013(test, PUDS_STATUS_OK, False))

def ConvertNai(nai, can_msgtype):
	"""
	Helper: convert cantp network address information into uds network information

	parameters:
		nai: cantp network address information structure
		can_msgtype: cantp message type
	returns: UDS network address information structure
	"""
	result = uds_netaddrinfo ()
	is_29bits = ((can_msgtype & PCANTP_CAN_MSGTYPE_EXTENDED.value) == PCANTP_CAN_MSGTYPE_EXTENDED.value)

	result.source_addr = nai.source_addr
	result.target_addr = nai.target_addr
	result.target_type = nai.target_type
	result.extension_addr = nai.extension_addr

	if nai.format == PCANTP_ISOTP_FORMAT_ENHANCED.value:
		result.protocol = PUDS_MSGPROTOCOL_ISO_15765_3_29B_ENHANCED
	elif nai.format == PCANTP_ISOTP_FORMAT_EXTENDED.value:
		result.protocol = PUDS_MSGPROTOCOL_ISO_15765_2_29B_EXTENDED if is_29bits else PUDS_MSGPROTOCOL_ISO_15765_2_11B_EXTENDED
	elif nai.format ==  PCANTP_ISOTP_FORMAT_FIXED_NORMAL.value:
		result.protocol = PUDS_MSGPROTOCOL_ISO_15765_2_29B_FIXED_NORMAL
	elif nai.format ==  PCANTP_ISOTP_FORMAT_MIXED.value:
		result.protocol = PUDS_MSGPROTOCOL_ISO_15765_2_29B_REMOTE if is_29bits else PUDS_MSGPROTOCOL_ISO_15765_2_11B_REMOTE
	elif nai.format ==  PCANTP_ISOTP_FORMAT_NORMAL.value:
		result.protocol = PUDS_MSGPROTOCOL_ISO_15765_2_29B_NORMAL if is_29bits else PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL
	else:
		result.protocol = PUDS_MSGPROTOCOL_NONE

	return result

def HasSubFunction(service_identifier):
	"""
	Helper: check if a service has a subfunction

	parameters:
		service_identifier: service identifier
	returns: boolean: yes (true) or not (false)
	"""

	if service_identifier == objPCANUds.PUDS_SI_DiagnosticSessionControl:
		res = True
	elif service_identifier ==  objPCANUds.PUDS_SI_ECUReset:
		res = True
	elif service_identifier ==  objPCANUds.PUDS_SI_SecurityAccess:
		res = True
	elif service_identifier ==  objPCANUds.PUDS_SI_CommunicationControl:
		res = True
	elif service_identifier ==  objPCANUds.PUDS_SI_TesterPresent:
		res = True
	elif service_identifier ==  objPCANUds.PUDS_SI_AccessTimingParameter:
		res = True
	elif service_identifier ==  objPCANUds.PUDS_SI_SecuredDataTransmission:
		res = False
	elif service_identifier ==  objPCANUds.PUDS_SI_ControlDTCSetting:
		res = True
	elif service_identifier ==  objPCANUds.PUDS_SI_ResponseOnEvent:
		res = True
	elif service_identifier ==  objPCANUds.PUDS_SI_LinkControl:
		res = True
	elif service_identifier ==  objPCANUds.PUDS_SI_ReadDataByIdentifier:
		res = False
	elif service_identifier ==  objPCANUds.PUDS_SI_ReadMemoryByAddress:
		res = False
	elif service_identifier == objPCANUds.PUDS_SI_ReadScalingDataByIdentifier:
		res = False
	elif service_identifier == objPCANUds.PUDS_SI_ReadDataByPeriodicIdentifier:
		res = False
	elif service_identifier == objPCANUds.PUDS_SI_DynamicallyDefineDataIdentifier:
		res = True
	elif service_identifier == objPCANUds.PUDS_SI_WriteDataByIdentifier:
		res = False
	elif service_identifier == objPCANUds.PUDS_SI_WriteMemoryByAddress:
		res = False
	elif service_identifier == objPCANUds.PUDS_SI_ClearDiagnosticInformation:
		res = False
	elif service_identifier == objPCANUds.PUDS_SI_ReadDTCInformation:
		res = True
	elif service_identifier == objPCANUds.PUDS_SI_InputOutputControlByIdentifier:
		res = False
	elif service_identifier == objPCANUds.PUDS_SI_RoutineControl:
		res = True
	elif service_identifier == objPCANUds.PUDS_SI_RequestDownload:
		res = False
	elif service_identifier == objPCANUds.PUDS_SI_RequestUpload:
		res = False
	elif service_identifier == objPCANUds.PUDS_SI_TransferData:
		res = False
	elif service_identifier == objPCANUds.PUDS_SI_RequestTransferExit:
		res = False
	elif service_identifier == objPCANUds.PUDS_SI_RequestFileTransfer:
		res = False
	elif service_identifier == objPCANUds.PUDS_SI_Authentication:
		res = True
	else:
		res = False

	return res

def create_SvcECUReset_response(config, request_msg, response_msg):
	"""
	Create the ECUReset response message

	parameters:
		config: Input, UDS configuration
		request_msg: Input, received request, containing ECUReset parameters
		response_msg: Output, contains the generated response
	returns: A uds_status code. PUDS_STATUS_OK is returned on success
	"""

	# Allocate response message
	if request_msg.links.param[0] == objPCANUds.PUDS_SVC_PARAM_ER_ERPSD:
		status = objPCANUds.MsgAlloc_2013(response_msg, config, 3)
		response_msg.links.param[1] = 0x66	# Power down time
	else:
		status = objPCANUds.MsgAlloc_2013(response_msg, config, 2)

	print("Allocate ECUReset response: %s" %(STATUS_OK_KO(status)))

	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		# Fill parameters
		response_msg.links.service_id[0] = c_ubyte(objPCANUds.PUDS_SI_ECUReset + objPCANUds.PUDS_SI_POSITIVE_RESPONSE)
		response_msg.links.param[0] = request_msg.links.param[0]

	return status

def create_SvcWriteMemoryByAddress_response(config, request_msg, response_msg):
	"""
	Create the WriteMemoryByAddress response message

	parameters:
		config: Input, UDS configuration
		request_msg: Input, received request, containing request parameters
		response_msg: Output, contains the generated response
	returns: A uds_status code. PUDS_STATUS_OK is returned on success
	"""

	# Read memory_size_size & memory_address_size
	memory_size_size = (request_msg.links.param[0] >> 4) & 0xF
	memory_address_size = request_msg.links.param[0] & 0xF
	length = 2 + memory_size_size + memory_address_size

	# Allocate message
	status = objPCANUds.MsgAlloc_2013(response_msg, config, length)
	print("Allocate WriteMemoryByAddress response: %s" %(STATUS_OK_KO(status)))

	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		# copy Address and Memory parameters
		response_msg.links.param[0] = request_msg.links.param[0]
		for i in range(memory_address_size):
			response_msg.links.param[1+i] = request_msg.links.param[1+i]
		for i in range(memory_size_size):
			response_msg.links.param[1+memory_address_size+i] = request_msg.links.param[1+memory_address_size+i]
		response_msg.links.service_id[0] = c_ubyte(objPCANUds.PUDS_SI_WriteMemoryByAddress + objPCANUds.PUDS_SI_POSITIVE_RESPONSE)

	return status

def create_SvcDynamicallyDefineDataIdentifier_response(config, request_msg, response_msg):
	"""
	Create the DynamicallyDefineDataIdentifier response message

	parameters:
		config: Input, UDS configuration
		request_msg: Input, received request, containing request parameters
		response_msg: Output, contains the generated response
	returns: A uds_status code. PUDS_STATUS_OK is returned on success
	"""

	# Allocate and fill message
	if request_msg.msg.msgdata.any.contents.length>=4:
		status = objPCANUds.MsgAlloc_2013(response_msg, config, 4)
		print("Allocate DynamicallyDefineDataIdentifier response: %s" %(STATUS_OK_KO(status)))

		if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
			# copy Data
			response_msg.links.service_id[0] = c_ubyte(objPCANUds.PUDS_SI_DynamicallyDefineDataIdentifier + objPCANUds.PUDS_SI_POSITIVE_RESPONSE)
			response_msg.links.param[0] = request_msg.links.param[0]
			response_msg.links.param[1] = request_msg.links.param[1]
			response_msg.links.param[2] = request_msg.links.param[2]

	else:
		# Clear all data identifier request
		status = objPCANUds.MsgAlloc_2013(response_msg, config, 2)
		print("Allocate DynamicallyDefineDataIdentifier response: %s" %(STATUS_OK_KO(status)))

		if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
			# Copy subfunction
			response_msg.links.service_id[0] = c_ubyte(objPCANUds.PUDS_SI_DynamicallyDefineDataIdentifier + objPCANUds.PUDS_SI_POSITIVE_RESPONSE)
			response_msg.links.param[0] = request_msg.links.param[0]


	return status

def create_dummy_response(config, request_msg, response_msg):
	"""
	Create the dummy response message

	parameters:
		config: Input, UDS configuration
		request_msg: Input, received request, containing request parameters
		response_msg: Output, contains the generated response
	returns: A uds_status code. PUDS_STATUS_OK is returned on success
	"""

	# Set a random message size
	length = 1 + request_msg.links.service_id[0] + random.randrange(1, 200) % 200
	length = PCANTP_MAX_LENGTH_ISOTP2004 if length > PCANTP_MAX_LENGTH_ISOTP2004 else length

	# Allocate response message
	status = objPCANUds.MsgAlloc_2013(response_msg, config, length)
	print("Allocate dummy response: %s" %(STATUS_OK_KO(status)))

	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		# Fake a positive response
		response_msg.links.service_id[0] = request_msg.links.service_id[0] + objPCANUds.PUDS_SI_POSITIVE_RESPONSE

		# Fill with dummy data
		response_msg.links.param[0] = request_msg.links.service_id[0]
		for i in range(1, length - 1, 1):	# (length - 1) as positive response SID uses 1 byte
			response_msg.links.param[i] = i + 1

		# Do not set PUDS_FLAG_SUPPRESS_POSITIVE_RESPONSE flag when service has a subfunction
		if HasSubFunction(request_msg.links.service_id[0]):
			response_msg.msg.msgdata.any.contents.data[1] = 0

	return status

def create_SvcRequestTransferExit_response(config, response_msg):
	"""
	Create the RequestTransferExit response message

	parameters:
		config: Input, UDS configuration
		request_msg: Input, received request, containing request parameters
		response_msg: Output, contains the generated response
	returns: A uds_status code. PUDS_STATUS_OK is returned on success
	"""

	# Allocate message
	length = 1 + 1 + random.randrange(1,200) % 50
	status = objPCANUds.MsgAlloc_2013(response_msg, config, length)
	print("Allocate RequestTransferExit response: %s" %(STATUS_OK_KO(status)))

	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		# Fill with dummy data
		for i in range(0, length - 1, 1):	# (length - 1) as positive response SID uses 1 byte
			response_msg.links.param[i] = i + 1
		response_msg.links.service_id[0] = c_ubyte(objPCANUds.PUDS_SI_RequestTransferExit + objPCANUds.PUDS_SI_POSITIVE_RESPONSE)

	return status

def create_SvcRequestUpload_response(config, response_msg):
	"""
	Create the SvcRequestUpload response message

	parameters:
		config: Input, UDS configuration
		request_msg: Input, received request, containing request parameters
		response_msg: Output, contains the generated response
	returns: A uds_status code. PUDS_STATUS_OK is returned on success
	"""

	# Allocate message
	length = 2 + 1 + random.randrange(1,200) % 50
	status = objPCANUds.MsgAlloc_2013(response_msg, config, length)
	print("Allocate RequestUpload response: %s" %(STATUS_OK_KO(status)))

	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		# Max number of block length = 0xF
		response_msg.links.param[0] = 0xF0

		# Fill with dummy data
		for i in range(1, length - 1, 1):	# (length - 1) as positive response SID uses 1 byte
			response_msg.links.param[i] = i + 1

		response_msg.links.service_id[0] = c_ubyte(objPCANUds.PUDS_SI_RequestUpload + objPCANUds.PUDS_SI_POSITIVE_RESPONSE)

	return status

def create_SvcRequestDownload_response(config, response_msg):
	"""
	Create the SvcRequestDownload response message

	parameters:
		config: Input, UDS configuration
		request_msg: Input, received request, containing request parameters
		response_msg: Output, contains the generated response
	returns: A uds_status code. PUDS_STATUS_OK is returned on success
	"""

	# Allocate message
	length = 2 + 1 + random.randrange(1,200) % 50
	status = objPCANUds.MsgAlloc_2013(response_msg, config, length)
	print("Allocate RequestDownload response: %s" %(STATUS_OK_KO(status)))

	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		# Max number of block length = 0xF
		response_msg.links.param[0] = 0xF0

		# fill with dummy data
		for i in range(1, length - 1, 1):	# (length - 1) as positive response SID uses 1 byte
			response_msg.links.param[i] = i + 1

		response_msg.links.service_id[0] = c_ubyte(objPCANUds.PUDS_SI_RequestDownload + objPCANUds.PUDS_SI_POSITIVE_RESPONSE)

	return status

def create_SvcRoutineControl_response(config, request_msg, response_msg):
	"""
	Create the RoutineControl response message

	parameters:
		config: Input, UDS configuration
		request_msg: Input, received request, containing request parameters
		response_msg: Output, contains the generated response
	returns: A uds_status code. PUDS_STATUS_OK is returned on success
	"""

	# Allocate response
	status = objPCANUds.MsgAlloc_2013(response_msg, config, 4)
	print("Allocate RoutineControl response: %s" %(STATUS_OK_KO(status)))

	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		# Copy Data
		response_msg.links.param[0] = request_msg.links.param[0]
		response_msg.links.param[1] = request_msg.links.param[1]
		response_msg.links.param[2] = request_msg.links.param[2]

		# Routine status record not implemented

		response_msg.links.service_id[0] = c_ubyte(objPCANUds.PUDS_SI_RoutineControl + objPCANUds.PUDS_SI_POSITIVE_RESPONSE)

	return status

def create_SvcInputOutputControlByIdentifier_response(config, request_msg, response_msg):
	"""
	Create the InputOutputControlByIdentifier response message

	parameters:
		config: Input, UDS configuration
		request_msg: Input, received request, containing request parameters
		response_msg: Output, contains the generated response
	returns: A uds_status code. PUDS_STATUS_OK is returned on success
	"""

	status = objPCANUds.MsgAlloc_2013(response_msg, config, 3)
	print("Allocate InputOutputControlByIdentifier response: %s" %(STATUS_OK_KO(status)))

	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		# Copy Data
		response_msg.links.param[0] = request_msg.links.param[0]
		response_msg.links.param[1] = request_msg.links.param[1]

		# Control status record not implemented
		response_msg.links.service_id[0] = c_ubyte(objPCANUds.PUDS_SI_InputOutputControlByIdentifier + objPCANUds.PUDS_SI_POSITIVE_RESPONSE)

	return status

def create_SvcClearDiagnosticInformation_response(config, response_msg):
	"""
	Create the ClearDiagnosticInformation response message

	parameters:
		config: Input, UDS configuration
		request_msg: Input, received request, containing request parameters
		response_msg: Output, contains the generated response
	returns: A uds_status code. PUDS_STATUS_OK is returned on success
	"""

	status = objPCANUds.MsgAlloc_2013(response_msg, config, 1)
	print("Allocate ClearDiagnosticInformation response: %s" %(STATUS_OK_KO(status)))

	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		response_msg.links.service_id[0] = c_ubyte(objPCANUds.PUDS_SI_ClearDiagnosticInformation + objPCANUds.PUDS_SI_POSITIVE_RESPONSE)

	return status

def create_SvcWriteDataByIdentifier_response(config, request_msg, response_msg):
	"""
	Create the WriteDataByIdentifier response message

	parameters:
		config: Input, UDS configuration
		request_msg: Input, received request, containing request parameters
		response_msg: Output, contains the generated response
	returns: A uds_status code. PUDS_STATUS_OK is returned on success
	"""

	status = objPCANUds.MsgAlloc_2013(response_msg, config, 3)
	print("Allocate WriteDataByIdentifier response: %s" %(STATUS_OK_KO(status)))

	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		# Copy Data
		response_msg.links.param[0] = request_msg.links.param[0]
		response_msg.links.param[1] = request_msg.links.param[1]
		response_msg.links.service_id[0] = c_ubyte(objPCANUds.PUDS_SI_WriteDataByIdentifier + objPCANUds.PUDS_SI_POSITIVE_RESPONSE)

	return status

def create_SvcReadDataByPeriodicIdentifier_response(config, response_msg):
	"""
	Create the ReadDataByPeriodicIdentifier response message

	parameters:
		config: Input, UDS configuration
		request_msg: Input, received request, containing request parameters
		response_msg: Output, contains the generated response
	returns: A uds_status code. PUDS_STATUS_OK is returned on success
	"""

	status = objPCANUds.MsgAlloc_2013(response_msg, config, 1)
	print("Allocate ReadDataByPeriodicIdentifier response: %s" %(STATUS_OK_KO(status)))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		response_msg.links.service_id[0] = c_ubyte(objPCANUds.PUDS_SI_ReadDataByPeriodicIdentifier + objPCANUds.PUDS_SI_POSITIVE_RESPONSE)
	return status

def create_SvcReadMemoryByAddress_response(config, request_msg, response_msg):
	"""
	Create the ReadMemoryByAddress response message

	parameters:
		config: Input, UDS configuration
		request_msg: Input, received request, containing request parameters
		response_msg: Output, contains the generated response
	returns: A uds_status code. PUDS_STATUS_OK is returned on success
	"""

	# Read memory_size_size = bits [7..4]
	length = 1 + ((request_msg.links.param[1] >> 4) & 0xF)

	# Allocate message
	status = objPCANUds.MsgAlloc_2013(response_msg, config, length)
	print("Allocate ReadMemoryByAddress response: %s" %(STATUS_OK_KO(status)))

	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		# Fill with dummy data
		for i in range(0, length - 1, 1):	# (length - 1) as positive response SID uses 1 byte
			response_msg.links.param[i] = i + 1
		response_msg.links.service_id[0] = c_ubyte(objPCANUds.PUDS_SI_ReadMemoryByAddress + objPCANUds.PUDS_SI_POSITIVE_RESPONSE)

	return status

def create_SvcReadScalingDataByIdentifier_response(config, request_msg, response_msg):
	"""
	Create the ReadScalingDataByIdentifier response message

	parameters:
		config: Input, UDS configuration
		request_msg: Input, received request, containing request parameters
		response_msg: Output, contains the generated response
	returns: A uds_status code. PUDS_STATUS_OK is returned on success
	"""

	# Allocate message
	status = objPCANUds.MsgAlloc_2013(response_msg, config, 12)
	print("Allocate ReadScalingDataByIdentifier response: %s" %(STATUS_OK_KO(status)))

	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		response_msg.links.service_id[0] = c_ubyte(objPCANUds.PUDS_SI_ReadScalingDataByIdentifier + objPCANUds.PUDS_SI_POSITIVE_RESPONSE)

		# Copy Data
		response_msg.links.param[0] = request_msg.links.param[0]
		response_msg.links.param[1] = request_msg.links.param[1]

		# Create a formula Vehicule Speed = (0.75*x+30) km/h
		response_msg.links.param[2] = (0x0 << 4) | (0x1)	# unsigned numeric of 1 byte)
		response_msg.links.param[3] = 0x90	# formula, 0 data bytes
		response_msg.links.param[4] = 0x00	# formulaIdentifier = C0 * x + C1
		response_msg.links.param[5] = 0xE0	# C0 high byte
		response_msg.links.param[6] = 0x4B	# C0 low byte
		response_msg.links.param[7] = 0x00	# C1 high byte
		response_msg.links.param[8] = 0x1E	# C1 low byte
		response_msg.links.param[9] = 0xA0	# unit/format, 0 data bytes
		response_msg.links.param[10] = 0x30	# unit ID, km/h

	return status

def create_SvcReadDataByIdentifier_response(config, request_msg, response_msg):
	"""
	Create the ReadDataByIdentifier response message

	parameters:
		config: Input, UDS configuration
		request_msg: Input, received request, containing request parameters
		response_msg: Output, contains the generated response
	returns: A uds_status code. PUDS_STATUS_OK is returned on success
	"""

	# Allocate message
	length = ((request_msg.msg.msgdata.any.contents.length - 1) * 7 // 2) + 1
	status = objPCANUds.MsgAlloc_2013(response_msg, config, length)
	print("Allocate ReadDataByIdentifier response: %s" %(STATUS_OK_KO(status)))

	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		counter = 0
		for i in range(0,request_msg.msg.msgdata.any.contents.length - 1, 2):
			# Copy data
			response_msg.links.param[counter] = request_msg.links.param[i]
			counter += 1
			response_msg.links.param[counter] = request_msg.links.param[i + 1]
			counter += 1
			# Data record: fill with dummy data
			for j in range(5):
				response_msg.links.param[counter] = j + ord('A')
				counter += 1

		response_msg.links.service_id[0] = c_ubyte(objPCANUds.PUDS_SI_ReadDataByIdentifier + objPCANUds.PUDS_SI_POSITIVE_RESPONSE)

	return status

def create_SvcLinkControl_response(config, request_msg, response_msg):
	"""
	Create the LinkControl response message

	parameters:
		config: Input, UDS configuration
		request_msg: Input, received request, containing request parameters
		response_msg: Output, contains the generated response
	returns: A uds_status code. PUDS_STATUS_OK is returned on success
	"""

	status = objPCANUds.MsgAlloc_2013(response_msg, config, 2)
	print("Allocate LinkControl response: %s" %(STATUS_OK_KO(status)))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		response_msg.links.param[0] = request_msg.links.param[0]
		response_msg.links.service_id[0] = c_ubyte(objPCANUds.PUDS_SI_LinkControl + objPCANUds.PUDS_SI_POSITIVE_RESPONSE)
	return status

def create_SvcResponseOnEvent_response(config, request_msg, response_msg):
	"""
	Create the ResponseOnEvent response message

	parameters:
		config: Input, UDS configuration
		request_msg: Input, received request, containing request parameters
		response_msg: Output, contains the generated response
	returns: A uds_status code. PUDS_STATUS_OK is returned on success
	"""

	if request_msg.links.param[0] == objPCANUds.PUDS_SVC_PARAM_ROE_RAE:

		status = objPCANUds.MsgAlloc_2013(response_msg, config, 3)
		print("Allocate ResponseOnEvent response: %s" %(STATUS_OK_KO(status)))
		if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
			response_msg.links.param[1] = 0	# Number of activated events
											# Event type and service to respond to records not implemented
	else:

		status = objPCANUds.MsgAlloc_2013(response_msg, config, 4)
		print("Allocate ResponseOnEvent response: %s" %(STATUS_OK_KO(status)))
		if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
			response_msg.links.param[1] = 0								# Number of identified events
			response_msg.links.param[2] = request_msg.links.param[1]	# Event window time
																		# Event type and service to respond to records not implemented

	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		response_msg.links.param[0] = request_msg.links.param[0]
		response_msg.links.service_id[0] = c_ubyte(objPCANUds.PUDS_SI_ResponseOnEvent + objPCANUds.PUDS_SI_POSITIVE_RESPONSE)

	return status

def create_SvcControlDTCSetting_response(config, request_msg, response_msg):
	"""
	Create the ControlDTCSetting response message

	parameters:
		config: Input, UDS configuration
		request_msg: Input, received request, containing request parameters
		response_msg: Output, contains the generated response
	returns: A uds_status code. PUDS_STATUS_OK is returned on success
	"""

	status = objPCANUds.MsgAlloc_2013(response_msg, config, 2)
	print("Allocate ControlDTCSetting response: %s" %(STATUS_OK_KO(status)))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		response_msg.links.param[0] = request_msg.links.param[0]
		response_msg.links.service_id[0] = c_ubyte(objPCANUds.PUDS_SI_ControlDTCSetting + objPCANUds.PUDS_SI_POSITIVE_RESPONSE)
	return status

def create_SvcSecuredDataTransmission_response(config, response_msg):
	"""
	Create the SecuredDataTransmission response message

	parameters:
		config: Input, UDS configuration
		request_msg: Input, received request, containing request parameters
		response_msg: Output, contains the generated response
	returns: A uds_status code. PUDS_STATUS_OK is returned on success
	"""

	# Allocate message
	length = 1 + random.randrange(1,200) % 50
	status = objPCANUds.MsgAlloc_2013(response_msg, config, length)
	print("Allocate SecuredDataTransmission response: %s" %(STATUS_OK_KO(status)))

	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		# Fill with dummy data (check Security-SubLayer record defined in ISO-15764)
		for i in range(0, length - 1, 1):	# (length - 1) as positive response SID uses 1 byte
			response_msg.links.param[i] = i + 1

		response_msg.links.service_id[0] = c_ubyte(objPCANUds.PUDS_SI_SecuredDataTransmission + objPCANUds.PUDS_SI_POSITIVE_RESPONSE)

	return status

def create_SvcTesterPresent_response(config, request_msg, response_msg):
	"""
	Create the TesterPresent response message

	parameters:
		config: Input, UDS configuration
		request_msg: Input, received request, containing request parameters
		response_msg: Output, contains the generated response
	returns: A uds_status code. PUDS_STATUS_OK is returned on success
	"""

	status = objPCANUds.MsgAlloc_2013(response_msg, config, 2)
	print("Allocate TesterPresent response: %s" %(STATUS_OK_KO(status)))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		response_msg.links.param[0] = request_msg.links.param[0]
		response_msg.links.service_id[0] = c_ubyte(objPCANUds.PUDS_SI_TesterPresent + objPCANUds.PUDS_SI_POSITIVE_RESPONSE)
	return status

def create_SvcCommunicationControl_response(config, request_msg, response_msg):
	"""
	Create the CommunicationControl response message

	parameters:
		config: Input, UDS configuration
		request_msg: Input, received request, containing request parameters
		response_msg: Output, contains the generated response
	returns: A uds_status code. PUDS_STATUS_OK is returned on success
	"""

	status = objPCANUds.MsgAlloc_2013(response_msg, config, 2)
	print("Allocate CommunicationControl response: %s" %(STATUS_OK_KO(status)))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		response_msg.links.param[0] = request_msg.links.param[0]
		response_msg.links.service_id[0] = c_ubyte(objPCANUds.PUDS_SI_CommunicationControl + objPCANUds.PUDS_SI_POSITIVE_RESPONSE)
	return status

def create_SvcSecurityAccess_response(config, request_msg, response_msg):
	"""
	Create the SecurityAccess response message

	parameters:
		config: Input, UDS configuration
		request_msg: Input, received request, containing request parameters
		response_msg: Output, contains the generated response
	returns: A uds_status code. PUDS_STATUS_OK is returned on success
	"""

	# Allocate response message
	if request_msg.links.param[0] >= objPCANUds.PUDS_SVC_PARAM_SA_RSD_MIN\
		and request_msg.links.param[0] <= objPCANUds.PUDS_SVC_PARAM_SA_RSD_MAX\
		and request_msg.links.param[0] % 2 == 1:

		# Request security seed are Even values
		# Fill with dummy data
		length = PCANTP_MAX_LENGTH_ISOTP2004 if (1 + objPCANUds.PUDS_SI_SecurityAccess) > PCANTP_MAX_LENGTH_ISOTP2004 else 1 + objPCANUds.PUDS_SI_SecurityAccess
		status = objPCANUds.MsgAlloc_2013(response_msg, config, length)
		print("Allocate SecurityAccess response: %s" %(STATUS_OK_KO(status)))

		if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
			for i in range(1,length - 1, 1):	# (length - 1) as positive response SID uses 1 byteat
				response_msg.links.param[i] = i + 1

	else:
		status = objPCANUds.MsgAlloc_2013(response_msg, config, 2)
		print("Allocate SecurityAccess response: %s" %(STATUS_OK_KO(status)))

	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		response_msg.links.param[0] = request_msg.links.param[0]
		response_msg.links.service_id[0] = c_ubyte(objPCANUds.PUDS_SI_SecurityAccess + objPCANUds.PUDS_SI_POSITIVE_RESPONSE)

	return status

def create_SvcDiagnosticSessionControl_response(config, request_msg, response_msg):
	"""
	Create the DiagnosticSessionControl response message

	parameters:
		config: Input, UDS configuration
		request_msg: Input, received request, containing request parameters
		response_msg: Output, contains the generated response
	returns: A uds_status code. PUDS_STATUS_OK is returned on success
	"""

	# Allocate response message
	status = objPCANUds.MsgAlloc_2013(response_msg, config, 6)
	print("Allocate DiagnosticSessionControl response: %s" %(STATUS_OK_KO(status)))

	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		# Fill response
		response_msg.links.param[0] = request_msg.links.param[0]
		response_msg.links.param[1] = 0x00	# P2Can_Server_Max = 0x0010
		response_msg.links.param[2] = 0x10
		response_msg.links.param[3] = 0x03	# P2*Can_Server_Max = 0x03E8
		response_msg.links.param[4] = 0xE8
		response_msg.links.service_id[0] = c_ubyte(objPCANUds.PUDS_SI_DiagnosticSessionControl + objPCANUds.PUDS_SI_POSITIVE_RESPONSE)

	return status

def create_SvcTransferData_response(tp_handle,  config, request_msg, response_msg):
	"""
	Create the TransferData response message

	parameters:
		config: Input, UDS configuration
		request_msg: Input, received request, containing request parameters
		response_msg: Output, contains the generated response
	returns: A uds_status code. PUDS_STATUS_OK is returned on success
	"""

	# Initialize locale struct
	pending_response_msg = uds_msg()

	# Custom response to client_all_request example:
	#  a. service is requested functionally,
	#  b. 1st response is NRC response pending
	#  c. wait
	#  d. send correct response
	if request_msg.msg.msgdata.isotp.contents.netaddrinfo.target_type == PCANTP_ISOTP_ADDRESSING_FUNCTIONAL.value:

		# Transmit a NRC response pending response
		status = objPCANUds.MsgAlloc_2013(pending_response_msg, config, 3)
		print("Allocate TransferData pending response: %s" %(STATUS_OK_KO(status)))


		# Transmit a NRC response pending response
		if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
			pending_response_msg.links.service_id[0] = c_ubyte(objPCANUds.PUDS_NR_SI)
			pending_response_msg.links.param[0] = objPCANUds.PUDS_SI_TransferData
			pending_response_msg.links.param[1] = objPCANUds.PUDS_NRC_EXTENDED_TIMING
		status = objPCANUds.Write_2013(tp_handle, pending_response_msg)

		print("\n   ...Transmitting a NRC Response Pending message: %i" %(status.value))
		print("\n   ...simulating computation... (waiting ~%dms)" %(PUDS_P2CAN_ENHANCED_SERVER_MAX_DEFAULT))
		sleep((PUDS_P2CAN_ENHANCED_SERVER_MAX_DEFAULT - 100)/1000.0)

		# Reallocate response message
		status = objPCANUds.MsgFree_2013(pending_response_msg)
		print("Free pending response: %s" %(STATUS_OK_KO(status)))

		# Initialize real service response
		config.type = PUDS_MSGTYPE_USDT
		length = PCANTP_MAX_LENGTH_ISOTP2004
		status = objPCANUds.MsgAlloc_2013(response_msg, config, length)
		print("Allocate TransferData real response: %s" %(STATUS_OK_KO(status)))

	else:
		length = PCANTP_MAX_LENGTH_ISOTP2004 if (2 + request_msg.msg.msgdata.any.contents.length) > PCANTP_MAX_LENGTH_ISOTP2004 else 2 + request_msg.msg.msgdata.any.contents.length
		status = objPCANUds.MsgAlloc_2013(response_msg, config, length)
		print("Allocate TransferData response: %s" %(STATUS_OK_KO(status)))

	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		response_msg.links.param[0] = request_msg.links.param[0]

		# fill with dummy data
		for i in range(1, length - 1, 1):	# (length - 1) as positive response SID uses 1 byte
			response_msg.links.param[i] = i + 1
		response_msg.links.service_id[0] = c_ubyte(objPCANUds.PUDS_SI_TransferData + objPCANUds.PUDS_SI_POSITIVE_RESPONSE)


	return status

def process_request(tp_handle, server_address, request_msg):
	"""
	Process request and send a response

	parameters:
		tp_handle: CANTP channel
		server_address: Server address
		request_msg: Received request message
	"""

	# Initialize config from request message network address information
	config= uds_msgconfig()
	config.nai = ConvertNai(request_msg.msg.msgdata.isotp.contents.netaddrinfo, request_msg.msg.can_info.can_msgtype)
	config.can_msgtype = request_msg.msg.can_info.can_msgtype
	config.type = PUDS_MSGTYPE_USDT
	config.nai.extension_addr = request_msg.msg.msgdata.isotp.contents.netaddrinfo.extension_addr

	response_msg=uds_msg()

	# Set target and source addresses
	if request_msg.msg.msgdata.isotp.contents.netaddrinfo.target_type == PCANTP_ISOTP_ADDRESSING_FUNCTIONAL.value:
		# response to functional addressing is set to TEST_EQUIPMENT
		config.nai.target_addr = PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT
	else:
		config.nai.target_addr = request_msg.msg.msgdata.isotp.contents.netaddrinfo.source_addr

	config.nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL
	config.nai.source_addr = server_address

	# This is a valid request, switch services
	service_id = request_msg.links.service_id[0]
	if service_id == objPCANUds.PUDS_SI_DiagnosticSessionControl:
		status = create_SvcDiagnosticSessionControl_response(config, request_msg, response_msg)
	elif service_id == objPCANUds.PUDS_SI_SecurityAccess:
		status = create_SvcSecurityAccess_response(config, request_msg, response_msg)
	elif service_id == objPCANUds.PUDS_SI_CommunicationControl:
		status = create_SvcCommunicationControl_response(config, request_msg, response_msg)
	elif service_id == objPCANUds.PUDS_SI_TesterPresent:
		status = create_SvcTesterPresent_response(config, request_msg, response_msg)
	elif service_id == objPCANUds.PUDS_SI_SecuredDataTransmission:
		status = create_SvcSecuredDataTransmission_response(config, response_msg)
	elif service_id == objPCANUds.PUDS_SI_ControlDTCSetting:
		status = create_SvcControlDTCSetting_response(config, request_msg, response_msg)
	elif service_id == objPCANUds.PUDS_SI_ResponseOnEvent:
		status = create_SvcResponseOnEvent_response(config, request_msg, response_msg)
	elif service_id == objPCANUds.PUDS_SI_LinkControl:
		status = create_SvcLinkControl_response(config, request_msg, response_msg)
	elif service_id == objPCANUds.PUDS_SI_ReadDataByIdentifier:
		status = create_SvcReadDataByIdentifier_response(config, request_msg, response_msg)
	elif service_id == objPCANUds.PUDS_SI_ReadMemoryByAddress:
		status = create_SvcReadMemoryByAddress_response(config, request_msg, response_msg)
	elif service_id == objPCANUds.PUDS_SI_ReadScalingDataByIdentifier:
		status = create_SvcReadScalingDataByIdentifier_response(config, request_msg, response_msg)
	elif service_id == objPCANUds.PUDS_SI_ReadDataByPeriodicIdentifier:
		status = create_SvcReadDataByPeriodicIdentifier_response(config, response_msg)
	elif service_id == objPCANUds.PUDS_SI_DynamicallyDefineDataIdentifier:
		status = create_SvcDynamicallyDefineDataIdentifier_response(config, request_msg, response_msg)
	elif service_id == objPCANUds.PUDS_SI_WriteDataByIdentifier:
		status = create_SvcWriteDataByIdentifier_response(config, request_msg, response_msg)
	elif service_id == objPCANUds.PUDS_SI_WriteMemoryByAddress:
		status = create_SvcWriteMemoryByAddress_response(config, request_msg, response_msg)
	elif service_id == objPCANUds.PUDS_SI_ClearDiagnosticInformation:
		status = create_SvcClearDiagnosticInformation_response(config, response_msg)
	elif service_id == objPCANUds.PUDS_SI_InputOutputControlByIdentifier:
		status = create_SvcInputOutputControlByIdentifier_response(config, request_msg, response_msg)
	elif service_id == objPCANUds.PUDS_SI_RoutineControl:
		status = create_SvcRoutineControl_response(config, request_msg, response_msg)
	elif service_id == objPCANUds.PUDS_SI_RequestDownload:
		status = create_SvcRequestDownload_response(config, response_msg)
	elif service_id == objPCANUds.PUDS_SI_RequestUpload:
		status = create_SvcRequestUpload_response(config, response_msg)
	elif service_id == objPCANUds.PUDS_SI_TransferData:
		status = create_SvcTransferData_response(tp_handle, config, request_msg, response_msg)
	elif service_id == objPCANUds.PUDS_SI_RequestTransferExit:
		status = create_SvcRequestTransferExit_response(config, response_msg)
	elif service_id == objPCANUds.PUDS_SI_ECUReset:
		status = create_SvcECUReset_response(config, request_msg, response_msg)
	else:
		print("Unknown service (0x%02X)" %(request_msg.links.service_id[0]))
		status = create_dummy_response(config, request_msg, response_msg)

	# Print allocation result
	print("Allocate response message: %s" %(STATUS_OK_KO(status)))

	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		# Send response message
		status = objPCANUds.Write_2013(tp_handle, response_msg)
		print("\n   ...Transmitting response: %i\n" %(status.value))

	# Free response message
	status = objPCANUds.MsgFree_2013(response_msg)
	print("Free response message: %s" %(STATUS_OK_KO(status)))

def display_message(msg):
	"""
	Print an UDS message
	parameters:
		msg: Message to print
	"""
	# A function that displays UDS messages

	if (msg.msg.msgdata.isotp.contents.netaddrinfo.msgtype & PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION.value) == 0:

		print("\n[%lu] %s message from 0x%02X (to 0x%02X, with extension 0x%02X) - result: %i - %s" %(
			getTimeIndicationInMilliseconds(),
			"Received UDS" if (msg.msg.msgdata.any.contents.flags & PCANTP_MSGFLAG_LOOPBACK.value) == 0 else "Received Loopback",
			msg.msg.msgdata.isotp.contents.netaddrinfo.source_addr,
			msg.msg.msgdata.isotp.contents.netaddrinfo.target_addr,
			msg.msg.msgdata.isotp.contents.netaddrinfo.extension_addr,
			msg.msg.msgdata.any.contents.netstatus,
			"PCANTP_NETSTATUS_OK !!!" if msg.msg.msgdata.any.contents.netstatus != PCANTP_NETSTATUS_OK.value else "OK !"))
		# Display data
		s = "\t-> Length: {x1}, Data= ".format(x1=format(msg.msg.msgdata.any.contents.length, "d"))
		for i in range(msg.msg.msgdata.any.contents.length):
			s += "{x1} ".format(x1=format(msg.msg.msgdata.any.contents.data[i], "02X"))
		print(s)
		print("")

	else:

		print("\n[%lu] PENDING UDS message from 0x%02X (to 0x%02X, with extension 0x%02X) -> length=%i ..." %(
			getTimeIndicationInMilliseconds(),
			msg.msg.msgdata.isotp.contents.netaddrinfo.source_addr,
			msg.msg.msgdata.isotp.contents.netaddrinfo.target_addr,
			msg.msg.msgdata.isotp.contents.netaddrinfo.extension_addr,
			msg.msg.msgdata.any.contents.length))


# Definitions
BUFFER_SIZE = 256

#
# Main entry point of the program, start a CAN UDS server simulation
#

try:
	old_settings = install_keyboardHit()

	# Initialize variables
	tp_handle = PCANTP_HANDLE_USBBUS2 # TODO: modify the value according to your available PCAN devices.
	server_address = c_uint16(PUDS_ISO_15765_4_ADDR_ECU_1)

	# Print version informations
	buffer = create_string_buffer(BUFFER_SIZE)
	status = objPCANUds.GetValue_2013(PCANTP_HANDLE_NONEBUS, PUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE)
	print("PCAN-UDS API Version - %s: %s" %( buffer.value, STATUS_OK_KO(status)))

	# Initialize channel
	status = objPCANUds.Initialize_2013(tp_handle, PCANTP_BAUDRATE_500K, 0, 0, 0)
	print("Initialize channel: %s" %(STATUS_OK_KO(status)))

	# Set server address parameter
	status = objPCANUds.SetValue_2013(tp_handle, PUDS_PARAMETER_SERVER_ADDRESS, server_address, sizeof(server_address))
	print("Set server address: %s" %(STATUS_OK_KO(status)))

	# Set timeout values
	tmp_buffer = c_ubyte(PCANTP_ISO_TIMEOUTS_15765_4)
	status = objPCANUds.SetValue_2013(tp_handle, PUDS_PARAMETER_ISO_TIMEOUTS, tmp_buffer, sizeof(tmp_buffer))
	print("Set ISO 15765-4 timeouts values: %s" %(STATUS_OK_KO(status)))

	# Print timeout values
	timeout_value = c_uint32(0)
	status = objPCANUds.GetValue_2013(tp_handle, PUDS_PARAMETER_TIMEOUT_REQUEST, timeout_value, sizeof(timeout_value))
	print("Get request timeout value (%ums): %s" %(timeout_value.value, STATUS_OK_KO(status)))
	status = objPCANUds.GetValue_2013(tp_handle, PUDS_PARAMETER_TIMEOUT_RESPONSE, timeout_value, sizeof(timeout_value))
	print("Get response timeout value (%ums): %s" %(timeout_value.value, STATUS_OK_KO(status)))

	# Set a receive event
	if IS_WINDOWS:
		receive_event = c_void_p(windll.kernel32.CreateEventA(None, 0,0,None))
		res = objPCANUds.SetValue_2013(tp_handle, PUDS_PARAMETER_RECEIVE_EVENT, receive_event, sizeof(receive_event))
		print("Set receive event parameter: %s" %( STATUS_OK_KO(res)))
	else:
		receive_event = NULL_HANDLE()
		res = objPCANUds.GetValue_2013(tp_handle, PUDS_PARAMETER_RECEIVE_EVENT, receive_event, sizeof(receive_event))
		print("Get receive event parameter: %s" %( STATUS_OK_KO(res)))

	# Read while user do not press Q
	print("Start listening, press Q to quit.")
	stop = False
	while not stop:

		# Wait a receive event on receiver
		#	note: timeout is used to check keyboard hit.
		if IS_WINDOWS:
			windows_wait_result = c_uint64(windll.kernel32.WaitForSingleObject(receive_event, 2000))
			wait_result = True if windows_wait_result.value == WAIT_OBJECT_0 else False
		else:
			readable, _, _ = select.select([receive_event.value], [], [], 2)
			wait_result = True if len(readable) > 0 else False

		# If we get a receive event
		if wait_result:
			count_no_event=0
			# Read messages
			read_status = PUDS_STATUS_OK
			request_msg = uds_msg()
			while not objPCANUds.StatusIsOk_2013(read_status, PUDS_STATUS_NO_MESSAGE, False):
				read_status = objPCANUds.Read_2013(tp_handle, request_msg, None, None)
				print("[%lu] Try to read a message (status=%i): %s" %(getTimeIndicationInMilliseconds(), read_status.value, STATUS_OK_KO(read_status)))
				if objPCANUds.StatusIsOk_2013(read_status, PUDS_STATUS_OK, False):
					display_message(request_msg)

					# We receive a request, check if it is not a loopback message, if it is not a UUDT message, if the message is not pending, and do not respond if it is not ask
					if (request_msg.msg.msgdata.any.contents.flags & PCANTP_MSGFLAG_LOOPBACK.value) == 0\
						and (request_msg.msg.msgdata.isotp.contents.netaddrinfo.msgtype & PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION.value) == 0\
						and request_msg.msg.msgdata.any.contents.netstatus == PCANTP_NETSTATUS_OK.value\
						and (request_msg.type & PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE.value) != PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE.value:

						# Process response and send it
						process_request(tp_handle, server_address, request_msg)

					else:
						print("\n[%lu]   ...Skipping response...  (flags=0x%X, type=0x%X, netstatus=0x%X, msgtype=0x%X)\n" %(getTimeIndicationInMilliseconds(),request_msg.msg.msgdata.any.contents.flags, request_msg.type, request_msg.msg.msgdata.any.contents.netstatus, request_msg.msg.msgdata.isotp.contents.netaddrinfo.msgtype))


				# Free request message (and clean memory in order to reallocate later)
				status = objPCANUds.MsgFree_2013(request_msg)
				print("Free request message: %s" %(STATUS_OK_KO(status)))

		# Quit when user press Q
		if keyboardHit():
			keyboard_res = getInput()
			if keyboard_res == 'Q' or keyboard_res == 'q':
				stop = True


	# Close receive event
	if IS_WINDOWS:
		status = objPCANUds.SetValue_2013(tp_handle, PUDS_PARAMETER_RECEIVE_EVENT, NULL_HANDLE(), sizeof(NULL_HANDLE()))
		print("Stop receive event: %s" %(STATUS_OK_KO(status)))
		boolean_status = windll.kernel32.CloseHandle(receive_event)
		print("Close receive event: %s" %(OK_KO(boolean_status)))

	# Close server
	status = objPCANUds.Uninitialize_2013(tp_handle)
	print("Uninitialize channel: %s" %(STATUS_OK_KO(status)))

	# Exit
	print("Press any key to continue...")
	getInput()

finally:
	desinstall_keyboardHit(old_settings)


