import os
import sys
import copy
from time import sleep
from PCAN_UDS_2013 import *

IS_WINDOWS = platform.system() == 'Windows'

# Support events
if not IS_WINDOWS:
	import select
	__LIBC_OBJ = cdll.LoadLibrary("libc.so.6")

WAIT_OBJECT_0 = 0
INFINITE = 0xFFFFFFFF
def NULL_HANDLE():
	return c_void_p(0) if IS_WINDOWS else c_int(0)

# Support keyboard
def getInput():
	sys.stdin.read(1)

# Console handling
USE_GETCH = False

def CLEAR_CONSOLE():
	if USE_GETCH:
		if IS_WINDOWS:
			os.system("cls")
		else:
			os.system("clear")

# UDS library
objPCANUds = PCAN_UDS_2013()

# A global counter to keep track of the number of failed tests (see display_uds_msg function)
g_nbErr = 0

# Help functions
def waitGetch():
	"""
	A simple function that waits for user input
	"""
	if USE_GETCH:
		print("Press <Enter> to continue...")
		getInput()
		CLEAR_CONSOLE()

def display_uds_msg(request, response, no_response_expected):
	"""
	A function that displays UDS Request and Response messages (and count error if no response)

	parameters:
		request: Request message
		response: Received response message
		no_response_expected: if no response is expected, do not increment error counter
	"""
	global g_nbErr
	if request != None and request.msg.msgdata.isotp:
			print("\nUDS request from 0x%04X (to 0x%04X, with extension address 0x%02X) - result: %i - %s" %(
				request.msg.msgdata.isotp.contents.netaddrinfo.source_addr,
				request.msg.msgdata.isotp.contents.netaddrinfo.target_addr,
				request.msg.msgdata.isotp.contents.netaddrinfo.extension_addr,
				request.msg.msgdata.any.contents.netstatus,
				"ERROR !!!" if request.msg.msgdata.any.contents.netstatus != PCANTP_NETSTATUS_OK.value else "OK !"))
			# display data
			s = "\t-> Length: {x1}, Data= ".format(x1=format(request.msg.msgdata.any.contents.length, "d"))
			for i in range(request.msg.msgdata.any.contents.length):
				s += "{x1} ".format(x1=format(request.msg.msgdata.any.contents.data[i], "02X"))

			print(s)

	if response != None and response.msg.msgdata.isotp:
		print("\nUDS RESPONSE from 0x%04X (to 0x%04X, with extension address 0x%02X) - result: %i - %s" %(
			response.msg.msgdata.isotp.contents.netaddrinfo.source_addr, response.msg.msgdata.isotp.contents.netaddrinfo.target_addr,
			response.msg.msgdata.isotp.contents.netaddrinfo.extension_addr, response.msg.msgdata.any.contents.netstatus,
			("ERROR !!!" if response.msg.msgdata.any.contents.netstatus != PCANTP_NETSTATUS_OK.value else "OK !")))
		# display data
		s = "\t-> Length: {x1}, Data= ".format(x1=format(response.msg.msgdata.any.contents.length, "d"))
		for i in range(response.msg.msgdata.any.contents.length):
			s += "{x1} ".format(x1=format(response.msg.msgdata.any.contents.data[i], "02X"))

		print(s)

	elif not no_response_expected:
		print("\n /!\\ ERROR: NO UDS RESPONSE !!\n")
		g_nbErr += 1

def getCChar(c):
	"""
	Get a character adapted to a C string buffer depending on Python version
	"""
	return c if sys.version_info.major >= 3 else chr(c)

def Reverse32(v):
	"""
	Helper: Inverts the bytes of a 32 bits numeric value

	parameters:
		v: Value to revert
	returns: Reverted value
	"""
	res = create_string_buffer(4)
	res[3] = getCChar(v & 0x000000FF)
	res[2] = getCChar((v >> 8) & 0x000000FF)
	res[1] = getCChar((v >> 16) & 0x000000FF)
	res[0] = getCChar((v >> 24) & 0x000000FF)
	return res

def testDiagnosticSessionControl(channel, config):
	"""
	Call UDS Service DiagnosticSessionControl

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""

	request = uds_msg()
	response = uds_msg()
	session_info = uds_sessioninfo()
	confirmation = uds_msg()

	CLEAR_CONSOLE()
	print("\n\n*** UDS Service: DiagnosticSessionControl ***")

	# Read default session information Server is not yet known (status will be PUDS_ERROR_NOT_INITIALIZED)
	# yet the API will still set session info to default values
	session_info.nai = copy.deepcopy(config.nai)
	status = objPCANUds.GetValue_2013(channel, PUDS_PARAMETER_SESSION_INFO, session_info, sizeof(uds_sessioninfo))
	print(" Diagnostic Session Information: %i, 0x%04X => %d = [%04X; %04X]" %(status.value, session_info.nai.target_addr,
		session_info.session_type, session_info.timeout_p2can_server_max, session_info.timeout_enhanced_p2can_server_max))
	waitGetch()

	# Set Diagnostic session to DEFAULT (to get session information)
	print("\n\nSetting a DEFAULT Diagnostic Session:")
	status = objPCANUds.SvcDiagnosticSessionControl_2013(channel, config, request, objPCANUds.PUDS_SVC_PARAM_DSC_DS)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcDiagnosticSessionControl_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))

	# Read current session information
	session_info= uds_sessioninfo()
	session_info.nai = copy.deepcopy(config.nai)
	status = objPCANUds.GetValue_2013(channel, PUDS_PARAMETER_SESSION_INFO, session_info, sizeof(uds_sessioninfo))
	print(" Diagnostic Session Information: %i, 0x%04X => %d = [%04X; %04X]" %(status.value, session_info.nai.target_addr,
		session_info.session_type, session_info.timeout_p2can_server_max, session_info.timeout_enhanced_p2can_server_max))
	waitGetch()

	# Set Diagnostic session to PROGRAMMING
	print("\n\nSetting a ECUPS Diagnostic Session:")
	status = objPCANUds.SvcDiagnosticSessionControl_2013(channel, config, request, objPCANUds.PUDS_SVC_PARAM_DSC_ECUPS)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcDiagnosticSessionControl_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))

	# Read current session information
	session_info = uds_sessioninfo()
	session_info.nai = copy.deepcopy(config.nai)
	status = objPCANUds.GetValue_2013(channel, PUDS_PARAMETER_SESSION_INFO, session_info, sizeof(uds_sessioninfo))
	print(" Diagnostic Session Information: %i, 0x%04X => %d = [%04X; %04X]" %(status.value, session_info.nai.target_addr,
		session_info.session_type, session_info.timeout_p2can_server_max, session_info.timeout_enhanced_p2can_server_max))
	print(" Assert that Auto TesterPresent Frame is sent...")
	sleep(2)
	print(" Should transmit an Auto TesterPresent Frame")
	sleep(2)
	print(" Should transmit an Auto TesterPresent Frame")
	waitGetch()

	# Set Diagnostic session back to DEFAULT
	print("\n\nSetting a DEFAULT Diagnostic Session:")
	status = objPCANUds.SvcDiagnosticSessionControl_2013(channel, config, request, objPCANUds.PUDS_SVC_PARAM_DSC_DS)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcDiagnosticSessionControl_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	print(" Assert that NO Auto TesterPresent Frame is sent...")
	sleep(2)
	print(" Should NOT transmit an Auto TesterPresent Frame")
	sleep(2)
	print(" Should NOT transmit an Auto TesterPresent Frame")
	waitGetch()

def testECUReset(channel, config):
	"""
	Call UDS Service ECUReset

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""

	request = uds_msg()
	response = uds_msg()
	confirmation = uds_msg()

	CLEAR_CONSOLE()
	print("\n\n*** UDS Service: ECUReset ***")

	# Sends a physical ECUReset message
	print("\n\nSends a physical ECUReset message: ")
	status = objPCANUds.SvcECUReset_2013(channel, config, request, objPCANUds.PUDS_SVC_PARAM_ER_SR)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcECUReset_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

def testRequestFileTransfer(channel, config):
	"""
	Call UDS Service RequestFileTransfer

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""

	file_size_uncompressed = create_string_buffer(b'\x0D\x00')
	file_size_compressed = create_string_buffer(b'\x0A\x00')
	file_size_length = 2
	sfile_name = create_string_buffer(b'toto.txt')
	file_name_length = 8

	request = uds_msg()
	response = uds_msg()
	confirmation = uds_msg()

	CLEAR_CONSOLE()
	print("*** UDS Service: RequestFileTransfer ***")

	# Sends a physical RequestFileTransfer message
	print("\n\nSends a physical RequestFileTransfer message: ")
	status = objPCANUds.SvcRequestFileTransfer_2013(channel, config, request, objPCANUds.PUDS_SVC_PARAM_RFT_MOOP_REPLFILE, file_name_length, sfile_name, 0, 0, file_size_length, file_size_uncompressed, file_size_compressed)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcRequestFileTransfer_2013: %d" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %d" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %d" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %d" %(status.value))
	waitGetch()

def testAccessTimingParameter(channel, config):
	"""
	Call UDS Service AccessTimingParameter

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""

	request_record = create_string_buffer(b'\xAB\xCD')
	record_size = c_uint32(2)

	request = uds_msg()
	response = uds_msg()
	confirmation = uds_msg()
	request_record[0] = getCChar(0xAB)
	request_record[1] = getCChar(0xCD)

	CLEAR_CONSOLE()
	print("\n\n\n*** UDS Service: AccessTimingParameter ***")

	# Sends a physical AccessTimingParameter message
	print("\n\nSends a physical AccessTimingParameter message: ")
	status = objPCANUds.SvcAccessTimingParameter_2013(channel, config, request, objPCANUds.PUDS_SVC_PARAM_ATP_RCATP, request_record, record_size)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcAccessTimingParameter_2013: %d" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %d" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %d" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

def testSecurityAccess(channel, config):
	"""
	Call UDS Service SecurityAccess

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""

	request = uds_msg()
	response = uds_msg()
	confirmation = uds_msg()

	CLEAR_CONSOLE()
	print("\n\n*** UDS Service: SecurityAccess ***")

	# Sends a physical SecurityAccess message
	print("\n\nSends a physical SecurityAccess message: ")
	value_little_endian = 0xF0A1B2C3
	security_access_data = Reverse32(value_little_endian) # use helper function to set MSB as 1st byte in the buffer (Win32 uses little endian format)
	status = objPCANUds.SvcSecurityAccess_2013(channel, config, request, objPCANUds.PUDS_SVC_PARAM_SA_RSD_1, security_access_data, 4)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcSecurityAccess_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

def testCommunicationControl(channel, config):
	"""
	Call UDS Service CommunicationControl

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""

	request = uds_msg()
	response = uds_msg()
	confirmation = uds_msg()

	CLEAR_CONSOLE()
	print("\n\n*** UDS Service: CommunicationControl ***")

	# Sends a physical CommunicationControl message
	print("\n\nSends a physical CommunicationControl message: ")
	status = objPCANUds.SvcCommunicationControl_2013(channel, config, request,
		objPCANUds.PUDS_SVC_PARAM_CC_ERXTX, objPCANUds.PUDS_SVC_PARAM_CC_FLAG_APPL | objPCANUds.PUDS_SVC_PARAM_CC_FLAG_NWM | objPCANUds.PUDS_SVC_PARAM_CC_FLAG_DENWRIRO, 0);

	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcCommunicationControl_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

def testTesterPresent(channel, config):
	"""
	UDS Call Service TesterPresent

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""

	response_count = c_uint32(0)
	request = uds_msg()
	response = uds_msg()
	confirmation = uds_msg()
	config_copy = uds_msgconfig()
	config_copy = copy.deepcopy(config)

	CLEAR_CONSOLE()
	print("\n\n*** UDS Service: TesterPresent ***")

	# Sends a physical TesterPresent message
	print("\n\nSends a physical TesterPresent message: ")
	status = objPCANUds.SvcTesterPresent_2013(channel, config_copy, request, objPCANUds.PUDS_SVC_PARAM_TP_ZSUBF)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcTesterPresent_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a physical TesterPresent message with no positive response
	config_copy.type = PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE
	print("\n\nSends a physical TesterPresent message with no positive response:")
	status = objPCANUds.SvcTesterPresent_2013(channel, config_copy, request, objPCANUds.PUDS_SVC_PARAM_TP_ZSUBF)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcTesterPresent_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, True)
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a functional TesterPresent message
	config_copy.type = PUDS_MSGTYPE_USDT
	config_copy.nai.target_type = PCANTP_ISOTP_ADDRESSING_FUNCTIONAL
	config_copy.nai.target_addr = PUDS_ISO_15765_4_ADDR_OBD_FUNCTIONAL
	print("\n\nSends a functional TesterPresent message: ")
	status = objPCANUds.SvcTesterPresent_2013(channel, config_copy, request, objPCANUds.PUDS_SVC_PARAM_TP_ZSUBF)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForServiceFunctional_2013(channel, request, 1, True, response, response_count,confirmation)
	print(" UDS_SvcTesterPresent_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a functional TesterPresent message with no positive response
	config_copy.type = PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE
	config_copy.nai.target_type = PCANTP_ISOTP_ADDRESSING_FUNCTIONAL
	config_copy.nai.target_addr = PUDS_ISO_15765_4_ADDR_OBD_FUNCTIONAL
	print("\n\nSends a functional TesterPresent message with no positive response:")
	status = objPCANUds.SvcTesterPresent_2013(channel, config_copy, request, objPCANUds.PUDS_SVC_PARAM_TP_ZSUBF)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForServiceFunctional_2013(channel, request, 1, True, response, response_count,confirmation)
	print(" UDS_SvcTesterPresent_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False) and response_count.value != 0:
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, True)
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

def testSecuredDataTransmission(channel, config):
	"""
	Call UDS Service SecuredDataTransmission

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""

	# Initialize structures
	request = uds_msg()
	ecureset_request = uds_msg()
	response = uds_msg()
	confirmation = uds_msg()

	CLEAR_CONSOLE()
	print("\n\n*** UDS Service: SecuredDataTransmission ***")

	# Sends a physical SecuredDataTransmission/2013 message
	print("\n\nSends a physical SecuredDataTransmission/2013 message: ")
	value_little_endian = 0xF0A1B2C3
	security_data_request_record = Reverse32(value_little_endian) # use helper function to set MSB as 1st byte in the buffer (Win32 uses little endian format)
	status = objPCANUds.SvcSecuredDataTransmission_2013(channel, config, request, security_data_request_record, 4)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcSecuredDataTransmission_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a physical SecuredDataTransmission/2013 message prepared with PUDS_ONLY_PREPARE_REQUEST option
	print("\n\nSends a physical SecuredDataTransmission/2013 prepared with PUDS_ONLY_PREPARE_REQUEST option: ")
	status = objPCANUds.SvcECUReset_2013(PUDS_ONLY_PREPARE_REQUEST, config, ecureset_request, objPCANUds.PUDS_SVC_PARAM_ER_HR)
	print(" Prepare ECUReset request for SecuredDataTransmission: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.SvcSecuredDataTransmission_2013(channel, config, request, ecureset_request.msg.msgdata.any.contents.data.contents, ecureset_request.msg.msgdata.any.contents.length)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcSecuredDataTransmission_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(ecureset_request)
	print(" Free prepared message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response);
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation);
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a physical SecuredDataTransmission/2020 message
	print("\n\nSends a physical SecuredDataTransmission/2020 message: ")
	administrative_parameter = objPCANUds.PUDS_SVC_PARAM_APAR_REQUEST_MSG_FLAG | objPCANUds.PUDS_SVC_PARAM_APAR_REQUEST_RESPONSE_SIGNATURE_FLAG | objPCANUds.PUDS_SVC_PARAM_APAR_SIGNED_MSG_FLAG
	signature_encryption_calculation = 0x0
	anti_replay_counter = 0x0124
	internal_service_identifier = 0x2E
	service_specific_parameters = create_string_buffer(b'\xF1\x23\xAA\x55' )
	service_specific_parameters_size = 4
	signature_mac = create_string_buffer( b'\xDB\xD1\x0E\xDC\x55\xAA' )
	signature_size = 0x0006
	status = objPCANUds.SvcSecuredDataTransmission_2020(channel, config, request, administrative_parameter, signature_encryption_calculation, anti_replay_counter, internal_service_identifier,service_specific_parameters,service_specific_parameters_size,signature_mac,signature_size)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcSecuredDataTransmission_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

def testControlDTCSetting(channel, config):
	"""
	Call UDS Service ControlDTCSetting

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""

	request = uds_msg()
	response = uds_msg()
	confirmation = uds_msg()

	CLEAR_CONSOLE()
	print("\n\n*** UDS Service: ControlDTCSetting ***")

	# Sends a physical ControlDTCSetting message
	print("\n\nSends a physical ControlDTCSetting message: ")
	value_little_endian = 0xF1A1B2EE
	dtc_setting_control_option_record = Reverse32(value_little_endian) # use helper function to set MSB as 1st byte in the buffer (Win32 uses little endian format)
	status = objPCANUds.SvcControlDTCSetting_2013(channel, config, request, objPCANUds.PUDS_SVC_PARAM_CDTCS_OFF, dtc_setting_control_option_record, 3)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcControlDTCSetting_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

def testResponseOnEvent(channel, config):
	"""
	Call UDS Service ResponseOnEvent

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""

	# Initialize structures
	request = uds_msg()
	response = uds_msg()
	confirmation = uds_msg()

	CLEAR_CONSOLE()
	print("\n\n*** UDS Service: ResponseOnEvent ***")

	# Sends a physical ResponseOnEvent message
	print("\n\nSends a physical ResponseOnEvent message: ")
	event_type_record = create_string_buffer(50)
	event_type_record[0] = getCChar(0x08)
	service_to_respond_to_record = create_string_buffer(50)
	service_to_respond_to_record[0] = getCChar(objPCANUds.PUDS_SI_ReadDTCInformation)
	service_to_respond_to_record[1] = getCChar(objPCANUds.PUDS_SVC_PARAM_RDTCI_RNODTCBSM)
	service_to_respond_to_record[2] = getCChar(0x01)
	status = objPCANUds.SvcResponseOnEvent_2013(channel, config, request, objPCANUds.PUDS_SVC_PARAM_ROE_ONDTCS, False, 0x08, event_type_record,
		objPCANUds.PUDS_SVC_PARAM_ROE_OTI_LEN, service_to_respond_to_record, 3)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcResponseOnEvent_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

def testLinkControl(channel, config):
	"""
	Call UDS Service LinkControl

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""

	request = uds_msg()
	response = uds_msg()
	confirmation = uds_msg()

	CLEAR_CONSOLE()
	print("\n\n*** UDS Service: LinkControl ***")

	# Sends a physical LinkControl message
	print("\n\nSends a physical LinkControl message (Verify Fixed Baudrate): ")
	status = objPCANUds.SvcLinkControl_2013(channel, config, request, objPCANUds.PUDS_SVC_PARAM_LC_VBTWFBR, objPCANUds.PUDS_SVC_PARAM_LC_BAUDRATE_CAN_500K, 0)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcLinkControl_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))

	# Sends a physical LinkControl message
	print("\n\nSends a physical LinkControl message (Verify Specific Baudrate): ")
	status = objPCANUds.SvcLinkControl_2013(channel, config, request, objPCANUds.PUDS_SVC_PARAM_LC_VBTWSBR, 0, 500000) # 500K = 0x0007a120
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcLinkControl_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))

	# Sends a physical LinkControl message
	print("\n\nSends a physical LinkControl message (Transition): ")
	status = objPCANUds.SvcLinkControl_2013(channel, config, request, objPCANUds.PUDS_SVC_PARAM_LC_TB, 0, 0)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcLinkControl_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

def testReadDataByIdentifier(channel, config):
	"""
	Call UDS Service ReadDataByIdentifier

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""

	data_identifier = (c_uint16 * 2) (objPCANUds.PUDS_SVC_PARAM_DI_ADSDID, objPCANUds.PUDS_SVC_PARAM_DI_ECUMDDID )
	request = uds_msg()
	response = uds_msg()
	confirmation = uds_msg()

	CLEAR_CONSOLE()
	print("\n\n*** UDS Service: ReadDataByIdentifier ***")

	# Sends a physical ReadDataByIdentifier message
	print("\n\nSends a physical ReadDataByIdentifier message: ")
	status = objPCANUds.SvcReadDataByIdentifier_2013(channel, config, request, data_identifier, 2)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcReadDataByIdentifier_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

def testReadMemoryByAddress(channel, config):
	"""
	Call UDS Service ReadMemoryByAddress

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""

	memory_address_buffer = create_string_buffer(10)
	memory_size_buffer = create_string_buffer(10)
	memory_address_size = 10
	memory_size_size  = 3

	# Initialize structures
	request = uds_msg()
	response = uds_msg()
	confirmation = uds_msg()

	CLEAR_CONSOLE()
	print("\n\n*** UDS Service: ReadMemoryByAddress ***")

	# Sends a physical ReadMemoryByAddress message
	print("\n\nSends a physical ReadMemoryByAddress message: ")
	for i in range(memory_address_size):
		memory_address_buffer[i] = getCChar(ord('A') + i)
		memory_size_buffer[i] = getCChar(ord('1') + i)

	status = objPCANUds.SvcReadMemoryByAddress_2013(channel, config, request, memory_address_buffer, memory_address_size, memory_size_buffer, memory_size_size)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcReadMemoryByAddress_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

def testReadScalingDataByIdentifier(channel, config):
	"""
	Call UDS Service ReadScalingDataByIdentifier

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""

	request = uds_msg()
	response = uds_msg()
	confirmation = uds_msg()

	CLEAR_CONSOLE()
	print("\n\n*** UDS Service: ReadScalingDataByIdentifier ***")

	# Sends a physical ReadScalingDataByIdentifier message
	print("\n\nSends a physical ReadScalingDataByIdentifier message: ")
	status = objPCANUds.SvcReadScalingDataByIdentifier_2013(channel, config, request, objPCANUds.PUDS_SVC_PARAM_DI_BSFPDID)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcReadScalingDataByIdentifier_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

def testReadDataByPeriodicIdentifier(channel, config):
	"""
	Call UDS Service ReadDataByPeriodicIdentifier

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""
	periodic_data_identifier = create_string_buffer(10)
	periodic_data_identifier_size = 10

	# Initialize structures
	request = uds_msg()
	response = uds_msg()
	confirmation = uds_msg()

	CLEAR_CONSOLE()
	print("\n\n*** UDS Service: ReadDataByPeriodicIdentifier ***")

	# Sends a physical ReadScalingDataByIdentifier message
	print("\n\nSends a physical ReadDataByPeriodicIdentifier message: ")
	for i in range(periodic_data_identifier_size):
		periodic_data_identifier[i] = getCChar(ord('A') + i)
	status = objPCANUds.SvcReadDataByPeriodicIdentifier_2013(channel, config, request, objPCANUds.PUDS_SVC_PARAM_RDBPI_SAMR, periodic_data_identifier, periodic_data_identifier_size)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcReadDataByPeriodicIdentifier_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

def testDynamicallyDefineDataIdentifier(channel, config):
	"""
	Call UDS Service DynamicallyDefineDataIdentifier

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""

	source_data_identifier = (c_uint16 *20)()
	memory_size = create_string_buffer(20)
	position_in_source_data_record = create_string_buffer(20)
	number_of_elements = 10
	memory_address_buffer = create_string_buffer(15)
	memory_size_buffer = create_string_buffer(9)

	request = uds_msg()
	response = uds_msg()
	confirmation = uds_msg()

	CLEAR_CONSOLE()
	print("\n\n*** UDS Service: DynamicallyDefineDataIdentifier ***")

	# Sends a physical DynamicallyDefineDataIdentifierDBID message
	print("\n\nSends a physical DynamicallyDefineDataIdentifierDBID message: ")
	for i in range(number_of_elements):
		source_data_identifier[i] = ((0xF0 + i) << 8) + (ord('A') + i)
		memory_size[i] = getCChar(i + 1)
		position_in_source_data_record[i] = getCChar(100 + i)

	status = objPCANUds.SvcDynamicallyDefineDataIdentifierDBID_2013(channel, config, request, objPCANUds.PUDS_SVC_PARAM_DI_CDDID, source_data_identifier,
		memory_size, position_in_source_data_record, number_of_elements)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcDynamicallyDefineDataIdentifierDBID_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a physical UDS_SvcDynamicallyDefineDataIdentifierDBMA message
	print("\n\nSends a physical UDS_SvcDynamicallyDefineDataIdentifierDBMA_2013 message: ")
	number_of_elements = 3
	memory_address_size = 5
	memory_size_size = 3
	for j in range(number_of_elements):
		for i in range(memory_address_size):
			memory_address_buffer[memory_address_size * j + i] = getCChar((10 * j) + i + 1)
		for i in range(memory_size_size):
			memory_size_buffer[memory_size_size * j + i] = getCChar(100 + (10 * j) + i + 1)

	status = objPCANUds.SvcDynamicallyDefineDataIdentifierDBMA_2013(channel, config, request, objPCANUds.PUDS_SVC_PARAM_DI_CESWNDID, memory_address_size,
		memory_size_size, memory_address_buffer, memory_size_buffer, number_of_elements)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcDynamicallyDefineDataIdentifierDBMA_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a physical UDS_SvcDynamicallyDefineDataIdentifierCDDDI message
	print("\n\nSends a physical UDS_SvcDynamicallyDefineDataIdentifierCDDDI_2013 message: ")
	status = objPCANUds.SvcDynamicallyDefineDataIdentifierCDDDI_2013(channel, config, request, objPCANUds.PUDS_SVC_PARAM_DI_CESWNDID)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcDynamicallyDefineDataIdentifierCDDDI_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a physical UDS_SvcDynamicallyDefineDataIdentifierClearAllDDDI_2013 message
	print("\n\nSends a physical UDS_SvcDynamicallyDefineDataIdentifierClearAllDDDI_2013 message: ")
	status = objPCANUds.SvcDynamicallyDefineDataIdentifierClearAllDDDI_2013(channel, config, request)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcDynamicallyDefineDataIdentifierClearAllDDDI_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

def testWriteDataByIdentifier(channel, config):
	"""
	Call UDS Service WriteDataByIdentifier

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""

	data_record = create_string_buffer(10)
	data_record_size = 10

	request = uds_msg()
	response = uds_msg()
	confirmation = uds_msg()

	CLEAR_CONSOLE()
	print("\n\n*** UDS Service: WriteDataByIdentifier ***")

	# Sends a physical WriteDataByIdentifier message
	print("\n\nSends a physical WriteDataByIdentifier message: ")
	for i in range(data_record_size):
		data_record[i] = getCChar(ord('A') + i)
	status = objPCANUds.SvcWriteDataByIdentifier_2013(channel, config, request, objPCANUds.PUDS_SVC_PARAM_DI_ASFPDID, data_record, data_record_size)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcWriteDataByIdentifier_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

def testWriteMemoryByAddress(channel, config):
	"""
	Call UDS Service WriteMemoryByAddress

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""
	data_record = create_string_buffer(50)
	memory_address_buffer = create_string_buffer(50)
	memory_size_buffer = create_string_buffer(50)
	data_record_size = 50
	memory_address_size = 5
	memory_size_size = 3

	request = uds_msg()
	response = uds_msg()
	confirmation = uds_msg()

	CLEAR_CONSOLE()
	print("\n\n*** UDS Service: WriteMemoryByAddress ***")

	# Sends a physical WriteMemoryByAddress message
	print("\n\nSends a physical WriteMemoryByAddress message: ")
	for i in range(data_record_size):
		data_record[i] = getCChar(i + 1)
		memory_address_buffer[i] = getCChar(ord('A') + i)
		memory_size_buffer[i] = getCChar(10 + i)

	status = objPCANUds.SvcWriteMemoryByAddress_2013(channel, config, request, memory_address_buffer, memory_address_size, memory_size_buffer, memory_size_size,
		data_record, data_record_size)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcWriteMemoryByAddress_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

def testClearDiagnosticInformation(channel, config):
	"""
	Call UDS Service ClearDiagnosticInformation

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""

	request = uds_msg()
	response = uds_msg()
	confirmation = uds_msg()

	CLEAR_CONSOLE()
	print("\n\n*** UDS Service: ClearDiagnosticInformation ***")

	# Sends a physical ClearDiagnosticInformation message with memory selection parameter (2020)
	print("\n\nSends a physical ClearDiagnosticInformation message: ")
	status = objPCANUds.SvcClearDiagnosticInformation_2013(channel, config, request, 0xF1A2B3)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcClearDiagnosticInformation_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a physical ClearDiagnosticInformation message
	print("\n\nSends a physical ClearDiagnosticInformation message with memory selection parameter: ")
	status = objPCANUds.SvcClearDiagnosticInformation_2020(channel, config, request, 0xF1A2B3, 0x42)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcClearDiagnosticInformation_2020: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

def testReadDTCInformation(channel, config):
	"""
	Call UDS Service ReadDTCInformation

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""

	request = uds_msg()
	response = uds_msg()
	confirmation = uds_msg()

	CLEAR_CONSOLE()
	print("\n\n*** UDS Service: ReadDTCInformation ***")

	# Sends a physical ReadDTCInformation message
	print("\n\nSends a physical ReadDTCInformation message: ")
	status = objPCANUds.SvcReadDTCInformation_2013(channel, config, request, objPCANUds.PUDS_SVC_PARAM_RDTCI_RNODTCBSM, 0xF1)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcReadDTCInformation_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a physical ReadDTCInformationRDTCSSBDTC message
	print("\n\nSends a physical ReadDTCInformationRDTCSSBDTC message: ")
	status = objPCANUds.SvcReadDTCInformationRDTCSSBDTC_2013(channel, config, request, 0x00A1B2B3, 0x12)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcReadDTCInformationRDTCSSBDTC_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a physical ReadDTCInformationRDTCSSBRN message
	print("\n\nSends a physical ReadDTCInformationRDTCSSBRN message: ")
	status = objPCANUds.SvcReadDTCInformationRDTCSSBRN_2013(channel, config, request, 0x12)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcReadDTCInformationRDTCSSBRN_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a physical ReadDTCInformationReportExtended message
	print("\n\nSends a physical ReadDTCInformationReportExtended message: ")
	status = objPCANUds.SvcReadDTCInformationReportExtended_2013(channel, config, request, objPCANUds.PUDS_SVC_PARAM_RDTCI_RDTCEDRBDN, 0x00A1B2B3,
		0x12)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcReadDTCInformationReportExtended_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a physical ReadDTCInformationReportSeverity message
	print("\n\nSends a physical ReadDTCInformationReportSeverity message: ")
	status = objPCANUds.SvcReadDTCInformationReportSeverity_2013(channel, config, request, objPCANUds.PUDS_SVC_PARAM_RDTCI_RNODTCBSMR, 0xF1, 0x12)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcReadDTCInformationReportSeverity_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a physical ReadDTCInformationRSIODTC message
	print("\n\nSends a physical ReadDTCInformationRSIODTC message: ")
	status = objPCANUds.SvcReadDTCInformationRSIODTC_2013(channel, config, request, 0xF1A1B2B3)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcReadDTCInformationRSIODTC_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a physical ReadDTCInformationNoParam message
	print("\n\nSends a physical ReadDTCInformationNoParam message: ")
	status = objPCANUds.SvcReadDTCInformationNoParam_2013(channel, config, request, objPCANUds.PUDS_SVC_PARAM_RDTCI_RSUPDTC)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcReadDTCInformationNoParam_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a physical UDS_SvcReadDTCInformationRDTCEDBR_2013 message
	print("\n\nSends a physical UDS_SvcReadDTCInformationRDTCEDBR_2013 message: ")
	status = objPCANUds.SvcReadDTCInformationRDTCEDBR_2013(channel, config, request, 0x12)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcReadDTCInformationRDTCEDBR_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a physical UDS_SvcReadDTCInformationRUDMDTCBSM_2013 message
	print("\n\nSends a physical UDS_SvcReadDTCInformationRUDMDTCBSM_2013 message: ")
	status = objPCANUds.SvcReadDTCInformationRUDMDTCBSM_2013(channel, config, request, 0x12, 0x34)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcReadDTCInformationRUDMDTCBSM_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a physical UDS_SvcReadDTCInformationRUDMDTCSSBDTC_2013 message
	print("\n\nSends a physical UDS_SvcReadDTCInformationRUDMDTCSSBDTC_2013 message: ")
	status = objPCANUds.SvcReadDTCInformationRUDMDTCSSBDTC_2013(channel, config, request, 0x00123456, 0x78, 0x9A)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcReadDTCInformationRUDMDTCSSBDTC_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a physical UDS_SvcReadDTCInformationRUDMDTCEDRBDN_2013 message
	print("\n\nSends a physical UDS_SvcReadDTCInformationRUDMDTCEDRBDN_2013 message: ")
	status = objPCANUds.SvcReadDTCInformationRUDMDTCEDRBDN_2013(channel, config, request, 0x00123456, 0x78, 0x9A)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcReadDTCInformationRUDMDTCEDRBDN_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a physical UDS_SvcReadDTCInformationRDTCEDI_2020 message
	print("\n\nSends a physical UDS_SvcReadDTCInformationRDTCEDI_2020 message: ")
	status = objPCANUds.SvcReadDTCInformationRDTCEDI_2020(channel, config, request, 0x12)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcReadDTCInformationRDTCEDI_2020: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a physical UDS_SvcReadDTCInformationRWWHOBDDTCBMR_2013 message
	print("\n\nSends a physical UDS_SvcReadDTCInformationRWWHOBDDTCBMR_2013 message: ")
	status = objPCANUds.SvcReadDTCInformationRWWHOBDDTCBMR_2013(channel, config, request, 0x12, 0x34, 0x56)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcReadDTCInformationRWWHOBDDTCBMR_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a physical UDS_SvcReadDTCInformationRWWHOBDDTCWPS_2013 message
	print("\n\nSends a physical UDS_SvcReadDTCInformationRWWHOBDDTCWPS_2013 message: ")
	status = objPCANUds.SvcReadDTCInformationRWWHOBDDTCWPS_2013(channel, config, request, 0x12)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcReadDTCInformationRWWHOBDDTCWPS_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a physical UDS_SvcReadDTCInformationRDTCBRGI_2020 message
	print("\n\nSends a physical UDS_SvcReadDTCInformationRDTCBRGI_2020 message: ")
	status = objPCANUds.SvcReadDTCInformationRDTCBRGI_2020(channel, config, request, 0x12, 0x34)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcReadDTCInformationRDTCBRGI_2020: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

def testInputOutputControlByIdentifier(channel, config):
	"""
	Call UDS Service InputOutputControlByIdentifier

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""
	control_option_record = create_string_buffer(10)
	control_enable_mask_record = create_string_buffer(10)
	control_option_record_size = 10
	control_enable_mask_record_size = 5

	request = uds_msg()
	response = uds_msg()
	confirmation = uds_msg()

	CLEAR_CONSOLE()
	print("\n\n*** UDS Service: InputOutputControlByIdentifier ***")

	# Sends a physical InputOutputControlByIdentifier message
	print("\n\nSends a physical InputOutputControlByIdentifier message: ")
	for i in range(control_option_record_size):
		control_option_record[i] = getCChar(ord('A') + i)
		control_enable_mask_record[i] = getCChar(10 + i)

	status = objPCANUds.SvcInputOutputControlByIdentifier_2013(channel, config, request, objPCANUds.PUDS_SVC_PARAM_DI_SSECUSWVNDID, control_option_record,
		control_option_record_size, control_enable_mask_record, control_enable_mask_record_size)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcInputOutputControlByIdentifier_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

def testRoutineControl(channel, config):
	"""
	Call UDS Service RoutineControl

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""

	routine_control_option_record = create_string_buffer(10)
	routine_control_option_record_size = 10

	request = uds_msg()
	response = uds_msg()
	confirmation = uds_msg()

	CLEAR_CONSOLE()
	print("\n\n*** UDS Service: RoutineControl ***")

	# Sends a physical RoutineControl message
	print("\n\nSends a physical RoutineControl message: ")
	for i in range(routine_control_option_record_size):
		routine_control_option_record[i] = getCChar(ord('A') + i)

	status = objPCANUds.SvcRoutineControl_2013(channel, config, request, objPCANUds.PUDS_SVC_PARAM_RC_RRR, 0xF1A2, routine_control_option_record, routine_control_option_record_size)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcRoutineControl_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

def testRequestDownload(channel, config):
	"""
	Call UDS Service RequestDownload

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""
	memory_address_buffer = create_string_buffer(15)
	memory_addres_size= create_string_buffer(15)
	memory_address_size = 15
	memory_size_size = 8

	request = uds_msg()
	response = uds_msg()
	confirmation = uds_msg()

	CLEAR_CONSOLE()
	print("\n\n*** UDS Service: RequestDownload ***")

	# Sends a physical RequestDownload message
	print("\n\nSends a physical RequestDownload message: ")
	for i in range(memory_address_size):
		memory_address_buffer[i] = getCChar(ord('A') + i)
		memory_addres_size[i] = getCChar(10 + i)

	status = objPCANUds.SvcRequestDownload_2013(channel, config, request, 0x01, 0x02, memory_address_buffer, memory_address_size, memory_addres_size,
		memory_size_size)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcRequestDownload_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

def testRequestUpload(channel, config):
	"""
	Call UDS Service RequestUpload

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""
	memory_address_buffer = create_string_buffer(4)
	memory_size_buffer= create_string_buffer(4)
	memory_address_buffer_size = 4
	memory_size_size = 4

	request = uds_msg()
	response = uds_msg()
	confirmation = uds_msg()

	CLEAR_CONSOLE()
	print("\n\n*** UDS Service: RequestUpload ***")

	# Sends a physical RequestUpload message
	print("\n\nSends a physical RequestUpload message: ")
	for i in range(memory_size_size):
		memory_address_buffer[i] = getCChar(ord('A') + i)
		memory_size_buffer[i] = getCChar(10 + i)

	status = objPCANUds.SvcRequestUpload_2013(channel, config, request, 0x01, 0x02, memory_address_buffer, memory_address_buffer_size, memory_size_buffer,
		memory_size_size)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcRequestUpload_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

def testTransferData(channel, config):
	"""
	Call UDS Service TransferData

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""
	record = create_string_buffer(50)
	record_size = 50

	request = uds_msg()
	response = uds_msg()
	confirmation = uds_msg()

	CLEAR_CONSOLE()
	print("\n\n*** UDS Service: TransferData ***")

	# Sends a physical TransferData message
	print("\n\nSends a physical TransferData message: ")
	for i in range(record_size):
		record[i] = getCChar(ord('A') + i)

	status = objPCANUds.SvcTransferData_2013(channel, config, request, 0x01, record, record_size)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcTransferData_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

def testRequestTransferExit(channel, config):
	"""
	Call UDS Service RequestTransferExit

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""
	record = create_string_buffer(50)
	record_size = 20

	request = uds_msg()
	response = uds_msg()
	confirmation = uds_msg()

	CLEAR_CONSOLE()
	print("\n\n*** UDS Service: RequestTransferExit ***")

	# Sends a physical RequestTransferExit message
	print("\n\nSends a physical RequestTransferExit message: ")
	for i in range(record_size):
		record[i] = getCChar(ord('A') + i)

	status = objPCANUds.SvcRequestTransferExit_2013(channel, config, request, record, record_size)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcRequestTransferExit_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

def testTransferDataBigMessage(channel, config):
	"""
	Call UDS Service TransferData with MAX_DATA length

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""
	record = create_string_buffer(4095)
	record_size = 4093

	request = uds_msg()
	response = uds_msg()
	confirmation = uds_msg()

	CLEAR_CONSOLE()
	print("\n\n*** UDS Service: TransferData with MAX_DATA ***")

	# Sends a physical TransferData message with the maximum data available. The goal is to show that
	# UDS_WaitForService_2013 does not return a TIMEOUT error although the transmit and receive time of all the
	# data will be longer than the default time to get a response.
	print("\n\nSends a physical TransferData message (length=%d): " %(record_size))
	for i in range(record_size):
		record[i] = getCChar(c_ubyte((ord('A') + i)%256).value)
	status = objPCANUds.SvcTransferData_2013(channel, config, request, 0x01, record, record_size)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcTransferData_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

def testTransferDataMultipleFunctionalMessage(channel, config):
	"""
	Call UDS Service TransferData

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""

	record = create_string_buffer(5)
	record_size = 5
	response_count = c_uint32(0)

	request = uds_msg()
	response = uds_msg()
	confirmation = uds_msg()
	config_copy = uds_msgconfig()
	config_copy = copy.deepcopy(config)

	CLEAR_CONSOLE()
	print("\n\n*** UDS Service: TransferData with functional message***")

	# Initialize request message
	config_copy.nai.target_addr = PUDS_ISO_15765_4_ADDR_OBD_FUNCTIONAL
	config_copy.nai.target_type = PCANTP_ISOTP_ADDRESSING_FUNCTIONAL

	# Sends a functional TransferData message. The goal is to show that UDS_WaitForServiceFunctional_2013 waits long
	# enough to fetch all possible ECU responses.
	print("\n\nSends a functional TransferData message: ")
	for i in range(record_size):
		record[i] = getCChar(ord('A') + i)

	status = objPCANUds.SvcTransferData_2013(channel, config_copy, request, 0x01, record, record_size)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForServiceFunctional_2013(channel, request, 1, True,  response, response_count,confirmation)
	print(" UDS_SvcTransferData_2013: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, None, True)
		print("\n Received %u UDS responses:" %(response_count.value))
		display_uds_msg(None, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

def testUsingEvent(channel, config):
	"""
	Sample to use event

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""

	request = uds_msg()
	response = uds_msg()

	# set event handler
	if IS_WINDOWS:
		receive_event = c_void_p(windll.kernel32.CreateEventA(None, 0,0,None))
		status = objPCANUds.SetValue_2013(channel, PUDS_PARAMETER_RECEIVE_EVENT, receive_event, sizeof(receive_event))
	else:
		receive_event = NULL_HANDLE()
		status = objPCANUds.GetValue_2013(channel, PUDS_PARAMETER_RECEIVE_EVENT, receive_event, sizeof(receive_event))
	if not objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		print("Failed to set event, aborting...")
		if IS_WINDOWS:
			windll.kernel32.CloseHandle(receive_event)
		waitGetch()
		return

	CLEAR_CONSOLE()
	print("\n\n*** UDS Service with Event: TesterPresent ***")

	# Sends a physical TesterPresent message
	print("\n\nSends a physical TesterPresent message: ")
	status = objPCANUds.SvcTesterPresent_2013(channel, config, request, objPCANUds.PUDS_SVC_PARAM_TP_ZSUBF)
	print(" UDS_SvcTesterPresent_2013: %i" %(status.value))

	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):

		# Instead of calling WaitForService function, this sample demonstrates how event can be used. But note that
		# the use of a thread listening to notifications and making the read operations is preferred.
		stop = False

		# wait until we receive expected response
		while not stop:
			if IS_WINDOWS:
				windows_wait_result = c_uint64(windll.kernel32.WaitForSingleObject(receive_event, INFINITE))
				wait_result = True if windows_wait_result.value == WAIT_OBJECT_0 else False
			else:
				readable, _, _ = select.select([receive_event.value], [], [], None)
				wait_result = True if len(readable) > 0 else False

			if wait_result:

				# read all messages
				read_status = PUDS_STATUS_OK
				while not objPCANUds.StatusIsOk_2013(read_status, PUDS_STATUS_NO_MESSAGE, False):
					read_status = objPCANUds.Read_2013(channel, response, None, None)
					if objPCANUds.StatusIsOk_2013(read_status, PUDS_STATUS_OK, False):

						# this is a simple message check (type and sender/receiver address): to filter UDS request
						# confirmation and get first message from target, but real use-case should check that the UDS
						# service ID matches the request
						if response.msg.msgdata.isotp.contents.netaddrinfo.msgtype == PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC.value\
							and response.msg.msgdata.isotp.contents.netaddrinfo.source_addr == config.nai.target_addr\
							and response.msg.msgdata.isotp.contents.netaddrinfo.target_addr == config.nai.source_addr:
							stop = True
							display_uds_msg(request, response, False)
					# Free response message
					status = objPCANUds.MsgFree_2013(response)
					print(" Free response message: %i" %(status.value))

	waitGetch()

	# Free request message
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))

	# Uninitialize event
	if IS_WINDOWS:
		status = objPCANUds.SetValue_2013(channel, PUDS_PARAMETER_RECEIVE_EVENT, NULL_HANDLE(), sizeof(NULL_HANDLE()))
		print(" Remove receive event: %i" %(status.value))
		windll.kernel32.CloseHandle(receive_event)

def testAuthentication(channel, config):
	"""
	Call UDS Service Authentication

	parameters:
		channel: cantp channel handle
		config: Configuration of the request message (type, network address information...)
	"""

	communication_configuration = 0
	certificate_client = create_string_buffer(b'\x12\x34' )
	certificate_client_size = 2
	challenge_client= create_string_buffer(b'\x56\x78' )
	challenge_client_size = 2
	proof_of_ownership_client= create_string_buffer(b'\x9A\xBC' )
	proof_of_ownership_client_size = 2
	ephemeral_public_key_client= create_string_buffer(b'\xDE\xF0' )
	ephemeral_public_key_client_size = 2
	algorithm_indicator= (c_ubyte * 16)(0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15)
	additional_parameter= create_string_buffer(b'\xAA\xBB' )
	additional_parameter_size = 2

	request = uds_msg()
	response = uds_msg()
	confirmation = uds_msg()

	CLEAR_CONSOLE()
	print("\n\n*** UDS Service: Authentication ***")

	# Sends a physical Authentication/deAuthenticate message
	print("\n\nSends a physical Authentication/deAuthenticate message: ")
	status = objPCANUds.SvcAuthenticationDA_2020(channel, config, request)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcAuthenticationDA_2020: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a physical Authentication/verifyCertificateUnidirectional message
	print("\n\nSends a physical Authentication/verifyCertificateUnidirectional message: ")
	status = objPCANUds.SvcAuthenticationVCU_2020(channel, config, request,communication_configuration,certificate_client,certificate_client_size,challenge_client,challenge_client_size)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcAuthenticationVCU_2020: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a physical Authentication/verifyCertificateBidirectional message
	print("\n\nSends a physical Authentication/verifyCertificateBidirectional message: ")
	status = objPCANUds.SvcAuthenticationVCB_2020(channel, config, request,communication_configuration,certificate_client,certificate_client_size,challenge_client,challenge_client_size )
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcAuthenticationVCB_2020: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a physical Authentication/proofOfOwnership message
	print("\n\nSends a physical Authentication/proofOfOwnership message: ")
	status = objPCANUds.SvcAuthenticationPOWN_2020(channel, config, request,proof_of_ownership_client, proof_of_ownership_client_size, ephemeral_public_key_client, ephemeral_public_key_client_size)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcAuthenticationPOWN_2020: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a physical Authentication/requestChallengeForAuthentication message
	print("\n\nSends a physical Authentication/requestChallengeForAuthentication message: ")
	status = objPCANUds.SvcAuthenticationRCFA_2020(channel, config, request,communication_configuration,algorithm_indicator )
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcAuthenticationRCFA_2020: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a physical Authentication/verifyProofOfOwnershipUnidirectional message
	print("\n\nSends a physical Authentication/verifyProofOfOwnershipUnidirectional message: ")
	status = objPCANUds.SvcAuthenticationVPOWNU_2020(channel, config, request,algorithm_indicator,proof_of_ownership_client,proof_of_ownership_client_size,challenge_client,challenge_client_size,additional_parameter,additional_parameter_size )
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcAuthenticationVPOWNU_2020: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a physical Authentication/verifyProofOfOwnershipBidirectional message
	print("\n\nSends a physical Authentication/verifyProofOfOwnershipBidirectional message: ")
	status = objPCANUds.SvcAuthenticationVPOWNB_2020(channel, config, request, algorithm_indicator, proof_of_ownership_client, proof_of_ownership_client_size, challenge_client, challenge_client_size, additional_parameter,additional_parameter_size)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcAuthenticationVPOWNB_2020: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

	# Sends a physical Authentication/authenticationConfiguration message
	print("\n\nSends a physical Authentication/authenticationConfiguration message: ")
	status = objPCANUds.SvcAuthenticationAC_2020(channel, config, request)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		status = objPCANUds.WaitForService_2013(channel, request, response, confirmation)
	print(" UDS_SvcAuthenticationAC_2020: %i" %(status.value))
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
		display_uds_msg(confirmation, response, False)
	else:
		display_uds_msg(request, None, False)

	# Free messages
	status = objPCANUds.MsgFree_2013(request)
	print(" Free request message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(response)
	print(" Free response message: %i" %(status.value))
	status = objPCANUds.MsgFree_2013(confirmation)
	print(" Free confirmation message: %i" %(status.value))
	waitGetch()

#
# Main entry point of the program, start a CAN UDS client
#

# Set the PCAN-Channel to use
client_handle = PCANTP_HANDLE_USBBUS1 # TODO: modify the value according to your available PCAN devices.

# Initializing of the UDS Communication session
status = objPCANUds.Initialize_2013(client_handle, PCANTP_BAUDRATE_500K, 0, 0, 0)
print("Initialize UDS: %i" %(status.value))

# Define TimeOuts
tmp_buffer = c_ubyte(PCANTP_ISO_TIMEOUTS_15765_4)
status = objPCANUds.SetValue_2013(client_handle, PUDS_PARAMETER_ISO_TIMEOUTS, tmp_buffer, sizeof(tmp_buffer))
print("Set ISO 15765-4 timeouts values: %d" %(status.value))
waitGetch()

# Define Network Address Information used for all the tests
config = uds_msgconfig()
config.can_id = -1
config.can_msgtype = PCANTP_CAN_MSGTYPE_STANDARD
config.nai.protocol = PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL
config.nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL
config.type = PUDS_MSGTYPE_USDT
config.nai.source_addr = PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT
config.nai.target_addr = PUDS_ISO_15765_4_ADDR_ECU_1

# The following functions call UDS Services
testDiagnosticSessionControl(client_handle, config)
testECUReset(client_handle, config)
testSecurityAccess(client_handle, config)
testCommunicationControl(client_handle, config)
testTesterPresent(client_handle, config)
testSecuredDataTransmission(client_handle, config)
testControlDTCSetting(client_handle, config)
testResponseOnEvent(client_handle, config)
testLinkControl(client_handle, config)
testReadDataByIdentifier(client_handle, config)
testReadMemoryByAddress(client_handle, config)
testReadScalingDataByIdentifier(client_handle, config)
testReadDataByPeriodicIdentifier(client_handle, config)
testDynamicallyDefineDataIdentifier(client_handle, config)
testWriteDataByIdentifier(client_handle, config)
testWriteMemoryByAddress(client_handle, config)
testClearDiagnosticInformation(client_handle, config)
testReadDTCInformation(client_handle, config)
testInputOutputControlByIdentifier(client_handle, config)
testRoutineControl(client_handle, config)
testRequestDownload(client_handle, config)
testRequestUpload(client_handle, config)
testTransferData(client_handle, config)
testRequestTransferExit(client_handle, config)
testAccessTimingParameter(client_handle, config)
testRequestFileTransfer(client_handle, config)
testAuthentication(client_handle, config)

# Miscellaneous examples
testTransferDataBigMessage(client_handle, config)
testTransferDataMultipleFunctionalMessage(client_handle, config)
testUsingEvent(client_handle, config)

# Uninitialize channel
status = objPCANUds.Uninitialize_2013(client_handle)
print(" Unitialize channel: %i" %(status.value))

# Display a small report and quit
if g_nbErr > 0:
	print("\nERROR: %d errors occurred.\n" %(g_nbErr))
else:
	print("\nALL Transmissions succeeded !\n")

print("Press any key to continue...")
getInput()


