import copy
from PCAN_UDS_2013 import *

IS_WINDOWS = platform.system() == 'Windows'

# Support events
if not IS_WINDOWS:
	import select
	__LIBC_OBJ = cdll.LoadLibrary("libc.so.6")

WAIT_OBJECT_0 = 0
def NULL_HANDLE():
	return c_void_p(0) if IS_WINDOWS else c_int(0)

# Support keyboard
def getInput():
	sys.stdin.read(1)

# UDS library
objPCANUds = PCAN_UDS_2013()

# Help functions
def OK_KO(test):
	return "OK" if test else "KO"

def STATUS_OK_KO(test):
	return OK_KO(objPCANUds.StatusIsOk_2013(test, PUDS_STATUS_OK, False))

def mapping_list_contains(mapping_list, mapping_list_size, searched_mapping):
	"""
	Helper: Check if a mapping list contains a certain mapping
	prrameters:
		mapping_list : Mapping list
		mapping_list_size : Number of mapping in the mapping list
		searched_mapping : Searched mapping
	returns : Present (true) or not (false)
	"""
	res = False
	for i in range(mapping_list_size.value):

		# If unique identifier are the same, the mapping is in the list
		if mapping_list[i].uid == searched_mapping.uid:
			res = True
			break
	return res

# Definitions

# Define Bitrate: SAE J2284-4: High-Speed CAN (HSC) for Vehicle Applications at 500 kbps with CAN FD Data at 2 Mbps
PCAN_BITRATE_SAE_J2284_4 = create_string_buffer(b'f_clock=80000000,nom_brp=2,nom_tseg1=63,nom_tseg2=16,nom_sjw=16,data_brp=2,data_tseg1=15,data_tseg2=4,data_sjw=4')
BUFFER_SIZE = 256

MSG = "PEAK"
MSG_SIZE = 4
MAPPING_BUFFER_SIZE = 256

#
# Main entry point of the program, start a small CAN UDS read/write example
#

# Initialize variables
transmitter_handle = PCANTP_HANDLE_USBBUS1 # TODO : modify the value according to your available PCAN devices.
receiver_handle = PCANTP_HANDLE_USBBUS2 # TODO : modify the value according to your available PCAN devices.

# Print version informations
buffer = create_string_buffer(BUFFER_SIZE)
status = objPCANUds.GetValue_2013(PCANTP_HANDLE_NONEBUS, PUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE)
print("PCAN-UDS API Version: %s (%s)" %(buffer.value, STATUS_OK_KO(status)))

# Initialize transmitter and receiver
status = objPCANUds.InitializeFD_2013(transmitter_handle, PCAN_BITRATE_SAE_J2284_4)
print("Initialize transmitter: %s" %(STATUS_OK_KO(status)))
status = objPCANUds.InitializeFD_2013(receiver_handle, PCAN_BITRATE_SAE_J2284_4)
print("Initialize receiver: %s" %(STATUS_OK_KO(status)))

# Create a receive event on receiver
if IS_WINDOWS:
	receive_event = c_void_p(windll.kernel32.CreateEventA(None, 0,0,None))
	res = objPCANUds.SetValue_2013(receiver_handle, PUDS_PARAMETER_RECEIVE_EVENT, receive_event, sizeof(receive_event))
	print("Set receive event parameter: %s" %(STATUS_OK_KO(res)))
else:
	receive_event = NULL_HANDLE()
	res = objPCANUds.GetValue_2013(receiver_handle, PUDS_PARAMETER_RECEIVE_EVENT, receive_event, sizeof(receive_event))
	print("Get receive event parameter: %s" %(STATUS_OK_KO(res)))

# Initialize 29bits
msg_config = uds_msgconfig()
msg_config.can_id = 0x30
msg_config.can_msgtype = PCANTP_CAN_MSGTYPE_STANDARD
# To use CAN FD with bitrate switch uncomment:
# msg_config.can_msgtype = cantp_can_msgtype(msg_config.can_msgtype | (PCANTP_CAN_MSGTYPE_FD.value | PCANTP_CAN_MSGTYPE_BRS.value))
msg_config.nai.protocol = PUDS_MSGPROTOCOL_ISO_15765_2_29B_NORMAL
msg_config.nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL
msg_config.type = PUDS_MSGTYPE_USDT
msg_config.nai.source_addr = 0x10
msg_config.nai.target_addr = 0x20

# Allocate TX message with previous configuration
tx_msg = uds_msg()
status = objPCANUds.MsgAlloc_2013(tx_msg, msg_config, MSG_SIZE)
if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
	for i in range(MSG_SIZE):
		tx_msg.msg.msgdata.any.contents.data[i] = ord(MSG[i])
print("Allocate TX message: %s" %(STATUS_OK_KO(status)))


# Initialize source mapping
source_mapping = uds_mapping()
source_mapping.can_id = msg_config.can_id
source_mapping.can_id_flow_ctrl = msg_config.can_id + 1
source_mapping.can_msgtype = msg_config.can_msgtype
source_mapping.can_tx_dlc = 8
source_mapping.nai = copy.deepcopy(msg_config.nai)

# Initialize response mapping
response_mapping = uds_mapping()
response_mapping.can_id = source_mapping.can_id_flow_ctrl
response_mapping.can_id_flow_ctrl = source_mapping.can_id
response_mapping.can_msgtype = source_mapping.can_msgtype
response_mapping.can_tx_dlc = source_mapping.can_tx_dlc
response_mapping.nai = copy.deepcopy(source_mapping.nai)
response_mapping.nai.source_addr = source_mapping.nai.target_addr
response_mapping.nai.target_addr = source_mapping.nai.source_addr

# Add mappings on transmitter
status = objPCANUds.AddMapping_2013(transmitter_handle, source_mapping)
print("Add source mapping on transmitter: %s" %(STATUS_OK_KO(status)))
status = objPCANUds.AddMapping_2013(transmitter_handle, response_mapping)
print("Add response mapping on transmitter: %s" %(STATUS_OK_KO(status)))

# Check if mappings are added on transmitter
mapping_buffer = (uds_mapping * MAPPING_BUFFER_SIZE)()
mapping_count = c_uint16(0)
status = objPCANUds.GetMappings_2013(transmitter_handle, mapping_buffer, MAPPING_BUFFER_SIZE, mapping_count)
print("Get mappings on transmitter: %s" %(STATUS_OK_KO(status)))
print("Check added mappings on transmitter: %s" %(OK_KO(mapping_list_contains(mapping_buffer, mapping_count, source_mapping) and mapping_list_contains(mapping_buffer, mapping_count, response_mapping))))

# Add mappings on receiver
status = objPCANUds.AddMapping_2013(receiver_handle, source_mapping)
print("Add source mapping on receiver: %s" %(STATUS_OK_KO(status)))
status = objPCANUds.AddMapping_2013(receiver_handle, response_mapping)
print("Add response mapping on receiver: %s" %(STATUS_OK_KO(status)))

# Check if mappings are added on receiver
status = objPCANUds.GetMappings_2013(receiver_handle, mapping_buffer, MAPPING_BUFFER_SIZE, mapping_count)
print("Get mappings on receiver: %s" %(STATUS_OK_KO(status)))
print("Check added mappings on receiver: %s" %(OK_KO(mapping_list_contains(mapping_buffer, mapping_count, source_mapping) and mapping_list_contains(mapping_buffer, mapping_count, response_mapping))))

# Write message
status = objPCANUds.Write_2013(transmitter_handle, tx_msg)
print("Write \"PEAK\": %s" %(STATUS_OK_KO(status)))

# Wait a receive event
if IS_WINDOWS:
	windows_wait_result = c_uint64(windll.kernel32.WaitForSingleObject(receive_event, 3000))
	wait_result = True if windows_wait_result.value == WAIT_OBJECT_0 else False
else:
	readable, _, _ = select.select([receive_event.value], [], [], 3)
	wait_result = True if len(readable) > 0 else False
print("Wait a receive event: %s" %(OK_KO(wait_result)))

# If we get a receive event
rx_msg = uds_msg()
if wait_result:

	# Read message
	status = objPCANUds.Read_2013(receiver_handle, rx_msg, None, None)
	print("Receive message: %s" %(STATUS_OK_KO(status)))

	# Check message (should contains "PEAK")
	dataEqual = objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False) and rx_msg.msg.msgdata.any.contents.length == MSG_SIZE
	if dataEqual:
		for i in range(MSG_SIZE):
			if tx_msg.msg.msgdata.any.contents.data[i] != rx_msg.msg.msgdata.any.contents.data[i]:
				dataEqual = False
				break
	if dataEqual:
		print("Message contains \"%c%c%c%c\": OK" %(rx_msg.msg.msgdata.any.contents.data[0], rx_msg.msg.msgdata.any.contents.data[1], rx_msg.msg.msgdata.any.contents.data[2], rx_msg.msg.msgdata.any.contents.data[3]))
	else:
		print("Message is corrupted: KO")


# Close receive event
if IS_WINDOWS:
	status = objPCANUds.SetValue_2013(receiver_handle, PUDS_PARAMETER_RECEIVE_EVENT, NULL_HANDLE(), sizeof(NULL_HANDLE()))
	print("Stop receive event receiver: %s" %(STATUS_OK_KO(status)))
	boolean_status = windll.kernel32.CloseHandle(receive_event)
	print("Close receive event: %s" %(OK_KO(boolean_status)))

# Free messages
status = objPCANUds.MsgFree_2013(tx_msg)
print("Free TX message: %s" %(STATUS_OK_KO(status)))
status = objPCANUds.MsgFree_2013(rx_msg)
print("Free RX message: %s" %(STATUS_OK_KO(status)))

# Close transmitter and receiver
status = objPCANUds.Uninitialize_2013(transmitter_handle)
print("Uninitialize transmitter: %s" %(STATUS_OK_KO(status)))
status = objPCANUds.Uninitialize_2013(receiver_handle)
print("Uninitialize receiver: %s" %(STATUS_OK_KO(status)))

# Exit
print("Press any key to continue...")
getInput()
