import copy
from PCAN_UDS_2013 import *

IS_WINDOWS = platform.system() == 'Windows'

# Support events
if not IS_WINDOWS:
	import select
	__LIBC_OBJ = cdll.LoadLibrary("libc.so.6")

WAIT_OBJECT_0 = 0
def NULL_HANDLE():
	return c_void_p(0) if IS_WINDOWS else c_int(0)

# Support keyboard
def getInput():
	sys.stdin.read(1)

# UDS and Isotp libraries
objPCANUds = PCAN_UDS_2013()
objPCANIsotp = PCAN_ISO_TP_2016()

# Help functions
def OK_KO(test):
	return "OK" if test else "KO"

def STATUS_OK_KO(test):
	return OK_KO(objPCANUds.StatusIsOk_2013(test, PUDS_STATUS_OK, False))

# Definitions

# Define Bitrate: SAE J2284-4: High-Speed CAN (HSC) for Vehicle Applications at 500 kbps with CAN FD Data at 2 Mbps
PCAN_BITRATE_SAE_J2284_4 = create_string_buffer(b'f_clock=80000000,nom_brp=2,nom_tseg1=63,nom_tseg2=16,nom_sjw=16,data_brp=2,data_tseg1=15,data_tseg2=4,data_sjw=4')
BUFFER_SIZE = 256
MSG = create_string_buffer(b'PEAK')
MSG_SIZE = 4

#
# Main entry point of the program, start a small CAN UDS / ISOTP read-write example
# This example initialise two UDS channel with a specific mapping.
# Then it inits an ISOTP mapping and write an ISOTP message.
# In this case, PCAN-UDS should not receive the message, we need PCAN-ISOTP to read it.
#

# Initialize variables
transmitter_handle = PCANTP_HANDLE_USBBUS1 # TODO: modify the value according to your available PCAN devices.
receiver_handle = PCANTP_HANDLE_USBBUS2 # TODO: modify the value according to your available PCAN devices.
can_msgtype = PCANTP_CAN_MSGTYPE_STANDARD

# To use CAN FD with bitrate switch uncomment:
# can_msgtype = cantp_can_msgtype(can_msgtype | (PCANTP_CAN_MSGTYPE_FD.value | PCANTP_CAN_MSGTYPE_BRS.value))

# Print version informations
buffer = create_string_buffer(BUFFER_SIZE)
status = objPCANUds.GetValue_2013(PCANTP_HANDLE_NONEBUS, PUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE)
print("PCAN-UDS API Version: %s (%s)" %(buffer.value, STATUS_OK_KO(status)))

# Initialize transmitter and receiver
status = objPCANUds.InitializeFD_2013(transmitter_handle, PCAN_BITRATE_SAE_J2284_4)
print("Initialize transmitter: %s" %(STATUS_OK_KO(status)))
status = objPCANUds.InitializeFD_2013(receiver_handle, PCAN_BITRATE_SAE_J2284_4)
print("Initialize receiver: %s" %(STATUS_OK_KO(status)))

# Create a receive event on receiver
if IS_WINDOWS:
	receive_event = c_void_p(windll.kernel32.CreateEventA(None, 0,0,None))
	res = objPCANUds.SetValue_2013(receiver_handle, PUDS_PARAMETER_RECEIVE_EVENT, receive_event, sizeof(receive_event))
	print("Set receive event parameter: %s" %(STATUS_OK_KO(res)))
else:
	receive_event = NULL_HANDLE()
	res = objPCANUds.GetValue_2013(receiver_handle, PUDS_PARAMETER_RECEIVE_EVENT, receive_event, sizeof(receive_event))
	print("Get receive event parameter: %s" %(STATUS_OK_KO(res)))

# Initialize source mapping
source_mapping = uds_mapping()
source_mapping.can_id = 0x30
source_mapping.can_id_flow_ctrl = source_mapping.can_id + 1
source_mapping.can_msgtype = can_msgtype
source_mapping.can_tx_dlc = 8
source_mapping.nai.protocol = PUDS_MSGPROTOCOL_ISO_15765_2_29B_NORMAL
source_mapping.nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL
source_mapping.nai.source_addr = 0x10
source_mapping.nai.target_addr = 0x20

# Initialize response mapping
response_mapping = uds_mapping()
response_mapping.can_id = source_mapping.can_id_flow_ctrl
response_mapping.can_id_flow_ctrl = source_mapping.can_id
response_mapping.can_msgtype = source_mapping.can_msgtype
response_mapping.can_tx_dlc = source_mapping.can_tx_dlc
response_mapping.nai = copy.deepcopy(source_mapping.nai)
response_mapping.nai.source_addr = source_mapping.nai.target_addr
response_mapping.nai.target_addr = source_mapping.nai.source_addr


# Add UDS mappings on transmitter
status = objPCANUds.AddMapping_2013(transmitter_handle, source_mapping)
print("Add source mapping on transmitter: %s" %(STATUS_OK_KO(status)))
status = objPCANUds.AddMapping_2013(transmitter_handle, response_mapping)
print("Add response mapping on transmitter: %s" %(STATUS_OK_KO(status)))

# Add UDS mappings on and receiver
status = objPCANUds.AddMapping_2013(receiver_handle, source_mapping)
print("Add source mapping on receiver: %s" %(STATUS_OK_KO(status)))
status = objPCANUds.AddMapping_2013(receiver_handle, response_mapping)
print("Add response mapping on receiver: %s" %(STATUS_OK_KO(status)))

# Create a simple isotp physical mapping:
#    - Source 0xF1 (transmitter), target 0x01 (receiver), CAN id 0xA1, CAN ID flow control 0xA2
#    - Diagnostic message in a classic format
mapping = cantp_mapping()
mapping.can_id = 0xA1
mapping.can_id_flow_ctrl = 0xA2
mapping.can_msgtype = can_msgtype
mapping.can_tx_dlc = 0x0
mapping.netaddrinfo.extension_addr = 0x00
mapping.netaddrinfo.format = PCANTP_ISOTP_FORMAT_NORMAL
mapping.netaddrinfo.msgtype = PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC
mapping.netaddrinfo.source_addr = 0xF1
mapping.netaddrinfo.target_addr = 0x01
mapping.netaddrinfo.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL

# Create the associated isotp reversed mapping:
reverse_mapping = cantp_mapping()
reverse_mapping.can_id = mapping.can_id_flow_ctrl
reverse_mapping.can_id_flow_ctrl = mapping.can_id
reverse_mapping.can_msgtype = mapping.can_msgtype
reverse_mapping.can_tx_dlc = mapping.can_tx_dlc
reverse_mapping.netaddrinfo = copy.deepcopy(mapping.netaddrinfo)
reverse_mapping.netaddrinfo.source_addr = mapping.netaddrinfo.target_addr
reverse_mapping.netaddrinfo.target_addr = mapping.netaddrinfo.source_addr

# Add ISOTP mappings on channels
status = objPCANIsotp.AddMapping_2016(transmitter_handle, mapping)
print("Add a simple isotp mapping on transmitter: %s" %(STATUS_OK_KO(status)))
status = objPCANIsotp.AddMapping_2016(transmitter_handle, reverse_mapping)
print("Add the reverse isotp mapping on transmitter: %s" %(STATUS_OK_KO(status)))
status = objPCANIsotp.AddMapping_2016(receiver_handle, mapping)
print("Add a simple isotp mapping on receiver: %s" %(STATUS_OK_KO(status)))
status = objPCANIsotp.AddMapping_2016(receiver_handle, reverse_mapping)
print("Add the reverse isotp mapping on receiver: %s" %(STATUS_OK_KO(status)))

# Initialize ISOTP Tx message containing "PEAK"
tx_msg = cantp_msg()
status = objPCANIsotp.MsgDataAlloc_2016(tx_msg, PCANTP_MSGTYPE_ISOTP)
print("Allocate tx ISOTP message: %s" %(STATUS_OK_KO(status)))
status = objPCANIsotp.MsgDataInit_2016(tx_msg, mapping.can_id, mapping.can_msgtype, MSG_SIZE, MSG, mapping.netaddrinfo)
print("Initialize tx message: %s" %(STATUS_OK_KO(status)))

# Send "PEAK" ISOTP message
status = objPCANIsotp.Write_2016(transmitter_handle, tx_msg)
print("Write \"PEAK\": %s" %(STATUS_OK_KO(status)))

# Wait a receive event on uds message (should not work because we sent a ISOTP message)
if IS_WINDOWS:
	windows_wait_result = c_uint64(windll.kernel32.WaitForSingleObject(receive_event, 3000))
	wait_result = True if windows_wait_result.value == WAIT_OBJECT_0 else False
else:
	readable, _, _ = select.select([receive_event.value], [], [], 3)
	wait_result = True if len(readable) > 0 else False
print("Wait a receive event on UDS message (should not work because we sent an ISOTP message): %s" %(OK_KO(wait_result)))

# We should receive a ISOTP message instead
rx_msg = cantp_msg()
status = objPCANIsotp.Read_2016(receiver_handle, rx_msg, None, PCANTP_MSGTYPE_NONE)
print("Read ISOTP message on receiver: %s" %(STATUS_OK_KO(status)))

# Check message (should contains "PEAK")
msgOk = True
if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False) and rx_msg.msgdata.any.contents.length == MSG_SIZE:
	for i in range(MSG_SIZE):
		if tx_msg.msgdata.any.contents.data[i] != rx_msg.msgdata.any.contents.data[i]:
			msgOk = False
			break
else:
	msgOk = False
if msgOk:
	print("Message contains \"%c%c%c%c\": OK" %(rx_msg.msgdata.any.contents.data[0], rx_msg.msgdata.any.contents.data[1], rx_msg.msgdata.any.contents.data[2], rx_msg.msgdata.any.contents.data[3]))
else:
	print("Message is corrupted: KO")


# Close receive event
if IS_WINDOWS:
	status = objPCANUds.SetValue_2013(receiver_handle, PUDS_PARAMETER_RECEIVE_EVENT, NULL_HANDLE(), sizeof(NULL_HANDLE()))
	print("Stop receive event receiver: %s" %(STATUS_OK_KO(status)))
	boolean_status = windll.kernel32.CloseHandle(receive_event)
	print("Close receive event: %s" %(OK_KO(boolean_status)))

# Free messages
status = objPCANIsotp.MsgDataFree_2016(tx_msg)
print("Free TX message: %s" %(STATUS_OK_KO(status)))
status = objPCANIsotp.MsgDataFree_2016(rx_msg)
print("Free RX message: %s" %(STATUS_OK_KO(status)))

# Close transmitter and receiver
status = objPCANUds.Uninitialize_2013(transmitter_handle)
print("Uninitialize transmitter: %s" %(STATUS_OK_KO(status)))
status = objPCANUds.Uninitialize_2013(receiver_handle)
print("Uninitialize receiver: %s" %(STATUS_OK_KO(status)))

# Exit
print("Press any key to continue...")
getInput()


