import copy
import threading
from PCAN_UDS_2013 import *

IS_WINDOWS = platform.system() == 'Windows'

# Support events
if not IS_WINDOWS:
	import select
	__LIBC_OBJ = cdll.LoadLibrary("libc.so.6")

WAIT_OBJECT_0 = 0
def NULL_HANDLE():
	return c_void_p(0) if IS_WINDOWS else c_int(0)

# Support keyboard
if IS_WINDOWS:
	from msvcrt import getch, kbhit
	def install_keyboardHit():
		return 0
	def desinstall_keyboardHit(old_settings):
		pass
	def keyboardHit():
		return kbhit() != 0
	def getInput():
		key = getch()
		try:
			key_decode = key.decode('utf-8')
			return key_decode
		except UnicodeDecodeError:
			return key
else:
	import tty
	import termios
	def install_keyboardHit():
		old_settings = termios.tcgetattr(sys.stdin)
		tty.setcbreak(sys.stdin.fileno())
		return old_settings
	def desinstall_keyboardHit(old_settings):
		termios.tcsetattr(sys.stdin, termios.TCSADRAIN, old_settings)
	def keyboardHit():
		return select.select([sys.stdin], [], [], 0) == ([sys.stdin], [], [])
	def getInput():
		return sys.stdin.read(1)

# UDS and Isotp libraries
objPCANUds = PCAN_UDS_2013()
objPCANIsotp = PCAN_ISO_TP_2016()

# Help functions
def OK_KO(test):
	return "OK" if test else "KO"

def UDS_STATUS_OK_KO(test):
	return OK_KO(objPCANUds.StatusIsOk_2013(test, PUDS_STATUS_OK, False))

def ISOTP_STATUS_OK_KO(test):
	return OK_KO(objPCANIsotp.StatusIsOk_2016(test, PCANTP_STATUS_OK, False))

class task_params (Structure):
	"""
	Structure passed as thread parameters
	"""
	_fields_ = [("server_handle", cantp_handle),	# Server channel handle
				("server_address", c_uint32),		# Server address
				("stop_task", c_bool)]				# Determine if the thread should end or not

def isotp_server_task(parameters):
	"""
	ISOTP server task: respond "PING" message by sending "PONG"
		parameters: pointer on task_params structures
	"""

	# Create a isotp receive event

	if IS_WINDOWS:
		receive_event = c_void_p(windll.kernel32.CreateEventA(None, 0,0,None))
		status = objPCANIsotp.SetValue_2016(parameters.server_handle, PCANTP_PARAMETER_RECEIVE_EVENT, receive_event, sizeof(receive_event))
		print("[ISOTP] Set isotp receive event parameter: %s" %(ISOTP_STATUS_OK_KO(status)))
	else:
		receive_event = NULL_HANDLE()
		status = objPCANIsotp.GetValue_2016(parameters.server_handle, PCANTP_PARAMETER_RECEIVE_EVENT, receive_event, sizeof(receive_event))
		print("[ISOTP] Get isotp receive event parameter: %s" %(ISOTP_STATUS_OK_KO(status)))

	# Create a simple isotp physical mapping:
	#    - Source 0xA1 (client), target 0xA2 (server), CAN id 0xA1, CAN ID flow control 0xA2
	#    - Diagnostic message in a classic format
	mapping = cantp_mapping()
	mapping.can_id = 0xA1
	mapping.can_id_flow_ctrl = 0xA2
	mapping.can_msgtype = PCANTP_CAN_MSGTYPE_STANDARD
	mapping.netaddrinfo.extension_addr = 0x00
	mapping.netaddrinfo.format = PCANTP_ISOTP_FORMAT_NORMAL
	mapping.netaddrinfo.msgtype = PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC
	mapping.netaddrinfo.source_addr = 0xA1
	mapping.netaddrinfo.target_addr = 0xA2
	mapping.netaddrinfo.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL

	# Create the associated isotp reversed mapping:
	reverse_mapping = cantp_mapping()
	reverse_mapping.can_id = mapping.can_id_flow_ctrl
	reverse_mapping.can_id_flow_ctrl = mapping.can_id
	reverse_mapping.can_msgtype = PCANTP_CAN_MSGTYPE_STANDARD
	reverse_mapping.netaddrinfo = copy.deepcopy(mapping.netaddrinfo)
	reverse_mapping.netaddrinfo.extension_addr = 0x00
	reverse_mapping.netaddrinfo.source_addr = mapping.netaddrinfo.target_addr
	reverse_mapping.netaddrinfo.target_addr = mapping.netaddrinfo.source_addr

	# Add ISOTP mappings on channels
	status = objPCANIsotp.AddMapping_2016(parameters.server_handle, mapping)
	print("[ISOTP] Add a simple isotp mapping: %s" %(ISOTP_STATUS_OK_KO(status)))
	status = objPCANIsotp.AddMapping_2016(parameters.server_handle, reverse_mapping)
	print("[ISOTP] Add the reverse isotp mapping: %s" %(ISOTP_STATUS_OK_KO(status)))

	# Initialize ISOTP Tx message containing "PING"
	tx_msg = cantp_msg()
	status = objPCANIsotp.MsgDataAlloc_2016(tx_msg, PCANTP_MSGTYPE_ISOTP)
	print("[ISOTP] Allocate ISOTP tx message: %s" %(ISOTP_STATUS_OK_KO(status)))
	status = objPCANIsotp.MsgDataInit_2016(tx_msg, reverse_mapping.can_id, reverse_mapping.can_msgtype, MSG_SIZE, ISOTP_RESPONSE_MSG, reverse_mapping.netaddrinfo)
	print("[ISOTP] Initialize ISOTP tx message: %s" %(ISOTP_STATUS_OK_KO(status)))

	# Wait a receive event on isotp message
	while not parameters.stop_task:

		if IS_WINDOWS:
			windows_wait_result = c_uint64(windll.kernel32.WaitForSingleObject(receive_event, 1000))
			wait_result = True if windows_wait_result.value == WAIT_OBJECT_0 else False
		else:
			readable, _, _ = select.select([receive_event.value], [], [], 1)
			wait_result = True if len(readable) > 0 else False
		print("[ISOTP] Wait a receive event from isotp: %s" %(OK_KO(wait_result)))

		# Read ISOTP messages
		rx_msg = cantp_msg()
		status = PCANTP_STATUS_OK
		while not objPCANIsotp.StatusIsOk_2016(status, PCANTP_STATUS_NO_MESSAGE, False):
			status = objPCANIsotp.Read_2016(parameters.server_handle, rx_msg, None, PCANTP_MSGTYPE_NONE)
			print("[ISOTP] Read ISOTP message: %s" %(ISOTP_STATUS_OK_KO(status)))

			# Check message (should contains "PING") and send "PONG"
			if not objPCANIsotp.StatusIsOk_2016(status, PCANTP_STATUS_NO_MESSAGE, False):
				msgOk = objPCANIsotp.StatusIsOk_2016(status, PCANTP_STATUS_OK, False) and rx_msg.msgdata.any.contents.length == MSG_SIZE
				if msgOk:
					for i in range(MSG_SIZE):
						if rx_msg.msgdata.any.contents.data[i] != ord(ISOTP_REQUEST_MSG[i]):
							msgOk = False
							break
				if msgOk:
					print("[ISOTP] Received message contains \"%c%c%c%c\": OK" %(rx_msg.msgdata.any.contents.data[0], rx_msg.msgdata.any.contents.data[1], rx_msg.msgdata.any.contents.data[2], rx_msg.msgdata.any.contents.data[3]))
					status = objPCANIsotp.Write_2016(parameters.server_handle, tx_msg)
					print("[ISOTP] Send ISOTP \"PONG\" message: %s" %(ISOTP_STATUS_OK_KO(status)))

				# Free RX message
				status = objPCANIsotp.MsgDataFree_2016(rx_msg)
				print("[ISOTP] Free RX message: %s" %(ISOTP_STATUS_OK_KO(status)))

	# Close receive event
	if IS_WINDOWS:
		status = objPCANIsotp.SetValue_2016(parameters.server_handle, PCANTP_PARAMETER_RECEIVE_EVENT, NULL_HANDLE(), sizeof(NULL_HANDLE()))
		print("[ISOTP] Stop ISOTP receive event: %s" %(ISOTP_STATUS_OK_KO(status)))
		boolean_status = windll.kernel32.CloseHandle(receive_event)
		print("[ISOTP] Close ISOTP receive event: %s" %(OK_KO(boolean_status)))


	# Free messages
	status = objPCANIsotp.MsgDataFree_2016(tx_msg)
	print("[ISOTP] Free ISOTP TX message: %s" %(ISOTP_STATUS_OK_KO(status)))

	return 0

def uds_server_task(parameters):
	"""
	UDS server task: respond TesterPresent request
		parameters: pointer on task_params structures
	"""

	# Set server address parameter
	status = objPCANUds.SetValue_2013(parameters.server_handle, PUDS_PARAMETER_SERVER_ADDRESS, c_uint32(parameters.server_address), sizeof(c_uint32))
	print("[UDS] Set UDS server address: %s" %(UDS_STATUS_OK_KO(status)))

	# Set a receive event for UDS
	if IS_WINDOWS:
		receive_event = c_void_p(windll.kernel32.CreateEventA(None, 0,0,None))
		status = objPCANUds.SetValue_2013(parameters.server_handle, PUDS_PARAMETER_RECEIVE_EVENT, receive_event, sizeof(receive_event))
		print("[UDS] Set UDS receive event parameter: %s" %(UDS_STATUS_OK_KO(status)))
	else:
		receive_event = NULL_HANDLE()
		res = objPCANUds.GetValue_2013(parameters.server_handle, PUDS_PARAMETER_RECEIVE_EVENT, receive_event, sizeof(receive_event))
		print("[UDS] Get UDS receive event parameter: %s" %(UDS_STATUS_OK_KO(status)))

	# Initialize a physical configuration
	config_physical = uds_msgconfig()
	config_physical.can_id = -1
	config_physical.can_msgtype = PCANTP_CAN_MSGTYPE_STANDARD
	config_physical.nai.protocol = PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL
	config_physical.nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL
	config_physical.type = PUDS_MSGTYPE_USDT
	config_physical.nai.extension_addr = 0

	while not parameters.stop_task:

		# Wait a receive event on receiver
		if IS_WINDOWS:
			windows_wait_result = c_uint64(windll.kernel32.WaitForSingleObject(receive_event, 1000))
			wait_result = True if windows_wait_result.value == WAIT_OBJECT_0 else False
		else:
			readable, _, _ = select.select([receive_event.value], [], [], 1)
			wait_result = True if len(readable) > 0 else False

		# If we get a receive event
		if wait_result:
			response_msg = uds_msg()
			request_msg = uds_msg()
			read_status = PUDS_STATUS_OK
			while not objPCANUds.StatusIsOk_2013(read_status, PUDS_STATUS_NO_MESSAGE, False):

				# Read first available message (no filtering based on message's type is set):

				read_status = objPCANUds.Read_2013(parameters.server_handle, request_msg, None, None)
				print("[UDS] Try to read a message: %s" %(UDS_STATUS_OK_KO(read_status)))
				if objPCANUds.StatusIsOk_2013(read_status, PUDS_STATUS_OK, False):

					# We receive a request, check its length and if it is not a loopback message and if it is a USDT message
					if request_msg.type == PUDS_MSGTYPE_USDT.value and request_msg.msg.msgdata.any.contents.length >= 1 and (request_msg.msg.msgdata.any.contents.flags & PCANTP_MSGFLAG_LOOPBACK.value) == 0:

						# This is a valid request, switch services
						service_id = request_msg.links.service_id[0]
						if service_id == objPCANUds.PUDS_SI_TesterPresent:

							# Allocate response message
							status = objPCANUds.MsgAlloc_2013(response_msg, config_physical, 2)
							print("[UDS] Prepare response message for TesterPresent service: %s" %(UDS_STATUS_OK_KO(status)))

							if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):

								# Fill parameters
								response_msg.msg.msgdata.isotp.contents.netaddrinfo = copy.deepcopy(request_msg.msg.msgdata.isotp.contents.netaddrinfo)
								response_msg.links.service_id[0] = c_ubyte(objPCANUds.PUDS_SI_TesterPresent + objPCANUds.PUDS_SI_POSITIVE_RESPONSE)
								response_msg.links.param[0] = 0
								response_msg.msg.msgdata.isotp.contents.netaddrinfo.target_addr = request_msg.msg.msgdata.isotp.contents.netaddrinfo.source_addr
								response_msg.msg.msgdata.isotp.contents.netaddrinfo.source_addr = parameters.server_address

								# Write response message
								status = objPCANUds.Write_2013(parameters.server_handle, response_msg)
								print("[UDS] Write response message for TesterPresent service: %s" %(UDS_STATUS_OK_KO(status)))

							# Free response message (and clean memory in order to reallocate later)
							status = objPCANUds.MsgFree_2013(response_msg)
							print("[UDS] Free response message: %s" %(UDS_STATUS_OK_KO(status)))
						else:
							print("[UDS] Unknown service (0x%02x)" %(service_id))


				# Free request message (and clean memory in order to reallocate later)
				status = objPCANUds.MsgFree_2013(request_msg)
				print("[UDS] Free request message: %s" %(UDS_STATUS_OK_KO(status)))

	# Close receive event
	if IS_WINDOWS:
		status = objPCANUds.SetValue_2013(parameters.server_handle, PUDS_PARAMETER_RECEIVE_EVENT, NULL_HANDLE(), sizeof(NULL_HANDLE()))
		print("[UDS] Stop UDS receive event: %s" %(UDS_STATUS_OK_KO(status)))
		boolean_status = windll.kernel32.CloseHandle(receive_event)
		print("[UDS] Close UDS receive event: %s" %(OK_KO(boolean_status)))

	return 0

# Definitions
BUFFER_SIZE = 256
ISOTP_REQUEST_MSG = create_string_buffer(b'PING')
ISOTP_RESPONSE_MSG = create_string_buffer(b'PONG')
MSG_SIZE = 4

#
# Main entry point of the program, start a small server which handle UDS testerpresent request and isotp request
#

try:
	old_settings = install_keyboardHit()

	# Initialize variables
	t_params = task_params()
	t_params.server_handle = PCANTP_HANDLE_USBBUS2 # TODO: modify the value according to your available PCAN devices.
	t_params.server_address = PUDS_ISO_15765_4_ADDR_ECU_1
	t_params.stop_task = False

	# Print version informations
	buffer = create_string_buffer(BUFFER_SIZE)
	status = objPCANUds.GetValue_2013(PCANTP_HANDLE_NONEBUS, PUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE)
	print("PCAN-UDS API Version - %s: %s" %(buffer.value, UDS_STATUS_OK_KO(status)))

	# Initialize server
	status = objPCANUds.Initialize_2013(t_params.server_handle, PCANTP_BAUDRATE_500K, 0, 0, 0)
	print("Initialize channel: %s" %(UDS_STATUS_OK_KO(status)))

	# Start uds and isotp servers
	uds_server = threading.Thread(target = uds_server_task, args = (t_params,))
	uds_server.start()
	isotp_server = threading.Thread(target = isotp_server_task, args = (t_params,))
	isotp_server.start()

	# Read while user do not press Q
	print("Start listening, press Q to quit.")
	t_params.stop_task = False
	while not t_params.stop_task:
		# Quit when user press Q
		if keyboardHit():
			keyboard_res = getInput()
			if keyboard_res == 'Q' or keyboard_res == 'q':
				t_params.stop_task = True

	# Close threads
	uds_server.join()
	isotp_server.join()

	# Close channel
	status = objPCANUds.Uninitialize_2013(t_params.server_handle)
	print("Uninitialize channel: %s" %(UDS_STATUS_OK_KO(status)))

	#Exit
	print("Press any key to continue...")
	getInput()

finally:
	desinstall_keyboardHit(old_settings)

