import copy
import threading
from PCAN_UDS_2013 import *

IS_WINDOWS = platform.system() == 'Windows'

# Support events
if not IS_WINDOWS:
	import select
	__LIBC_OBJ = cdll.LoadLibrary("libc.so.6")

WAIT_OBJECT_0 = 0
def NULL_HANDLE():
	return c_void_p(0) if IS_WINDOWS else c_int(0)

# Support keyboard
def getInput():
	sys.stdin.read(1)

# Uds and Isotp libraries
objPCANUds = PCAN_UDS_2013()
objPCANIsotp = PCAN_ISO_TP_2016()

# Help functions
def OK_KO(test):
	return "OK" if test else "KO"

def UDS_STATUS_OK_KO(test):
	return OK_KO(objPCANUds.StatusIsOk_2013(test, PUDS_STATUS_OK, False))

def ISOTP_STATUS_OK_KO(test):
	return OK_KO(objPCANIsotp.StatusIsOk_2016(test, PCANTP_STATUS_OK, False))

def uds_client_task(param):
	"""
	UDS client task: request TesterPresent service several time

	parameters:
		param: client channel handle
	"""

	# Initialize variables
	client_handle = param

	# Initialize a physical configuration
	config_physical = uds_msgconfig()
	config_physical.can_id = PUDS_ISO_15765_4_CAN_ID_PHYSICAL_REQUEST_1
	config_physical.can_msgtype = PCANTP_CAN_MSGTYPE_STANDARD
	config_physical.nai.protocol = PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL
	config_physical.nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL
	config_physical.type = PUDS_MSGTYPE_USDT
	config_physical.nai.source_addr = PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT
	config_physical.nai.target_addr = PUDS_ISO_15765_4_ADDR_ECU_1
	config_physical.nai.extension_addr = 0

	# Execute TesterPresent and wait response
	msg_request = uds_msg()
	response = uds_msg()
	request_confirmation = uds_msg()
	for i in range(1, 11, 1):
		status = objPCANUds.SvcTesterPresent_2013(client_handle, config_physical, msg_request, objPCANUds.PUDS_SVC_PARAM_TP_ZSUBF)
		print("[UDS] Execute TesterPresent service (%d): %s" %(i, UDS_STATUS_OK_KO(status)))
		status = objPCANUds.WaitForService_2013(client_handle, msg_request, response, request_confirmation)
		print("[UDS] Wait for service (%d): %s" %(i, UDS_STATUS_OK_KO(status)))
		status = objPCANUds.MsgFree_2013(msg_request)
		print("[UDS] Free request message: %s" %(UDS_STATUS_OK_KO(status)))
		status = objPCANUds.MsgFree_2013(request_confirmation)
		print("[UDS] Free request confirmation message: %s" %(UDS_STATUS_OK_KO(status)))
		status = objPCANUds.MsgFree_2013(response)
		print("[UDS] Free response message: %s" %(UDS_STATUS_OK_KO(status)))
	return 0

def wait_isotp_pong_response(client_handle, repeat_wait_counter):
	"""
	ISOTP: wait an ISOTP "PONG" message
	parameters:
		client_handle: client channel handle
		repeat_wait_counter: maximum number of WaitForSingleObject to do (to avoid infinite loop)
	returns: if we received a PONG message (true) or not (false)
	"""

	# Init variables
	repeat = 0
	get_pong_msg = False

	# Get ISOTP receive event
	if IS_WINDOWS:
		receive_event = c_void_p(windll.kernel32.CreateEventA(None, 0,0,None))
	else:
		receive_event = NULL_HANDLE()
	status = objPCANIsotp.GetValue_2016(client_handle, PCANTP_PARAMETER_RECEIVE_EVENT, receive_event, sizeof(receive_event))
	print("[ISOTP] Get isotp receive event parameter: %s" %(ISOTP_STATUS_OK_KO(status)))

	# Wait a receive event on isotp message
	while not get_pong_msg and repeat < repeat_wait_counter:

		if IS_WINDOWS:
			windows_wait_result = c_uint64(windll.kernel32.WaitForSingleObject(receive_event, 1000))
			wait_result = True if windows_wait_result.value == WAIT_OBJECT_0 else False
		else:
			readable, _, _ = select.select([receive_event.value], [], [], 1)
			wait_result = True if len(readable) > 0 else False
		print("[ISOTP] Wait a receive event from isotp: %s" %(OK_KO(wait_result)))

		# Read ISOTP messages
		rx_msg = cantp_msg()
		status = PCANTP_STATUS_OK
		while not objPCANIsotp.StatusIsOk_2016(status, PCANTP_STATUS_NO_MESSAGE, False):

			status = objPCANIsotp.Read_2016(client_handle, rx_msg, None, PCANTP_MSGTYPE_NONE)

			# Check if we received a "PONG" message
			if not objPCANIsotp.StatusIsOk_2016(status, PCANTP_STATUS_NO_MESSAGE, False):
				print("[ISOTP] Read ISOTP message: %s" %(ISOTP_STATUS_OK_KO(status)))
				msgOk = objPCANIsotp.StatusIsOk_2016(status, PCANTP_STATUS_OK, False) and rx_msg.msgdata.any.contents.length == MSG_SIZE
				if msgOk:
					for i in range(MSG_SIZE):
						if rx_msg.msgdata.any.contents.data[i] != ord(ISOTP_RESPONSE_MSG[i]):
							msgOk = False
							break
				if msgOk:
					print("[ISOTP] Message contains \"%c%c%c%c\": OK" %(rx_msg.msgdata.any.contents.data[0], rx_msg.msgdata.any.contents.data[1], rx_msg.msgdata.any.contents.data[2], rx_msg.msgdata.any.contents.data[3]))
					get_pong_msg = True

				status = objPCANIsotp.MsgDataFree_2016(rx_msg)
				print("[ISOTP] Free RX message: %s" %(ISOTP_STATUS_OK_KO(status)))

		repeat += 1

	return get_pong_msg


def isotp_client_task(param):
	"""
	ISOTP client task: send "PING" and wait "PONG" message several times

	parameters:
		param: client channel handle
	"""

	# Initialize variables
	client_handle = param

	# Create a isotp receive event
	if IS_WINDOWS:
		receive_event = c_void_p(windll.kernel32.CreateEventA(None, 0,0,None))
		status = objPCANIsotp.SetValue_2016(client_handle, PCANTP_PARAMETER_RECEIVE_EVENT, receive_event, sizeof(receive_event))
		print("[ISOTP] Set isotp receive event parameter: %s" %(ISOTP_STATUS_OK_KO(status)))
	else:
		receive_event = NULL_HANDLE()
		status = objPCANIsotp.GetValue_2016(client_handle, PCANTP_PARAMETER_RECEIVE_EVENT, receive_event, sizeof(receive_event))
		print("[ISOTP] Get isotp receive event parameter: %s" %(ISOTP_STATUS_OK_KO(status)))

	# Create a simple isotp physical mapping:
	#    - Source 0xA1 (client), target 0xA2 (server), CAN id 0xA1, CAN ID flow control 0xA2
	#    - Diagnostic message in a classic format
	mapping = cantp_mapping()
	mapping.can_id = 0xA1
	mapping.can_id_flow_ctrl = 0xA2
	mapping.can_msgtype = PCANTP_CAN_MSGTYPE_STANDARD
	mapping.netaddrinfo.extension_addr = 0x00
	mapping.netaddrinfo.format = PCANTP_ISOTP_FORMAT_NORMAL
	mapping.netaddrinfo.msgtype = PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC
	mapping.netaddrinfo.source_addr = 0xA1
	mapping.netaddrinfo.target_addr = 0xA2
	mapping.netaddrinfo.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL

	# Create the associated isotp reversed mapping:
	reverse_mapping = cantp_mapping()
	reverse_mapping.can_id = mapping.can_id_flow_ctrl
	reverse_mapping.can_id_flow_ctrl = mapping.can_id
	reverse_mapping.can_msgtype = mapping.can_msgtype
	reverse_mapping.netaddrinfo = copy.deepcopy(mapping.netaddrinfo)
	reverse_mapping.netaddrinfo.source_addr = mapping.netaddrinfo.target_addr
	reverse_mapping.netaddrinfo.target_addr = mapping.netaddrinfo.source_addr

	# Add ISOTP mappings on channels
	status = objPCANIsotp.AddMapping_2016(client_handle, mapping)
	print("[ISOTP] Add a simple isotp mapping: %s" %(ISOTP_STATUS_OK_KO(status)))
	status = objPCANIsotp.AddMapping_2016(client_handle, reverse_mapping)
	print("[ISOTP] Add the reverse isotp mapping: %s" %(ISOTP_STATUS_OK_KO(status)))

	# Initialize ISOTP Tx message containing "PING"
	tx_msg = cantp_msg()
	status = objPCANIsotp.MsgDataAlloc_2016(tx_msg, PCANTP_MSGTYPE_ISOTP)
	print("[ISOTP] Allocate ISOTP tx message: %s" %(ISOTP_STATUS_OK_KO(status)))
	status = objPCANIsotp.MsgDataInit_2016(tx_msg, mapping.can_id, mapping.can_msgtype, MSG_SIZE, ISOTP_REQUEST_MSG, mapping.netaddrinfo)
	print("[ISOTP] Initialize ISOTP tx message: %s" %(ISOTP_STATUS_OK_KO(status)))

	# Send "PING" ISOTP message and wait "PONG" response
	for i in range(1, 11, 1):
		status = objPCANIsotp.Write_2016(client_handle, tx_msg)
		print("[ISOTP] Send ISOTP \"PING\" message (%d): %s" %(i, ISOTP_STATUS_OK_KO(status)))
		response = wait_isotp_pong_response(client_handle, 3)
		print("[ISOTP] Get ISOTP \"PONG\" response (%d): %s" %(i, OK_KO(response)))

	# Close receive event
	if IS_WINDOWS:
		status = objPCANIsotp.SetValue_2016(client_handle, PCANTP_PARAMETER_RECEIVE_EVENT, NULL_HANDLE(), sizeof(NULL_HANDLE()))
		print("[ISOTP] Stop ISOTP receive event: %s" %(ISOTP_STATUS_OK_KO(status)))
		boolean_status = windll.kernel32.CloseHandle(receive_event)
		print("[ISOTP] Close ISOTP receive event: %s" %(OK_KO(boolean_status)))

	# Free messages
	status = objPCANIsotp.MsgDataFree_2016(tx_msg)
	print("[ISOTP] Free ISOTP TX message: %s" %(ISOTP_STATUS_OK_KO(status)))

	return 0

# Definitions
BUFFER_SIZE = 256
ISOTP_REQUEST_MSG = create_string_buffer(b'PING')
ISOTP_RESPONSE_MSG = create_string_buffer(b'PONG')
MSG_SIZE = 4

#
# Main entry point of the program, start a UDS channel, ask in the same time UDS TesterPresent and isotp request
#

# Initialize variables
client_handle = PCANTP_HANDLE_USBBUS1 # TODO: modify the value according to your available PCAN devices.

# Print version informations
buffer = create_string_buffer(BUFFER_SIZE)
status = objPCANUds.GetValue_2013(PCANTP_HANDLE_NONEBUS, PUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE)
print("PCAN-UDS API Version - %s: %s" %(buffer.value, UDS_STATUS_OK_KO(status)))

# Initialize client channel
status = objPCANUds.Initialize_2013(client_handle, PCANTP_BAUDRATE_500K, 0, 0, 0)
print("Initialize channel: %s" %(UDS_STATUS_OK_KO(status)))

# Start uds and isotp clients
uds_client = threading.Thread(target = uds_client_task, args = (client_handle,))
uds_client.start()
isotp_client = threading.Thread(target = isotp_client_task, args = (client_handle,))
isotp_client.start()

uds_client.join()
isotp_client.join()

# Close channel
status = objPCANUds.Uninitialize_2013(client_handle)
print("Uninitialize channel: %s" %(UDS_STATUS_OK_KO(status)))

# Exit
print("Press any key to continue...")
getInput()


