from PCAN_UDS_2013 import *

IS_WINDOWS = platform.system() == 'Windows'

# Support events
if not IS_WINDOWS:
	import select
	__LIBC_OBJ = cdll.LoadLibrary("libc.so.6")

WAIT_OBJECT_0 = 0
def NULL_HANDLE():
	return c_void_p(0) if IS_WINDOWS else c_int(0)

# Support keyboard
if IS_WINDOWS:
	from msvcrt import getch, kbhit
	def install_keyboardHit():
		return 0
	def desinstall_keyboardHit(old_settings):
		pass
	def keyboardHit():
		return kbhit() != 0
	def getInput():
		key = getch()
		try:
			key_decode = key.decode('utf-8')
			return key_decode
		except UnicodeDecodeError:
			return key
else:
	import tty
	import termios
	def install_keyboardHit():
		old_settings = termios.tcgetattr(sys.stdin)
		tty.setcbreak(sys.stdin.fileno())
		return old_settings
	def desinstall_keyboardHit(old_settings):
		termios.tcsetattr(sys.stdin, termios.TCSADRAIN, old_settings)
	def keyboardHit():
		return select.select([sys.stdin], [], [], 0) == ([sys.stdin], [], [])
	def getInput():
		return sys.stdin.read(1)

# UDS library
objPCANUds = PCAN_UDS_2013()

# Help functions
def OK_KO(test):
	return "OK" if test else "KO"

def STATUS_OK_KO(test):
	return OK_KO(objPCANUds.StatusIsOk_2013(test, PUDS_STATUS_OK, False))

# Definitions
PCAN_BITRATE = create_string_buffer(b'f_clock=40000000,nom_brp=2,nom_tseg1=63,nom_tseg2=16,nom_sjw=16,data_brp=2,data_tseg1=7,data_tseg2=2,data_sjw=2')

#
# Main entry point of the program, start a small server wich only support ReadDataByPeriodicIdentifier service.
# This example use a specific addressing. It receives request from test equipement (0xF1 to 0xC1) in 29b fixed
# normal addressing and sends responses for each periodic data identifier with 0x1F22C1F1 can identifier (UUDT).
#

try:
	old_settings = install_keyboardHit()

	# Initialize variables
	server_handle = PCANTP_HANDLE_USBBUS2 # TODO: modify the value according to your available PCAN devices.
	server_address = c_uint16(0xC1)

	# Initialize server
	status = objPCANUds.InitializeFD_2013(server_handle, PCAN_BITRATE)
	print("Initialize channel: %s" %(STATUS_OK_KO(status)))

	# Set server address parameter
	status = objPCANUds.SetValue_2013(server_handle, PUDS_PARAMETER_SERVER_ADDRESS, server_address, sizeof(server_address))
	print("Set server address: %s" %(STATUS_OK_KO(status)))

	# Set a padding value
	padding_value = c_ubyte(0xFF)
	status = objPCANUds.SetValue_2013(server_handle, PUDS_PARAMETER_CAN_PADDING_VALUE, padding_value, sizeof(padding_value))
	print("Set padding value: %s" %(STATUS_OK_KO(status)))

	# Define CAN_TX_DL=15
	can_tx_dl = c_ubyte(15)
	status = objPCANUds.SetValue_2013(server_handle, PUDS_PARAMETER_CAN_TX_DL, can_tx_dl, sizeof(can_tx_dl))
	print("Set CAN TX DL: %s" %(STATUS_OK_KO(status)))

	# Set UDS timeouts
	timeout_value = c_uint32(5000)
	status = objPCANUds.SetValue_2013(server_handle, PUDS_PARAMETER_TIMEOUT_REQUEST, timeout_value, sizeof(timeout_value))
	print("Set request timeout(ms): %s" %(STATUS_OK_KO(status)))
	status = objPCANUds.SetValue_2013(server_handle, PUDS_PARAMETER_TIMEOUT_RESPONSE, timeout_value, sizeof(timeout_value))
	print("Set response timeout(ms): %s" %(STATUS_OK_KO(status)))

	# Set a receive event
	if IS_WINDOWS:
		receive_event = c_void_p(windll.kernel32.CreateEventA(None, 0,0,None))
		res = objPCANUds.SetValue_2013(server_handle, PUDS_PARAMETER_RECEIVE_EVENT, receive_event, sizeof(receive_event))
		print("Set receive event parameter: %s" %( STATUS_OK_KO(res)))
	else:
		receive_event = NULL_HANDLE()
		res = objPCANUds.GetValue_2013(server_handle, PUDS_PARAMETER_RECEIVE_EVENT, receive_event, sizeof(receive_event))
		print("Get receive event parameter: %s" %( STATUS_OK_KO(res)))

	# Initialize service response configuration
	service_response_config = uds_msgconfig()
	service_response_config.can_id = -1
	service_response_config.can_msgtype = cantp_can_msgtype(PCANTP_CAN_MSGTYPE_EXTENDED.value | PCANTP_CAN_MSGTYPE_FD.value | PCANTP_CAN_MSGTYPE_BRS.value)
	service_response_config.nai.protocol = PUDS_MSGPROTOCOL_ISO_15765_2_29B_FIXED_NORMAL
	service_response_config.nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL
	service_response_config.type = PUDS_MSGTYPE_USDT
	service_response_config.nai.source_addr = server_address
	service_response_config.nai.target_addr = PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT
	service_response_config.nai.extension_addr = 0

	# Initialize responses configuration (for each periodic data identifier contained in the request)
	periodic_msg_config = uds_msgconfig()
	periodic_msg_config.can_id = 0x1F22C1F1
	periodic_msg_config.can_msgtype = cantp_can_msgtype(PCANTP_CAN_MSGTYPE_EXTENDED.value | PCANTP_CAN_MSGTYPE_FD.value | PCANTP_CAN_MSGTYPE_BRS.value)
	periodic_msg_config.nai.protocol = PUDS_MSGPROTOCOL_ISO_15765_2_29B_NORMAL
	periodic_msg_config.nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL
	periodic_msg_config.type = PUDS_MSGTYPE_UUDT
	periodic_msg_config.nai.source_addr = server_address
	periodic_msg_config.nai.target_addr = PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT
	periodic_msg_config.nai.extension_addr = 0

	# Add a filter for 0x1F22C1F1 can id (in order to receive UUDT loopback messages)
	status = objPCANUds.AddCanIdFilter_2013(server_handle, periodic_msg_config.can_id)
	print("Add can identifier filter: %s" %(STATUS_OK_KO(status)))

	# Read while user do not press Q
	print("Start listening, press Q to quit.")
	stop = False
	while not stop:

		# Wait a receive event on receiver
		#	note: timeout is used to check keyboard hit.
		if IS_WINDOWS:
			windows_wait_result = c_uint64(windll.kernel32.WaitForSingleObject(receive_event, 1000))
			wait_result = True if windows_wait_result.value == WAIT_OBJECT_0 else False
		else:
			readable, _, _ = select.select([receive_event.value], [], [], 3)
			wait_result = True if len(readable) > 0 else False

		# If we get a receive event
		if wait_result:
			request_msg = uds_msg()
			service_response_msg = uds_msg()
			read_status = PUDS_STATUS_OK
			while not objPCANUds.StatusIsOk_2013(read_status, PUDS_STATUS_NO_MESSAGE, False):

				# Read first available message (no filtering based on message's type is set):
				read_status = objPCANUds.Read_2013(server_handle, request_msg, None, None)
				print("Try to read a message: %s" %(STATUS_OK_KO(read_status)))
				if objPCANUds.StatusIsOk_2013(read_status, PUDS_STATUS_OK, False):

					# We receive a request, check its length and if it is not a loopback message
					if request_msg.msg.msgdata.any.contents.length >= 1 and (request_msg.msg.msgdata.any.contents.flags & PCANTP_MSGFLAG_LOOPBACK.value) == 0:

						# This is a valid request, switch services
						service_id = request_msg.links.service_id[0]
						if service_id == objPCANUds.PUDS_SI_ReadDataByPeriodicIdentifier:

							# Allocates service response message
							status = objPCANUds.MsgAlloc_2013(service_response_msg, service_response_config, 1)
							if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
								service_response_msg.links.service_id[0] = c_ubyte(objPCANUds.PUDS_SI_ReadDataByPeriodicIdentifier + objPCANUds.PUDS_SI_POSITIVE_RESPONSE)
							print("Prepare response message for ReadDataByPeriodicIdentifier service: %s" %(STATUS_OK_KO(status)))

							# Write service response message
							status = objPCANUds.Write_2013(server_handle, service_response_msg)
							print("Write response message for ReadDataByPeriodicIdentifier service: %s" %(STATUS_OK_KO(status)))

							# Free response message (and clean memory in order to reallocate later)
							status = objPCANUds.MsgFree_2013(service_response_msg)
							print("Free response message: %s" %(STATUS_OK_KO(status)))

							# Sends a message for each data identifier with a specific addressing.
							periodic_data_identifier_length = request_msg.msg.msgdata.any.contents.length - 2
							for i in range(periodic_data_identifier_length):

								# Allocates and prepares message with dummy data
								periodic_response = uds_msg()
								status = objPCANUds.MsgAlloc_2013(periodic_response, periodic_msg_config, 5)
								if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False):
									periodic_data_identifier = request_msg.links.param[1 + i]
									periodic_response.msg.msgdata.any.contents.data[0] = periodic_data_identifier
									periodic_response.msg.msgdata.any.contents.data[1] = 0x12
									periodic_response.msg.msgdata.any.contents.data[2] = 0x34
									periodic_response.msg.msgdata.any.contents.data[3] = 0x56
									periodic_response.msg.msgdata.any.contents.data[4] = 0x78

									print("Allocates message for 0x%02x periodic data identifier: %s" %(periodic_data_identifier, STATUS_OK_KO(status)))
									status = objPCANUds.Write_2013(server_handle, periodic_response)
									print("Write message for 0x%02x periodic data identifier: %s" %(periodic_data_identifier, STATUS_OK_KO(status)))
									status = objPCANUds.MsgFree_2013(periodic_response)
									print("Free message for 0x%02x periodic data identifier: %s" %(periodic_data_identifier, STATUS_OK_KO(status)))
						else:
							print("Unknown service (0x%02x)" %(service_id))

				# Free request message (in order to reallocate later)
				status = objPCANUds.MsgFree_2013(request_msg)
				print("Free request message: %s" %(STATUS_OK_KO(status)))

		# Quit when user press Q
		if keyboardHit():
			keyboard_res = getInput()
			if keyboard_res == 'Q' or keyboard_res == 'q':
				stop = True


	# Close receive event
	if IS_WINDOWS:
		status = objPCANUds.SetValue_2013(server_handle, PUDS_PARAMETER_RECEIVE_EVENT, NULL_HANDLE(), sizeof(NULL_HANDLE()))
		print("Stop receive event: %s" %(STATUS_OK_KO(status)))
		boolean_status = windll.kernel32.CloseHandle(receive_event)
		print("Close receive event: %s" %(OK_KO(boolean_status)))

	# Close server
	status = objPCANUds.Uninitialize_2013(server_handle)
	print("Uninitialize channel: %s" %(STATUS_OK_KO(status)))

	# Exit
	print("Press any key to continue...")
	getInput()

finally:
	desinstall_keyboardHit(old_settings)
