import time
from PCAN_UDS_2013 import *

IS_WINDOWS = platform.system() == 'Windows'

# Support keyboard
def getInput():
	sys.stdin.read(1)

# UDS library
objPCANUds = PCAN_UDS_2013()

# Help functions
def OK_KO(test):
	return "OK" if test else "KO"

def STATUS_OK_KO(test):
	return OK_KO(objPCANUds.StatusIsOk_2013(test, PUDS_STATUS_OK, False))

def getTimeIndicationInMilliseconds():
	"""
	Get a time indication in milliseconds
	"""
	return time.time()*1000

# Definitions
PCAN_BITRATE = create_string_buffer(b'f_clock=40000000,nom_brp=2,nom_tseg1=63,nom_tseg2=16,nom_sjw=16,data_brp=2,data_tseg1=7,data_tseg2=2,data_sjw=2')
RESPONSES_TIMEOUT_MS = 5000

#
# Main entry point of the program, start a UDS channel, ask ReadDataByPeriodicIdentifier service.
# It sends a request from 0xF1 to 0xC1 address in 29b fixed with normal addressing. Then read
# responses from 0x1F22C1F1 can identifier (UUDT).
#

# Initialize variables
client_handle = PCANTP_HANDLE_USBBUS1 # TODO: modify the value according to your available PCAN devices.

# Initialize client
status = objPCANUds.InitializeFD_2013(client_handle, PCAN_BITRATE)
print("Initialize channel: %s" %(STATUS_OK_KO(status)))

# Define CAN_TX_DL
can_tx_dl = c_ubyte(15)
status = objPCANUds.SetValue_2013(client_handle, PUDS_PARAMETER_CAN_TX_DL, can_tx_dl, sizeof(can_tx_dl))
print("Set CAN TX DL: %s" %(STATUS_OK_KO(status)))

# Check CAN_TX_DL
can_tx_dl = c_ubyte(0)
status = objPCANUds.GetValue_2013(client_handle, PUDS_PARAMETER_CAN_TX_DL, can_tx_dl, sizeof(can_tx_dl))
print("Check new CAN TX DL value(%u): %s" %(can_tx_dl.value, STATUS_OK_KO(status)))

# Set UDS timeouts
timeout_value = c_uint32(5000)
status = objPCANUds.SetValue_2013(client_handle, PUDS_PARAMETER_TIMEOUT_REQUEST, timeout_value, sizeof(timeout_value))
print("Set request timeout(ms): %s" %(STATUS_OK_KO(status)))
status = objPCANUds.SetValue_2013(client_handle, PUDS_PARAMETER_TIMEOUT_RESPONSE, timeout_value, sizeof(timeout_value))
print("Set response timeout(ms): %s" %(STATUS_OK_KO(status)))

# Check timeouts
timeout_value = c_uint32(0)
status = objPCANUds.GetValue_2013(client_handle, PUDS_PARAMETER_TIMEOUT_REQUEST, timeout_value, sizeof(timeout_value))
print("Get new request timeout(%ums): %s" %(timeout_value.value, STATUS_OK_KO(status)))
timeout_value = c_uint32(0)
status = objPCANUds.GetValue_2013(client_handle, PUDS_PARAMETER_TIMEOUT_RESPONSE, timeout_value, sizeof(timeout_value))
print("Get new response timeout(%ums): %s" %(timeout_value.value, STATUS_OK_KO(status)))

# Initialize the request configuration
request_config = uds_msgconfig()
request_config.can_id = -1
request_config.can_msgtype =cantp_can_msgtype(PCANTP_CAN_MSGTYPE_EXTENDED.value | PCANTP_CAN_MSGTYPE_FD.value | PCANTP_CAN_MSGTYPE_BRS.value)
request_config.nai.protocol = PUDS_MSGPROTOCOL_ISO_15765_2_29B_FIXED_NORMAL
request_config.nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL
request_config.type = PUDS_MSGTYPE_USDT
request_config.nai.source_addr = PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT
request_config.nai.target_addr = 0xC1
request_config.nai.extension_addr = 0

# Add a filter for 0x1F22C1F1 can id (in order to receive UUDT messages)
status = objPCANUds.AddCanIdFilter_2013(client_handle, 0x1F22C1F1)
print("Add can identifier filter: %s" %(STATUS_OK_KO(status)))

# Execute ReadDataByPeriodicIdentifier and wait service response
msg_request = uds_msg()
service_response = uds_msg()
request_confirmation = uds_msg()
periodic_data_identifier = create_string_buffer(b'\x12\x34\x56\x78')
status = objPCANUds.SvcReadDataByPeriodicIdentifier_2013(client_handle, request_config, msg_request, objPCANUds.PUDS_SVC_PARAM_RDBPI_SAMR, periodic_data_identifier, 4)
print("Execute ReadDataByPeriodicIdentifier service: %s" %(STATUS_OK_KO(status)))
status = objPCANUds.WaitForService_2013(client_handle, msg_request, service_response, request_confirmation)
print("Wait for service (confirmation and response): %s" %(STATUS_OK_KO(status)))

# Read responses for each given periodic data identifier
periodic_response = uds_msg()
response_count = 0
start_reading = getTimeIndicationInMilliseconds()
while response_count < 4 and getTimeIndicationInMilliseconds() - start_reading < RESPONSES_TIMEOUT_MS:
	status = objPCANUds.Read_2013(client_handle, periodic_response, None, None)
	if objPCANUds.StatusIsOk_2013(status, PUDS_STATUS_OK, False) and periodic_response.msg.msgdata.any.contents.length > 0:
		response_count += 1
		s = "Read response for 0x{x1} periodic data identifier: [".format(x1=format(periodic_response.msg.msgdata.any.contents.data[0],'02x'))
		for i in range(periodic_response.msg.msgdata.any.contents.length):
			s += " 0x{x1}".format(x1=format(periodic_response.msg.msgdata.any.contents.data[i], '02x'))
		s += " ]"
		print(s)
		status = objPCANUds.MsgFree_2013(periodic_response)
		print("Free response buffer: %s" %(STATUS_OK_KO(status)))

	# Wait for 4 responses or timeout

# Free message structures
status = objPCANUds.MsgFree_2013(msg_request)
print("Free request message: %s" %(STATUS_OK_KO(status)))
status = objPCANUds.MsgFree_2013(request_confirmation)
print("Free request confirmation: %s" %(STATUS_OK_KO(status)))
status = objPCANUds.MsgFree_2013(service_response)
print("Free service response: %s" %(STATUS_OK_KO(status)))

# Close client
status = objPCANUds.Uninitialize_2013(client_handle)
print("Uninitialize channel: %s" %(STATUS_OK_KO(status)))

# Exit
print("Press any key to continue...")
getInput()

