from time import sleep
import threading
from PCAN_UDS_2013 import *

IS_WINDOWS = platform.system() == 'Windows'

# Support events
if not IS_WINDOWS:
	import select
	__LIBC_OBJ = cdll.LoadLibrary("libc.so.6")

WAIT_OBJECT_0 = 0
def NULL_HANDLE():
	return c_void_p(0) if IS_WINDOWS else c_int(0)

# Support keyboard
def getInput():
	sys.stdin.read(1)

# UDS and Isotp libraries
objPCANUds = PCAN_UDS_2013()
objPCANIsotp = PCAN_ISO_TP_2016()

# Help functions
def OK_KO(test):
	return "OK" if test else "KO"

def UDS_STATUS_OK_KO(test):
	return OK_KO(objPCANUds.StatusIsOk_2013(test, PUDS_STATUS_OK, False))

def ISOTP_STATUS_OK_KO(test):
	return OK_KO(objPCANIsotp.StatusIsOk_2016(test, PCANTP_STATUS_OK, False))

class task_params (Structure):
	"""
	Structure passed as thread parameters
	"""
	_fields_ = [("client_handle", cantp_handle),	# Client channel handle
				("stop_task", c_bool)]				# Determine if the thread should end or not

def uds_client_task(parameters):
	"""
	UDS client task: request TesterPresent service several time

	parameters:
		parameters: pointer on task_params structures
	"""

	# Initialize a physical configuration
	config_physical = uds_msgconfig()
	config_physical.can_id = PUDS_ISO_15765_4_CAN_ID_PHYSICAL_REQUEST_1
	config_physical.can_msgtype = PCANTP_CAN_MSGTYPE_STANDARD
	config_physical.nai.protocol = PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL
	config_physical.nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL
	config_physical.type = PUDS_MSGTYPE_USDT
	config_physical.nai.source_addr = PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT
	config_physical.nai.target_addr = PUDS_ISO_15765_4_ADDR_ECU_1
	config_physical.nai.extension_addr = 0

	# Execute TesterPresent and wait response
	msg_request = uds_msg()
	response = uds_msg()
	request_confirmation = uds_msg()
	while not parameters.stop_task:
		# Wait before transmitting the next message
		sleep(UDS_REQUEST_TIMING_MS/1000.0)

		status = objPCANUds.SvcTesterPresent_2013(parameters.client_handle, config_physical, msg_request, objPCANUds.PUDS_SVC_PARAM_TP_ZSUBF)
		print("[UDS] Execute TesterPresent service: %s" %(UDS_STATUS_OK_KO(status)))
		status = objPCANUds.WaitForService_2013(parameters.client_handle, msg_request, response, request_confirmation)
		print("[UDS] Wait for service: %s" %(UDS_STATUS_OK_KO(status)))
		status = objPCANUds.MsgFree_2013(msg_request)
		print("[UDS] Free request message: %s" %(UDS_STATUS_OK_KO(status)))
		status = objPCANUds.MsgFree_2013(request_confirmation)
		print("[UDS] Free request confirmation message: %s" %(UDS_STATUS_OK_KO(status)))
		status = objPCANUds.MsgFree_2013(response)
		print("[UDS] Free response message: %s" %(UDS_STATUS_OK_KO(status)))

	return 0

def isotp_client_task(parameters):
	"""
	ISOTP client task: read and print isotp messages

	parameters
		parameters: pointer on task_params structures
	"""

	# Configure isotp to get any messages (disable can identifier filtering)
	isotp_param = c_ubyte(0)
	status = objPCANIsotp.SetValue_2016(parameters.client_handle, PCANTP_PARAMETER_FILTER_CAN_ID, isotp_param, sizeof(isotp_param))
	print("[ISOTP] Disable can identifier filtering in isotp: %s" %(ISOTP_STATUS_OK_KO(status)))

	# Configure isotp to get a copy of UDS messages
	isotp_param = c_ubyte(1)
	status = objPCANIsotp.SetValue_2016(parameters.client_handle, PCANTP_PARAMETER_KEEP_HIGHER_LAYER_MESSAGES, isotp_param, sizeof(isotp_param))
	print("[ISOTP] Activate higher layer messages in isotp: %s" %(ISOTP_STATUS_OK_KO(status)))

	# Create a isotp receive event
	if IS_WINDOWS:
		receive_event = c_void_p(windll.kernel32.CreateEventA(None, 0,0,None))
		status = objPCANIsotp.SetValue_2016(parameters.client_handle, PCANTP_PARAMETER_RECEIVE_EVENT, receive_event, sizeof(receive_event))
		print("[ISOTP] Set isotp receive event parameter: %s" %(ISOTP_STATUS_OK_KO(status)))
	else:
		receive_event = NULL_HANDLE()
		status = objPCANIsotp.GetValue_2016(parameters.client_handle, PCANTP_PARAMETER_RECEIVE_EVENT, receive_event, sizeof(receive_event))
		print("[ISOTP] Get isotp receive event parameter: %s" %(ISOTP_STATUS_OK_KO(status)))

	# Read and print ISOTP messages
	rx_msg = cantp_msg()
	while not parameters.stop_task:
		if IS_WINDOWS:
			windows_wait_result = c_uint64(windll.kernel32.WaitForSingleObject(receive_event, 1000))
			wait_result = True if windows_wait_result.value == WAIT_OBJECT_0 else False
		else:
			readable, _, _ = select.select([receive_event.value], [], [], 1)
			wait_result = True if len(readable) > 0 else False
		print("[ISOTP] Wait a receive event from isotp: %s" %(OK_KO(wait_result)))

		# Read ISOTP messages
		status = PCANTP_STATUS_OK
		while not objPCANIsotp.StatusIsOk_2016(status, PCANTP_STATUS_NO_MESSAGE, False):
			status = objPCANIsotp.Read_2016(parameters.client_handle, rx_msg, None, PCANTP_MSGTYPE_NONE)

			# Check if we received a message
			if not objPCANIsotp.StatusIsOk_2016(status, PCANTP_STATUS_NO_MESSAGE, False):
				print("[ISOTP] Read ISOTP message: %s" %(ISOTP_STATUS_OK_KO(status)))
				if rx_msg.msgdata.any.contents.length > 0 and\
					(rx_msg.msgdata.any.contents.data[0] == objPCANUds.PUDS_SI_TesterPresent or rx_msg.msgdata.any.contents.data[0] == objPCANUds.PUDS_SI_TesterPresent + objPCANUds.PUDS_SI_POSITIVE_RESPONSE):
					# This message is a TesterPresent message, PCANTP_PARAMETER_KEEP_HIGHER_LAYER_MESSAGES
					# option must be activated to get these messages.
					print("[ISOTP] Message is a TesterPresent service message.")
				elif objPCANIsotp.StatusIsOk_2016(status, PCANTP_STATUS_OK, False) and rx_msg.msgdata.any.contents.length < BUFFER_SIZE:
					# This message is a CAN message received by ISOTP
					buffer = create_string_buffer(rx_msg.msgdata.any.contents.length)
					for i in range(rx_msg.msgdata.any.contents.length):
						buffer[i] = rx_msg.msgdata.any.contents.data[i] if sys.version_info.major >= 3 else chr(rx_msg.msgdata.any.contents.data[i])
					print("[ISOTP] Message from 0x%x can identifier contains \"%s\"" %(rx_msg.can_info.can_id, buffer.value))

				status = objPCANIsotp.MsgDataFree_2016(rx_msg)
				print("[ISOTP] Free RX message: %s" %(ISOTP_STATUS_OK_KO(status)))

	# Close receive event
	if IS_WINDOWS:
		status = objPCANIsotp.SetValue_2016(parameters.client_handle, PCANTP_PARAMETER_RECEIVE_EVENT, NULL_HANDLE(), sizeof(NULL_HANDLE()))
		print("[ISOTP] Stop ISOTP receive event: %s" %(ISOTP_STATUS_OK_KO(status)))
		boolean_status = windll.kernel32.CloseHandle(receive_event)
		print("[ISOTP] Close ISOTP receive event: %s" %(OK_KO(boolean_status)))

	return 0

# Definitions
BUFFER_SIZE =256
UDS_REQUEST_TIMING_MS = 500
CLIENT_EXECUTION_TIME_MS = 10000

#
# Main entry point of the program, start a UDS channel, ask in the same time UDS TesterPresent and isotp request
#

# Initialize variables
client_handle = PCANTP_HANDLE_USBBUS1 # TODO: modify the value according to your available PCAN devices.
t_params = task_params()
t_params.client_handle = client_handle
t_params.stop_task = False

# Print version informations
buffer = create_string_buffer(BUFFER_SIZE)
status = objPCANUds.GetValue_2013(PCANTP_HANDLE_NONEBUS, PUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE)
print("PCAN-UDS API Version - %s: %s" %(buffer.value, UDS_STATUS_OK_KO(status)))

# Initialize client channel
status = objPCANUds.Initialize_2013(client_handle, PCANTP_BAUDRATE_500K, 0, 0, 0)
print("Initialize channel: %s" %(UDS_STATUS_OK_KO(status)))

# Start uds and isotp clients
uds_client = threading.Thread(target = uds_client_task, args = (t_params,))
uds_client.start()
isotp_client = threading.Thread(target = isotp_client_task, args = (t_params,))
isotp_client.start()

sleep(CLIENT_EXECUTION_TIME_MS/1000.0)
t_params.stop_task = True
uds_client.join()
isotp_client.join()

# Close channel
status = objPCANUds.Uninitialize_2013(client_handle)
print("Uninitialize channel: %s" %(UDS_STATUS_OK_KO(status)))

# Exit
print("Press any key to continue...")
getInput()


