﻿Imports System
Imports System.Text
Imports System.Runtime.InteropServices
Imports System.Threading

Imports Peak.Can.Basic
Imports Peak.Can.IsoTp
Imports Peak.Can.Uds


Module Module1
    Function OK_KO(ByVal test As Boolean) As String
        If test Then Return "OK"
        Return "KO"
    End Function

    Function STATUS_OK_KO(ByVal test As uds_status) As String
        Return OK_KO(UDSApi.StatusIsOk_2013(test))
    End Function

    ''' <summary>Helper: convert cantp network address information into uds network information</summary>
    ''' <param name="nai">cantp network address information structure</param>
    ''' <param name="can_msgtype">cantp message type</param>
    ''' <returns>UDS network address information structure</returns>
    Function ConvertNai(ByVal nai As cantp_netaddrinfo, ByVal can_msgtype As cantp_can_msgtype) As uds_netaddrinfo
        Dim result As uds_netaddrinfo
        Dim is_29bits As Boolean = (can_msgtype And cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED) = cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED
        result.source_addr = nai.source_addr
        result.target_addr = nai.target_addr
        result.target_type = nai.target_type
        result.extension_addr = nai.extension_addr

        Select Case nai.format
            Case cantp_isotp_format.PCANTP_ISOTP_FORMAT_ENHANCED
                result.protocol = uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_3_29B_ENHANCED
            Case cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED
                result.protocol = If(is_29bits, uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_29B_EXTENDED, uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_11B_EXTENDED)
            Case cantp_isotp_format.PCANTP_ISOTP_FORMAT_FIXED_NORMAL
                result.protocol = uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_29B_FIXED_NORMAL
            Case cantp_isotp_format.PCANTP_ISOTP_FORMAT_MIXED
                result.protocol = If(is_29bits, uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_29B_REMOTE, uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_11B_REMOTE)
            Case cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL
                result.protocol = If(is_29bits, uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_29B_NORMAL, uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL)
            Case Else
                result.protocol = uds_msgprotocol.PUDS_MSGPROTOCOL_NONE
        End Select

        Return result
    End Function

    ''' <summary>Helper: check if a service has a subfunction</summary>
    ''' <param name="service_identifier">service identifier</param>
    ''' <returns>boolean: yes (true) or not (false)</returns>
    Function HasSubFunction(ByVal service_identifier As Byte) As Boolean
        Dim res As Boolean

        Select Case service_identifier
            Case CByte(uds_service.PUDS_SERVICE_SI_DiagnosticSessionControl)
                res = True
            Case CByte(uds_service.PUDS_SERVICE_SI_ECUReset)
                res = True
            Case CByte(uds_service.PUDS_SERVICE_SI_SecurityAccess)
                res = True
            Case CByte(uds_service.PUDS_SERVICE_SI_CommunicationControl)
                res = True
            Case CByte(uds_service.PUDS_SERVICE_SI_TesterPresent)
                res = True
            Case CByte(uds_service.PUDS_SERVICE_SI_AccessTimingParameter)
                res = True
            Case CByte(uds_service.PUDS_SERVICE_SI_SecuredDataTransmission)
                res = False
            Case CByte(uds_service.PUDS_SERVICE_SI_ControlDTCSetting)
                res = True
            Case CByte(uds_service.PUDS_SERVICE_SI_ResponseOnEvent)
                res = True
            Case CByte(uds_service.PUDS_SERVICE_SI_LinkControl)
                res = True
            Case CByte(uds_service.PUDS_SERVICE_SI_ReadDataByIdentifier)
                res = False
            Case CByte(uds_service.PUDS_SERVICE_SI_ReadMemoryByAddress)
                res = False
            Case CByte(uds_service.PUDS_SERVICE_SI_ReadScalingDataByIdentifier)
                res = False
            Case CByte(uds_service.PUDS_SERVICE_SI_ReadDataByPeriodicIdentifier)
                res = False
            Case CByte(uds_service.PUDS_SERVICE_SI_DynamicallyDefineDataIdentifier)
                res = True
            Case CByte(uds_service.PUDS_SERVICE_SI_WriteDataByIdentifier)
                res = False
            Case CByte(uds_service.PUDS_SERVICE_SI_WriteMemoryByAddress)
                res = False
            Case CByte(uds_service.PUDS_SERVICE_SI_ClearDiagnosticInformation)
                res = False
            Case CByte(uds_service.PUDS_SERVICE_SI_ReadDTCInformation)
                res = True
            Case CByte(uds_service.PUDS_SERVICE_SI_InputOutputControlByIdentifier)
                res = False
            Case CByte(uds_service.PUDS_SERVICE_SI_RoutineControl)
                res = True
            Case CByte(uds_service.PUDS_SERVICE_SI_RequestDownload)
                res = False
            Case CByte(uds_service.PUDS_SERVICE_SI_RequestUpload)
                res = False
            Case CByte(uds_service.PUDS_SERVICE_SI_TransferData)
                res = False
            Case CByte(uds_service.PUDS_SERVICE_SI_RequestTransferExit)
                res = False
            Case CByte(uds_service.PUDS_SERVICE_SI_RequestFileTransfer)
                res = False
            Case CByte(uds_service.PUDS_SERVICE_SI_Authentication)
                res = True
            Case Else
                res = False
        End Select

        Return res
    End Function

    ''' <summary>
    ''' Create the ECUReset response message
    ''' </summary>
    ''' <param name="config">Input, UDS configuration</param>
    ''' <param name="request_msg">Input, received request, containing ECUReset parameters</param>
    ''' <param name="response_msg">Output, contains the generated response</param>
    ''' <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    Function create_SvcECUReset_response(ByVal config As uds_msgconfig, ByRef request_msg As uds_msg, ByRef response_msg As uds_msg) As uds_status
        Dim status As uds_status

        ' Allocate response message
        Dim param0 As Byte
        UDSApi.GetDataParameter_2013(request_msg, 0, param0)

        If param0 = CByte(UDSApi.uds_svc_param_er.PUDS_SVC_PARAM_ER_ERPSD) Then
            status = UDSApi.MsgAlloc_2013(response_msg, config, 3)
            UDSApi.SetDataParameter_2013(response_msg, 1, &H66)    ' Power down time
        Else
            status = UDSApi.MsgAlloc_2013(response_msg, config, 2)
        End If

        Console.WriteLine("Allocate ECUReset response: {0}", STATUS_OK_KO(status))

        If UDSApi.StatusIsOk_2013(status) Then
            ' Fill parameters
            UDSApi.SetDataServiceId_2013(response_msg, CByte(uds_service.PUDS_SERVICE_SI_ECUReset + UDSApi.PUDS_SI_POSITIVE_RESPONSE))
            UDSApi.SetDataParameter_2013(response_msg, 0, param0)
        End If

        Return status
    End Function


    ''' <summary>
    ''' Create the WriteMemoryByAddress response message
    ''' </summary>
    ''' <param name="config">Input, UDS configuration</param>
    ''' <param name="request_msg">Input, received request, containing request parameters</param>
    ''' <param name="response_msg">Output, contains the generated response</param>
    ''' <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    Function create_SvcWriteMemoryByAddress_response(ByVal config As uds_msgconfig, ByRef request_msg As uds_msg, ByRef response_msg As uds_msg) As uds_status
        Dim status As uds_status
        Dim memory_size_size As UInt32
        Dim memory_address_size As UInt32
        Dim length As UInt32

        ' Read memory_size_size & memory_address_size
        Dim param0 As Byte
        UDSApi.GetDataParameter_2013(request_msg, 0, param0)
        memory_size_size = CUInt(param0 >> 4 And &HF)
        memory_address_size = CUInt(param0 And &HF)
        length = 2 + memory_size_size + memory_address_size

        ' Allocate message
        status = UDSApi.MsgAlloc_2013(response_msg, config, length)
        Console.WriteLine("Allocate WriteMemoryByAddress response: {0}", STATUS_OK_KO(status))

        If UDSApi.StatusIsOk_2013(status) Then
            ' Copy address and memory parameters
            UDSApi.SetDataParameter_2013(response_msg, 0, param0)

            For i As Integer = 0 To memory_address_size - 1
                Dim val As Byte
                UDSApi.GetDataParameter_2013(request_msg, 1 + i, val)
                UDSApi.SetDataParameter_2013(response_msg, 1 + i, val)
            Next

            For i As Integer = 0 To memory_size_size - 1
                Dim val As Byte
                UDSApi.GetDataParameter_2013(request_msg, 1 + memory_address_size + i, val)
                UDSApi.SetDataParameter_2013(response_msg, 1 + memory_address_size + i, val)
            Next

            UDSApi.SetDataServiceId_2013(response_msg, CByte(uds_service.PUDS_SERVICE_SI_WriteMemoryByAddress + UDSApi.PUDS_SI_POSITIVE_RESPONSE))
        End If

        Return status
    End Function


    ''' <summary>
    ''' Create the DynamicallyDefineDataIdentifier response message
    ''' </summary>
    ''' <param name="config">Input, UDS configuration</param>
    ''' <param name="request_msg">Input, received request, containing request parameters</param>
    ''' <param name="response_msg">Output, contains the generated response</param>
    ''' <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    Function create_SvcDynamicallyDefineDataIdentifier_response(ByVal config As uds_msgconfig, ByRef request_msg As uds_msg, ByRef response_msg As uds_msg) As uds_status
        Dim status As uds_status

        ' Allocate and fill message
        If request_msg.msg.Msgdata_any_Copy.length >= 4 Then
            status = UDSApi.MsgAlloc_2013(response_msg, config, 4)
            Console.WriteLine("Allocate DynamicallyDefineDataIdentifier response: {0}", STATUS_OK_KO(status))

            If UDSApi.StatusIsOk_2013(status) Then
                ' Copy data
                Dim param0 As Byte
                UDSApi.GetDataParameter_2013(request_msg, 0, param0)
                Dim param1 As Byte
                UDSApi.GetDataParameter_2013(request_msg, 1, param1)
                Dim param2 As Byte
                UDSApi.GetDataParameter_2013(request_msg, 2, param2)
                UDSApi.SetDataParameter_2013(response_msg, 0, param0)
                UDSApi.SetDataParameter_2013(response_msg, 1, param1)
                UDSApi.SetDataParameter_2013(response_msg, 2, param2)
            End If
        Else
            ' Clear all data identifier request
            status = UDSApi.MsgAlloc_2013(response_msg, config, 2)
            Console.WriteLine("Allocate DynamicallyDefineDataIdentifier response: {0}", STATUS_OK_KO(status))

            If UDSApi.StatusIsOk_2013(status) Then
                ' Copy subfunction
                Dim param0 As Byte
                UDSApi.GetDataParameter_2013(request_msg, 0, param0)
                UDSApi.SetDataParameter_2013(response_msg, 0, param0)
            End If
        End If

        If UDSApi.StatusIsOk_2013(status) Then
            UDSApi.SetDataServiceId_2013(response_msg, CByte(uds_service.PUDS_SERVICE_SI_DynamicallyDefineDataIdentifier + UDSApi.PUDS_SI_POSITIVE_RESPONSE))
        End If

        Return status
    End Function


    ''' <summary>
    ''' Create the dummy response message
    ''' </summary>
    ''' <param name="config">Input, UDS configuration</param>
    ''' <param name="request_msg">Input, received request, containing request parameters</param>
    ''' <param name="response_msg">Output, contains the generated response</param>
    ''' <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    Function create_dummy_response(ByVal config As uds_msgconfig, ByRef request_msg As uds_msg, ByRef response_msg As uds_msg) As uds_status
        Dim status As uds_status
        Dim length As UInt32
        Dim service_id As Byte
        UDSApi.GetDataServiceId_2013(request_msg, service_id)

        ' Set a random message size
        Dim rand As Random = New Random
        length = CUInt(1) + service_id + rand.Next(0, 200)
        length = If(length > CanTpApi.PCANTP_MAX_LENGTH_ISOTP2004, CanTpApi.PCANTP_MAX_LENGTH_ISOTP2004, length)

        ' Allocate response message
        status = UDSApi.MsgAlloc_2013(response_msg, config, length)
        Console.WriteLine("Allocate dummy response: {0}", STATUS_OK_KO(status))

        If UDSApi.StatusIsOk_2013(status) Then
            ' Fake a positive response
            UDSApi.SetDataServiceId_2013(response_msg, CByte(service_id + UDSApi.PUDS_SI_POSITIVE_RESPONSE))

            ' Fill with dummy data
            UDSApi.SetDataParameter_2013(response_msg, 0, service_id)

            For i As Integer = 1 To length - 2   ' (length - 1) as positive response SID uses 1 byte
                UDSApi.SetDataParameter_2013(response_msg, i, CByte((i + 1) Mod 256))
            Next

            ' Do not set PUDS_FLAG_SUPPRESS_POSITIVE_RESPONSE flag when service has a subfunction
            If HasSubFunction(service_id) Then
                CanTpApi.setData_2016(response_msg.msg, 1, 0)
            End If
        End If

        Return status
    End Function



    ''' <summary>
    ''' Create the RequestTransferExit response message
    ''' </summary>
    ''' <param name="config">Input, UDS configuration</param>
    ''' <param name="request_msg">Input, received request, containing request parameters</param>
    ''' <param name="response_msg">Output, contains the generated response</param>
    ''' <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    Function create_SvcRequestTransferExit_response(ByVal config As uds_msgconfig, ByRef request_msg As uds_msg, ByRef response_msg As uds_msg) As uds_status
        Dim status As uds_status
        Dim length As UInt32

        ' Allocate message
        Dim rand As Random = New Random
        length = 1 + 1 + rand.Next(0, 50)
        status = UDSApi.MsgAlloc_2013(response_msg, config, length)
        Console.WriteLine("Allocate RequestTransferExit response: {0}", STATUS_OK_KO(status))

        If UDSApi.StatusIsOk_2013(status) Then
            ' Fill with dummy data
            For i As UInt32 = 0 To length - 2   ' (length - 1) as positive response SID uses 1 byte
                UDSApi.SetDataParameter_2013(response_msg, i, (i + 1) Mod 256)
            Next

            UDSApi.SetDataServiceId_2013(response_msg, CByte(uds_service.PUDS_SERVICE_SI_RequestTransferExit + UDSApi.PUDS_SI_POSITIVE_RESPONSE))
        End If

        Return status
    End Function


    ''' <summary>
    ''' Create the SvcRequestUpload response message
    ''' </summary>
    ''' <param name="config">Input, UDS configuration</param>
    ''' <param name="request_msg">Input, received request, containing request parameters</param>
    ''' <param name="response_msg">Output, contains the generated response</param>
    ''' <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    Function create_SvcRequestUpload_response(ByVal config As uds_msgconfig, ByRef request_msg As uds_msg, ByRef response_msg As uds_msg) As uds_status
        Dim status As uds_status
        Dim length As UInt32

        ' Allocate message
        Dim rand As Random = New Random
        length = 2 + 1 + rand.Next(0, 50)
        status = UDSApi.MsgAlloc_2013(response_msg, config, length)
        Console.WriteLine("Allocate RequestUpload response: {0}", STATUS_OK_KO(status))

        If UDSApi.StatusIsOk_2013(status) Then
            ' Max number of block length = 0xF
            UDSApi.SetDataParameter_2013(response_msg, 0, &HF0)


            ' Fill with dummy data
            For i As UInt32 = 1 To length - 2   ' (length - 1) as positive response SID uses 1 byte
                UDSApi.SetDataParameter_2013(response_msg, i, (i + 1) Mod 256)
            Next

            UDSApi.SetDataServiceId_2013(response_msg, CByte(uds_service.PUDS_SERVICE_SI_RequestUpload + UDSApi.PUDS_SI_POSITIVE_RESPONSE))
        End If

        Return status
    End Function


    ''' <summary>
    ''' Create the SvcRequestDownload response message
    ''' </summary>
    ''' <param name="config">Input, UDS configuration</param>
    ''' <param name="request_msg">Input, received request, containing request parameters</param>
    ''' <param name="response_msg">Output, contains the generated response</param>
    ''' <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    Function create_SvcRequestDownload_response(ByVal config As uds_msgconfig, ByRef request_msg As uds_msg, ByRef response_msg As uds_msg) As uds_status
        Dim status As uds_status
        Dim length As UInt32

        ' Allocate message
        Dim rand As Random = New Random
        length = 2 + 1 + rand.Next(0, 50)
        status = UDSApi.MsgAlloc_2013(response_msg, config, length)
        Console.WriteLine("Allocate RequestDownload response: {0}", STATUS_OK_KO(status))

        If UDSApi.StatusIsOk_2013(status) Then
            ' Max number of block length = 0xF
            UDSApi.SetDataParameter_2013(response_msg, 0, &HF0)


            ' Fill with dummy data
            For i As UInt32 = 1 To length - 2   ' (length - 1) as positive response SID uses 1 byte
                UDSApi.SetDataParameter_2013(response_msg, i, (i + 1) Mod 256)
            Next

            UDSApi.SetDataServiceId_2013(response_msg, CByte(uds_service.PUDS_SERVICE_SI_RequestDownload + UDSApi.PUDS_SI_POSITIVE_RESPONSE))
        End If

        Return status
    End Function


    ''' <summary>
    ''' Create the RoutineControl response message
    ''' </summary>
    ''' <param name="config">Input, UDS configuration</param>
    ''' <param name="request_msg">Input, received request, containing request parameters</param>
    ''' <param name="response_msg">Output, contains the generated response</param>
    ''' <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    Function create_SvcRoutineControl_response(ByVal config As uds_msgconfig, ByRef request_msg As uds_msg, ByRef response_msg As uds_msg) As uds_status
        Dim status As uds_status

        ' Allocate response
        status = UDSApi.MsgAlloc_2013(response_msg, config, 4)
        Console.WriteLine("Allocate RoutineControl response: {0}", STATUS_OK_KO(status))

        If UDSApi.StatusIsOk_2013(status) Then
            ' Copy data
            Dim param0 As Byte
            UDSApi.GetDataParameter_2013(request_msg, 0, param0)
            Dim param1 As Byte
            UDSApi.GetDataParameter_2013(request_msg, 1, param1)
            Dim param2 As Byte
            UDSApi.GetDataParameter_2013(request_msg, 2, param2)
            UDSApi.SetDataParameter_2013(response_msg, 0, param0)
            UDSApi.SetDataParameter_2013(response_msg, 1, param1)
            UDSApi.SetDataParameter_2013(response_msg, 2, param2)

            ' Routine status record not implemented
            UDSApi.SetDataServiceId_2013(response_msg, CByte(uds_service.PUDS_SERVICE_SI_RoutineControl + UDSApi.PUDS_SI_POSITIVE_RESPONSE))
        End If

        Return status
    End Function

    ''' <summary>
    ''' Create the InputOutputControlByIdentifier response message
    ''' </summary>
    ''' <param name="config">Input, UDS configuration</param>
    ''' <param name="request_msg">Input, received request, containing request parameters</param>
    ''' <param name="response_msg">Output, contains the generated response</param>
    ''' <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    Function create_SvcInputOutputControlByIdentifier_response(ByVal config As uds_msgconfig, ByRef request_msg As uds_msg, ByRef response_msg As uds_msg) As uds_status
        Dim status As uds_status
        status = UDSApi.MsgAlloc_2013(response_msg, config, 3)
        Console.WriteLine("Allocate InputOutputControlByIdentifier response: {0}", STATUS_OK_KO(status))

        If UDSApi.StatusIsOk_2013(status) Then
            ' Copy data
            Dim val As Byte
            UDSApi.GetDataParameter_2013(request_msg, 0, val)
            UDSApi.SetDataParameter_2013(response_msg, 0, val)
            UDSApi.GetDataParameter_2013(request_msg, 1, val)
            UDSApi.SetDataParameter_2013(response_msg, 1, val)

            ' Control status record not implemented
            UDSApi.SetDataServiceId_2013(response_msg, CByte(uds_service.PUDS_SERVICE_SI_InputOutputControlByIdentifier + UDSApi.PUDS_SI_POSITIVE_RESPONSE))
        End If

        Return status
    End Function


    ''' <summary>
    ''' Create the ClearDiagnosticInformation response message
    ''' </summary>
    ''' <param name="config">Input, UDS configuration</param>
    ''' <param name="request_msg">Input, received request, containing request parameters</param>
    ''' <param name="response_msg">Output, contains the generated response</param>
    ''' <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    Function create_SvcClearDiagnosticInformation_response(ByVal config As uds_msgconfig, ByRef request_msg As uds_msg, ByRef response_msg As uds_msg) As uds_status
        Dim status As uds_status
        status = UDSApi.MsgAlloc_2013(response_msg, config, 1)
        Console.WriteLine("Allocate ClearDiagnosticInformation response: {0}", STATUS_OK_KO(status))

        If UDSApi.StatusIsOk_2013(status) Then
            UDSApi.SetDataServiceId_2013(response_msg, CByte(uds_service.PUDS_SERVICE_SI_ClearDiagnosticInformation + UDSApi.PUDS_SI_POSITIVE_RESPONSE))
        End If

        Return status
    End Function


    ''' <summary>
    ''' Create the WriteDataByIdentifier response message
    ''' </summary>
    ''' <param name="config">Input, UDS configuration</param>
    ''' <param name="request_msg">Input, received request, containing request parameters</param>
    ''' <param name="response_msg">Output, contains the generated response</param>
    ''' <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    Function create_SvcWriteDataByIdentifier_response(ByVal config As uds_msgconfig, ByRef request_msg As uds_msg, ByRef response_msg As uds_msg) As uds_status
        Dim status As uds_status
        status = UDSApi.MsgAlloc_2013(response_msg, config, 3)
        Console.WriteLine("Allocate WriteDataByIdentifier response: {0}", STATUS_OK_KO(status))

        If UDSApi.StatusIsOk_2013(status) Then
            ' Copy data
            Dim param0 As Byte
            UDSApi.GetDataParameter_2013(request_msg, 0, param0)
            Dim param1 As Byte
            UDSApi.GetDataParameter_2013(request_msg, 1, param1)
            UDSApi.SetDataParameter_2013(response_msg, 0, param0)
            UDSApi.SetDataParameter_2013(response_msg, 1, param1)
            UDSApi.SetDataServiceId_2013(response_msg, CByte(uds_service.PUDS_SERVICE_SI_WriteDataByIdentifier + UDSApi.PUDS_SI_POSITIVE_RESPONSE))
        End If

        Return status
    End Function

    ''' <summary>
    ''' Create the ReadDataByPeriodicIdentifier response message
    ''' </summary>
    ''' <param name="config">Input, UDS configuration</param>
    ''' <param name="request_msg">Input, received request, containing request parameters</param>
    ''' <param name="response_msg">Output, contains the generated response</param>
    ''' <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    Function create_SvcReadDataByPeriodicIdentifier_response(ByVal config As uds_msgconfig, ByRef request_msg As uds_msg, ByRef response_msg As uds_msg) As uds_status
        Dim status As uds_status
        status = UDSApi.MsgAlloc_2013(response_msg, config, 1)
        Console.WriteLine("Allocate ReadDataByPeriodicIdentifier response: {0}", STATUS_OK_KO(status))

        If UDSApi.StatusIsOk_2013(status) Then
            UDSApi.SetDataServiceId_2013(response_msg, CByte(uds_service.PUDS_SERVICE_SI_ReadDataByPeriodicIdentifier) + UDSApi.PUDS_SI_POSITIVE_RESPONSE)
        End If

        Return status
    End Function


    ''' <summary>
    ''' Create the ReadMemoryByAddress response message
    ''' </summary>
    ''' <param name="config">Input, UDS configuration</param>
    ''' <param name="request_msg">Input, received request, containing request parameters</param>
    ''' <param name="response_msg">Output, contains the generated response</param>
    ''' <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    Function create_SvcReadMemoryByAddress_response(ByVal config As uds_msgconfig, ByRef request_msg As uds_msg, ByRef response_msg As uds_msg) As uds_status
        Dim status As uds_status
        Dim length As UInt32

        ' Read memory_size_size = bits [7..4]
        Dim param As Byte
        UDSApi.GetDataParameter_2013(request_msg, 1, param)
        length = CUInt(1 + (param >> 4 And &HF))

        ' Allocate message
        status = UDSApi.MsgAlloc_2013(response_msg, config, length)
        Console.WriteLine("Allocate ReadMemoryByAddress response: {0}", STATUS_OK_KO(status))

        If UDSApi.StatusIsOk_2013(status) Then
            ' Fill with dummy data
            For i As Integer = 0 To length - 2   ' (length - 1) as positive response SID uses 1 byte
                UDSApi.SetDataParameter_2013(response_msg, i, (i + 1) Mod 256)
            Next

            UDSApi.SetDataServiceId_2013(response_msg, CByte(uds_service.PUDS_SERVICE_SI_ReadMemoryByAddress) + UDSApi.PUDS_SI_POSITIVE_RESPONSE)
        End If

        Return status
    End Function


    ''' <summary>
    ''' Create the ReadScalingDataByIdentifier response message
    ''' </summary>
    ''' <param name="config">Input, UDS configuration</param>
    ''' <param name="request_msg">Input, received request, containing request parameters</param>
    ''' <param name="response_msg">Output, contains the generated response</param>
    ''' <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    Function create_SvcReadScalingDataByIdentifier_response(ByVal config As uds_msgconfig, ByRef request_msg As uds_msg, ByRef response_msg As uds_msg) As uds_status
        Dim status As uds_status

        ' Allocate message
        status = UDSApi.MsgAlloc_2013(response_msg, config, 12)
        Console.WriteLine("Allocate ReadScalingDataByIdentifier response: {0}", STATUS_OK_KO(status))

        If UDSApi.StatusIsOk_2013(status) Then
            UDSApi.SetDataServiceId_2013(response_msg, CByte(uds_service.PUDS_SERVICE_SI_ReadScalingDataByIdentifier + UDSApi.PUDS_SI_POSITIVE_RESPONSE))

            ' Copy data
            Dim param0 As Byte
            UDSApi.GetDataParameter_2013(request_msg, 0, param0)
            Dim param1 As Byte
            UDSApi.GetDataParameter_2013(request_msg, 1, param1)
            UDSApi.SetDataParameter_2013(response_msg, 0, param0)
            UDSApi.SetDataParameter_2013(response_msg, 1, param1)

            ' Create a formula Vehicule Speed = (0.75*x+30) km/h
            UDSApi.SetDataParameter_2013(response_msg, 2, &H0 << 4 Or &H1)  ' unsigned numeric of 1 Bytes)
            UDSApi.SetDataParameter_2013(response_msg, 3, &H90)    ' formula, 0 data bytes
            UDSApi.SetDataParameter_2013(response_msg, 4, &H0)    ' formulaIdentifier = C0 * x + C1
            UDSApi.SetDataParameter_2013(response_msg, 5, &HE0)    ' C0 high byte
            UDSApi.SetDataParameter_2013(response_msg, 6, &H4B)    ' C0 low byte
            UDSApi.SetDataParameter_2013(response_msg, 7, &H0)    ' C1 high byte
            UDSApi.SetDataParameter_2013(response_msg, 8, &H1E)    ' C1 low byte
            UDSApi.SetDataParameter_2013(response_msg, 9, &HA0)    ' unit/format, 0 data bytes
            UDSApi.SetDataParameter_2013(response_msg, 10, &H30)   ' unit ID, km/h
        End If

        Return status
    End Function

    ''' <summary>
    ''' Create the ReadDataByIdentifier response message
    ''' </summary>
    ''' <param name="config">Input, UDS configuration</param>
    ''' <param name="request_msg">Input, received request, containing request parameters</param>
    ''' <param name="response_msg">Output, contains the generated response</param>
    ''' <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    Function create_SvcReadDataByIdentifier_response(ByVal config As uds_msgconfig, ByRef request_msg As uds_msg, ByRef response_msg As uds_msg) As uds_status
        Dim status As uds_status
        Dim length As UInt32
        Dim counter As Integer

        Dim req_data As cantp_msgdata = New cantp_msgdata()
        If request_msg.msg.Msgdata <> IntPtr.Zero Then
            req_data = request_msg.msg.Msgdata_any_Copy
        End If

        ' Allocate message
        length = (req_data.length - 1) * 7 / 2 + 1
        status = UDSApi.MsgAlloc_2013(response_msg, config, length)
        Console.WriteLine("Allocate ReadDataByIdentifier response: {0}", STATUS_OK_KO(status))

        If UDSApi.StatusIsOk_2013(status) Then
            counter = 0

            For i As Integer = 0 To req_data.length - 2 Step 2
                ' Copy data
                Dim param As Byte
                UDSApi.GetDataParameter_2013(request_msg, i, param)
                UDSApi.SetDataParameter_2013(response_msg, counter, param)
                counter = counter + 1
                UDSApi.GetDataParameter_2013(request_msg, i + 1, param)
                UDSApi.SetDataParameter_2013(response_msg, counter, param)
                counter = counter + 1

                ' Data record: fill with dummy data
                For j As Integer = 0 To 4
                    UDSApi.SetDataParameter_2013(response_msg, counter, Asc("A"c) + j)
                    counter = counter + 1
                Next
            Next

            UDSApi.SetDataServiceId_2013(response_msg, CByte(uds_service.PUDS_SERVICE_SI_ReadDataByIdentifier) + UDSApi.PUDS_SI_POSITIVE_RESPONSE)
        End If

        Return status
    End Function


    ''' <summary>
    ''' Create the LinkControl response message
    ''' </summary>
    ''' <param name="config">Input, UDS configuration</param>
    ''' <param name="request_msg">Input, received request, containing request parameters</param>
    ''' <param name="response_msg">Output, contains the generated response</param>
    ''' <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    Function create_SvcLinkControl_response(ByVal config As uds_msgconfig, ByRef request_msg As uds_msg, ByRef response_msg As uds_msg) As uds_status
        Dim status As uds_status
        status = UDSApi.MsgAlloc_2013(response_msg, config, 2)
        Console.WriteLine("Allocate LinkControl response: {0}", STATUS_OK_KO(status))

        If UDSApi.StatusIsOk_2013(status) Then
            Dim param0 As Byte
            UDSApi.GetDataParameter_2013(request_msg, 0, param0)
            UDSApi.SetDataParameter_2013(response_msg, 0, param0)
            UDSApi.SetDataServiceId_2013(response_msg, CByte(uds_service.PUDS_SERVICE_SI_LinkControl + UDSApi.PUDS_SI_POSITIVE_RESPONSE))
        End If

        Return status
    End Function


    ''' <summary>
    ''' Create the ResponseOnEvent response message
    ''' </summary>
    ''' <param name="config">Input, UDS configuration</param>
    ''' <param name="request_msg">Input, received request, containing request parameters</param>
    ''' <param name="response_msg">Output, contains the generated response</param>
    ''' <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    Function create_SvcResponseOnEvent_response(ByVal config As uds_msgconfig, ByRef request_msg As uds_msg, ByRef response_msg As uds_msg) As uds_status
        Dim status As uds_status
        Dim param0 As Byte
        UDSApi.GetDataParameter_2013(request_msg, 0, param0)
        Dim param1 As Byte
        UDSApi.GetDataParameter_2013(request_msg, 1, param1)

        If param0 = CByte(UDSApi.uds_svc_param_roe.PUDS_SVC_PARAM_ROE_RAE) Then
            status = UDSApi.MsgAlloc_2013(response_msg, config, 3)
            Console.WriteLine("Allocate ResponseOnEvent response: {0}", STATUS_OK_KO(status))

            If UDSApi.StatusIsOk_2013(status) Then
                UDSApi.SetDataParameter_2013(response_msg, 1, 0) ' Number of activated events
                ' Event type and service to respond to records not implemented
            End If
        Else
            status = UDSApi.MsgAlloc_2013(response_msg, config, 4)
            Console.WriteLine("Allocate ResponseOnEvent response: {0}", STATUS_OK_KO(status))

            If UDSApi.StatusIsOk_2013(status) Then
                UDSApi.SetDataParameter_2013(response_msg, 1, 0)   ' Number of identified events
                UDSApi.SetDataParameter_2013(response_msg, 2, param1) ' Event window time
                ' Event type and service to respond to records not implemented
            End If
        End If

        If UDSApi.StatusIsOk_2013(status) Then
            UDSApi.SetDataParameter_2013(response_msg, 0, param0)
            UDSApi.SetDataServiceId_2013(response_msg, CByte(uds_service.PUDS_SERVICE_SI_ResponseOnEvent + UDSApi.PUDS_SI_POSITIVE_RESPONSE))
        End If

        Return status
    End Function


    ''' <summary>
    ''' Create the ControlDTCSetting response message
    ''' </summary>
    ''' <param name="config">Input, UDS configuration</param>
    ''' <param name="request_msg">Input, received request, containing request parameters</param>
    ''' <param name="response_msg">Output, contains the generated response</param>
    ''' <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    Function create_SvcControlDTCSetting_response(ByVal config As uds_msgconfig, ByRef request_msg As uds_msg, ByRef response_msg As uds_msg) As uds_status
        Dim status As uds_status
        status = UDSApi.MsgAlloc_2013(response_msg, config, 2)
        Console.WriteLine("Allocate ControlDTCSetting response: {0}", STATUS_OK_KO(status))

        If UDSApi.StatusIsOk_2013(status) Then
            Dim param0 As Byte
            UDSApi.GetDataParameter_2013(request_msg, 0, param0)
            UDSApi.SetDataParameter_2013(response_msg, 0, param0)
            UDSApi.SetDataServiceId_2013(response_msg, CByte(uds_service.PUDS_SERVICE_SI_ControlDTCSetting + UDSApi.PUDS_SI_POSITIVE_RESPONSE))
        End If

        Return status
    End Function


    ''' <summary>
    ''' Create the SecuredDataTransmission response message
    ''' </summary>
    ''' <param name="config">Input, UDS configuration</param>
    ''' <param name="request_msg">Input, received request, containing request parameters</param>
    ''' <param name="response_msg">Output, contains the generated response</param>
    ''' <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    Function create_SvcSecuredDataTransmission_response(ByVal config As uds_msgconfig, ByRef request_msg As uds_msg, ByRef response_msg As uds_msg) As uds_status
        Dim status As uds_status
        Dim length As UInt32

        ' Allocate message
        Dim rand As Random = New Random
        length = 1 + rand.Next(0, 50)
        status = UDSApi.MsgAlloc_2013(response_msg, config, length)
        Console.WriteLine("Allocate SecuredDataTransmission response: {0}", STATUS_OK_KO(status))

        If UDSApi.StatusIsOk_2013(status) Then
            ' Fill with dummy data (check Security-SubLayer record defined in ISO-15764)
            For i As UInt32 = 0 To length - 2   ' (length - 1) as positive response SID uses 1 byte
                UDSApi.SetDataParameter_2013(response_msg, i, (i + 1) Mod 256)
            Next

            UDSApi.SetDataServiceId_2013(response_msg, CByte(uds_service.PUDS_SERVICE_SI_SecuredDataTransmission + UDSApi.PUDS_SI_POSITIVE_RESPONSE))
        End If

        Return status
    End Function


    ''' <summary>
    ''' Create the TesterPresent response message
    ''' </summary>
    ''' <param name="config">Input, UDS configuration</param>
    ''' <param name="request_msg">Input, received request, containing request parameters</param>
    ''' <param name="response_msg">Output, contains the generated response</param>
    ''' <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    Function create_SvcTesterPresent_response(ByVal config As uds_msgconfig, ByRef request_msg As uds_msg, ByRef response_msg As uds_msg) As uds_status
        Dim status As uds_status
        status = UDSApi.MsgAlloc_2013(response_msg, config, 2)
        Console.WriteLine("Allocate TesterPresent response: {0}", STATUS_OK_KO(status))

        If UDSApi.StatusIsOk_2013(status) Then
            Dim param0 As Byte
            UDSApi.GetDataParameter_2013(request_msg, 0, param0)
            UDSApi.SetDataParameter_2013(response_msg, 0, param0)
            UDSApi.SetDataServiceId_2013(response_msg, CByte(uds_service.PUDS_SERVICE_SI_TesterPresent + UDSApi.PUDS_SI_POSITIVE_RESPONSE))
        End If

        Return status
    End Function


    ''' <summary>
    ''' Create the CommunicationControl response message
    ''' </summary>
    ''' <param name="config">Input, UDS configuration</param>
    ''' <param name="request_msg">Input, received request, containing request parameters</param>
    ''' <param name="response_msg">Output, contains the generated response</param>
    ''' <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    Function create_SvcCommunicationControl_response(ByVal config As uds_msgconfig, ByRef request_msg As uds_msg, ByRef response_msg As uds_msg) As uds_status
        Dim status As uds_status
        status = UDSApi.MsgAlloc_2013(response_msg, config, 2)
        Console.WriteLine("Allocate CommunicationControl response: {0}", STATUS_OK_KO(status))

        If UDSApi.StatusIsOk_2013(status) Then
            Dim param0 As Byte
            UDSApi.GetDataParameter_2013(request_msg, 0, param0)
            UDSApi.SetDataParameter_2013(response_msg, 0, param0)
            UDSApi.SetDataServiceId_2013(response_msg, CByte(uds_service.PUDS_SERVICE_SI_CommunicationControl + UDSApi.PUDS_SI_POSITIVE_RESPONSE))
        End If

        Return status
    End Function


    ''' <summary>
    ''' Create the SecurityAccess response message
    ''' </summary>
    ''' <param name="config">Input, UDS configuration</param>
    ''' <param name="request_msg">Input, received request, containing request parameters</param>
    ''' <param name="response_msg">Output, contains the generated response</param>
    ''' <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    Function create_SvcSecurityAccess_response(ByVal config As uds_msgconfig, ByRef request_msg As uds_msg, ByRef response_msg As uds_msg) As uds_status
        Dim status As uds_status
        Dim length As UInt32
        Dim i As UInt32

        ' Allocate response message
        Dim param0 As Byte
        UDSApi.GetDataParameter_2013(request_msg, 0, param0)

        If param0 >= UDSApi.PUDS_SVC_PARAM_SA_RSD_MIN AndAlso param0 <= UDSApi.PUDS_SVC_PARAM_SA_RSD_MAX AndAlso param0 Mod 2 = 1 Then

            ' Request security seed are Even values
            ' Fill with dummy data
            length = If(CUInt(1 + uds_service.PUDS_SERVICE_SI_SecurityAccess) > CUInt(CanTpApi.PCANTP_MAX_LENGTH_ISOTP2004), CUInt(CanTpApi.PCANTP_MAX_LENGTH_ISOTP2004), CUInt(1 + uds_service.PUDS_SERVICE_SI_SecurityAccess))
            status = UDSApi.MsgAlloc_2013(response_msg, config, length)
            Console.WriteLine("Allocate SecurityAccess response: {0}", STATUS_OK_KO(status))

            If UDSApi.StatusIsOk_2013(status) Then
                For i = 1 To length - 2    ' (length - 1) as positive response SID uses 1 byte
                    UDSApi.SetDataParameter_2013(response_msg, i, (i + 1) Mod 256)
                Next
            End If
        Else
            status = UDSApi.MsgAlloc_2013(response_msg, config, 2)
            Console.WriteLine("Allocate SecurityAccess response: {0}", STATUS_OK_KO(status))
        End If

        If UDSApi.StatusIsOk_2013(status) Then
            UDSApi.SetDataParameter_2013(response_msg, 0, param0)
            UDSApi.SetDataServiceId_2013(response_msg, CByte(uds_service.PUDS_SERVICE_SI_SecurityAccess + UDSApi.PUDS_SI_POSITIVE_RESPONSE))
        End If

        Return status
    End Function



    ''' <summary>
    ''' Create the DiagnosticSessionControl response message
    ''' </summary>
    ''' <param name="config">Input, UDS configuration</param>
    ''' <param name="request_msg">Input, received request, containing request parameters</param>
    ''' <param name="response_msg">Output, contains the generated response</param>
    ''' <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    Function create_SvcDiagnosticSessionControl_response(ByVal config As uds_msgconfig, ByRef request_msg As uds_msg, ByRef response_msg As uds_msg) As uds_status
        Dim status As uds_status

        ' Allocate response message
        status = UDSApi.MsgAlloc_2013(response_msg, config, 6)
        Console.WriteLine("Allocate DiagnosticSessionControl response: {0}", STATUS_OK_KO(status))

        If UDSApi.StatusIsOk_2013(status) Then
            ' Fill response
            Dim param0 As Byte
            UDSApi.GetDataParameter_2013(request_msg, 0, param0)
            UDSApi.SetDataParameter_2013(response_msg, 0, param0)
            UDSApi.SetDataParameter_2013(response_msg, 1, &H0)    ' P2Can_Server_Max = 0x0010
            UDSApi.SetDataParameter_2013(response_msg, 2, &H10)
            UDSApi.SetDataParameter_2013(response_msg, 3, &H3)    ' P2*Can_Server_Max = 0x03E8
            UDSApi.SetDataParameter_2013(response_msg, 4, &HE8)
            UDSApi.SetDataServiceId_2013(response_msg, CByte(uds_service.PUDS_SERVICE_SI_DiagnosticSessionControl + UDSApi.PUDS_SI_POSITIVE_RESPONSE))
        End If

        Return status
    End Function



    ''' <summary>
    ''' Create the TransferData response message
    ''' </summary>
    ''' <param name="config">Input, UDS configuration</param>
    ''' <param name="request_msg">Input, received request, containing request parameters</param>
    ''' <param name="response_msg">Output, contains the generated response</param>
    ''' <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    Function create_SvcTransferData_response(ByVal tp_handle As cantp_handle, ByVal config As uds_msgconfig, ByRef request_msg As uds_msg, ByRef response_msg As uds_msg) As uds_status
        Dim status As uds_status
        Dim length As UInt32
        Dim pending_response_msg As uds_msg = New uds_msg()

        ' Initialize locale struct

        Dim req_data As cantp_msgdata_isotp = New cantp_msgdata_isotp()
        If request_msg.msg.Msgdata <> IntPtr.Zero Then
            req_data = request_msg.msg.Msgdata_isotp_Copy
        End If

        ' Custom response to client_all_request example:
        '  a. service is requested functionally,
        '  b. 1st response is NRC response pending
        '  c. wait
        '  d. send correct response
        If req_data.netaddrinfo.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_FUNCTIONAL Then

            ' Transmit a NRC response pending
            status = UDSApi.MsgAlloc_2013(pending_response_msg, config, 3)
            Console.WriteLine("Allocate TransferData pending response: {0}", STATUS_OK_KO(status))


            ' Transmit a NRC response pending
            If UDSApi.StatusIsOk_2013(status) Then
                UDSApi.SetDataServiceId_2013(pending_response_msg, CByte(uds_service.PUDS_SERVICE_NR_SI))
                UDSApi.SetDataParameter_2013(pending_response_msg, 0, CByte(uds_service.PUDS_SERVICE_SI_TransferData))
                UDSApi.SetDataParameter_2013(pending_response_msg, 1, UDSApi.PUDS_NRC_EXTENDED_TIMING)
            End If
            status = UDSApi.Write_2013(tp_handle, pending_response_msg)
            Console.WriteLine()
            Console.WriteLine("   ...Transmitting a NRC Response Pending message: {0}", CInt(status))
            Console.WriteLine()
            Console.Write("   ...simulating computation... (waiting ~{0}ms)", CInt(UDSApi.PUDS_P2CAN_ENHANCED_SERVER_MAX_DEFAULT))
            Thread.Sleep(CInt(UDSApi.PUDS_P2CAN_ENHANCED_SERVER_MAX_DEFAULT) - 100)

            ' Reallocate response message
            status = UDSApi.MsgFree_2013(pending_response_msg)
            Console.WriteLine("Free pending response: {0}", STATUS_OK_KO(status))

            ' Initialize real service response
            config.type = uds_msgtype.PUDS_MSGTYPE_USDT
            length = CanTpApi.PCANTP_MAX_LENGTH_ISOTP2004
            status = UDSApi.MsgAlloc_2013(response_msg, config, length)
            Console.WriteLine("Allocate TransferData real response: {0}", STATUS_OK_KO(status))
        Else
            length = If(2 + req_data.length > CanTpApi.PCANTP_MAX_LENGTH_ISOTP2004, CanTpApi.PCANTP_MAX_LENGTH_ISOTP2004, 2 + req_data.length)
            status = UDSApi.MsgAlloc_2013(response_msg, config, length)
            Console.WriteLine("Allocate TransferData response: {0}", STATUS_OK_KO(status))
        End If

        If UDSApi.StatusIsOk_2013(status) Then
            Dim param As Byte
            UDSApi.GetDataParameter_2013(request_msg, 0, param)
            UDSApi.SetDataParameter_2013(response_msg, 0, param)


            ' Fill with dummy data
            For i As Integer = 1 To length - 2   ' (length - 1) as positive response SID uses 1 byte
                UDSApi.SetDataParameter_2013(response_msg, i, ((i + 1) And &HFF) Mod 256)
            Next

            UDSApi.SetDataServiceId_2013(response_msg, CByte(uds_service.PUDS_SERVICE_SI_TransferData) + UDSApi.PUDS_SI_POSITIVE_RESPONSE)
        End If

        Return status
    End Function

    ''' <summary>
    ''' Process request and send a response
    ''' </summary>
    ''' <param name="tp_handle">CANTP channel</param>
    ''' <param name="server_address">Server address</param>
    ''' <param name="request_msg">Received request message</param>
    Sub process_request(ByVal tp_handle As cantp_handle, ByVal server_address As UInt16, ByRef request_msg As uds_msg)
        Dim config As uds_msgconfig = New uds_msgconfig()
        Dim response_msg As uds_msg = New uds_msg()
        Dim status As uds_status = uds_status.PUDS_STATUS_OK


        ' Initialize config from request message network address information
        Dim req_isotp As cantp_msgdata_isotp = New cantp_msgdata_isotp()
        If request_msg.msg.Msgdata <> IntPtr.Zero Then
            req_isotp = request_msg.msg.Msgdata_isotp_Copy
        End If



        config.nai = ConvertNai(req_isotp.netaddrinfo, request_msg.msg.can_info.can_msgtype)
        config.can_msgtype = request_msg.msg.can_info.can_msgtype
        config.type = uds_msgtype.PUDS_MSGTYPE_USDT
        config.nai.extension_addr = req_isotp.netaddrinfo.extension_addr


        ' Set target and source addresses
        If req_isotp.netaddrinfo.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_FUNCTIONAL Then
            ' response to functional addressing is set to TEST_EQUIPMENT
            config.nai.target_addr = CType(uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT, UInt16)
        Else
            config.nai.target_addr = req_isotp.netaddrinfo.source_addr
        End If

        config.nai.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL
        config.nai.source_addr = server_address

        ' This is a valid request, switch services
        Dim service_id As Byte
        UDSApi.GetDataServiceId_2013(request_msg, service_id)

        Select Case service_id
            Case CByte(uds_service.PUDS_SERVICE_SI_DiagnosticSessionControl)
                status = create_SvcDiagnosticSessionControl_response(config, request_msg, response_msg)
            Case CByte(uds_service.PUDS_SERVICE_SI_SecurityAccess)
                status = create_SvcSecurityAccess_response(config, request_msg, response_msg)
            Case CByte(uds_service.PUDS_SERVICE_SI_CommunicationControl)
                status = create_SvcCommunicationControl_response(config, request_msg, response_msg)
            Case CByte(uds_service.PUDS_SERVICE_SI_TesterPresent)
                status = create_SvcTesterPresent_response(config, request_msg, response_msg)
            Case CByte(uds_service.PUDS_SERVICE_SI_SecuredDataTransmission)
                status = create_SvcSecuredDataTransmission_response(config, request_msg, response_msg)
            Case CByte(uds_service.PUDS_SERVICE_SI_ControlDTCSetting)
                status = create_SvcControlDTCSetting_response(config, request_msg, response_msg)
            Case CByte(uds_service.PUDS_SERVICE_SI_ResponseOnEvent)
                status = create_SvcResponseOnEvent_response(config, request_msg, response_msg)
            Case CByte(uds_service.PUDS_SERVICE_SI_LinkControl)
                status = create_SvcLinkControl_response(config, request_msg, response_msg)
            Case CByte(uds_service.PUDS_SERVICE_SI_ReadDataByIdentifier)
                status = create_SvcReadDataByIdentifier_response(config, request_msg, response_msg)
            Case CByte(uds_service.PUDS_SERVICE_SI_ReadMemoryByAddress)
                status = create_SvcReadMemoryByAddress_response(config, request_msg, response_msg)
            Case CByte(uds_service.PUDS_SERVICE_SI_ReadScalingDataByIdentifier)
                status = create_SvcReadScalingDataByIdentifier_response(config, request_msg, response_msg)
            Case CByte(uds_service.PUDS_SERVICE_SI_ReadDataByPeriodicIdentifier)
                status = create_SvcReadDataByPeriodicIdentifier_response(config, request_msg, response_msg)
            Case CByte(uds_service.PUDS_SERVICE_SI_DynamicallyDefineDataIdentifier)
                status = create_SvcDynamicallyDefineDataIdentifier_response(config, request_msg, response_msg)
            Case CByte(uds_service.PUDS_SERVICE_SI_WriteDataByIdentifier)
                status = create_SvcWriteDataByIdentifier_response(config, request_msg, response_msg)
            Case CByte(uds_service.PUDS_SERVICE_SI_WriteMemoryByAddress)
                status = create_SvcWriteMemoryByAddress_response(config, request_msg, response_msg)
            Case CByte(uds_service.PUDS_SERVICE_SI_ClearDiagnosticInformation)
                status = create_SvcClearDiagnosticInformation_response(config, request_msg, response_msg)
            Case CByte(uds_service.PUDS_SERVICE_SI_InputOutputControlByIdentifier)
                status = create_SvcInputOutputControlByIdentifier_response(config, request_msg, response_msg)
            Case CByte(uds_service.PUDS_SERVICE_SI_RoutineControl)
                status = create_SvcRoutineControl_response(config, request_msg, response_msg)
            Case CByte(uds_service.PUDS_SERVICE_SI_RequestDownload)
                status = create_SvcRequestDownload_response(config, request_msg, response_msg)
            Case CByte(uds_service.PUDS_SERVICE_SI_RequestUpload)
                status = create_SvcRequestUpload_response(config, request_msg, response_msg)
            Case CByte(uds_service.PUDS_SERVICE_SI_TransferData)
                status = create_SvcTransferData_response(tp_handle, config, request_msg, response_msg)
            Case CByte(uds_service.PUDS_SERVICE_SI_RequestTransferExit)
                status = create_SvcRequestTransferExit_response(config, request_msg, response_msg)
            Case CByte(uds_service.PUDS_SERVICE_SI_ECUReset)
                status = create_SvcECUReset_response(config, request_msg, response_msg)
            Case CByte(uds_service.PUDS_SERVICE_SI_ReadDTCInformation)
                ' Not yet implemented
                Console.WriteLine("Unknown service (0x{0:x})", service_id)
                status = create_dummy_response(config, request_msg, response_msg)
            Case CByte(uds_service.PUDS_SERVICE_SI_AccessTimingParameter)
                ' Not yet implemented
                Console.WriteLine("Unknown service (0x{0:x})", service_id)
                status = create_dummy_response(config, request_msg, response_msg)
            Case CByte(uds_service.PUDS_SERVICE_SI_RequestFileTransfer)
                ' Not yet implemented
                Console.WriteLine("Unknown service (0x{0:x})", service_id)
                status = create_dummy_response(config, request_msg, response_msg)
            Case CByte(uds_service.PUDS_SERVICE_SI_Authentication)
                ' Not yet implemented
                Console.WriteLine("Unknown service (0x{0:x})", service_id)
                status = create_dummy_response(config, request_msg, response_msg)
            Case Else
                Console.WriteLine("Unknown service (0x{0:x})", service_id)
                status = create_dummy_response(config, request_msg, response_msg)
        End Select


        ' Print allocation result
        Console.WriteLine("Allocate response message: {0}", STATUS_OK_KO(status))

        ' Send response message
        If UDSApi.StatusIsOk_2013(status) Then
            status = UDSApi.Write_2013(tp_handle, response_msg)
            Console.WriteLine()
            Console.WriteLine("   ...Transmitting response: {0}", CInt(status))
        End If

        ' Free response message
        status = UDSApi.MsgFree_2013(response_msg)
        Console.WriteLine("Free response message: {0}", STATUS_OK_KO(status))
    End Sub

    ''' <summary>Print an UDS message</summary>
    ''' <param name="msg">Message to print</param>
    Sub display_message(ByRef msg As uds_msg)
        ' A function that displays UDS messages

        Dim msg_isotp As cantp_msgdata_isotp = New cantp_msgdata_isotp()
        If msg.msg.Msgdata <> IntPtr.Zero Then
            msg_isotp = msg.msg.Msgdata_isotp_Copy
        End If

        If (msg_isotp.netaddrinfo.msgtype And cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION) = 0 Then
            Console.WriteLine()
            Console.WriteLine(" {0} message from 0x{1:X2} (to 0x{2:X2}, with extension 0x{3:X2}) - result: {4} - {5}",
                              If((msg_isotp.flags And cantp_msgflag.PCANTP_MSGFLAG_LOOPBACK) = 0, "Received UDS", "Received Loopback"),
                              CInt(msg_isotp.netaddrinfo.source_addr),
                              CInt(msg_isotp.netaddrinfo.target_addr), CInt(msg_isotp.netaddrinfo.extension_addr),
                              CInt(msg_isotp.netstatus),
                              If(msg_isotp.netstatus <> cantp_netstatus.PCANTP_NETSTATUS_OK, "PCANTP_NETSTATUS_OK !!!", "OK !"))
            ' Display data
            Console.Write("  -> Length: {0}, Data= ", CInt(msg_isotp.length))
            Dim val As Byte = 0

            For i As Integer = 0 To msg_isotp.length - 1
                CanTpApi.getData_2016(msg.msg, i, val)
                Console.Write("{0:X2} ", val)
            Next

            Console.WriteLine()
        Else
            Console.WriteLine()
            Console.WriteLine(" PENDING UDS message from 0x{0:X2} (to 0x{1:X2}, with extension 0x{2:X2}) -> length={3} ...",
                              CInt(msg_isotp.netaddrinfo.source_addr),
                              CInt(msg_isotp.netaddrinfo.target_addr), CInt(msg_isotp.netaddrinfo.extension_addr), CInt(msg_isotp.length))
        End If
    End Sub

    ''' <summary>Entry point of the program, start a CAN UDS server simulation</summary>
    Sub Main(ByVal args As String())
        Const BUFFER_SIZE As Integer = 256
        Dim buffer As StringBuilder = New StringBuilder(BUFFER_SIZE)
        Dim status As uds_status
        Dim tp_handle As cantp_handle
        Dim server_address As UInt32
        Dim stopit As Boolean
        Dim wait_result As Boolean
        Dim read_status As uds_status
        Dim request_msg As uds_msg = New uds_msg()
        Dim timeout_value As UInt32

        ' Initialize variables
        tp_handle = cantp_handle.PCANTP_HANDLE_USBBUS2 ' TODO: modify the value according to your available PCAN devices.
        server_address = CUInt(uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_1)

        ' Print version informations
        status = UDSApi.GetValue_2013(cantp_handle.PCANTP_HANDLE_NONEBUS, uds_parameter.PUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE)
        Console.WriteLine("PCAN-UDS API Version - {0}: {1}", buffer, STATUS_OK_KO(status))

        ' Initialize channel
        status = UDSApi.Initialize_2013(tp_handle, cantp_baudrate.PCANTP_BAUDRATE_500K)
        Console.WriteLine("Initialize channel: {0}", STATUS_OK_KO(status))

        ' Set server address parameter
        status = UDSApi.SetValue_2013(tp_handle, uds_parameter.PUDS_PARAMETER_SERVER_ADDRESS, server_address, CType(Marshal.SizeOf(server_address), UInt32))
        Console.WriteLine("Set server address: {0}", STATUS_OK_KO(status))

        ' Set timeout values
        timeout_value = CanTpApi.PCANTP_ISO_TIMEOUTS_15765_4
        status = UDSApi.SetValue_2013(tp_handle, uds_parameter.PUDS_PARAMETER_ISO_TIMEOUTS, timeout_value, CType(Marshal.SizeOf(timeout_value), UInt32))
        Console.WriteLine("Set ISO 15765-4 timeouts values: {0}", STATUS_OK_KO(status))

        ' Print timeout values
        status = UDSApi.GetValue_2013(tp_handle, uds_parameter.PUDS_PARAMETER_TIMEOUT_REQUEST, timeout_value, CType(Marshal.SizeOf(timeout_value), UInt32))
        Console.WriteLine("Get request timeout value ({0}ms): {1}", timeout_value, STATUS_OK_KO(status))
        status = UDSApi.GetValue_2013(tp_handle, uds_parameter.PUDS_PARAMETER_TIMEOUT_RESPONSE, timeout_value, CType(Marshal.SizeOf(timeout_value), UInt32))
        Console.WriteLine("Get response timeout value ({0}ms): {1}", timeout_value, STATUS_OK_KO(status))

        ' Set a receive event
        Dim receive_event As AutoResetEvent = New AutoResetEvent(False)

        If IntPtr.Size = 4 Then
            Dim tmp_buffer As UInt32 = Convert.ToUInt32(receive_event.SafeWaitHandle.DangerousGetHandle().ToInt32())
            status = UDSApi.SetValue_2013(tp_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, tmp_buffer, CType(Marshal.SizeOf(tmp_buffer), UInt32))

        ElseIf IntPtr.Size = 8 Then
            Dim tmp_buffer As Long = receive_event.SafeWaitHandle.DangerousGetHandle().ToInt64()
            Dim byte_array As Byte() = BitConverter.GetBytes(tmp_buffer)
            status = UDSApi.SetValue_2013(tp_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, byte_array, CType(Marshal.SizeOf(tmp_buffer), UInt64))

        End If

        Console.WriteLine("Set receive event parameter: {0}", STATUS_OK_KO(status))


        ' Read while user do not press Q
        Console.WriteLine("Start listening, press Q to quit.")
        stopit = False

        While Not stopit

            ' Wait a receive event on receiver
            '	note: timeout is used to check keyboard hit.
            wait_result = receive_event.WaitOne(2000)


            ' If we get a receive event
            If wait_result Then


                ' Read messages
                Do
                    read_status = UDSApi.Read_2013(tp_handle, request_msg)
                    Console.WriteLine(" Try to read a message (status={0}): {1}", CInt(read_status), STATUS_OK_KO(read_status))

                    If UDSApi.StatusIsOk_2013(read_status) Then
                        display_message(request_msg)

                        Dim req_isotp As cantp_msgdata_isotp = New cantp_msgdata_isotp()
                        If request_msg.msg.Msgdata <> IntPtr.Zero Then
                            req_isotp = request_msg.msg.Msgdata_isotp_Copy
                        End If


                        ' We receive a request, check if it is not a loopback message, if it is not a UUDT message, if the message is not pending, and do not respond if it is not ask
                        If (req_isotp.flags And cantp_msgflag.PCANTP_MSGFLAG_LOOPBACK) = 0 _
                            And (req_isotp.netaddrinfo.msgtype And cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION) = 0 _
                            And req_isotp.netstatus = cantp_netstatus.PCANTP_NETSTATUS_OK _
                            And (request_msg.type And uds_msgtype.PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE) <> uds_msgtype.PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE Then

                            ' Process response and send it
                            process_request(tp_handle, CType(server_address, UInt16), request_msg)
                        Else
                            Console.WriteLine()
                            Console.WriteLine("   ...Skipping response...  (flags=0x{0:X}, type=0x{1:X}, netstatus=0x{2:X}, msgtype=0x{3:X})", CInt(req_isotp.flags), CInt(request_msg.type), CInt(req_isotp.netstatus), CInt(req_isotp.netaddrinfo.msgtype))
                        End If
                    End If


                    ' Free request message (and clean memory in order to reallocate later)
                    status = UDSApi.MsgFree_2013(request_msg)
                    Console.WriteLine("Free request message: {0}", STATUS_OK_KO(status))
                Loop While Not UDSApi.StatusIsOk_2013(read_status, uds_status.PUDS_STATUS_NO_MESSAGE)
            End If


            ' Quit when user press Q
            If Console.KeyAvailable Then
                Dim keyboard_res As Char = Console.ReadKey().KeyChar

                If keyboard_res = "Q"c OrElse keyboard_res = "q"c Then
                    stopit = True
                End If
            End If
        End While

        ' Close receive event
        If IntPtr.Size = 4 Then
            Dim tmp_buffer As UInt32 = 0
            status = UDSApi.SetValue_2013(tp_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, tmp_buffer, CType(Marshal.SizeOf(tmp_buffer), UInt32))
        ElseIf IntPtr.Size = 8 Then
            Dim tmp_buffer As Long = 0
            Dim byte_array As Byte() = BitConverter.GetBytes(tmp_buffer)
            status = UDSApi.SetValue_2013(tp_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, byte_array, CType(Marshal.SizeOf(tmp_buffer), UInt64))
        End If
        Console.WriteLine("Stop receive event: {0}", STATUS_OK_KO(status))
        receive_event.Close()
        Console.WriteLine("Close receive event: {0}", STATUS_OK_KO(status))

        ' Close server
        status = UDSApi.Uninitialize_2013(tp_handle)
        Console.WriteLine("Uninitialize channel: {0}", STATUS_OK_KO(status))

        'Exit
        Console.WriteLine("Press any key to continue...")
        Console.In.Read()
    End Sub
End Module
