﻿Imports System
Imports System.Runtime.InteropServices
Imports System.Threading

Imports Peak.Can.Basic
Imports Peak.Can.IsoTp
Imports Peak.Can.Uds


Module Module1

    ' Console handling
    ReadOnly USE_GETCH As Boolean = False
    Sub CLEAR_CONSOLE()
        If USE_GETCH Then Console.Clear()
    End Sub

    ' A global counter to keep track of the number of failed tests (see display_uds_msg function)
    Dim g_nbErr As Integer = 0

    ''' <summary> A simple function that waits for user input</summary>
    Sub waitGetch()
        If USE_GETCH Then
            Console.WriteLine()
            Console.WriteLine(" Press <Enter> to continue...")
            Console.In.Read()
            CLEAR_CONSOLE()
        End If
    End Sub

    ''' <summary>A function that displays UDS Request and Response messages (and count error if no response)</summary>
    ''' <param name="request">Request message</param>
    ''' <param name="response">Received response message</param>
    ''' <param name="no_response_expected">if no response is expected, do not increment error counter</param>
    Sub display_uds_msg(ByRef request As uds_msg, ByRef response As uds_msg, ByVal no_response_expected As Boolean)
        display_uds_msg_request(request)
        display_uds_msg_response(response, no_response_expected)
    End Sub

    Sub display_uds_msg_response(ByRef response As uds_msg, ByVal no_response_expected As Boolean)
        If response.msg.Msgdata <> IntPtr.Zero Then
            Dim resp_isotp As cantp_msgdata_isotp = New cantp_msgdata_isotp()
            If response.msg.Msgdata <> IntPtr.Zero Then resp_isotp = response.msg.Msgdata_isotp_Copy
            Console.WriteLine()
            Console.WriteLine("UDS RESPONSE from 0x{0:X4} (to 0x{1:X4}, with extension address 0x{2:X2}) - result: {3} - {4}", resp_isotp.netaddrinfo.source_addr, resp_isotp.netaddrinfo.target_addr, resp_isotp.netaddrinfo.extension_addr, CInt(resp_isotp.netstatus), If(resp_isotp.netstatus <> cantp_netstatus.PCANTP_NETSTATUS_OK, "ERROR !!!", "OK !"))
            ' display data
            Console.Write("-> Length: {0}, Data= ", resp_isotp.length)

            For i As Integer = 0 To resp_isotp.length - 1
                Dim val As Byte = 0
                CanTpApi.getData_2016(response.msg, i, val)
                Console.Write("{0:X2} ", val)
            Next

            Console.WriteLine("")
        ElseIf Not no_response_expected Then
            Console.WriteLine()
            Console.WriteLine(" /!\ ERROR: NO UDS RESPONSE !!")
            g_nbErr += 1
        End If
    End Sub

    Sub display_uds_msg_request(ByRef request As uds_msg, ByVal no_response_expected As Boolean)
        display_uds_msg_request(request)

        If Not no_response_expected Then
            Console.WriteLine()
            Console.WriteLine(" /!\ ERROR: NO UDS RESPONSE !!")
            g_nbErr += 1
        End If
    End Sub

    Sub display_uds_msg_request(ByRef request As uds_msg)
        Dim req_isotp As cantp_msgdata_isotp = New cantp_msgdata_isotp()
        If request.msg.Msgdata <> IntPtr.Zero Then req_isotp = request.msg.Msgdata_isotp_Copy
        Console.WriteLine()
        Console.WriteLine("UDS request from 0x{0:X4} (to 0x{1:X4}, with extension address 0x{2:X2}) - result: {3} - {4}", req_isotp.netaddrinfo.source_addr, req_isotp.netaddrinfo.target_addr, req_isotp.netaddrinfo.extension_addr, CInt(req_isotp.netstatus), If(req_isotp.netstatus <> cantp_netstatus.PCANTP_NETSTATUS_OK, "ERROR !!!", "OK !"))
        ' display data
        Console.Write("-> Length: {0}, Data= ", req_isotp.length)

        For i As Integer = 0 To req_isotp.length - 1
            Dim val As Byte = 0
            CanTpApi.getData_2016(request.msg, i, val)
            Console.Write("{0:X2} ", val)
        Next
        Console.WriteLine("")
    End Sub

    ''' <summary>Helper: Inverts the bytes of a 32 bits numeric value</summary>
    ''' <param name="v">Value to revert</param>
    ''' <returns>Reverted value</returns>
    Function Reverse32(ByVal v As UInt32) As UInt32
        Return (v And &HFF) << 24 Or (v And &HFF00) << 8 Or (v And &HFF0000) >> 8 Or (v And &HFF000000) >> 24 And &HFF
    End Function

    Sub UInt32ToBytes(ByVal dw_buffer As UInt32, ByVal byte_buffer As Byte())
        byte_buffer(0) = CByte(dw_buffer And &HFF)
        byte_buffer(1) = CByte((dw_buffer And &HFF00) >> 8)
        byte_buffer(2) = CByte((dw_buffer And &HFF0000) >> 16)
        byte_buffer(3) = CByte((dw_buffer And &HFF000000) >> 24)
    End Sub

    ''' <summary>Call UDS Service DiagnosticSessionControl</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testDiagnosticSessionControl(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim session_info As uds_sessioninfo = New uds_sessioninfo()
        Dim request As uds_msg = New uds_msg()
        Dim response As uds_msg = New uds_msg()
        Dim confirmation As uds_msg = New uds_msg()

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service: DiagnosticSessionControl ***")

        ' Read default session information Server is not yet known (status will be PUDS_ERROR_NOT_INITIALIZED)
        ' yet the API will still set session info to default values
        session_info.nai = config.nai
        Dim session_size As Integer = Marshal.SizeOf(session_info)
        Dim session_ptr As IntPtr = Marshal.AllocHGlobal(session_size)
        Marshal.StructureToPtr(session_info, session_ptr, True)
        status = UDSApi.GetValue_2013(channel, uds_parameter.PUDS_PARAMETER_SESSION_INFO, session_ptr, CType(session_size, UInt32))
        session_info = CType(Marshal.PtrToStructure(session_ptr, GetType(uds_sessioninfo)), uds_sessioninfo)
        Marshal.FreeHGlobal(session_ptr)
        Console.WriteLine(" Diagnostic Session Information: {0}, 0x{1:X4} => {2} = [{3:X4}; {4:X4}]", CInt(status), session_info.nai.target_addr, session_info.session_type, session_info.timeout_p2can_server_max, session_info.timeout_enhanced_p2can_server_max)
        waitGetch()

        ' Set Diagnostic session to DEFAULT (to get session information)
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Setting a DEFAULT Diagnostic Session:")
        status = UDSApi.SvcDiagnosticSessionControl_2013(channel, config, request, UDSApi.uds_svc_param_dsc.PUDS_SVC_PARAM_DSC_DS)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcDiagnosticSessionControl_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))

        ' Read current session information
        session_info = New uds_sessioninfo()
        session_info.nai = config.nai
        session_ptr = Marshal.AllocHGlobal(session_size)
        Marshal.StructureToPtr(session_info, session_ptr, True)
        status = UDSApi.GetValue_2013(channel, uds_parameter.PUDS_PARAMETER_SESSION_INFO, session_ptr, CType(session_size, UInt32))
        session_info = CType(Marshal.PtrToStructure(session_ptr, GetType(uds_sessioninfo)), uds_sessioninfo)
        Marshal.FreeHGlobal(session_ptr)
        Console.WriteLine(" Diagnostic Session Information: {0}, 0x{1:X4} => {2} = [{3:X4}; {4:X4}]", CInt(status), session_info.nai.target_addr, session_info.session_type, session_info.timeout_p2can_server_max, session_info.timeout_enhanced_p2can_server_max)
        waitGetch()

        ' Set Diagnostic session to PROGRAMMING
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Setting a ECUPS Diagnostic Session:")
        status = UDSApi.SvcDiagnosticSessionControl_2013(channel, config, request, UDSApi.uds_svc_param_dsc.PUDS_SVC_PARAM_DSC_ECUPS)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcDiagnosticSessionControl_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))

        ' Read current session information
        session_info = New uds_sessioninfo()
        session_info.nai = config.nai
        session_ptr = Marshal.AllocHGlobal(session_size)
        Marshal.StructureToPtr(session_info, session_ptr, True)
        status = UDSApi.GetValue_2013(channel, uds_parameter.PUDS_PARAMETER_SESSION_INFO, session_ptr, CType(session_size, UInt32))
        session_info = CType(Marshal.PtrToStructure(session_ptr, GetType(uds_sessioninfo)), uds_sessioninfo)
        Marshal.FreeHGlobal(session_ptr)
        Console.WriteLine(" Diagnostic Session Information: {0}, 0x{1:X4} => {2} = [{3:X4}; {4:X4}]", CInt(status), session_info.nai.target_addr, session_info.session_type, session_info.timeout_p2can_server_max, session_info.timeout_enhanced_p2can_server_max)
        Console.WriteLine(" Assert that Auto TesterPresent Frame is sent...")
        Thread.Sleep(2000)
        Console.WriteLine(" Should transmit an Auto TesterPresent Frame")
        Thread.Sleep(2000)
        Console.WriteLine(" Should transmit an Auto TesterPresent Frame")
        waitGetch()

        ' Set Diagnostic session back to DEFAULT
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Setting a DEFAULT Diagnostic Session:")
        status = UDSApi.SvcDiagnosticSessionControl_2013(channel, config, request, UDSApi.uds_svc_param_dsc.PUDS_SVC_PARAM_DSC_DS)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcDiagnosticSessionControl_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        Console.WriteLine(" Assert that NO Auto TesterPresent Frame is sent...")
        Thread.Sleep(2000)
        Console.WriteLine(" Should NOT transmit an Auto TesterPresent Frame")
        Thread.Sleep(2000)
        Console.WriteLine(" Should NOT transmit an Auto TesterPresent Frame")
        waitGetch()
    End Sub


    ''' <summary>Call UDS Service ECUReset</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testECUReset(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim request As uds_msg = New uds_msg()
        Dim response As uds_msg = New uds_msg()
        Dim confirmation As uds_msg = New uds_msg()

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service: ECUReset ***")

        ' Sends a physical ECUReset message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical ECUReset message: ")
        status = UDSApi.SvcECUReset_2013(channel, config, request, UDSApi.uds_svc_param_er.PUDS_SVC_PARAM_ER_SR)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcECUReset_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If


        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()
    End Sub


    ''' <summary>Call UDS Service SecurityAccess</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testSecurityAccess(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim request As uds_msg = New uds_msg()
        Dim response As uds_msg = New uds_msg()
        Dim confirmation As uds_msg = New uds_msg()
        Dim dw_buffer As UInt32
        Dim security_access_data(3) As Byte
        Dim value_little_endian As UInt32

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service: SecurityAccess ***")

        ' Sends a physical SecurityAccess message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical SecurityAccess message: ")
        value_little_endian = &HF0A1B2C3UI
        dw_buffer = Reverse32(value_little_endian) ' use helper function to set MSB as 1st byte in the buffer (Win32 uses little endian format)
        UInt32ToBytes(dw_buffer, security_access_data)
        status = UDSApi.SvcSecurityAccess_2013(channel, config, request, UDSApi.PUDS_SVC_PARAM_SA_RSD_1, security_access_data, 4)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcSecurityAccess_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If


        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()
    End Sub


    ''' <summary>Call UDS Service CommunicationControl</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testCommunicationControl(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim request As uds_msg = New uds_msg()
        Dim response As uds_msg = New uds_msg()
        Dim confirmation As uds_msg = New uds_msg()

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service: CommunicationControl ***")

        ' Sends a physical CommunicationControl message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical CommunicationControl message: ")
        status = UDSApi.SvcCommunicationControl_2013(channel, config, request, UDSApi.uds_svc_param_cc.PUDS_SVC_PARAM_CC_ERXTX, UDSApi.PUDS_SVC_PARAM_CC_FLAG_APPL Or UDSApi.PUDS_SVC_PARAM_CC_FLAG_NWM Or UDSApi.PUDS_SVC_PARAM_CC_FLAG_DENWRIRO)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcCommunicationControl_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If


        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()
    End Sub


    ''' <summary>UDS Call Service TesterPresent</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testTesterPresent(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim request As uds_msg = New uds_msg()
        Dim tresponse(1) As uds_msg
        Dim response As uds_msg = New uds_msg()
        Dim confirmation As uds_msg = New uds_msg()
        Dim response_count As UInt32 = 0

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service: TesterPresent ***")

        ' Sends a physical TesterPresent message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical TesterPresent message: ")
        status = UDSApi.SvcTesterPresent_2013(channel, config, request, UDSApi.uds_svc_param_tp.PUDS_SVC_PARAM_TP_ZSUBF)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcTesterPresent_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()

        ' Sends a physical TesterPresent message with no positive response
        config.type = uds_msgtype.PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical TesterPresent message with no positive response:")
        status = UDSApi.SvcTesterPresent_2013(channel, config, request, UDSApi.uds_svc_param_tp.PUDS_SVC_PARAM_TP_ZSUBF)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcTesterPresent_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, True)
        End If

        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()

        ' Sends a functional TesterPresent message
        config.type = uds_msgtype.PUDS_MSGTYPE_USDT
        config.nai.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_FUNCTIONAL
        config.nai.target_addr = CType(uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_OBD_FUNCTIONAL, UInt16)
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a functional TesterPresent message: ")
        status = UDSApi.SvcTesterPresent_2013(channel, config, request, UDSApi.uds_svc_param_tp.PUDS_SVC_PARAM_TP_ZSUBF)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForServiceFunctional_2013(channel, request, 1, True, tresponse, response_count, confirmation)
        Console.WriteLine(" UDS_SvcTesterPresent_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(request, tresponse(0), False)
        Else
            display_uds_msg_request(request, False)
        End If

        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(tresponse(0))
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()

        ' Sends a functional TesterPresent message with no positive response
        config.type = uds_msgtype.PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE
        config.nai.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_FUNCTIONAL
        config.nai.target_addr = CType(uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_OBD_FUNCTIONAL, UInt16)
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a functional TesterPresent message with no positive response:")
        status = UDSApi.SvcTesterPresent_2013(channel, config, request, UDSApi.uds_svc_param_tp.PUDS_SVC_PARAM_TP_ZSUBF)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForServiceFunctional_2013(channel, request, 1, True, tresponse, response_count, confirmation)
        Console.WriteLine(" UDS_SvcTesterPresent_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) AndAlso response_count <> 0 Then
            display_uds_msg(request, tresponse(0), False)
        Else
            display_uds_msg_request(request, True)
        End If

        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(tresponse(0))
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()
    End Sub


    ''' <summary>Call UDS Service SecuredDataTransmission</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testSecuredDataTransmission(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim request As uds_msg = New uds_msg()
        Dim response As uds_msg = New uds_msg()
        Dim ecureset_request As uds_msg = New uds_msg()
        Dim confirmation As uds_msg = New uds_msg()
        Dim dw_buffer As UInt32
        Dim security_data_request_record(3) As Byte
        Dim value_little_endian As UInt32
        Dim administrative_parameter As UInt16
        Dim signature_encryption_calculation As Byte
        Dim anti_replay_counter As UInt16
        Dim internal_service_identifier As Byte
        Dim service_specific_parameters(4) As Byte
        Dim service_specific_parameters_size As UInt32
        Dim signature_mac(6) As Byte
        Dim signature_size As UInt16

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service: SecuredDataTransmission ***")

        ' Sends a physical SecuredDataTransmission/2013 message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical SecuredDataTransmission/2013 message: ")
        value_little_endian = &HF0A1B2C3UI
        dw_buffer = Reverse32(value_little_endian) ' use helper function to set MSB as 1st byte in the buffer (Win32 uses little endian format)
        UInt32ToBytes(dw_buffer, security_data_request_record)
        status = UDSApi.SvcSecuredDataTransmission_2013(channel, config, request, security_data_request_record, 4)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcSecuredDataTransmission_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If


        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()

        ' Sends a physical SecuredDataTransmission/2013 message prepared with PUDS_ONLY_PREPARE_REQUEST option
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical SecuredDataTransmission/2013 prepared with PUDS_ONLY_PREPARE_REQUEST option: ")
        status = UDSApi.SvcECUReset_2013(UDSApi.PUDS_ONLY_PREPARE_REQUEST, config, ecureset_request, UDSApi.uds_svc_param_er.PUDS_SVC_PARAM_ER_HR)
        Console.WriteLine(" Prepare ECUReset request for SecuredDataTransmission: {0}", CInt(status))
        If (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, False)) Then
            Dim ecureset_data(2) As Byte
            Dim i As UInt32
            For i = 0 To ecureset_request.msg.Msgdata_any_Copy.length - 1

                Dim val As Byte = 0
                CanTpApi.getData_2016(ecureset_request.msg, i, val)
                ecureset_data(i) = val
            Next
            status = UDSApi.SvcSecuredDataTransmission_2013(channel, config, request, ecureset_data, ecureset_request.msg.Msgdata_any_Copy.length)
        End If
        If (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, False)) Then
            status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        End If
        Console.WriteLine(" UDS_SvcSecuredDataTransmission_2013: {0}", CInt(status))
        If (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, False)) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        ' Free messages
        status = UDSApi.MsgFree_2013(ecureset_request)
        Console.WriteLine(" Free prepared message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()

        ' Sends a physical SecuredDataTransmission/2020 message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical SecuredDataTransmission/2020 message: ")
        administrative_parameter = UDSApi.PUDS_SVC_PARAM_APAR_REQUEST_MSG_FLAG Or UDSApi.PUDS_SVC_PARAM_APAR_REQUEST_RESPONSE_SIGNATURE_FLAG Or UDSApi.PUDS_SVC_PARAM_APAR_SIGNED_MSG_FLAG
        signature_encryption_calculation = &H0
        anti_replay_counter = &H124
        internal_service_identifier = &H2E
        service_specific_parameters(0) = &HF1
        service_specific_parameters(1) = &H23
        service_specific_parameters(2) = &HAA
        service_specific_parameters(3) = &H55
        service_specific_parameters_size = 4
        signature_mac(0) = &HDB
        signature_mac(1) = &HD1
        signature_mac(2) = &HE
        signature_mac(3) = &HDC
        signature_mac(4) = &H55
        signature_mac(5) = &HAA
        signature_size = &H6
        status = UDSApi.SvcSecuredDataTransmission_2020(channel, config, request, administrative_parameter, signature_encryption_calculation, anti_replay_counter, internal_service_identifier, service_specific_parameters, service_specific_parameters_size, signature_mac, signature_size)
        If (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, False)) Then
            status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        End If
        Console.WriteLine(" UDS_SvcSecuredDataTransmission_2013: {0}", CInt(status))
        If (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, False)) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()
    End Sub


    ''' <summary>Call UDS Service ControlDTCSetting</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testControlDTCSetting(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim request As uds_msg = New uds_msg()
        Dim response As uds_msg = New uds_msg()
        Dim confirmation As uds_msg = New uds_msg()
        Dim dw_buffer As UInt32
        Dim dtc_setting_control_option_record(3) As Byte
        Dim value_little_endian As UInt32

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service: ControlDTCSetting ***")

        ' Sends a physical ControlDTCSetting message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical ControlDTCSetting message: ")
        value_little_endian = &HF1A1B2EEUI
        dw_buffer = Reverse32(value_little_endian) ' use helper function to set MSB as 1st byte in the buffer (Win32 uses little endian format)
        UInt32ToBytes(dw_buffer, dtc_setting_control_option_record)
        status = UDSApi.SvcControlDTCSetting_2013(channel, config, request, UDSApi.uds_svc_param_cdtcs.PUDS_SVC_PARAM_CDTCS_OFF, dtc_setting_control_option_record, 3)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcControlDTCSetting_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If


        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()
    End Sub


    ''' <summary>Call UDS Service ResponseOnEvent</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testResponseOnEvent(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim request As uds_msg = New uds_msg()
        Dim response As uds_msg = New uds_msg()
        Dim confirmation As uds_msg = New uds_msg()
        Dim event_type_record(50) As Byte
        Dim service_to_respond_to_record(50) As Byte

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service: ResponseOnEvent ***")

        ' Sends a physical ResponseOnEvent message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical ResponseOnEvent message: ")
        event_type_record(0) = &H8
        service_to_respond_to_record(0) = CByte(uds_service.PUDS_SERVICE_SI_ReadDTCInformation)
        service_to_respond_to_record(1) = CByte(UDSApi.uds_svc_param_rdtci.PUDS_SVC_PARAM_RDTCI_RNODTCBSM)
        service_to_respond_to_record(2) = &H1
        status = UDSApi.SvcResponseOnEvent_2013(channel, config, request, UDSApi.uds_svc_param_roe.PUDS_SVC_PARAM_ROE_ONDTCS, False, &H8, event_type_record, UDSApi.PUDS_SVC_PARAM_ROE_OTI_LEN, service_to_respond_to_record, 3)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcResponseOnEvent_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If


        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()
    End Sub


    ''' <summary>Call UDS Service LinkControl</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testLinkControl(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim request As uds_msg = New uds_msg()
        Dim response As uds_msg = New uds_msg()
        Dim confirmation As uds_msg = New uds_msg()

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service: LinkControl ***")

        ' Sends a physical LinkControl message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical LinkControl message (Verify Fixed Baudrate): ")
        status = UDSApi.SvcLinkControl_2013(channel, config, request, UDSApi.uds_svc_param_lc.PUDS_SVC_PARAM_LC_VBTWFBR, UDSApi.uds_svc_param_lc_baudrate_identifier.PUDS_SVC_PARAM_LC_BAUDRATE_CAN_500K, 0)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcLinkControl_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))

        ' Sends a physical LinkControl message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical LinkControl message (Verify Specific Baudrate): ")
        status = UDSApi.SvcLinkControl_2013(channel, config, request, UDSApi.uds_svc_param_lc.PUDS_SVC_PARAM_LC_VBTWSBR, 0, 500000) ' 500K = 0x0007a120
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcLinkControl_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))

        ' Sends a physical LinkControl message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical LinkControl message (Transition): ")
        status = UDSApi.SvcLinkControl_2013(channel, config, request, UDSApi.uds_svc_param_lc.PUDS_SVC_PARAM_LC_TB, 0, 0)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcLinkControl_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()
    End Sub

    ''' <summary>Call UDS Service ReadDataByIdentifier</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testReadDataByIdentifier(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim request As uds_msg = New uds_msg()
        Dim response As uds_msg = New uds_msg()
        Dim confirmation As uds_msg = New uds_msg()
        Dim buffer As UDSApi.uds_svc_param_di() = {UDSApi.uds_svc_param_di.PUDS_SVC_PARAM_DI_ADSDID, UDSApi.uds_svc_param_di.PUDS_SVC_PARAM_DI_ECUMDDID}

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service: ReadDataByIdentifier ***")

        ' Sends a physical ReadDataByIdentifier message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical ReadDataByIdentifier message: ")
        status = UDSApi.SvcReadDataByIdentifier_2013(channel, config, request, buffer, 2)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcReadDataByIdentifier_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If


        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()
    End Sub

    ''' <summary>Call UDS Service ReadMemoryByAddress</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testReadMemoryByAddress(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim request As uds_msg = New uds_msg()
        Dim response As uds_msg = New uds_msg()
        Dim confirmation As uds_msg = New uds_msg()
        Dim memory_address_buffer(10) As Byte
        Dim memory_size_buffer(10) As Byte
        Dim memory_address_size As Byte = 10
        Dim memory_size_size As Byte = 3

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service: ReadMemoryByAddress ***")

        ' Sends a physical ReadMemoryByAddress message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical ReadMemoryByAddress message: ")

        For i As Integer = 0 To memory_address_size - 1
            memory_address_buffer(i) = Asc("A"c) + i
            memory_size_buffer(i) = Asc("1"c) + i
        Next

        status = UDSApi.SvcReadMemoryByAddress_2013(channel, config, request, memory_address_buffer, memory_address_size, memory_size_buffer, memory_size_size)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcReadMemoryByAddress_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If


        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()
    End Sub

    ''' <summary>Call UDS Service ReadScalingDataByIdentifier</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testReadScalingDataByIdentifier(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim request As uds_msg = New uds_msg()
        Dim response As uds_msg = New uds_msg()
        Dim confirmation As uds_msg = New uds_msg()

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service: ReadScalingDataByIdentifier ***")

        ' Sends a physical ReadScalingDataByIdentifier message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical ReadScalingDataByIdentifier message: ")
        status = UDSApi.SvcReadScalingDataByIdentifier_2013(channel, config, request, UDSApi.uds_svc_param_di.PUDS_SVC_PARAM_DI_BSFPDID)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcReadScalingDataByIdentifier_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If


        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()
    End Sub

    ''' <summary>Call UDS Service ReadDataByPeriodicIdentifier</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testReadDataByPeriodicIdentifier(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim request As uds_msg = New uds_msg()
        Dim response As uds_msg = New uds_msg()
        Dim confirmation As uds_msg = New uds_msg()
        Dim periodic_data_identifier(10) As Byte
        Dim periodic_data_identifier_size As UInt16 = 10

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service: ReadDataByPeriodicIdentifier ***")

        ' Sends a physical ReadScalingDataByIdentifier message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical ReadDataByPeriodicIdentifier message: ")

        For i As Integer = 0 To periodic_data_identifier_size - 1
            periodic_data_identifier(i) = Asc("A"c) + i
        Next

        status = UDSApi.SvcReadDataByPeriodicIdentifier_2013(channel, config, request, UDSApi.uds_svc_param_rdbpi.PUDS_SVC_PARAM_RDBPI_SAMR, periodic_data_identifier, periodic_data_identifier_size)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcReadDataByPeriodicIdentifier_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If


        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()
    End Sub

    ''' <summary>Call UDS Service DynamicallyDefineDataIdentifier</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testDynamicallyDefineDataIdentifier(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim request As uds_msg = New uds_msg()
        Dim response As uds_msg = New uds_msg()
        Dim confirmation As uds_msg = New uds_msg()
        Dim source_data_identifier(20) As UInt16
        Dim memory_size(20) As Byte
        Dim position_in_source_data_record(20) As Byte
        Dim number_of_elements As UInt16 = 10
        Dim memory_address_buffer(15) As Byte
        Dim memory_size_buffer(9) As Byte
        Dim memory_address_size As Byte
        Dim memory_size_size As Byte

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service: DynamicallyDefineDataIdentifier ***")

        ' Sends a physical DynamicallyDefineDataIdentifierDBID message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical DynamicallyDefineDataIdentifierDBID message: ")

        For i As Integer = 0 To number_of_elements - 1
            source_data_identifier(i) = CType((&HF0 + i << 8), UInt16) + Asc("A"c) + i
            memory_size(i) = CByte(i + 1)
            position_in_source_data_record(i) = CByte(100 + i)
        Next

        status = UDSApi.SvcDynamicallyDefineDataIdentifierDBID_2013(channel, config, request, UDSApi.uds_svc_param_di.PUDS_SVC_PARAM_DI_CDDID, source_data_identifier, memory_size, position_in_source_data_record, number_of_elements)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcDynamicallyDefineDataIdentifierDBID_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()

        ' Sends a physical UDS_SvcDynamicallyDefineDataIdentifierDBMA message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical UDS_SvcDynamicallyDefineDataIdentifierDBMA_2013 message: ")
        number_of_elements = 3
        memory_address_size = 5
        memory_size_size = 3

        For j As Integer = 0 To number_of_elements - 1

            For i As Integer = 0 To memory_address_size - 1
                memory_address_buffer(memory_address_size * j + i) = CByte(10 * j + i + 1)
            Next

            For i As Integer = 0 To memory_size_size - 1
                memory_size_buffer(memory_size_size * j + i) = CByte(100 + 10 * j + i + 1)
            Next
        Next

        status = UDSApi.SvcDynamicallyDefineDataIdentifierDBMA_2013(channel, config, request, UDSApi.uds_svc_param_di.PUDS_SVC_PARAM_DI_CESWNDID, memory_address_size, memory_size_size, memory_address_buffer, memory_size_buffer, number_of_elements)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcDynamicallyDefineDataIdentifierDBMA_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()

        ' Sends a physical UDS_SvcDynamicallyDefineDataIdentifierCDDDI message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical UDS_SvcDynamicallyDefineDataIdentifierCDDDI_2013 message: ")
        status = UDSApi.SvcDynamicallyDefineDataIdentifierCDDDI_2013(channel, config, request, UDSApi.uds_svc_param_di.PUDS_SVC_PARAM_DI_CESWNDID)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcDynamicallyDefineDataIdentifierCDDDI_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()

        ' Sends a physical UDS_SvcDynamicallyDefineDataIdentifierClearAllDDDI_2013 message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical UDS_SvcDynamicallyDefineDataIdentifierClearAllDDDI_2013 message: ")
        status = UDSApi.SvcDynamicallyDefineDataIdentifierClearAllDDDI_2013(channel, config, request)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcDynamicallyDefineDataIdentifierClearAllDDDI_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()
    End Sub


    ''' <summary>Call UDS Service WriteDataByIdentifier</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testWriteDataByIdentifier(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim request As uds_msg = New uds_msg()
        Dim response As uds_msg = New uds_msg()
        Dim confirmation As uds_msg = New uds_msg()
        Dim data_record(10) As Byte
        Dim data_record_size As UInt16 = 10

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service: WriteDataByIdentifier ***")

        ' Sends a physical WriteDataByIdentifier message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical WriteDataByIdentifier message: ")

        For i As Integer = 0 To data_record_size - 1
            data_record(i) = Asc("A"c) + i
        Next

        status = UDSApi.SvcWriteDataByIdentifier_2013(channel, config, request, UDSApi.uds_svc_param_di.PUDS_SVC_PARAM_DI_ASFPDID, data_record, data_record_size)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcWriteDataByIdentifier_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If


        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()
    End Sub

    ''' <summary>Call UDS Service WriteMemoryByAddress</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testWriteMemoryByAddress(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim request As uds_msg = New uds_msg()
        Dim response As uds_msg = New uds_msg()
        Dim confirmation As uds_msg = New uds_msg()
        Dim data_record(50) As Byte
        Dim memory_address_buffer(50) As Byte
        Dim memory_size_buffer(50) As Byte
        Dim data_record_size As UInt16 = 50
        Dim memory_address_size As Byte = 5
        Dim memory_size_size As Byte = 3

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service: WriteMemoryByAddress ***")

        ' Sends a physical WriteMemoryByAddress message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical WriteMemoryByAddress message: ")

        For i As Integer = 0 To data_record_size - 1
            data_record(i) = CByte(i + 1)
            memory_address_buffer(i) = Asc("A"c) + i
            memory_size_buffer(i) = CByte(10 + i)
        Next

        status = UDSApi.SvcWriteMemoryByAddress_2013(channel, config, request, memory_address_buffer, memory_address_size, memory_size_buffer, memory_size_size, data_record, data_record_size)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcWriteMemoryByAddress_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If


        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()
    End Sub


    ''' <summary>Call UDS Service ClearDiagnosticInformation</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testClearDiagnosticInformation(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim request As uds_msg = New uds_msg()
        Dim response As uds_msg = New uds_msg()
        Dim confirmation As uds_msg = New uds_msg()

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service: ClearDiagnosticInformation ***")

        ' Sends a physical ClearDiagnosticInformation message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical ClearDiagnosticInformation message: ")
        status = UDSApi.SvcClearDiagnosticInformation_2013(channel, config, request, &HF1A2B3)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcClearDiagnosticInformation_2013: {0}", CInt(status))
        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()


        ' Sends a physical ClearDiagnosticInformation message with memory selection parameter (2020)
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical ClearDiagnosticInformation message with memory selection parameter: ")
        status = UDSApi.SvcClearDiagnosticInformation_2020(channel, config, request, &HF1A2B3, &H42)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcClearDiagnosticInformation_2020: {0}", CInt(status))
        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()
    End Sub


    ''' <summary>Call UDS Service ReadDTCInformation</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testReadDTCInformation(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim request As uds_msg = New uds_msg()
        Dim response As uds_msg = New uds_msg()
        Dim confirmation As uds_msg = New uds_msg()

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service: ReadDTCInformation ***")

        ' Sends a physical ReadDTCInformation message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical ReadDTCInformation message: ")
        status = UDSApi.SvcReadDTCInformation_2013(channel, config, request, UDSApi.uds_svc_param_rdtci.PUDS_SVC_PARAM_RDTCI_RNODTCBSM, &HF1)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcReadDTCInformation_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()

        ' Sends a physical ReadDTCInformationRDTCSSBDTC message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical ReadDTCInformationRDTCSSBDTC message: ")
        status = UDSApi.SvcReadDTCInformationRDTCSSBDTC_2013(channel, config, request, &HA1B2B3, &H12)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcReadDTCInformationRDTCSSBDTC_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()

        ' Sends a physical ReadDTCInformationRDTCSSBRN message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical ReadDTCInformationRDTCSSBRN message: ")
        status = UDSApi.SvcReadDTCInformationRDTCSSBRN_2013(channel, config, request, &H12)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcReadDTCInformationRDTCSSBRN_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()

        ' Sends a physical ReadDTCInformationReportExtended message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical ReadDTCInformationReportExtended message: ")
        status = UDSApi.SvcReadDTCInformationReportExtended_2013(channel, config, request, UDSApi.uds_svc_param_rdtci.PUDS_SVC_PARAM_RDTCI_RDTCEDRBDN, &HA1B2B3, &H12)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcReadDTCInformationReportExtended_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()

        ' Sends a physical ReadDTCInformationReportSeverity message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical ReadDTCInformationReportSeverity message: ")
        status = UDSApi.SvcReadDTCInformationReportSeverity_2013(channel, config, request, UDSApi.uds_svc_param_rdtci.PUDS_SVC_PARAM_RDTCI_RNODTCBSMR, &HF1, &H12)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcReadDTCInformationReportSeverity_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()

        ' Sends a physical ReadDTCInformationRSIODTC message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical ReadDTCInformationRSIODTC message: ")
        status = UDSApi.SvcReadDTCInformationRSIODTC_2013(channel, config, request, &HF1A1B2B3UI)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcReadDTCInformationRSIODTC_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()

        ' Sends a physical ReadDTCInformationNoParam message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical ReadDTCInformationNoParam message: ")
        status = UDSApi.SvcReadDTCInformationNoParam_2013(channel, config, request, UDSApi.uds_svc_param_rdtci.PUDS_SVC_PARAM_RDTCI_RSUPDTC)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcReadDTCInformationNoParam_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()

        '' Sends a physical UDS_SvcReadDTCInformationRDTCEDBR_2013 message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical UDS_SvcReadDTCInformationRDTCEDBR_2013 message: ")
        status = UDSApi.SvcReadDTCInformationRDTCEDBR_2013(channel, config, request, &H12)
        If (UDSApi.StatusIsOk_2013(status)) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcReadDTCInformationRDTCEDBR_2013: {0}", CInt(status))
        If (UDSApi.StatusIsOk_2013(status)) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()

        '' Sends a physical UDS_SvcReadDTCInformationRUDMDTCBSM_2013 message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical UDS_SvcReadDTCInformationRUDMDTCBSM_2013 message: ")
        status = UDSApi.SvcReadDTCInformationRUDMDTCBSM_2013(channel, config, request, &H12, &H34)
        If (UDSApi.StatusIsOk_2013(status)) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcReadDTCInformationRUDMDTCBSM_2013: {0}", CInt(status))
        If (UDSApi.StatusIsOk_2013(status)) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()

        '' Sends a physical UDS_SvcReadDTCInformationRUDMDTCSSBDTC_2013 message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical UDS_SvcReadDTCInformationRUDMDTCSSBDTC_2013 message: ")
        status = UDSApi.SvcReadDTCInformationRUDMDTCSSBDTC_2013(channel, config, request, &H123456, &H78, &H9A)
        If (UDSApi.StatusIsOk_2013(status)) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcReadDTCInformationRUDMDTCSSBDTC_2013: {0}", CInt(status))
        If (UDSApi.StatusIsOk_2013(status)) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()

        '' Sends a physical UDS_SvcReadDTCInformationRUDMDTCEDRBDN_2013 message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical UDS_SvcReadDTCInformationRUDMDTCEDRBDN_2013 message: ")
        status = UDSApi.SvcReadDTCInformationRUDMDTCEDRBDN_2013(channel, config, request, &H123456, &H78, &H9A)
        If (UDSApi.StatusIsOk_2013(status)) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcReadDTCInformationRUDMDTCEDRBDN_2013: {0}", CInt(status))
        If (UDSApi.StatusIsOk_2013(status)) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()

        '' Sends a physical UDS_SvcReadDTCInformationRDTCEDI_2020 message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical UDS_SvcReadDTCInformationRDTCEDI_2020 message: ")
        status = UDSApi.SvcReadDTCInformationRDTCEDI_2020(channel, config, request, &H12)
        If (UDSApi.StatusIsOk_2013(status)) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcReadDTCInformationRDTCEDI_2020: {0}", CInt(status))
        If (UDSApi.StatusIsOk_2013(status)) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()

        '' Sends a physical UDS_SvcReadDTCInformationRWWHOBDDTCBMR_2013 message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical UDS_SvcReadDTCInformationRWWHOBDDTCBMR_2013 message: ")
        status = UDSApi.SvcReadDTCInformationRWWHOBDDTCBMR_2013(channel, config, request, &H12, &H34, &H56)
        If (UDSApi.StatusIsOk_2013(status)) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcReadDTCInformationRWWHOBDDTCBMR_2013: {0}", CInt(status))
        If (UDSApi.StatusIsOk_2013(status)) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()

        '' Sends a physical UDS_SvcReadDTCInformationRWWHOBDDTCWPS_2013 message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical UDS_SvcReadDTCInformationRWWHOBDDTCWPS_2013 message: ")
        status = UDSApi.SvcReadDTCInformationRWWHOBDDTCWPS_2013(channel, config, request, &H12)
        If (UDSApi.StatusIsOk_2013(status)) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcReadDTCInformationRWWHOBDDTCWPS_2013: {0}", CInt(status))
        If (UDSApi.StatusIsOk_2013(status)) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()

        '' Sends a physical UDS_SvcReadDTCInformationRDTCBRGI_2020 message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical UDS_SvcReadDTCInformationRDTCBRGI_2020 message: ")
        status = UDSApi.SvcReadDTCInformationRDTCBRGI_2020(channel, config, request, &H12, &H34)
        If (UDSApi.StatusIsOk_2013(status)) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcReadDTCInformationRDTCBRGI_2020: {0}", CInt(status))
        If (UDSApi.StatusIsOk_2013(status)) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
    End Sub

    ''' <summary>Call UDS Service InputOutputControlByIdentifier</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testInputOutputControlByIdentifier(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim request As uds_msg = New uds_msg()
        Dim response As uds_msg = New uds_msg()
        Dim confirmation As uds_msg = New uds_msg()
        Dim control_option_record(10) As Byte
        Dim control_enable_mask_record(10) As Byte
        Dim control_option_record_size As UInt16 = 10
        Dim control_enable_mask_record_size As UInt16 = 5

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service: InputOutputControlByIdentifier ***")

        ' Sends a physical InputOutputControlByIdentifier message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical InputOutputControlByIdentifier message: ")

        For i As Integer = 0 To control_option_record_size - 1
            control_option_record(i) = Asc("A"c) + i
            control_enable_mask_record(i) = CByte(10 + i)
        Next

        status = UDSApi.SvcInputOutputControlByIdentifier_2013(channel, config, request, UDSApi.uds_svc_param_di.PUDS_SVC_PARAM_DI_SSECUSWVNDID, control_option_record, control_option_record_size, control_enable_mask_record, control_enable_mask_record_size)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcInputOutputControlByIdentifier_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If


        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()
    End Sub

    ''' <summary>Call UDS Service RoutineControl</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testRoutineControl(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim request As uds_msg = New uds_msg()
        Dim response As uds_msg = New uds_msg()
        Dim confirmation As uds_msg = New uds_msg()
        Dim routine_control_option_record(10) As Byte
        Dim routine_control_option_record_size As UInt16 = 10

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service: RoutineControl ***")

        ' Sends a physical RoutineControl message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical RoutineControl message: ")

        For i As Integer = 0 To routine_control_option_record_size - 1
            routine_control_option_record(i) = Asc("A"c) + i
        Next

        status = UDSApi.SvcRoutineControl_2013(channel, config, request, UDSApi.uds_svc_param_rc.PUDS_SVC_PARAM_RC_RRR, CType(&HF1A2, UDSApi.uds_svc_param_rc_rid), routine_control_option_record, routine_control_option_record_size)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcRoutineControl_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If


        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()
    End Sub


    ''' <summary>Call UDS Service RequestDownload</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testRequestDownload(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim request As uds_msg = New uds_msg()
        Dim response As uds_msg = New uds_msg()
        Dim confirmation As uds_msg = New uds_msg()
        Dim memory_address_buffer(15) As Byte
        Dim memory_size_buffer(15) As Byte
        Dim memory_address_size As Byte = 15
        Dim memory_size_size As Byte = 8

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service: RequestDownload ***")

        ' Sends a physical RequestDownload message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical RequestDownload message: ")

        For i As Integer = 0 To memory_address_size - 1
            memory_address_buffer(i) = Asc("A"c) + i
            memory_size_buffer(i) = CByte(10 + i)
        Next

        status = UDSApi.SvcRequestDownload_2013(channel, config, request, &H1, &H2, memory_address_buffer, memory_address_size, memory_size_buffer, memory_size_size)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcRequestDownload_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If


        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()
    End Sub


    ''' <summary>Call UDS Service RequestUpload</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testRequestUpload(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim request As uds_msg = New uds_msg()
        Dim response As uds_msg = New uds_msg()
        Dim confirmation As uds_msg = New uds_msg()
        Dim memory_address_buffer(4) As Byte
        Dim memory_size_buffer(4) As Byte
        Dim memory_address_size As Byte = 4
        Dim memory_size_size As Byte = 4

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service: RequestUpload ***")

        ' Sends a physical RequestUpload message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical RequestUpload message: ")

        For i As Integer = 0 To memory_size_size - 1
            memory_address_buffer(i) = Asc("A"c) + i
            memory_size_buffer(i) = CByte(10 + i)
        Next

        status = UDSApi.SvcRequestUpload_2013(channel, config, request, &H1, &H2, memory_address_buffer, memory_address_size, memory_size_buffer, memory_size_size)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcRequestUpload_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If


        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()
    End Sub

    ''' <summary>Call UDS Service RequestTransferExit</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testRequestTransferExit(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim request As uds_msg = New uds_msg()
        Dim response As uds_msg = New uds_msg()
        Dim confirmation As uds_msg = New uds_msg()
        Dim record(20) As Byte
        Dim record_size As Byte = 20

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service: RequestTransferExit ***")

        ' Sends a physical RequestTransferExit message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical RequestTransferExit message: ")

        For i As Integer = 0 To record_size - 1
            record(i) = Asc("A"c) + i
        Next

        status = UDSApi.SvcRequestTransferExit_2013(channel, config, request, record, record_size)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcRequestTransferExit_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If


        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()
    End Sub

    ''' <summary>Call UDS Service TransferData</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testTransferData(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim request As uds_msg = New uds_msg()
        Dim response As uds_msg = New uds_msg()
        Dim confirmation As uds_msg = New uds_msg()
        Dim record(50) As Byte
        Dim record_size As Byte = 50

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service: TransferData ***")

        ' Sends a physical TransferData message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical TransferData message: ")

        For i As Integer = 0 To record_size - 1
            record(i) = Asc("A"c) + i
        Next

        status = UDSApi.SvcTransferData_2013(channel, config, request, &H1, record, record_size)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcTransferData_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If


        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()
    End Sub

    ''' <summary>Call UDS Service TransferData with MAX_DATA length</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testTransferDataBigMessage(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim request As uds_msg = New uds_msg()
        Dim response As uds_msg = New uds_msg()
        Dim confirmation As uds_msg = New uds_msg()
        Dim record(4096) As Byte
        Dim record_size As UInt16 = 4093

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service: TransferData with MAX_DATA ***")

        ' Sends a physical TransferData message with the maximum data available. The goal is to show that
        ' WaitForService_2013 does not return a TIMEOUT error although the transmit and receive time of all the
        ' data will be longer than the default time to get a response.
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical TransferData message (length={0}): ", record_size)

        For i As Integer = 0 To record_size - 1
            record(i) = (Asc("A"c) + i) And &HFF
        Next

        status = UDSApi.SvcTransferData_2013(channel, config, request, &H1, record, record_size)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcTransferData_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If


        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()
    End Sub

    ''' <summary>Call UDS Service TransferData</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testTransferDataMultipleFunctionalMessage(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim request As uds_msg = New uds_msg()
        Dim confirmation As uds_msg = New uds_msg()
        Dim tresponse(1) As uds_msg
        Dim record(4) As Byte
        Dim record_size As UInt16 = 5
        Dim response_count As UInt32 = 0

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service: TransferData with functional message***")

        ' Initialize request message
        config.nai.target_addr = CType(uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_OBD_FUNCTIONAL, UInt16)
        config.nai.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_FUNCTIONAL

        ' Sends a functional TransferData message. The goal is to show that UDS_WaitForServiceFunctional_2013 waits long
        ' enough to fetch all possible ECU responses.
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a functional TransferData message: ")

        For i As Integer = 0 To record_size - 1
            record(i) = Asc("A"c) + i
        Next

        status = UDSApi.SvcTransferData_2013(channel, config, request, &H1, record, record_size)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForServiceFunctional_2013(channel, request, 1, True, tresponse, response_count, confirmation)
        Console.WriteLine(" UDS_SvcTransferData_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg_request(request, True)
            Console.WriteLine()
            Console.WriteLine(" Received {0} UDS responses:", response_count)
            display_uds_msg_response(tresponse(0), False)
        Else
            display_uds_msg_request(request, False)
        End If


        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(tresponse(0))
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()
    End Sub


    ''' <summary>Sample to use event</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testUsingEvent(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim read_status As uds_status
        Dim request As uds_msg = New uds_msg()
        Dim response As uds_msg = New uds_msg()
        Dim receive_event As System.Threading.AutoResetEvent
        Dim tmp_event_handle As UInt32
        Dim res As Boolean
        Dim stop_it As Boolean

        ' set event handler
        receive_event = New System.Threading.AutoResetEvent(False)
        tmp_event_handle = Convert.ToUInt32(receive_event.SafeWaitHandle.DangerousGetHandle().ToInt32())
        status = UDSApi.SetValue_2013(channel, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, tmp_event_handle, CType(Len(tmp_event_handle), UInt32))
        If Not UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, False) Then
            Console.WriteLine("Failed to set event, aborting...")
            receive_event.Close()
            waitGetch()
            Return
        End If

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service with Event: TesterPresent ***")

        ' Sends a physical TesterPresent message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical TesterPresent message: ")
        status = UDSApi.SvcTesterPresent_2013(channel, config, request, UDSApi.uds_svc_param_tp.PUDS_SVC_PARAM_TP_ZSUBF)
        Console.WriteLine(" UDS_SvcTesterPresent_2013: {0}", CType(status, UInt32))

        If (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, False)) Then

            ' Instead of calling WaitForService function, this sample demonstrates how event can be used. But note that
            ' the use of a thread listening to notifications and making the read operations is preferred.
            stop_it = False

            ' wait until we receive expected response
            Do
                res = receive_event.WaitOne(System.Threading.Timeout.Infinite)
                If res Then

                    ' read all messages
                    Do
                        read_status = UDSApi.Read_2013(channel, response)
                        If UDSApi.StatusIsOk_2013(read_status, uds_status.PUDS_STATUS_OK, False) Then
                            ' this is a simple message check (type and sender/receiver address): to filter UDS request
                            ' confirmation and get first message from target, but real use-case should check that the UDS
                            ' service ID matches the request
                            If (response.msg.Msgdata_isotp_Copy.netaddrinfo.msgtype = cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC And response.msg.Msgdata_isotp_Copy.netaddrinfo.source_addr = config.nai.target_addr And response.msg.Msgdata_isotp_Copy.netaddrinfo.target_addr = config.nai.source_addr) Then
                                stop_it = True
                                display_uds_msg(request, response, False)
                            End If
                        End If

                        ' Free response message
                        status = UDSApi.MsgFree_2013(response)
                        Console.WriteLine(" Free response message: {0}", CInt(status))
                    Loop While Not UDSApi.StatusIsOk_2013(read_status, uds_status.PUDS_STATUS_NO_MESSAGE)
                End If
            Loop While Not stop_it
        End If

        waitGetch()

        ' Free request message
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))

        ' Uninitialize event
        tmp_event_handle = 0
        status = UDSApi.SetValue_2013(channel, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, tmp_event_handle, CType(Len(tmp_event_handle), UInt32))
        Console.WriteLine(" Remove receive event: {0}", CInt(status))
        receive_event.Close()
    End Sub

    ''' <summary>Call UDS Service RequestFileTransfer</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testRequestFileTransfer(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim request As uds_msg = New uds_msg()
        Dim response As uds_msg = New uds_msg()
        Dim confirmation As uds_msg = New uds_msg()
        Dim file_size_uncompressed As Byte() = {&HD, &H0}
        Dim file_size_compressed As Byte() = {&HA, &H0}
        Dim file_size_length As Byte = 2
        Dim sfile_name As String = "toto.txt"
        Dim file_name_length As UInt16 = sfile_name.Length

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service: RequestFileTransfer ***")

        ' Sends a physical RequestFileTransfer message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical RequestFileTransfer message: ")
        status = UDSApi.SvcRequestFileTransfer_2013(channel, config, request, UDSApi.uds_svc_param_rft_moop.PUDS_SVC_PARAM_RFT_MOOP_REPLFILE, file_name_length, sfile_name, 0, 0, file_size_length, file_size_uncompressed, file_size_compressed)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcRequestFileTransfer_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If


        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()
    End Sub


    ''' <summary>Call UDS Service AccessTimingParameter</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testAccessTimingParameter(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim request As uds_msg = New uds_msg()
        Dim response As uds_msg = New uds_msg()
        Dim confirmation As uds_msg = New uds_msg()
        Dim request_record As Byte() = {&HAB, &HCD}
        Dim record_size As UInt32 = 2

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service: AccessTimingParameter ***")

        ' Sends a physical AccessTimingParameter message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical AccessTimingParameter message: ")
        status = UDSApi.SvcAccessTimingParameter_2013(channel, config, request, UDSApi.uds_svc_param_atp.PUDS_SVC_PARAM_ATP_RCATP, request_record, record_size)
        If UDSApi.StatusIsOk_2013(status) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcAccessTimingParameter_2013: {0}", CInt(status))

        If UDSApi.StatusIsOk_2013(status) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If


        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()
    End Sub

    ''' <summary>Call UDS Service Authentication</summary>
    ''' <param name="channel">cantp channel handle</param>
    ''' <param name="config">Configuration of the request message (type, network address information...)</param>
    Sub testAuthentication(ByVal channel As cantp_handle, ByVal config As uds_msgconfig)
        Dim status As uds_status
        Dim request As uds_msg = New Peak.Can.Uds.uds_msg()
        Dim response As uds_msg = New uds_msg()
        Dim confirmation As uds_msg = New uds_msg()

        Dim communication_configuration As Byte = 0
        Dim certificate_client() As Byte = {&H12, &H34}
        Dim certificate_client_size As UInt16 = 2
        Dim challenge_client() As Byte = {&H56, &H78}
        Dim challenge_client_size As UInt16 = 2
        Dim proof_of_ownership_client() As Byte = {&H9A, &HBC}
        Dim proof_of_ownership_client_size As UInt16 = 2
        Dim ephemeral_public_key_client() As Byte = {&HDE, &HF0}
        Dim ephemeral_public_key_client_size As UInt16 = 2
        Dim algorithm_indicator() As Byte = {&H0, &H1, &H2, &H3, &H4, &H5, &H6, &H7, &H8, &H9, &HA, &HB, &HC, &HD, &HE, &HF}
        Dim additional_parameter() As Byte = {&HAA, &HBB}
        Dim additional_parameter_size As UInt16 = 2

        CLEAR_CONSOLE()
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("*** UDS Service: Authentication ***")

        ' Sends a physical Authentication/deAuthenticate message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical Authentication/deAuthenticate message: ")
        status = UDSApi.SvcAuthenticationDA_2020(channel, config, request)
        If (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, False)) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcAuthenticationDA_2020: {0}", CInt(status))
        If (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, False)) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()


        ' Sends a physical Authentication/verifyCertificateUnidirectional message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical Authentication/verifyCertificateUnidirectional message: ")
        status = UDSApi.SvcAuthenticationVCU_2020(channel, config, request, communication_configuration, certificate_client, certificate_client_size, challenge_client, challenge_client_size)
        If (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, False)) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcAuthenticationVCU_2020: {0}", CInt(status))
        If (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, False)) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()


        ' Sends a physical Authentication/verifyCertificateBidirectional message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical Authentication/verifyCertificateBidirectional message: ")
        status = UDSApi.SvcAuthenticationVCB_2020(channel, config, request, communication_configuration, certificate_client, certificate_client_size, challenge_client, challenge_client_size)
        If (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, False)) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcAuthenticationVCB_2020: {0}", CInt(status))
        If (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, False)) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()


        ' Sends a physical Authentication/proofOfOwnership message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical Authentication/proofOfOwnership message: ")
        status = UDSApi.SvcAuthenticationPOWN_2020(channel, config, request, proof_of_ownership_client, proof_of_ownership_client_size, ephemeral_public_key_client, ephemeral_public_key_client_size)
        If (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, False)) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcAuthenticationPOWN_2020: {0}", CInt(status))
        If (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, False)) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()


        ' Sends a physical Authentication/requestChallengeForAuthentication message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical Authentication/requestChallengeForAuthentication message: ")
        status = UDSApi.SvcAuthenticationRCFA_2020(channel, config, request, communication_configuration, algorithm_indicator)
        If (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, False)) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcAuthenticationRCFA_2020: {0}", CInt(status))
        If (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, False)) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()


        ' Sends a physical Authentication/verifyProofOfOwnershipUnidirectional message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical Authentication/verifyProofOfOwnershipUnidirectional message: ")
        status = UDSApi.SvcAuthenticationVPOWNU_2020(channel, config, request, algorithm_indicator, proof_of_ownership_client, proof_of_ownership_client_size, challenge_client, challenge_client_size, additional_parameter, additional_parameter_size)
        If (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, False)) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcAuthenticationVPOWNU_2020: {0}", CInt(status))
        If (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, False)) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()


        ' Sends a physical Authentication/verifyProofOfOwnershipBidirectional message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical Authentication/verifyProofOfOwnershipBidirectional message: ")
        status = UDSApi.SvcAuthenticationVPOWNB_2020(channel, config, request, algorithm_indicator, proof_of_ownership_client, proof_of_ownership_client_size, challenge_client, challenge_client_size, additional_parameter, additional_parameter_size)
        If (UDSApi.StatusIsOk_2013(status)) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcAuthenticationVPOWNB_2020: {0}", CInt(status))
        If (UDSApi.StatusIsOk_2013(status)) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()


        ' Sends a physical Authentication/authenticationConfiguration message
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Sends a physical Authentication/authenticationConfiguration message: ")
        status = UDSApi.SvcAuthenticationAC_2020(channel, config, request)
        If (UDSApi.StatusIsOk_2013(status)) Then status = UDSApi.WaitForService_2013(channel, request, response, confirmation)
        Console.WriteLine(" UDS_SvcAuthenticationAC_2020: {0}", CInt(status))
        If (UDSApi.StatusIsOk_2013(status)) Then
            display_uds_msg(confirmation, response, False)
        Else
            display_uds_msg_request(request, False)
        End If

        ' Free messages
        status = UDSApi.MsgFree_2013(request)
        Console.WriteLine(" Free request message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(response)
        Console.WriteLine(" Free response message: {0}", CInt(status))
        status = UDSApi.MsgFree_2013(confirmation)
        Console.WriteLine(" Free confirmation message: {0}", CInt(status))
        waitGetch()
    End Sub

    ''' <summary>Entry point of the program, start a CAN UDS client</summary>
    Sub Main(ByVal args As String())
        Dim client_handle As cantp_handle
        Dim status As uds_status
        Dim timeout_value As UInt32
        Dim config As uds_msgconfig = New uds_msgconfig()

        ' Set the PCAN-Channel to use
        client_handle = cantp_handle.PCANTP_HANDLE_USBBUS1 ' TODO: modify the value according to your available PCAN devices.

        ' Initializing of the UDS Communication session
        status = UDSApi.Initialize_2013(client_handle, cantp_baudrate.PCANTP_BAUDRATE_500K)
        Console.WriteLine("Initialize UDS: {0}", CInt(status))

        ' Define TimeOuts
        timeout_value = CanTpApi.PCANTP_ISO_TIMEOUTS_15765_4
        status = UDSApi.SetValue_2013(client_handle, uds_parameter.PUDS_PARAMETER_ISO_TIMEOUTS, timeout_value, CType(Marshal.SizeOf(timeout_value), UInt32))
        Console.WriteLine("Set ISO 15765-4 timeouts values: {0}", CInt(status))
        waitGetch()

        ' Define Network Address Information used for all the tests
        config.can_id = &HFFFFFFFFUI
        config.can_msgtype = cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD
        config.nai.protocol = uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL
        config.nai.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL
        config.type = uds_msgtype.PUDS_MSGTYPE_USDT
        config.nai.source_addr = CType(uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT, UInt16)
        config.nai.target_addr = CType(uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_1, UInt16)

        ' The following functions call UDS Services
        testDiagnosticSessionControl(client_handle, config)
        testECUReset(client_handle, config)
        testSecurityAccess(client_handle, config)
        testCommunicationControl(client_handle, config)
        testTesterPresent(client_handle, config)
        testSecuredDataTransmission(client_handle, config)
        testControlDTCSetting(client_handle, config)
        testResponseOnEvent(client_handle, config)
        testLinkControl(client_handle, config)
        testReadDataByIdentifier(client_handle, config)
        testReadMemoryByAddress(client_handle, config)
        testReadScalingDataByIdentifier(client_handle, config)
        testReadDataByPeriodicIdentifier(client_handle, config)
        testDynamicallyDefineDataIdentifier(client_handle, config)
        testWriteDataByIdentifier(client_handle, config)
        testWriteMemoryByAddress(client_handle, config)
        testClearDiagnosticInformation(client_handle, config)
        testReadDTCInformation(client_handle, config)
        testInputOutputControlByIdentifier(client_handle, config)
        testRoutineControl(client_handle, config)
        testRequestDownload(client_handle, config)
        testRequestUpload(client_handle, config)
        testTransferData(client_handle, config)
        testRequestTransferExit(client_handle, config)
        testAccessTimingParameter(client_handle, config)
        testRequestFileTransfer(client_handle, config)
        testAuthentication(client_handle, config)

        ' Miscellaneous examples
        testTransferDataBigMessage(client_handle, config)
        testTransferDataMultipleFunctionalMessage(client_handle, config)
        testUsingEvent(client_handle, config)

        ' Uninitialize channel
        status = UDSApi.Uninitialize_2013(client_handle)
        Console.WriteLine(" Unitialize channel: {0}", CInt(status))


        ' Display a small report and quit
        If g_nbErr > 0 Then
            Console.WriteLine()
            Console.WriteLine("ERROR: {0} errors occurred.", g_nbErr)
        Else
            Console.WriteLine()
            Console.WriteLine("ALL Transmissions succeeded !")
        End If

        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Press <Enter> to quit...")
        Console.In.Read()
    End Sub
End Module
