﻿Imports System
Imports System.Text

Imports Peak.Can.Basic
Imports Peak.Can.IsoTp
Imports Peak.Can.Uds
Module Module1

    ' Define Bitrate: SAE J2284-4: High-Speed CAN (HSC) for Vehicle Applications at 500 kbps with CAN FD Data at 2 Mbps
    Const PCAN_BITRATE_SAE_J2284_4 As String = "f_clock=80000000,nom_brp=2,nom_tseg1=63,nom_tseg2=16,nom_sjw=16,data_brp=2,data_tseg1=15,data_tseg2=4,data_sjw=4"

    ''' <summary>Helper: Check if a mapping list contains a certain mapping</summary>
    ''' <param name="mapping_list">Mapping list</param>
    ''' <param name="mapping_list_size">Number of mapping in the mapping list</param>
    ''' <param name="searched_mapping">Searched mapping</param>
    ''' <returns>Present (true) or not (false)</returns>
    Function mapping_list_contains(ByRef mapping_list As uds_mapping(), ByVal mapping_list_size As UShort, ByRef searched_mapping As uds_mapping) As Boolean
        Dim res As Boolean = False

        For i As Integer = 0 To mapping_list_size - 1
            ' If unique identifier are the same, the mapping is in the list
            If mapping_list(i).uid = searched_mapping.uid Then
                res = True
                Exit For
            End If
        Next

        Return res
    End Function

    ''' <summary>Entry point of the program, start a small CAN UDS read/write example</summary>
    Sub Main(ByVal args As String())
        Const MSG As String = "PEAK"
        Const MSG_SIZE As UInteger = 4
        Const MAPPING_BUFFER_SIZE As UShort = 256
        Const BUFFER_SIZE As Integer = 256
        Dim buffer As StringBuilder = New StringBuilder(BUFFER_SIZE)
        Dim status As uds_status
        Dim transmitter_handle As cantp_handle
        Dim receiver_handle As cantp_handle
        Dim tx_msg As uds_msg = New uds_msg()
        Dim rx_msg As uds_msg = New uds_msg()
        Dim msg_config As uds_msgconfig = New uds_msgconfig()
        Dim source_mapping As uds_mapping = New uds_mapping()
        Dim response_mapping As uds_mapping = New uds_mapping()
        Dim mapping_buffer As uds_mapping() = New uds_mapping(MAPPING_BUFFER_SIZE - 1) {}
        Dim mapping_count As UShort
        Dim wait_result As Boolean

        ' Initialize variables
        transmitter_handle = cantp_handle.PCANTP_HANDLE_USBBUS1 ' TODO: modify the value according to your available PCAN devices.
        receiver_handle = cantp_handle.PCANTP_HANDLE_USBBUS2 ' TODO: modify the value according to your available PCAN devices.
        mapping_count = 0

        ' Print version informations
        status = UDSApi.GetValue_2013(cantp_handle.PCANTP_HANDLE_NONEBUS, uds_parameter.PUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE)
        Console.WriteLine("PCAN-UDS API Version: {0} ({1})", buffer, STATUS_OK_KO(status))

        ' Initialize transmitter and receiver
        status = UDSApi.InitializeFD_2013(transmitter_handle, PCAN_BITRATE_SAE_J2284_4)
        Console.WriteLine("Initialize transmitter: {0}", STATUS_OK_KO(status))
        status = UDSApi.InitializeFD_2013(receiver_handle, PCAN_BITRATE_SAE_J2284_4)
        Console.WriteLine("Initialize receiver: {0}", STATUS_OK_KO(status))

        ' Create a receive event on receiver
        Dim receive_event As Threading.AutoResetEvent = New Threading.AutoResetEvent(False)

        If IntPtr.Size = 4 Then
            Dim tmp_buffer As UInteger = Convert.ToUInt32(receive_event.SafeWaitHandle.DangerousGetHandle().ToInt32())
            status = UDSApi.SetValue_2013(receiver_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, tmp_buffer, CType(System.Runtime.InteropServices.Marshal.SizeOf(tmp_buffer), UInt32))
        ElseIf IntPtr.Size = 8 Then
            Dim tmp_buffer As Long = receive_event.SafeWaitHandle.DangerousGetHandle().ToInt64()
            Dim byte_array As Byte() = BitConverter.GetBytes(tmp_buffer)
            status = UDSApi.SetValue_2013(receiver_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, byte_array, CType(System.Runtime.InteropServices.Marshal.SizeOf(tmp_buffer), UInt64))
        End If
        Console.WriteLine("Set receive event parameter: {0}", STATUS_OK_KO(status))

        ' Initialize 29bits
        msg_config.can_id = &H30
        msg_config.can_msgtype = cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD
        ' To use CAN FD with bitrate switch uncomment:
        ' msg_config.can_msgtype = msg_config.can_msgtype Or cantp_can_msgtype.PCANTP_CAN_MSGTYPE_FD Or cantp_can_msgtype.PCANTP_CAN_MSGTYPE_BRS
        msg_config.nai.protocol = uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_29B_NORMAL
        msg_config.nai.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL
        msg_config.type = uds_msgtype.PUDS_MSGTYPE_USDT
        msg_config.nai.source_addr = &H10
        msg_config.nai.target_addr = &H20

        ' Allocate TX message with previous configuration
        status = UDSApi.MsgAlloc_2013(tx_msg, msg_config, MSG_SIZE)
        Dim resb As Boolean = False
        If UDSApi.StatusIsOk_2013(status) Then
            Dim tx_byte_array As Byte() = New Byte(MSG_SIZE - 1) {}

            For i As Integer = 0 To MSG_SIZE - 1
                tx_byte_array(i) = Microsoft.VisualBasic.AscW(MSG(i))
            Next

            resb = CanTpApi.setData_2016(tx_msg.msg, 0, tx_byte_array, MSG_SIZE)
        End If
        Console.WriteLine("Allocate TX message: {0}", OK_KO(UDSApi.StatusIsOk_2013(status) And resb))

        ' Initialize source mapping
        source_mapping.can_id = msg_config.can_id
        source_mapping.can_id_flow_ctrl = msg_config.can_id + 1
        source_mapping.can_msgtype = msg_config.can_msgtype
        source_mapping.can_tx_dlc = 8
        source_mapping.nai = msg_config.nai

        ' Initialize response mapping
        response_mapping = source_mapping
        response_mapping.can_id = source_mapping.can_id_flow_ctrl
        response_mapping.can_id_flow_ctrl = source_mapping.can_id
        response_mapping.nai.source_addr = source_mapping.nai.target_addr
        response_mapping.nai.target_addr = source_mapping.nai.source_addr

        ' Add mappings on transmitter
        status = UDSApi.AddMapping_2013(transmitter_handle, source_mapping)
        Console.WriteLine("Add source mapping on transmitter: {0}", STATUS_OK_KO(status))
        status = UDSApi.AddMapping_2013(transmitter_handle, response_mapping)
        Console.WriteLine("Add response mapping on transmitter: {0}", STATUS_OK_KO(status))

        ' Check if mappings are added on transmitter
        status = UDSApi.GetMappings_2013(transmitter_handle, mapping_buffer, MAPPING_BUFFER_SIZE, mapping_count)
        Console.WriteLine("Get mappings on transmitter: {0}", STATUS_OK_KO(status))
        Console.WriteLine("Check added mappings on transmitter: {0}", OK_KO(mapping_list_contains(mapping_buffer, mapping_count, source_mapping) AndAlso mapping_list_contains(mapping_buffer, mapping_count, response_mapping)))

        ' Add mappings on receiver
        status = UDSApi.AddMapping_2013(receiver_handle, source_mapping)
        Console.WriteLine("Add source mapping on receiver: {0}", STATUS_OK_KO(status))
        status = UDSApi.AddMapping_2013(receiver_handle, response_mapping)
        Console.WriteLine("Add response mapping on receiver: {0}", STATUS_OK_KO(status))

        ' Check if mappings are added on receiver
        status = UDSApi.GetMappings_2013(receiver_handle, mapping_buffer, MAPPING_BUFFER_SIZE, mapping_count)
        Console.WriteLine("Get mappings on receiver: {0}", STATUS_OK_KO(status))
        Console.WriteLine("Check added mappings on receiver: {0}", OK_KO(mapping_list_contains(mapping_buffer, mapping_count, source_mapping) AndAlso mapping_list_contains(mapping_buffer, mapping_count, response_mapping)))

        ' Write message
        status = UDSApi.Write_2013(transmitter_handle, tx_msg)
        Console.WriteLine("Write ""PEAK"": {0}", STATUS_OK_KO(status))

        ' Wait a receive event
        wait_result = receive_event.WaitOne(3000)
        Console.WriteLine("Wait a receive event: {0}", OK_KO(wait_result))


        ' If we get a receive event
        If wait_result Then

            ' Read message
            status = UDSApi.Read_2013(receiver_handle, rx_msg)
            Console.WriteLine("Receive message: {0}", STATUS_OK_KO(status))

            ' Check message (should contains "PEAK")
            Dim msgok As Boolean = False
            Dim rx_byte_array As Byte() = New Byte(MSG_SIZE - 1) {}

            If UDSApi.StatusIsOk_2013(status) AndAlso rx_msg.msg.Msgdata_any_Copy.length = MSG_SIZE Then
                msgok = CanTpApi.getData_2016(rx_msg.msg, 0, rx_byte_array, MSG_SIZE)
                If msgok Then
                    Dim i As Integer = 0
                    While msgok AndAlso i < MSG_SIZE
                        If Chr(rx_byte_array(i)) = MSG(i) Then
                            msgok = True
                        Else
                            msgok = False
                        End If
                        i += 1
                    End While
                End If
            End If

            If msgok Then
                Console.WriteLine("Message contains ""{0}{1}{2}{3}"": OK", Microsoft.VisualBasic.ChrW(rx_byte_array(0)), Microsoft.VisualBasic.ChrW(rx_byte_array(1)), Microsoft.VisualBasic.ChrW(rx_byte_array(2)), Microsoft.VisualBasic.ChrW(rx_byte_array(3)))
            Else
                Console.WriteLine("Message is corrupted: KO" & vbLf)
            End If
        End If


        ' Close receive event
        If IntPtr.Size = 4 Then
            Dim tmp_buffer As UInteger = 0
            status = UDSApi.SetValue_2013(receiver_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, tmp_buffer, CType(System.Runtime.InteropServices.Marshal.SizeOf(tmp_buffer), UInt32))
        ElseIf IntPtr.Size = 8 Then
            Dim tmp_buffer As Long = 0
            Dim byte_array As Byte() = BitConverter.GetBytes(tmp_buffer)
            status = UDSApi.SetValue_2013(receiver_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, byte_array, CType(System.Runtime.InteropServices.Marshal.SizeOf(tmp_buffer), UInt64))
        End If

        Console.WriteLine("Stop receive event receiver: {0}", STATUS_OK_KO(status))
        receive_event.Close()
        Console.WriteLine("Close receive event: {0}", STATUS_OK_KO(status))

        ' Free messages
        status = UDSApi.MsgFree_2013(tx_msg)
        Console.WriteLine("Free TX message: {0}", STATUS_OK_KO(status))
        status = UDSApi.MsgFree_2013(rx_msg)
        Console.WriteLine("Free RX message: {0}", STATUS_OK_KO(status))

        ' Close transmitter and receiver
        status = UDSApi.Uninitialize_2013(transmitter_handle)
        Console.WriteLine("Uninitialize transmitter: {0}", STATUS_OK_KO(status))
        status = UDSApi.Uninitialize_2013(receiver_handle)
        Console.WriteLine("Uninitialize receiver: {0}", STATUS_OK_KO(status))

        ' Exit
        Console.WriteLine("Press any key to continue...")
        Console.In.Read()
    End Sub

    Function OK_KO(ByVal test As Boolean) As String
        If test Then Return "OK"
        Return "KO"
    End Function

    Function STATUS_OK_KO(ByVal test As uds_status) As String
        Return OK_KO(UDSApi.StatusIsOk_2013(test))
    End Function
End Module
