﻿Imports System
Imports System.Runtime.InteropServices
Imports System.Threading
Imports System.Text

Imports Peak.Can.Basic
Imports Peak.Can.IsoTp
Imports Peak.Can.Uds
Imports cantp_timestamp = System.UInt64

Module Module1
    Const BUFFER_SIZE As Integer = 256
    Const ISOTP_REQUEST_MSG As String = "PING"
    Const ISOTP_RESPONSE_MSG As String = "PONG"
    Const MSG_SIZE As Integer = 4

    ''' <summary>Structure passed as thread parameters</summary>
    Public Structure task_params
        ''' <summary>Server channel handle</summary>
        Public server_handle As cantp_handle
        ''' <summary>Server address</summary>
        Public server_address As UInt32
        ''' <summary>Determine if the thread should end or not</summary>
        Public stop_task As Boolean
    End Structure

    ''' <summary>ISOTP server task: respond "PING" message by sending "PONG"</summary>
    ''' <param name="parameters">pointer on task_params structures</param>
    Sub isotp_server_task(ByRef parameters As task_params)

        ' Init variables
        Dim rx_msg As cantp_msg = New cantp_msg()
        Dim tx_msg As cantp_msg = New cantp_msg()
        Dim mapping As cantp_mapping = New cantp_mapping()
        Dim reverse_mapping As cantp_mapping = New cantp_mapping()
        Dim timestamp_buffer As cantp_timestamp
        Dim wait_result As Boolean
        Dim status As cantp_status = cantp_status.PCANTP_STATUS_UNKNOWN

        ' Create a isotp receive event
        Dim receive_event As System.Threading.AutoResetEvent = New System.Threading.AutoResetEvent(False)
        If (IntPtr.Size = 4) Then
            Dim tmp_buffer As UInt32 = Convert.ToUInt32(receive_event.SafeWaitHandle.DangerousGetHandle().ToInt32())
            status = CanTpApi.SetValue_2016(parameters.server_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, tmp_buffer, CType(System.Runtime.InteropServices.Marshal.SizeOf(tmp_buffer), UInt32))
        ElseIf (IntPtr.Size = 8) Then
            Dim tmp_buffer As Int64 = receive_event.SafeWaitHandle.DangerousGetHandle().ToInt64()
            Dim byte_array As Byte() = BitConverter.GetBytes(tmp_buffer)
            status = CanTpApi.SetValue_2016(parameters.server_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, byte_array, CType(System.Runtime.InteropServices.Marshal.SizeOf(tmp_buffer), UInt64))
        End If
        Console.WriteLine("[ISOTP] Set isotp receive event parameter: {0}", ISOTP_STATUS_OK_KO(status))

        ' Create a simple isotp physical mapping:
        '    - Source 0xA1 (client), target 0xA2 (server), CAN id 0xA1, CAN ID flow control 0xA2
        '    - Diagnostic message in a classic format
        mapping.can_id = &HA1
        mapping.can_id_flow_ctrl = &HA2
        mapping.can_msgtype = cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD
        mapping.netaddrinfo.extension_addr = &H0
        mapping.netaddrinfo.format = cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL
        mapping.netaddrinfo.msgtype = cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC
        mapping.netaddrinfo.source_addr = &HA1
        mapping.netaddrinfo.target_addr = &HA2
        mapping.netaddrinfo.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL

        ' Create the associated isotp reversed mapping:
        reverse_mapping = mapping
        reverse_mapping.can_id = mapping.can_id_flow_ctrl
        reverse_mapping.can_id_flow_ctrl = mapping.can_id
        reverse_mapping.netaddrinfo.source_addr = mapping.netaddrinfo.target_addr
        reverse_mapping.netaddrinfo.target_addr = mapping.netaddrinfo.source_addr

        ' Add ISOTP mappings on channels
        status = CanTpApi.AddMapping_2016(parameters.server_handle, mapping)
        Console.WriteLine("[ISOTP] Add a simple isotp mapping: {0}", ISOTP_STATUS_OK_KO(status))
        status = CanTpApi.AddMapping_2016(parameters.server_handle, reverse_mapping)
        Console.WriteLine("[ISOTP] Add the reverse isotp mapping: {0}", ISOTP_STATUS_OK_KO(status))

        ' Initialize ISOTP Tx message containing "PING"
        status = CanTpApi.MsgDataAlloc_2016(tx_msg, cantp_msgtype.PCANTP_MSGTYPE_ISOTP)
        Console.WriteLine("[ISOTP] Allocate ISOTP tx message: {0}", ISOTP_STATUS_OK_KO(status))
        Dim bytes_array As Byte() = Encoding.ASCII.GetBytes(ISOTP_RESPONSE_MSG)
        status = CanTpApi.MsgDataInit_2016(tx_msg, reverse_mapping.can_id, reverse_mapping.can_msgtype, MSG_SIZE, bytes_array, reverse_mapping.netaddrinfo)
        Console.WriteLine("[ISOTP] Initialize ISOTP tx message: {0}", ISOTP_STATUS_OK_KO(status))

        ' Wait a receive event on isotp message
        Do
            wait_result = receive_event.WaitOne(1000)
            Console.WriteLine("[ISOTP] Wait a receive event from isotp: {0}", OK_KO(wait_result))

            ' Read ISOTP messages
            Do
                status = CanTpApi.Read_2016(parameters.server_handle, rx_msg, timestamp_buffer, cantp_msgtype.PCANTP_MSGTYPE_NONE)
                Console.WriteLine("[ISOTP] Read ISOTP message: {0}", ISOTP_STATUS_OK_KO(status))

                ' Check message (should contains "PING") and send "PONG"
                If Not CanTpApi.StatusIsOk_2016(status, cantp_status.PCANTP_STATUS_NO_MESSAGE) Then
                    Dim msgok As Boolean = False
                    Dim rx_byte_array As Byte() = New Byte(MSG_SIZE - 1) {}
                    If (CanTpApi.StatusIsOk_2016(status, cantp_status.PCANTP_STATUS_OK, False) And (rx_msg.Msgdata_any_Copy.length = MSG_SIZE)) Then
                        msgok = CanTpApi.getData_2016(rx_msg, 0, rx_byte_array, MSG_SIZE)
                        If msgok Then
                            Dim i As Integer = 0
                            While msgok AndAlso i < MSG_SIZE
                                If Chr(rx_byte_array(i)) = ISOTP_REQUEST_MSG(i) Then
                                    msgok = True
                                Else
                                    msgok = False
                                End If
                                i += 1
                            End While
                        End If
                    End If
                    If (msgok) Then
                        Console.WriteLine("[ISOTP] Received message contains ""{0}{1}{2}{3}"": OK", Microsoft.VisualBasic.ChrW(rx_byte_array(0)), Microsoft.VisualBasic.ChrW(rx_byte_array(1)), Microsoft.VisualBasic.ChrW(rx_byte_array(2)), Microsoft.VisualBasic.ChrW(rx_byte_array(3)))
                        status = CanTpApi.Write_2016(parameters.server_handle, tx_msg)
                        Console.WriteLine("[ISOTP] Send ISOTP ""PONG"" message: {0}", ISOTP_STATUS_OK_KO(status))
                    End If

                    ' Free RX message
                    status = CanTpApi.MsgDataFree_2016(rx_msg)
                    Console.WriteLine("[ISOTP] Free RX message: {0}", ISOTP_STATUS_OK_KO(status))
                End If
            Loop While Not CanTpApi.StatusIsOk_2016(status, cantp_status.PCANTP_STATUS_NO_MESSAGE)
        Loop While parameters.stop_task = False

        ' Close receive event
        If IntPtr.Size = 4 Then
            Dim tmp_buffer As UInt32 = 0
            status = CanTpApi.SetValue_2016(parameters.server_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, tmp_buffer, CType(System.Runtime.InteropServices.Marshal.SizeOf(tmp_buffer), UInt32))
        ElseIf IntPtr.Size = 8 Then
            Dim tmp_buffer As Int64 = 0
            Dim byte_array As Byte() = BitConverter.GetBytes(tmp_buffer)
            status = CanTpApi.SetValue_2016(parameters.server_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, byte_array, CType(System.Runtime.InteropServices.Marshal.SizeOf(tmp_buffer), UInt64))
        End If
        Console.WriteLine("[ISOTP] Stop ISOTP receive event: {0}", ISOTP_STATUS_OK_KO(status))
        receive_event.Close()
        Console.WriteLine("[ISOTP] Close ISOTP receive event: {0}", ISOTP_STATUS_OK_KO(status))

        ' Free messages
        status = CanTpApi.MsgDataFree_2016(tx_msg)
        Console.WriteLine("[ISOTP] Free ISOTP TX message: {0}", ISOTP_STATUS_OK_KO(status))
    End Sub

    ''' <summary>UDS server task: respond TesterPresent request</summary>
    ''' <param name="parameters">pointer on task_params structures</param>
    Sub uds_server_task(ByRef parameters As task_params)

        ' Init variables
        Dim config_physical As uds_msgconfig = New uds_msgconfig()
        Dim request_msg As uds_msg = New uds_msg()
        Dim response_msg As uds_msg = New uds_msg()
        Dim wait_result As Boolean
        Dim status As uds_status
        Dim read_status As uds_status

        ' Set server address parameter
        status = UDSApi.SetValue_2013(parameters.server_handle, uds_parameter.PUDS_PARAMETER_SERVER_ADDRESS, parameters.server_address, CType(System.Runtime.InteropServices.Marshal.SizeOf(parameters.server_address), UInt32))
        Console.WriteLine("[UDS] Set UDS server address: {0}", UDS_STATUS_OK_KO(status))

        ' Set a receive event for UDS
        Dim receive_event As System.Threading.AutoResetEvent = New System.Threading.AutoResetEvent(False)
        If (IntPtr.Size = 4) Then
            Dim tmp_buffer As UInt32 = Convert.ToUInt32(receive_event.SafeWaitHandle.DangerousGetHandle().ToInt32())
            status = UDSApi.SetValue_2013(parameters.server_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, tmp_buffer, CType(System.Runtime.InteropServices.Marshal.SizeOf(parameters.server_address), UInt32))
        ElseIf (IntPtr.Size = 8) Then
            Dim tmp_buffer As Int64 = receive_event.SafeWaitHandle.DangerousGetHandle().ToInt64()
            Dim byte_array As Byte() = BitConverter.GetBytes(tmp_buffer)
            status = UDSApi.SetValue_2013(parameters.server_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, byte_array, CType(System.Runtime.InteropServices.Marshal.SizeOf(parameters.server_address), UInt64))
        End If
        Console.WriteLine("[UDS] Set UDS receive event parameter: {0}", UDS_STATUS_OK_KO(status))

        ' Initialize a physical configuration
        config_physical.can_id = &HFFFFFFFFUI
        config_physical.can_msgtype = cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD
        config_physical.nai.protocol = uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL
        config_physical.nai.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL
        config_physical.type = uds_msgtype.PUDS_MSGTYPE_USDT
        config_physical.nai.extension_addr = 0

        Do

            ' Wait a receive event on receiver
            wait_result = receive_event.WaitOne(1000)

            ' If we get a receive event
            If (wait_result) Then

                Do

                    ' Read first available message (no filtering based on message's type is set):
                    read_status = UDSApi.Read_2013(parameters.server_handle, request_msg)

                    Console.WriteLine("[UDS] Try to read a message: {0}", UDS_STATUS_OK_KO(read_status))
                    If (UDSApi.StatusIsOk_2013(read_status)) Then

                        ' We receive a request, check its length and if it is not a loopback message and if it is a USDT message
                        If (request_msg.type = uds_msgtype.PUDS_MSGTYPE_USDT) And (request_msg.msg.Msgdata_any_Copy.length >= 1) And ((request_msg.msg.Msgdata_any_Copy.flags And cantp_msgflag.PCANTP_MSGFLAG_LOOPBACK) = 0) Then

                            ' This is a valid request, switch services
                            Dim service_id As Byte
                            Dim get_sid As Boolean = UDSApi.GetDataServiceId_2013(request_msg, service_id)
                            If get_sid Then
                                Select Case CType(service_id, uds_service)
                                    Case uds_service.PUDS_SERVICE_SI_TesterPresent

                                        ' Allocate response message
                                        status = UDSApi.MsgAlloc_2013(response_msg, config_physical, 2)
                                        Console.WriteLine("[UDS] Prepare response message for TesterPresent service: {0}", UDS_STATUS_OK_KO(status))

                                        If UDSApi.StatusIsOk_2013(status) Then
                                            ' Fill parameters
                                            Dim nadr As cantp_netaddrinfo = request_msg.msg.Msgdata_isotp_Copy.netaddrinfo
                                            nadr.target_addr = nadr.source_addr
                                            nadr.source_addr = parameters.server_address
                                            CanTpApi.setNetaddrinfo_2016(response_msg.msg, nadr)
                                            UDSApi.SetDataServiceId_2013(response_msg, uds_service.PUDS_SERVICE_SI_TesterPresent + UDSApi.PUDS_SI_POSITIVE_RESPONSE)
                                            UDSApi.SetDataParameter_2013(response_msg, 0, 0)

                                            ' Write response message
                                            status = UDSApi.Write_2013(parameters.server_handle, response_msg)
                                            Console.WriteLine("[UDS] Write response message for TesterPresent service: {0}", UDS_STATUS_OK_KO(status))
                                        End If

                                        ' Free response message (and clean memory in order to reallocate later)
                                        status = UDSApi.MsgFree_2013(response_msg)
                                        Console.WriteLine("[UDS] Free response message: {0}", UDS_STATUS_OK_KO(status))
                                    Case Else
                                        Console.WriteLine("[UDS] Unknown service (0x{0:x2})", service_id)
                                End Select
                            Else
                                Console.WriteLine("[UDS] Cannot get service identifier")
                            End If
                        End If
                    End If

                    ' Free request message (and clean memory in order to reallocate later)
                    status = UDSApi.MsgFree_2013(request_msg)

                    Console.WriteLine("[UDS] Free request message: {0}", UDS_STATUS_OK_KO(status))
                Loop While Not UDSApi.StatusIsOk_2013(read_status, uds_status.PUDS_STATUS_NO_MESSAGE)
            End If
        Loop While parameters.stop_task = False

        ' Close receive event
        If IntPtr.Size = 4 Then
            Dim tmp_buffer As UInt32 = 0
            status = UDSApi.SetValue_2013(parameters.server_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, tmp_buffer, CType(Marshal.SizeOf(tmp_buffer), UInt32))
        ElseIf IntPtr.Size = 8 Then
            Dim tmp_buffer As Int64 = 0
            Dim byte_array As Byte() = BitConverter.GetBytes(tmp_buffer)
            status = UDSApi.SetValue_2013(parameters.server_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, byte_array, CType(Marshal.SizeOf(tmp_buffer), UInt64))
        End If

        Console.WriteLine("[UDS] Stop UDS receive event: {0}", UDS_STATUS_OK_KO(status))
        receive_event.Close()
        Console.WriteLine("[UDS] Close UDS receive event: {0}", UDS_STATUS_OK_KO(status))
    End Sub

    ''' <summary>Entry point of the program, start a small server which handle UDS testerpresent request and isotp request</summary>
    ''' <returns>By convention, return success.</returns>
    Function Main(ByVal args As String()) As Integer
        Dim status As uds_status
        Dim uds_server As Thread
        Dim isotp_server As Thread
        Dim t_params As task_params
        Dim Buffer As StringBuilder = New StringBuilder(BUFFER_SIZE)

        ' Initialize variables
        t_params.server_handle = cantp_handle.PCANTP_HANDLE_USBBUS2 ' TODO: modify the value according to your available PCAN devices.
        t_params.server_address = uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_1
        t_params.stop_task = False

        ' Print version informations
        status = UDSApi.GetValue_2013(cantp_handle.PCANTP_HANDLE_NONEBUS, uds_parameter.PUDS_PARAMETER_API_VERSION, Buffer, BUFFER_SIZE)

        Console.WriteLine("PCAN-UDS API Version - {0}: {1}", Buffer, UDS_STATUS_OK_KO(status))

        ' Initialize server
        status = UDSApi.Initialize_2013(t_params.server_handle, cantp_baudrate.PCANTP_BAUDRATE_500K, 0, 0, 0)

        Console.WriteLine("Initialize channel: {0}", UDS_STATUS_OK_KO(status))

        ' Start uds and isotp servers
        uds_server = New Thread(Sub() uds_server_task(t_params))
        uds_server.Start()
        isotp_server = New Thread(Sub() isotp_server_task(t_params))
        isotp_server.Start()

        ' Read while user do not press Q
        Console.WriteLine("Start listening, press Q to quit.")
        t_params.stop_task = False
        Do

            ' Quit when user press Q
            If Console.KeyAvailable Then
                Dim keyboard_res As Char = Console.ReadKey().KeyChar

                If keyboard_res = "Q"c OrElse keyboard_res = "q"c Then
                    t_params.stop_task = True
                End If
            End If
        Loop While t_params.stop_task = False

        ' Close threads
        isotp_server.Join()
        uds_server.Join()


        ' Close channel
        status = UDSApi.Uninitialize_2013(t_params.server_handle)
        Console.WriteLine("Uninitialize channel: {0}", UDS_STATUS_OK_KO(status))

        ' Exit
        Console.WriteLine("Press any key to continue...")
        Console.In.Read()
        Return 0
    End Function

    Function OK_KO(test As Boolean) As String
        If test Then Return "OK"
        Return "KO"
    End Function
    Function UDS_STATUS_OK_KO(test As uds_status) As String
        Return OK_KO(UDSApi.StatusIsOk_2013(test))
    End Function
    Function ISOTP_STATUS_OK_KO(test As cantp_status) As String
        Return OK_KO(CanTpApi.StatusIsOk_2016(test))
    End Function
End Module
