﻿Imports System
Imports System.Runtime.InteropServices
Imports System.Threading
Imports System.Text

Imports Peak.Can.Basic
Imports Peak.Can.IsoTp
Imports Peak.Can.Uds
Imports cantp_timestamp = System.UInt64

Module Module1
    Const BUFFER_SIZE As Integer = 256
    Const ISOTP_REQUEST_MSG As String = "PING"
    Const ISOTP_RESPONSE_MSG As String = "PONG"
    Const MSG_SIZE As Integer = 4

    ''' <summary>UDS client task: request TesterPresent service several time</summary>
    ''' <param name="param">client channel handle</param>
    Sub uds_client_task(param As cantp_handle)

        ' Initialize variables
        Dim config_physical As uds_msgconfig = New uds_msgconfig()
        Dim msg_request As uds_msg = New uds_msg()
        Dim request_confirmation As uds_msg = New uds_msg()
        Dim response As uds_msg = New uds_msg()
        Dim client_handle As cantp_handle = param
        Dim status As uds_status
        Dim i As Integer

        ' Initialize a physical configuration
        config_physical.can_id = uds_can_id.PUDS_CAN_ID_ISO_15765_4_PHYSICAL_REQUEST_1
        config_physical.can_msgtype = cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD
        config_physical.nai.protocol = uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL
        config_physical.nai.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL
        config_physical.type = uds_msgtype.PUDS_MSGTYPE_USDT
        config_physical.nai.source_addr = uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT
        config_physical.nai.target_addr = uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_1
        config_physical.nai.extension_addr = 0

        ' Execute TesterPresent and wait response
        For i = 1 To 10
            status = UDSApi.SvcTesterPresent_2013(client_handle, config_physical, msg_request)
            Console.WriteLine("[UDS] Execute TesterPresent service ({0}): {1}", i, UDS_STATUS_OK_KO(status))
            status = UDSApi.WaitForService_2013(client_handle, msg_request, response, request_confirmation)
            Console.WriteLine("[UDS] Wait for service ({0}): {1}", i, UDS_STATUS_OK_KO(status))
            status = UDSApi.MsgFree_2013(msg_request)
            Console.WriteLine("[UDS] Free request message: {0}", UDS_STATUS_OK_KO(status))
            status = UDSApi.MsgFree_2013(request_confirmation)
            Console.WriteLine("[UDS] Free request confirmation message: {0}", UDS_STATUS_OK_KO(status))
            status = UDSApi.MsgFree_2013(response)
            Console.WriteLine("[UDS] Free response message: {0}", UDS_STATUS_OK_KO(status))
        Next
    End Sub

    Private Class ReceptionWaitHandleWrapper
        Inherits System.Threading.WaitHandle
        Public Sub New(handle As IntPtr)
            Me.SafeWaitHandle = New Microsoft.Win32.SafeHandles.SafeWaitHandle(handle, False)
        End Sub
    End Class

    ''' <summary>ISOTP: wait an ISOTP "PONG" message</summary>
    ''' <param name="client_handle">client channel handle</param>
    ''' <param name="repeat_wait_counter">maximum number of WaitForSingleObject to do (to avoid infinite loop)</param>
    ''' <returns>if we received a PONG message (true) or not (false)</returns>
    Function wait_isotp_pong_response(client_handle As cantp_handle, repeat_wait_counter As Integer) As Boolean

        ' Init variables
        Dim rx_msg As cantp_msg = New cantp_msg()
        Dim repeat As Integer = 0
        Dim get_pong_msg As Boolean = False
        Dim status As cantp_status
        Dim wait_result As Boolean
        Dim timestamp_buffer As cantp_timestamp

        ' Get ISOTP receive event
        Dim byte_array As Byte() = New Byte(IntPtr.Size) {}
        status = CanTpApi.GetValue_2016(client_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, byte_array, IntPtr.Size)
        Dim handle_ptr As IntPtr = IntPtr.Zero
        If IntPtr.Size = 4 Then
            handle_ptr = New IntPtr(BitConverter.ToInt32(byte_array, 0))
        Else
            handle_ptr = New IntPtr(BitConverter.ToInt64(byte_array, 0))
        End If
        Dim receive_event As ReceptionWaitHandleWrapper = New ReceptionWaitHandleWrapper(handle_ptr)
        Console.WriteLine("[ISOTP] Get isotp receive event parameter: {0}", ISOTP_STATUS_OK_KO(status))

        ' Wait a receive event on isotp message
        Do
            wait_result = receive_event.WaitOne(1000)
            Console.WriteLine("[ISOTP] Wait a receive event from isotp: {0}", OK_KO(wait_result))

            ' Read ISOTP messages
            Do
                status = CanTpApi.Read_2016(client_handle, rx_msg, timestamp_buffer, cantp_msgtype.PCANTP_MSGTYPE_NONE)

                ' Check if we received a "PONG" message
                If Not CanTpApi.StatusIsOk_2016(status, cantp_status.PCANTP_STATUS_NO_MESSAGE) Then
                    Console.WriteLine("[ISOTP] Read ISOTP message: {0}", ISOTP_STATUS_OK_KO(status))

                    Dim msgok As Boolean = False
                    Dim rx_byte_array As Byte() = New Byte(MSG_SIZE - 1) {}
                    If (UDSApi.StatusIsOk_2013(CType(status, uds_status)) And (rx_msg.Msgdata_any_Copy.length = MSG_SIZE)) Then
                        msgok = CanTpApi.getData_2016(rx_msg, 0, rx_byte_array, MSG_SIZE)
                        If msgok Then
                            Dim i As Integer = 0
                            While msgok AndAlso i < MSG_SIZE
                                If Chr(rx_byte_array(i)) = ISOTP_RESPONSE_MSG(i) Then
                                    msgok = True
                                Else
                                    msgok = False
                                End If
                                i += 1
                            End While
                        End If
                    End If
                    If msgok Then
                        Console.WriteLine("[ISOTP] Message contains ""{0}{1}{2}{3}"": OK", Microsoft.VisualBasic.ChrW(rx_byte_array(0)), Microsoft.VisualBasic.ChrW(rx_byte_array(1)), Microsoft.VisualBasic.ChrW(rx_byte_array(2)), Microsoft.VisualBasic.ChrW(rx_byte_array(3)))
                        get_pong_msg = True
                    End If

                    status = CanTpApi.MsgDataFree_2016(rx_msg)
                    Console.WriteLine("[ISOTP] Free RX message: {0}", ISOTP_STATUS_OK_KO(status))
                End If
            Loop While Not CanTpApi.StatusIsOk_2016(status, cantp_status.PCANTP_STATUS_NO_MESSAGE)
            repeat = repeat + 1
        Loop While (get_pong_msg = False) And (repeat < repeat_wait_counter)
        Return get_pong_msg
    End Function

    ''' <summary>ISOTP client task: send "PING" and wait "PONG" message several times</summary>
    ''' <param name="param">client channel handle</param>
    Sub isotp_client_task(param As cantp_handle)

        ' Initialize variables
        Dim tx_msg As cantp_msg = New cantp_msg()
        Dim mapping As cantp_mapping = New cantp_mapping()
        Dim reverse_mapping As cantp_mapping = New cantp_mapping()
        Dim status As cantp_status = cantp_status.PCANTP_STATUS_UNKNOWN
        Dim client_handle As cantp_handle = param
        Dim response As Boolean
        Dim i As Integer

        ' Create a isotp receive event
        Dim receive_event As System.Threading.AutoResetEvent = New System.Threading.AutoResetEvent(False)
        If IntPtr.Size = 4 Then
            Dim tmp_buffer As UInt32 = Convert.ToUInt32(receive_event.SafeWaitHandle.DangerousGetHandle().ToInt32())
            status = CanTpApi.SetValue_2016(client_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, tmp_buffer, CType(System.Runtime.InteropServices.Marshal.SizeOf(tmp_buffer), UInt32))
        ElseIf IntPtr.Size = 8 Then
            Dim tmp_buffer As Int64 = receive_event.SafeWaitHandle.DangerousGetHandle().ToInt64()
            Dim byte_array As Byte() = BitConverter.GetBytes(tmp_buffer)
            status = CanTpApi.SetValue_2016(client_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, byte_array, CType(System.Runtime.InteropServices.Marshal.SizeOf(tmp_buffer), UInt64))
        End If
        Console.WriteLine("[ISOTP] Set isotp receive event parameter: {0}", ISOTP_STATUS_OK_KO(status))

        ' Create a simple isotp physical mapping:
        '    - Source 0xA1 (client), target 0xA2 (server), CAN id 0xA1, CAN ID flow control 0xA2
        '    - Diagnostic message in a classic format
        mapping.can_id = &HA1
        mapping.can_id_flow_ctrl = &HA2
        mapping.can_msgtype = cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD
        mapping.netaddrinfo.extension_addr = &H0
        mapping.netaddrinfo.format = cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL
        mapping.netaddrinfo.msgtype = cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC
        mapping.netaddrinfo.source_addr = &HA1
        mapping.netaddrinfo.target_addr = &HA2
        mapping.netaddrinfo.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL

        ' Create the associated isotp reversed mapping:
        reverse_mapping = mapping
        reverse_mapping.can_id = mapping.can_id_flow_ctrl
        reverse_mapping.can_id_flow_ctrl = mapping.can_id
        reverse_mapping.netaddrinfo.source_addr = mapping.netaddrinfo.target_addr
        reverse_mapping.netaddrinfo.target_addr = mapping.netaddrinfo.source_addr

        ' Add ISOTP mappings on channels
        status = CanTpApi.AddMapping_2016(client_handle, mapping)
        Console.WriteLine("[ISOTP] Add a simple isotp mapping: {0}", ISOTP_STATUS_OK_KO(status))
        status = CanTpApi.AddMapping_2016(client_handle, reverse_mapping)
        Console.WriteLine("[ISOTP] Add the reverse isotp mapping: {0}", ISOTP_STATUS_OK_KO(status))

        ' Initialize ISOTP Tx message containing "PING"
        status = CanTpApi.MsgDataAlloc_2016(tx_msg, cantp_msgtype.PCANTP_MSGTYPE_ISOTP)
        Console.WriteLine("[ISOTP] Allocate ISOTP tx message: {0}", ISOTP_STATUS_OK_KO(status))
        Dim msg_array As Byte() = Encoding.ASCII.GetBytes(ISOTP_REQUEST_MSG)
        status = CanTpApi.MsgDataInit_2016(tx_msg, mapping.can_id, mapping.can_msgtype, MSG_SIZE, msg_array, mapping.netaddrinfo)
        Console.WriteLine("[ISOTP] Initialize ISOTP tx message: {0}", ISOTP_STATUS_OK_KO(status))

        ' Send "PING" ISOTP message and wait "PONG" response
        For i = 1 To 10
            status = CanTpApi.Write_2016(client_handle, tx_msg)
            Console.WriteLine("[ISOTP] Send ISOTP ""PING"" message ({0}): {1}", i, ISOTP_STATUS_OK_KO(status))
            response = wait_isotp_pong_response(client_handle, 3)
            Console.WriteLine("[ISOTP] Get ISOTP ""PONG"" response ({0}): {1}", i, OK_KO(response))
        Next

        ' Close receive event
        If IntPtr.Size = 4 Then
            Dim tmp_buffer As UInt32 = 0
            status = CanTpApi.SetValue_2016(client_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, tmp_buffer, CType(System.Runtime.InteropServices.Marshal.SizeOf(tmp_buffer), UInt32))
        ElseIf IntPtr.Size = 8 Then
            Dim tmp_buffer As Int64 = 0
            Dim byte_array As Byte() = BitConverter.GetBytes(tmp_buffer)
            status = CanTpApi.SetValue_2016(client_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, byte_array, CType(System.Runtime.InteropServices.Marshal.SizeOf(tmp_buffer), UInt64))
        End If
        Console.WriteLine("[ISOTP] Stop ISOTP receive event: {0}", ISOTP_STATUS_OK_KO(status))
        receive_event.Close()
        Console.WriteLine("[ISOTP] Close ISOTP receive event: {0}", ISOTP_STATUS_OK_KO(status))

        ' Free messages
        status = CanTpApi.MsgDataFree_2016(tx_msg)
        Console.WriteLine("[ISOTP] Free ISOTP TX message: {0}", ISOTP_STATUS_OK_KO(status))
    End Sub

    ''' <summary>Entry point of the program, start a UDS channel, ask in the same time UDS TesterPresent and isotp request</summary>
    ''' <returns>By convention, return success.</returns>
    Function Main(ByVal args As String()) As Integer
        Dim buffer As StringBuilder
        Dim status As uds_status
        Dim client_handle As cantp_handle
        Dim uds_client As Thread
        Dim isotp_client As Thread

        ' Initialize variables
        client_handle = cantp_handle.PCANTP_HANDLE_USBBUS1 ' TODO: modify the value according to your available PCAN devices.
        buffer = New StringBuilder(BUFFER_SIZE)

        ' Print version informations
        status = UDSApi.GetValue_2013(cantp_handle.PCANTP_HANDLE_NONEBUS, uds_parameter.PUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE)
        Console.WriteLine("PCAN-UDS API Version - {0}: {1}", buffer, UDS_STATUS_OK_KO(status))

        ' Initialize client channel
        status = UDSApi.Initialize_2013(client_handle, cantp_baudrate.PCANTP_BAUDRATE_500K, CType(0, cantp_hwtype), 0, 0)
        Console.WriteLine("Initialize channel: {0}", UDS_STATUS_OK_KO(status))

        ' Start uds and isotp clients
        uds_client = New Thread(Sub() uds_client_task(client_handle))
        uds_client.Start()

        isotp_client = New Thread(Sub() isotp_client_task(client_handle))
        isotp_client.Start()

        isotp_client.Join()
        uds_client.Join()

        ' Close channel
        status = UDSApi.Uninitialize_2013(client_handle)
        Console.WriteLine("Uninitialize channel: {0}", UDS_STATUS_OK_KO(status))

        ' Exit
        Console.WriteLine("Press any key to continue...")
        Console.In.Read()
        Return 0
    End Function

    Function OK_KO(test As Boolean) As String
        If test Then Return "OK"
        Return "KO"
    End Function
    Function UDS_STATUS_OK_KO(test As uds_status) As String
        Return OK_KO(UDSApi.StatusIsOk_2013(test))
    End Function
    Function ISOTP_STATUS_OK_KO(test As cantp_status) As String
        Return OK_KO(CanTpApi.StatusIsOk_2016(test))
    End Function
End Module
